/* Copyright (c) 1994 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Author: Timothy Rowley */

/* {{{ Includes */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include "mgP.h"
#include "mgx11P.h"

/* }}} */

static endPoint *mug=NULL;
static mugSize = 0;
static int redmask;
static int greenmask;
static int bluemask;

#ifdef __GNUC__
inline
#endif
static int
RGBtoVal(int r, int g, int b)
{
    return ((redmask  & (r|(r<<8)|(r<<16)|(r<<24))) |
            (greenmask& (g|(g<<8)|(g<<16)|(g<<24))) |
            (bluemask & (b|(b<<8)|(b<<16)|(b<<24))));
}

static int
swap32(unsigned long word)
{
    return (word >> 24) | (word >> 8) & 0xFF00 | ((word&0xFF00) << 8) | ((word&0xFF) << 24);
}

void
Xmgr_24init(int rmask, int gmask, int bmask)
{
    long word = 1;
    if((*(char *)&word != 0) == (ImageByteOrder(_mgx11c->mgx11display) == LSBFirst)) {
        redmask = rmask;
        greenmask = gmask;
        bluemask = bmask;
    } else {
        /* Flip byte order to server's own format. */
        redmask = swap32(rmask);
        greenmask = swap32(gmask);
        bluemask = swap32(bmask);
    }
}


/* {{{ clearing routine */

void
Xmgr_24clear(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, int *color, int flag,
	     int fullclear, int xmin, int ymin, int xmax, int ymax)
{
    int *ptr = (int *)buf;
    int i, fill, end, x, length, pos;
    fill = RGBtoVal(color[0], color[1], color[2]);

    if (mug==NULL)
    {
	mug = (endPoint *)malloc(sizeof(endPoint)*height);
	mugSize = height;
    }
    if (height>mugSize)
    {
	mug = (endPoint *)realloc(mug, sizeof(endPoint)*height);
	mugSize = height;
    }

    if (fullclear)
    {
	end = (width*height)/4;
	for (i=0; i<end; i++)
	    ptr[i] = fill;
	
	if (flag)
	    for (i=0; i<zwidth*height; i++)
		zbuf[i] = 1.0;
    }

    xmin = MAX(xmin,0);
    length = (MIN(zwidth-1,xmax)-xmin+1);
    ymin = MAX(ymin,0);
    ymax = MIN(height-1,ymax);
    for (i=ymin; i<=ymax; i++)
    {
	ptr = (int *)(buf+width*i+xmin*4);
	for (x=0; x<length; x++)
	    ptr[x] = fill;
    }
    length = MIN(zwidth-1,xmax)-xmin+1; 
    if (flag)
	for (i=ymin; i<=ymax; i++)
	{
	    pos = i*zwidth+xmin;
	    for (x=0; x<length; x++)
	        zbuf[pos+x] = 1.0;
	}
}

/* }}} */

/* {{{ single lines */

static void
wideline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1, int x2, int y2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int i, end;
    int col = RGBtoVal(color[0], color[1], color[2]);
    int width4 = width>>2;
    int *ptr = (int *)buf;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth))
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		    ptr[i*width4+x] = col;
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
		for (i=MAX(0,x-lwidth/2), end=MIN(zwidth-1,x-lwidth/2+lwidth); i<end; i++)
		    ptr[y*width4+i] = col;
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
}


void
Xmgr_24line(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	   int x1, int y1, float z1, int x2, int y2, float z2, int lwidth, int *color)
{
    int d, x, y, ax, ay, sx, sy, dx, dy;
    int width4 = width>>2;
    int *ptr=(int *)(buf+y1*width+4*x1);
    int col=RGBtoVal(color[0], color[1], color[2]);

    if (lwidth > 1)
    {
	wideline(buf, zbuf, zwidth, width, height, x1, y1, x2, y2,
		   lwidth, color);
	return;
    }

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		*ptr = col;
	    if (x==x2) return;
	    if (d>=0)
	    {
		y+=sy;
		if (sy==1)
		    ptr += width4;
		else
		    ptr -= width4;
		d -= ax;
	    }
	    x += sx;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		*ptr = col;
	    if (y==y2) return;
	    if (d>=0)
	    {
		x += sx;
		ptr += sx;
		d -= ay;
	    }
	    y+=sy;
	    if (sy==1)
		ptr += width4;
	    else
		ptr -= width4;
	    d += ax;
	}
    }
}


static void
wideZline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1, float z1,
	 int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int jumps, total;
    int i, end;
    int col = RGBtoVal(color[0], color[1], color[2]);
    int width4 = width>>2;
    int *ptr = (int *)buf;

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    jumps = 0;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth))
	    {
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[i*zwidth+x]) && (z>-1.0))
		    {
			ptr[i*width4+x] = col;
			zbuf[i*zwidth+x] = z;
		    }
		}
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(zwidth-1,x-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[y*zwidth+i]) && (z>-1.0))
		    {
			ptr[y*width4+i] = col;
			zbuf[y*zwidth+i] = z;
		    }
		}
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    d += ax;
	}
    }
}

void
Xmgr_24Zline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
	 int x1, int y1, float z1,
	 int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int width4=width>>2;
    int *ptr=(int *)(buf+y1*width+4*x1);
    double delta=0, z;
    int jumps, total;
    int col = RGBtoVal(color[0], color[1], color[2]);

    if (lwidth > 1)
    {
	wideZline(buf, zbuf, zwidth, width, height, x1, y1, z1, x2, y2,
		   z2, lwidth, color);
	return;
    }

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    jumps = 0;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height) 
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		*ptr = col;
		zbuf[y*zwidth+x] = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		if (sy < 0)
		    ptr -= width4;
		else
		    ptr += width4;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height)
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		*ptr = col;
		zbuf[y*zwidth+x] = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		ptr += sx;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
   	    if (sy < 0)
	        ptr -= width4;
	    else
		ptr += width4;
	    d += ax;
	}
    }
}

static void
wide24Gline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, 
	     int x1, int y1, double z1, int r1, int g1, int b1,
	     int x2, int y2, double z2, int r2, int g2, int b2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    double r, g, b, rdelta, gdelta, bdelta;
    int total, i, end;
    int *ptr=(int *)buf;
    int width4 = width>>2;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x >= 0) && (x < width))
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		    ptr[i*width4+x] =  RGBtoVal(r, g, b);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		    ptr[y*width4+i] =  RGBtoVal(r, g, b);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ay;
	    }
	    y++;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ax;
	}
    }
}

void
Xmgr_24Gline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    int width4 = width>>2;
    int *ptr;
    int r1, r2, g1, g2, b1, b2;
    double r, g, b, rdelta, gdelta, bdelta, z1, z2;
    int total, x1, y1, x2, y2;

    if (p2->y<p1->y)
    {
	x1 = p2->x; y1 = p2->y;
	x2 = p1->x; y2 = p1->y;
	r1 = 255*p2->vcol.r; g1 = 255*p2->vcol.g; b1 = 255*p2->vcol.b;
	r2 = 255*p1->vcol.r; g2 = 255*p1->vcol.g; b2 = 255*p1->vcol.b;
    }
    else
    {
	x1 = p1->x; y1 = p1->y;
	x2 = p2->x; y2 = p2->y;
	r1 = 255*p1->vcol.r; g1 = 255*p1->vcol.g; b1 = 255*p1->vcol.b;
	r2 = 255*p2->vcol.r; g2 = 255*p2->vcol.g; b2 = 255*p2->vcol.b;
    }

    if (lwidth > 1)
    {
	wide24Gline(buf, zbuf, zwidth, width, height, x1, y1, z1, r1, g1, b1,
		   x2, y2, z2, r2, g2, b2, lwidth);
	return;
    }

    ptr = (int *)(buf+y1*width+x1*4);

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = RGBtoVal(r, g, b);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += width4;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return; 
	    if ((x>=0) && (x<width) && (y>=0))
		*ptr = RGBtoVal(r, g, b);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += sx;
		d -= ay;
	    }
	    y++;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += width4;
	    d += ax;
	}
    }
}

static void
wide24GZline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, 
	     int x1, int y1, double z1, int r1, int g1, int b1,
	     int x2, int y2, double z2, int r2, int g2, int b2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    double delta=0, z;
    double r, g, b, rdelta, gdelta, bdelta;
    int total, i, end;
    int *ptr=(int *)buf;
    int width4 = width>>2;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    delta = (z2-z1)/total;
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x >= 0) && (x < width))
		for (i=MAX(0,y-lwidth/2), end=MIN(height-1,y-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[i*zwidth+x]) && (z>-1.0))
		    {
		    	ptr[i*width4+x] =  RGBtoVal(r, g, b);
		    	zbuf[i*zwidth+x] = z;
		    } 
		}
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((y>=0) && (y<height))
	    {
		for (i=MAX(0,x-lwidth/2), end=MIN(width-1,x-lwidth/2+lwidth); i<end; i++)
		{
		    if ((z<zbuf[y*zwidth+i]) && (z>-1.0))
		    {
		    	ptr[y*width4+i] =  RGBtoVal(r, g, b);
		    	zbuf[y*zwidth+i] = z;
		    }
		}
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    d += ax;
	}
    }
}

void
Xmgr_24GZline(unsigned char *buf, float *zbuf, int zwidth, 
	      int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, dx, dy;
    int width4 = width>>2;
    int *ptr;
    float *zptr;
    double delta=0, z;
    int r1, r2, g1, g2, b1, b2;
    double r, g, b, rdelta, gdelta, bdelta, z1, z2;
    int total, x1, y1, x2, y2;

    if (p2->y<p1->y)
    {
	x1 = p2->x; y1 = p2->y; z1 = p2->z - _mgc->zfnudge;
	x2 = p1->x; y2 = p1->y; z2 = p1->z - _mgc->zfnudge;
	r1 = 255*p2->vcol.r; g1 = 255*p2->vcol.g; b1 = 255*p2->vcol.b;
	r2 = 255*p1->vcol.r; g2 = 255*p1->vcol.g; b2 = 255*p1->vcol.b;
    }
    else
    {
	x1 = p1->x; y1 = p1->y; z1 = p1->z - _mgc->zfnudge;
	x2 = p2->x; y2 = p2->y; z2 = p2->z - _mgc->zfnudge;
	r1 = 255*p1->vcol.r; g1 = 255*p1->vcol.g; b1 = 255*p1->vcol.b;
	r2 = 255*p2->vcol.r; g2 = 255*p2->vcol.g; b2 = 255*p2->vcol.b;
    }
    if (lwidth > 1)
    {
	wide24GZline(buf, zbuf, zwidth, width, height, x1, y1, z1, r1, g1, b1,
		   x2, y2, z2, r2, g2, b2, lwidth);
	return;
    }
    ptr = (int *)(buf+y1*width+x1*4);
    zptr = zbuf+y1*zwidth+x1;

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    g = g1;
    b = b1;
    total = ABS(dx)+ABS(dy);
    delta = (z2-z1)/total;
    rdelta = (r2-r1)/(double)total;
    gdelta = (g2-g1)/(double)total;
    bdelta = (b2-b1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if (y>=height) return;
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = RGBtoVal(r, g, b);
		*zptr = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y++;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += width4;
		zptr += zwidth;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += sx;
	    zptr += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if (y>=height) return; 
	    if ((x>=0) && (x<width) && (y>=0) && (z<*zptr) && (z>-1.0))
	    {
		*ptr = RGBtoVal(r, g, b);
		*zptr = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta; g += gdelta; b += bdelta;
		ptr += sx;
		zptr += sx;
		d -= ay;
	    }
	    y++;
	    z += delta;
	    r += rdelta; g += gdelta; b += bdelta;
	    ptr += width4;
	    zptr += zwidth;
	    d += ax;
	}
    }
}

/* }}} */

/* {{{ polygon scan conversion */

static void
Xmgr_doLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	       int height, int miny, int maxy, int *color, endPoint *mug)
{
    register int *ptr;           /* pixel pointers   */
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    int col=RGBtoVal(color[0], color[1], color[2]);

    for(y = miny; y <= maxy; y++)
    {
	if (mug[y].P1x < 0)
	    x1 = 0;
	else
	    x1 = mug[y].P1x;
	if (mug[y].P2x >= zwidth)
	    x2 = zwidth-1;
	else
	    x2 = mug[y].P2x;
	ptr = (int *)(buf+width*y+x1*4);
	for (i=x1; i<=x2; i++, ptr++)
		*ptr = col;
    }
}

static void
Xmgr_ZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	       int height, int miny, int maxy, int *color, endPoint *mug)
{
    register int *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int col=RGBtoVal(color[0], color[1], color[2]);

    for(y = miny; y <= maxy; y++)
    {
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(zwidth-1-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	}
	zdelta = (x1==x2) ? 0 : ((z2-z)/(x2-x1));
	ptr = (int *)(buf+width*y+x1*4); zptr = zbuf+zwidth*y+x1;
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	{
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
		if ((z < *zptr) && (z>-1.0))
		{
		    *ptr = col;
		    *zptr = z;
		}
	}
	else
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
		if (z < *zptr)
		{
		    *ptr = col;
		    *zptr = z;
		}
    }
}

static void
Xmgr_GdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
		int height, int miny, int maxy, endPoint *mug)
{
    int *ptr;           /* pixel pointers   */
    register int i, x2;           /* dithering junk   */
    int y, x1;                          /* current line     */
    int sr, sg, sb, er, eg, eb, dr, dg, db, r, g, b, dx;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    g = mug[y].P1g - (mug[y].P2g-mug[y].P1g) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    b = mug[y].P1b - (mug[y].P2b-mug[y].P1b) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    r = mug[y].P1r;
	    g = mug[y].P1g;
	    b = mug[y].P1b;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    dr = (mug[y].P1r + (mug[y].P2r-mug[y].P1r) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-r;
	    dg = (mug[y].P1g + (mug[y].P2g-mug[y].P1g) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-g;
	    db = (mug[y].P1b + (mug[y].P2b-mug[y].P1b) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-b;
	}
	else
	{
	    x2 = mug[y].P2x;
	    dr = mug[y].P2r-r;
	    dg = mug[y].P2g-g;
	    db = mug[y].P2b-b;
	}
	dx = x2-x1;
	er = (dr << 1) - dx;
	eg = (dg << 1) - dx;
	eb = (db << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	sg = SGN(dg); dg = abs(dg)<<1;
	sb = SGN(db); db = abs(db)<<1;
	dx <<= 1;
	ptr = (int *)(buf+width*y+x1*4);
	for (i=x1; i<=x2; i++, ptr++)
	{
	    *ptr = RGBtoVal(r, g, b);
	    if (dx) {
	    while (er>0)
	    {
		r += sr;
		er -= dx;
	    }
	    while (eg>0)
	    {
		g += sg;
		eg -= dx;
	    }
	    while (eb>0)
	    {
		b += sb;
		eb -= dx;
	    }
	    }
	    er += dr; eg += dg; eb += db;
	}
    }
}

static void
Xmgr_GZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
		int height, int miny, int maxy, endPoint *mug)
{
    int *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, x2;           /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int sr, sg, sb, er, eg, eb, dr, dg, db, r, g, b, dx;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    g = mug[y].P1g - (mug[y].P2g-mug[y].P1g) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    b = mug[y].P1b - (mug[y].P2b-mug[y].P1b) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	    r = mug[y].P1r;
	    g = mug[y].P1g;
	    b = mug[y].P1b;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	    dr = (mug[y].P1r + (mug[y].P2r-mug[y].P1r) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-r;
	    dg = (mug[y].P1g + (mug[y].P2g-mug[y].P1g) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-g;
	    db = (mug[y].P1b + (mug[y].P2b-mug[y].P1b) *
		   (x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x))-b;
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	    dr = mug[y].P2r-r;
	    dg = mug[y].P2g-g;
	    db = mug[y].P2b-b;
	}
	dx = x2-x1;
	zdelta = dx ? (z2-z)/dx : 0;
	er = (dr << 1) - dx;
	eg = (dg << 1) - dx;
	eb = (db << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	sg = SGN(dg); dg = abs(dg)<<1;
	sb = SGN(db); db = abs(db)<<1;
	dx <<= 1;
	ptr = (int *)(buf+width*y+x1*4); zptr = zbuf+zwidth*y+x1;
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
	    {
		if ((z < *zptr) && (z>-1.0))
		{
		    *ptr = RGBtoVal(r, g, b);
		    *zptr = z;
		}
		if (dx) {
		while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		while (eg>0)
		{
		    g += sg;
		    eg -= dx;
		}
		while (eb>0)
		{
		    b += sb;
		    eb -= dx;
		}
		}
		er += dr; eg += dg; eb += db;
	    }
	else
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++, ptr++)
	    {
		if (z < *zptr)
		{
		    *ptr = RGBtoVal(r, g, b);
		    *zptr = z;
		}
		if (dx) {
		while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		while (eg>0)
		{
		    g += sg;
		    eg -= dx;
		}
		while (eb>0)
		{
		    b += sb;
		    eb -= dx;
		}
		}
		er += dr; eg += dg; eb += db;
	    }
    }
}

void
Xmgr_24Zpoly(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_Zpolyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		   Xmgr_ZdoLines);
}

void
Xmgr_24poly(unsigned char *buf, float *zbuf, int zwidth, 
	    int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_polyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		  Xmgr_doLines);
}

void
Xmgr_24GZpoly(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	     CPoint3 *p, int n, int *dummy)
{
    Xmgr_GZpolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		    Xmgr_GZdoLines);
}

void
Xmgr_24Gpoly(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	     CPoint3 *p, int n, int *dummy)
{
    Xmgr_Gpolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		    Xmgr_GdoLines);
}

/* }}} */

/* {{{ multi-line scan conversion */

void
Xmgr_24polyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    ((int *)buf)[(int)(p0->y)*(width/4)+(int)p0->x] = 
		RGBtoVal(color[0], color[1], color[2]);
	return;
    }
    p1 = &p[1];
    Xmgr_24line(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0, 
	        p1->x, p1->y, 0, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_24line(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0,
		    p1->x, p1->y, 0, lwidth, color);
    }
}


void
Xmgr_24Zpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y)*zwidth+(int)p0->x]) && (p0->z>-1.0))
		((int *)buf)[(int)(p0->y)*(width/4)+(int)p0->x] = 
		    RGBtoVal(color[0], color[1], color[2]);
	return;
    }
    p1 = &p[1];
    Xmgr_24Zline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, 
		 p1->x, p1->y, p1->z, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_24Zline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, 
		     p1->x, p1->y, p1->z, lwidth, color);
    }
}

void
Xmgr_24Gpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    ((int *)buf)[(int)(p0->y)*(width/4)+(int)p0->x] = 
		RGBtoVal(color[0], color[1], color[2]);
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
		     Xmgr_24line, Xmgr_24Gline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
	Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			 Xmgr_24line, Xmgr_24Gline);
    }
}

void
Xmgr_24GZpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y)*zwidth+(int)p0->x]) && (p0->z>-1.0))
		((int *)buf)[(int)(p0->y)*(width/4)+(int)p0->x] = 
		    RGBtoVal(color[0], color[1], color[2]);
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
		     Xmgr_24Zline, Xmgr_24GZline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
	Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			 Xmgr_24Zline, Xmgr_24GZline);
    }
}

/* }}} */

/*
Local variables:
folded-file: t
*/

