/* Copyright (c) 1994 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Author: Timothy Rowley */

/* {{{ Includes */

#include <stdio.h>
#include "mgP.h"
#include "mgx11P.h"

/* }}} */

/* {{{ dither and masking tables */

static unsigned char dither[65][8] = {
	{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00},
	{0x80, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x00, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x02, 0x00},
	{0x88, 0x00, 0x20, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0x88, 0x00, 0x22, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0x88, 0x00, 0xA2, 0x00, 0x88, 0x00, 0x22, 0x00},
	{0x88, 0x00, 0xA2, 0x00, 0x88, 0x00, 0x2A, 0x00},
	{0x88, 0x00, 0xA2, 0x00, 0x88, 0x00, 0xAA, 0x00},
	{0x88, 0x00, 0xAA, 0x00, 0x88, 0x00, 0xAA, 0x00},
	{0xA8, 0x00, 0xAA, 0x00, 0x88, 0x00, 0xAA, 0x00},
	{0xA8, 0x00, 0xAA, 0x00, 0x8A, 0x00, 0xAA, 0x00},
	{0xA8, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x00, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x04, 0xAA, 0x00},
	{0xAA, 0x40, 0xAA, 0x00, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x00, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x00},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x01},
	{0xAA, 0x44, 0xAA, 0x10, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x44, 0xAA, 0x11, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x44, 0xAA, 0x51, 0xAA, 0x44, 0xAA, 0x11},
	{0xAA, 0x44, 0xAA, 0x51, 0xAA, 0x44, 0xAA, 0x15},
	{0xAA, 0x44, 0xAA, 0x51, 0xAA, 0x44, 0xAA, 0x55},
	{0xAA, 0x44, 0xAA, 0x55, 0xAA, 0x44, 0xAA, 0x55},
	{0xAA, 0x54, 0xAA, 0x55, 0xAA, 0x44, 0xAA, 0x55},
	{0xAA, 0x54, 0xAA, 0x55, 0xAA, 0x45, 0xAA, 0x55},
	{0xAA, 0x54, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xAA, 0xD5, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55},
	{0xAA, 0xD5, 0xAA, 0x55, 0xAA, 0x5D, 0xAA, 0x55},
	{0xAA, 0xD5, 0xAA, 0x55, 0xAA, 0xDD, 0xAA, 0x55},
	{0xAA, 0xDD, 0xAA, 0x55, 0xAA, 0xDD, 0xAA, 0x55},
	{0xAA, 0xDD, 0xAA, 0x75, 0xAA, 0xDD, 0xAA, 0x55},
	{0xAA, 0xDD, 0xAA, 0x75, 0xAA, 0xDD, 0xAA, 0x57},
	{0xAA, 0xDD, 0xAA, 0x75, 0xAA, 0xDD, 0xAA, 0x77},
	{0xAA, 0xDD, 0xAA, 0x77, 0xAA, 0xDD, 0xAA, 0x77},
	{0xAA, 0xDD, 0xAA, 0xF7, 0xAA, 0xDD, 0xAA, 0x77},
	{0xAA, 0xDD, 0xAA, 0xF7, 0xAA, 0xDD, 0xAA, 0x7F},
	{0xAA, 0xDD, 0xAA, 0xF7, 0xAA, 0xDD, 0xAA, 0xFF},
	{0xAA, 0xDD, 0xAA, 0xFF, 0xAA, 0xDD, 0xAA, 0xFF},
	{0xAA, 0xFD, 0xAA, 0xFF, 0xAA, 0xDD, 0xAA, 0xFF},
	{0xAA, 0xFD, 0xAA, 0xFF, 0xAA, 0xDF, 0xAA, 0xFF},
	{0xAA, 0xFD, 0xAA, 0xFF, 0xAA, 0xFF, 0xAA, 0xFF},
	{0xAA, 0xFF, 0xAA, 0xFF, 0xAA, 0xFF, 0xAA, 0xFF},
	{0xEA, 0xFF, 0xAA, 0xFF, 0xAA, 0xFF, 0xAA, 0xFF},
	{0xEA, 0xFF, 0xAA, 0xFF, 0xAE, 0xFF, 0xAA, 0xFF},
	{0xEA, 0xFF, 0xAA, 0xFF, 0xEE, 0xFF, 0xAA, 0xFF},
	{0xEE, 0xFF, 0xAA, 0xFF, 0xEE, 0xFF, 0xAA, 0xFF},
	{0xEE, 0xFF, 0xBA, 0xFF, 0xEE, 0xFF, 0xAA, 0xFF},
	{0xEE, 0xFF, 0xBA, 0xFF, 0xEE, 0xFF, 0xAB, 0xFF},
	{0xEE, 0xFF, 0xBA, 0xFF, 0xEE, 0xFF, 0xBB, 0xFF},
	{0xEE, 0xFF, 0xBB, 0xFF, 0xEE, 0xFF, 0xBB, 0xFF},
	{0xEE, 0xFF, 0xFB, 0xFF, 0xEE, 0xFF, 0xBB, 0xFF},
	{0xEE, 0xFF, 0xFB, 0xFF, 0xEE, 0xFF, 0xBF, 0xFF},
	{0xEE, 0xFF, 0xFB, 0xFF, 0xEE, 0xFF, 0xFF, 0xFF},
	{0xEE, 0xFF, 0xFF, 0xFF, 0xEE, 0xFF, 0xFF, 0xFF},
	{0xFE, 0xFF, 0xFF, 0xFF, 0xEE, 0xFF, 0xFF, 0xFF},
	{0xFE, 0xFF, 0xFF, 0xFF, 0xEF, 0xFF, 0xFF, 0xFF},
	{0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
	{0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF},
};

static unsigned char bits[8] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};

static unsigned char startBits[8] = {
    0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01};

static unsigned char endBits[8] = {
    0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF};

/* }}} */

static endPoint *mug=NULL;
static int mugSize = 0;
static int flipped = 0;

static int
RGB2gray(int *color)
{
    int n;
    n = (64.0*(color[0]*0.299 + color[1]*0.587 + color[2]*0.114))/255.0;
    if (n>64)
	return 64;
    else
	return n;
}

static int
RGB2gray2(float r, float g, float b)
{
    int n;
    n = 64.0*(r*0.299 + g*0.587 + b*0.114);
    if (n>64)
	return 64;
    else
	return n;
}

#ifdef __GNUC__
inline
#endif
static void
setPixel(unsigned char *buf, int zwidth, int width, int height, int x, int y,
	 int color)
{
    if ((x<0) || (x>=zwidth) || (y<0) || (y>=height))
	return;
    buf[y*width+(x>>3)] = 
	(buf[y*width+(x>>3)] & ~bits[x&0x07]) | 
	(bits[x&0x07] & dither[color][y&0x07]);
}

void
Xmgr_1init(int blackPixel)
{
    int col, i;

    if (blackPixel && (!flipped))
    {
	for (col=0; col<65; col++)
	    for (i=0; i<8; i++)
		dither[col][i] = ~dither[col][i];
	flipped = 1;
    }
}

/* {{{ vertical lines */

static void
Vline(unsigned char *buf, int zwidth, int width, int height, int x,
      int y1, int y2, int color)
{
    int i;
    if ((x<0) || (x>=zwidth))
	return;
    for (i=y1; i<=y2; i++)
	setPixel(buf, zwidth, width, height, x, i, color);
}

static void
VZline(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
       int x, int y1, int y2, float z, int color)
{
    int i;
    if ((x<0) || (x>=zwidth))
	return;
    for (i=y1; i<=y2; i++)
	if ((i>=0) && (i<height) && (z<zbuf[i*zwidth+x]) && (z>-1.0))
	{
	    setPixel(buf, zwidth, width, height, x, i, color);
	    zbuf[i*zwidth+x] = z;
	}
}

/* }}} */

/* {{{ horizontal lines */

static void
Hline(unsigned char *buf, int zwidth, int width, int height, 
     int y, int a, int b, int color)
{
    int x, i;

    if ((y>=0) && (y<height))
    {
        x = a;

	if ((a>=zwidth) || (b<0))
	    return;
	a = MAX(a, 0);
        b = MIN(b, zwidth-1);

	for (i=a; i<=b; i++)
	    setPixel(buf, zwidth, width, height, i, y, color);
    }
}



static void
HZline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
       int y, int a, int b, float z, int color)
{
    register int i, pos, zpos;

    if ((y>=0) && (y<height))
    {
	if ((a>=zwidth) || (b<0))
	    return;

        zpos = y*zwidth+a;
	for (i=a; i<=b; i++, zpos++)
	    if ((i>=0) && (i< zwidth) && (z < zbuf[zpos]) && (z>-1.0))
	    {
		setPixel(buf, zwidth, width, height, i, y, color);
		zbuf[zpos] = z;
	    }
    }
}

/* }}} */

void
Xmgr_1clear(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, int *color, int flag,
	    int fullclear, int xmin, int ymin, int xmax, int ymax)
{
    int i, length, pos, x;
    int col = RGB2gray(color);

    if (mug==NULL)
    {
	mug = (endPoint *)malloc(sizeof(endPoint)*height);
	mugSize = height;
    }
    if (height>mugSize)
    {
	mug = (endPoint *)realloc(mug, sizeof(endPoint)*height);
	mugSize = height;
    }

    if (fullclear)
    {
        for (i=0; i<height; i++)
            memset(buf+i*width, dither[col][i&0x07], width);

        if (flag)
            for (i=0; i<zwidth*height; i++)
                zbuf[i] = 1.0;
	return;
    }
    xmin = MAX(xmin,0) >> 3;
    length = (MIN(zwidth-1,xmax)-xmin+8) >> 3;
    ymin = MAX(ymin,0);
    ymax = MIN(height-1,ymax);

    for (i=ymin; i<=ymax; i++)
	memset(buf+i*width+xmin, dither[col][i&0x07], length);

    length = MIN(zwidth-1,xmax)-xmin+1; 
    if (flag)
	for (i=ymin; i<=ymax; i++)
	{
	    pos = i*zwidth+xmin;
	    for (x=0; x<length; x++)
	        zbuf[pos+x] = 1.0;
	}
}

/* {{{ single lines */

static void
wide1Dline(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    int x1, int y1, int x2, int y2,
	    int lwidth, int *color)
{
    int d, x, y, ax, ay, sx, sy, dx, dy;
    int col = RGB2gray(color);

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    Vline(buf, zwidth, width, height, x, y-lwidth/2,
		  y-lwidth/2+lwidth, col);
	    if (x==x2) return;
	    if (d>=0)
	    {
		y+=sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    Hline(buf, zwidth, width, height, y, x-lwidth/2, x-lwidth/2+lwidth,
		 col);
	    if (y==y2) return;
	    if (d>=0)
	    {
		x += sx;
		d -= ay;
	    }
	    y+=sy;
	    d += ax;
	}
    }
}


void
Xmgr_1Dline(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    int x1, int y1, float z1, int x2, int y2, float z2,
	    int lwidth, int *color)
{
    int d, x, y, ax, ay, sx, sy, dx, dy;
    int col = RGB2gray(color);

    if (lwidth > 1)
    {
	wide1Dline(buf, zbuf, zwidth, width, height, x1, y1, x2, y2,
		   lwidth, color);
	return;
    }

    dx = x2-x1;
    dy = y2-y1;

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		setPixel(buf, zwidth, width, height, x, y, col);
	    if (x==x2) return;
	    if (d>=0)
	    {
		y+=sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		setPixel(buf, zwidth, width, height, x, y, col);
	    if (y==y2) return;
	    if (d>=0)
	    {
		x += sx;
		d -= ay;
	    }
	    y+=sy;
	    d += ax;
	}
    }
}

static void
wide1DZline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, int x1, int y1, float z1,
	     int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int total;
    int col = RGB2gray(color);

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    VZline(buf, zbuf, zwidth, width, height, x, y-lwidth/2,
		   y-lwidth/2+lwidth, z, col);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    HZline(buf, zbuf, zwidth, width, height, y, x-lwidth/2,
		  x-lwidth/2+lwidth, z, col);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    d += ax;
	}
    }
}


void
Xmgr_1DZline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, int x1, int y1, float z1,
	     int x2, int y2, float z2, int lwidth, int *color)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int total;
    int col = RGB2gray(color);

    if (lwidth > 1)
    {
	wide1DZline(buf, zbuf, zwidth, width, height, x1, y1, z1, x2, y2, z2,
		    lwidth, color);
	return;
    }

    z1 -= _mgc->zfnudge;
    z2 -= _mgc->zfnudge;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    if (total)
        delta = (z2-z1)/total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height) 
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		setPixel(buf, zwidth, width, height, x, y, col);
		zbuf[y*zwidth+x] = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height)
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		setPixel(buf, zwidth, width, height, x, y, col);
		zbuf[y*zwidth+x] = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    d += ax;
	}
    }
}

void
wide1DGZline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, int x1, int y1, double z1, int r1,
	     int x2, int y2, double z2, int r2, int lwidth)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int total;
    double r, rdelta;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    rdelta = (r2-r1)/(double)total;
    delta = (z2-z1)/total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    VZline(buf, zbuf, zwidth, width, height, x, y-lwidth/2,
		   y-lwidth/2+lwidth, z, r);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		r += rdelta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    HZline(buf, zbuf, zwidth, width, height, y, x-lwidth/2,
		  x-lwidth/2+lwidth, z, r);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    r += rdelta;
	    d += ax;
	}
    }
}

void
Xmgr_1DGZline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    float delta=0, z;
    int total;
    int x1, y1, x2, y2, r1, r2;
    double z1, z2;
    double r, rdelta;

    x1 = p1->x; x2 = p2->x;
    y1 = p1->y; y2 = p2->y;
    r1 = RGB2gray2(p1->vcol.r, p1->vcol.g, p1->vcol.b);
    r2 = RGB2gray2(p2->vcol.r, p2->vcol.g, p2->vcol.b);
    z1 = p1->z-_mgc->zfnudge; z2 = p2->z-_mgc->zfnudge;

    if (lwidth > 1)
    {
	wide1DGZline(buf, zbuf, zwidth, width, height, x1, y1, z1, r1,
		     x2, y2, z2, r2, lwidth);
	return;
    }

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    z = z1;
    r = r1;
    rdelta = (r2-r1)/(double)total;
    delta = (z2-z1)/total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height) 
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		setPixel(buf, zwidth, width, height, x, y, r);
		zbuf[y*zwidth+x] = z;
	    }
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		z += delta;
		r += rdelta;
		d -= ax;
	    }
	    x += sx;
	    z += delta;
	    r += rdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height)
		&& (z<zbuf[y*zwidth+x]) && (z>-1.0))
	    {
		setPixel(buf, zwidth, width, height, x, y, r);
		zbuf[y*zwidth+x] = z;
	    }
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		z += delta;
		r += rdelta;
		d -= ay;
	    }
	    y += sy;
	    z += delta;
	    r += rdelta;
	    d += ax;
	}
    }
}

void
wide1DGline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, int x1, int y1, int r1,
	     int x2, int y2, int r2, int lwidth)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int total;
    double r, rdelta;

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    r = r1;
    rdelta = (r2-r1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    Vline(buf, zwidth, width, height, x, y-lwidth/2,
		   y-lwidth/2+lwidth, r);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		r += rdelta;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    Hline(buf, zwidth, width, height, y, x-lwidth/2,
		  x-lwidth/2+lwidth, r);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta;
		d -= ay;
	    }
	    y += sy;
	    r += rdelta;
	    d += ax;
	}
    }
}

void
Xmgr_1DGline(unsigned char *buf, float *zbuf, int zwidth,
	     int width, int height, CPoint3 *p1, CPoint3 *p2, int lwidth)
{
    register int d, x, y, ax, ay, sx, sy, dx, dy;
    int total;
    int x1, y1, x2, y2, r1, r2;
    double r, rdelta;

    x1 = p1->x; x2 = p2->x;
    y1 = p1->y; y2 = p2->y;
    r1 = RGB2gray2(p1->vcol.r, p1->vcol.g, p1->vcol.b);
    r2 = RGB2gray2(p2->vcol.r, p2->vcol.g, p2->vcol.b);

    if (lwidth > 1)
    {
	wide1DGline(buf, zbuf, zwidth, width, height, x1, y1, r1,
		     x2, y2, r2, lwidth);
	return;
    }

    dx = x2-x1;
    dy = y2-y1;
    total = ABS(dx)+ABS(dy);

    ax = ABS(dx)<<1;
    ay = ABS(dy)<<1;

    sx = SGN(dx);
    sy = SGN(dy);

    x = x1;
    y = y1;
    r = r1;
    rdelta = (r2-r1)/(double)total;
    if (ax>ay)
    {		/* x dominant */
	d = ay-(ax>>1);
	for (;;)
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		setPixel(buf, zwidth, width, height, x, y, r);
	    if (x==x2) break;
	    if (d>=0)
	    {
		y += sy;
		r += rdelta;
		d -= ax;
	    }
	    x += sx;
	    r += rdelta;
	    d += ay;
	}
    }
    else 
    {			/* y dominant */
	d = ax-(ay>>1);
	for (;;) 
	{
	    if ((x>=0) && (x<zwidth) && (y>=0) && (y<height))
		setPixel(buf, zwidth, width, height, x, y, r);
	    if (y==y2) break;
	    if (d>=0)
	    {
		x += sx;
		r += rdelta;
		d -= ay;
	    }
	    y += sy;
	    r += rdelta;
	    d += ax;
	}
    }
}

/* }}} */

/* {{{ polygon scan convers */

static void
Xmgr_DdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	      int height, int miny, int maxy, int *color, endPoint *mug)
{
    register int i, x2;                   /* dithering junk   */
    int y, x1;                           /* current line     */
    unsigned char *dith = dither[RGB2gray(color)];
    unsigned char pattern;
    unsigned char *ptr;

    for(y = miny; y <= maxy; y++)
    {
	if ((x1=mug[y].P1x) < 0)
	    x1 = 0;
	if ((x2=mug[y].P2x) >= zwidth)
	    x2 = zwidth-1;
	pattern = dith[y&0x07];
	ptr = buf+y*width;
	for (i=x1; i<=x2; i++)
	    ptr[i>>3] = (ptr[i>>3] & ~bits[i&0x07]) | (bits[i&0x07] & pattern);
    }
}

static void
Xmgr_DZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
	       int height, int miny, int maxy, int *color, endPoint *mug)
{
    register unsigned char *ptr;           /* pixel pointers   */
    register float *zptr;                 /* zbuff pointers   */
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    unsigned char *dith = dither[RGB2gray(color)];
    unsigned char pattern;

    for(y = miny; y <= maxy; y++)
    {
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(zwidth-1-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	}
	zdelta = (x2==x1) ? 0 : ((z2-z)/(x2-x1));
	pattern = dith[y&0x07];
	ptr = buf+y*width; zptr = zbuf+zwidth*y+x1;
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	{
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++)
		if ((z < *zptr) && (z>-1.0))
		{
		    ptr[i>>3] = (ptr[i>>3] & ~bits[i&0x07]) | 
			(bits[i&0x07] & pattern);
		    *zptr = z;
		}
	}
	else
	for (i=x1; i<=x2; i++, z+=zdelta, zptr++)
	    if (z < *zptr)
	    {
		ptr[i>>3] = (ptr[i>>3] & ~bits[i&0x07]) | 
		    (bits[i&0x07] & pattern);
		*zptr = z;
	    }
    }
}

static void
Xmgr_DGdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
		int height, int miny, int maxy, endPoint *mug)
{
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    int sr, er, dr, r, dx;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) * mug[y].P1x /
		(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    r = mug[y].P1r;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    dr = mug[y].P1r+(mug[y].P2r-mug[y].P1r)*(x2-mug[y].P1x) /
		(mug[y].P2x-mug[y].P1x)-r;
	}
	else
	{
	    x2 = mug[y].P2x;
	    dr = mug[y].P2r-r;
	}
	dx = x2-x1;
	er = (dr << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	dx <<= 1;
	for (i=x1; i<=x2; i++)
	{
	    setPixel(buf, zwidth, width, height, i, y, r);
	    if (dx) while (er>0)
	    {
		r += sr;
		er -= dx;
	    }
	    er += dr;
	}
    }
}

static void
Xmgr_DGZdoLines(unsigned char *buf, float *zbuf, int zwidth, int width, 
		int height, int miny, int maxy, endPoint *mug)
{
    register float *zptr;                 /* zbuff pointers   */
    register int i, x2;                  /* dithering junk   */
    int y, x1;                          /* current line     */
    double z, z2, zdelta;              /* z-buffer stuff   */
    int sr, er, dr, r, dx;

    for(y = miny; y <= maxy; y++)
    {
	if ((mug[y].P2x-mug[y].P1x)==0) continue;
	if (mug[y].P1x < 0)
	{
	    x1 = 0;
	    z = mug[y].P1z - (mug[y].P2z-mug[y].P1z) *
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	    r = mug[y].P1r - (mug[y].P2r-mug[y].P1r) * 
		mug[y].P1x/(mug[y].P2x-mug[y].P1x);
	}
	else
	{
	    x1 = mug[y].P1x;
	    z = mug[y].P1z;
	    r = mug[y].P1r;
	}
	if (mug[y].P2x >= zwidth)
	{
	    x2 = zwidth-1;
	    z2 = mug[y].P1z + (mug[y].P2z-mug[y].P1z) *
		(x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x);
	    dr = mug[y].P1r + (mug[y].P2r-mug[y].P1r) *
		(x2-mug[y].P1x)/(mug[y].P2x-mug[y].P1x)-r;
	}
	else
	{
	    x2 = mug[y].P2x;
	    z2 = mug[y].P2z;
	    dr = mug[y].P2r-r;
	}
	dx = x2-x1;
	zdelta = dx ? (z2-z)/dx : 0;
	er = (dr << 1) - dx;
	sr = SGN(dr); dr = abs(dr)<<1;
	dx <<= 1;
	zptr = zbuf+zwidth*y+x1;
	if (((z<-1) && (z2<-1)) || ((z>1) && (z2>1)))
	    continue;
	if ((z<-1) || (z2<-1))
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++)
	    {
		if ((z < *zptr) && (z>-1.0))
		{
		    setPixel(buf, zwidth, width, height, i, y, r);
		    *zptr = z;
		}
		if (dx) while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		er += dr;
	    }
	else
	    for (i=x1; i<=x2; i++, z+=zdelta, zptr++)
	    {
		if (z < *zptr)
		{
		    setPixel(buf, zwidth, width, height, i, y, r);
		    *zptr = z;
		}
		if (dx) while (er>0)
		{
		    r += sr;
		    er -= dx;
		}
		er += dr;
	    }
    }
}

void
Xmgr_1Dpoly(unsigned char *buf, float *zbuf, int zwidth, int width, int height,
	    CPoint3 *p, int n, int *color)
{
    Xmgr_polyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		  Xmgr_DdoLines);
}

void
Xmgr_1DZpoly(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_Zpolyscan(buf, zbuf, zwidth, width, height, p, n, color, mug,
		  Xmgr_DZdoLines);
}

void
Xmgr_1DGpoly(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_Graypolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		       Xmgr_DGdoLines);
}

void
Xmgr_1DGZpoly(unsigned char *buf, float *zbuf, int zwidth, 
	     int width, int height, CPoint3 *p, int n, int *color)
{
    Xmgr_GrayZpolyscan(buf, zbuf, zwidth, width, height, p, n, mug,
		       Xmgr_DGZdoLines);
}

/* }}} */

/* {{{ multi-line scan conversion */

void
Xmgr_1Dpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    setPixel(buf, zwidth, width, height, p0->x, p0->y, RGB2gray(color));
	return;
    }
    p1 = &p[1];
    Xmgr_1Dline(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0, 
	        p1->x, p1->y, 0, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_1Dline(buf, zbuf, zwidth, width, height, p0->x, p0->y, 0,
		    p1->x, p1->y, 0, lwidth, color);
    }
}


void
Xmgr_1DZpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y)*zwidth+(int)p0->x]) && (p0->z>-1.0))
		setPixel(buf, zwidth, width, height, p0->x, p0->y, RGB2gray(color));
	return;
    }
    p1 = &p[1];
    Xmgr_1DZline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, 
		 p1->x, p1->y, p1->z, lwidth, color);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
        Xmgr_1DZline(buf, zbuf, zwidth, width, height, p0->x, p0->y, p0->z, 
		     p1->x, p1->y, p1->z, lwidth, color);
    }
}

void
Xmgr_1DGpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    setPixel(buf, zwidth, width, height, p0->x, p0->y, RGB2gray(color));
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
		     Xmgr_1Dline, Xmgr_1DGline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
	Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			 Xmgr_1Dline, Xmgr_1DGline);
    }
}

void
Xmgr_1DGZpolyline(unsigned char *buf, float *zbuf, int zwidth, int width, int height, 
		 CPoint3 *p, int n, int lwidth, int *color)
{
    CPoint3 *p0, *p1;
    int i;
    p0 = &p[0];
    if (n == 1)
    {
	if ((p0->x > 0) && (p0->x < width) && (p0->y > 0) && (p0->y <height))
	    if ((p0->z < zbuf[(int)(p0->y)*zwidth+(int)p0->x]) && (p0->z>-1.0))
		setPixel(buf, zwidth, width, height, p0->x, p0->y, RGB2gray(color));
	return;
    }
    p1 = &p[1];
    Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
		     Xmgr_1DZline, Xmgr_1DGZline);
    for (i=2; i<n; i++)
    {
	p0 = p1;
        p1 = &p[i];
	Xmgr_gradWrapper(buf, zbuf, zwidth, width, height, p0, p1, lwidth,
			 Xmgr_1DZline, Xmgr_1DGZline);
    }
}

/* }}} */

/*
Local variables:
folded-file: t
*/

