/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

 /*
  * Geometry Routines
  * 
  * Geometry Supercomputer Project
  * 
  * ROUTINE DESCRIPTION:  Create and load a vert object from a file.
  * 
  */

#include "vectP.h"

/* why isn't VectCreate being called here ? */

Vect *
VectFLoad(file, fname)
    FILE *file;
    char *fname;
{
    register Vect *v;
    int	 binary = 0, dimn = 3;
    char *p;
    int i;
    static char badvert[] = "Reading VECT from \"%s\": bad %dth vertex (of %d)";

    if (file == NULL) return NULL;

    /* new format possibility: 4-d vectors */
    if (fnextc(file, 1) == '4')	{
	dimn = 4;	
	if (fexpectstr(file,"4VECT")) return NULL;
	}

    else if (fnextc(file, 0) != 'V' || fexpectstr(file, "VECT"))
	return NULL;

    if(fnextc(file, 1) == 'B') {
	if(fexpectstr(file, "BINARY"))
	    return NULL;
	binary = 1;
	if(fnextc(file, 1) == '\n')
	    (void) fgetc(file);		/* Toss \n */
    }

    v = OOGLNewE(Vect, "VectFLoad: Vect");

    GGeomInit(v, VectMethods(), VECTMAGIC, NULL);
    v->geomflags = (dimn == 4) ? VERT_4D : 0;
    v->vnvert = NULL;
    v->vncolor = NULL;
    v->p = NULL;
    v->c = NULL;

    if(fgetni(file, 1, &v->nvec, binary) <= 0 ||
       fgetni(file, 1, &v->nvert, binary) <= 0 ||
       fgetni(file, 1, &v->ncolor, binary) <= 0) {
	OOGLSyntax(file, "Reading VECT from \"%s\": can't read header counts", fname);
	goto bogus;
    }
    if(!vSane(v)) {
	OOGLSyntax(file,
	 "Reading VECT from \"%s\": inconsistent VECT header counts %d %d %d",
		fname, v->nvec, v->nvert, v->ncolor);
	goto bogus;
    }

    v->vnvert = OOGLNewNE(short, 2*v->nvec, "VECT nvec counts");
    v->p = OOGLNewNE(HPoint3, v->nvert, "VECT vertices");
    v->c = OOGLNewNE(ColorA, (v->ncolor>0) ? v->ncolor : 1, "VECT colors");

    v->vncolor = v->vnvert + v->nvec;

    if((i = fgetns(file, v->nvec, v->vnvert, binary)) < v->nvec) {
	OOGLSyntax(file,
	 "Reading VECT from \"%s\": bad vertex count in %d'th polyline (of %d)",
		fname, i, v->nvec);
	goto bogus; 
    }
    if((i = fgetns(file, v->nvec, v->vncolor, binary)) < v->nvec) {
	OOGLSyntax(file,
	 "Reading VECT from \"%s\": bad color count in %d'th polyline (of %d)",
		fname, i, v->nvec);
	goto bogus;
    }
    /* if the points are 3D points, we have to convert them to the native
	4D data structure */
    if (dimn == 3)	{
	register HPoint3 *p;

	for(i = v->nvert, p = v->p; --i >= 0; p++) {
	    if (fgetnf(file, 3, (float *)p, binary) < 3) {
		OOGLSyntax(file, badvert, fname, v->nvert - i, v->nvert);
		goto bogus;
	    }
	    p->w = 1;
	  }
	}
    else {
	i = fgetnf(file, 4*v->nvert, (float *)v->p, binary);
	if(i < 4*v->nvert) {
	    OOGLSyntax(file, badvert, fname, i/4, v->nvert);
	    goto bogus;
	}
      }
    if (v->ncolor > 0 &&
	(i = fgetnf(file, 4*v->ncolor, (float *)v->c, binary)) < 4*v->ncolor) {
	    OOGLSyntax(file, "Reading VECT from \"%s\": bad %dth color (of %d)",
		fname, i/4, v->ncolor);
	    goto bogus;
	}

    if(!VectSane(v)) {
	OOGLError(0, "Reading VECT from \"%s\": VECT polyline/color counts inconsistent with header", fname);
	goto bogus;
    }

    return v;

  bogus:
    GeomDelete((Geom *)v);
    return NULL;
}
