/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <ctype.h>
#include "quadP.h"

static int
getquads( file, pquad, off, binary, dimn )
    FILE	*file;
    register Quad *pquad;
    int		off;
    int		binary;
    int 	dimn;
{
    HPoint3 *p;
    Point3 *n;
    ColorA *c;
    register int k;

    p = &pquad->p[off][0];
    n = (pquad->flag & QUAD_N) ? &pquad->n[off][0] : NULL;
    c = (pquad->flag & QUAD_C) ? &pquad->c[off][0] : NULL;
    for(k = 4 * (pquad->maxquad - off); --k >= 0; ) {
	if(fgetnf(file, dimn, (float *)p, binary) < dimn)
	    break;
	/* set the w-coordinate if the points are 3 dim'nal */
	if (dimn == 3) p->w = 1.0;
	p++;
	if(n != NULL) {
	    if(fgetnf(file, 3, (float *)n, binary) < 3)
		return -1;
	    n++;
	}
	if(c != NULL) {
	    if(fgetnf(file, 4, (float *)c, binary) < 4)
		return -1;
	    c++;
	}
    }
    k++;
    if(k % 4 != 0)
	return(-1);
    return(pquad->maxquad - k/4);
}


Quad *
QuadFLoad( FILE *file, char *fname )
{
    Quad q;
    int binary = 0;
    register int ch;
    int ngot;
    int dimn = 3;

    q.flag = 0;
    q.p = NULL;
    q.n = NULL;
    q.c = NULL;

    (void) fnextc(file, 0);	/* Skip white space and comments */

    /* Parse [C][N][4]{QUAD|POLY}[ BINARY]\n */

    if((ch = getc(file)) == 'C') {
	q.flag = QUAD_C;
	ch = getc(file);
    }
    if(ch == 'N') {
	q.flag |= QUAD_N;
	ch = getc(file);
    }

    if(ch == '4') {
	q.flag |= QUAD_4D;
	dimn = 4;
	ch = getc(file);
    }

    if(!( (ch == 'P' && fexpectstr(file, "OLY") == 0) ||
	 (ch == 'Q' && fexpectstr(file, "UAD") == 0))) {
	return NULL;
    }

    if(fnextc(file, 1) == 'B' && fexpectstr(file, "BINARY") == 0) {
	binary = 1;
	if(fnextc(file, 1) != '\n') {	/* Expect \n after BINARY */
            GeomError(1,"QuadFLoad: bad QUAD file heading on %s", fname);
	    return NULL;
        }
	(void) getc(file);		/* Toss \n */
    }

    if(binary) {
	/*
	 * Binary case is easy.
	 * Read the q count, then the P, N and C arrays.
	 * Error if we get less than the q count promised.
	 */
	if(fgetni(file, 1, &q.maxquad, 1) <= 0) 
	    return NULL;
	if(q.maxquad <= 0 || q.maxquad > 10000000) {
	    OOGLError(0, "Reading QUAD BINARY from \"%s\": incredible q count 0x%x",
		fname, q.maxquad);
	    return NULL;
	}
	q.p = OOGLNewNE(QuadP, q.maxquad, "QuadFLoad: vertices");
	if(q.flag & QUAD_N) q.n = OOGLNewNE(QuadN, q.maxquad, "QuadFLoad: normals");
	if(q.flag & QUAD_C) q.c = OOGLNewNE(QuadC, q.maxquad, "QuadFLoad: colors");
	ngot = getquads(file, &q, 0, 1, dimn);
	if(ngot != q.maxquad) {
	    OOGLFree(q.p);
	    if(q.n) OOGLFree(q.n);
	    if(q.c) OOGLFree(q.c);
	    goto fail;
	}
    } else {
	/*
	 * ASCII case is harder.
	 * Guess how many quads we need, and realloc() if we need more.
	 */
#define  TRYQUADS	1000

	QuadP qp[TRYQUADS];
	QuadN qn[TRYQUADS];
	QuadC qc[TRYQUADS];
	vvec vp, vn, vc;

	VVINIT(vp, QuadP, TRYQUADS);
	vvuse(&vp, qp, TRYQUADS);
	if(q.flag & QUAD_N) {
	    VVINIT(vn, QuadN, TRYQUADS);
	    vvuse(&vn, qn, TRYQUADS);
	}
	if(q.flag & QUAD_C) {
	    VVINIT(vc, QuadC, TRYQUADS);
	    vvuse(&vc, qc, TRYQUADS);
	}
	q.maxquad = TRYQUADS;
	ngot = 0;
	for(;;) {
	    q.p = VVEC(vp, QuadP);
	    if(q.flag & QUAD_N) q.n = VVEC(vn, QuadN);
	    if(q.flag & QUAD_C) q.c = VVEC(vc, QuadC);

	    ngot = getquads(file, &q, ngot, 0, dimn);

	    VVCOUNT(vp) = VVCOUNT(vn) = VVCOUNT(vc) = ngot;

	    if(ngot < q.maxquad)
		break;

	    q.maxquad *= 2;
	    vvneeds(&vp, q.maxquad);
	    if(q.flag & QUAD_N) vvneeds(&vn, q.maxquad);
	    if(q.flag & QUAD_C) vvneeds(&vc, q.maxquad);
	}
	if(ngot <= 0) {
	    vvfree(&vp);
	    vvfree(&vn);
	    vvfree(&vc);
	    goto fail;
	}

	/* Release unneeded storage */

	q.maxquad = ngot;
	vvtrim(&vp);
	q.p = VVEC(vp, QuadP);
	if (q.flag & QUAD_N) {
	    vvtrim(&vn);
	    q.n = VVEC(vn, QuadN);
	}
	if (q.flag & QUAD_C) {
	    vvtrim(&vc);
	    q.c = VVEC(vc, QuadC);
	}
    }

    return (Quad *) GeomCCreate(NULL, QuadMethods(),
	CR_4D, (dimn == 4) ? 1 : 0,
	CR_NOCOPY, CR_FLAG, q.flag, CR_NELEM, q.maxquad,
	CR_POINT4, q.p, CR_NORMAL, q.n, CR_COLOR, q.c, 0);

  fail:
    OOGLSyntax(file, "Error reading QUADs in \"%s\"", fname);
    return NULL;
}
