/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

 /*
  * Load a PolyList object from an "off"-format file.
  * 
  */

#include <string.h>
#include "npolylistP.h"

#define	SIZEOF_CMAP	256
static ColorA	*colormap;

static int
LoadCmap(char *file)
{
    FILE *fp;
    colormap = OOGLNewNE(ColorA, 256, "PolyList colormap");
    if((file = findfile(NULL, file)) != NULL &&
			   (fp = fopen(file,"r")) != NULL) {
	fgetnf(fp, SIZEOF_CMAP*4, (float *)colormap, 0);
	fclose(fp);
    }
}

NPolyList *
NPolyListFLoad(FILE *file, char *fname)
{
    register NPolyList *pl;
    int edges;
    register int i;
    float *v;
    int binary = 0;
    int headerseen = 0;
    int flags = 0;
    int makenorm = 0;
    int dimn = 3, pdim;
    static ColorA black = { 0,0,0,0 };
    vvec vi;

    if (file == NULL)
	return NULL;

    (void) fnextc(file, 0);
    i = fgetc(file);
    if(i == 'C') {
	flags = PL_HASVCOL;	/* Per-vertex colors */
	i = fgetc(file);
    }
    if(i == '4') {
	dimn = 4;
	i = fgetc(file);
    }
    if(i == 'S') {		/* "smooth surface": we compute vtx normals */
	makenorm = 1;
	i = fgetc(file);
    }
    if(i == 'n' && !fexpectstr(file, "OFF")) {	/* Skip blanks & comments */
	headerseen = 1;
	if(fgetni(file, 1, &pdim, binary) == 0) {
	    OOGLSyntax(file, "nOFF %s: expected dimension", fname);
	    return NULL;
	}
	if(fnextc(file, 1) == 'B' && fexpectstr(file, "BINARY") == 0) {
	    binary = 1;
	    if(fnextc(file, 1) != '\n')	/* Expect \n after BINARY */
		return NULL;
	    (void) getc(file);		/* Toss \n */
	}
    } else {
	return NULL;
    }


    pl = OOGLNewE(NPolyList, "NPolyListFLoad NPolyList");
    GGeomInit(pl, NPolyListMethods(), NPLMAGIC, NULL);
    pl->p = NULL;
    pl->v = NULL;
    pl->vi = NULL;
    pl->vcol = NULL;

    pl->flags = flags;
    /* this is very ugly, but this whole routine is, too.  
       e.g., Why isn't GeomCreate being called here to guarantee that the
       returned structure is valid? */
    pl->geomflags = (dimn == 4) ? VERT_4D : 0;

    pl->pdim = pdim + (dimn == 4 ? 0 : 1);


    if(fgetni(file, 1, &pl->n_verts, binary) <= 0 ||
       fgetni(file, 1, &pl->n_polys, binary) <= 0 ||
       fgetni(file, 1, &edges, binary) <= 0) {
		if(headerseen)
		    OOGLSyntax(file, "PolyList %s: Bad vertex/face/edge counts", fname);
		goto bogus;
    }
    VVINIT(vi, int, 5*pl->n_polys);

    pl->v = OOGLNewNE(float, pl->n_verts*pl->pdim, "NPolyListFLoad vertices");
    if(pl->flags & PL_HASVCOL)
	pl->vcol = OOGLNewNE(ColorA, pl->n_verts, "NPolyListFLoad vertex colors");

    for(v = pl->v, i = 0; i < pl->n_verts; v += pl->pdim, i++) {
	if(fgetnf(file, pdim, v, binary) < pdim ||
	   flags & PL_HASVCOL && fgetnf(file, 4, (float *)&pl->vcol[i], binary) < 4) {
		OOGLSyntax(file, "nOFF %s: Bad vertex %d (of %d)", fname, i, pl->n_verts);
		goto bogus;
	}
	if (dimn == 3) v[pl->pdim-1] = 1.0;
    }

    pl->p = OOGLNewNE(NPoly, pl->n_polys, "NPolyListFLoad polygons");
    for(i = 0; i < pl->n_polys; ) {
	register NPoly *p;
	register int k,index,m;
	float rgba[4];

	p = &pl->p[i];
	if(fgetni(file, 1, &p->n_vertices, binary) <= 0 || p->n_vertices <= 0) {
	   OOGLSyntax(file, "PolyList %s: bad %d'th polygon (of %d)",
		fname, i, pl->n_polys);
	   goto bogus_face;
	}
	p->vi0 = VVCOUNT(vi);
	VVCOUNT(vi) += p->n_vertices;
	vvneeds(&vi, VVCOUNT(vi));
	i++;

	for(k = 0; k < p->n_vertices; k++) {
	    int index;

	    if(fgetni(file, 1, &index, binary) <= 0 ||
	       index < 0 || index >= pl->n_verts) {
		    OOGLSyntax(file, "PolyList: %s: bad index %d on %d'th polygon (of %d)", 
			fname, index, i, p->n_vertices);
		    goto bogus_face;
	    }
	    VVEC(vi, int)[k + p->vi0] = index;
	}

	/* Pick up the color, if any.
	 * In ASCII format, take whatever's left before end-of-line
	 */
	p->pcol = black;
	if(binary) {
	    int ncol;

	    if(fgetni(file, 1, &ncol, 1) <= 0
	       || fgetnf(file, ncol, (float *)&p->pcol, 1) < ncol)
		goto bogus_face_color;
	    k = ncol;
	} else {
	    for(k = 0; k < 4 && fnextc(file, 1) != '\n'; k++) {
		if(fgetnf(file, 1, ((float *)(&p->pcol))+k, 0) < 1)
		    goto bogus_face_color;
	    }
	}

	if((flags & PL_HASVCOL) == 0) {
	    if(k > 0)
		pl->flags |= PL_HASPCOL;

	    if(k != 1 && (p->pcol.r>1||p->pcol.g>1||p->pcol.b>1||p->pcol.a>1)) {
		p->pcol.r /= 255, p->pcol.g /= 255,
		p->pcol.b /= 255, p->pcol.a /= 255;
	    }

	    switch(k) {
	    case 0:
		p->pcol.r = 170/255.0;		/* Gray */
		p->pcol.g = p->pcol.r;
	    case 2:
		p->pcol.b = p->pcol.g;
	    case 3:
		p->pcol.a = 170/255.0;		/* Semitransparent */
		break;
	    case 1:				/* use colormap */
		if ( colormap == NULL )
			LoadCmap("cmap.fmap");
		index = p->pcol.r;
		if((unsigned)index >= SIZEOF_CMAP) index = 0;
		p->pcol = colormap[index];
	    }				/* case 4, all components supplied */
	}
    }

	/* Hand the list of vertex-indices on to the new object */
    vvtrim(&vi);
    pl->nvi = VVCOUNT(vi);
    pl->vi = VVEC(vi, int);
    return pl;

  
  bogus_face_color:
    OOGLSyntax(file, "PolyList %s: bad face color on face %d (of %d)",
	fname, i, pl->n_polys);
  bogus_face:
    pl->n_polys = i;
  bogus:
    GeomDelete((Geom *)pl);
    return NULL;
}
