#include "mg.h"
#include "../common/drawer.h"
#include "../common/ui.h"
#include "../common/comm.h"
#include "../common/space.h"
#include "../common/lights.h"
#include "../common/event.h"
#include "../common/motion.h"
#include "../common/worldio.h"
#include "../common/version.h"
#include "../common/lang.h"

/* Geomview Icon */
#include "bitmaps/icon"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
/* #include <sys/dir.h> Do we need this? - slevy 94.03.09 */
#include <sys/signal.h>

#include "mibload.h"
#include "gvui.h"

/******************************* Index defines *******************************/

#define BLACKINDEX      0
#define WHITEINDEX      7

#define FACEINDEX       250
#define EDGEINDEX       251
#define NORMALINDEX     252
#define BBOXINDEX       253
#define BACKINDEX       254
#define LIGHTINDEX      255

/****************************** Table of Panels ******************************/

static struct panel {
  char *name;
  char key[2];
  void (*show)();
  void (*hide)();
  void (*move)(int, int, int);
  int  shown;
  int  position;
  int  x, y;
} panels[] = {
  { "\0", "  ", NULL, NULL, 0 },
  { "geomview", "Pm", ui_show_mainpanel, ui_hide_mainpanel,
		ui_move_mainpanel, 1, 0, 0, 0 },
  { "Tools", "Pt", ui_show_toolpanel, ui_hide_toolpanel,
		ui_move_toolpanel, 1, 0, 0, 0 },
  { "Appearance", "Pa", ui_show_appearancepanel, ui_hide_appearancepanel,
		ui_move_appearancepanel, 0, 0, 0, 0 },
  { "Cameras", "Pc", ui_show_cameraspanel, ui_hide_cameraspanel,
		ui_move_cameraspanel, 0, 0, 0, 0 },
  { "Lighting", "Pl", ui_show_lightspanel, ui_hide_lightspanel,
		ui_move_lightspanel, 0, 0, 0, 0 },
  { "Commands", "PC", ui_show_commandspanel, ui_hide_commandspanel,
		ui_move_commandspanel, 0, 0, 0, 0 },
  { "Materials", "PM", ui_show_materialpanel, ui_hide_materialpanel,
		ui_move_materialpanel, 0, 0, 0, 0 },
  { "Credits", "", ui_show_creditspanel, ui_hide_creditspanel,
		ui_move_creditspanel, 0, 0, 0, 0 },
  { "Save", ">", ui_show_savepanel, ui_hide_savepanel,
		ui_move_savepanel, 0, 0, 0, 0 },
  { "Files", "Pf", ui_show_loadpanel, ui_hide_loadpanel,
		ui_move_loadpanel, 0, 0, 0, 0 },
  { "Load", "<", ui_show_loadpanel, ui_hide_loadpanel,
		ui_move_loadpanel, 0, 0, 0, 0 },
  { "Color", "", ui_show_colorpanel, ui_hide_colorpanel,
		ui_move_colorpanel, 0, 0, 0, 0 },
};

/************************** Global Widget Variables **************************/

Widget	TopLevel,
	MainWindow,
        MainMenu,
        MainForm;

Atom	AProtocol;
Arg	args[20];
int     n;
XtAppContext App;
extern	Display	*dpy;
extern  GC	mib_gc;
extern int real_id(int);
Pixmap geomicon;

/******************************* Global Variables ****************************/

int	 gwidth=381;
int      gheight=234;
int      YScrnSize, XScrnSize;
UIState  uistate;
int      gvbitdepth;
Visual  *gvvisual;
Colormap gvcolormap;
int	 gvpriv=0;

/*****************************************************************************/

static  int panelsinit = 0;
static  int oldstderr;

void ui_select(int id);
void ui_choose_depth();

/*****************************************************************************/
	
void load_interface(argc, argv)
unsigned int argc;
char **argv;
{
  String fallbacks[] = {
    "gv*fontList:        -adobe-helvetica-medium-r-normal--14-*-*-*-p-*-iso8859-1",
    "*BorderWidth:   0",

#ifdef dark_ui
    "*Foreground:    gray20",
    "*Background:    gray70",
/*
    "*Foreground:     #000000",
    "*Background:     #848484",
*/
#endif

#ifdef light_ui
    "*Foreground:     #000000000000",
    "*Background:     #afafafafafaf",
#endif

    "*XmToggleButton.selectColor:    yellow",
    "*XmToggleButton.indicatorSize:    16",
    "*XmToggleButtonGadget.selectColor:    yellow",
    "*XmToggleButtonGadget.indicatorSize:    16",
    "*fontList:        -adobe-helvetica-medium-r-normal--14-*-*-*-p-*-iso8859-1",
    "gv*Load*fontList:    -adobe-helvetica-medium-r-normal--12-120-75-75-p-67-iso8859-1",
    "gv*Save*fontList:    -adobe-helvetica-medium-r-normal--12-120-75-75-p-67-iso8859-1",
    "Commands*fontList:    -adobe-helvetica-medium-r-normal--12-120-75-75-p-67-iso8859-1",
    "Tools*fontList:    -adobe-helvetica-medium-r-normal--12-120-75-75-p-67-iso8859-1",
    NULL
  };

  int argcblah = 0;
  char *argvblah[1] = { "Geomview" };


/* Code for TopLevel */
/*****************************************************************************/
	
  n = 0;

  TopLevel = XtAppInitialize(&App, "gv", NULL, 0, &argcblah /*&argc*/, argvblah
			/*argv*/, fallbacks, args, n);

  dpy = XtDisplay(TopLevel);
  YScrnSize = XtScreen(TopLevel)->height;
  XScrnSize = XtScreen(TopLevel)->width;

  ui_choose_depth();

  geomicon = XCreatePixmapFromBitmapData(dpy, DefaultRootWindow(dpy),
		oogl_bits, oogl_width, oogl_height,
		WhitePixel(dpy, DefaultScreen(dpy)),
		BlackPixel(dpy, DefaultScreen(dpy)),
		DefaultDepth(dpy, DefaultScreen(dpy)));


}

/*****************************************************************************/

void ui_highlight(int newid)
{
  int newtype = TYPEOF(newid);
  DGeom *dg;
  DView *v;
  static Color black = {0.0, 0.0, 0.0};
  Color c;
  extern int mg_ctxselect(mgcontext *);


  if ( newtype == T_GEOM &&
      (dg = (DGeom *) drawer_get_object(newid)) && dg->bboxap ) {
    c = uistate.pickedbboxcolor;
    gv_bbox_color( GEOMID(uistate.targetgeom), black.r, black.g, black.b );
    gv_bbox_color( newid, c.r, c.g, c.b );
  } else if (newtype == T_CAM) { /* pop cam window in case it's obscured */
    v = (DView *) drawer_get_object(newid);
    if (v) {
      mg_ctxselect(v->mgctx); /* winset happens in here */
      /* winpop */
    }
  }

}

/*****************************************************************************/

void ui_action(int val)
{
}

/*****************************************************************************/

void ui_mousefocus(int index)
{
  uistate.mousefocus = index;
  if (uistate.targetcam == INDEXOF(FOCUSID)) {
    ui_select(CAMID(uistate.mousefocus));
  }

}

/*****************************************************************************/

void ui_objectchange()
{
  ui_adjustpicklist(-1);
  ui_adjustpicklist(uistate.targetid);
}

/*****************************************************************************/

void ui_maybe_refresh(int id)
{
   if ( id==WORLDGEOM || id==GEOMID(uistate.targetgeom) || id==ALLGEOMS
       || id==TARGETGEOMID) {
    ui_select(GEOMID(uistate.targetgeom));
  } else if( id==ALLCAMS || id==FOCUSID || id==TARGETCAMID
        || id==CAMID(uistate.targetcam) || id==CAMID(uistate.mousefocus) ) {
    ui_select(CAMID(uistate.targetcam));
  } else {
    ui_select(NOID);
  }
}

/*****************************************************************************/

void ui_select(int id)
{
  DObject *obj;

  if ((obj = drawer_get_object(uistate.targetid)))
  {
    ui_tool_targetset(drawer_id2name(obj->id));
    ui_tool_centerset(drawer_id2name(uistate.centerid));

    if (ISCAM(id))
      ui_target_cameraspanel(id);
    if (ISGEOM(id))
    {
      ui_target_appearancepanel(id);
      ui_target_materialpanel(id);
    }

    ui_refresh_mainpanel(id);
  }
}

/*****************************************************************************/

static void ui_delete_camwin(Window parent)
{
  XtDestroyWidget(XtWindowToWidget(dpy, parent));
}

/*****************************************************************************/

void ui_windowWillOpen(DView *dv)
{
  Widget shell, cam;
  int    depth;
  Visual *visual;

  shell = ui_create_camera(MainWindow, &cam, dv, &depth, &visual);

  oldstderr = dup(fileno(stderr));
  freopen("/dev/null", "w", stderr);

  XtRealizeWidget(shell);

  freopen("/dev/null", "w", stderr);
  dup2(oldstderr, fileno(stderr));
  close(oldstderr);

#ifdef MGX11

  mgctxset(MG_X11DISPLAY, dpy,
           MG_X11COLORMAP, gvcolormap,
	   MG_BITDEPTH, gvbitdepth,
           MG_X11VISUAL, gvvisual,
	   MG_X11WINID, XtWindow(cam),
	   MG_X11PARENT, XtWindow(shell),
	   MG_X11DELETE, ui_delete_camwin,
	   MG_X11SIZELOCK, 1,
	   MG_END);

  oldstderr = dup(fileno(stderr));
  freopen("/dev/null", "w", stderr);

  XtAddCallback(cam, XmNexposeCallback, (XtCallbackProc) cam_expose,
			(XtPointer) dv);
  XtAddCallback(cam, XmNresizeCallback, (XtCallbackProc) cam_expose,
			(XtPointer) dv);

#endif

#ifdef MGGL

  oldstderr = dup(fileno(stderr));
  freopen("/dev/null", "w", stderr);

  XtAddCallback(cam, GlxNginitCallback, (XtCallbackProc) cam_initgl,
                        (XtPointer) NULL);
  XtAddCallback(cam, GlxNexposeCallback, (XtCallbackProc) cam_exposegl,
                        (XtPointer) dv);

  XtManageChild(cam);

  mgctxset(MG_GLXDISPLAY, XtDisplay(cam),
           MG_GLWINID, XtWindow(cam),
           MG_END);

#endif

  XtAddCallback(cam, XmNinputCallback, (XtCallbackProc) cam_input,
			(XtPointer) dv);
  XtAddEventHandler(cam, Button1MotionMask |
			 Button2MotionMask |
			 Button3MotionMask, False, (XtEventHandler) cam_mouse,
			(XtPointer) dv);
  XtAddEventHandler(cam, EnterWindowMask, False, (XtEventHandler) cam_mousecross,
			(XtPointer) dv);

  freopen("/dev/null", "w", stderr);
  dup2(oldstderr, fileno(stderr));
  close(oldstderr);
}

/*****************************************************************************/

void ui_windowDidOpen(DView *dv)
{
}

/*****************************************************************************/

void ui_showpanel(int index, int show)
{

  int i, invis = 1;
  
  if (( index > P_MAX ) || ( index < 0 ))	/* protect us from showing */
    return;					/* non-existent panels */

  if (panels[index].show == NULL)		/* or panels which have no */
    return;					/* show/hide methods */

  switch (show)
  {
    case -1:
      if (panels[index].shown)
      {
        panels[index].shown = 0;
        (* panels[index].hide)();
      }
      else
      {
        panels[index].shown = 1;
        (* panels[index].show)();
      }
      break;
    case 0:
        panels[index].shown = 0;
        (* panels[index].hide)();
      break;
    case 1:
        panels[index].shown = 1;
        (* panels[index].show)();
      break;
  }
}

/*****************************************************************************/

void ui_keyboard(int ch)
{
}

/*****************************************************************************/

void ui_event_mode(char *mode)
{
  int index = ui_mode_index(mode);
  ui_select_tool(index);
}

/*****************************************************************************/

LDEFINE(ui_target, LVOID,
       "(ui-target      ID [yes|no])\n\
        Set the target of user actions (the selected line of the\n\
        target object browser) to ID.  The second argument specifies\n\
        whether to make ID the current object regardless of its type.\n\
        If \"no\", then ID becomes the current object of its type\n\
        (geom or camera).  The default is \"yes\".  This command may\n\
        result in a change of motion modes based on target choice.")
{
  DObject *obj;
  int id, newtype;
  int immediate=YES_KEYWORD;

  LDECLARE(("ui-target", LBEGIN,
            LID, &id,
            LOPTIONAL,
            LKEYWORD, &immediate,
            LEND));
  immediate = boolval("ui-target", immediate);
  newtype = TYPEOF(id);

  if (id == uistate.targetid ) return Lt;
  if ( !(obj = drawer_get_object(id))
      || (id == TARGETID) || (id == CENTERID)
      || (id == SELF) || (id == UNIVERSE) || (id == PRIMITIVE)) {
    return Lt;
  }
  if (immediate) {
    /* ui_highlight must be called before set_ui_target_id */
    /* freeze the browser so old target name doesn't flash
       during highlight. */

    ui_highlight(id);
    set_ui_target_id( id );
    ui_adjustpicklist(id);
    if (ISGEOM(id))
    {
      uistate.targetcam = INDEXOF(FOCUSID);
      ui_select(real_id(FOCUSID));
    }
    ui_select( id );
  } else {
    /* immediate == NOIMMEDIATE: for cases like deleting geometry
     * with the command language when current target is a camera.
     * update targettype but don't change browser.
     */
    switch (newtype) {
    case T_GEOM: uistate.targetgeom = INDEXOF(id); break;
    case T_CAM: uistate.targetcam = INDEXOF(id); break;
    }
  }
  return Lt;
}

LDEFINE(ui_center, LVOID,
       "(ui-center      ID)\n\
        Set the center for user interface (i.e. mouse) controlled\n\
        motions to object ID.")
{
  int id;
  LDECLARE(("ui-center", LBEGIN,
            LID, &id,
            LEND));

  if ( (id == NOID) || (id == CENTERID) ) {
    id = uistate.centerid;
  } else {
    set_ui_center(id);
  }
  ui_tool_centerset(drawer_id2name(id));
  return Lt;
}

/*****************************************************************************/

void ui_pickcolor(int val)
{
  DView      *dv;
  Color      *old;
  char       *name;
  Appearance *ap;
  DGeom      *dg;
  int         index;


  ap = drawer_get_ap(GEOMID(uistate.targetgeom));

  switch (val)
  {
    case DRAWER_BACKCOLOR:
	if ((dv = (DView *)drawer_get_object(real_id(CAMID(uistate.targetcam)))))
	{
	  ui_config_colorpanel("Background", &(dv->backcolor), BACKINDEX, val,
		real_id(CAMID(uistate.targetcam)));
	  ui_showpanel(P_COLOR, 1);
	}
	return;

    case DRAWER_DIFFUSE:
	name = "Faces"; index = FACEINDEX;
	old = &ap->mat->diffuse;
	break;

    case DRAWER_EDGECOLOR:
	name = "Edges"; index = EDGEINDEX;
	old = &ap->mat->edgecolor;
	break;

    case DRAWER_NORMALCOLOR:
	name = "Normals"; index = NORMALINDEX;
	old = &ap->mat->normalcolor;
	break;

    case DRAWER_BBOXCOLOR:
	name = "BBoxes"; index = BBOXINDEX;
	dg = (DGeom*)drawer_get_object(GEOMID(uistate.targetgeom));
	old = &dg->bboxap->mat->edgecolor;
	break;

    default:
      break;
  }

  ui_config_colorpanel(name, old, index, val, GEOMID(uistate.targetgeom));
  ui_showpanel(P_COLOR, 1);
}

/*****************************************************************************/

void ui_curson(int on)
{
}

/*****************************************************************************/

int ui_name2panel(char *name)
{
  extern int      strcasecmp(const char *, const char *);
  extern int      strncasecmp(const char *, const char *, size_t);
  int i;

  if (strcasecmp(name, "main") == 0) return P_MAIN;
  for (i=P_MAX; --i > 0 && strcasecmp(name, panels[i].name) != 0;);
  if (i == 0)
    for(i=P_MAX; --i > 0 && strncasecmp(name, panels[i].name, 4) != 0; ) ;

  return i;
}

/*****************************************************************************/

void ui_add_mode(char *name, PFI proc, int type)
{

  ui_install_mode(name, proc, type);
}

/*****************************************************************************/

void ui_remove_mode(char *name)
{
  int current;

  current = ui_mode_index(name);
  ui_uninstall_mode(name);

  if (uistate.mode_current == current)
    gv_event_mode(uistate.modenames[0]);
}

/*****************************************************************************/

void ui_init()
{
  oldstderr = dup(fileno(stderr));
  freopen("/dev/null", "w", stderr);

  ui_load_mainpanel();
  ui_load_appearancepanel();
  ui_load_materialpanel();
  ui_load_cameraspanel();
  ui_load_lightspanel();

  freopen("/dev/null", "w", stderr);
  dup2(oldstderr, fileno(stderr));
  close(oldstderr);

  cui_init();
  clight_init();

  gv_event_mode(OBJROTATE);
  uistate.cursor_on = 1;
}

/*****************************************************************************/

LDEFINE(ui_panel, LVOID,
            "(ui-panel       PANELNAME  {on|off} [ WINDOW ] )\n\
        Do or don't display the given user-interface panel.\n\
        Case is ignored in panel names.  Current PANELNAMEs are:\n\
                geomview        main panel\n\
                tools           motion controls\n\
                appearance      appearance controls\n\
                cameras         camera controls\n\
                lighting        lighting controls\n\
                obscure         obscure controls\n\
                materials       material properties controls\n\
                command         command entry box\n\
                credits         geomview credits\n\
        By default, the \"geomview\" and \"tools\" panels appear when\n\
        geomview starts.  If the optional Window is supplied, a\n\
        \"position\" clause (e.g. (ui-panel obscure on { position xmin\n\
        xmax ymin ymax }) sets the panel's default position.  (Only\n\
        xmin and ymin values are actually used.)  A present but empty\n\
        Window, e.g.  \"(ui-panel obscure on {})\" causes interactive\n\
        positioning.")
{
  char *panelname;
  int index, on;
  struct panel *p;
  WindowStruct *ws=NULL;
  WnPosition wp;

  LDECLARE(("ui-panel", LBEGIN,
            LSTRING, &panelname,
            LKEYWORD, &on,
            LOPTIONAL,
            LWINDOW, &ws,
            LEND));

  if((index = ui_name2panel(panelname)) == 0) {
    if (!strcmp(panelname, "Obscure"))
      return Lnil;
    fprintf(stderr, "ui-panel: expected name of a panel, got \"%s\"\n",
            panelname);
    return Lnil;
  }

  if (ws)
  {
    p = &(panels[index]);
    if (WnGet(ws->wn, WN_PREFPOS, &wp) > 0)
    {
      p->x = wp.xmin;
      p->y = wp.ymin;
    }
    if (panels[index].shown && panelsinit)
    {
      p->position = 0;
      (* p->move) (p->position, p->x, p->y);
    }
    else
    {
      p->position = 1; /* position window when it shows up later */
      (* p->move) (p->position, p->x, p->y);
    }
  } 

  if (!panelsinit)
    panels[index].shown = boolval("ui-panel", on);
  else
    ui_showpanel( index, boolval("ui-panel", on));

  return Lt;
}

/*****************************************************************************/

/* Dummy -- we don't bother to freeze the UI */

LDEFINE(ui_freeze, LVOID,
        "(ui-freeze [on|off])\n\
        Toggle updating user interface panels. Off by default."
)
{
  int kw = -1;
  LDECLARE(("ui-freeze", LBEGIN,
        LOPTIONAL,
        LKEYWORD, &kw,
        LEND));
  return Lnil;
}

/*****************************************************************************/

int ui_panelshown(int index)
{

  return panels[index].shown;
}

/*****************************************************************************/

void ui_final_init()
{
  int i;

  oldstderr = dup(fileno(stderr));
  freopen("/dev/null", "w", stderr);

  ui_load_toolpanel();  /* we load toolpanel here because it relies on mgx11
			   for colors which is activated AFTER ui_init() */
  ui_load_colorpanel(); /* ditto for the color panel */

  freopen("/dev/null", "w", stderr);
  dup2(oldstderr, fileno(stderr));
  close(oldstderr);

  panelsinit = 1;
  ui_select(CAMID(uistate.targetcam));
  ui_select(GEOMID(uistate.targetgeom));
  lights_changed_check();
  for(i = 1; i < P_MAX; i++)
    if (panels[i].shown)
      ui_showpanel(i, 1);
  ui_objectchange();

}

/*****************************************************************************/

void ui_update()
{
  int i;
  emodule *em;

  if(uistate.emod_check) {
    for(i=0, em=VVEC(uistate.emod,emodule); i<VVCOUNT(uistate.emod); ) {
        if(VVEC(uistate.emod,emodule)[i].pid < 0 &&
            (em->link == NULL || PoolInputFile(em->link) == NULL)) {
                emodule_reap(em);
        } else {
            i++, em++;
        }
    }
    uistate.emod_check = 0;
  }
  if(uistate.emod_changed) {
    ui_adjustemodulelist(uistate.emod_changed);
    uistate.emod_changed = 0;
  }
}

/*****************************************************************************/

LDEFINE(ui_emodule_define, LVOID,
"(emodule-define  NAME  SHELL-COMMAND ...)\n\
        Define an external module called NAME, which then appears in the\n\
        external-module browser.  The SHELL-COMMAND string\n\
        is a UNIX shell command which invokes the module.\n\
        See emodule-run for discussion of external modules.")
{
  emodule *em;

  char *title, *cmd;
  LDECLARE(("emodule-define", LBEGIN,
            LSTRING, &title,
            LSTRING, &cmd,
            LEND));

  em = ui_emodule_install(0, strdup(title), (PFI)emodule_run);
  em->text = strdup(cmd);
  ui_adjustemodulelist(1);
  return Lt;
}

/*****************************************************************************/

static void ename_fix(c,e,n)
char *c,*e;
int n;
{
  char num[80];
  int ni=0, ei=0;
  if (e[0] != '[') goto copy;
  ei = 1;
  num[0] = '\0';
  while (ei<n && e[ei++] != ']') num[ni++] = e[ei-1];
  if (ei == n) goto copy;
  num[ni] = '\0';
  strncpy(c,e+ei,n);
  c[strlen(c)-1] = '0';
  strcat(c,"[");
  strcat(c,num);
  strcat(c,"]");
  return;
 copy:
  strncpy(c,e,n);
  return;
}

static int emod_compare(e1, e2)
emodule *e1, *e2;
{
  char c1[80], c2[80];

  ename_fix(c1,e1->name,80);
  ename_fix(c2,e2->name,80);

  return strcmp(c1,c2);
}

LDEFINE(emodule_sort, LVOID,
"(emodule-sort)\n\
        Sorts the modules in the application browser alphabetically.")
{
  emodule *em;
  size_t nel, size;
  LDECLARE(("emodule-sort", LBEGIN,
            LEND));

  em = VVEC(uistate.emod, emodule);
  nel = VVCOUNT(uistate.emod);
  size = sizeof(emodule);

  qsort(em, nel, size, emod_compare);
  ui_adjustemodulelist(1);
  return Lt;
}

/*****************************************************************************/

LDEFINE(ui_emodule_start, LVOID,
"(emodule-start  NAME)\n\
        Starts the external module NAME, defined by emodule-define.\n\
        Equivalent to clicking on the corresponding module-browser entry.")
{
  emodule *em;
  int i;
  char *name;
  LDECLARE(("emodule-start", LBEGIN,
            LSTRING, &name,
            LEND));
  i = ui_emodule_index(name, &em);
  if(i < 0) return Lnil;
  (*em->func)(em);
  return Lt;
}

/*****************************************************************************/

void gv_cull_backface(int a1)
{
  LObject *val = LEvalFunc("cull-backface",
                           LKEYWORD, a1,
                           LEND);
  LFree(val);
}


/*****************************************************************************/

void ui_build_browser(Widget a_list, int count, char *names[])
{
  XmString str;
  int      i;

  XmListDeleteAllItems(a_list);
  for (i = 0; i < count; i++)
  {
    str = XmStringCreateSimple(names[i]);
    XmListAddItemUnselected(a_list, str, 0);
    XmStringFree(str);
  }
}

/*****************************************************************************/

void ui_set_ftext(Widget w, float val)
{
  char buf[80], *str;

  sprintf(buf,"%5g",val);
  str = XmTextFieldGetString(w);
  if (strcmp(str, buf))
    XmTextFieldSetString(w, buf);
  XtFree(str);
}

/*****************************************************************************/

void ui_set_itext(Widget w, int val)
{
  char buf[80], *str;

  sprintf(buf,"%3d",val);
  str = XmTextFieldGetString(w);
  if (strcmp(str, buf))
    XmTextFieldSetString(w, buf);
  XtFree(str);
}

/*****************************************************************************/

void ui_choose_depth()
{

  static XVisualInfo vTemplate;
  static XVisualInfo *visualList;
  unsigned int nplanes = 0;
  unsigned long planemasks[1];
  unsigned long testcols[30];
  int         visualsMatched;


  vTemplate.screen = DefaultScreen(dpy);
  vTemplate.class = TrueColor;
  gvbitdepth = vTemplate.depth = 24;
  visualList = XGetVisualInfo(dpy, VisualScreenMask | VisualClassMask |
                              VisualDepthMask, &vTemplate, &visualsMatched);
  if (visualsMatched == 0)
  {
      vTemplate.class = PseudoColor;
      gvbitdepth = vTemplate.depth = 8;
      visualList =
          XGetVisualInfo(dpy, VisualScreenMask | VisualClassMask |
                         VisualDepthMask, &vTemplate, &visualsMatched);
      if (visualsMatched == 0)
      {
          gvbitdepth = vTemplate.depth = 1;
          visualList = XGetVisualInfo(dpy, VisualScreenMask | VisualDepthMask,
                                      &vTemplate, &visualsMatched);
          if (visualsMatched == 0)
          {
            fprintf(stderr,
                    "X-Geomview currently supports only 24, 8, and 1 bit\n"
                    "\tdeep displays.  No supported display depth was found.\n"
                    "\tPlease mail the output of \"xdpyinfo\" to\n"
                    "\tsoftware@geom.umn.edu\n");
            exit(1);
          }
      }
  }

  gvvisual = visualList[0].visual;

  if (gvbitdepth == 8)
  {
    if (!XAllocColorCells(dpy,
		DefaultColormapOfScreen(DefaultScreenOfDisplay(dpy)),
		False, planemasks, nplanes, testcols, (unsigned int)30))
      /* there aren't enough colors so use private colormap instead */
      gvpriv = 1;
    else
    {
      /* there *are* enough colors in shared colormap */
      gvpriv = 0;
      XFreeColors(dpy, DefaultColormapOfScreen(DefaultScreenOfDisplay(dpy)),
		testcols, 30, nplanes);
    }
  }

  

  if (gvbitdepth == 24)
    gvcolormap = XCreateColormap(dpy, RootWindow(dpy, DefaultScreen(dpy)),
                                gvvisual, AllocNone);
  else
    if (gvbitdepth == 1)
      gvcolormap = DefaultColormapOfScreen(DefaultScreenOfDisplay(dpy));
    else
      if (gvbitdepth == 8) {
        if (!gvpriv)
          gvcolormap = DefaultColormapOfScreen(DefaultScreenOfDisplay(dpy));
        else {
	  fprintf(stderr,
		"Not enough colors available. Using private colormap.\n");
          gvcolormap = XCreateColormap(dpy,
                RootWindow(dpy, DefaultScreen(dpy)), gvvisual, AllocNone);
	}
      }

}

/*****************************************************************************/
