/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Stuart Levy, Tamara Munzner, Mark Phillips */

#include "mg.h"
#include "../common/main.h"
#include "../common/drawer.h"
#include "../common/ui.h"
#include "../common/comm.h"
#include "../common/lang.h"
#include <gl/gl.h>
#include <gl/get.h>
#include <gl/device.h>
#include <string.h>

int leftmouse_only = 0;	/* Hack to simulate just 1 mouse button (glevent.c) */

static void parse_args(int argc, char *argv[]);

extern HandleOps CommandOps, GeomOps;

main(int argc, char *argv[])
{
  int i;
  CameraStruct cs;

  mgdevice_GL();
  foreground();

  init_geomview(argc, argv);		/* (Just scans for "-noinit" arg) */

  parse_args(argc, argv);		/* Load command & data file(s) */

  cs.h = NULL;
  cs.cam = NULL;

  for(i = 0; i < nwins; i++)
    if (!dview[i]) gv_new_camera(NULL,&cs);

  ui_final_init();

  MainLoop();
  return 0;
}

/*-----------------------------------------------------------------------
 * Function:	env_init
 * Description:	initialize the geomview environment
 * Returns:	
 * Author:	mbp
 * Date:	Thu Jan 21 17:36:01 1993
 * Notes:	This procedure should set the following global variables:
 *		  vv_load_path
 *		  vv_emodule_path
 *		  geomview_system_initfile
 *		On the IRISes it does this by getting them from the
 *		shell environment.  On NeXTs it may use a different
 *		mechanism.  This is the IRIS version.
 */
void
env_init()
{
  extern char *getenv();
#define MAXDIRS 100
  char *dirs[MAXDIRS];
  char *geomview_load_path = getenv("GEOMVIEW_LOAD_PATH");
  char *p, *geomview_emodule_path = getenv("GEOMVIEW_EMODULE_PATH");
  int i;

  if (geomview_load_path == NULL) {
    env_usage();
    exit();
  }
  
  geomview_system_initfile = getenv("GEOMVIEW_SYSTEM_INITFILE");

  if (geomview_load_path) {
    char *path = strdup(geomview_load_path);
    for(i=0, p = strtok(path, ":"); p; p = strtok(NULL, ":"), i++)
	dirs[i] = p;
    dirs[i] = NULL;
    set_load_path(dirs);
    free(path);
  }
  
  if (geomview_emodule_path) {
    char *path = strdup(geomview_emodule_path);
    for(i=0, p = strtok(path, ":"); p; p = strtok(NULL, ":"), i++)
	dirs[i] = p;
    dirs[i] = NULL;
    set_emodule_path(dirs);
    gv_emodule_sort();
    free(path);
  }
  
#undef MAXDIRS
}

env_usage()
{
  fprintf(stderr,
"The shell environment variable GEOMVIEW_LOAD_PATH is not set;\n\
are you trying to run the geomview executable, gvx, directly?\n\
If so, try running the main geomview shell script, called\n\
\"geomview\" instead.  It takes care of setting this and other\n\
relevant environment variables for you.\n\
\n\
If for some reason you do not have the \"geomview\" shell\n\
script you should obtain a copy of geomview from the Geometry\n\
Center's ftp directory on internet host geom.umn.edu.\n\
\n\
Geomview expects the following environment variables to be\n\
set\n\
\n\
GEOMVIEW_LOAD_PATH\n\
    colon-separated list of directories that geomview should\n\
    look in for files you load\n\
\n\
GEOMVIEW_EMODULE_PATH\n\
    colon-separated list of directories that geomview should\n\
    look in for external modules (these appear in the\n\
    \"Applications\" browser on the main panel).\n\
\n\
GEOMVIEW_SYSTEM_INITFILE\n\
    pathname of a system-wide initialization file (.geomview)\n\
    to read upon startup\n");
}

#define STREQ(s1,s2) (strcmp(s1,s2)==0)

static void
parse_args(int argc, char *argv[])
{
  char **av;
  int i, ac;

  for(ac = argc, av = argv+1; --ac > 0; av++) {
    if(STREQ(*av, "-leftmouse")) {
	leftmouse_only = 1;
    } else if (!parse_common_arg(&ac, &av)) {
	usage(argv[0], "");
    }
  }
}

/*
 * isntsc() -- returns 1 if NTSC mode enabled, else 0.
 */
int
isntsc()
{  return getmonitor() == NTSC; }

/*
 * ntsc() -- turns NTSC mode on (1) or off (0), or toggle (-1).
 */
ntsc(int on)
{
    if(on < 0)		/* Toggle */
	on = !isntsc();

    setmonitor(on ? NTSC : HZ60);
    setvideo(GL_VIDEO_REG, getvideo(GL_VIDEO_REG) | GLV_SGREEN);
}

LDEFINE(cull_backface, LVOID,	
"(cull-backface [on|off])\n\
	Select whether back-facing polygons should be displayed.\n\
	Initially on: all polygons are displayed.  When off, polygons whose\n\
	vertices are arranged clockwise on the screen are hidden.  Useful for\n\
	simulating two-sided surface coloring.")
{
  int on;
  LDECLARE(("cull-backface", LBEGIN,
    LKEYWORD, &on,
    LEND));
  backface(on == YES_KEYWORD || on == ON_KEYWORD);
  gv_redraw(ALLCAMS);
  return Lt;
}

static void
loadcursor(char *pbmfile, int xorigin, int yorigin)
{
  FILE *f = fopen(pbmfile, "r");
  int i, j, byte, xlen, size, sz[2];
  char *err;
  unsigned char cs[128];

  if(f == NULL) { err = sperror(); goto nope; }
  if(fexpectstr(f, "P4") || fgetni(f, 2, sz, 0) != 2)
        { err = "Not a binary PBM file"; goto nope; }
  fnextc(f, 1);  getc(f);       /* Skip newline */
  bzero(&cs, sizeof(cs));
  size = sz[0]>sz[1] ? sz[0] : sz[1];
  size = (size > 16) ? 32 : 16;
  xlen = size / 8;
  for(i = (size - sz[1])/2; i < sz[1] && i < size; i++) {
    for(j = 0; j < (sz[0]+7)/8; j++) {
        byte = fgetc(f);
        if(i >= 0 && i < size && j >= 0 && j < xlen)
            cs[(size-1-i)*xlen + j] = byte;
    }
  }
  curstype(size==32 ? C32X1 : C16X1);
  defcursor(1, (unsigned short *)cs);
  curorigin(1, xorigin, yorigin);
  setcursor(1,0,0);
  return;
nope:
  OOGLError(0, "cursor: %s: %s", pbmfile, err);
}



LDEFINE(cursor, LVOID,
"(cursor CAM-ID  {on|off} [pbmfile xorigin yorigin])\n\
	Turns the given window's graphics cursor on or off.\n\
	Optionally sets the 16x16 pixel cursor glyph from the given file,\n\
	which must be in binary (P4) PBM format.  Can only be applied to\n\
	actual windows, not e.g. \"allcams\" or \"default\".  Sorry.")
{
  int on;
  char *pbmfile = NULL;
  DView *dv;
  int xorigin = 0, yorigin = 0;
  int id;
  LDECLARE(("cursor", LBEGIN,
    LID, &id,
    LKEYWORD, &on,
    LOPTIONAL,
    LSTRING, &pbmfile,
    LINT, &xorigin,
    LINT, &yorigin,
    LEND));
  if((dv = (DView *)drawer_get_object(id)) == NULL || !ISCAM(dv->id)
	|| dv->mgctx == NULL)
    return Lnil;

  mgctxselect(dv->mgctx);	/* Implies winset() to the graphics window */

  if(on == NO_KEYWORD || on == OFF_KEYWORD) cursoff(); else curson();

  if(pbmfile) loadcursor(pbmfile, xorigin, yorigin);
  return Lt;
}
