/* NDview.c:  external module for geomview
 *             allowing n-dimensional objects  
 *  version 2.0: Nov. 8, 1993        by Olaf Holt
 */

#include<stdio.h>
#include<stdlib.h>
#include<strings.h>
#include "forms.h"
#include "NDpanels.h"
#include<transformn.h>
#include<camera.h>

#define MAXCHARSPERLINE  	200
#define NAMESIZE  			20
#define TYPESIZE  			20
#define FILENAMESIZE  		160
#define MAXENTRIES			300
#define LENSVALCONST		50
#define ROT1VALCONST		.5
#define DEFAULT				1
#define USER-ADDED			2
#define EPSILON				.00001
#define SLIDERSIZE          .05

int buttondimensions[8] = { 1, 2, 3, 0, 0, 0, 0, 0 };
int buttonvalues[7] = { 0, 1, 2, 3, 4, 5, 6 };
int dimension = 3;
int numbuttonsdown;
int debug = 0;
long winid;
char *clustername;
char *name[MAXENTRIES], *type[MAXENTRIES], *filename[MAXENTRIES];
int maxnames, maxdefaultnames;
int typenum[MAXENTRIES];
int envline, colline, demoline, objectline;
int envreference[MAXENTRIES], colreference[MAXENTRIES], demoreference[MAXENTRIES], objreference[MAXENTRIES];
char *newenvname = NULL, *newenvfilename = NULL;
char *newcolname = NULL, *newcolfilename = NULL;
char *newobjname = NULL, *newobjfilename = NULL;
int CONSTUPDATEFLAG = 1;
int userlevel;

extern char *getenv();
extern size_t strlen(const char *);
char NDviewdir[512];

char *NDviewfile(char *file)
{
static char path[512];

    sprintf(path,"%s/%s",NDviewdir,file);
    return(path);
}

void update_dimension()
{
	char buf[10];
	int newdimension;

	rewind(stdin);
	printf("(echo (dimension) \"\\n\")\n"); fflush(stdout);
	fgets(buf,10,stdin); newdimension = atoi(buf);
	if(newdimension < 3) newdimension = 3;   /* if gv in not ND yet */
	if(dimension != newdimension) {
		dimension = newdimension;
		fl_set_counter_value(DimensionCounter,(float) dimension);
	}
}

main( argc, argv )
int argc; char ** argv;
{
	char *geomroot;

	foreground();

	if(argc > 1 && argv[1][1] == 'd' )  { /* debugging turned on */
		fprintf(stderr, "Debugging on.\n");
		debug = 1;
	}

    if( (geomroot = getenv("GEOMROOT")) == NULL ) {
        fprintf(stderr, "Normally, modules are run in an environment in which the GEOMROOT environment\nvariable is set, which has not been done.  This is done by the geomview\nshell script:  please check your shell script.\n");
        if (debug != 1) exit(1);
    }

    sprintf(NDviewdir,"%s/data/NDview",geomroot);

	create_the_forms();

/* INITIALIZE STUFF ON MAIN PANEL */
	fl_hide_object(notndgroup);
	fl_hide_object(prefabgroup);
	fl_hide_object(toolkitgroup);
	fl_hide_object(axesgroup);
	fl_hide_object(eraser);
	fl_show_object(introgroup);

	fl_set_button(BeginnerButton,1);  /* presses this button */
	fl_set_button(updateconstantlybutton,1);  /* presses this button */

	winid = fl_show_form(NDviewMainPanel,FL_PLACE_SIZE,TRUE,"NDview");
	winset(winid);  winpop();

	update_dimension();
	fl_set_counter_value(DimensionCounter,(float) dimension);
	fl_set_counter_return(DimensionCounter,1);
	fl_set_counter_bounds(DimensionCounter,3.,100.);
	fl_set_counter_step(DimensionCounter,1.,1.);
	fl_set_counter_precision(DimensionCounter,0);

	fl_set_slider_value(rotxslider,0.0);
	fl_set_slider_value(rotyslider,0.0);
	fl_set_slider_value(rotzslider,0.0);
	fl_set_slider_value(lensslider,0.245);

	fl_set_slider_size(rotxslider,SLIDERSIZE);
	fl_set_slider_size(rotyslider,SLIDERSIZE);
	fl_set_slider_size(rotzslider,SLIDERSIZE);
	fl_set_slider_size(lensslider,SLIDERSIZE);

	initializebrowsers();	


	DimensionCounterproc(NULL,NULL); /* initializes the buttons */
	fl_set_input(Clusterbox,"cluster1");

	if((fl_load_browser(IntroHelpBrowser,NDviewfile("text/introhelp.txt"))) == NULL)
		fprintf(stderr, "Couldn't load %s\n", NDviewfile("text/introhelp.txt"));
	if((fl_load_browser(PrefabHelpBrowser,NDviewfile("text/prefabhelp.txt"))) == NULL)
		fprintf(stderr, "Couldn't load %s\n", NDviewfile("text/prefabhelp.txt"));
	if((fl_load_browser(ToolkitHelpBrowser,NDviewfile("text/toolkithelp.txt"))) == NULL)
		fprintf(stderr, "Couldn't load %s\n",NDviewfile("text/toolkithelp.txt"));
	if((fl_load_browser(AxesHelpBrowser,NDviewfile("text/axeshelp.txt"))) == NULL)
		fprintf(stderr, "Couldn't load %s\n",NDviewfile("text/axeshelp.txt"));


	printf("(bbox-draw allgeoms off)\n");
	printf("(dimension %d)\n",dimension);
	fflush(stdout);

	while(1) {
		if (CONSTUPDATEFLAG) {
			if ( userlevel == 4 ) {
				/* first check if we are in ND-mode */
			    /* NOTE: this doesn't seem to work all the time */
				if (dimension <= 3) {
					fl_hide_object(axesgroup);
					fl_show_object(notndgroup);
				} else { spanproc(NULL,0); }
				fl_check_forms();
				sginap(20);
			} else fl_do_forms();
		} else {
			fl_do_forms(); 
		}
	}
}

initializebrowsers()
{
	FILE *f;
	int i = 0,j;
	char buf[MAXCHARSPERLINE];

	/* read file .ndview in current directory for names, also check *
	*		system default file .ndview                             */
    for ( j = 1; j < 3; j++) {
		if ( j == 1 ) {  /* check default file first! */
			if ( (f = fopen(NDviewfile("scripts/.ndview"), "r")) == NULL) {
				fprintf(stderr,"No system default .ndview file found!\n");
				break;
			}
		} else {
			maxdefaultnames = i;
			if ( (f = fopen(".ndview", "r")) == NULL) break;
		}

		while(fgets(buf, MAXCHARSPERLINE, f) != NULL) {
			if( strchr(buf,':') == NULL) {
				fprintf(stderr, "Wieird line in .ndview file encountered: no :'s\n");
			} else if( (name[i] = strdup(strtok(buf,":"))) == NULL) {
				fprintf(stderr, "Wieird line in .ndview file encountered:\n%s\n",buf);
			} else if ( (type[i] = strdup(strtok(NULL,":"))) == NULL) {
				fprintf(stderr, "Wieird line in .ndview file encountered:\n%s\n",buf);
			} else if ( (filename[i] = strdup(strtok(NULL,":"))) == NULL) {
				fprintf(stderr, "Wieird line in .ndview file encountered:\n%s\n",buf);
			} else {
				if ( filename[i][(int)strlen((const char *)filename[i])-1] == '\n') 
					filename[i][(int)strlen((const char *)filename[i])-1] = NULL;
				if ( strcmp(type[i], "environment") == 0 ) {
					typenum[i] = 0;
					i++;
				} else if ( strcmp(type[i], "colormap") == 0 ) {
					typenum[i] = 1;
					i++;
				} else if ( strcmp(type[i], "demo") == 0 ) {
					typenum[i] = 2;
					i++;
				} else if ( strcmp(type[i], "object") == 0 ) {
					typenum[i] = 3;
					i++;
				} else { 
					fprintf(stderr, "Wieird line in .ndview file encountered:\n%s\n",buf);
				}
			}
		}
		fclose(f);
	}
	maxnames = i;

	envline = 0; colline = 0; demoline = 0; objectline = 0;

	fl_freeze_object(EnvBrowser);
	fl_freeze_object(ColBrowser);
	fl_freeze_object(DemoBrowser);
	fl_freeze_object(ObjBrowser);
	for(i = 0; i< maxnames; i++) {
		if ( typenum[i] == 0 ) {
			fl_add_browser_line(EnvBrowser,name[i]);
			envreference[envline++] = i;
		} else if ( typenum[i] == 1 ) {
			fl_add_browser_line(ColBrowser,name[i]);
			colreference[colline++] = i;
		} else if ( typenum[i] == 2 ) {
			fl_add_browser_line(DemoBrowser,name[i]);
			demoreference[demoline++] = i;
		} else if ( typenum[i] == 3 ) {
			fl_add_browser_line(ObjBrowser,name[i]);
			objreference[objectline++] = i;
		} else fprintf(stderr, "Please check source: #29348\n");
	}
	fl_unfreeze_object(EnvBrowser);
	fl_unfreeze_object(ColBrowser);
	fl_unfreeze_object(DemoBrowser);
	fl_unfreeze_object(ObjBrowser);
}

void newcolormapbuttonproc(FL_OBJECT *obj, long i)
{
    /* NOTE:  remove hard-coding here */
    /*        system("/u/gcg/ngrap/src/bin/NDview/coledit/colormap &");  */
/* NOTE2: colormap isn't being distributed yet  -- TOR 12/2/94 */
/*	printf("(emodule-run colormap)\n"); fflush(stdout); */
	printf("(emodule-run smoothcolor)\n"); fflush(stdout);
    update_dimension();
}

void MainPanelColorproc(FL_OBJECT *obj, long i)
{
	system("/u/gcg/ngrap/src/bin/ndviewer/smoothcolor &");
	update_dimension();
}

void newwindowbuttonproc(FL_OBJECT *obj, long i)
{
	update_dimension();
	winid = fl_show_form(NewWindowPanel,FL_PLACE_MOUSE,TRUE,"New Window");
	winset(winid);  winpop();
}

void DimensionCounterproc(FL_OBJECT *obj, long i)
{
	int j;


	if( (dimension = (int)(fl_get_counter_value(DimensionCounter)+.1))
			< 3) 
		fprintf(stderr, "Please check source, #19872\n");

	for(j = 0; j< 7; j++) {
		buttonvalues[j] = j;
	}

	numbuttonsdown = 0;

	buttondimensions[0] = 1;
	buttondimensions[1] = 2;
	buttondimensions[2] = 3;

	fl_freeze_form(NewWindowPanel);

	fl_hide_object(NewWindowQuit);

	fl_set_object_label(b1, "1");
	fl_set_object_label(b2, "2");
	fl_set_object_label(b3, "3");

	fl_set_button(b1, 0);  /* not pressed */
	fl_set_button(b2, 0);  /* not pressed */
	fl_set_button(b3, 0);  /* not pressed */

	if ( dimension >= 4) {
		fl_set_object_label(b4, "4");
		fl_show_object(b4);
		fl_set_button(b4, 0);  /* not pressed */
	} else
		fl_hide_object(b4);

	if ( dimension >= 5) {
		fl_set_object_label(b5, "5");
		fl_show_object(b5);
		fl_set_button(b5, 0);  /* not pressed */
	} else
		fl_hide_object(b5);

	if ( dimension >= 6) {
		fl_set_object_label(b6, "6");
		fl_set_button(b6, 0);  /* not pressed */
		fl_show_object(b6);
	} else
		fl_hide_object(b6);

	if ( dimension >= 7)
		fl_show_object(b7);
	else
		fl_hide_object(b7);

	fl_unfreeze_form(NewWindowPanel);

	/* send message to geomview */
	if( obj != NULL ) {
		printf("(dimension %d)\n",dimension);
		fflush(stdout);
	}
}


void EnvBrowserproc(FL_OBJECT *obj, long i)
{
	int line = fl_get_browser(obj);

	if (line > 0) {
			printf("(load %s)\n",filename[envreference[line-1]]);
			fflush(stdout);
	} else fprintf(stderr, "Please check source: #87622\n");
	winid = fl_show_form(NDviewMainPanel,FL_PLACE_SIZE,TRUE,"N-D Viewer");
	winset(winid);  winpop();
	update_dimension();
}

void ColBrowserproc(FL_OBJECT *obj, long i)
{
	int line = fl_get_browser(obj);

	if (line > 0) {
			printf("(load %s)\n",filename[colreference[line-1]]);
			fflush(stdout);
	} else fprintf(stderr, "Please check source: #87632\n");
	update_dimension();
}

void DemoBrowserproc(FL_OBJECT *obj, long i)
{
	int line = fl_get_browser(obj);
	char buf[10];

	if (line > 0) {
			/* run selected demo */
			printf("(emodule-run %s)\n",filename[demoreference[line-1]]);
			fflush(stdout);
	} else fprintf(stderr, "Please check source: #83632\n");
	update_dimension();
}

void ObjBrowserproc(FL_OBJECT *obj, long i)
{
	int line = fl_get_browser(obj);

	if (line > 0) {
			/* load selected object */
			printf("(load %s)\n",filename[objreference[line-1]]);
			fflush(stdout);
	} else fprintf(stderr, "Please check source: #24387\n");
	update_dimension();
}

void updateproc(FL_OBJECT *obj, long i)
{
	switch (i) {
		case 1:
			CONSTUPDATEFLAG = 0;
			/* first check if we are in ND-mode */
			if (dimension <= 3) {
				fl_hide_object(axesgroup);
				fl_show_object(notndgroup);
			} else spanproc(NULL,0);
			break;
		case 2:
			CONSTUPDATEFLAG = 1;
			if (dimension <= 3) {
				fl_hide_object(axesgroup);
				fl_show_object(notndgroup);
			} else spanproc(NULL,0);
			break;
	}
}

void rotproc(FL_OBJECT *obj, long i)
{
	float rotval = fl_get_slider_value(obj);

	if ( rotval == 0 ) {
		printf("(transform target focus focus rotate 0 0 0)\n");
		fl_set_slider_value(rotxslider,0);
		fl_set_slider_value(rotyslider,0);
		fl_set_slider_value(rotzslider,0);
	} else {
		switch ( i ) {
			case 1:
				printf("(transform-incr target focus focus rotate 1.57 0 0 %f)\n",ROT1VALCONST/rotval);
				fl_set_slider_value(rotyslider,0);
				fl_set_slider_value(rotzslider,0);
				break;
			case 2:
				printf("(transform-incr target focus focus rotate 0 1.57 0 %f)\n",ROT1VALCONST/rotval);
				fl_set_slider_value(rotxslider,0);
				fl_set_slider_value(rotzslider,0);
				break;
			case 3:
				printf("(transform-incr target focus focus rotate 0 0 1.57 %f)\n",ROT1VALCONST/rotval);
				fl_set_slider_value(rotxslider,0);
				fl_set_slider_value(rotyslider,0);
				break;
		}
	}
	fflush(stdout);
	update_dimension();
}

void lenssliderproc(FL_OBJECT *obj, long i)
{
	float lensval = fl_get_slider_value(obj);

	printf("(merge camera allcams {focus %f})\n",lensval*lensval*LENSVALCONST);
	fflush(stdout);
	update_dimension();
}

void userlevelproc(FL_OBJECT *obj, long i)
{
	userlevel = i;
	update_dimension();
	switch (i) {
		case 1:  /* beginner */
			fl_freeze_form(NDviewMainPanel);
			fl_hide_object(notndgroup);
			fl_hide_object(prefabgroup);
			fl_hide_object(toolkitgroup);
			fl_hide_object(axesgroup);
			fl_show_object(introgroup);
			fl_unfreeze_form(NDviewMainPanel);
			break;
		case 2:  /* intermediate */
			fl_freeze_form(NDviewMainPanel);
			fl_hide_object(notndgroup);
			fl_hide_object(introgroup);
			fl_hide_object(toolkitgroup);
			fl_hide_object(axesgroup);
			fl_show_object(prefabgroup);
			fl_unfreeze_form(NDviewMainPanel);
			break;
		case 3:  /* toolkitgroup */
			fl_freeze_form(NDviewMainPanel);
			fl_hide_object(notndgroup);
			fl_hide_object(introgroup);
			fl_hide_object(prefabgroup);
			fl_hide_object(axesgroup);
			fl_show_object(toolkitgroup);
			fl_unfreeze_form(NDviewMainPanel);
			break;
		case 4:  /* axesgroup */
			fl_freeze_form(NDviewMainPanel);
			fl_hide_object(introgroup);
			fl_hide_object(prefabgroup);
			fl_hide_object(toolkitgroup);
			if (dimension <= 3) {
				fl_hide_object(axesgroup);
				fl_show_object(notndgroup);
			} else {
				spanproc(NULL,0);
				fl_show_object(axesgroup);
			}
			fl_unfreeze_form(NDviewMainPanel);
			break;
		default:
			fprintf(stderr,"Please check source: #198324\n");
			break;
	}
}

void TessaractDemoproc(FL_OBJECT *obj, long i)
{
	printf("(emodule-run NDdemo)\n");
	fflush(stdout);
	update_dimension();
}

void Helpproc(FL_OBJECT *obj, long i)
{
	switch(i) {
		case 1:  /* show Intro help */
			winid = fl_show_form(IntroHelpPanel,FL_PLACE_MOUSE,TRUE,"N-D Viewer Introductory Help");
			winset(winid);  winpop();
	        minsize(10,10); maxsize(1024,800);
   	        winconstraints();
			update_dimension();
			break;
		case 2:  /* show Prefab help */
			winid = fl_show_form(PrefabHelpPanel,FL_PLACE_MOUSE,TRUE,"N-D Viewer Prefabricated Help");
			winset(winid);  winpop();
	        minsize(10,10); maxsize(1024,800);
   	        winconstraints();
			update_dimension();
			break;
		case 3:  /* show Toolkit help */
			winid = fl_show_form(ToolkitHelpPanel,FL_PLACE_MOUSE,TRUE,"N-D Viewer Toolkit Help");
			winset(winid);  winpop();
	        minsize(10,10); maxsize(1024,800);
   	        winconstraints();
			update_dimension();
			break;
		case 4:  /* show Axes help */
			winid = fl_show_form(AxesHelpPanel,FL_PLACE_MOUSE,TRUE,"N-D Viewer Basis Vectors Help");
			winset(winid);  winpop();
	        minsize(10,10); maxsize(1024,800);
   	        winconstraints();
			update_dimension();
			break;
		case 11: /* hide Intro help */
			fl_hide_form(IntroHelpPanel);
			update_dimension();
			break;
		case 12: /* hide Prefab help */
			fl_hide_form(PrefabHelpPanel);
			update_dimension();
			break;
		case 13: /* hide Toolkit help */
			fl_hide_form(ToolkitHelpPanel);
			update_dimension();
			break;
		case 14: /* hide Axes help */
			fl_hide_form(AxesHelpPanel);
			update_dimension();
			break;
		default:
			fprintf(stderr, "Please check source: #293842348798\n");
	}
}

void MainPanelQuitproc(FL_OBJECT *obj, long i)
{
	/*  Note:  some kind of graceful exit?  */
		exit(0);
}

void NewWindowCancelproc(FL_OBJECT *obj, long i)
{
	fl_hide_form(NewWindowPanel);
	update_dimension();
}

void Clusterproc(FL_OBJECT *obj, long i)
{
	/* this input specifies which cluster we are attaching to */
	clustername = fl_get_input(Clusterbox);
	update_dimension();
}

void NewWindowQuitproc(FL_OBJECT *obj, long i)
{
	clustername = fl_get_input(Clusterbox);
	if ( strlen(clustername) == 0 )  clustername = "cluster1";
	fl_hide_form(NewWindowPanel);
	/*  generate_new_window(buttondimensions[0]-1, buttondimensions[1]-1,
			buttondimensions[2]-1);   */
	printf("(new-camera %s:%d_%d_%d)\n",clustername,buttondimensions[0], buttondimensions[1],buttondimensions[2]);
	printf("(ND-axes %s:%d_%d_%d %s %d %d %d)\n", clustername, buttondimensions[0], buttondimensions[1], buttondimensions[2], clustername, buttondimensions[0]-1, buttondimensions[1]-1,buttondimensions[2]-1);
	fflush(stdout);
	if(debug) 
		fprintf(stderr, "returned dims: %d %d %d\n", buttondimensions[0]-1, 
			buttondimensions[1]-1, buttondimensions[2]-1); 
	update_dimension();
}

void DimensionButtonproc(FL_OBJECT *obj, long i)
{
	int shift, j, k;
	char *otherstring;

		/* Note:  need to make sure numbuttonsdown doesn't become
				too large */
	

	if( i != 7 ) {  /* one of the buttons was pressed */
		if(fl_get_button(obj))
			buttondimensions[numbuttonsdown++] = buttonvalues[i];
		else {  /* move all the numbers down the array */
			numbuttonsdown--;
			shift = 0;
			for(j= 0; buttondimensions[j] != 0; j++) {
				if (shift) {
					buttondimensions[j] = buttondimensions[j+1];
				} else if( buttondimensions[j] == buttonvalues[i] ) {
					shift = 1;
					buttondimensions[j] = buttondimensions[j+1];
				}
			}
			if (!shift)
				fprintf(stderr, "Please check source: #76345\n");
/*  Note:  IT GOT HERE!!!*/
		}
	} else {	
		/* the category "Other" was used */

		otherstring = fl_get_input(b7);

		i = atoi(otherstring);
		if( (i < 1) || (i > dimension) || (i == buttonvalues[1]) || (i == buttonvalues[2]) || (i == buttonvalues[3]) || (i == buttonvalues[4]) || (i == buttonvalues[5]) || (i == buttonvalues[6]))
			return;

		if(fl_get_button(b6) == 0) {
			fl_set_object_label(b6,otherstring);
			fl_set_button(b6, 1);   /* pushes the button in */
			buttonvalues[6] = 
			buttondimensions[numbuttonsdown++] = i;
		} else if(fl_get_button(b5) == 0) {
			fl_set_object_label(b5,otherstring);
			fl_set_button(b5, 1);   /* pushes the button in */
			buttonvalues[5] = 
			buttondimensions[numbuttonsdown++] = i;
		} else if(fl_get_button(b4) == 0) {
			fl_set_object_label(b4,otherstring);
			fl_set_button(b4, 1);   /* pushes the button in */
			buttonvalues[4] = 
			buttondimensions[numbuttonsdown++] = i;
		} else if(fl_get_button(b3) == 0) {
			fl_set_object_label(b3,otherstring);
			fl_set_button(b3, 1);   /* pushes the button in */
			buttonvalues[3] = 
			buttondimensions[numbuttonsdown++] = i;
		} else if(fl_get_button(b2) == 0) {
			fl_set_object_label(b2,otherstring);
			fl_set_button(b2, 1);   /* pushes the button in */
			buttonvalues[2] = 
			buttondimensions[numbuttonsdown++] = i;
		} else if(fl_get_button(b1) == 0) {
			fl_set_object_label(b1,otherstring);
			fl_set_button(b1, 1);   /* pushes the button in */
			buttonvalues[1] = 
			buttondimensions[numbuttonsdown++] = i;
		} else {  /* all the buttons have been used! */
			fl_set_object_label(b6,otherstring);
			fl_set_button(b6, 1);   /* pushes the button in */
			k = 0;
			while (buttondimensions[k] != buttonvalues[6]) k++;
			buttonvalues[6] = 
			buttondimensions[k] = i;
		}
	}

	if (numbuttonsdown == 3) {
		fl_show_object(NewWindowQuit);
	} else {
		fl_hide_object(NewWindowQuit);
	}

	if(debug)
		fprintf(stderr, "numdown = %d, buttondimensions are %d %d %d %d %d %d %d\n",numbuttonsdown,buttondimensions[0],buttondimensions[1],buttondimensions[2],buttondimensions[3],buttondimensions[4],buttondimensions[5],buttondimensions[6]);

	update_dimension();
}

void spanproc(FL_OBJECT *obj, long i)
{
	TransformN *O2W=NULL, *W2U=NULL, *C2U=NULL, *U2C=NULL;
	TransformN *O2C=NULL, *C2O=NULL, *tmp=NULL;
	HPointN *camvect=NULL, *translation=NULL;
	char *buf, buf1[256], buf2[256], ch, *inputstring, whatcamera[80];
	int objectflag = 0, xdim = -1, ydim = -1, zdim = -1;
	int j, k;
	Camera *focus = NULL;
	float focallen;
	static int what1;

	/* first check if we are in ND-mode */
	if (dimension <= 3) {
		fprintf(stderr, "Please check source: #982848484848\n");
		return;
	}

	/* Flush stdin */
/*
	rewind(stdin);
*/
		printf("(echo x)\n");
		fflush(stdout);
	    ch = '\0';
/*
	    fprintf(stderr,"buffer contents=");
*/
	    while (ch != 'x')
        {
	      ch = (char)fgetc(stdin);
/*
	      fprintf(stderr,"%c", ch);
*/
		}
/*
		fprintf(stderr,"\n");
*/

	/* Find out if the target is the world */
	printf("(echo (real-id targetgeom))\n");
	printf("(echo \"\\n\")\n");
	fflush(stdout);
	if( fgets(buf1, sizeof(buf1), stdin) == NULL) {
		fprintf(stderr, "Please check source: #293847238\n");
	}
	if( strcmp(buf1, "\"World\"\n") != 0) {
		/* the world is NOT selected */
/*
fprintf(stderr, "The world is NOT selected: %s\n",buf1);
*/
		objectflag = 1;
		printf("(echo (ND-xform targetgeom))\n");
		fflush(stdout);
		O2W = TmNRead(stdin);
	} else  {
/*
fprintf(stderr, "The world is selected\n");
*/
			;
	}

	printf("(echo (ND-xform world))\n");
	fflush(stdout);
	W2U = TmNRead(stdin);
	printf("(echo (ND-xform focus))\n");
	fflush(stdout);
	C2U = TmNRead(stdin);

	/* Then multiply the right ones together */
	U2C = TmNInvert(C2U, U2C);
	if ( objectflag ) {
		tmp = TmNConcat(O2W, W2U, tmp);
		O2C = TmNConcat(tmp, U2C, O2C);
	} else {
		O2C = TmNConcat(W2U, U2C, O2C);
	}

	/* next find out what camera this is */
	printf("(echo (real-id focus) \"\\n\")\n"); fflush(stdout);
    fnextc(stdin, 0);
	if ( scanf("\"%[^\"]\"",buf1) == NULL) {
		fprintf(stderr, "Please check source: #2388433\n");;
	}
	sprintf(whatcamera,"Information for\n%s",buf1);

	/* next find out what the x, y, z, projections are */
	printf("(echo (ND-axes focus))\n");
	printf("(echo \"\\n\")\n");
	fflush(stdout);
    fnextc(stdin, 0);
	if ( scanf("(\"%[^\"]\"%d%d%d%[^\)]\)",buf1,&xdim,&ydim,&zdim,buf2) == NULL) {
		fprintf(stderr, "Please check source: #2938423\n");;
	}

	if ( xdim < 0 || ydim < 0 || zdim < 0 || xdim > dimension || ydim > dimension || zdim > dimension ) {
		/* something is wrong */
fprintf(stderr, "buf1 = %s\n", buf1);
		fprintf(stderr, "Please check source: #293842938 (%d %d %d)\n", xdim, ydim, zdim);
		TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
		return;
	}
/*
fprintf(stderr, "The axes for this camera are: %d %d %d\n", xdim, ydim, zdim);
*/

	/* Now add the camera displacement */
	/* the camera is displaced in the zdim direction by focallen */
	printf("(write camera - focus)\n"); 
    fflush(stdout);
	focus = CamFLoad(NULL, stdin, NULL);
	CamGet(focus, CAM_FOCUS, &focallen);

	/* Give the camera position as it is seen in the projected space? */
	C2O = TmNInvert(O2C, C2O);
	translation = HPtNCreate(dimension+1, &(O2C->a[(dimension+1)*dimension]));
	for ( j = 0; j< dimension; j++) translation->v[j] *= -1;
	translation->v[zdim] += focallen;
	translation->v[dimension] = 0;  /* we don't want to translate it _again_ */
	camvect = HPtNTransform(C2O, translation, camvect);

	switch(i) {
		case 0:   /* just update */
			buf1[0] = '\('; buf1[1] = NULL;
			for ( j = 0; j< dimension-1; j++)  {
				sprintf(buf2, "%.2f", O2C->a[xdim*(dimension+1)+j]);
				strcat(buf1, buf2);
				strcat(buf1, ",");
			}
			sprintf(buf2, "%.2f", O2C->a[xdim*(dimension+1)+dimension-1]);
			strcat(buf1, buf2);
			strcat(buf1, ")");
			fl_set_input(xdirinput, buf1);

			buf1[0] = '\('; buf1[1] = NULL;
			for ( j = 0; j< dimension-1; j++)  {
				sprintf(buf2, "%.2f", O2C->a[ydim*(dimension+1)+j]);
				strcat(buf1, buf2);
				strcat(buf1, ",");
			}
			sprintf(buf2, "%.2f", O2C->a[ydim*(dimension+1)+dimension-1]);
			strcat(buf1, buf2);
			strcat(buf1, ")");
			fl_set_input(ydirinput, buf1);

			buf1[0] = '\('; buf1[1] = NULL;
			for ( j = 0; j< dimension-1; j++)  {
				sprintf(buf2, "%.2f", O2C->a[zdim*(dimension+1)+j]);
				strcat(buf1, buf2);
				strcat(buf1, ",");
			}
			sprintf(buf2, "%.2f", O2C->a[zdim*(dimension+1)+dimension-1]);
			strcat(buf1, buf2);
			strcat(buf1, ")");
			fl_set_input(zdirinput, buf1);

			buf1[0] = '\('; buf1[1] = NULL;
			/* add in the adjustment of the camera for printing purposes */
			O2C->a[(dimension+1)*dimension+zdim] -= focallen;
			for ( j = 0; j< dimension-1; j++)  {
				sprintf(buf2, "%.1f", -O2C->a[dimension*(dimension+1)+j]);
				strcat(buf1, buf2);
				strcat(buf1, ",");
			}
			sprintf(buf2, "%.1f", -O2C->a[dimension*(dimension+1)+dimension-1]);
			strcat(buf1, buf2);
			strcat(buf1, ")");
			fl_set_input(transinput, buf1);
			/* remove the adjustment of the camera */
			O2C->a[(dimension+1)*dimension+zdim] += focallen;

			buf1[0] = '\('; buf1[1] = NULL;
			sprintf(buf2, "%.2f", camvect->v[xdim]);
			strcat(buf1, buf2); strcat(buf1, ",");
			sprintf(buf2, "%.2f", camvect->v[ydim]);
			strcat(buf1, buf2); strcat(buf1, ",");
			sprintf(buf2, "%.2f", camvect->v[zdim]);
			strcat(buf1, buf2); strcat(buf1, ")");
			fl_set_input(camviewinput, buf1);
			fl_show_object(eraser);
			fl_set_object_label(whatcameratext,whatcamera);
			fl_hide_object(eraser);

			break;
		case 1:   /* x direction */
			buf = strdup(fl_get_input(xdirinput));
			if ( buf[0] != '\(' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			if ( buf[strlen(buf)-1] != '\)' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			buf[strlen(buf)-1] = NULL;  /* chop off last char */
			if ( (inputstring = strtok(&(buf[1]),",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
				spanproc(NULL,0);
TmNDelete(U2C); TmNDelete(tmp);
                return;
			}
			O2C->a[xdim*(dimension+1)+0] = atof(inputstring);
			for( j = 1; j < dimension; j++ ) {
	            if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                	fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
   	            	TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                	return;
            	}
				O2C->a[xdim*(dimension+1)+j] = atof(inputstring);
			}
			if ( return_new_matrix(O2C, O2W, C2U, W2U, xdim, ydim, zdim, objectflag) == 0 ) {
				/* reinstate old values */
				;
			}
			spanproc(NULL,0);
			break;
		case 2:   /* y direction */
			buf = strdup(fl_get_input(ydirinput));
			if ( buf[0] != '\(' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			if ( buf[strlen(buf)-1] != '\)' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			buf[strlen(buf)-1] = NULL;  /* chop off last char */
            if ( (inputstring = strtok(&(buf[1]),",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			O2C->a[ydim*(dimension+1)+0] = atof(inputstring);
			for( j = 1; j < dimension; j++ ) {
                if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                    fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                    TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
					spanproc(NULL,0);
                    return;
                }
				O2C->a[ydim*(dimension+1)+j] = atof(inputstring);
			}
			if ( return_new_matrix(O2C, O2W, C2U, W2U, ydim, xdim, zdim, objectflag) == 0 ) {
				/* reinstate old values */
				;
			}
			spanproc(NULL,0);
			break;
		case 3:   /* z direction */
			buf = strdup(fl_get_input(zdirinput));
			if ( buf[0] != '\(' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			if ( buf[strlen(buf)-1] != '\)' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			buf[strlen(buf)-1] = NULL;  /* chop off last char */
            if ( (inputstring = strtok(&(buf[1]),",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			O2C->a[zdim*(dimension+1)+0] = atof(inputstring);
			for( j = 1; j < dimension; j++ ) {
                if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                    fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                    TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
					spanproc(NULL,0);
                    return;
                }
				O2C->a[zdim*(dimension+1)+j] = atof(inputstring);
			}
			if ( return_new_matrix(O2C, O2W, C2U, W2U, zdim, xdim, ydim, objectflag) == 0 ) {
				/* reinstate old values */
				;
			}
			spanproc(NULL,0);
			break;
		case 4:   /* displacement */
			buf = strdup(fl_get_input(transinput));
			if ( buf[0] != '\(' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			if ( buf[strlen(buf)-1] != '\)' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			buf[strlen(buf)-1] = NULL;  /* chop off last char */

            if ( (inputstring = strtok(&(buf[1]),",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			O2C->a[dimension*(dimension+1)+0] = -(atof(inputstring));
			for( j = 1; j < dimension; j++ ) {
                if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                    fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                    TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
					spanproc(NULL,0);
                    return;
                }
				O2C->a[dimension*(dimension+1)+j] = -(atof(inputstring));
			}
			O2C->a[(dimension+1)*dimension+zdim] += focallen;

			if ( return_new_matrix(O2C, O2W, C2U, W2U, xdim, ydim, zdim, objectflag) == 0) {
				/* reinstate old values */
				;
			}
			spanproc(NULL,0);
			break;
		case 5:   /* camera viewpoint */
			buf = strdup(fl_get_input(camviewinput));
			if ( buf[0] != '\(' ) {
				fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			if ( buf[strlen(buf)-1] != '\)' ) {
				fprintf(stderr, "Format of input string must be (num1, num2, num3)\n");
				TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
				return;
			}
			buf[strlen(buf)-1] = NULL;  /* chop off last char */

            if ( (inputstring = strtok(&(buf[1]),",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			camvect->v[xdim] = atof(inputstring);
            if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			camvect->v[ydim] = atof(inputstring);
            if ( (inputstring = strtok(NULL,",")  ) == NULL ) {
                fprintf(stderr, "Format of input string must be (a, b, . . . )\n");
                TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U);
TmNDelete(U2C); TmNDelete(tmp);
				spanproc(NULL,0);
                return;
            }
			camvect->v[zdim] = atof(inputstring);
			/* ok, we have the new elements of camvect */

			HPtNTransform(O2C, camvect, translation);
			if ( translation->v[dimension] != 0. ) {
				fprintf(stderr, "Please check source: (%f) #293842338948\n",translation->v[dimension]);
			}
			translation->v[zdim] -= focallen;
		    for ( j = 0; j< dimension; j++) translation->v[j] *= -1;
			for ( j = 0; j < dimension; j++) {
				O2C->a[dimension*(dimension+1) + j] = translation->v[j];
			}
			if ( return_new_matrix(O2C, O2W, C2U, W2U, xdim, ydim, zdim, objectflag) == 0) {
				/* reinstate old values */
				;
			}
			spanproc(NULL,0);
			break;
		default:
			fprintf(stderr, "Please check source: #293848239\n");
	}
	TmNDelete(O2C); TmNDelete(O2W); TmNDelete(W2U); TmNDelete(C2U); TmNDelete(U2C); TmNDelete(tmp);
}

int return_new_matrix(TransformN *O2C, TransformN *O2W, TransformN *C2U, TransformN *W2U, int xdim, int ydim, int zdim, int objectflag)
{
	TransformN *U2W = NULL, *tmp = NULL;
	float **rearranged;
	int j = 0, k;

	rearranged = calloc(dimension, sizeof(float *));
	rearranged[j++] = &(O2C->a[xdim*(dimension+1)]);
	rearranged[j++] = &(O2C->a[ydim*(dimension+1)]);
	rearranged[j++] = &(O2C->a[zdim*(dimension+1)]);
	for ( k = 0; k < dimension; k++)
		if ( k != xdim && k != ydim && k != zdim ) 
			rearranged[j++] = &(O2C->a[k*(dimension+1)]);
	if ( gram_schmidt(O2C,rearranged) == 0) {
		/* got an almost zero matrix, don't return anything! */
		free(rearranged);
		return 0;
	}
	free(rearranged);

	U2W = TmNInvert(W2U, U2W);
	if ( objectflag ) {   /* compute a new O2W */
		/* we get this from O2C * C2U * U2W  */
		tmp = TmNConcat(O2C, C2U, tmp);
		O2W = TmNConcat(tmp, U2W, O2W);
		printf("(ND-xform targetgeom %d %d ",dimension+1,dimension+1);
		for ( j = 0; j < dimension + 1; j++ )
			for ( k = 0; k < dimension + 1; k++ )
				printf("%.2f ",O2W->a[j*(dimension+1)+k]);
		printf(")\n");
		fflush(stdout);
	} else {              /* compute a new C2U */
		/* this we get as the inverse of U2W * 
					O2C  (which is really W2C in this case)  */
		tmp = TmNConcat(U2W, O2C, tmp);
		C2U = TmNInvert(tmp, C2U);
		printf("(ND-xform focus %d %d ",dimension+1,dimension+1);
		for ( j = 0; j < dimension + 1; j++ )
			for ( k = 0; k < dimension + 1; k++ )
				printf("%.2f ",C2U->a[j*(dimension+1)+k]);
		printf(")\n");
		fflush(stdout);
	}
	return 1;
}

int gram_schmidt( TransformN *T, float **order )
{
	float *a, r, *point;
	int i, j, k, currentrow = 0;

	a = order[currentrow];
	for ( k = 0; k< dimension; k++) {
		for( r = 0.0, i = dimension; i; i--, a++)
			r += *a * *a;
		if ( r < EPSILON ) return 0;
		r = sqrt(r);
		a--;
		for (  i = dimension; i; i--)
			*a-- /= r;
		point = order[currentrow];
		for ( j = k + 1; j < dimension; j++) {
			r = 0.0;
			a = order[currentrow + j - k];
			for ( i = dimension; i; i--)
				r += *point++ * *a++;
			for ( i = dimension; i; i--)
				*(--a) -= *(--point) * r;
		}
		a = order[++currentrow];
	}
	return 1;
}
