/********************************************************************************
 *  Projektname		: AERO - Teilprojekt FSB
 *  Filename		: io.c
 *  Filetyp		: C-Source
 ********************************************************************************
 *  Modulname		: io.o
 *  letzte Aenderung	: Dienstag, 27. August 1996, 17:53:45
 *  Autor		: Horst Stolz (HUS)
 *  
 *  Beschreibung:
 *    Dieses Modul stellt Funktionen zur Verfuegung um die Zustandsstruktur in 
 *    ein File abzulegen bzw. um sie von dort wieder zu laden.
 *    Des Weiteren werden Routinen zum Schreiben und Lesen der Materialtabelle 
 *    zur verfuegung gestellt.
 *  
 *  Noch zu machen:
 *
 *  Versionsgeschichte:
 *  15.02.93	Erstellungsdatum
 *  13.03.93    Support zum Lesen und Schreiben von Materialtabellen 
 *              Bruecksichtigung von Kraeften beim Zustand!
 *  19.03.93    Bruecksichtigung von Koerperattributen Masselos&Gravlos
 *  23.03.93    der Massen-Eintrag m... wird beim Laden beruecksichtigt!
 *  31.03.93    Engl. Begriffe verbessert: cuboid->box, plain->plane
 *              und Engl. Fehlermeldungen
 *  01.04.93    zusammengesetzer Koerper wird unterstuetzt,
 *              Fehler-spez. Meldung bei Auftreten von Fehlern per perror(..)
 *  26.04.93    Abspeichern und Laden der FBB-Konfiguration
 *  06.05.93    Unterstuetzung der neu eingefuehrten KoerperId-Nummer (Bloed!)
 *  03.10.93    defaultwerte fuer RGB, durch,ref,rauh aus Tabelle nehmen.
 *  17.01.94    Fehler: Composed immer mit BewTyp FREI|KOLL behoben!
 *  10.02.94    Fehler beim einlesen von guided Forces behoben, F<nr> obsolent!!
 *  
 ********************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#include "io.h"
#include "vektor.h"



/***************************************************************************
 * Format der Zustands-Datei:
 *-------------------------------------------------------------------------
 * # Dies ist eine Bemerkung!!
 * # klein geschriebene Schluesselwoerter sind optional!
 * # Default-Werte sind
 * #   fuer einen Real oder Integer-Wert: 0
 * #   fuer einen Vektor:  0 0 0 
 * #   fuer ein Quaterion: 1 0 0 0           (Betrag muss 1.0 ergeben!)
 * # auch Werte in []-Klammern sind optional!
 * # | bedeutet eine Wahlmoeglichkeit.
 * 
 * Z                     # Anfangsschluesselwort fuer einen Zustand
 * t <real>              # Startzeit (>=0)
 * g <vektor>            # Gravitation
 * s <integer>           # Seedwert fuer Zufallszahlen
 *
 *
 * # beliebige Anzahl von Koerper, Nummer muss fortlaufend sein!
 *
 * K<nr> <koerperart> <form> <material>      # Koerperart, Geometrie und Material
 * m <real>              # Masse [kg]
 * p <vektor>            # Position des Schwerpunktes bzgl. Inertialsystem in [m]
 * v <vektor>            # Geschwindigkeit [m/s]
 * a <vektor>            # Beschleunigung  [m/s^2]
 * q <quaternion>        # Drehung des Inertialkoordsys zum Koerperkoordsys
 * w <vektor>            # Drehgeschwindigkeit [rad/s]
 * u <vektor>            # Drehbeschleunigung [rad/s^2]
 * c <R:ushort> <G:ushort> <B:ushort>   # Farbe nur fuer Kugel, Quader und Zylinder
 * n <durchsichtigkeit:real> <reflexion:real> <rauhigeit:real>  # fuer Raytracer
 * b -[g][m][k]          # Bewegungstyp: keine Grav, Masselos, Kollisionslos
 * i <idnr:long int>     # Identifikationsnr
 * 
 * # falls Koerper "composed" ist kommen noch die dazugehoerigen Eingebetteten
 * # Koerper dazu, ausserdem werden die optionalen Parameter m, p, q und c
 * # wie beim Koerper K<nr> unterstuetzt. (bel. Anzahl erlaubt)
 * E <koerperart> <form> <material>    # nur Kugel, Quader, Zylinder und Massepunkt!
 * m <real>              # Masse
 * p <vektor>            # Position in Weltkoord  [m]
 * q <quaternion>        # Quaternion bzgl. Inertialsys.(Weltkoordsys.)
 * c <R:ushort> <G:ushort> <B:ushort>   # Farbe
 * # Die Eingebetteten Koerper zusammen =>Schwerpunkt <p'> muss mit <p> von
 * # K? composed p <p> uebereinstimmen!
 *
 * X                     # Abschluss der eingebetteten Koerper
 * 
 *
 * V<nr> K<nr1> K<nr2> <verbindart> <verparameter>  # bel. Anzahl Verbindungen
 * a <vektor>            # koerperfeste Pos. der Verbindung in Koerper<nr1>
 * b <vektor>            # und bei Koerper<nr2>
 *
 * # Kraft an Koerper<knr> wirkt zwischen t0 und t1, koerperfester Angriffspunkt<vektor>
 * # Wenn Angriffspunkt<vektor>=(0,0,0) kann er auch weggelassen werden
 * F <t0:real> <t1:real> K<nr> [<vektor>] <kraftpar>
 * 
 * Q  # Ende der Zustandsstruktur
 * 
 * mit
 * <nr> = Integerzahl
 * <real> = Realzahl
 * <vektor> = <real> <real> <real>
 * <quaternion> = <real> <real> <real> <real>
 * <koerperart> = sphere | box | cylinder | plane | nail | mpoint | point | composed
 * <form> fuer eine "sphere" = <radius:real>
 * <form> fuer eine "cylinder" = <radius:real> <height:real>
 * <form> fuer eine "box" = <vektor>
 * <form> fuer e
 * <material> = Materialname aus Tabelle (z.b.: Eisen)
 * #      (allerdings nur fuer sphere, box, cylinder und plane zu setzen)
 * <verbindart> = spring | damper | joint | rod
 * <verparameter> fuer "spring" = <ruhelaenge:real> <federkonst:real>
 * <verparameter> fuer "damper" = <daempfungskonst:real>
 * <verparameter> fuer "rod" = <stangenlaenge:real>
 * <kraftpar> = inertial <kraft:vektor>  # konst. Kraftvektor in Inertial-Koordinaten [N]
 * <kraftpar> = local <kraft':vektor>    # konst. Kraftvektor in Koerperkoord. [N]
 * <kraftpar> = guided <kraftbetrag:real> K<nr> [<zielpunkt:vektor>] 
 *                   # konstante Kraft [N] und Punkt am Zielkoerper(in Koerperkoord) [m]
 *-------------------------------------------------------------------------
 * Es werden beim Einlesen auch einige Plausibilitaetstests gemacht, also
 * zum Bsp. ob der Kugelradius >0 ist, oder der BETRAG(Quaternion)=1.0.
 * Es ist aber nicht ganz sichergestellt das jeder Fehler abgefangen wird.
 *
 * Beispiel fuer ein gueltiges Zustands-File: 
 *
 * Z
 * g 0 -9.81 0                       # Mit Gravitation
 *
 * K1 box 0.1 0.2 0.3 iron           # Eisen-Quader fliegt irgendwie rum
 * v -1.0 0 0
 *
 * K2 sphere 0.05 plastic            # Plastikkugel als Pendel
 * p -2.0 10.0 0
 *
 * K3 nail
 * p 0.0 10.0 0.0
 *
 * V1 K2 K3 rod 2.0
 *
 * F1 K1 0 1 0 0 0 inertial 10 0 0       # Kraft im Schwerpunkt von K1 in X-Richtung 10N
 *                                       # wirkt zwischen 0s <= t <= 1s
 * Q
 *
 *
 *
 ***************************************************************************
 * Format einer Material-Datei:
 *-------------------------------------------------------------------------
 * # Dies ist eine Bemerkung
 * M <MaxMatAnz:Integer>
 * m <name:string> <dichte:real> <gleitfaktor:real> <haftfaktor:real> 
 *    <stossfaktor:real> <dehnfaktor:real> <daempfungsfaktor:real>
 *   [<default_red:ushort> <default_green:ushort> <default_blue:ushort> 
      <durchsichtigkeit:real> <reflexion:real> <rauhigkeit:real>]
 * # Dichte in [kg/m^3]
 * # Durchsichtigkeit... im Bereich 0..1
 * Q
 *
 *
 ***************************************************************************
 * Format einer Konfigurationsdatei:
 *-------------------------------------------------------------------------
 * P
 * h <Integrationsschrittweite> <min> <max>
 * e (Y|N) <eps>  # Fehlersteuerung an Y oder aus N
 * k (Y|N) <Kollisionsschrittweite> #Kollisionen An oder Aus
 * b <MaximaleBeruehrungsgeschw> <MinimaleBeruehrungsTiefe>
 * u <MinGleitGeschw> <MinHaftGeschw>
 * g (Y|N) <ZufGravAnteil in %>
 * r <Federkonstante> <Daempferkonstante>     # Stange
 * j <Federkonstante> <Daempferkonstante>     # Gelenk
 * m <MinFederDehnung>
 * q <MaxQuaderQuerTiefe>
 * s (Y|N)   # Federstoss YES, NO
 * a (Y|N) <Beiwert-Kugel> <Beiwert-Quader> <Beiwert-Zylinder>
 * Q
 * 
 */

extern char fsb_version_string[];

static char **mname;		/* Feld der Materialnamen */
static int manz;		/* Anzahl der Materialnamen */
static FILE *fh;		/* Filehandle */

static TKoerper **pk;		/* vars nur fuer ReadZustand */
static TKoerper **pek;		/* var nur fuer ReadZustand */
static TVerbindung **pv;        
static int konr;		/* Koerpernr fuer ReadZustand */

static long Id;



#define READVEKTOR(r,s) if(ReadReals(3,r)){fprintf(stderr,"::%s\n", s);goto fehler;}

enum modes {READZ, READK, READV, READE, UNKNOWN}; 
/* Lesemoden: Zustand, Koeper, Eingebetteter Koerper, Verbindung */



/* Schreibt den Vektor v nach (fh) falls v kein Nullvektor ist
 * Das Zeichen c wird vorangestellt und v wird mit Zwischenraum und
 * Zeilentrenner am Ende ausgeben.
 */
static void WriteVektor(char c, TVektor v)
{
    TReal r;
    
    r = V_PRODUKT(v,v);
    if (r!=0.0) {
	fprintf(fh, "%c %.8e %.8e %.8e\n", c, v[0], v[1], v[2]);
    }
}


/* Schreibt Quaternion q nach (fh) falls q kein Standardquat. 1 0 0 0 ist.
 * Auch hier wird c vorangestellt und mit Zeilentrenner abgeschlossen.
 */
static void WriteQuaternion(char c, TQuaternion q)
{
    if (q[0]!=1.0) {
	fprintf(fh, "%c %.8e %.8e %.8e %.8e\n", c, q[0], q[1], q[2], q[3]);
    }
}


/* Schreibt alle Verbindungen in v ins File (fh) unter Verwendung der Koerperliste kk
 * (zwecks Koerpernummern)
 * Ausgabe V<nr> K<nr1> K<nr2> <verbart> <verbpar> sowie Verb-Punkte 
 *         a <vektor> und b <vektor>
 * Fehlermeldung bei: Kann Koerpernr nicht ermitteln oder unbekannte Verb.art
 */
static void WriteVerbindungen(TVerbindung *v, TKoerper *kk)
{
    int vnr, knr1, knr2;
    TKoerper *k;

    for (vnr=1; v; v=v->Naechste, vnr++) {
	
	/* Ermittle nr des 1.Koerpers
	 */
	for (k=kk, knr1=1; k; k=k->Naechster, knr1++) if (k==v->Koerper1) break;
	if (k==NULL) {
	    fprintf(stderr, "::1.body not found in body-list (wrong Knr?)\n");
	    return ;
	}
	/* Ermittle nr des 1.Koerpers
	 */
	for (k=kk, knr2=1; k; k=k->Naechster, knr2++) if (k==v->Koerper2) break;
	if (k==NULL) {
	    fprintf(stderr, "::can't find 2nd body of link (wrong Knr?)\n");
	    return ;
	}
	fprintf(fh, "V%d K%d K%d ", vnr, knr1, knr2);

	/* Verbindungs-spezifischer Teil
	 */
	switch(v->Art) {
	case FEDER:
	    fprintf(fh, "spring %.8e %.8e\n", v->VerParameter.Feder.Ruhelaenge,
		    v->VerParameter.Feder.Federkonstante);
	    break;
	case DAEMPFER:
	    fprintf(fh, "damper %.8e\n", v->VerParameter.Daempfer.Daempfungskonstante);
	    break;
	case STANGE:
	    fprintf(fh, "rod %.8e\n", v->VerParameter.Stange.Stangenlaenge);
	    break;
	case GELENK:
	    fprintf(fh, "joint\n");
	    break;
	default:
	    fprintf(stderr, "::can't save unknown link-type!\n");
	    return ;
	    break;
	}
	/* Verbindungspunkte ausgeben
         */
	WriteVektor('a', v->VPunkt1);
	WriteVektor('b', v->VPunkt2);
	    
	fputc('\n', fh);
    }
    
}


/* Ausgabe aller Krafte der Koerper k ins File (fh).
 * Form wie oben beschrieben!
 *
 */
static void WriteKraefte(TKoerper *koerper)
{
    TKraft *f;
    TKoerper *k, *kk;
    int knr, fnr, knr2;

    fnr = 1;
    for (knr=1, k=koerper; k; k=k->Naechster, knr++) 
	for (f = k->Kraefte; f; f = f->Naechste) {
	    fprintf(fh, "F %.8e %.8e K%d ", f->StartZeit, f->EndZeit, knr);
	    if (f->Angriffspunkt[0] != 0.0 ||
		f->Angriffspunkt[1] != 0.0 ||
		f->Angriffspunkt[2] != 0.0)
		fprintf(fh, " %.8e %.8e %.8e ",
			f->Angriffspunkt[0], f->Angriffspunkt[1], f->Angriffspunkt[2]);

	    switch(f->Typ) {
	    case RAUMFEST:
		fprintf(fh, "inertial %.8e %.8e %8e\n\n",
			     f->KD.Kraft[0], f->KD.Kraft[1], f->KD.Kraft[2]);
		break;
	    case KOERPERFEST:
		fprintf(fh, "local %.8e %.8e %8e\n\n",
			     f->KD.Kraft[0], f->KD.Kraft[1], f->KD.Kraft[2]);
		break;
	    case GERICHTET:
		for (kk=koerper, knr2=1; kk; kk=kk->Naechster, knr2++)
		    if (kk == f->KD.GerKraft.ZielKoerper) break;
		if (kk==NULL) {
		    fprintf(stderr, "::can't find target-body of guided force (wrong Knr?)\n");
		    return ;
		}
		fprintf(fh, "guided %.8e K%d", f->KD.GerKraft.Kraftbetrag, knr2);

		if (f->KD.GerKraft.Zielpunkt[0]!=0.0 ||
		    f->KD.GerKraft.Zielpunkt[1]!=0.0 ||
		    f->KD.GerKraft.Zielpunkt[2]!=0.0)
		    fprintf(fh, " %.8e %.8e %.8e\n\n",
			     f->KD.GerKraft.Zielpunkt[0],
			     f->KD.GerKraft.Zielpunkt[1],
			     f->KD.GerKraft.Zielpunkt[2]);
		else fprintf(fh, "\n\n");

		break;
	    default:
	        fprintf(stderr, "::unknown force-type %d\n", (int)(f->Typ));
		return;
	    }
	}
}


/* Ausgabe aller Koerper in k ins File (fh).
 * Form wie oben beschrieben!
 *
 */
static void WriteKoerper(TKoerper *k)
{
    int knr;
    TKoerper *ek;

    for (knr=1; k; k=k->Naechster, knr++) {
	/* Koerperart beruecksichtigen */
        /* Koerperart, Geometrische Daten, Material und Masse ausgeben */
	switch(k->Art) {
	case KUGEL:
	    fprintf(fh, "K%d sphere %.8e %s\nm %.8e\n", 
		    knr, k->Form.Kugel.Radius, mname[k->Material], k->Masse);
	    break;
	case ZYLINDER:
	    fprintf(fh, "K%d cylinder %.8e %.8e %s\nm %.8e\n", 
		    knr, k->Form.Zylinder.Radius, k->Form.Zylinder.Hoehe,
                    mname[k->Material], k->Masse);
	    break;
	case QUADER:
	    fprintf(fh, "K%d box %.8e %.8e %.8e %s\nm %.8e\n", knr,
		    k->Form.Quader.KantenLaenge[0],
		    k->Form.Quader.KantenLaenge[1],
		    k->Form.Quader.KantenLaenge[2],
		    mname[k->Material], k->Masse);
	    break;
	case EBENE:
	    fprintf(fh, "K%d plane %s\n\n", 
		    knr, mname[k->Material]);
	    break;
	case NAGEL:
	    fprintf(fh, "K%d nail\n", knr);
	    break;
	case MPUNKT:
	    fprintf(fh, "K%d mpoint\nm %.8e\n", knr, k->Masse);
	    break;
	case PUNKT:
	    fprintf(fh, "K%d point\n", knr);
	    break;
	case ZUSGESOBJ:
	    fprintf(fh, "K%d composed\nm %.8e\n", knr, k->Masse);
	    break;
	default:
	    fprintf(stderr, "::can't save unknown body-typ!\n");
	    break;
	}

	/* nur die fuer die Koerperart notwendigen Daten werden Ausgegeben  */
	WriteVektor('p', k->Position);
	if (k->Art != NAGEL) WriteQuaternion('q', k->q);
	if (k->Art != EBENE && k->Art != NAGEL) {
	    WriteVektor('v', k->Geschwindigkeit);
	    WriteVektor('w', k->Drehgeschwindigkeit);
	    WriteVektor('a', k->Beschleunigung);
	    WriteVektor('u', k->Drehbeschleunigung);
	}

	if (k->Art != ZUSGESOBJ) {
	    if (k->R || k->G || k->B) {
		fprintf( fh,"c %hu %hu %hu\n", k->R, k->G, k->B );
		
	    }
	    if (k->Durchsichtigkeit != 0.0 || k->Reflexion != 0.0 || 
		k->Rauhigkeit != 0.0) {
		fprintf( fh,"n %.4e %.4e %.4e\n", 
			k->Durchsichtigkeit, k->Reflexion, k->Rauhigkeit);
	    }
	}

        /* Bewegungstyp abspeichern
         */
	switch(k->Art) {
	  case KUGEL:
	  case QUADER:
	  case ZYLINDER:
	  case ZUSGESOBJ:
	    if ((k->BewTyp & MASSELOS) || (k->BewTyp & GRAV0)
		|| ((k->BewTyp & KOLL)==0)) {
		fprintf(fh, "b -%s%s%s\n",
			(k->BewTyp & MASSELOS) ? "m" : "",
			(k->BewTyp & GRAV0) ? "g" : "",
			(k->BewTyp & KOLL) ? "" : "k");
	    }
	    break;
          case EBENE:
            if ((k->BewTyp & KOLL)==0)
	      fprintf(fh, "b -k\n");
	    break;
	  default:
	    break;
	}

        /* KoerperIdentifikationsnr (BLOED!)
         */
        fprintf(fh, "i %ld\n", k->KoerperId);


	/* in Zusammengesetzen Koerper eingebettete Koerper ausgeben!
         */
	if (k->Art == ZUSGESOBJ) {
	    for (ek=k->Form.ZusGesObj.KoerperListe; ek; ek=ek->Naechster) {
	        /* Art, geom. Parameter, Material und Masse ausgeben */
		switch(ek->Art) {
		case KUGEL:
		    fprintf(fh, "E sphere %.8e %s\nm %.8e\n", 
		    	    ek->Form.Kugel.Radius, mname[ek->Material], ek->Masse);
		    break;
		    
		case ZYLINDER:
		    fprintf(fh, "E cylinder %.8e %.8e %s\nm %.8e\n", 
			    ek->Form.Zylinder.Radius, ek->Form.Zylinder.Hoehe,
			    mname[ek->Material], ek->Masse);
		    break;
		case QUADER:
		    fprintf(fh, "E box %.8e %.8e %.8e %s\nm %.8e\n",
		    		ek->Form.Quader.KantenLaenge[0],
		    		ek->Form.Quader.KantenLaenge[1],
		    		ek->Form.Quader.KantenLaenge[2],
		    		mname[ek->Material], ek->Masse);
		    break;
		case MPUNKT:
		    fprintf(fh, "E mpoint\nm %.8e\n", ek->Masse);
		    break;
		default:
		    fprintf(stderr, "::incorrect body-typ of embedded object\n");
		    break;
		}

		/* Position und Drehwinkel ausgeben */
		WriteVektor('p', ek->Position);
		WriteQuaternion('q', ek->q);
	        if (ek->R || ek->G || ek->B)
		    fprintf(fh, "c %hu %hu %hu\n", ek->R, ek->G, ek->B);
		if (ek->Durchsichtigkeit != 0.0 || ek->Reflexion != 0.0 || 
		    ek->Rauhigkeit != 0.0) {
		    fprintf( fh,"n %.4e %.4e %.4e\n", 
			    ek->Durchsichtigkeit, ek->Reflexion, ek->Rauhigkeit);
		}
		fprintf(fh, "i %ld\n", ek->KoerperId);
	    }
	    fprintf(fh, "X\n");
	}

	fputc('\n', fh);
    }
}


/* Schreibt Zustandsstruktur in Datei "fname".
 * Wenn fname==NULL benutze den Filehandle des schon geoffneten Files mit
 * Filehandle fh;
 * Liefert TRUE wenn erfolgreich ohne Fehler gelesen wurde
 */
TBoolean WriteZustand(TZustand *z, char *fname, FILE *handle)
{
    int err;

    fh = handle;
    if (fname) 	fh = fopen(fname, "w");
    if (fh==NULL) return FALSE;

    err=fprintf(fh, "# FSB V%s, p:HUS, io-Modul\n", fsb_version_string);
    if (ferror(fh)) {
	perror("::error while writing");
	goto cleanup;
    }

    if (err==EOF) goto cleanup;
	
    if (z) {
	err = fprintf(fh, "Z\n");
	if (err==EOF) goto cleanup;

	if (z->Zeit != 0.0) err = fprintf(fh, "t %.8e\n", z->Zeit);
	if (err==EOF) goto cleanup;
	if (z->GravitationAn) 
	    err = fprintf(fh, "g %.8e %.8e %.8e\n", z->GravitationsVektor[0],
			z->GravitationsVektor[1], z->GravitationsVektor[2]);
	if (err==EOF) goto cleanup;

        if (z->Seed != 0)  fprintf(fh, "s %d\n", z->Seed);

	mname = MaterialNamen(z->MaterialDaten);

	fputc('\n', fh);
	WriteKoerper(z->Koerper);

	if (err==EOF) goto cleanup;

	WriteVerbindungen(z->Verbindungen, z->Koerper);
	if (err==EOF) goto cleanup;

	WriteKraefte(z->Koerper);
    }


    err = fprintf(fh, "Q\n");

cleanup:
    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }

    return (err==EOF) ? FALSE : TRUE;
}




TBoolean WriteMaterialien(TMaterialTabelle *mt, char *fname, FILE *handle)
/****************************************************************************
 * Schreibt die Materialtabelle ins File "fname" bzw. falls fname==NULL in
 * den Filehandle "handle".
 * Bei einem Fehler wird FALSE zurueckgeliefert. (Ansonsten TRUE)
 */
{
    int err;
    int i;

    fh = handle;
    if (fname) 	fh = fopen(fname, "w");
    if (fh==NULL) return FALSE;

    err=fprintf(fh, "# FSB V%s, p:HUS, io-Modul\n\nM %d\n",
		fsb_version_string, (mt) ? mt->MaxMatAnz : 30 );

    if (mt && err!=EOF) {
	for (i=0; i<mt->AnzahlMaterialien; i++) {
	    err = fprintf(fh, "m %s %.4e %.4e %.4e %.4e %.4e %.4e %hu %hu %hu %.4e %.4e %.4e\n",
			  mt->MaterialNamen[i], 
			  mt->Dichte[i],
			  mt->Gleitreibungsfaktor[i],
			  mt->Haftreibungsfaktor[i],
			  mt->Stossfaktor[i],
			  mt->Dehnungsfaktor[i],
			  mt->Daempfungsfaktor[i],
			  mt->DefaultR[i],
			  mt->DefaultG[i],
			  mt->DefaultB[i],
			  mt->Durchsichtigkeit[i],
			  mt->Reflexion[i],
			  mt->Rauhigkeit[i]);
	    if (err == EOF) break;
	}
    }

    if (err != EOF) err = fprintf(fh, "\nQ\n");

    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }

    return (err==EOF) ? FALSE : TRUE;
}






TBoolean WriteKonfiguration(TKonfigFBB *k, char *fname, FILE *handle)
/****************************************************************************
 * Schreibt die Konfiguration der FBB ins File "fname" bzw. falls fname==NULL
 * in den Filehandle "handle".
 * Bei einem Fehler wird FALSE zurueckgeliefert. (Ansonsten TRUE)
 */
{
    int err;


    fh = handle;
    if (fname) 	fh = fopen(fname, "w");
    if (fh==NULL) return FALSE;

    err=fprintf(fh, "# FSB V%s, p:HUS, io-Modul\nP\n", fsb_version_string);

    if (k && err!=EOF)
      err = fprintf(fh, "h %.8e %.8e %.8e\n"
		    "e %c %.8e\n"
		    "k %c %.8e\n"
		    "b %.8e %.8e\n"
		    "u %.8e %.8e\n"
		    "g %c %.8e\n"
		    "r %.8e %.8e\n"
		    "j %.8e %.8e\n"
		    "m %.8e\n"
		    "q %.8e\n"
		    "s %c\n"
		    "a %c %.8e %.8e %.8e\nQ\n",
		    k->I_Schritt, k->I_MinSchritt, k->I_MaxSchritt,
		    (k->I_Fehlersteuerung)? 'Y':'N', k->I_Eps,
		    (k->K_Kollisionserkennung)? 'Y':'N', k->K_KollSchritt,
		    k->B_MaxBerGeschw, k->B_MinBerTiefe,
		    k->R_MinGleitGeschw, k->R_MinRollGeschw,
		    (k->G_Zufall)? 'Y':'N', k->G_ZufGravAnteil,
		    k->C_Stange_c, k->C_Stange_d,
		    k->C_Gelenk_c, k->C_Gelenk_d,
		    k->F_MinDehnung,
		    k->K_MaxQuaderQuerTiefe,
		    (k->K_FederStoss)? 'Y':'N',
		    (k->L_Luftwiderstand)? 'Y':'N',
		    k->L_Beiwert_Kugel, k->L_Beiwert_Quader, k->L_Beiwert_Zylinder);
    
    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }

    return (err==EOF) ? FALSE : TRUE;
}



/* liest das naechste Zeichen aus (fh) welches kein Zwischenraumzeichen ist
 * und gibt es als Rueckgabewert zurueck.
 */
static int ReadCode()
{
    int c;

    do {
    	c = fgetc(fh);
	if (c == '#') {
	    do c=fgetc(fh); while (c != EOF && c != '\n' && c != '\r');
	}
    } while (isspace(c));

    return c;
}


/* Liest den naechsten zusammenhaengenden Zeichenstring der Max. Laenge n in
 * das char-array s ein. (aus dem File (fh)).
 * vorangestellte Zwischenraumzeichen werden dabei ueberlesen
 * der String in s wird mit '\0' abgeschlossen!
 */
static TBoolean ReadString(int n, char *s)
{
    int c;

    /* voranstehende Leerzeichen ueberlesen */
    do {
	c = fgetc(fh);
    } while (isspace(c));

    if (c==EOF) {
	*s=0;
	return TRUE;
    }

    /* gelese Zeichen in String kopieren */
    do {
	*(s++) = tolower(c);
	if (--n<=0) break;
	c = fgetc(fh);
    } while (!isspace(c) && c!=EOF);
    *s = 0;
    return FALSE;
}



/* liest n Realzahlen ins Real-Array r.
 * Bei Fehler wird TRUE zurueckgeliefert. Ansonsten FALSE.
 */
static TBoolean ReadReals(int n, TReal *r)
{
    int i;

    for (i=0; i<n; i++) {
	if (fscanf(fh, "%lf", r) != 1) return TRUE;
	r++;
    }
    return FALSE;
}


/* Liest 4 Real-Zahlen in den Quaternion-Vektor ein.
 * bei Fehlerhaftem Lesen oder wenn das Quaternion nicht den Betrag 1.0
 * hat (bzw. quad.Diff. >=0.001 ist!) wird als Fehler TRUE geliefert.
 * Ansonsten wird FALSE zurueckgeliefert.
 */
static TBoolean ReadQuaternion(TQuaternion q)
{
    if (ReadReals(4, q)) {
	fprintf(stderr, "::quaternion incorrect\n");
	return TRUE; 
    }
/* Quaternioncheck entfernt HUS 26-Nov-93
    if (fabs(q[0]*q[0]+q[1]*q[1]+q[2]*q[2]+q[3]*q[3]-1.0)>=0.01) {
	fprintf(stderr, ":: |quaternion| != 1.0\n");
	return TRUE;
    }
*/
    return FALSE;
}



/* Allokiert eine Zustandsstruktur und belegt sie mit Defaultwerten.
 * Fehler (TRUE) wird zurueckgeliefert falls man versucht eine zweite
 * Struktur anzulegen.
 * Falls die Materialtabell==NULL ist so wird die Grundmaterialtabelle
 * eingesetzt.
 * Gravitation wird defaultmaessig auf AUS gestellt!
 * Auch werden die Var. "mname" und "manz" gesetzt
 * FALSE wird zurueckgegeben wenn alles korrekt abgelaufen ist.
 */
static TBoolean InitZ(TZustand **z, TMaterialTabelle *mt)
{
    if (*z  != NULL) {
	fprintf(stderr, "::can read only 1 state a time!!\n");
	return TRUE;
    }

    *z = (TZustand *)malloc(sizeof(TZustand));
    if (*z == NULL) {
	fprintf(stderr, "::not enough memory for state-structure\n");
	return TRUE;
    }

    (*z)->Zeit = 0.0;
    (*z)->GravitationAn = FALSE;
    (*z)->GravitationsVektor[0] = 0.0;
    (*z)->GravitationsVektor[1] = -9.81;
    (*z)->GravitationsVektor[2] = 0.0;
    (*z)->MaterialDaten = mt ? mt : &GrundMaterialien;
    (*z)->Seed = 0;
    (*z)->Koerper = NULL;
    (*z)->Verbindungen = NULL;
    pk = &((*z)->Koerper);
    pv = &((*z)->Verbindungen);
    mname = MaterialNamen((*z)->MaterialDaten);
    manz = AnzahlMaterialien((*z)->MaterialDaten);

    konr = 1;

    return FALSE;
}


/* Allokiert eine Koerperstruktur und haengt sie and die Struktur in pk!.
 * Die in *pk befindliche Koe-Struktur wird mit KoeIniMech() initialisiert!!!!
 * Grundparameter des Koerpers werden aus (fh) gelesen und entsprechen eingetragen
 * Bei Fehlern wird eine Meldung ausgegeben und TRUE geliefert, ansonsten FALSE.
 * Die Var. konr wird benutzt um eine aufsteigende Folge von Koerpernummern
 * zu gewaehrleisten.
 */
static TBoolean InitK(TZustand **z) 
{
    int nr;
    int i;
    char s[42];
    char *s1=NULL, *s2=NULL; /* Initialisierung, um Warning zu vermeiden */

    if (pk==NULL) {
	fprintf(stderr, "::no state declared above????\n");
	return TRUE;
    }
/*
    if (*pk) {
	KoeIniMech(*pk, (*z)->MaterialDaten);
	pk = &((*pk)->Naechster);
    }
*/
    if (*pk) {
	pk = &((*pk)->Naechster);
    }

    *pk = (TKoerper *)malloc(sizeof(TKoerper));
    if (*pk == NULL) {
	fprintf(stderr, "::not enough memory for body-structure\n");
	return TRUE;
    }

    /* Grundbelegung */
    V_SET((*pk)->Position, 0.0, 0.0, 0.0);
    V_SET((*pk)->Geschwindigkeit, 0.0, 0.0, 0.0);
    V_SET((*pk)->Beschleunigung, 0.0, 0.0, 0.0);
    V_SET((*pk)->Drehgeschwindigkeit, 0.0, 0.0, 0.0);
    V_SET((*pk)->Drehbeschleunigung, 0.0, 0.0, 0.0);
    (*pk)->q[0] = 1.0;
    (*pk)->q[1] = 0.0;
    (*pk)->q[2] = 0.0;
    (*pk)->q[3] = 0.0;
    (*pk)->Masse = 0.0;
    (*pk)->Angriffsflaeche = 0.0;
    (*pk)->Kugelhuellenradius = 0.0;
    (*pk)->Material = 0;
    (*pk)->Naechster = NULL;
    (*pk)->Kraefte = NULL;
    (*pk)->Art = NAGEL;
    (*pk)->R = 0;
    (*pk)->G = 0;
    (*pk)->B = 0;
    (*pk)->Durchsichtigkeit = 0.0;
    (*pk)->Reflexion = 0.0;
    (*pk)->Rauhigkeit = 0.0;
    (*pk)->AStatus = 0;
    (*pk)->KoerperId = Id++;
/*
    (*pk)->Masselos = FALSE;
    (*pk)->Gravitationslos = FALSE;
*/
    /* Koerpernr aufsteigend ! 1 2 3 4 ... */
    if (fscanf(fh, "%d", &nr) == 0 || nr != konr) {
	fprintf(stderr, "::wrong body-nr (Knr)\n");
	return TRUE;
    }

    if (ReadString(10, s)) {
	fprintf(stderr, "::can't read body-type\n");
	return TRUE;
    }

    /* Struktur nach Koerperart belegen */
    if (strcmp(s, "sphere")==0) {
	(*pk)->Art = KUGEL;
	if (ReadReals(1, &((*pk)->Form.Kugel.Radius)) || (*pk)->Form.Kugel.Radius <=0) {
	    fprintf(stderr, "::error in sphere-radix!\n");
	    return TRUE;
	}
    }
    else if ((strcmp(s, "cuboid")==0)||(strcmp(s, "box")==0)) {
	(*pk)->Art = QUADER;
	if (ReadReals(3, &((*pk)->Form.Quader.KantenLaenge[0]))) {
	    fprintf(stderr, "::shape-parameter of box incorrect!\n");
	    return TRUE;
	}
        if ((*pk)->Form.Quader.KantenLaenge[0]<=0.0 || 
	    (*pk)->Form.Quader.KantenLaenge[1]<=0.0 || 
	    (*pk)->Form.Quader.KantenLaenge[2]<=0.0) {
	    fprintf(stderr, "::box-vertex<=0?!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "cylinder")==0) {
	(*pk)->Art = ZYLINDER;
	if (ReadReals(1, &((*pk)->Form.Zylinder.Radius)) ||
            (*pk)->Form.Zylinder.Radius <=0) {
	    fprintf(stderr, "::cylinder-radix incorrect!\n");
	    return TRUE;
	}
	if (ReadReals(1, &((*pk)->Form.Zylinder.Hoehe)) ||
            (*pk)->Form.Zylinder.Hoehe <=0) {
	    fprintf(stderr, "::cylinder-height incorrect!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "plain")==0) {
	(*pk)->Art = EBENE;
    }
    else if (strcmp(s, "plane")==0) {
	(*pk)->Art = EBENE;
    }
    else if (strcmp(s, "nail")==0) {
	(*pk)->Art = NAGEL;
    }
    else if (strcmp(s, "mpoint")==0) {
	(*pk)->Art = MPUNKT;
    }
    else if (strcmp(s, "point")==0) {
	(*pk)->Art = PUNKT;
    }
    else if (strcmp(s, "composed")==0) {
	(*pk)->Art = ZUSGESOBJ;
	(*pk)->Form.ZusGesObj.KoerperListe = NULL;
	pek = &((*pk)->Form.ZusGesObj.KoerperListe);
    }
    else {
	fprintf(stderr, "::unknown body-type %s\n", s);
	return TRUE;
    }

    /* Material falls noetig eintragen */
    if ((*pk)->Art == KUGEL || (*pk)->Art == QUADER
        || (*pk)->Art == ZYLINDER || (*pk)->Art == EBENE) {
	if (ReadString(40, s)) {
	    fprintf(stderr, "::error while reading material name\n");
	    return TRUE;
	}
	for (i=0; i<manz; i++) {
	    s1 = s;
	    s2 = mname[i];
	    while (*s1 && *s2 && *s1==tolower(*s2)) {
		s1++;
		s2++;
	    }
	    if (*s1 == *s2) {
		(*pk)->Material = i;
		(*pk)->R = DefaultR((*z)->MaterialDaten, i);
		(*pk)->G = DefaultG((*z)->MaterialDaten, i);
		(*pk)->B = DefaultB((*z)->MaterialDaten, i);
		(*pk)->Durchsichtigkeit = DefaultDurchsichtigkeit((*z)->MaterialDaten, i);
		(*pk)->Reflexion = DefaultReflexion((*z)->MaterialDaten, i);
		(*pk)->Rauhigkeit = DefaultRauhigkeit((*z)->MaterialDaten, i);


		break;
	    }
	}
        if (*s1 != *s2) {
	    fprintf(stderr, "::unknown material %s\n", s);
	    return TRUE;
	}
    }

    konr++;
    
    KoeIniMech(*pk, (*z)->MaterialDaten);

    return FALSE;
}


static TBoolean InitEK(TZustand **z) 
{
    int i;
    char s[42];
    char *s1=NULL, *s2=NULL; /* Initialisierung, um Warning zu vermeiden */

    if (pek==NULL) {
	fprintf(stderr, "::no composed object declared above!\n");
	return TRUE;
    }

    if (*pek) {
	pek = &((*pek)->Naechster);
    }

    *pek = (TKoerper *)malloc(sizeof(TKoerper));
    if (*pek == NULL) {
	fprintf(stderr, "::not enough memory for embedded-body-structure\n");
	return TRUE;
    }

    /* Grundbelegung */
    V_SET((*pek)->Position, 0.0, 0.0, 0.0);
    V_SET((*pek)->Geschwindigkeit, 0.0, 0.0, 0.0);
    V_SET((*pek)->Beschleunigung, 0.0, 0.0, 0.0);
    V_SET((*pek)->Drehgeschwindigkeit, 0.0, 0.0, 0.0);
    V_SET((*pek)->Drehbeschleunigung, 0.0, 0.0, 0.0);
    (*pek)->q[0] = 1.0;
    (*pek)->q[1] = 0.0;
    (*pek)->q[2] = 0.0;
    (*pek)->q[3] = 0.0;
    (*pek)->Masse = 1.0;
    (*pek)->Material = 0;
    (*pek)->Naechster = NULL;
    (*pek)->Kraefte = NULL;
    (*pek)->Art = MPUNKT;
    (*pek)->R = 0;
    (*pek)->G = 0;
    (*pek)->B = 0;
    (*pek)->Durchsichtigkeit = 0.0;
    (*pek)->Reflexion = 0.0;
    (*pek)->Rauhigkeit = 0.0;
    (*pek)->AStatus = 0;
    (*pek)->KoerperId = Id++;

    if (ReadString(10, s)) {
	fprintf(stderr, "::can't read body-type\n");
	return TRUE;
    }

    /* Struktur je nach Koerperart belegen */
    if (strcmp(s, "sphere")==0) {
	(*pek)->Art = KUGEL;
	if (ReadReals(1, &((*pek)->Form.Kugel.Radius)) || (*pek)->Form.Kugel.Radius <=0) {
	    fprintf(stderr, "::error in sphere-radix!\n");
	    return TRUE;
	}
    }
    else if ((strcmp(s, "cuboid")==0)||(strcmp(s, "box")==0)) {
	(*pek)->Art = QUADER;
	if (ReadReals(3, &((*pek)->Form.Quader.KantenLaenge[0]))) {
	    fprintf(stderr, "::shape-parameter of box incorrect!\n");
	    return TRUE;
	}
        if ((*pek)->Form.Quader.KantenLaenge[0]<=0.0 || 
	    (*pek)->Form.Quader.KantenLaenge[1]<=0.0 || 
	    (*pek)->Form.Quader.KantenLaenge[2]<=0.0) {
	    fprintf(stderr, "::box-vertex<=0?!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "cylinder")==0) {
	(*pek)->Art = ZYLINDER;
	if (ReadReals(1, &((*pek)->Form.Zylinder.Radius)) ||
            (*pek)->Form.Zylinder.Radius <=0) {
	    fprintf(stderr, "::cylinder-radix incorrect!\n");
	    return TRUE;
	}
	if (ReadReals(1, &((*pek)->Form.Zylinder.Hoehe)) ||
            (*pek)->Form.Zylinder.Hoehe <=0) {
	    fprintf(stderr, "::cylinder-height incorrect!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "mpoint")==0) {
	(*pek)->Art = MPUNKT;
    }
    else {
	fprintf(stderr, "::unknown or not allowded body-type %s\n", s);
	return TRUE;
    }

    /* Material falls noetig eintragen */
    if ((*pek)->Art != MPUNKT) {
	if (ReadString(40, s)) {
	    fprintf(stderr, "::error while reading material name\n");
	    return TRUE;
	}
	for (i=0; i<manz; i++) {
	    s1 = s;
	    s2 = mname[i];
	    while (*s1 && *s2 && *s1==tolower(*s2)) {
		s1++;
		s2++;
	    }
	    if (*s1 == *s2) {
		(*pek)->Material = i;
		break;
	    }
	}
        if (*s1 != *s2) {
	    fprintf(stderr, "::unknown material %s\n", s);
	    return TRUE;
	}
    }

    KoeIniMech(*pek, (*z)->MaterialDaten);

    return FALSE;
}


static TBoolean ReinitEK(TZustand **z) 
{
    TBewTyp AlterTyp;

    if (pek==NULL) {
	fprintf(stderr, "::no embedded object declared!\n");
	return TRUE;
    }

    if (pk==NULL) {
	fprintf(stderr, "::internal error - where is the root-object(composed)?\n");
	return TRUE;
    }

    AlterTyp = (*pk)->BewTyp;        /* sonst wird BewTyp auf FREI|KOLL gesetzt */
    KoeIniMech(*pk, (*z)->MaterialDaten);
    (*pk)->BewTyp = AlterTyp;

    return FALSE;
}


/* Allokiert eine Kraftstruktur und haengt sie zu dem entsprechendem Koerper.
 * Dann werden entsprechen der obigen "Grammatik" dir Werte eingelesen und
 * eingetragen.
 * Bei einem Fehler wird TRUE gelesen, wenn alles ok ist wird FALSE zurueckgegeben.
 */
static TBoolean ReadKraft(TZustand **z) 
{
    TKraft *f;
    TKoerper *k;
    TVektor  v;
    TReal h;
    int knr;
    char s[12];
    char c;


    if (*z==NULL || pk==NULL) {
	fprintf(stderr, "::no state declared above????\n");
	return TRUE;
    }

    f = (TKraft *)malloc(sizeof(TKraft));
    if (f == NULL) {
	fprintf(stderr, "::can't allocate memory for force-structure\n");
	return TRUE;
    }
 
    /* Grundinitialisierung
     */
    f->Naechste = NULL;
    V_SET(f->Angriffspunkt, 0.0, 0.0, 0.0);

    /* Wirkungszeitbereich lesen
     */
    if (ReadReals(1, &(f->StartZeit))) {
	fprintf(stderr, "::no begin-time\n");
	return TRUE;
    }
    if (ReadReals(1, &(f->EndZeit))) {
	fprintf(stderr, "::no end-time\n");
	return TRUE;
    }

    while (isspace(c=fgetc(fh)))
	;

    if (c != 'K' || fscanf(fh, "%d", &knr)== 0) {
	fprintf(stderr, "::no body-number (Knr)\n");
	return TRUE;
    }

    for (k=(*z)->Koerper; k; k=k->Naechster)   /* Kraftstruktur zur Koerpernr suchen */
	if (--knr == 0) break;

    if (k && knr==0) {               /* Kraftstruktur zum Koerper haengen */
	f->Naechste = k->Kraefte;
	k->Kraefte = f;
    }
    else {
	fprintf(stderr, "::incorrect body-nr (Knr)\n");
	return TRUE;
    }

    /* Angriffsort lesen (optional)
     */    
    if (ReadReals(3, &(v[0]))==FALSE) {
	V_LET(f->Angriffspunkt, v);
    }

    /* Krafttyp lesen
     */
    if (ReadString(10, s)) {
	fprintf(stderr, "::incorrect force-type\n");
	return TRUE;
    }

    /* Parameter des entsprechenden Krafttypes lesen */
    if (strcmp(s, "inertial")==0) {
	f->Typ = RAUMFEST;
	if (ReadReals(3, &(f->KD.Kraft[0]))) {
	    fprintf(stderr, "::inertial force-vector expected!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "local")==0) {
	f->Typ = KOERPERFEST;
	if (ReadReals(3, &(f->KD.Kraft[0]))) {
	    fprintf(stderr, "::local force-vector expected!\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "guided")==0) {
	f->Typ = GERICHTET;

	if (ReadReals(1, &(f->KD.GerKraft.Kraftbetrag))) {
	    fprintf(stderr, "::no guided-force to read\n");
	    return TRUE;
	}

	while (isspace(c=fgetc(fh)))
	    ;

	if (c != 'K' || fscanf(fh, "%d", &knr)==0) {
	    fprintf(stderr, "::no target-body (Knr?)\n");
	    return TRUE;
	}

	for (k=(*z)->Koerper; k; k=k->Naechster)
	    if (--knr == 0) break;

	if (k==NULL || knr!=0) {
	    fprintf(stderr, "::target-body not in body-list (wrong Knr!)\n");
	    return TRUE;
	}

        f->KD.GerKraft.ZielKoerper = k;

	V_SET(v, 0.0, 0.0, 0.0);
	
	if (ReadReals(1, &h)==FALSE) {
	    v[0] = h;
	    if (ReadReals(2, &(v[1]))) {
		fprintf(stderr, "::target-point-vector incorrect\n");
		return TRUE;
	    }
	}
	V_LET(f->KD.GerKraft.Zielpunkt, v);

    }
    else {
	fprintf(stderr, "::unknown force-type\n");
	return TRUE;
    }

    return FALSE;
}



/* Allokiert eine Verbindungsstruktur und haengt sie in pv ein (als Naechstes).
 * Die Koerpernummern und Verbindungsart+Parameter werden von (fh) gelesen 
 * und in die Struktur eingetragen.
 * Bei Fehler wird eine Meldung ausgeben und TRUE geliefert. ansonsten FALSE.
 *
 */
static TBoolean InitV(TZustand **z)
{
    int nr, i;
    TKoerper *k;
    char s[40];

    if (pv == NULL) {
	fprintf(stderr, "::double state-declaration ????\n");
	return TRUE;
    }

    if (*pv) pv = &((*pv)->Naechste);

    *pv = (TVerbindung *)malloc(sizeof(TVerbindung));
    if (*pv == NULL) {
	fprintf(stderr, "::can't allocate memory for link-structure\n");
	return TRUE;
    }

    fscanf(fh, "%d", &nr);

    (*pv)->Naechste = NULL;
    (*pv)->Art = DAEMPFER;
    (*pv)->AStatus = 0;
    (*pv)->VerParameter.Daempfer.Daempfungskonstante = 1000.0;
    V_SET((*pv)->VPunkt1, 0.0, 0.0, 0.0);
    V_SET((*pv)->VPunkt2, 0.0, 0.0, 0.0);

    /* Nummern von Koerper1 und Koerper2 in Koerperstruktur suchen */
    if (ReadCode()!='K' || fscanf(fh, "%d", &nr) !=1) {
	fprintf(stderr, "::link to which body? (wrong Knr)?\n");
	return TRUE;
    }
    i = nr;
    for (k=(*z)->Koerper; k; k=k->Naechster)
	if (--i==0) break;
    (*pv)->Koerper1 = k;
    if (k==NULL) {
	fprintf(stderr, "::can't find body %d of link\n", nr);
	return TRUE;
    }

    if (ReadCode()!='K' || fscanf(fh, "%d", &nr) !=1) {
	fprintf(stderr, "::no 2nd body of link specificated!\n");
	return TRUE;
    }
    i = nr;
    for (k=(*z)->Koerper; k; k=k->Naechster)
	if (--i==0) break;
    (*pv)->Koerper2 = k;
    if (k==NULL) {
	fprintf(stderr, "::can't find body %d of link!\n", nr);
	return TRUE;
    }

    if (ReadString(10, s)) {
	fprintf(stderr, "::where is the link-type (damper, rod,..)???\n");
	return TRUE;
    }

    if (strcmp(s, "spring")==0) {
	(*pv)->Art = FEDER;
	if (ReadReals(1, &((*pv)->VerParameter.Feder.Ruhelaenge))) {
	    fprintf(stderr, "::spring-still-length incorrect\n");
	    return TRUE;
	}
	if (ReadReals(1, &((*pv)->VerParameter.Feder.Federkonstante))) {
	    fprintf(stderr, "::spring-constant incorrect\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "damper")==0) {
	(*pv)->Art = DAEMPFER;
	if (ReadReals(1, &((*pv)->VerParameter.Daempfer.Daempfungskonstante))) {
	    fprintf(stderr, "::damper-constant incorrect\n");
	    return TRUE;
	}
    }
    else if (strcmp(s, "joint")==0) {
	(*pv)->Art = GELENK;
    }
    else if (strcmp(s, "rod")==0) {
	(*pv)->Art = STANGE;
	if (ReadReals(1, &((*pv)->VerParameter.Stange.Stangenlaenge))) {
	    fprintf(stderr, "::rod-length incorrect\n");
	    return TRUE;
	}
    }
    else {
	fprintf(stderr, "::unknown link-type %s\n", s);
	return TRUE;
    }
      
    return FALSE;
}


TBoolean ReadBewTyp(TKoerper *k)
{
    char s[10];
    int i;

    if (ReadString(5, s)==TRUE || s[0] != '-') {
	fprintf(stderr, "::wrong movement-type %s\n", s);
    }
    
    for (i=strlen(s)-1; i>=1; i--)
      switch(s[i]) {
	case 'm':
	  SetzeKoerperMasselos(k, TRUE);
	  break;
	case 'g':
	  SetzeKoerperGravlos(k, TRUE);
	  break;
	case 'k':
	  SetzeKoerperKollmoeglich(k, FALSE);
	  break;
	default:
	  fprintf(stderr, "::unknown movement-parameter %c\n", s[i]);
	  return TRUE;
      }


    return FALSE;
}


/* Liest eine Zustand aus dem File mit Name "fname" bzw. wenn fname==NULL aus
 * File (handle) und gibt die Zustandsstruktur in z zurueck!
 * Falls keine Materialtabelle(mt==NULL) angegeben wird so wird die
 * GrundMaterialtabelle benutzt.
 * Die Routine liefert TRUE wenn alles in Ordnung war und ansonsten FALSE.
 * (und eine Fehlermeldung im stderr-Kanal.
 * Falls ein Fehler auftritt ist die Zurueckgelieferte Struktur nicht gueltig
 * und sollte nicht weiterbenutzt werden.
 * Also am besten wieder freigegeben! (Verkettung ist in Ordnung!!!)
 */
TBoolean ReadZustand(TZustand **z, TMaterialTabelle *mt, char *fname, FILE *handle)
{
    int c;
    int mode = UNKNOWN;

    *z = NULL;    /* Zustand */
    pk = NULL;    /* Koerperliste */
    pek = NULL;   /* Liste von eingebetteten Koerpern (in ZusGesObj) */
    pv = NULL;    /* Verbindungs-Zeiger */
    fh = handle; 
    Id = 1;

    if (fname) fh = fopen(fname, "r");
    if (fh==NULL) return FALSE;


    do {
	c = ReadCode(fh);
	if (ferror(fh)) {
	    perror("::error while reading");
	    goto fehler;
	}

	switch(c) {
	case 'Z':
	    if (InitZ(z, mt)) goto fehler;
	    mode = READZ;
	    break;
	case 'K':
	    if (InitK(z)) goto fehler;
	    mode = READK;
	    break;
	case 'V':
	    if (InitV(z)) goto fehler;
	    mode = READV;
	    break;
	case 'F':
	    if (ReadKraft(z)) goto fehler;
	    break;
	case 'E':
	    if (InitEK(z)) goto fehler;
	    mode = READE;
	    break;
	case 'X':
	    if (ReinitEK(z)) goto fehler;
	    break;
	case EOF:
	    fprintf(stderr, "::unexpected end of file");
	    goto fehler;
	    break;
        case 'Q':
	    break;
	default:
	    switch(mode) {
	    case READZ:
		switch(c) {
		case 't':
		    ReadReals(1, &((*z)->Zeit));
		    if ((*z)->Zeit<0.0) {
			fprintf(stderr, "::Fehler -Startzeit<0\n");
			goto fehler;
		    }
		    break;
	        case 'g':
		    READVEKTOR((*z)->GravitationsVektor, "gravity-parameter incorrect");
		    (*z)->GravitationAn = TRUE;
		    break;
	        case 's':
		    if (fscanf(fh, "%d", &((*z)->Seed)) != 1) {
		        fprintf(stderr, "::error - no seed-value\n");
		        goto fehler;
		    }
		    break;

	        default:
		    fprintf(stderr, "::unknown scan-code %c\n", c);
		    goto fehler;
		}
		break;

	    case READK:
	        switch(c) {
	        case 'm':
		    if (ReadReals(1, &((*pk)->Masse))) {
			fprintf(stderr, "::mass incorrect\n");
			goto fehler;
	    	    }
	            if ((*pk)->Masse < 0.0) {
		        fprintf(stderr, "::mass %f <=0 !\n", (*pk)->Masse);
			goto fehler;
		    }
	            SetzeKoerperMasse((*pk), (*pk)->Masse);
		    break;
	        case 'p':
		    READVEKTOR((*pk)->Position, "position-vector incorrect");
		    break;
		case 'v':
		    READVEKTOR((*pk)->Geschwindigkeit, "velocity-vector incorrect");
		    break;
	        case 'a':
		    READVEKTOR((*pk)->Beschleunigung, "acceleration-vector incorrect");
		    break;
	        case 'q':
		    if (ReadQuaternion((*pk)->q)) goto fehler;
		    break;
	        case 'w': 
		    READVEKTOR((*pk)->Drehgeschwindigkeit, "angular-velocity-vector incorrect");
		    break;
	        case 'u':
		    READVEKTOR((*pk)->Drehbeschleunigung, "angular-accelaration-vector incorrect");
		    break;
	        case 'b':
                    if (ReadBewTyp((*pk))) goto fehler;
	      	    break;
	        case 'c':
	            if( fscanf( fh,"%hu%hu%hu", &(*pk)->R, &(*pk)->G, &(*pk)->B ) != 3 ) {
		        fprintf(stderr, "::3 color-values expected!!!\n");
		        goto fehler;
		    }
	            break;
	        case 'n':
	            if( fscanf( fh,"%lf%lf%lf",
			       &(*pk)->Durchsichtigkeit, &(*pk)->Reflexion, 
                               &(*pk)->Rauhigkeit ) != 3 ) {
		        fprintf(stderr, "::3 mat-values expected!!!\n");
		        goto fehler;
		    }
	            break;
	        case 'i':
	            if( fscanf( fh,"%ld", &(*pk)->KoerperId)!=1 ) {
		        fprintf(stderr, "::body-id expected\n");
		        goto fehler;
		    }
 		    if ((*pk)->KoerperId >Id) Id = (*pk)->KoerperId;
	            break;
	        default:
		     fprintf(stderr, "::unknown code %c during READK-mode\n", c);
		     goto fehler;
		}
		break;

	    case READV:
		switch(c) {
	        case 'a':
                    READVEKTOR((*pv)->VPunkt1, "link-point#1(vector) incorrect"); break;
	        case 'b':
		    READVEKTOR((*pv)->VPunkt2, "link-point#2(vector) incorrect"); break;
	        default:
		    fprintf(stderr, "::unknown code %c during READV-mode\n", c);
                    goto fehler;
		}
		break;

	    case READE:
	        switch(c) {
	        case 'm':
		    if (ReadReals(1, &((*pek)->Masse))) {
			fprintf(stderr, "::mass incorrect\n");
			goto fehler;
	    	    }
	            if ((*pek)->Masse < 0.0) {
		        fprintf(stderr, "::mass %f <=0 !\n", (*pek)->Masse);
			goto fehler;
		    }
	            SetzeKoerperMasse((*pek), (*pek)->Masse);
		    break;
	        case 'p':
		    READVEKTOR((*pek)->Position, "position-vector incorrect");
		    break;
	        case 'q':
		    if (ReadQuaternion((*pek)->q)) goto fehler;
		    break;
	        case 'b':
                    if (ReadBewTyp((*pek))) goto fehler;
	      	    break;
	        case 'c':
	            if( fscanf( fh,"%hu%hu%hu", 
                               &(*pek)->R, &(*pek)->G, &(*pek)->B ) != 3 ) {
		        fprintf(stderr, "::3 color-values expected!!!\n");
		        goto fehler;
		    }
	            break;
	        case 'n':
	            if( fscanf( fh,"%lf%lf%lf",
			       &(*pk)->Durchsichtigkeit, &(*pk)->Reflexion, 
                               &(*pk)->Rauhigkeit ) != 3 ) {
		        fprintf(stderr, "::3 mat-values expected!!!\n");
		        goto fehler;
		    }
	            break;
	        case 'i':
	            if( fscanf( fh,"%ld", &(*pek)->KoerperId)!=1 ) {
		        fprintf(stderr, "::body-id expected\n");
		        goto fehler;
		    }
 		    if ((*pek)->KoerperId >Id) Id = (*pek)->KoerperId;
	            break;
	        default:
		     fprintf(stderr, "::unknown code %c during READE-mode\n", c);
		     goto fehler;
	        }
		break;
		
	    default:
		fprintf(stderr, "::unknown code %c during read-mode\n", c);
            	goto fehler;
	    }
	}
    } while (c != 'Q');


/*
cleanup:
    if (*pk) KoeIniMech(*pk, (*z)->MaterialDaten);
*/
    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }
    return TRUE;


fehler:
    if (fname) fclose(fh);
    return FALSE;
}


TBoolean ReadMaterialien(TMaterialTabelle **mt, char *fname, FILE *handle)
{
    int c;
    int max;
    char name[42];
    TReal dichte, haft, gleit, stoss, dehn, daempf;
    unsigned short defr, defg, defb;
    TReal durch, reflex, rauh;


    if (fname) fh = fopen(fname, "r"); else fh = handle;
    if (fh==NULL) return FALSE;

    c = ReadCode(fh);
    if (ferror(fh)) {
	perror("::read-error");
	goto fehler;
    }
    if (c != 'M') {
	    fprintf(stderr, "::wrong signature (M expected!)\n");
	    goto fehler;
    }
    if (fscanf(fh, "%d", &max) != 1 || max<=0) {
	    fprintf(stderr, "::no or incorrect maximum number of material-entries\n");
	    max = 32;
    }

    LoescheMatTab(*mt);

    *mt = AllokiereMatTab(max);

    do {
	c = ReadCode(fh);
	if (ferror(fh)) {
	    perror("::read-error");
	    goto fehler;
	}

	switch(c) {
	case 'm':
	    if (fscanf(fh, "%40s%lf%lf%lf%lf%lf%lf", 
                name, &dichte, &gleit, &haft, &stoss, &dehn, &daempf) != 7) {
		    fprintf(stderr, "::material-parameters incorrect!\n");
	    }
	    if (fscanf(fh, "%hu", &defr)!=1) {
		if (ErgaenzeMatTab(*mt, name, dichte, gleit, haft, stoss, dehn, daempf)) {
		    fprintf(stderr, "::material-tabular overflow!\n");
		}
	    }
	    else {
		if (fscanf(fh, "%hu%hu%lf%lf%lf",
			   &defg, &defb, &durch, &reflex, &rauh)
		    != 5) {
		    fprintf(stderr, "::extended material-parameters incorrect!\n");
		}
		if (ErgaenzeMatTab2(*mt, name, dichte, gleit, haft, stoss,
				    dehn, daempf, defr, defg, defb,
				    durch, reflex, rauh)) {
		    fprintf(stderr, "::material-tabular overflow!\n");
		}

	    }
	    break;

	case EOF:
	    fprintf(stderr, "::unexpected end of file\n");
	    goto fehler;
	    break;

        case 'Q':
	    break;

	default:
	    fprintf(stderr, "::unknown scan-code %c\n", c);
	    goto fehler;
	}
    } while (c != 'Q');

/*
cleanup:
*/
    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }
    return TRUE;


fehler:
    if (fname) fclose(fh);
    return FALSE;
}




TBoolean ReadKonfiguration(TKonfigFBB *k, char *fname, FILE *handle)
{
    int c;
    int i;



    if (fname) fh = fopen(fname, "r"); else fh = handle;
    if (fh==NULL) return FALSE;

    c = ReadCode(fh);
    if (ferror(fh)) {
	perror("::read-error");
	goto fehler;
    }
    if (c != 'P') {
	    fprintf(stderr, "::wrong signature (M expected!)\n");
	    goto fehler;
    }

    do {
	c = ReadCode(fh);
	if (ferror(fh)) {
	    perror("::read-error");
	    goto fehler;
	}

	switch(c) {
	  case 'h':
	    i = fscanf(fh, "%lf%lf%lf", &(k->I_Schritt), 
		       &(k->I_MinSchritt), &(k->I_MaxSchritt));
	    if (i<3) k->I_MaxSchritt = k->I_Schritt;
	    if (i<2) k->I_MinSchritt = k->I_Schritt;
	    if (i<=0 || i>3) {
		fprintf(stderr, "::stepsizes expected\n");
		goto fehler;
	    }
	    break;

	  case 'e':
	    k->I_Fehlersteuerung = (tolower(ReadCode(fh))=='n')?FALSE:TRUE;
	    if (fscanf(fh, "%lf", &(k->I_Eps))!=1) {
		fprintf(stderr, "::Eps expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'k':
	    k->K_Kollisionserkennung = (tolower(ReadCode(fh))=='n')?FALSE:TRUE;
	    if (fscanf(fh, "%lf", &(k->K_KollSchritt))!=1) {
		fprintf(stderr, "::(Y/N) <CollStep:Real> expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'b':
	    if (fscanf(fh, "%lf%lf", 
		       &(k->B_MaxBerGeschw), &(k->B_MinBerTiefe))!=2) {
		fprintf(stderr, "::MaxRestVel and MinRestDepth expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'u':
	    if (fscanf(fh, "%lf%lf", 
		       &(k->R_MinGleitGeschw), &(k->R_MinRollGeschw))!=2) {
		fprintf(stderr, "::Friction Velocities expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'g':
	    k->G_Zufall = (tolower(ReadCode(fh))=='n')?FALSE:TRUE;
	    if (fscanf(fh, "%lf", &(k->G_ZufGravAnteil))!=1) {
		fprintf(stderr, "::(Y/N) <RandomGrav%%:Real> expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'r':
	    if (fscanf(fh, "%lf%lf", 
		       &(k->C_Stange_c), &(k->C_Stange_d))!=2) {
		fprintf(stderr, "::Rod parameters (spring c, damper d) expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'j':
	    if (fscanf(fh, "%lf%lf", 
		       &(k->C_Gelenk_c), &(k->C_Gelenk_d))!=2) {
		fprintf(stderr, "::Joint parameters (spring c, damper d) expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'm':
	    if (fscanf(fh, "%lf", &(k->F_MinDehnung))!=1) {
		fprintf(stderr, "::minimal spring-offset expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 'q':
	    if (fscanf(fh, "%lf", &(k->K_MaxQuaderQuerTiefe))!=1) {
		fprintf(stderr, "::minimal box depth expected\n");
		goto fehler;
	    }
	    break;
	    
	  case 's':
	    k->K_FederStoss = (tolower(ReadCode(fh))=='n')?FALSE:TRUE;
	    break;
	    
	  case 'a':
	    k->L_Luftwiderstand = (tolower(ReadCode(fh))=='n')?FALSE:TRUE;
	    if (fscanf(fh, "%lf%lf%lf", 
		       &(k->L_Beiwert_Kugel), &(k->L_Beiwert_Quader),
		       &(k->L_Beiwert_Zylinder))!=3) {
		fprintf(stderr, "::(Y/N) and cw-values for sphere,box and cylinder expected\n");
		goto fehler;
	    }
	    break;
	    
	  case EOF:
	    fprintf(stderr, "::unexpected end of file\n");
	    goto fehler;
	    break;
	    
	  case 'Q':
	    break;
	    
	  default:
	    fprintf(stderr, "::unknown scan-code %c\n", c);
	    goto fehler;
	}
    } while (c != 'Q');

/*
cleanup:
*/
    if (fname) {
	if (fclose(fh)==EOF) return FALSE;
    }
    return TRUE;


fehler:
    if (fname) fclose(fh);
    return FALSE;
}
