/*    
**  ShellyLibV2.0 - the ShellShapeGenerator 
**
**  Copyright (C) 1996 Randolf Schultz (rschultz@informatik.uni-rostock.de)
**
**  This software is shareware!
**  Read the file "License" for further information.
*/

#include <math.h>
#include <stdlib.h>

#include "shellyl.h"


/********************************************/
/*SLFreeGC:                                 */
/********************************************/
void
SLFreeGC (struct SLarguments *arg)
{
  struct SLgcpoint *hp1, *hp2 = NULL;

  if(arg == NULL)
   return;
  else
    hp1 = arg->GCurve;

  while(hp1)
    {
      hp2 = hp1->next;
      free(hp1);
      hp1 = hp2;
    }
  
  return;
}				/* SLFreeGC */


/********************************************/
/*SLFreePindex:                             */
/********************************************/
void
SLFreePindex (struct SLpindex *hp1)
{
  struct SLpindex *hp2;

  if(hp1==NULL)
    return;

  do
    {
      hp2 = hp1->next;
      free(hp1);
      hp1 = hp2;
    }
  while(hp1);

  return;
}				/* SLFreePindex */


/********************************************/
/*SLFreeLindex:                             */
/********************************************/
void
SLFreeLindex (struct SLlindex *hp1)
{
  struct SLlindex *hp2;

  if(hp1==NULL)
    return;

  while(hp1)
    {
      hp2 = hp1->next;
      free(hp1);
      hp1 = hp2;
    }


  return;
}				/* SLFreeLindex */


/********************************************/
/*SLFreeTindex:                             */
/********************************************/
void
SLFreeTindex (struct SLtindex *hp)
{
  struct SLtindex *hp1;

  if(hp==NULL)
    return;

  while(hp)
    {
      hp1 = hp->next;
      free(hp);
      hp = hp1;
    };


  return;
}				/* SLFreeTindex */


/********************************************/
/* register_point:                          */
/********************************************/
int
register_point(struct SLpindex **ganchor, struct SLpindex **end, 
	       struct SLpoint *p1, unsigned int index)
{
  struct SLpindex *hp;

  if ((hp = calloc (1, sizeof (struct SLpindex))) != NULL)
    {
      hp->point = p1;
      hp->pid = index;

      if(*end)
	(*end)->next = hp;
      *end = hp;

      if(*ganchor == NULL) /* very first point to register? */
	*ganchor = hp;
    }
  else
    return(5);

  return(0);
}				/* register_point */


/********************************************/
/* is_point_used:                           */
/********************************************/
unsigned int
is_point_used (struct SLpindex *anchor, struct SLpindex *hp, struct SLpoint *p)
{

  if(hp == NULL)
    {
      if(anchor == NULL)
	return (0);
      else
	hp = anchor;
    }

  do
    {
      if (hp->point->x == p->x)
	if (hp->point->y == p->y)
	  if (hp->point->z == p->z)
	    return (hp->pid);

      hp = hp->next;
    }
  while(hp);

  return (0);
}                              /* is_point_used */


/********************************************/
/* register_line:                           */
/********************************************/
int
register_line (struct SLlindex **ganchor, struct SLlindex **end,
	       unsigned int lid, unsigned int pid1, unsigned int pid2)
{
  struct SLlindex *hp;

  if ((hp = calloc (1, sizeof (struct SLlindex))) != NULL)
    {
      hp->pid1 = pid1;
      hp->pid2 = pid2;
      hp->lid = lid;

      if(*end)
	(*end)->next = hp;
      *end = hp;

      if(*ganchor == NULL) /* very first line to register? */
	*ganchor = hp; 
    }
  else
    return(5);

  return(0);

}				/* register_line */


/********************************************/
/* is_line_used:                            */
/********************************************/
unsigned int
is_line_used (struct SLlindex *anchor, struct SLlindex *hp, unsigned int pid1,
 unsigned int pid2)
{

  if(hp == NULL)
    {
      if(anchor == NULL)
	return(0);
      else
	hp = anchor;
    }

  do
    {
      if (hp->pid1 == pid1)
	if (hp->pid2 == pid2)
	  return (hp->lid);
      /* reverse orientation */
      if (hp->pid2 == pid1)
	if (hp->pid1 == pid2)
	  return (hp->lid);

      hp = hp->next;
    }
  while(hp);

  return (0);
}				/* is_line_used */


/********************************************/
/*SLFreeTriangles:                          */
/********************************************/
void
SLFreeTriangles (struct SLtriangle *hp)
{
  struct SLtriangle *hp1;
 
  if(hp == NULL)
    return;

  while(hp)
    {
      hp1 = hp->next;
      free(hp);
      hp = hp1;
    };

  return;
} /* SLFreeTriangles */


/********************************************/
/* SLTriangulate:                           */
/********************************************/
int
SLTriangulate(struct SLpoint *list1, struct SLpoint *list2, 
	      struct SLtriangle **result)
{
  struct SLpoint *p11, *p12, *p21, *p22;
  struct SLtriangle *tp, *tp2 = NULL, *anchor = NULL;
  int end = 1;

  if ((list1 == NULL) || (list2 == NULL))
    return(10);

  p11 = list1;
  p21 = list2;

  while (end)
    {
      p12 = p11->next;
      p22 = p21->next;

      if ((p11->next == NULL) && (p21->next == NULL))
	end = 0;
      else if (p11->next == NULL)
	{
	  tp = calloc (1, sizeof (struct SLtriangle));
	  if (tp != NULL)
	    {
	      tp->p1 = p21;
	      tp->p2 = p22;
	      tp->p3 = p11;
	      if (anchor == NULL)
		anchor = tp;
	      else
		tp2->next = tp;
	      tp2 = tp;
	    }
	  else
	    return(5);

	  p21 = p22;
	}
      else if (p21->next == NULL)
	{
	  tp = calloc (1, sizeof (struct SLtriangle));
	  if (tp != NULL)
	    {
	      tp->p1 = p11;
	      tp->p2 = p21;
	      tp->p3 = p12;
	      if (anchor == NULL)
		anchor = tp;
	      else
		tp2->next = tp;
	      tp2 = tp;
	    }
	  else
	    return(5);

	  p11 = p12;
	}
      else if (p11->s == p21->s)
	{
	  if (fabs (p11->s - p12->s) < fabs (p21->s - p22->s))
	    {
	      tp = calloc (1, sizeof (struct SLtriangle));
	      if (tp != NULL)
		{
		  tp->p1 = p11;
		  tp->p2 = p21;
		  tp->p3 = p12;
		  if (anchor == NULL)
		    anchor = tp;
		  else
		    tp2->next = tp;
		  tp2 = tp;
		}
	      else
		return(5);
	      p11 = p12;
	    }
	  else
	    {
	      tp = calloc (1, sizeof (struct SLtriangle));
	      if (tp != NULL)
		{
		  tp->p1 = p21;
		  tp->p2 = p22;
		  tp->p3 = p11;
		  if (anchor == NULL)
		    anchor = tp;
		  else
		    tp2->next = tp;
		  tp2 = tp;
		}
	      else
		return(5);

	      p21 = p22;
	    }
	}
      else if (p11->s < p21->s)
	{
	  tp = calloc (1, sizeof (struct SLtriangle));
	  if (tp != NULL)
	    {
	      tp->p1 = p11;
	      tp->p2 = p21;
	      tp->p3 = p12;
	      if (anchor == NULL)
		anchor = tp;
	      else
		tp2->next = tp;
	      tp2 = tp;
	    }
	  else
	    return(5);

	  p11 = p12;

	}
      else if (p11->s > p21->s)
	{
	  tp = calloc (1, sizeof (struct SLtriangle));
	  if (tp != NULL)
	    {
	      tp->p1 = p21;
	      tp->p2 = p22;
	      tp->p3 = p11;
	      if (anchor == NULL)
		anchor = tp;
	      else
		tp2->next = tp;
	      tp2 = tp;
	    }
	  else
	    return(5);

	  p21 = p22;

	}			/* big if-cascade ;) */

    }				/* while */

  *result = anchor;

  return(0);
}				/* SLTriangulate */


/********************************************/
/* SLCreateIndex:                           */
/* create index structure for a given shell */
/********************************************/
int
SLCreateIndex(struct SLarguments *arg, struct SLshell *shell, struct 
	      SLtindex **tind, struct SLlindex **lind, struct SLpindex **pind)
{
  struct SLtriangle *trp;
  struct SLpindex *pp = NULL, *plocal = NULL;
  struct SLlindex *lp = NULL, *llocal = NULL;
  struct SLtindex *tip = NULL, *tip2;
  unsigned int pcount = 0, lcount = 0, tcount = 0;
  unsigned int p1id, p2id, p3id, l1id, l2id, l3id;

  if(shell == NULL)
    return(10);

  while(shell->next)
    {

      if(SLTriangulate(shell->line, shell->next->line, &trp) == 0)
	{

	  while(trp)
	    {

	      /* new triangle-index node */
	      if((tip2 = calloc(1, sizeof(struct SLtindex))) == NULL)
		return(5);

	      tcount++;

	      if (tip == NULL)
		*tind = tip2;
	      else
		tip->next = tip2;
	      
	      tip = tip2;

	      /* check if points are used in other triangles */

	      /* p1 */
	      if ((p1id = is_point_used(*pind, plocal, trp->p1)) == 0)
		{
		  p1id = ++pcount;
		  if(register_point(pind, &pp, trp->p1, pcount) != 0)
		    return(5);
		}
	      /* p2 */
	      if ((p2id = is_point_used(*pind, plocal, trp->p2)) == 0)
		{
		  p2id = ++pcount;
		  if(register_point(pind, &pp, trp->p2, pcount) != 0)
		    return(5);
		}
	      /* p3 */
	      if ((p3id = is_point_used(*pind, plocal, trp->p3)) == 0)
		{
		  p3id = ++pcount;
		  if(register_point(pind, &pp, trp->p3, pcount) != 0)
		    return(5);
		}

	      /* check if lines are used in other triangles */

	      /* l1 */
	      if ((l1id = is_line_used(*lind, llocal, p1id, p2id)) == 0)
		{
		  l1id = ++lcount;
		  if(register_line(lind, &lp, lcount, p1id, p2id) != 0)
		    return(5);
		}
	      /* l2 */
	      if ((l2id = is_line_used(*lind, llocal, p2id, p3id)) == 0)
		{
		  l2id = ++lcount;
		  if(register_line(lind, &lp, lcount, p2id, p3id) != 0)
		    return(5);
		}
	      /* l3 */
	      if ((l3id = is_line_used(*lind, llocal, p3id, p1id)) == 0)
		{
		  l3id = ++lcount;
		  if(register_line(lind, &lp, lcount, p3id, p1id) != 0)
		    return(5);
		}

	      tip->lid1 = l1id;
	      tip->lid2 = l2id;
	      tip->lid3 = l3id;

	      tip->pid1 = p1id;
	      tip->pid2 = p2id;
	      tip->pid3 = p3id;

	      trp = trp->next;
	    }; /* while */

	} /* triangulate */
      else
	return(5);

      SLFreeTriangles (trp);
      
      if(plocal == NULL)
	plocal = *pind;
      else
	plocal = pp; 

      if(llocal == NULL)
	llocal = *lind;
      else
	llocal = lp;

      shell = shell->next;
    }; /* while shell->next */

  arg->points = pcount;
  arg->lines = lcount;
  arg->triangles = tcount;

  return(0);
} /* SLCreateIndex */


/********************************************/
/*cot:                                      */
/********************************************/
double
cot (double in)
{
  if (sin (in) == 0)
    return (0);
  else
    return (cos (in) / sin (in));
}


/********************************************/
/*round:                                    */
/********************************************/
double
round (double in)
{
  if (in - floor (in) < 0.5)
    return (floor (in));
  else
    return (floor (in) + 1);
}


/********************************************/
/*SLFreeLine:                               */
/********************************************/
void
SLFreeLine (struct SLpoint *hp)
{
  struct SLpoint *hp1;
 
  while(hp)
    {
      hp1 = hp->next;
      free(hp);
      hp = hp1;
    }
  return;
} /* SLFreeLine */


/********************************************/
/*SLFreeShell:                              */
/********************************************/
void
SLFreeShell (struct SLshell *hp)
{
  struct SLshell *hp1;
 
  while(hp)
    {
      hp1 = hp->next;
      SLFreeLine(hp->line);
      free(hp);
      hp = hp1;
    }

  return;
} /* SLFreeShell */


/********************************************/
/*calcg:                                    */
/********************************************/
double
calcg (double O, double N)
{

  if (N == 0.0)
    return (0);
  else
    return ((pi * 2.0) / N * (O * N / (pi * 2.0) - round(O * N / (pi * 2.0))));

}				/* calcg */


/********************************************/
/*calck:                                    */
/********************************************/
double
calck (struct SLarguments *arg, double S, double O)
{
  double N, N2, N3, L, L2, L3, W1, W2, W12, W13, W22, W23, P, P2, P3, Off2,
    Off3;
  double k = 0.0;
  double g = 0.0;

  N = arg->N;
  W1 = arg->W1;
  W2 = arg->W2;
  P = arg->P;
  L = arg->L;
  N2 = arg->N2;
  W12 = arg->W12;
  W22 = arg->W22;
  P2 = arg->P2;
  L2 = arg->L2;

  N3 = arg->N3;
  W13 = arg->W13;
  W23 = arg->W23;
  P3 = arg->P3;
  L3 = arg->L3;

  Off2 = arg->Off2;
  Off3 = arg->Off3;

  if ((N != 0.0) && (O >= arg->Nstart))
    {
      g = calcg (O, N);
      
      k = L * exp (-(2.0 * (S - P) / W1) * (2.0 * (S - P) / W1)) *
	exp (-(2.0 * g / W2) * (2.0 * g / W2));
    }
  
  if ((N2 != 0.0) && (O >= arg->Nstart2))
    {
      g = calcg (O + Off2, N2); 

      k = k + (L2 * exp (-(2.0 * (S - P2) / W12) * (2.0 * (S - P2) / W12)) *
	       exp (-(2.0 * g / W22) * (2.0 * g / W22)));
    }
  
  if ((N3 != 0.0) && (O >= arg->Nstart3))
    {
      g = calcg (O + Off3, N3);

      k = k + (L3 * exp (-(2.0 * (S - P3) / W13) * (2.0 * (S - P3) / W13)) * 
	       exp (-(2.0 * g / W23) * (2.0 * g / W23)));
    }

  return (k);
}				/* calck */


/*********************************************/
/*innodule2: is Line O 
  part of a Nodule?                         */
/*********************************************/
double
innodule2 (struct SLarguments *arg, double O, double O2)
{
  double k1 = 0.0, k2 = 0.0, g;

  if ((arg->N != 0.0) && (O >= arg->Nstart))
    {
      g = calcg (O, arg->N);
      k1 = arg->L * exp (-(2.0 * g / arg->W2) * (2.0 * g / arg->W2));
      g = calcg (O2, arg->N);
      k2 = arg->L * exp (-(2.0 * g / arg->W2) * (2.0 * g / arg->W2));
    }
  if ((arg->N2 != 0.0) && (O >= arg->Nstart2)) 
    {
      g = calcg (O + arg->Off2, arg->N2);
      k1 = k1 + (arg->L2 * exp (-(2.0 * g / arg->W22) * (2.0 * g / arg->W22)));
      g = calcg (O2 + arg->Off2, arg->N2);
      k2 = k2 + (arg->L2 * exp (-(2.0 * g / arg->W22) * (2.0 * g / arg->W22)));
    }
  if ((arg->N3 != 0.0) && (O >= arg->Nstart2))
    {
      g = calcg (O + arg->Off3, arg->N3);
      k1 = k1 + (arg->L3 * exp (-(2.0 * g / arg->W23) * (2.0 * g / arg->W23)));
      g = calcg (O2 + arg->Off3, arg->N3);
      k2 = k2 + (arg->L3 * exp (-(2.0 * g / arg->W23) * (2.0 * g / arg->W23)));
    }

  return (fabs(k1) - fabs(k2));
}


/*********************************************/
/*innodule: is Point(S+sc,O) 
  part of a Nodule?                          */
/*********************************************/
double
innodule (struct SLarguments *arg, double O, double S, double S2)
{
  double k1 = 0.0, k2 = 0.0;

  k1 = calck (arg, S, O);

  k2 = calck (arg, S2, O);

  return (fabs(k1) - fabs(k2));

}				/* innodule */


/**********************************************/
/* SLCalcPoint:                               */
/*                                            */
/**********************************************/
int
SLCalcPoint (double O, double S, struct SLarguments *arg, 
	     struct SLpoint **SLpoint)
{
  struct SLpoint *SLpoint1 = NULL;
  double x, y, z, R, Re, k;
  double a, b, A, beta, phi, omega, alpha, my, scale;

  alpha = arg->alpha;
  beta = arg->beta;
  phi = arg->phi;
  omega = arg->omega;
  my = arg->my;
  scale = arg -> scale;
  a = arg->a;
  b = arg->b;
  A = arg->A;

  arg->countp++;

  Re = pow ((pow (a, -2.0) * (cos (S) * cos (S)) + pow (b, -2.0) * 
	     (sin (S) * sin (S))), -0.5);

  k = calck (arg, S, O);

  R = Re + k;

  x = scale * (A * sin (beta) * cos (O) + R * cos (S + phi) * cos (O + omega)
	       - R * sin (my) * sin (S + phi) * sin (O)) * exp (O * cot (alpha));
  y = scale * (-A * sin (beta) * sin (O) - R * cos (S + phi) * sin (O + omega)
	       - R * sin (my) * sin (S + phi) * cos (O)) * exp (O * cot (alpha));
  z = scale * (-A * cos (beta) + R * sin (S + phi) * cos (my))
    * exp (O * cot (alpha));

  if ((SLpoint1 = calloc (1, sizeof (struct SLpoint))) == NULL)
    return (5);

  (*SLpoint1).x = x;
  (*SLpoint1).y = y;
  (*SLpoint1).z = z;
  (*SLpoint1).s = S;

  *SLpoint = SLpoint1;

  return (0);

}				/* SLCalcPoint */


/**********************************************/
/* calcnews:                                  */
/**********************************************/
void
calcnews (struct SLarguments *arg, double O, double Sold, double *newS)
{
  double S = Sold, sd;
  short midnodule = 0;

  sd = arg->sd;

  /* for speedup, check the big step first */  
  if(fabs(innodule(arg, O, Sold, Sold+sd)) < arg->heightdiffs)
    {
      if(innodule(arg, O, Sold, Sold+arg->Scans) < 0.0)
	if(innodule(arg, O, Sold+sd, Sold+sd+arg->Scans) < 0.0)
	  {
	    *newS = Sold+sd;
	    return;
	  }
	
      if(innodule(arg, O, Sold, Sold+arg->Scans) > 0.0)
	if(innodule(arg, O,Sold+sd, Sold+sd+arg->Scans) > 0.0)
	  {
	    *newS = Sold+sd;
	    return;
	  }   
    }

  S += arg->Scans;

  while ((fabs (innodule (arg, O, Sold, S)) < arg->heightdiffs) && 
	 (S - Sold < sd) && (!midnodule))
    {
      S += arg->Scans;
      if ((innodule (arg, O, Sold, S) < 0.0) && 
	  (innodule (arg, O, Sold, S+arg->Scans) > 0.0))
	midnodule = 1;
    }
  *newS = S;

  return;

}				/* calcnews */


/**********************************************/
/* calcnewo:                                  */
/**********************************************/
void
calcnewo (struct SLarguments *arg, double Oold, double *newO)
{
  double O = Oold;
  short midnodule = 0;

  /* for speedup, check the big step first */  
  if(fabs(innodule2(arg, Oold, Oold+arg->od)) < arg->heightdiffo)
    {
      if(innodule2(arg, Oold, Oold+arg->Scano) < 0.0)
	if(innodule2(arg,Oold+arg->od, Oold+arg->od+arg->Scano) < 0.0)
	  {
	    *newO = Oold+arg->od;
	    return;
	  }
	
      if(innodule2(arg, Oold, Oold+arg->Scano) > 0.0)
	if(innodule2(arg,Oold+arg->od, Oold+arg->od+arg->Scano) > 0.0)
	  {
	    *newO = Oold+arg->od;
	    return;
	  }   
    }

  O += arg->Scano;

  while ((fabs (innodule2 (arg, Oold, O)) < arg->heightdiffo) && 
	 (O - Oold < arg->od) && (!midnodule))
    {
      O += arg->Scano;
      if ((innodule2 (arg, Oold, Oold+arg->Scano) < 0.0) && 
	  (innodule2 (arg, O, O+arg->Scano) > 0.0))
	midnodule = 1;	
    }

  *newO = O;

  return;
}				/* calcnewo */


/**********************************************/
/* SLCalcNodLine:                             */
/*                                            */
/**********************************************/
int
SLCalcNodLine (double O, struct SLarguments *arg, struct SLpoint **list)
{
  struct SLpoint *list1 = NULL, *L1p1, *L1p2;
  double S, newS;
  double smin, smax;

  smin = arg->smin;
  smax = arg->smax;

  S = smin;

  arg->countw++;

  SLCalcPoint (O, smin, arg, &list1);
  L1p1 = list1;
  L1p2 = list1;

  while (S <= smax)
    {
      calcnews (arg, O, S, &newS);
      S = newS;

      /*      if(!(((S > arg->SkipSStart) && (S < arg->SkipSEnd)) &&
	   ((O > arg->SkipOStart) && (O < arg->SkipOEnd))))*/
	if(S < smax)
	  SLCalcPoint (O, S, arg, &L1p1);
	else
	  SLCalcPoint (O, smax, arg, &L1p1);

      L1p2->next = L1p1;
      L1p2 = L1p1;

    }				/* while */

  *list = list1;

  return (0);

}				/* SLCalcNodLine */


/**********************************************/
/* SLCalcLine:                                */
/*                                            */
/**********************************************/
int
SLCalcLine (double O, struct SLarguments *arg, struct SLpoint **list)
{
  struct SLpoint *list1 = NULL, *L1p1 = NULL, *L1p2 = NULL;
  double x, y, z, R = 1.0, S, Re, k;
  double my, scale;
  double smin, smax, sd, a, b, A, beta, phi, omega, alpha;

  smin = arg->smin;
  smax = arg->smax;
  sd = arg->sd;
  scale = arg->scale;
  alpha = arg->alpha;
  beta = arg->beta;
  phi = arg->phi;
  omega = arg->omega;
  my = arg->my;

  a = arg->a;
  b = arg->b;
  A = arg->A;

  arg->countw = 0;

  for (S = smin; (S + sd) < smax; S = S + sd)
    {

      /*      if(!(((S > arg->SkipSStart) && (S < arg->SkipSEnd)) &&
	   ((O > arg->SkipOStart) && (O < arg->SkipOEnd))))*/
	{
	  arg->countw++;

 	  Re = pow ((pow (a, -2.0) * (cos (S) * cos (S)) + pow (b, -2.0) *  
 		     (sin (S) * sin (S))), -0.5); 

 	  k = calck (arg, S, O); 

 	  R = Re + k; 


	  x = scale * (A * sin(beta) * cos(O) + R * cos(S + phi) * 
		       cos(O + omega) - R * sin(my) * sin(S + phi) * 
		       sin(O)) * exp(O * cot(alpha));
	  y = scale * (-A * sin(beta) * sin(O) - R * cos(S + phi) *
		       sin(O + omega) - R * sin (my) * sin (S + phi) * 
		       cos (O)) * exp (O * cot (alpha));
	  z = scale * (-A * cos (beta) + R * sin (S + phi) * cos (my)) * 
	    exp (O * cot (alpha));

	  if ((L1p1 = calloc (1, sizeof (struct SLpoint))) == NULL)
	    return (5);

	  if (S == smin)
	    list1 = L1p1;
	  else
	    (*L1p2).next = L1p1;
	  L1p2 = L1p1;

	  (*L1p1).x = x;
	  (*L1p1).y = y;
	  (*L1p1).z = z;
	  (*L1p1).s = S;
	}
    }				/* for */

  *list = list1;

  return (0);
}				/* SLCalcLine */


/**********************************************/
/* SLCalcLineFromGCurve:                      */
/*                                            */
/**********************************************/
int
SLCalcLineFromGCurve (double O, struct SLarguments *arg, struct SLpoint **list)
{
  struct SLpoint *list1 = NULL, *L1p1 = NULL, *L1p2 = NULL;
  struct SLgcpoint *gcp = arg->GCurve;
  double x, y, z, S, R;
  double my, scale;
  double a, b, A, beta, phi, omega, alpha;
  
  scale = arg->scale;
  alpha = arg->alpha;
  beta = arg->beta;
  phi = arg->phi;
  omega = arg->omega;
  my = arg->my;

  a = arg->a;
  b = arg->b;
  A = arg->A;

  arg->countw = 0;

  while(gcp)
    {
      arg->countw++;

      R = sqrt(a*gcp->x*a*gcp->x+b*gcp->y*b*gcp->y);
      
      S = atan2(b*gcp->y,a*gcp->x);

      x = scale * (A * sin(beta) * cos(O) + R * cos(S + phi) * 
		   cos(O + omega) - R * sin(my) * sin(S + phi) * 
		   sin(O)) * exp(O * cot(alpha));
      y = scale * (-A * sin(beta) * sin(O) - R * cos(S + phi) *
		   sin(O + omega) - R * sin (my) * sin (S + phi) * 
		   cos (O)) * exp (O * cot (alpha));
      z = scale * (-A * cos (beta) + R * sin (S + phi) * cos (my)) * 
	exp (O * cot (alpha));

      if ((L1p1 = calloc (1, sizeof (struct SLpoint))) == NULL)
	return (5);

      if (gcp == arg->GCurve)
	list1 = L1p1;
      else
	(*L1p2).next = L1p1;
      L1p2 = L1p1;

      (*L1p1).x = x;
      (*L1p1).y = y;
      (*L1p1).z = z;
      (*L1p1).s = S;

      gcp = gcp->next;
    }				/* while */

  *list = list1;

  return (0);
}				/* SLCalcLineFromGCurve */


/**********************************************/
/* SLCalcShellNodule:                         */
/* Nodule-mode                                */
/**********************************************/
int
SLCalcShellNodule (struct SLarguments *arg, struct SLshell **shell)
{
  struct SLpoint *list = NULL;
  struct SLshell *hp = NULL;
  double O, newO, omin, omax;


  omin = arg->omin;
  omax = arg->omax;
  O = omin;

  arg->countp = arg->counth = arg->countw = 0;

  if (SLCalcNodLine (omin, arg, &list) != 0)
    return (5);

  arg->counth++;

  if (!(hp = calloc (1, sizeof (struct SLshell))))
    {
      return (5);
    }

  hp->line = list;
  *shell = hp;

  while (O <= omax)
    {
      calcnewo (arg, O, &newO);
      O = newO;

      if (SLCalcNodLine (O, arg, &list) != 0)
	return (5);
      
      arg->counth++;

      if (!(hp->next = calloc (1, sizeof (struct SLshell))))
	return (5);
	
      hp = hp->next;
      hp->line = list;
    }

  return (0);
}				/* SLCalcShellNodule */


/**********************************************/
/* SLCalcShell:                               
   Basic Shell generation on a regular grid   */
/**********************************************/
int
SLCalcShell (struct SLarguments *arg, struct SLshell **shell)
{
  struct SLpoint *list = NULL;
  struct SLshell *hp = NULL;
  double O, omin, omax, od;

  omin = arg->omin;
  omax = arg->omax;
  od = arg->od;

  arg->countp = arg->counth = arg->countw = 0;

  if (SLCalcLine (omin, arg, &list) != 0)
    return (5);

  arg->counth++;

  if (!(hp = calloc (1, sizeof (struct SLshell))))
    return (5);

  hp->line = list;
  *shell = hp;

  for (O = omin + od; (O + od) < omax; O = O + od)
    {

      if (SLCalcLine (O, arg, &list) != 0)
	return (5);

      arg->counth++;

      if (!(hp->next = calloc (1, sizeof (struct SLshell))))
	return (5);

      hp = hp->next;
      hp->line = list;
    }				/* for */

  return (0);
}				/* SLCalcShell */



/**********************************************/
/* SLCalcShellGC:                               
   Shell generation using an user specified
   generating curve                           */
/**********************************************/
int
SLCalcShellGC (struct SLarguments *arg, struct SLshell **shell)
{
  struct SLpoint *list = NULL;
  struct SLshell *hp = NULL;
  double O, omin, omax, od;

  omin = arg->omin;
  omax = arg->omax;
  od = arg->od;

  arg->countp = arg->counth = arg->countw = 0;

  if (SLCalcLineFromGCurve (omin, arg, &list) != 0)
    return (5);

  arg->counth++;

  if (!(hp = calloc (1, sizeof (struct SLshell))))
    return (5);

  hp->line = list;
  *shell = hp;

  for (O = omin + od; (O + od) < omax; O = O + od)
    {

      if (SLCalcLineFromGCurve (O, arg, &list) != 0)
	return (5);

      arg->counth++;

      if (!(hp->next = calloc (1, sizeof (struct SLshell))))
	return (5);

      hp = hp->next;
      hp->line = list;
    }				/* for */

  return (0);
}				/* SLCalcShellGC */











