/*    ShellyLibV2.0 - the ShellShapeGenerator 
**
**    Copyright (C) 1996 Randolf Schultz (rschultz@informatik.uni-rostock.de)
**
**    This software is shareware!
**    Read the file "License" for further information.
*/

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "shellyl.h"
#include "frontend.h"
#include "rdwrt.h"

/********************************************/
/*ReadInfile:                               */
/********************************************/
void
ReadInfile (struct SLarguments *arg, FILE * fp)
{
  char readline[laenge], hilf[laenge], *hp;
  struct SLgcpoint *gcp = NULL, *gcpn = NULL;
  /* set some defaults */
  arg->scale = 1.0;
  arg->N2 = 0.0;
  arg->N3 = 0.0;
  arg->N = 0.0;
  arg->Nstart = 0.0;
  arg->Nstart2 = 0.0;
  arg->Nstart3 = 0.0;
  arg->mode = NORMAL;
  arg->output = DEFAULTOUT;

  if(arg->GCurve)
    SLFreeGC(arg);
  arg->GCurve = NULL;

  while (fgets (readline, laenge, fp) != NULL)
    {
      if(readline[0] != '#')
      {
      hp = strstr (readline, "alpha:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 6);
	  arg->alpha = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "beta:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->beta = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "phi:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->phi = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "my:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->my = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "omega:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 6);
	  arg->omega = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "smin:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->smin = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "smax:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->smax = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "sd:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->sd = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "A:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->A = atof (hilf);
	}			/* if */
      hp = strstr (readline, "Scale:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 6);
	  arg->scale = atof (hilf);
	}			/* if */
      hp = strstr (readline, "a:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->a = atof (hilf);
	}			/* if */
      hp = strstr (readline, "b:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->b = atof (hilf);
	}			/* if */
      hp = strstr (readline, "P:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->P = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W1:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->W1 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->W2 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "N:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->N = atof (hilf);
	}			/* if */
      hp = strstr (readline, "L:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 2);
	  arg->L = atof (hilf);
	}			/* if */
      hp = strstr (readline, "omin:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->omin = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "omax:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->omax = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "od:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->od = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "SkipOStart:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 11);
	  arg->SkipOStart = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "SkipOEnd:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 9);
	  arg->SkipOEnd = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "SkipSStart:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 11);
	  arg->SkipSStart = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "SkipSEnd");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 9);
	  arg->SkipSEnd = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Hdo:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->heightdiffo = atof (hilf);
	}			/* if */
      hp = strstr (readline, "Hds:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->heightdiffs = atof (hilf);
	}			/* if */
      hp = strstr (readline, "Scans:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 6);
	  arg->Scans = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Scano:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 6);
	  arg->Scano = atof (hilf) * pi / 180.0;
	}			/* if */

      hp = strstr (readline, "P2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->P2 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W12:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->W12 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W22:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->W22 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "N2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->N2 = atof (hilf);
	}			/* if */
      hp = strstr (readline, "L2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->L2 = atof (hilf);
	}			/* if */
      hp = strstr (readline, "P3:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->P3 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W13:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->W13 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "W23:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 4);
	  arg->W23 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "N3:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->N3 = atof (hilf);
	}			/* if */
      hp = strstr (readline, "L3:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 3);
	  arg->L3 = atof (hilf);
	}			/* if */
      hp = strstr (readline, "Off2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->Off2 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Off3:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 5);
	  arg->Off3 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Nstart:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 7);
	  arg->Nstart = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Nstart2:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 8);
	  arg->Nstart2 = atof (hilf) * pi / 180.0;
	}			/* if */
      hp = strstr (readline, "Nstart3:");
      if (hp != NULL)
	{
	  strcpy (hilf, hp + 8);
	  arg->Nstart3 = atof (hilf) * pi / 180.0;
	}			/* if */

      /* modes of operation */
      hp = strstr (readline, "NORMAL");
      if (hp != NULL)
	{
	  arg->mode = NORMAL;
	}			/* if */
      hp = strstr (readline, "NODULE");
      if (hp != NULL)
	{
	  arg->mode = NODULE;
	}			/* if */
      hp = strstr (readline, "GCURVE");
      if (hp != NULL)
	{
	  arg->mode = GCURVE;
	}			/* if */

      /* Output types */

      hp = strstr (readline, "RAW");
      if (hp != NULL)
	{
	  arg->output = RAW;
	}			/* if */
      hp = strstr (readline, "RPL");
      if (hp != NULL)
	{
	  arg->output = RPL;
	}			/* if */
      hp = strstr (readline, "T3D");
      if (hp != NULL)
	{
	  arg->output = T3D;
	}			/* if */
      hp = strstr (readline, "X3D");
      if (hp != NULL)
	{
	  arg->output = X3D;
	}			/* if */
      hp = strstr (readline, "TSP");
      if (hp != NULL)
	{
	  arg->output = TSP;
	}			/* if */
      hp = strstr (readline, "SCD");
      if (hp != NULL)
	{
	  arg->output = SCD;
	}			/* if */
      hp = strstr (readline, "RIB");
      if (hp != NULL)
	{
	  arg->output = RIB;
	}			/* if */
      hp = strstr (readline, "SCB");
      if (hp != NULL)
	{
	  arg->output = SCB;
	}			/* if */
      hp = strstr (readline, "BEZ");
      if (hp != NULL)
	{
	  arg->output = BEZ;
	}			/* if */
      hp = strstr (readline, "POV");
      if (hp != NULL)
	{
	  arg->output = BEZ;
	}			/* if */
      }

      hp = strstr (readline,"GCx:");
      if (hp != NULL)
	{
	  if((gcpn = calloc(1, sizeof(struct SLgcpoint))) == NULL)
	   exit(5);

	  if(gcp)
	    gcp->next = gcpn;
	  else
	    arg->GCurve = gcpn;

	  gcpn->next = NULL;

	  gcpn->x = 0.0;
	  gcpn->y = 0.0;

	  strcpy (hilf, hp + 4);
	  gcpn->x = atof(hilf);

	  gcp = gcpn;    
	}			/* if */
      hp = strstr (readline,"GCy:");
      if (hp != NULL)
	{
	  if(gcp)
	    {
	      strcpy (hilf, hp + 4);
	      gcp->y = atof(hilf);
	    }

	}			/* if */

    }				/* while */

  return;
}				/* ReadInfile */


/********************************************/
/*WriteRIBFile:                             */
/* Write a File in RIB-Format               */
/********************************************/
void
WriteRIBFile (struct SLarguments *arg, struct SLshell *shell, FILE *fp)
{
  struct SLpoint *hp;
  double scale = SCALERIB;

  fprintf(fp,"# RIB generated by ShellyLibV2.0\n\n");

  fprintf(fp,"Display \"shell.tiff\" \"file\" \"rgb\"\n");
  fprintf(fp,"Projection \"perspective\"\n");
  fprintf(fp,"Translate 0.0 0.0 2.0\n");
  fprintf(fp,"WorldBegin\n");
  fprintf(fp,"LightSource \"distantlight\" 1\n");
  fprintf(fp,"AttributeBegin\n");
  fprintf(fp,"Color [ 0.9 0.9 0.8 ]\n");

  fprintf(fp,"#Surface \"shell\"\n");
  fprintf(fp,"Surface \"plastic\"\n");

  fprintf(fp,"# mirror shell to ensure unique coiling direction\n");
  fprintf(fp,"ConcatTransform [-1 0 0 0  0 1 0 0  0 0 1 0  0 0 0 1]\n");

  fprintf(fp,"#Basis \"b-spline\" 1 \"b-spline\" 1\n");
  fprintf(fp,"Basis \"catmull-rom\" 1 \"catmull-rom\" 1\n");

  fprintf(fp,"PatchMesh \"bicubic\" %d \"nonperiodic\" %d \"nonperiodic\" \"P\"\
[ \n", arg->countw, arg->counth);
  
  while(shell)
    {
      hp = shell->line;
      while (hp)
	{
	  fprintf(fp,"%f %f %f ", hp->x*scale, hp->y*scale,
		  hp->z*scale);
	  hp = hp->next;
	}
      shell = shell->next;
    }

  fprintf(fp,"]\nAttributeEnd\nWorldEnd\n");
  return;
}  /* WriteRIBFile */


/* VSub */
void
VSub(struct SLpoint *p1, struct SLpoint *p2, struct SLpoint *res)
{
  res->x = p1->x - p2->x; 
  res->y = p1->y - p2->y;
  res->z = p1->z - p2->z;
  return;
}  /* VSub */


/* VAdd */
void
VAdd(struct SLpoint *p1, struct SLpoint *p2, struct SLpoint *res)
{
  res->x = p1->x + p2->x; 
  res->y = p1->y + p2->y;
  res->z = p1->z + p2->z;
  return;
}  /* VAdd */


/* VLength */
void
VLength(struct SLpoint *v, double *length)
{
 *length = sqrt(v->x * v->x + v->y * v->y + v->z * v->z);
 return;
}


/* VNorm */
void
VNorm(struct SLpoint *v, double length)
{
double l;

l = sqrt(v->x * v->x + v->y * v->y + v->z * v->z);

if(l != 0.0)
  {
    if(length == 0.0)
      length = 1.0;

    v->x = v->x * (length / l);
    v->y = v->y * (length / l);
    v->z = v->z * (length / l);
  }
return;
} /* VNorm */

/**********************************************/
/* CalcControlpoints                          */
/* calc bezier controlpoints using            */
/* tangents                                   */
/* hp & res _must_ point to 16-element-arrays */
/**********************************************/
void
CalcControlpoints(struct SLpoint *hp, struct SLpoint *res)
{
  struct SLpoint i, i2, i3;
  double length;

  /* corner points */
  res[0] = hp[3];
  res[3] = hp[4];
  res[12] = hp[7];
  res[15] = hp[8];

  /* outer control-points */
  VSub(&hp[4], &hp[3], &i);
  VLength(&i, &length);
  /* cp 1 */
  VSub(&hp[2], &hp[4], &i);
  VNorm(&i, length/BEZCON);
  VSub(&hp[3], &i, &res[1]);
  /* cp 2 */
  VSub(&hp[3], &hp[5], &i);
  VNorm(&i, length/BEZCON);
  VAdd(&hp[4], &i, &res[2]);

  VSub(&hp[8], &hp[7], &i);
  VLength(&i, &length);
  /* cp 13 */
  VSub(&hp[6], &hp[8], &i);
  VNorm(&i, length/BEZCON);
  VSub(&hp[7], &i, &res[13]);
  /* cp 14 */
  VSub(&hp[7], &hp[9], &i);
  VNorm(&i, length/BEZCON);
  VAdd(&hp[8], &i, &res[14]);

  VSub(&hp[7], &hp[3], &i);
  VLength(&i, &length);
  /* cp 4 */
  VSub(&hp[0], &hp[7], &i);
  VNorm(&i, length/BEZCON);
  VSub(&hp[3], &i, &res[4]);
  /* cp 8 */
  VSub(&hp[3], &hp[10], &i);
  VNorm(&i, length/BEZCON);
  VAdd(&hp[7], &i, &res[8]);

  VSub(&hp[8], &hp[4], &i);
  VLength(&i, &length);
  /* cp 7 */
  VSub(&hp[1], &hp[8], &i);
  VNorm(&i, length/BEZCON);
  VSub(&hp[4], &i, &res[7]);
  /* cp 11 */
  VSub(&hp[4], &hp[11], &i);
  VNorm(&i, length/BEZCON);
  VAdd(&hp[8], &i, &res[11]);

  /* inner control-points */
  /* cp 5 */
  VSub(&res[4], &res[0], &i);
  VSub(&res[1], &res[0], &i2);
  VAdd(&i, &i2, &i3);
  VAdd(&res[0], &i3, &res[5]);
  /* cp 6 */
  VSub(&res[7], &res[3], &i);
  VSub(&res[2], &res[3], &i2);
  VAdd(&i, &i2, &i3);
  VAdd(&res[3], &i3, &res[6]);  
  /* cp 9 */
  VSub(&res[8], &res[12], &i);
  VSub(&res[13], &res[12], &i2);
  VAdd(&i, &i2, &i3);
  VAdd(&res[12], &i3, &res[9]);  
  /* cp 10 */
  VSub(&res[11], &res[15], &i);
  VSub(&res[14], &res[15], &i2);
  VAdd(&i, &i2, &i3);
  VAdd(&res[15], &i3, &res[10]);  

 return;
} /* CalcControlpoints */

/****************************************/
/* writes a POVRay-Bezier-Patch         */
/* p _must_ point to a 16-Element-Array */
/****************************************/
void
WriteBEZPatch(struct SLarguments *arg, struct SLpoint *p, FILE *fp)
{
  double scale = SCALEBEZ;

  fprintf(fp,"bicubic_patch{type %d flatness %f u_steps %d v_steps %d\n",
	 BEZTYPE, BEZFLAT, BEZSTEPU, BEZSTEPV);

  fprintf(fp,"<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,\n",
	 p[0].x*scale, p[0].y*scale, p[0].z*scale, p[1].x*scale, 
	 p[1].y*scale, p[1].z*scale, p[2].x*scale, p[2].y*scale, 
	 p[2].z*scale, p[3].x*scale, p[3].y*scale,p[3].z*scale);
  fprintf(fp,"<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,\n",
	  p[4].x*scale, p[4].y*scale, p[4].z*scale, p[5].x*scale,
	  p[5].y*scale, p[5].z*scale, p[6].x*scale, p[6].y*scale, 
	  p[6].z*scale, p[7].x*scale, p[7].y*scale, p[7].z*scale);
  fprintf(fp,"<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,\n",
	  p[8].x*scale, p[8].y*scale, p[8].z*scale, p[9].x*scale,
	  p[9].y*scale, p[9].z*scale, p[10].x*scale, p[10].y*scale,
	  p[10].z*scale, p[11].x*scale, p[11].y*scale, p[11].z*scale);
  fprintf(fp,"<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>,<%f,%f,%f>\n",
	  p[12].x*scale, p[12].y*scale, p[12].z*scale, p[13].x*scale,
	  p[13].y*scale, p[13].z*scale, p[14].x*scale, p[14].y*scale,
	  p[14].z*scale, p[15].x*scale, p[15].y*scale, p[15].z*scale);
       
  fprintf(fp,"pigment{SLte}}\n"); 

  return;
}  /* WriteBEZPatch */


/****************************************************/
/* writes a SCED-Bezier-Patch                       */
/* p _must_ point to a 16-Element-Array             */  
/* note: Yes yes the order of the                   */
/* points _is_ correct!                             */
/****************************************************/
void
WriteSCDPatch(struct SLarguments *arg, struct SLpoint *p, FILE *fp)
{
  double scale = SCALESCD;

  fprintf(fp,"Object \"patch\" \"bezier\"\n");

  fprintf(fp,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	 p[0].x*scale, p[0].y*scale, p[0].z*scale, p[3].x*scale, 
	 p[3].y*scale, p[3].z*scale, p[15].x*scale, p[15].y*scale, 
	 p[15].z*scale, p[12].x*scale, p[12].y*scale,p[12].z*scale);
  fprintf(fp,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  p[1].x*scale, p[1].y*scale, p[1].z*scale, p[2].x*scale,
	  p[2].y*scale, p[2].z*scale, p[7].x*scale, p[7].y*scale, 
	  p[7].z*scale, p[11].x*scale, p[11].y*scale, p[11].z*scale);
  fprintf(fp,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  p[14].x*scale, p[14].y*scale, p[14].z*scale, p[13].x*scale,
	  p[13].y*scale, p[13].z*scale, p[8].x*scale, p[8].y*scale,
	  p[8].z*scale, p[4].x*scale, p[4].y*scale, p[4].z*scale);
  fprintf(fp,"%f %f %f\n%f %f %f\n%f %f %f\n%f %f %f\n",
	  p[5].x*scale, p[5].y*scale, p[5].z*scale, p[6].x*scale,
	  p[6].y*scale, p[6].z*scale, p[10].x*scale, p[10].y*scale,
	  p[10].z*scale, p[9].x*scale, p[9].y*scale, p[9].z*scale);
       
  return;
}  /* WriteSCDPatch */

/********************************************/
/*WriteBEZFile:                             */
/* Write Bezier Meshes for POVRay           */
/********************************************/
void
WriteBEZFile (struct SLarguments *arg, struct SLshell *shell, FILE *fp)
{
  struct SLpoint *p[16];
  int i = 0;
  struct SLpoint cp[16], hp[12]; 

  fprintf(fp,"/* POV2.x data, generated by ShellyLibV2.0 */\n");

  while(shell && shell->next && shell->next->next && shell->next->next->next)
    { 
      p[0] = shell->line; 
      p[1] = shell->next->line; 
      p[2] = shell->next->next->line; 
      p[3] = shell->next->next->next->line;

      do
	{
	  if (p[0] && p[1] && p[2] && p[3])
	    for (i = 0; i < 4; i++)
	      p[i + 4] = p[i]->next;

	  if (p[4] && p[5] && p[6] && p[7])
	    for (i = 0; i < 4; i++)
	      p[i + 8] = p[i + 4]->next;

	  if (p[8] && p[9] && p[10] && p[11])
	    for (i = 0; i < 4; i++)
	      p[i + 12] = p[i + 8]->next;

	  if(p[0] && p[1] && p[2] && p[3] && p[4] && p[5] && p[6] && p[7] && 
	     p[8]&& p[9] && p[10] && p[11] && p[12] && p[13] && p[14] && p[15])
	    {
	      
	      hp[0] = *p[1];
	      hp[1] = *p[2];
	      hp[2] = *p[4];
	      hp[3] = *p[5];
	      hp[4] = *p[6];
	      hp[5] = *p[7];
	      hp[6] = *p[8];
	      hp[7] = *p[9];
	      hp[8] = *p[10];
	      hp[9] = *p[11];
	      hp[10] = *p[13];
	      hp[11] = *p[14];

	      CalcControlpoints(hp,cp);

	      WriteBEZPatch(arg,cp,fp);
	    } 
	  
	  if (p[0] && p[1] && p[2] && p[3])
	    for (i = 0; i < 4; i++)
	      p[i] = p[i]->next;

	}
      while (p[0] && p[1] && p[2] && p[3] && p[4] && p[5] && p[6] && p[7] && 
	     p[8] && p[9] && p[10] && p[11] && p[12] && p[13] && p[14] &&
	     p[15]);
      shell = shell->next;
    }
  

  return;
}  /* WriteBEZFile */


/********************************************/
/*WriteSCDFile:                             */
/* Write a triangle mesh for SCED           */
/********************************************/
void
WriteSCDFile (struct SLarguments *arg, struct SLshell *shell, struct
	      SLpindex *pind, struct SLlindex *lind, struct SLtindex *tind,
	      FILE *fp)
{
  double scale = SCALESCD;

  if((pind == NULL) || (tind == NULL) || (lind == NULL))
    return;

  fprintf(fp,"Version 0.8\n");

  fprintf(fp,"Wireframe \"Shell\" %d %d\n", arg->points, arg->triangles);

  while(pind)      /* vertices */
    { 
      fprintf(fp,"%f %f %f\n", pind->point->x*scale, pind->point->y*scale,
	     pind->point->z*scale);
      pind = pind->next;
    }  

  while(tind)     /* triangles */
    {
      fprintf(fp," 3 %d %d %d\n",tind->pid1-1, tind->pid2-1, tind->pid3-1); 
      tind = tind->next;
    }


  return;
}  /* WriteSCDFile */


/********************************************/
/*WriteSCDBFile:                            */
/* Write Bezier patches for SCED            */
/********************************************/
void
WriteSCDBFile (struct SLarguments *arg, struct SLshell *shell, FILE *fp)
{
  struct SLpoint *p[16];
  int i = 0;
  struct SLpoint cp[16], hp[12]; 

  fprintf(fp,"Version 0.8\n");

  while(shell && shell->next && shell->next->next && shell->next->next->next)
    { 
      p[0] = shell->line; 
      p[1] = shell->next->line; 
      p[2] = shell->next->next->line; 
      p[3] = shell->next->next->next->line;

      do
	{
	  if (p[0] && p[1] && p[2] && p[3])
	    for (i = 0; i < 4; i++)
	      p[i + 4] = p[i]->next;

	  if (p[4] && p[5] && p[6] && p[7])
	    for (i = 0; i < 4; i++)
	      p[i + 8] = p[i + 4]->next;

	  if (p[8] && p[9] && p[10] && p[11])
	    for (i = 0; i < 4; i++)
	      p[i + 12] = p[i + 8]->next;

	  if(p[0] && p[1] && p[2] && p[3] && p[4] && p[5] && p[6] && p[7] && 
	     p[8]&& p[9] && p[10] && p[11] && p[12] && p[13] && p[14] && p[15])
	    {
	      
	      hp[0] = *p[1];
	      hp[1] = *p[2];
	      hp[2] = *p[4];
	      hp[3] = *p[5];
	      hp[4] = *p[6];
	      hp[5] = *p[7];
	      hp[6] = *p[8];
	      hp[7] = *p[9];
	      hp[8] = *p[10];
	      hp[9] = *p[11];
	      hp[10] = *p[13];
	      hp[11] = *p[14];

	      CalcControlpoints(hp,cp);

	      WriteSCDPatch(arg,cp,fp);
	    } 
	  
	  if (p[0] && p[1] && p[2] && p[3])
	    for (i = 0; i < 4; i++)
	      p[i] = p[i]->next;

	}
      while (p[0] && p[1] && p[2] && p[3] && p[4] && p[5] && p[6] && p[7] && 
	     p[8] && p[9] && p[10] && p[11] && p[12] && p[13] && p[14] &&
	     p[15]);
      shell = shell->next;
    }

  return;
}  /* WriteSCDBFile */


/********************************************/
/*WriteTSPFile:                             */
/* Write a Caligari TrueSPace-file          */
/********************************************/
void
WriteTSPFile (struct SLarguments *arg, struct SLshell *shell, struct
	      SLpindex *pind, struct SLlindex *lind, struct SLtindex *tind,
	      FILE *fp)
{
  double scale = SCALETSP;

  if((pind == NULL) || (tind == NULL) || (lind == NULL))
    return;

  fprintf(fp,"Caligari V00.01ALH             \n");  /*File Header */
  fprintf(fp,"PolH V0.02 Id 1 Parent 0 Size -1");   /*Data-Chunk Header */
  fprintf(fp,"\nName %s","Shell");                  /* Name */

  fprintf(fp,"\ncenter %g %g %g", 0.0, 0.0, 0.0);    /* local Axes */
  fprintf(fp,"\nx axis %g %g %g", 1.0, 0.0, 0.0);
  fprintf(fp,"\ny axis %g %g %g", 0.0, 1.0, 0.0);
  fprintf(fp,"\nz axis %g %g %g", 0.0, 0.0, 1.0);
  
  fprintf(fp,"\nTransform" );                        /* current position */
  fprintf(fp,"\n%g %g %g %g", 1.0, 0.0, 0.0, 0.0);
  fprintf(fp,"\n%g %g %g %g", 0.0, 1.0, 0.0, 0.0);
  fprintf(fp,"\n%g %g %g %g", 0.0, 0.0, 1.0, 0.0);
  fprintf(fp,"\n%g %g %g %g", 0.0, 0.0, 0.0, 1.0 );

  fprintf(fp,"\nWorld Vertices %d", arg->points);    /* No. of vertices */

  while(pind)      /* vertices */
    { 
      fprintf(fp,"\n%f %f %f", pind->point->x*scale, pind->point->y*scale,
	     pind->point->z*scale);
      pind = pind->next;
    }

  fprintf(fp,"\nTexture Vertices %d", 1);  /* No. of UV-vertices ??? */
  fprintf(fp,"\n%f %f", 1.0, 1.0); /* buggy? */

  fprintf(fp,"\nFaces %d", arg->triangles);  /* No. of triangles */

  while(tind) /* triangles */
    {
      fprintf(fp,"\nFace verts %hd flags %hd mat %hd\n", 3, 0x10, 1); 
      fprintf(fp,"<%d,%d> ", tind->pid1-1, 1);
      fprintf(fp,"<%d,%d> ", tind->pid2-1, 1);
      fprintf(fp,"<%d,%d>", tind->pid3-1, 1);

      tind = tind->next;
    }

  fprintf(fp,"\nMat1 V0.05 Id 2 Parent 1 Size -1");
                                                          /* Material chunk */
  fprintf(fp,"\nmat# %d",1);
  fprintf(fp,"\nshader: %s  facet: %s", "phong", "smooth");
  fprintf(fp,"\nrgb %g,%g,%g", 0.8, 0.8, 0.8);
  fprintf(fp,"\nalpha %g  ka %g  ks %g  exp %g  ior %g", 0.0, 0.0, 0.0, 0.0, 0.0);
  
  fprintf(fp,"\nEND  V1.00 Id 0 Parent 0 Size 0\n"); 

  return;
} /* WriteTSPFile */


/********************************************/
/*WriteT3DFile:                             */
/********************************************/
void
WriteT3DFile (struct SLarguments *arg, struct SLshell *shell, struct
	      SLpindex *pind, struct SLlindex *lind, struct SLtindex *tind,
	      FILE *fp)
{
  char percent = '%';
  unsigned int countt = 0, countp = 0, countl = 0;
  double scale = SCALET3D;

  if((pind == NULL) || (tind == NULL) || (lind == NULL))
    return;

  fprintf(fp," %c T3Dlib-data-file generated by ShellyLibV2.0\n", percent);

  fprintf(fp,"OBJ Begin \"Hierarchy 1\"\n");
  fprintf(fp," DESC Begin \"Object 1 at level 1 of hierarchy 1\"\n");
  fprintf(fp,"  NAME \"SHELL\"\n");
  fprintf(fp,"  SHAP Shape = 2\n");
  fprintf(fp,"  SHAP Lamp  = 0\n");
  fprintf(fp,"  POSI X=0 Y=0 Z=0\n");
  fprintf(fp,"  AXIS XAxis X=1 Y=0 Z=0\n");
  fprintf(fp,"  AXIS YAxis X=0 Y=1 Z=0\n");
  fprintf(fp,"  AXIS ZAxis X=0 Y=0 Z=1\n");
  fprintf(fp,"  SIZE X=32 Y=32 Z=32\n");

  fprintf(fp,"  PNTS PCount %d\n", arg->points);

  while(pind)
    {
      fprintf(fp,"  PNTS Point[%d] X=%f Y=%f Z=%f\n", countp++,
	     pind->point->x*scale, pind->point->y*scale,
	     pind->point->z*scale);
      pind = pind->next;
    }
  
  fprintf(fp,"  EDGE ECount %d\n", arg->lines); /* No. of edges */    
  
  while(lind)
    {
      fprintf(fp,"  EDGE Edge[%d] %d %d\n", countl++, lind->pid1-1, lind->pid2-1);
      lind = lind->next;
    }
  
  fprintf(fp,"  FACE TCount %d\n", arg->triangles);   /* No. of triangles */     
  
  while(tind)
    {
      fprintf(fp,"  FACE Connect[%d] %d %d %d\n", countt++, tind->lid1-1,
	     tind->lid2-1, tind->lid3-1);
      tind = tind->next;
    }

  fprintf(fp,"  COLR R=98 G=68 B=58\n");
  fprintf(fp,"  REFL R=0 G=0 B=0\n");
  fprintf(fp,"  TRAN R=0 G=0 B=0\n");
  fprintf(fp,"  SPC1 R=0 G=0 B=0\n");
  fprintf(fp,"  End DESC   \"Object 1 at level 1 of hierarchy 1\"\n");
  fprintf(fp," TOBJ       \"Object 0 at level 1 of hierarchy 1\"\n");
  fprintf(fp,"End OBJ  \"Hierarchy 1\"\n");

  return;
}  /* WriteT3DFile */


/********************************************/
/*WriteRAWFile:                             */
/********************************************/
void
WriteRAWFile (struct SLarguments *arg, struct SLshell *shell, FILE *fp)
{
  struct SLtriangle *trp;
  struct SLpoint *hp;
  double scale = SCALERAW;

  if(shell == NULL)
    return;

  while (shell->next)
    {
      hp = shell->line;
      SLTriangulate(shell->line, shell->next->line, &trp);

      while(trp)
	{
	  fprintf(fp,"%f %f %f", trp->p1->x*scale, trp->p1->y*scale,
		 trp->p1->z*scale);
	  fprintf(fp," %f %f %f", trp->p2->x*scale, trp->p2->y*scale,
		 trp->p2->z*scale);
	  fprintf(fp," %f %f %f\n", trp->p3->x*scale, trp->p3->y*scale,
		 trp->p3->z*scale );
	  trp = trp->next;
	}

      SLFreeTriangles (trp);

      shell = shell->next;
    }

  return;
}				/* WriteRAWFile */


/********************************************/
/*WriteRPLFile:                             */
/* mode == Nodule -> write a triset         */
/* else -> a mesh                           */
/********************************************/
void
WriteRPLFile (struct SLarguments *arg, struct SLshell *shell, struct
	      SLpindex *pind, struct SLtindex *tind, FILE *fp)
{
  struct SLpoint *hp;
  double scale = SCALERPL;

  fprintf(fp,"( RPL-macro generated by ShellyLibV2.0\n");

  /* objects.rpl might not be included at startup by Real3D since V3! */

  fprintf(fp,"\"objects.rpl\" LOAD\n");

  if(arg->mode == NODULE)
    {
      if((pind == NULL) || (tind == NULL))
	return;
     
      while(pind)
	{
	  fprintf(fp,"%f %f %f\n", pind->point->x*scale,
		 pind->point->y*scale, pind->point->z*scale);
	  pind = pind->next;
	}

      fprintf(fp,"%d ( Points\n", arg->points); /* No. of Points */

	while(tind)
	{
	  fprintf(fp,"%d %d %d\n", tind->lid1, tind->lid2,
		 tind->lid3);
	  tind = tind->next;
	}
  
      fprintf(fp,"%d ( Faces\n", arg->triangles);   /* No. of Faces */

      fprintf(fp,"wFT_PHONG\n");
      fprintf(fp,"255 255 255 0\n");	/* RGBA */
      fprintf(fp,"\"Shell\"\n0\n");	/* Name */
      fprintf(fp,"\"CEND\"\n");
      fprintf(fp,"C_TRISET DROP\n");   /* create Triset */

    }
  else
    {
      if (shell == NULL)
	return;

      while (shell)
	{
	  hp = shell->line;
	  while (hp)
	    {
	      fprintf(fp,"%f %f %f\n", hp->x*scale, hp->y*scale,
		      hp->z*scale);
	      hp = hp->next;
	    }
	  shell = shell->next;
	}

      fprintf(fp,"%d ( height\n", arg->counth); /* height, width of the mesh */
      fprintf(fp,"%d ( width\n", arg->countw);

      fprintf(fp,"wFT_BSPLINE\n0\n"); /* GeomType, GeomFlags */
      fprintf(fp,"255 255 255 0\n");  /* RGBA */
      fprintf(fp,"\"Shell\"\n0\n");   /* Name */
      fprintf(fp,"\"CEND\"\n");
      fprintf(fp,"C_MESH DROP\n");    /* create a Mesh */
    }

  return;
}				/* WriteRPLFile */


/********************************************/
/* WriteX3DFile:                            */
/********************************************/
void
WriteX3DFile (struct SLarguments *arg, struct SLshell *shell, struct 
	      SLpindex *pind, struct SLlindex *lind, struct SLtindex *tind,
	      FILE *fp)
{
  unsigned int countt = 0, countp = 0, countl = 0;
  double scale = SCALEX3D;

  if((pind == NULL) || (tind == NULL) || (lind == NULL))
    return;

  fprintf(fp,"2\n");		/* No. of colors */
  fprintf(fp,"0  0 0 0\n");	/* color 0 Black */
  fprintf(fp,"1  255 0 0\n");	/* color 1 Red */
  
  fprintf(fp,"%d\n", arg->points);    /* No. of points */

  while(pind)
    {
      fprintf(fp,"%d %f %f %f\n", countp++, pind->point->x*scale,
	     pind->point->y*scale, pind->point->z*scale);
      pind = pind->next;
    }
  
  fprintf(fp,"%d\n", arg->lines); /* No. of lines */    
  
  while(lind)
    {
      fprintf(fp,"%d 1 %d %d\n", countl++, lind->pid1-1, lind->pid2-1);
      lind = lind->next;
    }
  
  fprintf(fp,"%d\n", arg->triangles);   /* No. of triangles */     
  
  while(tind)
    {
      fprintf(fp,"%d 1 3 %d %d %d\n", countt++, tind->lid1-1, tind->lid2-1,
	     tind->lid3-1);
      tind = tind->next;
    }

  return;
}






