/*    
**  ShellyLibV2.0 - the ShellShapeGenerator 
**
**  Copyright (C) 1996 Randolf Schultz (rschultz@informatik.uni-rostock.de)
**
**  This software is shareware!
**  Read the file "License" for further information.
*/

/* The Shell-Laboratory, a Tcl/Tk - OpenGL GUI for ShellyLib */
/* C-part */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <tcl.h>
#include <tk.h>

#include <togl.h>

#include "shellyl.h"
#include "rdwrt.h"

/* Global: */
static struct SLshell *shell = NULL;
static struct SLarguments arg = {0};

static GLfloat Grotx = 0.0, Groty = 0.0, Grotz = 0.0;
static GLfloat Gmovx = 0.0, Gmovy = 0.0, Gmovz = 0.0;
static GLfloat Gscale = 1.0;

static GLfloat light_pos1[4] = {0.0, 0.0, 100.0, 0.0};
static GLfloat shell_color[4] = {0.7, 0.7, 0.6, 1.0};

/* display-mode wire,flat,smooth? */
static int mode = 0;

/* draw coordsys? */
static int coordsys = 1;


/*
 * The following variable is a special hack that is needed in order for
 * Sun shared libraries to be used for Tcl.
 */
extern int matherr();
int *tclDummyMathPtr = (int *) matherr;


/********************************************/
/*drawshell:
  draws a shell in different modes,
  using OpenGL                              */
/********************************************/
void
drawshell (void)
{
  struct SLshell *lshell = shell;
  struct SLpoint *hp = NULL, *hp2 = NULL;
  struct SLtriangle *triangles;

  float normx, normy, normz;

  if (mode == 0)
    {
      glDisable (GL_LIGHTING);
      while (lshell && lshell->next)
	{
	  hp = lshell->line;
	  hp2 = lshell->next->line;

	  glBegin (GL_LINE_STRIP);
	  while (hp)
	    {
	      if (hp)
		{
		  glVertex3f ((GLfloat) hp->x, (GLfloat) hp->y,
			      (GLfloat) hp->z);
		}
	      hp = hp->next;
	    }
	  glEnd ();

	  hp = lshell->line;
	  hp2 = lshell->next->line;

	  glBegin (GL_LINES);
	  while (hp && hp2)
	    {
	      glVertex3f ((GLfloat) hp->x, (GLfloat) hp->y,
			  (GLfloat) hp->z);
	      glVertex3f ((GLfloat) hp2->x, (GLfloat) hp2->y,
			  (GLfloat) hp2->z);


	      hp2 = hp2->next;
	      hp = hp->next;
	    }
	  glEnd ();

	  lshell = lshell->next;
	}
      hp = lshell->line;
      glBegin (GL_LINE_STRIP);
      while (hp)
	{
	  if (hp)
	    {
	      glVertex3f ((GLfloat) hp->x, (GLfloat) hp->y, (GLfloat) hp->z);
	    }
	  hp = hp->next;
	}
      glEnd ();
    }
  else
    /* mode != 0 */
    {
      glEnable (GL_LIGHTING);
      if ((arg.mode == 0) || (arg.mode == 2))
	while (lshell && lshell->next)
	  {
	    hp = lshell->line;
	    hp2 = lshell->next->line;

	    glBegin (GL_QUAD_STRIP);
	    while (hp && hp2)
	      {

		glVertex3f ((GLfloat) hp->x, (GLfloat) hp->y,
			    (GLfloat) hp->z);

		glVertex3f ((GLfloat) hp2->x, (GLfloat) hp2->y,
			    (GLfloat) hp2->z);
		if(hp->next)
		  {
		/* normal calculation, only valid for flat shading! */
		    normx = (hp2->y - hp->y) * (hp->next->z - hp->z) -
		      (hp2->z - hp->z) * (hp->next->y - hp->y);
		    normy = (hp2->z - hp->z) * (hp->next->x - hp->x) -
		      (hp2->x - hp->x) * (hp->next->z - hp->z);
		    normz = (hp2->x - hp->x) * (hp->next->y - hp->y) -
		      (hp2->y - hp->y) * (hp->next->x - hp->x);

		    glNormal3f (normx, normy, normz);

		  }
		hp2 = hp2->next;
		hp = hp->next;
	      }
	    glEnd ();

	    lshell = lshell->next;
	  }
      else			/* Nodule Mode */
	while (lshell && lshell->next)
	  {
	    hp = lshell->line;
	    hp2 = lshell->next->line;

	    SLTriangulate (hp, hp2, &triangles);

	    glBegin (GL_TRIANGLES);
	    while (triangles)
	      {
		/* normal calculation, only valid for flat shading! */
		normx = (triangles->p1->y - triangles->p2->y) *
		  (triangles->p3->z - triangles->p2->z) -
		  (triangles->p1->z - triangles->p2->z) *
		  (triangles->p3->y - triangles->p2->y);
		normy = (triangles->p1->z - triangles->p2->z) *
		  (triangles->p3->x - triangles->p2->x) -
		  (triangles->p1->x - triangles->p2->x) *
		  (triangles->p3->z - triangles->p2->z);
		normz = (triangles->p1->x - triangles->p2->x) *
		  (triangles->p3->y - triangles->p2->y) -
		  (triangles->p1->y - triangles->p2->y) *
		  (triangles->p3->x - triangles->p2->x);

		glNormal3f (-normx, -normy, -normz);

		glVertex3f ((GLfloat) triangles->p1->x, 
			    (GLfloat) triangles->p1->y,
			    (GLfloat) triangles->p1->z);
		glVertex3f ((GLfloat) triangles->p2->x,
			    (GLfloat) triangles->p2->y,
			    (GLfloat) triangles->p2->z);
		glVertex3f ((GLfloat) triangles->p3->x,
			    (GLfloat) triangles->p3->y,
			    (GLfloat) triangles->p3->z);

		triangles = triangles->next;
	      }
	    glEnd ();

	    SLFreeTriangles (triangles);
	    triangles = NULL;

	    lshell = lshell->next;
	  }

    }

  return;
}				/* drawshell */


/********************************************/
/*drawcoordsys:
  draws a coordinate system 
  using OpenGL                              */
/********************************************/
void
drawcoordsys (void)
{

  if (coordsys > 0)
    {
      glDisable (GL_LIGHTING);
      glBegin (GL_LINES);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (1.0, 0.0, 0.0);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (0.0, 1.0, 0.0);
      glVertex3f (0.0, 0.0, 0.0);
      glVertex3f (0.0, 0.0, 1.0);
      glEnd ();
      glEnable (GL_LIGHTING);
    }

  return;
} /* drawcoordsys */


/********************************************/
/*create_cb:
 Togl Callback, called on creation
                                            */
/********************************************/
void
create_cb (struct Togl *togl)
{

  glEnable (GL_DEPTH_TEST);
  glEnable (GL_NORMALIZE);
  glShadeModel (GL_FLAT);

  glLightfv (GL_LIGHT0, GL_POSITION, light_pos1);
  glEnable (GL_LIGHT0);

  glEnable (GL_LIGHTING);

  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, shell_color);
  glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 1.0);

  return;
} /* create_cb */


/********************************************/
/*reshape_cb:
 Togl Callback, called when window is 
 exposed or reshaped                        */
/********************************************/
void
reshape_cb (struct Togl *togl)
{
  int width = Togl_Width (togl);
  int height = Togl_Height (togl);
  float aspect = (float) width / (float) height;

  glViewport (0, 0, width, height);

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glFrustum (-aspect, aspect, -1.0, 1.0, 1.0, 100.0);

  glMatrixMode (GL_MODELVIEW);

  return;
} /* reshape_cb */


/********************************************/
/*display_cb:
 Togl Callback, displays the shell
                                            */
/********************************************/
void
display_cb (struct Togl *togl)
{

  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glLightfv (GL_LIGHT0, GL_POSITION, light_pos1);

  glPushMatrix ();
  glTranslatef (0.0, 0.0, -10.0);
  glRotatef (Grotx, 1.0, 0.0, 0.0);
  glRotatef (Groty, 0.0, 1.0, 0.0);
  glRotatef (Grotz, 0.0, 0.0, 1.0);
  glScalef (Gscale, Gscale, Gscale);
  glTranslatef (Gmovx, Gmovy, Gmovz);

  if(shell)
    drawshell ();

  drawcoordsys ();

  glPopMatrix ();

  Togl_SwapBuffers (togl);

  return;
} /* display_cb */


/********************************************/
/*setmode_cb:
 Togl Callback, set's rendering mode
                                            */
/********************************************/
int
setmode_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setMode 1|2|3\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  mode = atoi (argv[2]);

  if (mode == 2)
    glShadeModel (GL_SMOOTH);
  else
    glShadeModel (GL_FLAT);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* setmode_cb */


/********************************************/
/*drawsys_cb:
 Togl Callback, toggles drawing of the
 coordinate system                          */
/********************************************/
int
drawsys_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName drawSys\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if (coordsys == 1)
    coordsys = 0;
  else
    coordsys = 1;

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* drawsys_cb */


/********************************************/
/*setXrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setXrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setXrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Grotx = atof (argv[2]);

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setXrot_cb */


/********************************************/
/*setYrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setYrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setYrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Groty = atof (argv[2]);

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setYrot_cb */


/********************************************/
/*setZrot_cb:
 Togl Callback, rotate display
                                            */
/********************************************/
int
setZrot_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);

  /* error checking */
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName setZrot ?angle?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }


  Grotz = Grotz + atof (argv[2]);
  if (Grotz > 360)
    Grotz -= 360;
  if (Grotz < 360)
    Grotz += 360;

  Togl_PostRedisplay (togl);

  /* Let result string equal value */
  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* setZrot_cb */


/********************************************/
/*zoomin_cb:
 Togl Callback, zoom display
                                            */
/********************************************/
int
zoomin_cb (struct Togl *togl, int argc, char *argv[])
{

  Gscale = Gscale + (Gscale / 10.0);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* zoomin_cb */


/********************************************/
/*zoomout_cb:
 Togl Callback, zoom display
                                            */
/********************************************/
int
zoomout_cb (struct Togl *togl, int argc, char *argv[])
{

  Gscale = Gscale - (Gscale / 10.0);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* zoomout_cb */


/********************************************/
/*movez_cb:
 Togl Callback, move shell
                                            */
/********************************************/
int
movez_cb (struct Togl *togl, int argc, char *argv[])
{
  Tcl_Interp *interp = Togl_Interp (togl);
  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"pathName moveZ ?value?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  Gmovz = Gmovz + atof (argv[2]);

  Togl_PostRedisplay (togl);

  strcpy (interp->result, argv[2]);
  return TCL_OK;
} /* movez_cb */


/********************************************/
/*resetview_cb:
 Togl Callback, reset all transformations
                                            */
/********************************************/
int
resetview_cb (struct Togl *togl, int argc, char *argv[])
{
  Grotx = 0.0;
  Groty = 0.0;
  Grotz = 0.0;
  Gmovx = 0.0;
  Gmovy = 0.0;
  Gmovz = 0.0;
  Gscale = 1.0;

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* resetview_cb */


/********************************************/
/*fetchSLarg:
  fetch variables from Tcl                    
  This one crashes badly, if the variables    
  referred to here are not 'to be seen'       
  (accessible) in the current Tcl-context   */
/********************************************/
void
fetchSLarg (Tcl_Interp * interp)
{
  char *tclvar;

  tclvar = Tcl_GetVar (interp, "alpha", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.alpha);
  arg.alpha = arg.alpha * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "beta", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.beta);
  arg.beta = arg.beta * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "phi", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.phi);
  arg.phi = arg.phi * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "my", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.my);
  arg.my = arg.my * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "omega", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.omega);
  arg.omega = arg.omega * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_A", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.A);

  tclvar = Tcl_GetVar (interp, "a", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.a);

  tclvar = Tcl_GetVar (interp, "b", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.b);


  tclvar = Tcl_GetVar (interp, "omin", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.omin);
  arg.omin = arg.omin * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "omax", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.omax);
  arg.omax = arg.omax * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "od", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.od);
  arg.od = arg.od * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "smin", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.smin);
  arg.smin = arg.smin * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "smax", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.smax);
  arg.smax = arg.smax * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "sd", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.sd);
  arg.sd = arg.sd * pi / 180.0;


  tclvar = Tcl_GetVar (interp, "_P1", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.P);
  arg.P = arg.P * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_Nstart1", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Nstart);
  arg.Nstart = arg.Nstart * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W11", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W1);
  arg.W1 = arg.W1 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W21", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W2);
  arg.W2 = arg.W2 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_N1", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.N);

  tclvar = Tcl_GetVar (interp, "_L1", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.L);


  tclvar = Tcl_GetVar (interp, "_P2", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.P2);
  arg.P2 = arg.P2 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_Nstart2", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Nstart2);
  arg.Nstart2 = arg.Nstart2 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W12", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W12);
  arg.W12 = arg.W12 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W22", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W22);
  arg.W22 = arg.W22 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_N2", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.N2);

  tclvar = Tcl_GetVar (interp, "_L2", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.L2);

  tclvar = Tcl_GetVar (interp, "_Off2", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Off2);
  arg.Off2 = arg.Off2 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_P3", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.P3);
  arg.P3 = arg.P3 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_Nstart3", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Nstart3);
  arg.Nstart3 = arg.Nstart3 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W13", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W13);
  arg.W13 = arg.W13 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_W23", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.W23);
  arg.W23 = arg.W23 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "_N3", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.N3);

  tclvar = Tcl_GetVar (interp, "_L3", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.L3);

  tclvar = Tcl_GetVar (interp, "_Off3", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Off3);
  arg.Off3 = arg.Off3 * pi / 180.0;

  tclvar = Tcl_GetVar (interp, "hdo", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.heightdiffo);

  tclvar = Tcl_GetVar (interp, "scano", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Scano);
  arg.Scano = arg.Scano;

  tclvar = Tcl_GetVar (interp, "hds", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.heightdiffs);

  tclvar = Tcl_GetVar (interp, "scans", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.Scans);
  arg.Scans = arg.Scans;

  tclvar = Tcl_GetVar (interp, "scale", TCL_LEAVE_ERR_MSG);
  Tcl_GetDouble (interp, tclvar, &arg.scale);

  tclvar = Tcl_GetVar (interp, "slmode", TCL_LEAVE_ERR_MSG);
  Tcl_GetInt (interp, tclvar, &arg.mode);

  return;
}				/* fetchSLarg */


/********************************************/
/*update3DView_cb:
 Togl Callback, called on slider movement
 calculates a new shell and redisplays      */
/********************************************/
int
update3DView_cb (struct Togl *togl, int argc, char *argv[])
{
  double value = 0.0;

  value = atof (argv[3]);
  if (strcmp (argv[2], "scale") == 0)
    arg.scale = value;
  else if (strcmp (argv[2], "alpha") == 0)
    arg.alpha = value * pi / 180.0;
  else if (strcmp (argv[2], "beta") == 0)
    arg.beta = value * pi / 180.0;
  else if (strcmp (argv[2], "phi") == 0)
    arg.phi = value * pi / 180.0;
  else if (strcmp (argv[2], "my") == 0)
    arg.my = value * pi / 180.0;
  else if (strcmp (argv[2], "omega") == 0)
    arg.omega = value * pi / 180.0;
  else if (strcmp (argv[2], "a") == 0)
    arg.a = value;
  else if (strcmp (argv[2], "b") == 0)
    arg.b = value;
  else if (strcmp (argv[2], "_A") == 0)
    arg.A = value;
  else if (strcmp (argv[2], "omin") == 0)
    arg.omin = value * pi / 180.0;
  else if (strcmp (argv[2], "omax") == 0)
    arg.omax = value * pi / 180.0;
  else if (strcmp (argv[2], "od") == 0)
    arg.od = value * pi / 180.0;
  else if (strcmp (argv[2], "smin") == 0)
    arg.smin = value * pi / 180.0;
  else if (strcmp (argv[2], "smax") == 0)
    arg.smax = value * pi / 180.0;
  else if (strcmp (argv[2], "sd") == 0)
    arg.sd = value * pi / 180.0;
  else if (strcmp (argv[2], "scale") == 0)
    arg.scale = value;
  else if (strcmp (argv[2], "_P1") == 0)
    arg.P = value * pi / 180.0;
  else if (strcmp (argv[2], "_Nstart1") == 0)
    arg.Nstart = value * pi / 180.0;
  else if (strcmp (argv[2], "_W11") == 0)
    arg.W1 = value * pi / 180.0;
  else if (strcmp (argv[2], "_W21") == 0)
    arg.W2 = value * pi / 180.0;
  else if (strcmp (argv[2], "_N1") == 0)
    arg.N = value;
  else if (strcmp (argv[2], "_L1") == 0)
    arg.L = value;
  else if (strcmp (argv[2], "_P2") == 0)
    arg.P2 = value * pi / 180.0;
  else if (strcmp (argv[2], "_Nstart2") == 0)
    arg.Nstart2 = value * pi / 180.0;
  else if (strcmp (argv[2], "_W12") == 0)
    arg.W12 = value * pi / 180.0;
  else if (strcmp (argv[2], "_W22") == 0)
    arg.W22 = value * pi / 180.0;
  else if (strcmp (argv[2], "_N2") == 0)
    arg.N2 = value;
  else if (strcmp (argv[2], "_L2") == 0)
    arg.L2 = value;
  else if (strcmp (argv[2], "_Off2") == 0)
    arg.Off2 = value * pi / 180.0;
  else if (strcmp (argv[2], "_P3") == 0)
    arg.P3 = value * pi / 180.0;
  else if (strcmp (argv[2], "_Nstart3") == 0)
    arg.Nstart3 = value * pi / 180.0;
  else if (strcmp (argv[2], "_W13") == 0)
    arg.W13 = value * pi / 180.0;
  else if (strcmp (argv[2], "_W23") == 0)
    arg.W23 = value * pi / 180.0;
  else if (strcmp (argv[2], "_N3") == 0)
    arg.N3 = value;
  else if (strcmp (argv[2], "_L3") == 0)
    arg.L3 = value;
  else if (strcmp (argv[2], "_Off3") == 0)
    arg.Off3 = value * pi / 180.0;
  else if (strcmp (argv[2], "hdo") == 0)
    arg.heightdiffo = value;
  else if (strcmp (argv[2], "scano") == 0)
    arg.Scano = value;
  else if (strcmp (argv[2], "hds") == 0)
    arg.heightdiffs = value;
  else if (strcmp (argv[2], "scans") == 0)
    arg.Scans = value;

  if (strcmp (argv[2], "slmode") == 0)
    arg.mode = atoi (argv[3]);

  SLFreeShell (shell);
  shell=NULL;

  if (arg.mode == 0) 
    SLCalcShell (&arg, &shell); 
  if (arg.mode == 1)
    SLCalcShellNodule (&arg, &shell); 
  if (arg.mode == 2)
    SLCalcShellGC (&arg, &shell);

  Togl_PostRedisplay (togl);

  return TCL_OK;
} /* update3DView */


/********************************************/
/*cmdCreateShell:
  a Tcl-command that interfaces to 
  Shelly-Lib                                */
/********************************************/
int
cmdCreateShell (ClientData clientData, Tcl_Interp * interp,
		int argc, char *argv[])
{
  char *tclvar;

  tclvar = Tcl_GetVar (interp, "slmode", TCL_LEAVE_ERR_MSG);

  fetchSLarg (interp);

  arg.SkipSStart = 0;
  arg.SkipSEnd = 0;
  arg.SkipOStart = 0;
  arg.SkipOEnd = 0;

  SLFreeShell (shell);
  shell = NULL;

  /* create shell */
  if (arg.mode == 0) 
    SLCalcShell (&arg, &shell); 
  if (arg.mode == 1)
    SLCalcShellNodule (&arg, &shell); 
  if (arg.mode == 2)
    SLCalcShellGC (&arg, &shell);

  return TCL_OK;
}				/* cmdCreateShell */


/********************************************/
/*cmdDeleteShell:
  a Tcl-command that interfaces to 
  Shelly-Lib                                */
/********************************************/
int
cmdDeleteShell (ClientData clientData, Tcl_Interp * interp,
		int argc, char *argv[])
{

  SLFreeShell (shell);
  shell = NULL;

  return TCL_OK;
}				/* cmdDeleteShell */


/********************************/
/* setslider:
  sets the values from
  SLarg into Tcl-Vars          */
/********************************/
void
setslider (Tcl_Interp * interp, char *var, double val)
{
  char valstring[250], *getval, varname[255];
  double minslider = 0.0, maxslider = 0.0;

  if((fabs(val) < 0.01) && (val != 0.0))
    sprintf (valstring, "%f", val);
  else
    sprintf (valstring, "%.2f", val);

  /* before setting we must check (and correct) the bounds */
  sprintf (varname, "%smin", var);
  if((getval = Tcl_GetVar (interp, varname, TCL_LEAVE_ERR_MSG)) == NULL)
    fprintf(stderr,"oops, cannot get variable %s\n", varname);
  else
    Tcl_GetDouble (interp, getval, &minslider);

  if (val < minslider)
    {
      /* We don't configure the slider here, but we set the
       * associated variable */
      if(Tcl_SetVar (interp, varname, valstring, TCL_LEAVE_ERR_MSG) == NULL)
	fprintf(stderr,"oops, cannot set variable %s\n", varname);
    }

  sprintf (varname, "%smax", var);
  if((getval = Tcl_GetVar (interp, varname, TCL_LEAVE_ERR_MSG)) == NULL)
    fprintf(stderr,"oops, cannot get variable %s\n", varname);
  else
    Tcl_GetDouble (interp, getval, &maxslider);

  if (val > maxslider)
    {
      if(Tcl_SetVar (interp, varname, valstring, TCL_LEAVE_ERR_MSG) == NULL)
	fprintf(stderr,"oops, cannot set variable %s\n", varname);
    }

  if((fabs(val) < 0.01) && (val != 0.0))
    sprintf (valstring, "%f", val);
  else
    sprintf (valstring, "%.2f", val);

  /* Now, set the value */
  if(Tcl_SetVar (interp, var, valstring, TCL_LEAVE_ERR_MSG) == NULL)
    fprintf(stderr,"oops, cannot set variable %s\n", var);
  getval = Tcl_GetVar (interp, var, TCL_LEAVE_ERR_MSG);

  /*  fprintf(stderr,"Set value: %s , get value %s\n",valstring,getval);*/

  return;
} /* setslider */


/********************************************/
/*cmdReadShy:
  a Tcl-command that loads a .shy 
  (Datafile for Shelly-Lib)                 */
/********************************************/
int
cmdReadShy (ClientData clientData, Tcl_Interp * interp,
	    int argc, char *argv[])
{
  FILE *fileptr = NULL;

  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"readShy ?filename?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }


  if ((argv[1] != NULL) && ((fileptr = fopen (argv[1], "r")) == NULL))
    {
      Tcl_SetResult (interp,
		     "unable to open file",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  SLFreeGC(&arg);
  arg.GCurve = NULL;

  ReadInfile (&arg, fileptr);

  /* set Tcl-Vars */
  setslider (interp, "alpha", arg.alpha * 180.0 / pi);
  setslider (interp, "beta", arg.beta * 180.0 / pi);
  setslider (interp, "phi", arg.phi * 180.0 / pi);
  setslider (interp, "my", arg.my * 180.0 / pi);
  setslider (interp, "omega", arg.omega * 180.0 / pi);

  setslider (interp, "_A", arg.A);
  setslider (interp, "a", arg.a);
  setslider (interp, "b", arg.b);

  setslider (interp, "scale", arg.scale);
  setslider (interp, "omin", arg.omin * 180.0 / pi);
  setslider (interp, "omax", arg.omax * 180.0 / pi);
  setslider (interp, "od", arg.od * 180.0 / pi);
  setslider (interp, "smin", arg.smin * 180.0 / pi);
  setslider (interp, "smax", arg.smax * 180.0 / pi);
  setslider (interp, "sd", arg.sd * 180.0 / pi);

  setslider (interp, "_P1", arg.P * 180.0 / pi);
  setslider (interp, "_L1", arg.L);
  setslider (interp, "_N1", arg.N);
  setslider (interp, "_W11", arg.W1 * 180.0 / pi);
  setslider (interp, "_W21", arg.W2 * 180.0 / pi);
  setslider (interp, "_Nstart1", arg.Nstart * 180.0 / pi);

  setslider (interp, "_P2", arg.P2 * 180.0 / pi);
  setslider (interp, "_L2", arg.L2);
  setslider (interp, "_N2", arg.N2);
  setslider (interp, "_W12", arg.W12 * 180.0 / pi);
  setslider (interp, "_W22", arg.W22 * 180.0 / pi);
  setslider (interp, "_Nstart2", arg.Nstart2 * 180.0 / pi);
  setslider (interp, "_Off2", arg.Off2 * 180.0 / pi);

  setslider (interp, "_P3", arg.P3 * 180.0 / pi);
  setslider (interp, "_L3", arg.L3);
  setslider (interp, "_N3", arg.N3);
  setslider (interp, "_W13", arg.W13 * 180.0 / pi);
  setslider (interp, "_W23", arg.W23 * 180.0 / pi);
  setslider (interp, "_Nstart3", arg.Nstart3 * 180.0 / pi);
  setslider (interp, "_Off3", arg.Off3 * 180.0 / pi);

  return TCL_OK;
}				/* cmdReadShy */


/********************************************/
/*cmdWriteShy:
  a Tcl-command that writes a .shy 
  (Datafile for Shelly-Lib)                 */
/********************************************/
int
cmdWriteShy (ClientData clientData, Tcl_Interp * interp,
	     int argc, char *argv[])
{
  FILE *fileptr = NULL;
  struct SLgcpoint *gcp = NULL;

  if (argc != 2)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"writeShy ?filename?\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if ((fileptr = fopen (argv[1], "w")) == NULL)
    {
      Tcl_SetResult (interp,
		     "unable to open file",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  fetchSLarg (interp);

  fprintf (fileptr, "# Datafile for ShellyLib, the ShellShapeGenerator.\n");
  fprintf (fileptr, "# Generated by Shell-Laboratory.\n");

  fprintf (fileptr, "alpha:%f\n", arg.alpha * 180.0 / pi);
  fprintf (fileptr, "beta:%f\n", arg.beta * 180.0 / pi);
  fprintf (fileptr, "phi:%f\n", arg.phi * 180.0 / pi);
  fprintf (fileptr, "my:%f\n", arg.my * 180.0 / pi);
  fprintf (fileptr, "omega:%f\n", arg.omega * 180.0 / pi);

  fprintf (fileptr, "omin:%f\n", arg.omin * 180.0 / pi);
  fprintf (fileptr, "omax:%f\n", arg.omax * 180.0 / pi);
  fprintf (fileptr, "od:%f\n", arg.od * 180.0 / pi);
  fprintf (fileptr, "smin:%f\n", arg.smin * 180.0 / pi);
  fprintf (fileptr, "smax:%f\n", arg.smax * 180.0 / pi);
  fprintf (fileptr, "sd:%f\n", arg.sd * 180.0 / pi);

  fprintf (fileptr, "A:%f\n", arg.A);
  fprintf (fileptr, "a:%f\n", arg.a);
  fprintf (fileptr, "b:%f\n", arg.b);

  fprintf (fileptr, "P:%f\n", arg.P * 180.0 / pi);
  fprintf (fileptr, "L:%f\n", arg.L);
  fprintf (fileptr, "N:%f\n", arg.N);
  fprintf (fileptr, "W1:%f\n", arg.W1 * 180.0 / pi);
  fprintf (fileptr, "W2:%f\n", arg.W2 * 180.0 / pi);
  fprintf (fileptr, "Nstart:%f\n", arg.Nstart * 180.0 / pi);

  fprintf (fileptr, "P2:%f\n", arg.P2 * 180.0 / pi);
  fprintf (fileptr, "L2:%f\n", arg.L2);
  fprintf (fileptr, "N2:%f\n", arg.N2);
  fprintf (fileptr, "W12:%f\n", arg.W12 * 180.0 / pi);
  fprintf (fileptr, "W22:%f\n", arg.W22 * 180.0 / pi);
  fprintf (fileptr, "Off2:%f\n", arg.Off2 * 180.0 / pi);
  fprintf (fileptr, "Nstart2:%f\n", arg.Nstart2 * 180.0 / pi);

  fprintf (fileptr, "P3:%f\n", arg.P3 * 180.0 / pi);
  fprintf (fileptr, "L3:%f\n", arg.L3);
  fprintf (fileptr, "N3:%f\n", arg.N3);
  fprintf (fileptr, "W13:%f\n", arg.W13 * 180.0 / pi);
  fprintf (fileptr, "W23:%f\n", arg.W23 * 180.0 / pi);
  fprintf (fileptr, "Off3:%f\n", arg.Off3 * 180.0 / pi);
  fprintf (fileptr, "Nstart3:%f\n", arg.Nstart3 * 180.0 / pi);

  fprintf (fileptr, "Scale:%f\n", arg.scale);

  fprintf (fileptr, "SkipOStart:%f\n", arg.SkipOStart * 180.0 / pi);
  fprintf (fileptr, "SkipOEnd:%f\n", arg.SkipOEnd * 180.0 / pi);
  fprintf (fileptr, "SkipSStart:%f\n", arg.SkipSStart * 180.0 / pi);
  fprintf (fileptr, "SkipSEnd:%f\n", arg.SkipSEnd * 180.0 / pi);

  fprintf (fileptr, "Hdo:%f\n", arg.heightdiffo);
  fprintf (fileptr, "Hds:%f\n", arg.heightdiffs);
  fprintf (fileptr, "Scano:%f\n", arg.Scano);
  fprintf (fileptr, "Scans:%f\n", arg.Scans);

  gcp = arg.GCurve;
  while(gcp)
    {
      fprintf (fileptr, "GCx:%f\n", gcp->x);
      fprintf (fileptr, "GCy:%f\n\n", gcp->y);
      gcp = gcp->next;
    }


  fclose(fileptr);

  return TCL_OK;
}				/* cmdWriteShy */


/********************************************/
/*cmdWrite3D:
  a Tcl-command that writes a 3D Object 
  from the current shell                    */
/********************************************/
int
cmdWrite3D (ClientData clientData, Tcl_Interp * interp,
	    int argc, char *argv[])
{
  FILE *fp = NULL;

  struct SLtindex *tind = NULL;
  struct SLlindex *lind = NULL;
  struct SLpindex *pind = NULL;

  if (argc != 3)
    {
      Tcl_SetResult (interp,
		     "wrong # args: should be \"write3D type filename\"",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  if ((fp = fopen (argv[2], "w")) == NULL)
    {
      Tcl_SetResult (interp,
		     "unable to open file",
		     TCL_STATIC);
      return TCL_ERROR;
    }

  /* get type of file */
  arg.output = atoi (argv[1]);

  switch (arg.output)
    {
    case 1:
      SLCreateIndex (&arg, shell, &tind, &lind, &pind);
      WriteRPLFile (&arg, shell, pind, tind, fp);
      break;

    case 5:
      WriteRAWFile (&arg, shell, fp);
      break;

    case 7:
      SLCreateIndex (&arg, shell, &tind, &lind, &pind);
      WriteX3DFile (&arg, shell, pind, lind, tind, fp);
      break;

    case 6:
      SLCreateIndex (&arg, shell, &tind, &lind, &pind);
      WriteT3DFile (&arg, shell, pind, lind, tind, fp);
      break;

    case 8:
      SLCreateIndex (&arg, shell, &tind, &lind, &pind);
      WriteTSPFile (&arg, shell, pind, lind, tind, fp);
      break;

    case 3:
      SLCreateIndex (&arg, shell, &tind, &lind, &pind);
      WriteSCDFile (&arg, shell, pind, lind, tind, fp);
      break;

    case 4:
      WriteSCDBFile (&arg, shell, fp);
      break;

    case 2:
      WriteRIBFile (&arg, shell, fp);
      break;

    case 0:
      WriteBEZFile (&arg, shell, fp);
      break;
    }

  fclose(fp);

  SLFreePindex (pind);
  SLFreeLindex (lind);
  SLFreeTindex (tind);

  return TCL_OK;
}				/* cmdWrite3D */


/********************************************/
/*Tcl_AppInit:                              */
/********************************************/
int
Tcl_AppInit (Tcl_Interp * interp)
{
  if (Tcl_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  if (Tk_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  if (Togl_Init (interp) == TCL_ERROR)
    return TCL_ERROR;

  Togl_CreateFunc (create_cb);
  Togl_DisplayFunc (display_cb);
  Togl_ReshapeFunc (reshape_cb);

  Togl_CreateCommand ("setXrot", setXrot_cb);
  Togl_CreateCommand ("setYrot", setYrot_cb);
  Togl_CreateCommand ("setZrot", setZrot_cb);
  Togl_CreateCommand ("zoomIn", zoomin_cb);
  Togl_CreateCommand ("zoomOut", zoomout_cb);
  Togl_CreateCommand ("moveZ", movez_cb);
  Togl_CreateCommand ("update3DView", update3DView_cb);
  Togl_CreateCommand ("setMode", setmode_cb);
  Togl_CreateCommand ("drawSys", drawsys_cb);
  Togl_CreateCommand ("resetView", resetview_cb);

  Tcl_CreateCommand (interp, "createShell", cmdCreateShell,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "deleteShell", cmdDeleteShell,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "readShy", cmdReadShy,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "writeShy", cmdWriteShy,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
  Tcl_CreateCommand (interp, "write3D", cmdWrite3D,
		     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  return TCL_OK;
} /* Tcl_AppInit */

/********************************************/
/*main:                                     */
/********************************************/
int
main (int argc, char **argv)
{
  Tk_Main (argc, argv, Tcl_AppInit);
  return 0;
} /* main */


