/*
    pmosc - recreates an image out of smaller image tiles
    Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<math.h>
#include"image.h"
#include"misc.h"

static void badTGA(const char *path,const char *msg)
{
	fprintf(stderr,"%s %s\nonly uncompressed unmapped non-interleaved "
		"24 bit RGB TGA format image files\nwith (0,0) lower "
		"left-hand origins and non-zero widths and "
		"heights are supported\n",path,msg);
	exit(1);
}

static void readTGAheader(const char *path,FILE *stream,image_t *img)
{
	unsigned char idfs,data;

	xfread(&idfs,1,1,stream);

	xfread(&data,1,1,stream);
	if(data) badTGA(path,"has a color map");

	xfread(&data,1,1,stream);
	if(data != 2) badTGA(path,"is not in uncompressed RGB format");

	xfseek(stream,5,SEEK_CUR);

	xfread(&data,1,1,stream);
	if(data) badTGA(path,"does not have a (0,0) origin");

	xfread(&data,1,1,stream);
	if(data) badTGA(path,"does not have a (0,0) origin");

	xfread(&data,1,1,stream);
	if(data) badTGA(path,"does not have a (0,0) origin");

	xfread(&data,1,1,stream);
	if(data) badTGA(path,"does not have a (0,0) origin");

	img->width = 0;
	xfread(&data,1,1,stream);
	img->width |= data;
	xfread(&data,1,1,stream);
	img->width |= (data << 8);
	if(!img->width) badTGA(path,"has a width of 0");

	img->height = 0;
	xfread(&data,1,1,stream);
	img->height |= data;
	xfread(&data,1,1,stream);
	img->height |= (data << 8);
	if(!img->height) badTGA(path,"has a height of 0");

	xfread(&data,1,1,stream);
	if(data != 24) badTGA(path,"does not in 24 bits per pixel");

	xfread(&data,1,1,stream);
	if(data & 0x07) badTGA(path,"does not have 0 attribute bits");
	if(data & 0x10) badTGA(path,"has reserved bit non-zero");
	if(data & 0x20) badTGA(path,"has upper left-hand corner origin");
	if(data & 0xc0) badTGA(path,"is interleaved");

	xfseek(stream,idfs,SEEK_CUR);
}

static void readTGAdata(FILE *stream,image_t *img)
{
	unsigned long x,y;

	for(y=0;y<img->height;y++)
	{
		for(x=0;x<img->width;x++)
		{
			xfread(&img->data[x + y*img->width].b,1,1,stream);
			xfread(&img->data[x + y*img->width].g,1,1,stream);
			xfread(&img->data[x + y*img->width].r,1,1,stream);
		}
	}
}

image_t *imageRead(const char *path)
{
	FILE *stream;
	image_t *img;

	stream = xfopen(path,"rb");
	img = (image_t*)xmalloc(sizeof(image_t));

	readTGAheader(path,stream,img);
	img->data = (rgb_t*)xmalloc(img->width * img->height * sizeof(rgb_t));

	readTGAdata(stream,img);
	fclose(stream);

	return img;
}

void imageWrite(const char *path,const image_t *img)
{
	FILE *stream;
	unsigned long x,y;
	unsigned char header[18];

	stream = xfopen(path,"wb");

	memset(header,0,sizeof(header));
	header[2] = 2;
	header[12] = img->width & 0xff;
	header[13] = (img->width & 0xff00) >> 8;
	header[14] = img->height & 0xff;
	header[15] = (img->height & 0xff00) >> 8;
	header[16] = 24;
	xfwrite(header,sizeof(header),1,stream);

	for(y=0;y<img->height;y++)
	{
		for(x=0;x<img->width;x++)
		{
			xfwrite(&img->data[x + y*img->width].b,1,1,stream);
			xfwrite(&img->data[x + y*img->width].g,1,1,stream);
			xfwrite(&img->data[x + y*img->width].r,1,1,stream);
		}
	}

	fclose(stream);
}

void imageFree(image_t *img)
{
	free(img->data);
	free(img);
}

static rgb_t average(const image_t *img,
	double x1,double y1,double x2,double y2)
{
	rgb_t c;
	unsigned long i;
	double x,y,xn,yn;
	double r=0.0,g=0.0,b=0.0;
	double n=0.0;

	for(y=y1;y<floor(y2);y+=yn)
	{
		yn = MIN(floor(y + 1.0),y2) - y;

		for(x=x1;x<floor(x2);x+=xn)
		{
			xn = MIN(floor(x + 1.0),x2) - x;

			i = (unsigned long)floor(x);
			i += img->width * (unsigned long)floor(y);

			r += xn * yn * img->data[i].r;
			g += xn * yn * img->data[i].g;
			b += xn * yn * img->data[i].b;

			n += xn * yn;
		}
	}

	c.r = (unsigned char)floor(0.5 + (r / n));
	c.g = (unsigned char)floor(0.5 + (g / n));
	c.b = (unsigned char)floor(0.5 + (b / n));

	return c;
}

image_t* imageScale(const image_t *img,
	unsigned long width,unsigned long height)
{
	image_t *scaled;
	unsigned long x,y;
	double xf,yf;

	scaled = (image_t*)xmalloc(sizeof(image_t));
	scaled->data = (rgb_t*)xmalloc(width * height * sizeof(rgb_t));

	scaled->width = width;
	scaled->height = height;

	xf = (double)img->width / (double)scaled->width;
	yf = (double)img->height / (double)scaled->height;

	for(y=0;y<scaled->height;y++)
	{
		for(x=0;x<scaled->width;x++)
		{
			scaled->data[x + scaled->width * y] =
				average(img,x * xf,y * yf,
					(x + 1) * xf,(y + 1) * yf);
		}
	}

	return scaled;
}

image_t* imageLoad(const char *path,unsigned long width,unsigned long height)
{
	image_t *raw,*img;
	raw = imageRead(path);
	img = imageScale(raw,width,height);
	imageFree(raw);
	return img;
}

image_t* imageDup(const image_t *img)
{
	image_t *imgdup;
	unsigned long size;

	imgdup = (image_t*)xmalloc(sizeof(image_t));
	imgdup->width = img->width;
	imgdup->height = img->height;

	size = img->width * img->height * sizeof(rgb_t);
	imgdup->data = (rgb_t*)xmalloc(size);
	memcpy(imgdup->data,img->data,size);

	return imgdup;
}

void imageCopy(image_t *dest,const image_t *src,
		unsigned long dx,unsigned long dy,
		unsigned long sx,unsigned long sy,
		unsigned long width,unsigned long height)
{
	unsigned long x,y;

	for(y=0;y<height;y++)
	{
		for(x=0;x<width;x++)
		{
			dest->data[x+dx + (y+dy) * dest->width]
				= src->data[x+sx + (y+sy) * src->width];
		}
	}
}

