/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: StateDefVisitor.cc,v $
$Revision: 1.9 $
$Date: 1997/07/02 04:45:26 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: StateDefVisitor.cc,v $
Revision 1.9  1997/07/02 04:45:26  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.8  1997/07/01 03:54:34  andrewm
Changes in the way code is generated.  Now there is an initial event
function that must be called after construction but before any
other events are triggered.  This is to help integration of generated
classes into a class hierarchy.

Revision 1.7  1997/05/20 05:15:37  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.6  1997/05/15 04:14:49  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.5  1997/01/12 02:17:48  andrewm
Changed the compiler so that state machine is defined as nested
classes in order to prevent polluting the global class name space.

Revision 1.4  1996/12/27 02:11:02  andrewm
Checkpoint before attempting to restructure the compiler to generate
nested class declarations.

Revision 1.3  1996/12/26 05:55:30  andrewm
Checkpoint, the compiler is working again.

Revision 1.2  1996/10/01 04:39:38  andrewm
checkpoint and revision

// Revision 1.1  1996/09/22  01:18:50  andrewm
// pre-alpha release
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "StateDefVisitor.h"
#include "MachineGroup.h"
#include "Smachine.h"
#include "MachEvent.h"
#include "State.h"
#include "Transition.h"
#include "Parameter.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: StateDefVisitor.cc,v $ $Revision: 1.9 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

StateDefVisitor::
StateDefVisitor(
	string& filename,
	ostream& stream,
	bool line_directives) :
		_filename(filename),
		_codefile(stream),
		_line_directives(line_directives)
{
}

void StateDefVisitor::
visit(
	Smachine *machine)
{
	_prefix_name = machine->name().var_name() ;
	_event_class = _prefix_name + "::" + _prefix_name + "_Event" ;

	_codefile << "// State definition for " <<
		_prefix_name << endl ;

	State *initial_state = machine->initial_state() ;
	if (initial_state)
	{
		string state_name = initial_state->name().var_name() ;
		string init_state_class =
			_prefix_name + "::" + _prefix_name + "_" + state_name + "_State" ;

		_codefile <<
			"void " << init_state_class << "::\n"
			"initial_event(\n" <<
				'\t' << _prefix_name << "& machine,\n" <<
				"\tconst " << _event_class << "& event" ;
		const State::ParameterList& param_list =
			initial_state->parameter_list() ;
		for (State::ParameterListConstIter param_iter = param_list.begin() ;
			param_iter != param_list.end() ; param_iter++)
		{
			Parameter *param = *param_iter ;
			param->accept_visitor(*this) ;
		}
		_codefile <<
			")\n"
			"{\n"
				"\t _" << state_name << ".action(machine, event" ;
		for (State::ParameterListConstIter param_iter = param_list.begin() ;
			param_iter != param_list.end() ; param_iter++)
		{
			Parameter *param = *param_iter ;
			_codefile << ", " << param->name() ;
		}
		_codefile <<
			") ;\n"
			"}" << endl ;
	}

	Smachine::StateList& state_list = machine->state_list() ;
	for (Smachine::StateListIter st_iter = state_list.begin() ;
		st_iter != state_list.end() ; st_iter++)
	{
		State *st = *st_iter ;
		st->accept_visitor(*this) ;
	}
}

void StateDefVisitor::
visit(
	State *state)
{
	string state_name = state->name().var_name() ;
	_state_class =
		_prefix_name + "::" + _prefix_name + "_" + state_name + "_State" ;

	const State::TransitionList& trans_list = state->transition_list() ;
	for (State::TransitionListConstIter tr_iter = trans_list.begin() ;
		tr_iter != trans_list.end() ; tr_iter++)
	{
		Transition *tr = *tr_iter ;
		tr->accept_visitor(*this) ;
	}

	_codefile <<
			"void " << _state_class << "::\n"
			"action(\n" << 
				'\t' << _prefix_name << "& machine,\n" <<
				"\tconst " << _event_class << "& event" ;
	const State::ParameterList& param_list = state->parameter_list() ;
	for (State::ParameterListConstIter param_iter = param_list.begin() ;
		param_iter != param_list.end() ; param_iter++)
	{
		Parameter *param = *param_iter ;
		param->accept_visitor(*this) ;
	}
	_codefile <<
		")\n"
		"{\n"
			"\tlog(machine, event, _" << state_name << ") ;\n"
			"\tset_state(machine, &_" << state_name << ") ;\n"
			"\t{" << endl ;

	const ChioTerm& code_string = state->code() ;
	int line_no = code_string.lineno() ;
	if (_line_directives && line_no > 0)
		_codefile <<
			"#\tline " << line_no << " \"" << _filename << '"' << endl ;

	_codefile <<
				(const string&)code_string <<
			"\t}\n"
		"}" << endl ;

	_codefile <<
		_state_class << ' ' << _prefix_name << "::"
			<< _prefix_name << "_State::_" << state_name <<
			" ;" << endl ;
}

void StateDefVisitor::
visit(
	Transition *transition)
{
	MachEvent *event = transition->event() ;
	string event_name = event->name().var_name() ;
	string dest_name = transition->destination()->name().var_name() ;

	bool is_pseudo_state = transition->destination()->is_pseudo_state() ;
	string action_func(is_pseudo_state ?
		dest_name + "_action" : "_" + dest_name + ".action") ;

	_codefile <<
		"void " << _state_class << "::\n" <<
		event_name << "(\n" << 
			'\t' << _prefix_name << "& machine,\n" <<
			"\tconst " << _event_class << "& event" ;

	MachEvent::ParameterList& param_list = event->parameter_list() ;
	for (MachEvent::ParameterListIter param_iter = param_list.begin() ;
		param_iter != param_list.end() ; param_iter++)
	{
		Parameter *param = *param_iter ;
		param->accept_visitor(*this) ;
	}

	_codefile <<
		")\n"
		"{\n"
			"\t " << action_func << "(machine, event" ;
	if (!is_pseudo_state)
		for (MachEvent::ParameterListIter param_iter = param_list.begin() ;
			param_iter != param_list.end() ; param_iter++)
		{
			Parameter *param = *param_iter ;
			_codefile << ", " << param->name() ;
		}
	_codefile <<
		") ;\n"
		"}" << endl ;
}

void StateDefVisitor::
visit(
	Parameter *parameter)
{
	_codefile << ",\n\t" << (const string&)parameter->type() << ' '
		<< (const string&)parameter->name() ;
}
