/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: StateClassDefVisitor.cc,v $
$Revision: 1.11 $
$Date: 1997/07/02 04:45:25 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: StateClassDefVisitor.cc,v $
Revision 1.11  1997/07/02 04:45:25  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.10  1997/07/01 03:54:34  andrewm
Changes in the way code is generated.  Now there is an initial event
function that must be called after construction but before any
other events are triggered.  This is to help integration of generated
classes into a class hierarchy.

Revision 1.9  1997/05/20 05:15:37  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.8  1997/05/15 04:14:48  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.7  1997/03/04 06:32:59  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.6  1997/01/12 02:17:47  andrewm
Changed the compiler so that state machine is defined as nested
classes in order to prevent polluting the global class name space.

Revision 1.5  1997/01/11 23:55:34  andrewm
Removed the inclusion of "Iterator.h".  This is no longer used
since the conversion to use STL.

Revision 1.4  1996/12/27 02:11:02  andrewm
Checkpoint before attempting to restructure the compiler to generate
nested class declarations.

Revision 1.3  1996/12/26 05:55:29  andrewm
Checkpoint, the compiler is working again.

Revision 1.2  1996/10/01 04:39:38  andrewm
checkpoint and revision

// Revision 1.1  1996/09/22  01:18:50  andrewm
// pre-alpha release
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <assert.h>

#include "StateClassDefVisitor.h"
#include "MachineGroup.h"
#include "Smachine.h"
#include "MachEvent.h"
#include "State.h"
#include "Parameter.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: StateClassDefVisitor.cc,v $ $Revision: 1.11 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

StateClassDefVisitor::
StateClassDefVisitor(
	string& filename,
	ostream& stream,
	bool line_directives) :
		_filename(filename),
		_codefile(stream),
		_line_directives(line_directives)
{
}

void StateClassDefVisitor::
visit(
	Smachine *machine)
{
	string prefix_name(machine->name().var_name()) ;
	string event_class(prefix_name + "::" + prefix_name + "_Event") ;
	string state_class(prefix_name + "::" + prefix_name + "_State") ;

	_codefile << "// Abstract State definition for " <<
		prefix_name << endl ;

	_codefile <<
		"ostream&\n"
		"operator <<(\n"
			"\tostream& s,\n"
			"\t" << state_class << "& state)\n"
		"{\n"
			"\treturn s << state.name() ;\n"
		"}" << endl ;

	_codefile <<
		"void " << state_class << "::\n"
		"error_action(\n" <<
			'\t' << prefix_name << "& machine,\n" <<
			"\tconst " << event_class << "& event)\n"
		"{\n"
			"\tif (machine.log())\n"
				"\t\tcerr << \"On \\\"\" << event << \"\\\", \\\"\" << "
				"*machine.current_state()\n"
					"\t\t\t<< \"\\\" entered the error state.\" << endl ;"
				<< endl ;

	State *error_state = machine->find_state("error") ;
	assert(error_state != 0) ;
	const ChioTerm& error_action = error_state->code() ;
	if (error_action.empty())
	{
		_codefile <<
			"\tcerr << \"Event \\\"\" << event\n"
	"\t\t<< \"\\\" caused the error action to be entered from state \\\"\"\n"
				"\t\t<< *machine.current_state() << \"\\\".\" << endl ;\n"
			"\texit(1) ;" << endl ;
	}
	else
	{
		if (_line_directives)
			_codefile <<
				"#\tline " << error_action.lineno()
				<< " \"" << _filename << '"' << endl ;
		_codefile << "\t{\n" << (const string&)error_action << "\t}" << endl ;
	}

	_codefile <<
		"}" << endl ;

	_codefile <<
		"void " << state_class << "::\n"
		"ignore_action(\n" <<
			'\t' << prefix_name << "& machine,\n" <<
			"\tconst " << event_class << "& event)\n"
		"{\n"
			"\tif (machine.log())\n"
				"\t\tcerr << \"On \\\"\" << event << \"\\\", \\\"\" << "
				"*machine.current_state()\n"
					"\t\t\t<< \"\\\" ignored the event.\" << endl ;"
				<< endl ;

	State *ignore_state = machine->find_state("ignore") ;
	assert(ignore_state != 0) ;
	const ChioTerm& ignore_action = ignore_state->code() ;
	if (ignore_action.empty())
	{
		_codefile <<
	"\tcerr << \"Event \\\"\" << event << \"\\\" is ignored in state \\\"\"\n"
				"\t\t<< *machine.current_state() << \"\\\".\" << endl ;"
				<< endl ;
	}
	else
	{
		if (_line_directives)
			_codefile <<
				"#\tline " << ignore_action.lineno()
				<< " \"" << _filename << '"' << endl ;
		_codefile << "\t{\n" << (const string&)ignore_action << "\t}" << endl ;
	}

	_codefile <<
		"}" << endl ;

	_codefile <<
		"void " << state_class << "::\n"
		"log(\n" <<
			'\t' << prefix_name << "& machine,\n" <<
			"\tconst " << event_class << "& event,\n" <<
			'\t' << state_class << "& new_state)\n"
		"{\n"
			"\tif (machine.log())\n"
				"\t\tcerr << \"On \\\"\" << event << \"\\\", \\\"\""
					" << *machine.current_state()\n"
			"\t\t\t<< \"\\\" -> \\\"\" << new_state << \"\\\".\" << endl ;\n"
		"}" << endl ;
}
