/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: XGraphicRenderer.cc,v $
$Revision: 1.8 $
$Date: 1997/07/02 04:45:20 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: XGraphicRenderer.cc,v $
Revision 1.8  1997/07/02 04:45:20  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.7  1997/06/21 02:21:35  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.6  1997/06/15 00:43:57  andrewm
Another checkpoint.  Reworked the way text is specfied to be drawn
for the benefit of the post script renderer.

Revision 1.5  1997/05/31 21:12:44  andrewm
Checkpoint.  Things are working well.

Revision 1.4  1997/03/18 06:51:05  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.3  1997/03/12 03:13:09  andrewm
Checkpoint.  Things are working rather well.

Revision 1.2  1997/02/23 23:44:17  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.1  1997/01/23 06:20:57  andrewm
Checkpoint as base and graphics classes are operating together.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "XGraphicRenderer.h"
#include "CSutils.h"
#include <Xm/Xm.h>
#include <math.h>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: XGraphicRenderer.cc,v $ $Revision: 1.8 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/
XGraphicRenderer::
XGraphicRenderer() :
		_widget(0),
		_display(0),
		_screen(0),
		_window(0),
		_gc(0),
		_clear_gc(0),
		_render(0),
		_width(0),
		_height(0),
		_expose_region(XCreateRegion())
{
}

XGraphicRenderer::
~XGraphicRenderer()
{
	if (_gc) XFreeGC(_display, _gc) ;
	if (_clear_gc) XFreeGC(_display, _clear_gc) ;
	if (_expose_region) XDestroyRegion(_expose_region) ;
}

void XGraphicRenderer::
set_context(
	Widget widget,
	XmRenderTable rt)
{
	_widget = widget ;
	_display = XtDisplay(_widget) ;
	_screen = XtScreen(_widget) ;
	XtVaGetValues(_widget,
		XmNwidth, &_width,
		XmNheight, &_height,
		NULL) ;
	_render = rt ;

	_width_pixel_per_mm =
		float(DisplayWidth(_display, DefaultScreen(_display))) /
		float(DisplayWidthMM(_display, DefaultScreen(_display))) ;
	_height_pixel_per_mm = 
		float(DisplayHeight(_display, DefaultScreen(_display))) /
		float(DisplayHeightMM(_display, DefaultScreen(_display))) ;
}

void XGraphicRenderer::
realize_context()
{
	int foreground = BlackPixelOfScreen(_screen) ;
	int background = WhitePixelOfScreen(_screen) ;
	XtVaGetValues(_widget,
		XmNforeground, &foreground,
		XmNbackground, &background,
		NULL) ;

	assert(_display != NULL) ;
	_window = XtWindow(_widget) ;
	assert(_window != 0) ;

	XGCValues gcval ;
	gcval.function = GXxor ;
	gcval.foreground = foreground ^ background ;
	gcval.background = background ;
	_gc = XCreateGC(_display, _window,
		GCFunction | GCForeground | GCBackground, &gcval) ;

	gcval.foreground = background ;
	gcval.background = background ;
	_clear_gc = XCreateGC(_display, _window,
		GCForeground | GCBackground, &gcval) ;
}

Point XGraphicRenderer::
scale_to_mm(
	const Point& point)
{
	Point scaled ;
	scaled.x() = point.x() / _width_pixel_per_mm ;
	scaled.y() = point.y() / _height_pixel_per_mm ;
	return scaled ;
}

void XGraphicRenderer::
circle(
	const Circle& circle)
{
	circular_arc(circle, 0.0, 2.0 * M_PI) ;
}

void XGraphicRenderer::
circular_arc(
	const Circle& circle,
	float start,
	float extent)
{
	Circle scaled(scale_to_pixels(circle)) ;
	float radius = scaled.radius() ;
	int x = int(scaled.center().x() - radius) ;
	int y = int(scaled.center().y() - radius) ;
	int diameter = int(radius * 2.0) ;
	int start_deg = int(start * 180.0 / M_PI) * _arc_scale_factor ;
	int extent_deg = int(extent * 180.0 / M_PI) * _arc_scale_factor ;
	XDrawArc(_display, _window, _gc, x, y, diameter, diameter,
		start_deg, extent_deg) ;
}

void XGraphicRenderer::
rect(
	const Rectangle& rectangle)
{
	Rectangle scaled(scale_to_pixels(rectangle)) ;
	XDrawRectangle(_display, _window, _gc,
		(int)scaled.origin().x(), (int)scaled.origin().y(),
		(unsigned int)scaled.width(), (unsigned int)scaled.height()) ;
}

void XGraphicRenderer::
fillrect(
	const Rectangle& rectangle)
{
	Rectangle scaled(scale_to_pixels(rectangle)) ;
	XFillRectangle(_display, _window, _gc,
		int(scaled.origin().x()), int(scaled.origin().y()),
		(unsigned int)scaled.width(), (unsigned int)scaled.height()) ;
}

void XGraphicRenderer::
line(
	const Point& begin_pt,
	const Point& end_pt)
{
	Point scaled_begin(scale_to_pixels(begin_pt)) ;
	Point scaled_end(scale_to_pixels(end_pt)) ;
	XDrawLine(_display, _window, _gc,
		int(scaled_begin.x()), int(scaled_begin.y()),
		int(scaled_end.x()), int(scaled_end.y())) ;
}

void XGraphicRenderer::
lines(
	const PointList& points)
{
	XPoint *xpts = cnv_points(points) ;
	int nxpts = points.size() ;
	XDrawLines(_display, _window, _gc, xpts, nxpts, CoordModeOrigin) ;
	delete [] xpts ;
}

void XGraphicRenderer::
fillpolygon(
	const PointList& points)
{
	XPoint *xpts = cnv_points(points) ;
	int nxpts = points.size() ;
	XFillPolygon(_display, _window, _gc, xpts, nxpts,
		Complex, CoordModeOrigin) ;
	delete [] xpts ;
}

void XGraphicRenderer::
text(
	const string& text,
	const Point& location,
	TextOrigin origin)
{
	XmString xm_text = TextToCS(text.c_str()) ;
	Point extent(X_text_extent(text)) ;
		// add a few pixels of margin for text, enough to cover
		// line widths and other concerns.
	float lw_adjust = 6.0 * line_width() * _width_pixel_per_mm ;
	extent.x() += lw_adjust ;
	extent.y() += lw_adjust ;
		// adjust the location to so that it translates between
		// the X coordinates and the desired origin of the text extent
	Point text_origin(scale_to_pixels(location)) ;
	switch (origin)
	{
		case Origin_Center:
			// Move 1/2 to the left and 1/2 down
			text_origin.x() -= extent.x() / 2 ;
			text_origin.y() -= extent.y() / 2 ;
			break ;

		case Origin_NorthWest:
			// This is the natural coordinate system so ...
			// Move 0 to the left and 0 down
			break ;

		case Origin_North:
			// Move 1/2 to the left and 0 down
			text_origin.x() -= extent.x() / 2 ;
			break ;

		case Origin_NorthEast:
			// Move 1 to the left and 0 down
			text_origin.x() -= extent.x() ;
			break ;

		case Origin_East:
			// Move 1 to the left and 1/2 down
			text_origin.x() -= extent.x() ;
			text_origin.y() -= extent.y() / 2 ;
			break ;

		case Origin_SouthEast:
			// Move 1 to the left and 1 down
			text_origin.x() -= extent.x() ;
			text_origin.y() -= extent.y() ;
			break ;

		case Origin_South:
			// Move 1/2 to the left and 1 down
			text_origin.x() -= extent.x() / 2 ;
			text_origin.y() -= extent.y() ;
			break ;

		case Origin_SouthWest:
			// Move 0 to the left and 1 down
			text_origin.y() -= extent.y() ;
			break ;

		case Origin_West:
			// Move 0 to the left and 1/2 down
			text_origin.y() -= extent.y() / 2 ;
			break ;
	}
	XmStringDraw(_display, _window, _render, xm_text, _gc,
		int(text_origin.x()), int(text_origin.y() + lw_adjust / 2.0),
		int(extent.x()), XmALIGNMENT_CENTER, XmDEFAULT_DIRECTION, NULL) ;
	XmStringFree(xm_text) ;
}

Point XGraphicRenderer::
text_extent(
	const string& text)
{
	return scale_to_mm(X_text_extent(text)) ;
}

float XGraphicRenderer::
line_width()
{
	unsigned long valuemask = GCLineWidth ;
	XGCValues values ;
	XGetGCValues(_display, _gc, valuemask, &values) ;
	int width = values.line_width ;
	if (width == 0)
		width = 1 ;

	return float(width / _width_pixel_per_mm) ;
}

GraphicRenderer::LineDrawingStyle XGraphicRenderer::
line_style(
	LineDrawingStyle new_style)
{
	unsigned long valuemask = GCLineStyle ;
	XGCValues values ;
	LineDrawingStyle old_style = LS_NORMAL ;
	XGetGCValues(_display, _gc, valuemask, &values) ;
	switch (values.line_style)
	{
	case LineSolid:
		old_style = LS_NORMAL ;
		break ;

	case LineOnOffDash:
	case LineDoubleDash:
		old_style = LS_DASHED ;
		break ;
	}

	values.line_style = LineSolid ;
	switch (new_style)
	{
	case LS_NORMAL:
		values.line_style = LineSolid ;
		break ;

	case LS_DASHED:
		values.line_style = LineOnOffDash ;
		break ;
	}
	XChangeGC(_display, _gc, valuemask, &values) ;
}

void XGraphicRenderer::
clear()
{
	if (XtIsRealized(_widget))
		XClearWindow(_display, _window) ;
}

void XGraphicRenderer::
clear_clip()
{
		/*
		Set up the region for next time.
		*/
	XDestroyRegion(_expose_region) ;
	_expose_region = XCreateRegion() ;
		/*
		Restore the graphics context to allow drawing anywhere
		on the window.
		*/
	XSetClipMask(_display, _gc, None) ;
}

void XGraphicRenderer::
add_rectangle_to_clip(
	const Rectangle& rect)
{
	XRectangle xrect ;

	xrect.x = short(rect.origin().x()) ;
	xrect.y = short(rect.origin().y()) ;
	xrect.width = (unsigned short)(rect.width()) ;
	xrect.height = (unsigned short)(rect.height()) ;
	XUnionRectWithRegion(&xrect, _expose_region, _expose_region) ;
}

void XGraphicRenderer::
set_clip()
{
	XSetRegion(_display, _gc, _expose_region) ;
}

Point XGraphicRenderer::
scale_to_pixels(
	const Point& point)
{
	Point scaled ;
	scaled.x() = point.x() * _width_pixel_per_mm ;
	scaled.y() = point.y() * _height_pixel_per_mm ;
	return scaled ;
}

Circle XGraphicRenderer::
scale_to_pixels(
	const Circle& circle)
{
	Circle scaled ;
	scaled.center() = scale_to_pixels(circle.center()) ;
	scaled.radius() = circle.radius() * _width_pixel_per_mm ;
	return scaled ;
}

Rectangle XGraphicRenderer::
scale_to_pixels(
	const Rectangle& rectangle)
{
	Rectangle scaled ;
	scaled.origin() = scale_to_pixels(rectangle.origin()) ;
	scaled.width() = rectangle.width() * _width_pixel_per_mm ;
	scaled.height() = rectangle.height() * _height_pixel_per_mm ;
	return scaled ;
}

Point XGraphicRenderer::
X_text_extent(
	const string& text)
{
	XmString xm_text = TextToCS(text.c_str()) ;
	Dimension text_width = 0 ;
	Dimension text_height = 0 ;
	XmStringExtent(_render, xm_text, &text_width, &text_height) ;
	XmStringFree(xm_text) ;
	return Point(text_width, text_height) ;
}

XPoint *XGraphicRenderer::
cnv_points(
	const PointList& points)
{
	int npoints = points.size() ;
	XPoint *pt_array = new XPoint[npoints] ;
	XPoint *pt_ptr = pt_array ;
	for (PointListConstIter pl_iter = points.begin() ;
		pl_iter != points.end() ; ++pl_iter)
	{
		Point p = scale_to_pixels(*pl_iter) ;
		pt_ptr->x = short(p.x()) ;
		pt_ptr->y = short(p.y()) ;
		pt_ptr++ ;
	}
	return pt_array ;
}
