/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: TransitionGraphic.cc,v $
$Revision: 1.20 $
$Date: 1997/07/02 04:45:20 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: TransitionGraphic.cc,v $
Revision 1.20  1997/07/02 04:45:20  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.19  1997/06/21 02:21:35  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.18  1997/06/15 00:43:57  andrewm
Another checkpoint.  Reworked the way text is specfied to be drawn
for the benefit of the post script renderer.

Revision 1.17  1997/06/06 04:34:29  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.16  1997/05/20 05:15:36  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.15  1997/03/18 06:51:05  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.14  1997/03/12 03:13:08  andrewm
Checkpoint.  Things are working rather well.

Revision 1.13  1997/03/04 06:32:58  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.12  1997/02/23 23:44:16  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.11  1997/02/08 04:37:44  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.10  1997/01/23 06:20:56  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.9  1996/11/27 01:25:54  andrewm
Another checkpoint before I go off and figure out what to do about
the graphical representations so that I'll be able to get some post script
output one of these days.

Revision 1.8  1996/11/14 06:19:54  andrewm
checkpoint

// Revision 1.7  1996/10/01  04:39:26  andrewm
// checkpoint and revision
//
// Revision 1.6  1996/09/22  01:18:34  andrewm
// pre-alpha release
//
// Revision 1.5  1996/08/18  17:58:13  andrewm
// checkpoint
//
// Revision 1.4  1996/08/05  01:25:56  andrewm
// checkpoint
//
// Revision 1.3  1996/07/27  20:57:09  andrewm
// checkpoint
//
// Revision 1.2  1996/07/15  01:21:01  andrewm
// checkpoint again
//
// Revision 1.1  1996/06/26  03:16:15  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <math.h>

#include "MachEvent.h"
#include "StateGraphic.h"
#include "TransitionGraphic.h"
#include "PageGroup.h"

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: TransitionGraphic.cc,v $ $Revision: 1.20 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const float TransitionGraphic::_arrow_length = 4.0 ; // in millimeters
const float TransitionGraphic::_arrow_width = 2.7 ; // in millimeters
const float TransitionGraphic::_highlight_width = 2.7 ; // in millimeters

	// 240 deg
const float LoopTransitionGraphic::_loop_extent = 4.0 * M_PI / 3.0 ;
	// -120 deg
const float LoopTransitionGraphic::_loop_angle_start = -(2.0 * M_PI / 3.0) ;

/*
FUNCTION DEFINITIONS
*/

TransitionGraphic::
TransitionGraphic(
	StateGraphic *parent,
	const string& event_name) :
		_parent(parent),
		_event_name(event_name),
		_origin(GraphicRenderer::Origin_Center),
		_arrow_points(3)
{
}

TransitionGraphic::
~TransitionGraphic()
{
	if (_parent)
		_parent->remove_transition_graphic(this) ;
}

LoopTransitionGraphic::
LoopTransitionGraphic(
	StateGraphic *parent,
	LoopTransition *transition) :
		TransitionGraphic(parent, transition->event()->name().c_str()),
		_transition(transition),
		_attach_angle(transition->attach_angle()),
		_loop_circle(transition->compute_loop_circle())
{
	_transition->attach(this) ;
	_selected = _transition->selected() ;
	compute_arrow_head() ;
	compute_text_location() ;
}

LoopTransitionGraphic::
~LoopTransitionGraphic()
{
	draw() ;
	if (_transition)
		_transition->detach(this) ;
}

void LoopTransitionGraphic::
update(
	const Subject *subject)
{
	assert(subject == _transition) ;
	draw() ;
	_attach_angle = _transition->attach_angle() ;
	_loop_circle = _transition->compute_loop_circle() ;
	_event_name = _transition->event()->name() ;
	_selected = _transition->selected() ;
	compute_arrow_head() ;
	compute_text_location() ;
	draw() ;
}

void LoopTransitionGraphic::
subject_deleted(
	Subject *subject)
{
	assert(subject == _transition) ;
	_transition = NULL ;
	delete this ;
}

void LoopTransitionGraphic::
draw()
{
	GraphicRenderer& render = GraphicRenderer::instance() ;
		// Draw the arc
		// compute the amount of arc that is occupied by the arrow head
		// and exclude that from the circular arc that is used to
		// represent the transition
	float arrow_radians = asin(_arrow_length / _loop_circle.radius()) ;
	render.circular_arc(_loop_circle,
		_attach_angle + _loop_angle_start + arrow_radians,
		_loop_extent - arrow_radians) ;
		// Draw the arrow head
	render.fillpolygon(_arrow_points) ;
		// Draw the text
	render.text(_event_name, _text_location, _origin) ;
	highlight() ;
}

void LoopTransitionGraphic::
highlight()
{
	if (_selected)
	{
		GraphicRenderer& render = GraphicRenderer::instance() ;
		float pt_angle = _attach_angle + _loop_angle_start ;
		const int highlight_points = 4 ;
		const float angle_incr = _loop_extent / (highlight_points - 1) ;
		for (int i = 0 ; i < highlight_points ; i++, pt_angle += angle_incr)
		{
			Point p = _loop_circle.on_circle_at_angle(pt_angle) ;
			p.x() -= _highlight_width / 2 ;
			p.y() -= _highlight_width / 2 ;
			Rectangle rect(p, _highlight_width, _highlight_width) ;
			render.fillrect(rect) ;
		}
	}
}

void LoopTransitionGraphic::
compute_arrow_head()
{
	Point arrow_head(_loop_circle.on_circle_at_angle(_attach_angle
		+ _loop_angle_start)) ;
	Line tangent(_loop_circle.tangent_line_at_point(arrow_head)) ;
	Point p(tangent.point_along(_arrow_length)) ;
	Line norm(tangent.perpendicular_through_point(p)) ;
	norm.normalize() ;
	_arrow_points[0] = arrow_head ;
	_arrow_points[1] = norm.point_along(_arrow_width) ;
	_arrow_points[2] = norm.point_along(-_arrow_width) ;
}

void LoopTransitionGraphic::
compute_text_location()
{
	const float deg_90 = M_PI / 2 ;
	const float deg_180 = M_PI ;
	const float deg_270 = 3 * M_PI / 2 ;
	const float deg_360 = 2 * M_PI ;

	_text_location = _loop_circle.on_circle_at_angle(_attach_angle) ;
	GraphicRenderer& render = GraphicRenderer::instance() ;
	float line_width = render.line_width() * 2.0 ;
	if (_attach_angle >= 0 && _attach_angle < deg_90)
	{
		_origin = GraphicRenderer::Origin_SouthWest ;
	}
	else if (_attach_angle >= deg_90 && _attach_angle < deg_180)
	{
		_origin = GraphicRenderer::Origin_SouthEast ;
	}
	else if (_attach_angle >= deg_180 && _attach_angle < deg_270)
	{
		_origin = GraphicRenderer::Origin_NorthEast ;
	}
	else if (_attach_angle >= deg_270 && _attach_angle < deg_360)
	{
		_origin = GraphicRenderer::Origin_NorthWest ;
	}
}

SegmentTransitionGraphic::
SegmentTransitionGraphic(
	StateGraphic *parent,
	SegmentTransition *transition) :
		TransitionGraphic(parent, transition->event()->name().c_str()),
		_transition(transition),
		_vertices(transition->compute_segments(1.0, _arrow_length))
{
	_transition->attach(this) ;
	_selected = _transition->selected() ;
	compute_arrow_head() ;
	compute_text_location() ;
}

SegmentTransitionGraphic::
~SegmentTransitionGraphic()
{
	draw() ;
	if (_transition)
		_transition->detach(this) ;
}

void SegmentTransitionGraphic::
update(
	const Subject *subject)
{
	assert(subject == _transition) ;
	draw() ;
	float line_width = GraphicRenderer::instance().line_width() ;
	_vertices = _transition->compute_segments(line_width, _arrow_length) ;
	_event_name = _transition->event()->name() ;
	_selected = _transition->selected() ;
	compute_arrow_head() ;
	compute_text_location() ;
	draw() ;
}

void SegmentTransitionGraphic::
subject_deleted(
	Subject *subject)
{
	assert(subject == _transition) ;
	_transition = NULL ;
	delete this ;
}

void SegmentTransitionGraphic::
draw()
{
	GraphicRenderer& render = GraphicRenderer::instance() ;
		// Draw the line segments
	render.lines(_vertices) ;
		// Draw the arrow head
	render.fillpolygon(_arrow_points) ;
		// Draw the text
	render.text(_event_name, _text_location, _origin) ;
	highlight() ;
}

void SegmentTransitionGraphic::
highlight()
{
	if (_selected)
	{
		GraphicRenderer& render = GraphicRenderer::instance() ;
		for (SegmentTransition::TransitionVerticesIter v_iter =
			_vertices.begin() ; v_iter != _vertices.end() ; ++v_iter)
		{
			Point vertex(*v_iter) ;
			vertex.x() -= _highlight_width / 2 ;
			vertex.y() -= _highlight_width / 2 ;
			Rectangle rect(vertex, _highlight_width, _highlight_width) ;
			render.fillrect(rect) ;
		}
	}
}

void SegmentTransitionGraphic::
compute_arrow_head()
{
	Point last_vertex(*(_vertices.end() - 1)) ;
	Point arrow_pt(_transition->destination()->circle().out_from_circle(
		last_vertex, 1.0)) ;
	Line direction(arrow_pt, last_vertex) ;
	Point cross(direction.point_along(_arrow_length)) ;
	Line norm(direction.perpendicular_through_point(cross)) ;
	norm.normalize() ;
	_arrow_points[0] = arrow_pt ;
	_arrow_points[1] = norm.point_along(_arrow_width) ;
	_arrow_points[2] = norm.point_along(-_arrow_width) ;
}

void SegmentTransitionGraphic::
compute_text_location()
{
	_origin = GraphicRenderer::Origin_Center ;
		// Place text in the "middle" of the vertices
	int nvertices = _vertices.size()  ;
	int location_index = (nvertices - 1) / 2 ;
	_text_location = _vertices[location_index] ;
	Point next(_vertices[location_index + 1]) ;
	float x_diff = next.x() - _text_location.x() ;
	float y_diff = next.y() - _text_location.y() ;
	if ((nvertices & 1) == 0)
	{
		_text_location.x() += x_diff / 2 ;
		_text_location.y() += y_diff / 2 ;
	}
	GraphicRenderer& render = GraphicRenderer::instance() ;
	if (x_diff == 0) // vertical line
	{
		_origin = GraphicRenderer::Origin_West ;
	}
	else
	{
			// The minus sign for y_diff is to account for the
			// wierd y axis in X windows.
		float slope = -y_diff / x_diff ;
		if (slope > 0)
		{
			_origin = GraphicRenderer::Origin_NorthWest ;
		}
		else if (slope < 0)
		{
			_origin = GraphicRenderer::Origin_SouthWest ;
		}
		else // slope == 0
		{
			_origin = GraphicRenderer::Origin_South ;
		}
	}
}
