/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: StateGraphic.cc,v $
$Revision: 1.21 $
$Date: 1997/07/02 04:45:19 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: StateGraphic.cc,v $
Revision 1.21  1997/07/02 04:45:19  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.20  1997/06/15 00:43:57  andrewm
Another checkpoint.  Reworked the way text is specfied to be drawn
for the benefit of the post script renderer.

Revision 1.19  1997/06/06 04:34:28  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.18  1997/05/31 21:12:44  andrewm
Checkpoint.  Things are working well.

Revision 1.17  1997/05/20 05:15:35  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.16  1997/04/12 21:15:01  andrewm
Checkpoint after adding the event dialog.

Revision 1.15  1997/03/18 06:51:04  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.14  1997/03/12 03:13:08  andrewm
Checkpoint.  Things are working rather well.

Revision 1.13  1997/03/04 06:32:58  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.12  1997/02/23 23:44:16  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.11  1997/02/08 04:37:43  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.10  1997/01/23 06:20:56  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.9  1996/11/27 01:25:54  andrewm
Another checkpoint before I go off and figure out what to do about
the graphical representations so that I'll be able to get some post script
output one of these days.

Revision 1.8  1996/11/14 06:19:54  andrewm
checkpoint

// Revision 1.7  1996/10/01  04:39:26  andrewm
// checkpoint and revision
//
// Revision 1.6  1996/09/22  01:18:34  andrewm
// pre-alpha release
//
// Revision 1.5  1996/08/18  17:58:13  andrewm
// checkpoint
//
// Revision 1.4  1996/08/05  01:25:56  andrewm
// checkpoint
//
// Revision 1.3  1996/07/27  20:57:09  andrewm
// checkpoint
//
// Revision 1.2  1996/07/15  01:21:01  andrewm
// checkpoint again
//
// Revision 1.1  1996/06/26  03:16:15  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Smachine.h"
#include "StateGraphic.h"
#include "TransitionGraphic.h"
#include "PageGroup.h"
#include "Page.h"
#include "State.h"
#include "Transition.h"
#include "GraphicRenderer.h"

#include <math.h>
#include <algorithm>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: StateGraphic.cc,v $ $Revision: 1.21 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

StateGraphic::
StateGraphic(
	Page *parent,
	State *state) :
		_parent(parent),
		_state(state),
		_circle(state->circle()),
		_name(state->name()),
		_selected(state->selected()),
		_is_initial_state(state->is_initial_state())
{
	_state->attach(this) ;

	const State::TransitionList& tlist = _state->transition_list() ;
	for (State::TransitionListConstIter t_iter = tlist.begin() ;
		t_iter != tlist.end() ; ++t_iter)
	{
		Transition *trans = *t_iter ;
		if (!trans->is_pseudo_transition())
		{
			_transition_graphic_list.push_back(
				trans->is_loop_transition() ?
				new LoopTransitionGraphic(this, (LoopTransition *)trans) :
				new SegmentTransitionGraphic(this,
					(SegmentTransition *)trans)) ;
		}
	}
}

StateGraphic::
~StateGraphic()
{
	draw_state_graphic() ;

	if (_state)
		_state->detach(this) ;
	if (_parent)
		_parent->remove_state_graphic(this) ;

	for (TransitionGraphicListIter tg_iter = _transition_graphic_list.begin() ;
		tg_iter != _transition_graphic_list.end() ; ++tg_iter)
	{
		TransitionGraphic *tg = *tg_iter ;
		tg->orphan() ;
		delete tg ;
	}
}

TransitionGraphic *StateGraphic::
find_transition_graphic(
	Transition *transition) const
{
	class TransitionGraphicCmp :
		public binary_function<TransitionGraphic *, Transition *, bool>
	{
	public:
		bool operator ()(TransitionGraphic *tg, Transition *tr) const {
			return tg->transition() == tr ;
		}
	} ;

	TransitionGraphicListConstIter end = _transition_graphic_list.end() ;
	TransitionGraphicListConstIter found = find_if(
		_transition_graphic_list.begin(), end,
		bind2nd(TransitionGraphicCmp(), transition)) ;

	return found == end ? (TransitionGraphic *)NULL : *found ;
}

TransitionGraphic *StateGraphic::
find_transition_graphic(
	MachEvent *event) const
{
	Transition *trans = _state->find_transition(event) ;
	return (trans ?
		find_transition_graphic(trans) : (TransitionGraphic *)NULL) ;
}

void StateGraphic::
remove_transition_graphic(
	TransitionGraphic *transition_graphic)
{
	transition_graphic->orphan() ;
	_transition_graphic_list.remove(transition_graphic) ;
}

void StateGraphic::
update(
	const Subject *subject)
{
	assert(subject == _state) ;
	const State::TransitionList& tlist = _state->transition_list() ;
	for (State::TransitionListConstIter t_iter = tlist.begin() ;
		t_iter != tlist.end() ; ++t_iter)
	{
		Transition *trans = *t_iter ;
		if (!trans->is_pseudo_transition()
			&& !find_transition_graphic(trans))
		{
			TransitionGraphic *tg = trans->is_loop_transition() ?
				new LoopTransitionGraphic(this, (LoopTransition *)trans) :
				new SegmentTransitionGraphic(this, (SegmentTransition *)trans) ;
			_transition_graphic_list.push_back(tg) ;
			tg->draw() ;
		}
	}

	draw_state_graphic() ;
	_name = _state->name() ;

	GraphicRenderer& render = GraphicRenderer::instance() ;
	Point extent(render.text_extent(_name)) ;
	extent += render.text_extent("MM") ;  // give some margin
	float new_radius = hypot(extent.x(), extent.y()) / 2.0 ;
	_state->adjust_radius(new_radius) ;

	_circle = _state->circle() ;
	_selected = _state->selected() ;
	_is_initial_state = _state->is_initial_state() ;
	draw_state_graphic() ;
}

void StateGraphic::
subject_deleted(
	Subject *subject)
{
	assert(subject == _state) ;
	_state = NULL ;
	delete this ;
}

void StateGraphic::
draw()
{
	draw_state_graphic() ;

	class TransitionDraw
	{
	public:
		void operator ()(TransitionGraphic *tg) { tg->draw() ; }
	} ;

	for_each(_transition_graphic_list.begin(), _transition_graphic_list.end(),
		TransitionDraw()) ;
}

void StateGraphic::
highlight()
{
	if (_selected)
	{
		GraphicRenderer& render = GraphicRenderer::instance() ;
		GraphicRenderer::LineDrawingStyle style =
			render.line_style(GraphicRenderer::LS_DASHED) ;
		float line_width = render.line_width() ;
		render.rect(enclose_circle(_circle, line_width)) ;
		render.line_style(style) ;
	}
}

void StateGraphic::
draw_state_graphic()
{
	GraphicRenderer& render = GraphicRenderer::instance() ;
	render.circle(_circle) ;
	if (_is_initial_state)
	{
		Circle inner(_circle) ;
		inner.radius() -= render.line_width() * 3 ;
		render.circle(inner) ;
	}
	render.text(_name, _circle.center(), GraphicRenderer::Origin_Center) ;
	highlight() ;
}
