/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: PSGraphicRenderer.cc,v $
$Revision: 1.6 $
$Date: 1997/07/14 02:30:35 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: PSGraphicRenderer.cc,v $
Revision 1.6  1997/07/14 02:30:35  andrewm
Checkpoint.  Rework makefiles.  Rework man pages.  Fixed PostScript problems.

Revision 1.5  1997/07/02 04:45:18  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.4  1997/06/28 23:45:09  andrewm
Finished PostScript formatter.  Checkpoint.

Revision 1.3  1997/06/21 02:21:34  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.2  1997/06/15 00:43:56  andrewm
Another checkpoint.  Reworked the way text is specfied to be drawn
for the benefit of the post script renderer.

Revision 1.1  1997/06/12 03:20:26  andrewm
Checkpoint.  Crude version of the PostScript output running.  Need
to change the way text is manipulated in order to make PostScript generation
better.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "PSGraphicRenderer.h"
#include <stdlib.h>
#include <minmax.h>
#include <math.h>
#include <time.h>
#include <ctype.h>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: PSGraphicRenderer.cc,v $ $Revision: 1.6 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const float PSGraphicRenderer::_points_per_mm = 72.0 / 25.4 ;

/*
FUNCTION DEFINITIONS
*/
PSGraphicRenderer::
PSGraphicRenderer(
	ostream& s) :
		_stream(s),
		_running(false),
		_page_count(0),
		_orientation(BestFit),
		_page_width(8.5 * 25.4), 	// 8.5 inches in mm
		_page_height(11.0 * 25.4),	// 11 inches in mm
		_left_margin(0.5 * 25.4),	// 0.5 inch margins all round
		_right_margin(0.5 * 25.4),
		_top_margin(0.5 * 25.4),
		_bottom_margin(0.6 * 25.4)
{
}

PSGraphicRenderer::
~PSGraphicRenderer()
{
}

void PSGraphicRenderer::
start(
	const string& media,
	const char *creator,
	const Rectangle& bounding)
{
	_bounding = bounding ;
	_running = true ;
	if (_orientation == BestFit)
	{
		_orientation = _bounding.width() > _bounding.height() ?
			Landscape : Portrait ;
	}

	_stream << "%!PS-Adobe-3.0" << endl ;
	_stream << "%%Pages: (atend)" << endl ;
	_stream << "%%BoundingBox: (atend)" << endl ;
	_stream << "%%Creator: " << creator << endl ;
	time_t current_time = time(NULL) ;
	_stream << "%%CreationDate: " << ctime(&current_time) ;
	_stream << "%%For: " << getenv("USER") << endl ;
	_stream << "%%LanguageLevel: 1" << endl ;
	string media_name(media) ;
	string::value_type c(media_name[0]) ;
	if (islower(c))
		c = toupper(c) ;
	media_name[0] = c ;
	_stream << "%%DocumentMedia: " << media_name << ' ' <<
		int(ceil(_page_width * _points_per_mm)) << ' ' <<
		int(_page_height * _points_per_mm) << ' ' <<
		"75 white ()" << endl ;
	_stream << "%%Orientation: " << _orientation << endl ;
	_stream << "%%PageOrder: Ascend" << endl ;
	_stream << "%%EndComments" << endl ;
	setup() ;
}

void PSGraphicRenderer::
end()
{
	_stream << "end" << endl ;
	_stream << "%%Trailer" << endl << flush ;
	_stream << "%%Pages: " << _page_count << endl ;
	_stream << "%%BoundingBox: " <<
		int(ceil(_doc_bounding.origin().x())) << ' ' <<
		int(ceil(_doc_bounding.origin().y())) << ' ' <<
		int(ceil(_doc_bounding.origin().x() + _doc_bounding.width())) << ' ' <<
		int(ceil(_doc_bounding.origin().y() + _doc_bounding.height())) << endl ;
	_stream << "%%EOF" << endl << flush ;
	_running = false ;
}

void PSGraphicRenderer::
start_page(
	const Rectangle& bounding)
{
	++_page_count ;
	_bounding = bounding ;
	_stream << "%%Page: " << _page_count << ' ' << _page_count << endl ;

	float available_width ;
	float available_height ;
	if (_orientation == Landscape)
	{
		available_width = _page_height - _top_margin - _bottom_margin ;
		available_height = _page_width - _left_margin - _right_margin ;
	}
	else
	{
		available_width = _page_width - _left_margin - _right_margin ;
		available_height = _page_height - _top_margin - _bottom_margin ;
	}

	float x_scale = available_width / _bounding.width() ;
	float y_scale = available_height / _bounding.height() ;
	_page_scale = min(x_scale, y_scale) ;
	float scaled_width = _bounding.width() * _page_scale ;
	float scaled_height = _bounding.height() * _page_scale ;

	Rectangle scaled_bounding ;
	scaled_bounding.origin().x() = _bounding.origin().x() * _page_scale ;
	scaled_bounding.origin().y() = _bounding.origin().y() * _page_scale ;
	scaled_bounding.width() = scaled_width ;
	scaled_bounding.height() = scaled_height ;

	Rectangle ps_bounding(calc_ps_bounding()) ;
	if (_page_count == 1)
		_doc_bounding = ps_bounding ;
	else
		_doc_bounding.enclose(ps_bounding) ;
	_stream << "%%PageBoundingBox: " <<
		int(ceil(ps_bounding.origin().x())) << ' ' <<
		int(ceil(ps_bounding.origin().y())) << ' ' <<
		int(ceil(ps_bounding.origin().x() + ps_bounding.width())) << ' ' <<
		int(ceil(ps_bounding.origin().y() + ps_bounding.height())) << endl ;
	_stream << "gsave" << endl ;

	if (_orientation == Landscape)
	{
			/*
			The following reorients the coordinate system
			properly.  In landscape we reorient and then
			apply an additional translation to center the
			diagram.  All very mystical here and most of this
			was determined by trial and error.
			*/
		float height_points = _page_height * _points_per_mm ;
		_stream << "0 " << height_points << " translate" << endl ;
		_stream << "1 -1 scale -90 rotate" << endl ;
		_stream << -height_points << " 0 translate" << endl ;
		_stream << _page_scale << ' ' << _page_scale << " scale" << endl ;

		float x_trans = 
			_bottom_margin
			+ (available_width - scaled_width) / 2.0
			- scaled_bounding.origin().x() ;
		x_trans *= _points_per_mm ;
		float y_trans = 
			_left_margin
			+ (available_height - scaled_height) / 2.0
			- scaled_bounding.origin().y() ;
		y_trans *= _points_per_mm ;

		_stream << x_trans << ' ' << y_trans << " translate" << endl ;
	}
	else
	{
			/*
			For normal portrait stuff, this is the conventional
			way to transform from X coordinates to PostScript coordinates.
			*/
		float x_trans = 
			_left_margin
			+ (available_width - scaled_width) / 2.0
			- scaled_bounding.origin().x() ;
		x_trans *= _points_per_mm ;
		float y_trans = _page_height -
			(_top_margin
			+ (available_height - scaled_height) / 2.0
			- scaled_bounding.origin().y()) ;
		y_trans *= _points_per_mm ;

		_stream << x_trans << ' ' << y_trans << " translate" << endl ;
		_stream << _page_scale << ' ' << -_page_scale << " scale" << endl ;
	}

	_stream <<
		"/Helvetica findfont [ FontSize 0 0 FontSize neg 0 0 ] makefont setfont"
		<< endl ;
}

void PSGraphicRenderer::
end_page()
{
	_stream << "showpage grestore" << endl << flush ;
}

Point PSGraphicRenderer::
scale_to_mm(
	const Point& point)
{
	return point ;
}

void PSGraphicRenderer::
circle(
	const Circle& circle)
{
	if (_running == false)
		return ;

	Circle c(scale_to_points(circle)) ;
	Point center(c.center()) ;
	_stream <<
		center.x() << ' ' <<
		center.y() << ' ' <<
		c.radius() << ' ' <<
		"Circle" << endl ;
}

void PSGraphicRenderer::
circular_arc(
	const Circle& circle,
	float start,
	float extent)
{
	if (_running == false)
		return ;

		/*
		It's a little tricky here because of the coordinate system
		inversion about the Y axis that is necessary to translate
		from X coordinates to PS.  To do arcs, it is necessary to
		reflect the angles across the X axis and then use "arcn"
		to draw the arc. The negation of "arcn" combined with the
		reflection across the X axis and the scaling in the negative
		Y direction seems to work.
		*/
	Circle c(scale_to_points(circle)) ;
	Point center(c.center()) ;
	float angle1 = -start * 180.0 / M_PI ;
	float angle2 = start + extent ;
	angle2 = (M_PI - angle2) + M_PI ;
	angle2 = angle2 * 180.0 / M_PI ;
	_stream <<
		center.x() << ' ' <<
		center.y() << ' ' <<
		c.radius() <<  ' ' <<
		angle1 << ' ' <<
		angle2 << ' ' <<
		"CArc" << endl ;

}

void PSGraphicRenderer::
rect(
	const Rectangle& rectangle)
{
	if (_running == false)
		return ;

	Rectangle r(scale_to_points(rectangle)) ;
	Point origin(r.origin()) ;
	_stream <<
		origin.x() << ' ' <<
		origin.y() << ' ' <<
		"moveto" << endl ;
	_stream <<
		"0" << ' ' <<
		r.width() << ' ' <<
		"rlineto" << endl ;
	_stream <<
		r.height() << ' ' <<
		"0" << ' ' <<
		"rlineto" << endl ;
	_stream <<
		"closepath stroke" << endl ;
}

void PSGraphicRenderer::
fillrect(
	const Rectangle& rectangle)
{
	if (_running == false)
		return ;

	Rectangle r(scale_to_points(rectangle)) ;
	Point origin(r.origin()) ;
	_stream <<
		origin.x() << ' ' <<
		origin.y() << ' ' <<
		"moveto" << endl ;
	_stream <<
		"0" << ' ' <<
		r.width() << ' ' <<
		"rlineto" << endl ;
	_stream <<
		r.height() << ' ' <<
		"0" << ' ' <<
		"rlineto" << endl ;
	_stream <<
		"closepath fill" << endl ;
}

void PSGraphicRenderer::
line(
	const Point& begin_pt,
	const Point& end_pt)
{
	if (_running == false)
		return ;

	Point b(scale_to_points(begin_pt)) ;
	Point e(scale_to_points(end_pt)) ;
	_stream <<
		b.x() << ' ' <<
		b.y() << ' ' <<
		"moveto" << endl ;

	_stream <<
		e.x() << ' ' <<
		e.y() << ' ' <<
		"lineto" << endl ;

	_stream << "stroke" << endl ;
}

void PSGraphicRenderer::
lines(
	const PointList& points)
{
	if (_running == false)
		return ;

	PointListConstIter p_iter = points.begin() ;
	PointListConstIter p_end = points.end() ;
	if (p_iter != p_end)
	{
		Point p(scale_to_points(*p_iter)) ;
		_stream <<
			p.x() << ' ' <<
			p.y() << ' ' <<
			"moveto" << endl ;

		for (++p_iter ; p_iter != p_end ; ++p_iter)
		{
			p = scale_to_points(*p_iter) ;
			_stream <<
				p.x() << ' ' <<
				p.y() << ' ' <<
				"lineto" << endl ;
		}

		_stream << "stroke" << endl ;
	}
}

void PSGraphicRenderer::
fillpolygon(
	const PointList& points)
{
	if (_running == false)
		return ;

	PointListConstIter p_iter = points.begin() ;
	PointListConstIter p_end = points.end() ;
	if (p_iter != p_end)
	{
		Point p(scale_to_points(*p_iter)) ;
		_stream <<
			p.x() << ' ' <<
			p.y() << ' ' <<
			"moveto" << endl ;

		for (++p_iter ; p_iter != p_end ; ++p_iter)
		{
			p = scale_to_points(*p_iter) ;
			_stream <<
				p.x() << ' ' <<
				p.y() << ' ' <<
				"lineto" << endl ;
		}

		_stream << "closepath fill" << endl ;
	}
}

void PSGraphicRenderer::
text(
	const string& text,
	const Point& location,
	TextOrigin origin)
{
	if (_running == false)
		return ;

	Point scaled = scale_to_points(location) ;
	_stream <<
		'(' << text << ')'  << ' ' <<
		scaled.x() << ' ' <<
		scaled.y() << ' ' ;
	switch (origin)
	{
		default:
		case Origin_Center:
			// Move 1/2 to the left and 1/2 down
			_stream << "CShow" ;
			break ;

		case Origin_NorthWest:
			// This is the natural coordinate system so ...
			// Move 0 to the left and 0 down
			_stream << "NWShow" ;
			break ;

		case Origin_North:
			// Move 1/2 to the left and 0 down
			_stream << "NShow" ;
			break ;

		case Origin_NorthEast:
			// Move 1 to the left and 0 down
			_stream << "NEShow" ;
			break ;

		case Origin_East:
			// Move 1 to the left and 1/2 down
			_stream << "EShow" ;
			break ;

		case Origin_SouthEast:
			// Move 1 to the left and 1 down
			_stream << "SEShow" ;
			break ;

		case Origin_South:
			// Move 1/2 to the left and 1 down
			_stream << "SShow" ;
			break ;

		case Origin_SouthWest:
			// Move 0 to the left and 1 down
			_stream << "SWShow" ;
			break ;

		case Origin_West:
			// Move 0 to the left and 1/2 down
			_stream << "WShow" ;
			break ;
	}
	_stream << endl ;
}

Point PSGraphicRenderer::
text_extent(
	const string& text)
{
	return Point(1.0, 1.0) ;
}

float PSGraphicRenderer::
line_width()
{
	return 1.0 / _points_per_mm ;
}

GraphicRenderer::LineDrawingStyle PSGraphicRenderer::
line_style(
	LineDrawingStyle new_style)
{
}

void PSGraphicRenderer::
clear()
{
}

void PSGraphicRenderer::
clear_clip()
{
}

void PSGraphicRenderer::
add_rectangle_to_clip(
	const Rectangle& rect)
{
}

void PSGraphicRenderer::
set_clip()
{
}

void PSGraphicRenderer::
setup()
{
	_stream <<

	"%%BeginSetup\n"
	"/gsmdict 20 dict def\n"
	"gsmdict begin\n"

	"/TextBBox { %def  stack: (string)\n"
		"gsave\n"
			"/Helvetica findfont FontSize scalefont setfont\n"
			"newpath 0 0 moveto\n"
			"true charpath flattenpath\n"
			"pathbbox %stack: lx ly ux uy\n"
		"grestore\n"
	"} bind def\n"

	"/Circle { %def\n"
		"0 360 arc stroke\n"
	"} bind def\n"

	"/CArc { %def\n"
		"arcn stroke\n"
	"} bind def\n"

	"/TextExtent { %stack: (string)\n"
	"	gsave\n"
	"	newpath\n"
	"	0 0 moveto\n"
	"	{\n"
	"		(\012) search\n"
	"		{ \n"
	"			dup\n"
	"			true charpath\n"
	"			stringwidth pop neg FontSize\n"
	"			rmoveto\n"
	"			pop % newline character\n"
	"		}\n"
	"		{\n"
	"			true charpath\n"
	"			exit\n"
	"		}\n"
	"		ifelse\n"
	"	} loop\n"
	"	flattenpath pathbbox	% lx ly ux uy\n"
	"	4 -2 roll pop pop		% ux uy\n"
	//"	3 -1 roll				% lx ux uy ly\n"
	//"	sub						% lx ux height\n"
	//"	3 1 roll				% height lx ux\n"
	//"	exch sub				% height width\n"
	//"	exch					% width height\n"
	"	grestore\n"
	"} bind def\n"
	"/ShowCenteredInWidth { %stack: (string) width\n"
	"	1 index\n"
	"	stringwidth pop\n"
	"	sub\n"
	"	2 div 0 rmoveto\n"
	"	show\n"
	"} bind def\n"
	"/TextCJShow { %stack: (string) x y width\n"
	"	gsave\n"
	"	newpath\n"
	"	/BoxWidth exch def\n"
	"	/OriginY exch def\n"
	"	/OriginX exch def\n"
	"	{\n"
	"		(\012) search\n"
	"		{ \n"
	"									% post match pre\n"
	"			OriginX OriginY moveto\n"
	"			BoxWidth ShowCenteredInWidth\n"
	"			/OriginY OriginY FontSize add def\n"
	"			pop						% newline character\n"
	"		}\n"
	"		{\n"
	"			OriginX OriginY moveto\n"
	"			BoxWidth ShowCenteredInWidth\n"
	"			exit\n"
	"		}\n"
	"		ifelse\n"
	"	} loop\n"
	"	grestore\n"
	"} bind def\n"
	"/CShow { % stack: (string) x y\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	2 div					% (string) x y width height/2\n"
	"	3 -1 roll				% (string) x width height/2 y\n"
	"	exch sub				% (string) x width y\n"
	"	CharAdjust add			% (string) x width y\n"
	"	3 1 roll				% (string) y x width\n"
	"	dup 2 div				% (string) y x width width/2\n"
	"	3 -1 roll				% (string) y width width/2 x\n"
	"	exch sub				% (string) y width x\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/NWShow { % stack: (string) x y\n"
	"	CharHeight add\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	pop						% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/NShow { % stack: (string) x y\n"
	"	CharHeight add\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	pop						% (string) x y width\n"
	"	dup 2 div				% (string) x y width width/2\n"
	"	4 -1 roll				% (string) y width width/2 x\n"
	"	exch sub				% (string) y width x\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/NEShow { % stack: (string) x y\n"
	"	CharHeight add\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	pop						% (string) x y width\n"
	"	dup 					% (string) x y width width\n"
	"	4 -1 roll				% (string) y width width x\n"
	"	exch sub				% (string) y width x\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/EShow { % stack: (string) x y\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	2 div 3 -1 roll			% (string) x width height/2 y\n"
	"	exch sub				% (string) x width y\n"
	"	3 1 roll				% (string) y x width\n"
	"	dup						% (string) y x width width\n"
	"	3 -1 roll				% (string) y width width x\n"
	"	exch sub				% (string) y width x\n"
	"	CharAdjust sub\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/SEShow { % stack: (string) x y\n"
	"	CharAdjust sub\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	3 -1 roll				% (string) x width height y\n"
	"	exch sub				% (string) x width y\n"
	"	3 1 roll				% (string) y x width\n"
	"	dup						% (string) y x width width\n"
	"	3 -1 roll				% (string) y width width x\n"
	"	exch sub				% (string) y width x\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/SShow { % stack: (string) x y\n"
	"	CharAdjust sub\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	3 -1 roll				% (string) x width height y\n"
	"	exch sub				% (string) x width y\n"
	"	3 1 roll				% (string) y x width\n"
	"	dup 2 div				% (string) y x width width/2\n"
	"	3 -1 roll				% (string) y width width/2 x\n"
	"	exch sub				% (string) y width x\n"
	"	3 1 roll				% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/SWShow { % stack: (string) x y\n"
	"	CharAdjust sub\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	3 -1 roll				% (string) x width height y\n"
	"	exch sub				% (string) x width y\n"
	"	exch					% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"
	"/WShow { % stack: (string) x y\n"
	"	exch CharAdjust add exch\n"
	"	2 index TextExtent		% (string) x y width height\n"
	"	2 div 3 -1 roll			% (string) x width height/2 y\n"
	"	exch sub				% (string) x width y\n"
	"	exch					% (string) x y width\n"
	"	TextCJShow\n"
	"} bind def\n"


	"/FontSize 12 def\n"
	"(X) TextBBox /CharHeight exch def pop pop pop\n"
	"/CharAdjust FontSize CharHeight sub def\n"

	"%%EndSetup\n"

	;
}

Rectangle PSGraphicRenderer::
calc_ps_bounding()
{
	Rectangle ps_rect ;
	float available_width ;
	float available_height ;
	if (_orientation == Landscape)
	{
		available_width = _page_height - _top_margin - _bottom_margin ;
		available_height = _page_width - _left_margin - _right_margin ;
	}
	else
	{
		available_width = _page_width - _left_margin - _right_margin ;
		available_height = _page_height - _top_margin - _bottom_margin ;
	}
	float scaled_width = _bounding.width() * _page_scale ;
	float scaled_height = _bounding.height() * _page_scale ;
	ps_rect.origin().x() = 
		_left_margin
		+ (available_width - scaled_width) / 2.0 ;
	ps_rect.origin().y() =
		_bottom_margin
		+ (available_height - scaled_height) / 2.0 ;
	ps_rect.width() = _bounding.width() * _page_scale ;
	ps_rect.height() = _bounding.height() * _page_scale ;
	return scale_to_points(ps_rect) ;
}

Point PSGraphicRenderer::
scale_to_points(
	const Point& point)
{
	return point * _points_per_mm ;
}

Circle PSGraphicRenderer::
scale_to_points(
	const Circle& circle)
{
	Circle scaled ;
	scaled.center() = scale_to_points(circle.center()) ;
	scaled.radius() = circle.radius() * _points_per_mm ;
	return scaled ;
}

Rectangle PSGraphicRenderer::
scale_to_points(
	const Rectangle& rectangle)
{
	Rectangle scaled ;
	scaled.origin() = scale_to_points(rectangle.origin()) ;
	scaled.width() = rectangle.width() * _points_per_mm ;
	scaled.height() = rectangle.height() * _points_per_mm ;
	return scaled ;
}

ostream&
operator <<(
	ostream& s,
	PSGraphicRenderer::Orientation orientation)
{
	if (orientation == PSGraphicRenderer::Portrait)
		s << "Portrait" ;
	else if (orientation == PSGraphicRenderer::Landscape)
		s << "Landscape" ;
	else
		s << "Unknown Orientation" ;
	return s ;
}
