/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: Matrix3x3.h,v $
$Revision: 1.2 $
$Date: 1997/07/02 04:45:16 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Matrix3x3.h,v $
Revision 1.2  1997/07/02 04:45:16  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.1  1997/06/12 03:20:26  andrewm
Checkpoint.  Crude version of the PostScript output running.  Need
to change the way text is manipulated in order to make PostScript generation
better.

 *--
 */
#ifndef _Matrix3x3_h_
#define _Matrix3x3_h_

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma interface
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Geom2d.h"
#include <math.h>

/*
MACRO DEFINITIONS
*/

/*
CLASS DEFINITIONS
*/

template<class T>
class Matrix3x3
{
public:
		// Construction and destruction
	Matrix3x3() ;
	Matrix3x3(T angle) ;

		// Useful member functions
	Matrix3x3<T>& scale(const Point2D<T>& point) ;
	Matrix3x3<T>& translate(const Point2D<T>& point) ;

		// Friends
	friend Matrix3x3<T> operator +(
		const Matrix3x3<T>& m1, const Matrix3x3<T>& m2) ;
	friend Matrix3x3<T> operator -(
		const Matrix3x3<T>& m1, const Matrix3x3<T>& m2) ;
	friend Matrix3x3<T> operator *(
		const Matrix3x3<T>& m1, const Matrix3x3<T>& m2) ;
	friend Matrix3x3<T> operator *(T scalar, const Matrix3x3<T>& matrix) ;
	friend Point2D<T> operator *(
		const Point2D<T>& point, const Matrix3x3<T>& matrix) ;

		// Stream I/O
	friend ostream& operator <<(ostream& s, Matrix3x3<T>& matrix) ;

private:
	T m[3][3] ;
} ;

typedef Matrix3x3<float> FMatrix ;

template<class T> Matrix3x3<T>::
Matrix3x3()
{
	m[0][0] = 1 ;
	m[0][1] = 0 ;
	m[0][2] = 0 ;
	m[1][0] = 0 ;
	m[1][1] = 1 ;
	m[1][2] = 0 ;
	m[2][0] = 0 ;
	m[2][1] = 0 ;
	m[2][2] = 1 ;
}

template<class T> Matrix3x3<T>::
Matrix3x3(
	T angle)
{
	m[0][0] = cos(angle) ;
	m[0][1] = sin(angle) ;
	m[0][2] = 0 ;
	m[1][0] = -sin(angle) ;
	m[1][1] = cos(angle) ;
	m[1][2] = 0 ;
	m[2][0] = 0 ;
	m[2][1] = 0 ;
	m[2][2] = 1 ;
}

template<class T> Matrix3x3<T>& Matrix3x3<T>::
scale(
	const Point2D<T>& point)
{
	m[0][0] = point.x() ;
	m[1][1] = point.y() ;
	return *this ;
}

template<class T> Matrix3x3<T>& Matrix3x3<T>::
translate(
	const Point2D<T>& point)
{
	m[2][0] = point.x() ;
	m[2][1] = point.y() ;
	return *this ;
}

template<class T> Matrix3x3<T>
operator +(
	const Matrix3x3<T>& m1,
	const Matrix3x3<T>& m2)
{
	Matrix3x3<T> r ;
	r[0][0] = m1[0][0] + m2[0][0] ;
	r[0][1] = m1[0][1] + m2[0][1] ;
	r[0][2] = m1[0][2] + m2[0][2] ;
	r[1][0] = m1[1][0] + m2[1][0] ;
	r[1][1] = m1[1][1] + m2[1][1] ;
	r[1][2] = m1[1][2] + m2[1][2] ;
	r[2][0] = m1[2][0] + m2[2][0] ;
	r[2][1] = m1[2][1] + m2[2][1] ;
	r[2][2] = m1[2][2] + m2[2][2] ;
	return r ;
}

template<class T> Matrix3x3<T>
operator -(
	const Matrix3x3<T>& m1,
	const Matrix3x3<T>& m2)
{
	Matrix3x3<T> r ;
	r[0][0] = m1[0][0] - m2[0][0] ;
	r[0][1] = m1[0][1] - m2[0][1] ;
	r[0][2] = m1[0][2] - m2[0][2] ;
	r[1][0] = m1[1][0] - m2[1][0] ;
	r[1][1] = m1[1][1] - m2[1][1] ;
	r[1][2] = m1[1][2] - m2[1][2] ;
	r[2][0] = m1[2][0] - m2[2][0] ;
	r[2][1] = m1[2][1] - m2[2][1] ;
	r[2][2] = m1[2][2] - m2[2][2] ;
	return r ;
}

template<class T> Matrix3x3<T>
operator *(
	const Matrix3x3<T>& m1,
	const Matrix3x3<T>& m2)
{
	Matrix3x3<T> r ;

	r[0][0] = m1[0][0] * m2[0][0] +
			  m1[0][1] * m2[1][0] +
			  m1[0][2] * m2[2][0] ;
	r[1][0] = m1[1][0] * m2[0][0] +
			  m1[1][1] * m2[1][0] +
			  m1[1][2] * m2[2][0] ;
	r[2][0] = m1[2][0] * m2[0][0] +
			  m1[2][1] * m2[1][0] +
			  m1[2][2] * m2[2][0] ;

	r[0][1] = m1[0][0] * m2[0][1] +
			  m1[0][1] * m2[1][1] +
			  m1[0][2] * m2[2][1] ;
	r[1][1] = m1[1][0] * m2[0][1] +
			  m1[1][1] * m2[1][1] +
			  m1[1][2] * m2[2][1] ;
	r[2][1] = m1[2][0] * m2[0][1] +
			  m1[2][1] * m2[1][1] +
			  m1[2][2] * m2[2][1] ;

	r[0][2] = m1[0][0] * m2[0][2] +
			  m1[0][1] * m2[1][2] +
			  m1[0][2] * m2[2][2] ;
	r[1][2] = m1[1][0] * m2[0][2] +
			  m1[1][1] * m2[1][2] +
			  m1[1][2] * m2[2][2] ;
	r[2][2] = m1[2][0] * m2[0][2] +
			  m1[2][1] * m2[1][2] +
			  m1[2][2] * m2[2][2] ;

	return r ;
}

template<class T> Matrix3x3<T>
operator *(
	T scalar,
	const Matrix3x3<T>& matrix)
{
	Matrix3x3<T> r ;

	r[0][0] = scalar * matrix[0][0] ;
	r[1][0] = scalar * matrix[0][0] ;
	r[2][0] = scalar * matrix[0][0] ;

	r[0][1] = scalar * matrix[0][1] ;
	r[1][1] = scalar * matrix[0][1] ;
	r[2][1] = scalar * matrix[0][1] ;

	r[0][2] = scalar * matrix[0][2] ;
	r[1][2] = scalar * matrix[0][2] ;
	r[2][2] = scalar * matrix[0][2] ;

	return r ;
}

template<class T> Point2D<T>
operator *(
	const Point2D<T>& point,
	const Matrix3x3<T>& matrix)
{
	Point2D<T> p ;
	p.x() =
		point.x() * matrix.m[0][0] +
		point.y() * matrix.m[1][0] +
		/* 1 * */ matrix.m[2][0] ;
	p.y() =
		point.x() * matrix.m[0][1] +
		point.y() * matrix.m[1][1] +
		/* 1 * */ matrix.m[2][1] ;
	return p ;
}

template<class T> ostream&
operator <<(
	ostream& s,
	Matrix3x3<T>& matrix)
{
	s << '[' ;
	s << '[' <<
			matrix.m[0][0] << ", " <<
			matrix.m[0][1] << ", " <<
			matrix.m[0][2] <<
		"], " ;
	s << '[' <<
			matrix.m[1][0] << ", " <<
			matrix.m[1][1] << ", " <<
			matrix.m[1][2] <<
		"], " ;
	s << '[' <<
			matrix.m[2][0] << ", " <<
			matrix.m[2][1] << ", " <<
			matrix.m[2][2] <<
		']' ;
	return s ;
}


#endif /* _Matrix3x3_h_ */
