/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: Transition.cc,v $
$Revision: 1.18 $
$Date: 1997/07/14 02:30:34 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Transition.cc,v $
Revision 1.18  1997/07/14 02:30:34  andrewm
Checkpoint.  Rework makefiles.  Rework man pages.  Fixed PostScript problems.

Revision 1.17  1997/07/03 04:18:35  andrewm
Minor changes to deal with a few cases from inserting copyright notices.

Revision 1.16  1997/07/02 04:45:13  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.15  1997/06/21 02:21:33  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.14  1997/05/31 21:12:44  andrewm
Checkpoint.  Things are working well.

Revision 1.13  1997/05/20 05:15:35  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.12  1997/05/15 04:14:46  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.11  1997/04/24 03:20:49  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.10  1997/03/18 06:51:04  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.9  1997/03/12 03:13:07  andrewm
Checkpoint.  Things are working rather well.

Revision 1.8  1997/03/04 06:32:55  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.7  1997/02/23 23:44:14  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.6  1997/02/08 04:37:41  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.5  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.4  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.3  1996/12/24 05:20:12  andrewm
Checkpoint.

Revision 1.2  1996/09/22 01:18:22  andrewm
pre-alpha release

// Revision 1.1  1996/08/18  17:57:45  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Smachine.h"
#include "State.h"
#include "Transition.h"
#include "MachEvent.h"

#include <algorithm>
#include <math.h>
#include <minmax.h>
#include <builtin.h>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: Transition.cc,v $ $Revision: 1.18 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const char Transition::_destination_name[] = "destination" ;
const char LoopTransition::_attach_name[] = "attach" ;
const char SegmentTransition::_path_name[] = "path" ;

/*
FUNCTION DEFINITIONS
*/

Transition::
Transition(
	State *source,
	MachEvent *event) :
		_source(source),
		_event(event),
		_modified(true),
		_selected(false)
{
	ChioMap& parent_map = _source->transitions() ;
	ChioMap::pair_iterator_bool result = parent_map.insert(
		ChioAssignment(_event->name(), ChioValue(ChioValue::MapChioValue))) ;
	assert(result.second == true) ;
}

Transition::
Transition(
	State *source,
	ChioMapIter place) :
		_source(source),
		_modified(false),
		_selected(false)
{
	const ChioTerm& event_name = (*place).first ;
	_event = source->parent()->find_event(event_name) ;
	assert(_event != 0) ;
}

Transition::
~Transition()
{
	if (_source)
	{
		_source->transitions().erase(find_place()) ;
		_source->remove_transition(this) ;
	}
}

void Transition::
selected(
	bool new_selected)
{
	bool old_selected = _selected ;
	_selected = new_selected ;
	if (old_selected != new_selected)
		notify() ;
}

void Transition::
select_if_within(
	const Rectangle& box)
{
	bool is_within = box.contains(bounding_box()) ;
	if (is_within == true)
		selected(true) ;
}

void Transition::
insert_at_location(
	const Point& p)
{
}

bool Transition::
delete_at_location(
	const Point& p)
{
	return false ;
}

ChioTerm& Transition::
determine_destination(
	ChioMap& trans_map)
{
	ChioMapIter found = trans_map.find(_destination_name) ;
	assert(found != trans_map.end()) ;
	return (ChioTerm&)((*found).second) ;
}

ChioMapIter Transition::
find_place()
{
	ChioMap& trans_map = _source->transitions() ;
	ChioMapIter t_iter = trans_map.find(_event->name()) ;
	assert(t_iter != trans_map.end()) ;
	return t_iter ;
}

ostream&
operator <<(
	ostream& stream,
	const Transition& transition)
{
	transition.print_transition(stream) ;
	return stream ;
}

PseudoTransition::
PseudoTransition(
	State *source,
	MachEvent *event,
	State *destination) :
		Transition(source, event),
		_destination(destination)
{
	assert(_destination->is_pseudo_state()) ;
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	trans_map[_destination_name] = _destination->name() ;
}

PseudoTransition::
PseudoTransition(
	State *source,
	ChioMapIter place) :
		Transition(source, place)
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	ChioTerm& dest_name(trans_map[_destination_name]) ;
	_destination = _source->parent()->find_state(dest_name) ;
	assert(_destination != NULL) ;
	assert(_destination->is_pseudo_state()) ;
}

void PseudoTransition::
print_transition(
	ostream& s) const
{
	s << *_source << " ---" << *_event << "--> " << *_destination ;
}

void PseudoTransition::
sync()
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	trans_map[_destination_name] = _destination->name() ;
}

LoopTransition::
LoopTransition(
	State *source,
	MachEvent *event,
	float attach_angle) :
		Transition(source, event),
		_attach_angle(attach_angle)
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	trans_map[_destination_name] = _source->name() ;
	trans_map[_attach_name] = _attach_angle ;
}

LoopTransition::
LoopTransition(
	State *source,
	ChioMapIter place) :
		Transition(source, place)
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	ChioTerm& dest_name(trans_map[_destination_name]) ;
	assert(dest_name == _source->name()) ;

	ChioMapIter found = trans_map.find(_attach_name) ;
	if (found != trans_map.end())
		_attach_angle = (*found).second ;
	else
	{
		ChioMap::pair_iterator_bool result =
			trans_map.insert(ChioAssignment(_attach_name, 0.0)) ;
		assert(result.second == true) ;
	}
}

void LoopTransition::
print_transition(
	ostream& s) const
{
	s << *_source << " ---" << *_event << "--> " << *_source ;
}

void LoopTransition::
sync()
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	trans_map[_destination_name] = _source->name() ;
	trans_map[_attach_name] = _attach_angle ;
}

void LoopTransition::
move(
	const Point& p)
{
	Circle state_circle(_source->circle()) ;
	Point state_center(state_circle.center()) ;
	if (p != state_center)
	{
		Line l(state_center, p) ;
		Line m(state_center, state_circle.on_circle_at_angle(0)) ;
		_attach_angle = acos(cosine_between(l, m)) ;
		if (p.y() > state_center.y())
			_attach_angle = 2.0 * M_PI - _attach_angle ;
		_modified = true ;
		notify() ;
	}
}

void LoopTransition::
move_relative(
	const Point& p)
{
	/*
	Since "move" in the Loop sense changes the attach angle,
	"move_relative" really does nothing.
	*/
}

bool LoopTransition::
contains(
	const Point& p)
{
	return compute_loop_circle().is_point_within(p) ;
}

Rectangle LoopTransition::
bounding_box()
{
	Circle loop_circle(compute_loop_circle()) ;
	float radius = loop_circle.radius() ;
		// guess how much to allow for text
		// assume 12 pt font
	radius += _event->name().size() * 6.0 / 72.0 * 25.4 ;
	Point center = loop_circle.center() ;
	Point p(center.x() - radius, center.y() - radius) ;
	float diameter = radius * 2 ;
	return Rectangle(p, diameter, diameter) ;
}

void LoopTransition::
attach_angle(
	float new_angle)
{
	_attach_angle = new_angle ;
	_modified = true ;
	notify() ;
}

Circle LoopTransition::
compute_loop_circle() const
{
	const Circle& c(_source->circle()) ;
	Point center(c.on_circle_at_angle(_attach_angle)) ;
	return Circle(center, c.radius()) ;
}

SegmentTransition::
SegmentTransition(
	State *source,
	MachEvent *event,
	State *destination,
	const TransitionVertices& vertices) :
		Transition(source, event),
		_destination(destination),
		_vertices(vertices)
{
	ChioList path_list ;
	for (TransitionVerticesConstIter v_iter = _vertices.begin() ;
		v_iter != _vertices.end() ; ++v_iter)
	{
		Point p = *v_iter ;
		ChioList point_list ;
		point_list.push_back(p.x()) ;
		point_list.push_back(p.y()) ;

		path_list.push_back(point_list) ;
	}

	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	ChioMap::pair_iterator_bool result =
		trans_map.insert(ChioAssignment(_path_name, path_list)) ;
	assert(result.second == true) ;
	trans_map[_destination_name] = _destination->name() ;
	_current_vertex = _vertices.end() ;
}

SegmentTransition::
SegmentTransition(
	State *source,
	ChioMapIter place) :
		Transition(source, place)
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	ChioTerm& dest_name(trans_map[_destination_name]) ;
	_destination = source->parent()->find_state(dest_name) ;
	assert(_destination != 0) ;

	ChioMapIter found = trans_map.find(_path_name) ;
	if (found != trans_map.end())
	{
		ChioList& path_list = (*found).second ;
		for (ChioListIter p_iter = path_list.begin() ;
			p_iter != path_list.end() ; ++p_iter)
		{
			ChioList& point_list = *p_iter ;
			Point p(point_list.front(), point_list.back()) ;
			_vertices.push_back(p) ;
		}
	}
	else
	{
		ChioMap::pair_iterator_bool result =
			trans_map.insert(ChioAssignment(_path_name,
			ChioValue(ChioValue::ListChioValue))) ;
		assert(result.second == true) ;
	}
	_current_vertex = _vertices.end() ;
}

void SegmentTransition::
print_transition(
	ostream& s) const
{
	s << *_source << " ---" << *_event << "--> " << *_destination ;
}

void SegmentTransition::
sync()
{
	ChioMapIter place = find_place() ;
	ChioMap& trans_map = (*place).second ;
	trans_map[_destination_name] = _destination->name() ;
	ChioMapIter found = trans_map.find(_path_name) ;
	assert(found != trans_map.end()) ;
	ChioList& path_list = (*found).second ;
		// Erase all the path elements and recharge the
		// map with the current values from the vertices vector.
		// In this manner, we don't have to keep track of addtions or
		// deletions from the vertices vector.
	path_list.erase(path_list.begin(), path_list.end()) ;
	for (TransitionVerticesConstIter v_iter = _vertices.begin() ;
		v_iter != _vertices.end() ; ++v_iter)
	{
		Point p(*v_iter) ;
		ChioList point_list ;
		point_list.push_back(p.x()) ;
		point_list.push_back(p.y()) ;

		path_list.push_back(point_list) ;
	}
}

void SegmentTransition::
selected(
	bool new_selected)
{
	if (new_selected == false)
		_current_vertex = _vertices.end() ;
	Transition::selected(new_selected) ;
}

void SegmentTransition::
insert_at_location(
	const Point& p)
{
	_current_vertex = find_vertex_near(p) ;
	if (_current_vertex == _vertices.end())
	{
		TransitionVerticesIter v_iter = find_vertex_after(p) ;
		_current_vertex = _vertices.insert(v_iter, p) ;
		_modified = true ;
		notify() ;
	}
}

bool SegmentTransition::
delete_at_location(
	const Point& p)
{
	bool status = false ;
	_current_vertex = find_vertex_near(p) ;
	if (_current_vertex != _vertices.end())
	{
		_vertices.erase(_current_vertex) ;
		_modified = true ;
		notify() ;
		status = true ;
	}
	_current_vertex = _vertices.end() ;
	return status ;
}

void SegmentTransition::
move(
	const Point& p)
{
	if (_current_vertex != _vertices.end())
	{
		*_current_vertex = p ;
		_modified = true ;
		notify() ;
	}
}

void SegmentTransition::
move_relative(
	const Point& p)
{
	class VertexOffset :
		public binary_function<Point, Point, Point>
	{
	public:
		Point operator ()(const Point& v, const Point& p) const {
			return v + p ;
		}
	} ;

	if (_selected)
	{
		transform(_vertices.begin(), _vertices.end(), _vertices.begin(),
			bind1st(VertexOffset(), p)) ;
		_modified = true ;
		notify() ;
	}
}

bool SegmentTransition::
contains(
	const Point& p)
{
	TransitionVertices v(compute_segments(0.0, 0.0)) ;
	TransitionVerticesConstIter v_iter = v.begin() ;
	Point first_pt(*v_iter) ;
	while (v_iter != v.end())
	{
		Point next_pt(*++v_iter) ;

		if (point_on_line(first_pt, next_pt, p) == in_segment_PQ)
		{
			_current_vertex = find_vertex_near(p) ;
			return true ;
		}
		first_pt = next_pt ;
	}

	return false ;
}

Rectangle SegmentTransition::
bounding_box()
{
	float min_x = HUGE ;
	float max_x = 0.0 ;
	float min_y = HUGE ;
	float max_y = 0.0 ;
	TransitionVertices v(compute_segments(0.0, 0.0)) ;
	for (TransitionVerticesConstIter v_iter = v.begin() ;
		v_iter != v.end() ; ++v_iter)
	{
		Point p(*v_iter) ;
		float x = p.x() ;
		min_x = min(min_x, x) ;
		max_x = max(max_x, x) ;

		float y = p.y() ;
		min_y = min(min_y, y) ;
		max_y = max(max_y, y) ;
	}

	Rectangle trans_rect(Point(min_x, min_y), max_x - min_x, max_y - min_y) ;
	int nvertices = v.size() ;
	int location_index = (nvertices - 1) / 2 ;
	Point text_location = v[location_index] ;
	if (nvertices & 1)
	{
		Point next_location = v[location_index + 1] ;
		text_location.x() += (next_location.x() - text_location.x()) / 2 ;
		text_location.y() += (next_location.y() - text_location.y()) / 2 ;
	}
	Rectangle text_rect ;
	text_rect.origin() = text_location ;
	text_rect.height() = 12.0 / 72.0 * 25.4 ;
	text_rect.width() = _event->name().size() * 6.0 / 72.0 * 25.4 ;
	trans_rect.enclose(text_rect) ;
	return trans_rect ;
}

SegmentTransition::TransitionVertices SegmentTransition::
compute_segments(
	float line_width,
	float arrow_length) const
{
	const Circle& src_circle = _source->circle() ;
	const Circle& dst_circle = _destination->circle() ;

	Point first_vertex ;
	Point last_vertex ;
	int cnt = _vertices.size() ;
	if (cnt > 0)
	{
		first_vertex = _vertices[0] ;
		last_vertex = _vertices[cnt - 1] ;
	}
	else
	{
		first_vertex = dst_circle.center() ;
		last_vertex = src_circle.center() ;
	}

	TransitionVertices v(cnt + 2) ;
	copy(_vertices.begin(), _vertices.end(), v.begin() + 1) ;
	v[0] = src_circle.out_from_circle(first_vertex, line_width) ;
	v[cnt + 1] = dst_circle.out_from_circle(last_vertex,
		arrow_length + line_width) ;

	assert(v.size() >= 2) ;
	return v ;
}

SegmentTransition::TransitionVerticesIter SegmentTransition::
find_vertex_near(
	const Point& p)
{
	class VertexNear :
		public binary_function<Point, Point, bool>
	{
	public:
		bool operator ()(const Point& vertex, const Point& near) const {
			return (abs(vertex.x() - near.x()) <= 6.0
				&& abs(vertex.y() - near.y()) <= 6.0) ;
		}
	} ;

	return find_if(_vertices.begin(), _vertices.end(),
		bind2nd(VertexNear(), p)) ;
}

SegmentTransition::TransitionVerticesIter SegmentTransition::
find_vertex_after(
	const Point& p)
{
	TransitionVertices segments(compute_segments(0, 0)) ;
	TransitionVertices::size_type seg_size = segments.size() ;
	assert(seg_size > 1) ;
	int i ;
	for (i = 1 ; i < seg_size ; ++i)
	{
		Point left(segments[i - 1]) ;
		Point right(segments[i]) ;
		if (point_on_line(left, right, p) == in_segment_PQ)
			break ;
	}
	TransitionVertices::size_type v_size = _vertices.size() ;
	assert(v_size == seg_size - 2) ;
	return i > v_size  ? _vertices.end() : _vertices.begin() + i - 1 ;
}
