/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: State.cc,v $
$Revision: 1.18 $
$Date: 1997/07/02 04:45:13 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: State.cc,v $
Revision 1.18  1997/07/02 04:45:13  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.17  1997/06/21 02:21:33  andrewm
Checkpoint.  PostScript generator going well. A lot of small tweeks
all over to accomplish this.

Revision 1.16  1997/05/31 21:12:43  andrewm
Checkpoint.  Things are working well.

Revision 1.15  1997/05/20 05:15:34  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.14  1997/05/15 04:14:46  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.13  1997/04/24 03:20:48  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.12  1997/03/18 06:51:03  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.11  1997/03/12 03:13:07  andrewm
Checkpoint.  Things are working rather well.

Revision 1.10  1997/03/04 06:32:55  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.9  1997/02/23 23:44:13  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.8  1997/02/08 04:37:40  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.7  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.6  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.5  1996/12/24 05:20:11  andrewm
Checkpoint.

Revision 1.4  1996/09/22 01:18:22  andrewm
pre-alpha release

// Revision 1.3  1996/08/18  17:57:45  andrewm
// checkpoint
//
// Revision 1.2  1996/06/26  03:14:43  andrewm
// checkpoint
//
// Revision 1.1  1996/06/15  23:53:13  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Chio.h"
#include "State.h"
#include "Parameter.h"
#include "Transition.h"

#include <assert.h>
#include <algorithm>
#include <functional>
#include <minmax.h>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/
	/*
	Function object to issue notifications on a transition.
	*/
class TransNotifier
{
public:
	void operator ()(Transition *trans) const { trans->notify() ; }
} ;

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: State.cc,v $ $Revision: 1.18 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const char State::_code_name[] = "code" ;

const char RealState::_parameters_name[] = "parameters" ;
const char RealState::_transitions_name[] = "transitions" ;
const char RealState::_circle_name[] = "circle" ;
const char RealState::_center_name[] = "center" ;
const char RealState::_radius_name[] = "radius" ;
const float RealState::default_radius = 17.0 ; // in millimeters

/*
FUNCTION DEFINITIONS
*/

State::
State(
	Smachine *parent,
	const ChioTerm& name) :
		_parent(parent),
		_name(name),
		_code(ChioTerm()),
		_modified(true)
{
	ChioMap state_map ;
	ChioMap::pair_iterator_bool result = state_map.insert(
		ChioAssignment(_code_name, _code)) ;
	assert(result.second == true) ;
		// Add your own part to the parent's map of state machines.
	ChioMap& parent_map = _parent->states() ;
	result = parent_map.insert(ChioAssignment(_name, state_map)) ;
	assert(result.second == true) ;
}

State::
State(
	Smachine *parent,
	ChioMapIter place) :
		_parent(parent),
		_modified(false)
{
	_name = (*place).first ;
	ChioMap& file_def = (*place).second ;
	_code = file_def[_code_name] ;
}

State::
~State()
{
	if (_parent)
	{
		TransitionList in_trans(get_inbound_transitions()) ;
		for (TransitionListIter t_iter = in_trans.begin() ;
			t_iter != in_trans.end() ; ++t_iter)
		{
			delete *t_iter ;
		}
		_parent->states().erase(find_place()) ;
		_parent->remove_state(this) ;
	}
}

void State::
code(
	const ChioTerm& new_code)
{
	_code = new_code ;
	_modified = true ;
	notify() ;
}

State::TransitionList State::
get_inbound_transitions()
{
	TransitionList in_trans ;
	Smachine::StateList& state_list = _parent->state_list() ;
	for (Smachine::StateListIter state_iter = state_list.begin() ;
		state_iter != state_list.end() ; ++state_iter)
	{
		State *state = *state_iter ;
		if (state != this)
		{
			TransitionList trans_list = state->transition_list() ;
			for (TransitionListIter trans_iter = trans_list.begin() ;
				trans_iter != trans_list.end() ; ++trans_iter)
			{
				Transition *trans = *trans_iter ;
				if (trans->destination() == this)
				{
					in_trans.push_back(trans) ;
				}
			}
		}
	}
	return in_trans ;
}

void State::
sync()
{
	ChioMapIter place = find_place() ;
	assert(_name == (*place).first) ;
	ChioMap& m = (*place).second ;
	m[_code_name] = _code ;
}

void State::
print_state(
	ostream& s) const
{
	s << _name.c_str() ;
}

ChioMapIter State::
find_place()
{
	ChioMap& state_map = _parent->states() ;
	ChioMapIter found = state_map.find(_name) ;
	assert(found != state_map.end()) ;
	return found ;
}

ostream&
operator <<(
	ostream& stream,
	const State& state)
{
	state.print_state(stream) ;
	return stream ;
}

PseudoState::
PseudoState(
	Smachine *parent,
	const ChioTerm& name) :
		State(parent, name)
{
}

PseudoState::
PseudoState(
	Smachine *parent,
	ChioMapIter place) :
		State(parent, place)
{
}

PseudoState::
~PseudoState()
{
}

bool PseudoState::
rename(
	const ChioTerm& new_name)
{
	return false ;
}

bool PseudoState::
modified() const
{
	return _modified ;
}

void PseudoState::
modified(
	bool new_modified)
{
	_modified = new_modified ;
}

void PseudoState::
sync()
{
	State::sync() ;
}

bool PseudoState::
is_pseudo_state()
{
	return true ;
}

bool PseudoState::
is_real_state()
{
	return false ;
}

void PseudoState::
print_state(
	ostream& s) const
{
	State::print_state(s) ;
}

StateParameter *PseudoState::
find_parameter(
	const ChioTerm& type,
	const ChioTerm& name) const
{
	return NULL ;
}

StateParameter *PseudoState::
add_parameter(
	const ChioTerm& type,
	const ChioTerm& name)
{
	return NULL ;
}

StateParameter *PseudoState::
insert_parameter(
	StateParameter *current,
	const ChioTerm& type,
	const ChioTerm& name)
{
	return NULL ;
}

bool PseudoState::
change_parameter(
	StateParameter *parameter,
	const ChioTerm& type,
	const ChioTerm& name)
{
	return false ;
}

void PseudoState::
remove_parameter(
	StateParameter *parameter)
{
}

State::ParameterList PseudoState::
parameter_list()
{
	return ParameterList() ;
}

ChioList& PseudoState::
parameters()
{
	cerr << "Pseudo States have no parameters" << endl ;
	assert(0) ;
}

Transition *PseudoState::
find_transition(
	MachEvent *event) const
{
	return NULL ;
}

Transition *PseudoState::
find_transition_at_location(
	const Point& loc) const
{
	return NULL ;
}

PseudoTransition *PseudoState::
add_pseudo_transition(
	MachEvent *event,
	State *destination)
{
	return NULL ;
}

LoopTransition *PseudoState::
add_loop_transition(
	MachEvent *event)
{
	return NULL ;
}

SegmentTransition *PseudoState::
add_segment_transition(
	MachEvent *event,
	State *destination)
{
	return NULL ;
}

void PseudoState::
remove_transition(
	Transition *transition)
{
}

void PseudoState::
resolve_transitions()
{
}

State::TransitionList PseudoState::
transition_list()
{
	return TransitionList() ;
}

ChioMap& PseudoState::
transitions()
{
	cerr << "Pseudo States have no transitions" << endl ;
	assert(0) ;
}

const Circle& PseudoState::
circle() const
{
	static Circle pseudo_circle ;

	return pseudo_circle ;
}

void PseudoState::
adjust_radius(
	float new_radius)
{
}

void PseudoState::
move(
	const Point& p)
{
}

void PseudoState::
move_relative(
	const Point& p)
{
}

bool PseudoState::
contains(
	const Point& p)
{
	return false ;
}

Rectangle PseudoState::
bounding_box() const
{
	return Rectangle() ;
}

void PseudoState::
selected(
	bool new_selected)
{
}

bool PseudoState::
selected() const
{
	return false ;
}

void PseudoState::
select_all(
	bool new_selected)
{
}

void PseudoState::
select_if_within(
	const Rectangle& box)
{
}

void PseudoState::
accept_visitor(
	MachineVisitor& visitor)
{
}

RealState::
RealState(
	Smachine *parent,
	const ChioTerm& name) :
		State(parent, name),
		_circle(Circle(Point(0, 0), default_radius)),
		_selected(false)
{
	ChioList center_list ;
	center_list.push_back(_circle.center().x()) ;
	center_list.push_back(_circle.center().y()) ;

	ChioMap circle_map ;
	ChioMap::pair_iterator_bool result = 
		circle_map.insert(ChioAssignment(_center_name, center_list)) ;
	assert(result.second == true) ;
	result = circle_map.insert(ChioAssignment(_radius_name, _circle.radius())) ;
	assert(result.second == true) ;
	
	ChioMapIter place = find_place() ;
	ChioMap& state_map = (*place).second ;
	result = state_map.insert(ChioAssignment(_circle_name, circle_map)) ;
	assert(result.second == true) ;
	result = state_map.insert(
		ChioAssignment(_parameters_name, ChioValue(ChioValue::ListChioValue))) ;
	assert(result.second == true) ;
	result = state_map.insert(
		ChioAssignment(_transitions_name, ChioValue(ChioValue::MapChioValue))) ;
	assert(result.second == true) ;
}

RealState::
RealState(
	Smachine *parent,
	ChioMapIter place) :
		State(parent, place),
		_selected(false)
{
	ChioMap& state_map = (*place).second ;
	ChioMap& circle_map = state_map[_circle_name] ;
	ChioList& center_list = circle_map[_center_name] ;
	_circle.center() = Point(center_list.front(), center_list.back()) ;
	_circle.radius() = circle_map[_radius_name] ;

	ChioMapIter found = state_map.find(ChioTerm(_parameters_name)) ;
	if (found != state_map.end())
	{
		ChioList& param_list = (*found).second ;
		for (ChioListIter p_iter = param_list.begin() ;
			p_iter != param_list.end() ; ++p_iter)
		{
			_param_list.push_back(new StateParameter(this, p_iter)) ;
		}
	}
	else
	{
		ChioMap::pair_iterator_bool result = state_map.insert(
			ChioAssignment(_parameters_name,
			ChioValue(ChioValue::ListChioValue))) ;
		assert(result.second == true) ;
	}
		/*
		At this point we cannot create the transitions for the
		state because the there may be destination states that
		are referenced by the transition that have not yet been
		created as objects.  Therefore it is necessary to defer
		this to later and the "Smachine" will have to call
		"State::resolve_transitions()" to accomplish the transitions
		creation after all the states have been created.  However,
		we do make sure that there is some kind of transition map
		in the file backing store.
		*/
	found = state_map.find(_transitions_name) ;
	if (found == state_map.end())
	{
		ChioMap::pair_iterator_bool result = state_map.insert(
			ChioAssignment(_transitions_name,
			ChioValue(ChioValue::MapChioValue))) ;
		assert(result.second == true) ;
	}
}

RealState::
~RealState()
{
	for (ParameterListIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; ++param_iter)
	{
		StateParameter *param = *param_iter ;
		param->orphan() ;
		delete param ;
	}

	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		Transition *trans = *trans_iter ;
		trans->orphan() ;
		delete trans ;
	}
}

bool RealState::
rename(
	const ChioTerm& new_name)
{
	bool renamed = false ;
	State *state = _parent->find_state(new_name) ;
	if (state)
		cerr << "A state by the name of " << new_name << " already exists."
			<< endl ;
	else
	{
		ChioMapIter place = find_place() ;
		ChioValue& old_value = (*place).second ;
		ChioMap& state_map = _parent->states() ;
		ChioMap::pair_iterator_bool result = state_map.insert(
			ChioAssignment(new_name, old_value)) ;
		assert(result.second == true) ;

		state_map.erase(place) ;
		_name = new_name ;
		_modified = true ;
		renamed = true ;
		notify() ;
	}

	return renamed ;
}

bool RealState::
modified() const
{
	class TransMod
	{
	public:
		bool operator ()(bool needs, Transition *trans) {
			return needs || trans->modified() ;
		}
	} ;

	class ParamMod
	{
	public:
		bool operator ()(bool needs, StateParameter *param) {
			return needs || param->modified() ;
		}
	} ;

	return _modified
		|| accumulate(_trans_list.begin(), _trans_list.end(),
			false, TransMod())
		|| accumulate(_param_list.begin(), _param_list.end(),
			false, ParamMod()) ;
}

void RealState::
modified(
	bool new_modified)
{
	bool needs = false ;

	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; ++param_iter)
	{
		StateParameter *param = *param_iter ;
		needs = needs || param->modified() ;
	}

	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		Transition *trans = *trans_iter ;
		trans->modified(new_modified) ;
	}

	_modified = new_modified ;
}

void RealState::
sync()
{
	State::sync() ;

	ChioMapIter place = find_place() ;
	ChioMap& m = (*place).second ;
	ChioMap& circle_map = m[_circle_name] ;
	ChioList& center_list = circle_map[_center_name] ;
	center_list.erase(center_list.begin(), center_list.end()) ;
	center_list.push_back(_circle.center().x()) ;
	center_list.push_back(_circle.center().y()) ;
	circle_map[_radius_name] = _circle.radius() ;

	for (ParameterListIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; ++param_iter)
	{
		StateParameter *param = *param_iter ;
		param->sync() ;
	}

	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		Transition *trans = *trans_iter ;
		trans->sync() ;
	}
}

bool RealState::
is_pseudo_state()
{
	return false ;
}

bool RealState::
is_real_state()
{
	return true ;
}

void RealState::
print_state(
	ostream& s) const
{
	State::print_state(s) ;
	s << '(' ;

	const char *leader = "" ;
	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; ++param_iter)
	{
		StateParameter *param = *param_iter ;
		s << leader << *param ;
		leader = ", " ;
	}

	s << ')' ;
}

StateParameter *RealState::
find_parameter(
	const ChioTerm& type,
	const ChioTerm& name) const
{
	for (ParameterListConstIter param_iter = _param_list.begin() ;
		param_iter != _param_list.end() ; ++param_iter)
	{
		StateParameter *param = *param_iter ;
		if (param->type() == type && param->name() == name)
			return param ;
	}
	return NULL ;
}

StateParameter *RealState::
add_parameter(
	const ChioTerm& type,
	const ChioTerm& name)
{
	StateParameter *parameter = find_parameter(type, name) ;
	if (!parameter)
	{
		parameter = new StateParameter(this, parameters().end(), type, name) ;
		_param_list.push_back(parameter) ;
		_modified = true ;
		notify() ;
	}
	return parameter ;
}

StateParameter *RealState::
insert_parameter(
	StateParameter *current,
	const ChioTerm& type,
	const ChioTerm& name)
{
	StateParameter *parameter = find_parameter(type, name) ;
	if (parameter)
		return NULL ;

	parameter = new StateParameter(this, current->find_place(), type, name) ;
	ParameterListIter p_pos = find(_param_list.begin(), _param_list.end(),
		current) ;
	assert(p_pos != _param_list.end()) ;
	_param_list.insert(p_pos, parameter) ;
	_modified = true ;
	notify() ;
	return parameter ;
}

bool RealState::
change_parameter(
	StateParameter *parameter,
	const ChioTerm& type,
	const ChioTerm& name)
{
	StateParameter *p = find_parameter(type, name) ;
	if (p)
		return false ;

	parameter->type(type) ;
	parameter->name(name) ;
	notify() ;
	return true ;
}

void RealState::
remove_parameter(
	StateParameter *parameter)
{
	parameter->orphan() ;
	_param_list.remove(parameter) ;
	_modified = true ;
	notify() ;
}

State::ParameterList RealState::
parameter_list()
{
	return _param_list ;
}

ChioList& RealState::
parameters()
{
	ChioMapIter place = find_place() ;
	ChioMap& file_def = (*place).second ;
	ChioMapIter found = file_def.find(_parameters_name) ;
	assert(found != file_def.end()) ;
	return (ChioList&)((*found).second) ;
}

Transition *RealState::
find_transition(
	MachEvent *event) const
{
	class MachEventCmp :
		public binary_function<Transition *, MachEvent *, bool>
	{
	public:
		bool operator ()(Transition *t, MachEvent *e) const {
			return t->event() == e ;
		}
	} ;

	TransitionListConstIter end = _trans_list.end() ;
	TransitionListConstIter found = find_if(
		_trans_list.begin(), end, bind2nd(MachEventCmp(), event)) ;

	return found == end ? (Transition *)NULL : *found ;
}

Transition *RealState::
find_transition_at_location(
	const Point& loc) const
{
	class TransLocator :
		public binary_function<Transition *, const Point, bool>
	{
	public:
		bool operator ()(Transition *t, const Point p) const {
			return t->contains(p) ;
		}
	} ;

	TransitionListConstIter end = _trans_list.end() ;
	TransitionListConstIter found = find_if(
		_trans_list.begin(), end, bind2nd(TransLocator(), loc)) ;

	return found == end ? (Transition *)NULL : *found ;
}

PseudoTransition *RealState::
add_pseudo_transition(
	MachEvent *event,
	State *destination)
{
	PseudoTransition *trans = NULL ;
	if (!find_transition(event))
	{
		trans = new PseudoTransition(this, event, destination) ;
		_trans_list.push_back(trans) ;
		_modified = true ;
		notify() ;
	}
	return trans ;
}

LoopTransition *RealState::
add_loop_transition(
	MachEvent *event)
{
	LoopTransition *trans = NULL ;
	if (!find_transition(event))
	{
		trans = new LoopTransition(this, event) ;
		_trans_list.push_back(trans) ;
		_modified = true ;
		notify() ;
	}
	return trans ;
}

SegmentTransition *RealState::
add_segment_transition(
	MachEvent *event,
	State *destination)
{
	SegmentTransition *trans = NULL ;
	if (!find_transition(event))
	{
		trans = new SegmentTransition(this, event, destination) ;
		_trans_list.push_back(trans) ;
		_modified = true ;
		notify() ;
	}
	return trans ;
}

void RealState::
remove_transition(
	Transition *transition)
{
	transition->orphan() ;
	_trans_list.remove(transition) ;
	_modified = true ;
	notify() ;
}

void RealState::
resolve_transitions()
{
	ChioMapIter place = find_place() ;
	ChioMap& file_def = (*place).second ;
	ChioMapIter found = file_def.find(ChioTerm(_transitions_name)) ;
	assert(found != file_def.end()) ;
	ChioMap& trans_map = (*found).second ;
	for (ChioMapIter t_iter = trans_map.begin() ;
		t_iter != trans_map.end() ; ++t_iter)
	{
		ChioMap& trans = (*t_iter).second ;
		ChioTerm& dest_name(Transition::determine_destination(trans)) ;
		Transition *t ;
		if (dest_name == _name)
			t = new LoopTransition(this, t_iter) ;
		else
		{
			State *dest_state = _parent->find_state(dest_name) ;
			assert(dest_state != NULL) ;
			t = dest_state->is_pseudo_state() ?
				new PseudoTransition(this, t_iter) :
				new SegmentTransition(this, t_iter) ;
		}
		_trans_list.push_back(t) ;
	}
}

State::TransitionList RealState::
transition_list()
{
	return _trans_list ;
}

ChioMap& RealState::
transitions()
{
	ChioMapIter place = find_place() ;
	ChioMap& file_def = (*place).second ;
	ChioMapIter found = file_def.find(_transitions_name) ;
	assert(found != file_def.end()) ;
	return (ChioMap&)((*found).second) ;
}

const Circle& RealState::
circle() const
{
	return _circle ;
}

void RealState::
adjust_radius(
	float new_radius)
{
	float old_radius = _circle.radius() ;
	_circle.radius() = max(new_radius, default_radius) ;
		// Don't notify here, this is called during the "StateGraphic"
		// update function to set the circle radius to something that
		// encloses the text of the state name.
		// However, if the radius changes, we need to notify the
		// transitions so they can adjust themselves.
	if (old_radius != _circle.radius())
	{
		_modified = true ;
		notify_src_trans() ;
		notify_dest_trans() ;
	}
}

void RealState::
move(
	const Point& p)
{
	_circle.center() = p ;
	_modified = true ;
	notify() ;
	notify_src_trans() ;
	notify_dest_trans() ;
}

void RealState::
move_relative(
	const Point& p)
{
		/*
		Relative moves can happen with multiple items selected, so
		only move if you are currently selected.
		*/
	if (_selected)
	{
		_circle.center() += p ;
		_modified = true ;
		notify() ;
		notify_src_trans() ;
		notify_dest_trans() ;
	}
		/*
		Move on all the transitions that have this state as a source.
		*/
	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		(*trans_iter)->move_relative(p) ;
	}
}

bool RealState::
contains(
	const Point& p)
{
	return _circle.is_point_within(p) ;
}

Rectangle RealState::
bounding_box() const
{
	class TransEnclose
	{
	public:
		Rectangle operator ()(Rectangle r, Transition *t) {
			r.enclose(t->bounding_box()) ;
			return r ;
		}
	} ;

	return accumulate(_trans_list.begin(), _trans_list.end(),
		enclose_circle(_circle, float(0)), TransEnclose()) ;
}

void RealState::
selected(
	bool new_selected)
{
	bool old_selected = _selected ;
	_selected = new_selected ;
	if (old_selected != new_selected)
		notify() ;
}

bool RealState::
selected() const
{
	return _selected ;
}

void RealState::
select_all(
	bool new_selected)
{
	selected(new_selected) ;
	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		Transition *trans = *trans_iter ;
		trans->selected(new_selected) ;
	}
}

void RealState::
select_if_within(
	const Rectangle& box)
{
	bool is_within = box.contains(enclose_circle(_circle, float(0))) ;
	if (is_within == true)
		selected(true) ;
	for (TransitionListIter trans_iter = _trans_list.begin() ;
		trans_iter != _trans_list.end() ; ++trans_iter)
	{
		Transition *trans = *trans_iter ;
		trans->select_if_within(box) ;
	}
}

void RealState::
accept_visitor(
	MachineVisitor& visitor)
{
	visitor.visit(this) ;
}

void RealState::
notify_src_trans()
{
	for_each(_trans_list.begin(), _trans_list.end(), TransNotifier()) ;
}

void RealState::
notify_dest_trans()
{
	TransitionList in_trans(get_inbound_transitions()) ;
	for_each(in_trans.begin(), in_trans.end(), TransNotifier()) ;
}
