/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
Smachine.h -- definitions for the state machine object

$RCSfile: Smachine.h,v $
$Revision: 1.20 $
$Date: 1997/07/02 04:45:13 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Smachine.h,v $
Revision 1.20  1997/07/02 04:45:13  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.19  1997/05/20 05:15:34  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.18  1997/05/15 04:14:45  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.17  1997/04/24 03:20:48  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.16  1997/04/08 04:34:32  andrewm
Checkpoint as dialogs are added.
Decided to make sure that the name part of a name/value pair in
the hierarchical file is not used for id purposes.  Specifically
state machine names need to be contained within the machine not as
the key to the machine.

Revision 1.15  1997/03/18 06:51:03  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.14  1997/03/12 03:13:06  andrewm
Checkpoint.  Things are working rather well.

Revision 1.13  1997/03/04 06:32:55  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.12  1997/02/23 23:44:13  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.11  1997/02/08 04:37:40  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.10  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.9  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.8  1996/12/24 05:20:11  andrewm
Checkpoint.

Revision 1.7  1996/10/01 04:39:14  andrewm
checkpoint and revision

 * Revision 1.6  1996/09/22  01:18:22  andrewm
 * pre-alpha release
 *
 * Revision 1.5  1996/08/18  17:57:45  andrewm
 * checkpoint
 *
 * Revision 1.4  1996/07/27  20:56:54  andrewm
 * checkpoint
 *
 * Revision 1.3  1996/07/15  01:20:40  andrewm
 * checkpoint again
 *
 * Revision 1.2  1996/06/26  03:14:43  andrewm
 * checkpoint
 *
 * Revision 1.1  1996/06/15  23:53:13  andrewm
 * Initial revision
 *
 *--
 */
#ifndef _Smachine_h_
#define _Smachine_h_

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma interface
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "Chio.h"
#include "Observer.h"
#include "MachineVisitor.h"
#include "Geom2d.h"

#include <list>


/*
MACRO DEFINITIONS
*/

/*
CLASS DEFINITIONS
*/

class MachineGroup ;
class MachEvent ;
class State ;
class Transition ;

class Smachine :
	public Subject
{
	friend ostream& operator <<(ostream& stream, Smachine& machine) ;

public:
	typedef list<State *> StateList ;
	typedef list<State *>::iterator StateListIter ;
	typedef list<State *>::const_iterator StateListConstIter ;
	typedef list<MachEvent *> MachEventList ;
	typedef list<MachEvent *>::iterator MachEventListIter ;
	typedef list<MachEvent *>::const_iterator MachEventListConstIter ;

public:
	Smachine(MachineGroup *parent, ChioListIter position,
		const ChioTerm& name) ;
	Smachine(MachineGroup *parent, ChioListIter place) ;
	~Smachine() ;

	bool modified() const ;
	void modified(bool new_modified) ;
	const ChioTerm& name() const { return _name ; }
	bool rename(const ChioTerm& new_name) ;
	int counter() const { return _counter ; }
	void counter(int new_counter) ;
	const ChioTerm& prolog() const { return _prolog ; }
	void prolog(const ChioTerm& new_prolog) ;
	const ChioTerm& epilog() const { return _epilog ; }
	void epilog(const ChioTerm& new_epilog) ;
	State *initial_state() const { return _initial_state ; }
	void initial_state(State *new_initial_state) ;
	State *terminal_state() const { return _terminal_state ; }
	void terminal_state(State *new_terminal_state) ;
	State *default_state() const { return _default_state ; }
	void default_state(State *new_default_state) ;
	MachineGroup *parent() const { return _parent ; }
	void orphan() { _parent = NULL ; }

	ChioMap& events() ;
	ChioMap& states() ;
	void sync() ;
	ChioListIter find_place() ;

	State *find_state(const ChioTerm& name) const ;
	State *find_state_at_location(const Point& location) const ;
	Transition *find_transition_at_location(const Point& location) const ;
	State *add_state(const ChioTerm& name) ;
	State *add_state() ;
	State *add_state(const Point& center) ;
	void remove_state(State *state) ;
	StateList& state_list() { return _state_list ; }

	MachEvent *find_event(const ChioTerm& name) const ;
	MachEvent *add_event(const ChioTerm& name) ;
	void remove_event(MachEvent *event) ;
	MachEventList& event_list() { return _event_list ; }

	Rectangle bounding_box() ;
	void move_relative(const Point& p) ;
	void select_all(bool new_selected) ;
	void select_if_within(const Rectangle& box) ;

	void accept_visitor(MachineVisitor& visitor) { visitor.visit(this) ; }

private:
	MachineGroup *_parent ;
	ChioTerm _name ;
	bool _modified ;
	int _counter ;
	ChioTerm _prolog ;
	ChioTerm _epilog ;
	State *_initial_state ;
	State *_terminal_state ;
	State *_default_state ;
	StateList _state_list ;
	MachEventList _event_list ;

private:
	void check_itd_states(State *test_state, State *new_state) ;
	State *do_add_state(const ChioTerm& name) ;
	string new_state_name() ;
	Smachine& operator =(const Smachine& sm) ; // no assigments

private:
		// Names used as the search keys for the various sections
		// as they are stored in the backing file.
	static const char _name_name[] ;
	static const char _events_name[] ;
	static const char _states_name[] ;
	static const char _counter_name[] ;
	static const char _prolog_name[] ;
	static const char _epilog_name[] ;
	static const char _initial_state_name[] ;
	static const char _terminal_state_name[] ;
	static const char _default_state_name[] ;
	static const char _error_name[] ;
	static const char _ignore_name[] ;
} ;



#endif /* _Smachine_h_ */
