/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
Smachine.cc -- object that represents the entire state machine in
	internal form

$RCSfile: Smachine.cc,v $
$Revision: 1.19 $
$Date: 1997/07/02 04:45:12 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: Smachine.cc,v $
Revision 1.19  1997/07/02 04:45:12  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.18  1997/05/31 21:12:43  andrewm
Checkpoint.  Things are working well.

Revision 1.17  1997/05/20 05:15:34  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.16  1997/05/15 04:14:45  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.15  1997/04/24 03:20:48  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.14  1997/03/18 06:51:03  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.13  1997/03/12 03:13:06  andrewm
Checkpoint.  Things are working rather well.

Revision 1.12  1997/03/04 06:32:55  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.11  1997/02/23 23:44:12  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.10  1997/02/08 04:37:40  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.9  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.8  1996/12/26 05:55:25  andrewm
Checkpoint, the compiler is working again.

Revision 1.7  1996/12/24 05:20:11  andrewm
Checkpoint.

Revision 1.6  1996/10/01 04:39:14  andrewm
checkpoint and revision

// Revision 1.5  1996/09/22  01:18:22  andrewm
// pre-alpha release
//
// Revision 1.4  1996/08/18  17:57:45  andrewm
// checkpoint
//
// Revision 1.3  1996/07/27  20:56:54  andrewm
// checkpoint
//
// Revision 1.2  1996/06/26  03:14:43  andrewm
// checkpoint
//
// Revision 1.1  1996/06/15  23:53:13  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "MachineGroup.h"
#include "Smachine.h"
#include "MachEvent.h"
#include "State.h"
#include "Transition.h"
#include "CleanName.h"

#include <assert.h>
#include <strstream.h>
#include <iomanip.h>
#include <algorithm>
#include <functional>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: Smachine.cc,v $ $Revision: 1.19 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const char Smachine::_name_name[] = "name" ;
const char Smachine::_events_name[] = "events" ;
const char Smachine::_states_name[] = "states" ;
const char Smachine::_counter_name[] = "counter" ;
const char Smachine::_prolog_name[] = "prolog" ;
const char Smachine::_epilog_name[] = "epilog" ;
const char Smachine::_initial_state_name[] = "initial_state" ;
const char Smachine::_terminal_state_name[] = "terminal_state" ;
const char Smachine::_default_state_name[] = "default_state" ;
const char Smachine::_error_name[] = "error" ;
const char Smachine::_ignore_name[] = "ignore" ;

/*
FRIEND FUNCTIONS
*/

ostream&
operator<<(
	ostream& stream,
	Smachine& machine)
{
	return stream << machine._name.c_str() ;
}

/*
FUNCTION DEFINITIONS
*/

Smachine::
Smachine(
	MachineGroup *parent,
	ChioListIter position,
	const ChioTerm& name) :
		_parent(parent),
		_modified(false),
		_counter(0),
		_name(name),
		_prolog(ChioTerm()),
		_epilog(ChioTerm()),
		_initial_state(NULL),
		_terminal_state(NULL),
		_default_state(NULL)
{
	ChioMap machine_map ;
	ChioMap::pair_iterator_bool result ;
		// Create the sub maps that contains the events, states, and std_states.
	result = machine_map.insert(ChioAssignment(_name_name, name)) ;
	assert(result.second == true) ;
	result = machine_map.insert(ChioAssignment(_events_name,
		ChioValue(ChioValue::MapChioValue))) ;
	assert(result.second == true) ;
	result = machine_map.insert(ChioAssignment(_states_name,
		ChioValue(ChioValue::MapChioValue))) ;
	assert(result.second == true) ;
		// add those assignments at this level
	result = machine_map.insert(ChioAssignment(_counter_name, _counter)) ;
	assert(result.second == true) ;
	result = machine_map.insert(ChioAssignment(_prolog_name, _prolog)) ;
	assert(result.second == true) ;
	result = machine_map.insert(ChioAssignment(_epilog_name, _epilog)) ;
	assert(result.second == true) ;
		// Add your own part of the hierarchy
		// to the parent's map of state machines.
	ChioList& parent_list = _parent->state_machines() ;
	parent_list.insert(position, machine_map) ;
		// Create the "standard" states "ignore" and "error"
	_state_list.push_back(new PseudoState(this, _ignore_name)) ;
	_default_state = new PseudoState(this, _error_name) ;
	_state_list.push_back(_default_state) ;
}

Smachine::
Smachine(
	MachineGroup *parent,
	ChioListIter place) :
		_parent(parent),
		_modified(false)
{
	ChioMap& file_def = *place ;
	_name = file_def[_name_name] ;
	_counter = file_def[_counter_name] ;
	_prolog = file_def[_prolog_name] ;
	_epilog = file_def[_epilog_name] ;

	ChioMapIter found = file_def.find(_events_name) ;
	if (found != file_def.end())
	{
		ChioMap& event_map = (*found).second ;
		for (ChioMapIter e_iter = event_map.begin() ;
			e_iter != event_map.end() ; ++e_iter)
		{
			_event_list.push_back(new MachEvent(this, e_iter)) ;
		}
	}
	else
	{
		ChioMap::pair_iterator_bool result = file_def.insert(
			ChioAssignment(_events_name, ChioValue(ChioValue::MapChioValue))) ;
		assert(result.second == true) ;
	}

	found = file_def.find(_states_name) ;
	if (found != file_def.end())
	{
		ChioMap& state_map = (*found).second ;
		for (ChioMapIter s_iter = state_map.begin() ;
			s_iter != state_map.end() ; ++s_iter)
		{
			const ChioTerm state_name((*s_iter).first) ;
			_state_list.push_back(
				(state_name == _error_name || state_name == _ignore_name) ?
				new PseudoState(this, s_iter) :
				new RealState(this, s_iter)) ;
		}
	}
	else
	{
		ChioMap::pair_iterator_bool result = file_def.insert(
			ChioAssignment(_states_name, ChioValue(ChioValue::MapChioValue))) ;
		assert(result.second == true) ;
	}

		/*
		Check to make sure the "error" and "ignore" special states were
		in the file.  If not, create them so that any subsequent references
		to them can be resolved.
		*/
	if (!find_state(_error_name))
		_state_list.push_back(new PseudoState(this, _error_name)) ;
	if (!find_state(_ignore_name))
		_state_list.push_back(new PseudoState(this, _ignore_name)) ;

	for (StateListIter state_iter = _state_list.begin() ;
		state_iter != _state_list.end() ; ++state_iter)
	{
		State *state = *state_iter ;
		state->resolve_transitions() ;
	}

	const ChioTerm& isn = file_def[_initial_state_name] ;
	_initial_state = find_state(isn) ;
	const ChioTerm& tsn = file_def[_terminal_state_name] ;
	_terminal_state = find_state(tsn) ;
	const ChioTerm& dsn = file_def[_default_state_name] ;
	_default_state = find_state(dsn) ;
}

Smachine::
~Smachine(void)
{
	if (_parent)
	{
		_parent->state_machines().erase(find_place()) ;
		_parent->remove_machine(this) ;
	}

	for (MachEventListIter event_iter = _event_list.begin() ;
		event_iter != _event_list.end() ; ++event_iter)
	{
		MachEvent *event = *event_iter ;
		event->orphan() ;
		delete event ;
	}

	for (StateListIter state_iter = _state_list.begin() ;
		state_iter != _state_list.end() ; ++state_iter)
	{
		State *state = *state_iter ;
		state->orphan() ;
		delete state ;
	}
}

bool Smachine::
modified() const
{
	class MachEventMod
	{
	public:
		bool operator ()(bool needs, MachEvent *ev) {
			return needs || ev->modified() ;
		}
	} ;

	class StateMod
	{
	public:
		bool operator ()(bool needs, State *st) {
			return needs || st->modified() ;
		}
	} ;

	return _modified
		|| accumulate(_event_list.begin(), _event_list.end(),
			false, MachEventMod())
		|| accumulate(_state_list.begin(), _state_list.end(),
			false, StateMod()) ;
}

void Smachine::
modified(
	bool new_modified)
{
	for (MachEventListIter event_iter = _event_list.begin() ;
		event_iter != _event_list.end() ; ++event_iter)
	{
		MachEvent *event = *event_iter ;
		event->modified(new_modified) ;
	}

	for (StateListIter state_iter = _state_list.begin() ;
		state_iter != _state_list.end() ; ++state_iter)
	{
		State *state = *state_iter ;
		state->modified(new_modified) ;
	}

	_modified = new_modified ;
}

bool Smachine::
rename(
	const ChioTerm& new_name)
{
	bool renamed = false ;
	Smachine *machine = _parent->find_machine(new_name) ;
	if (machine)
		cerr << "A state machine by the name of "
			<< new_name << " already exists." << endl ;
	else
	{
		_name = new_name ;
		_modified = true ;
		renamed = true ;
		notify() ;
	}

	return renamed ;
}

void Smachine::
counter(
	int new_counter)
{
	_counter = new_counter ;
	_modified = true ;
	notify() ;
}

void Smachine::
prolog(
	const ChioTerm& new_prolog)
{
	_prolog = new_prolog ;
	_modified = true ;
	notify() ;
}

void Smachine::
epilog(
	const ChioTerm& new_epilog)
{
	_epilog = new_epilog ;
	_modified = true ;
	notify() ;
}

void Smachine::
initial_state(
	State *new_initial_state)
{
	State *old_initial_state = _initial_state ;
	_initial_state = new_initial_state ;
	_modified = true ;
	notify() ;
	if (old_initial_state)
		old_initial_state->notify() ;
	if (_initial_state)
		_initial_state->notify() ;
}

void Smachine::
terminal_state(
	State *new_terminal_state)
{
	_terminal_state = new_terminal_state ;
	_modified = true ;
	notify() ;
}

void Smachine::
default_state(
	State *new_default_state)
{
	_default_state = new_default_state ;
	_modified = true ;
	notify() ;
}

ChioMap& Smachine::
events()
{
	ChioListIter place = find_place() ;
	ChioMap& map = *place ;
	ChioMapIter found = map.find(_events_name) ;
	assert(found != map.end()) ;
	return (ChioMap&)((*found).second) ;
}

ChioMap& Smachine::
states()
{
	ChioListIter place = find_place() ;
	ChioMap& map = *place ;
	ChioMapIter found = map.find(_states_name) ;
	assert(found != map.end()) ;
	return (ChioMap&)((*found).second) ;
}

void Smachine::
sync()
{
	ChioListIter place = find_place() ;
	ChioMap& m = *place ;
	m[_name_name] = _name ;
	m[_counter_name] = _counter ;
	m[_prolog_name] = _prolog ;
	m[_epilog_name] = _epilog ;
	m[_initial_state_name] = _initial_state ?
		_initial_state->name() : ChioTerm() ;
	m[_terminal_state_name] = _terminal_state ?
		_terminal_state->name() : ChioTerm() ;
	m[_default_state_name] = _default_state ?
		_default_state->name() : ChioTerm() ;

	for (MachEventListIter event_iter = _event_list.begin() ;
		event_iter != _event_list.end() ; ++event_iter)
	{
		MachEvent *event = *event_iter ;
		event->sync() ;
	}

	for (StateListIter state_iter = _state_list.begin() ;
		state_iter != _state_list.end() ; ++state_iter)
	{
		State *state = *state_iter ;
		state->sync() ;
	}
}

ChioListIter Smachine::
find_place()
{
	ChioList& m_list = _parent->state_machines() ;
	ChioListIter ml_iter = m_list.begin() ;
	MachineGroup::SmachineList& sm_list = _parent->machine_list() ;
	for (MachineGroup::SmachineListIter sm_iter = sm_list.begin() ;
		sm_iter != sm_list.end() ; ++sm_iter, ++ml_iter)
	{
		assert(ml_iter != m_list.end()) ;
		if (*sm_iter == this)
			break ;
	}

	assert(ml_iter != m_list.end()) ;
	return ml_iter ;
}

State *Smachine::
find_state(
	const ChioTerm& name) const
{
	class StateNameCmp :
		public binary_function<State *, const ChioTerm, bool>
	{
	public:
		bool operator ()(State *s, const ChioTerm n) const {
			return s->name() == n ;
		}
	} ;

	StateListConstIter end = _state_list.end() ;
	StateListConstIter found = find_if(
		_state_list.begin(), end, bind2nd(StateNameCmp(), name)) ;

	return found == end ? (State *)NULL : *found ;
}

State *Smachine::
find_state_at_location(
	const Point& location) const
{
	class StateLocator :
		public binary_function<State *, const Point, bool>
	{
	public:
		bool operator ()(State *s, const Point l) const {
			return s->contains(l) ;
		}
	} ;

	StateListConstIter end = _state_list.end() ;
	StateListConstIter found = find_if(
		_state_list.begin(), end, bind2nd(StateLocator(), location)) ;

	return found == end ? (State *)NULL : *found ;
}

Transition *Smachine::
find_transition_at_location(
	const Point& location) const
{
	for (StateListConstIter s_iter = _state_list.begin() ;
		s_iter != _state_list.end() ; ++s_iter)
	{
		State *state = *s_iter ;
		if (Transition *trans = state->find_transition_at_location(location))
			return trans ;
	}

	return (Transition *)NULL ;
}

State *Smachine::
add_state(
	const ChioTerm& name)
{
	State *state = do_add_state(name) ;
	notify() ;
	return state ;
}

State *Smachine::
add_state()
{
	State *state = do_add_state(new_state_name()) ;
	notify() ;
	return state ;
}

State *Smachine::
add_state(
	const Point& center)
{
	State *state = do_add_state(new_state_name()) ;
	state->move(center) ;
	notify() ;
	return state ;
}

void Smachine::
remove_state(
	State *state)
{
	check_itd_states(state, NULL) ;
	state->orphan() ;
	_state_list.remove(state) ;
	_modified = true ;
	notify() ;
}

MachEvent *Smachine::
find_event(
	const ChioTerm& name) const
{
	class MachEventNameCmp :
		public binary_function<MachEvent *, const ChioTerm, bool>
	{
	public:
		bool operator ()(MachEvent *e, const ChioTerm n) const {
			return e->name() == n ;
		}
	} ;

	MachEventListConstIter end = _event_list.end() ;
	MachEventListConstIter found = find_if(
		_event_list.begin(), end, bind2nd(MachEventNameCmp(), name)) ;

	return found == end ? (MachEvent *)NULL : *found ;
}

MachEvent *Smachine::
add_event(
	const ChioTerm& name)
{
	MachEvent *event = find_event(name) ;
	if (!event)
	{
		event = new MachEvent(this, name) ;
		_event_list.push_back(event) ;
		_modified = true ;
		notify() ;
	}
	return event ;
}

void Smachine::
remove_event(
	MachEvent *event)
{
	event->orphan() ;
	_event_list.remove(event) ;
	_modified = true ;
	notify() ;
}

Rectangle Smachine::
bounding_box()
{
	Rectangle rect ;

	StateListIter state_iter = _state_list.begin() ;
	StateListIter end_iter = _state_list.end() ;
	if (state_iter != end_iter)
	{
		State *state = *state_iter ;
		rect = state->bounding_box() ;
		++state_iter ;
		for ( ; state_iter != end_iter ; ++state_iter)
		{
			state = *state_iter ;
			rect.enclose(state->bounding_box()) ;
		}
	}
	return rect ;
}

void Smachine::
move_relative(
	const Point& p)
{
	for (StateListIter s_iter = _state_list.begin() ;
		s_iter != _state_list.end() ; ++s_iter)
	{
		(*s_iter)->move_relative(p) ;
	}
}

void Smachine::
select_all(
	bool new_selected)
{
	for (StateListIter s_iter = _state_list.begin() ;
		s_iter != _state_list.end() ; ++s_iter)
	{
		(*s_iter)->select_all(new_selected) ;
	}
}

void Smachine::
select_if_within(
	const Rectangle& box)
{
	for (StateListIter s_iter = _state_list.begin() ;
		s_iter != _state_list.end() ; ++s_iter)
	{
		(*s_iter)->select_if_within(box) ;
	}
}

void Smachine::
check_itd_states(
	State *test_state,
	State *new_state)
{
	if (_initial_state == test_state)
		_initial_state = new_state ;
	if (_terminal_state == test_state)
		_terminal_state = new_state ;
	if (_default_state == test_state)
		_default_state = new_state ;
}

State *Smachine::
do_add_state(
	const ChioTerm& name)
{
	State *state = find_state(name) ;
	if (!state)
	{
		state = new RealState(this, name) ;
		_state_list.push_back(state) ;
		_modified = true ;
	}
	return state ;
}

string Smachine::
new_state_name()
{
	ostrstream str_buf ;
	str_buf << "State" << setw(3) << setfill('0') << _counter++ << ends ;
	return string(str_buf.str()) ;
}
