/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
MachineGroup.cc -- object that represents a set of related state machines.

$RCSfile: MachineGroup.cc,v $
$Revision: 1.16 $
$Date: 1997/07/02 04:45:12 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: MachineGroup.cc,v $
Revision 1.16  1997/07/02 04:45:12  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.15  1997/05/15 04:14:45  andrewm
Checkpoint.  Reworked the low level file format stuff to contain
proper lists rather than maps keyed to binary numbers.
This point represents the entire program working with this file format
change.

Revision 1.14  1997/04/24 03:20:47  andrewm
Checkpoint.  All features in.  Starting test cycle.

Revision 1.13  1997/04/08 04:34:31  andrewm
Checkpoint as dialogs are added.
Decided to make sure that the name part of a name/value pair in
the hierarchical file is not used for id purposes.  Specifically
state machine names need to be contained within the machine not as
the key to the machine.

Revision 1.12  1997/03/30 02:07:30  andrewm
Checkpoint.  Things working well. About to change the way
machines are stored in the backing file.

Revision 1.11  1997/03/12 03:13:06  andrewm
Checkpoint.  Things are working rather well.

Revision 1.10  1997/03/04 06:32:54  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

Revision 1.9  1997/02/23 23:44:12  andrewm
Checkpoint.  Things seem to be working reasonably well.

Revision 1.8  1997/02/08 04:37:39  andrewm
Checkpoint before returning to work on the GUI portion.

Revision 1.7  1997/01/23 06:20:29  andrewm
Checkpoint as base and graphics classes are operating together.

Revision 1.6  1996/12/24 05:20:10  andrewm
Checkpoint.

Revision 1.5  1996/11/27 01:25:53  andrewm
Another checkpoint before I go off and figure out what to do about
the graphical representations so that I'll be able to get some post script
output one of these days.

Revision 1.4  1996/09/22 01:45:26  andrewm
pre-alpha release

// Revision 1.3  1996/08/18  17:57:45  andrewm
// checkpoint
//
// Revision 1.3  1996/08/18  17:57:45  andrewm
// checkpoint
//
// Revision 1.2  1996/06/26  03:14:43  andrewm
// checkpoint
//
// Revision 1.1  1996/06/15  23:53:13  andrewm
// Initial revision
//
 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "MachineGroup.h"
#include "Smachine.h"

#include <strstream.h>
#include <algorithm>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: MachineGroup.cc,v $ $Revision: 1.16 $" ;

/*
STATIC MEMBER DEFINITIONS
*/
const char MachineGroup::_author_name[] = "author" ;
const char MachineGroup::_revision_name[] = "revision" ;
const char MachineGroup::_group_prolog_name[] = "group_prolog" ;
const char MachineGroup::_group_epilog_name[] = "group_epilog" ;
const char MachineGroup::_state_machines_name[] = "state_machines" ;

/*
FUNCTION DEFINITIONS
*/

MachineGroup::
MachineGroup(
	const char *author,
	const char *revision) :
		_modified(true),
		_author(ChioTerm(author ? author : getenv("USER"))),
		_revision(ChioTerm(revision ? revision : "$""Revision""$"))
{
	ChioMap::pair_iterator_bool result ;

		// Add the "author", "revision", etc. assignments, i.e. the
		// straight assignments at this level.
	result = _file_map.insert(ChioAssignment(_author_name, _author)) ;
	assert(result.second == true) ;
	result = _file_map.insert(ChioAssignment(_revision_name, _revision)) ;
	assert(result.second == true) ;
	result = _file_map.insert(ChioAssignment(_group_prolog_name,
		ChioValue(""))) ;
	assert(result.second == true) ;
	result = _file_map.insert(ChioAssignment(_group_epilog_name,
		ChioValue(""))) ;
	assert(result.second == true) ;
		// Create the list that contains the state machines.
	result = _file_map.insert(ChioAssignment(_state_machines_name,
		ChioValue(ChioValue::ListChioValue))) ;
	assert(result.second == true) ;
}

MachineGroup::
MachineGroup(
	const ChioMap& file_map) :
		_file_map(file_map),
		_modified(false)
{
	_author = _file_map[_author_name] ;
	_revision = _file_map[_revision_name] ;
	_group_prolog = _file_map[_group_prolog_name] ;
	_group_epilog = _file_map[_group_epilog_name] ;
	ChioMapIter found = _file_map.find(_state_machines_name) ;
	if (found != _file_map.end())
	{
		ChioList& mlist = (*found).second ;
		for (ChioListIter m_iter = mlist.begin() ;
			m_iter != mlist.end() ; ++m_iter)
		{
			_machine_list.push_back(new Smachine(this, m_iter)) ;
		}
	}
	else
	{
		ChioMap::pair_iterator_bool result = _file_map.insert(
			ChioAssignment(_state_machines_name,
			ChioValue(ChioValue::ListChioValue))) ;
		assert(result.second == true) ;
	}
}

MachineGroup::
~MachineGroup(void)
{
	for (SmachineListIter sm_iter = _machine_list.begin() ;
		sm_iter != _machine_list.end() ; ++sm_iter)
	{
		Smachine *sm = *sm_iter ;
		sm->orphan() ;
		delete sm ;
	}
}

bool MachineGroup::
modified() const
{
	class SmachineMod
	{
	public:
		bool operator ()(bool needs, Smachine *sm) {
			return needs || sm->modified() ;
		}
	} ;

	return _modified
		|| accumulate(_machine_list.begin(), _machine_list.end(),
			false, SmachineMod()) ;
}

void MachineGroup::
modified(
	bool new_modified)
{
	for (SmachineListIter sm_iter = _machine_list.begin() ;
		sm_iter != _machine_list.end() ; ++sm_iter)
	{
		Smachine *sm = *sm_iter ;
		sm->modified(new_modified) ;
	}
	_modified = new_modified ;
}

void MachineGroup::
sync()
{
		// Cause everything to be updated to the backing store.
	_file_map[_author_name] = _author ;
	_file_map[_revision_name] = _revision ;
	_file_map[_group_prolog_name] = _group_prolog ;
	_file_map[_group_epilog_name] = _group_epilog ;
	for (SmachineListIter sm_iter = _machine_list.begin() ;
		sm_iter != _machine_list.end() ; ++sm_iter)
	{
		Smachine *sm = *sm_iter ;
		sm->sync() ;
	}
}

void MachineGroup::
author(
	const ChioTerm& new_author)
{
	_author = new_author ;
	_modified = true ;
	notify() ;
}

void MachineGroup::
revision(
	const ChioTerm& new_revision)
{
	_revision = new_revision ;
	_modified = true ;
	notify() ;
}

void MachineGroup::
group_prolog(
	const ChioTerm& new_group_prolog)
{
	_group_prolog = new_group_prolog ;
	_modified = true ;
	notify() ;
}

void MachineGroup::
group_epilog(
	const ChioTerm& new_group_epilog)
{
	_group_epilog = new_group_epilog ;
	_modified = true ;
	notify() ;
}

ChioList& MachineGroup::
state_machines()
{
	ChioMapIter found = _file_map.find(_state_machines_name) ;
	assert(found != _file_map.end()) ;
	return (ChioList&)((*found).second) ;
}

Smachine *MachineGroup::
find_machine(
	const ChioTerm& name) const
{
	class SmachineNameCmp :
		public binary_function<Smachine *, const ChioTerm, bool>
	{
	public:
		bool operator ()(Smachine *sm, const ChioTerm n) const {
			return sm->name() == n ;
		}
	} ;

	SmachineListConstIter end = _machine_list.end() ;
	SmachineListConstIter found = find_if(
		_machine_list.begin(), end, bind2nd(SmachineNameCmp(), name)) ;

	return found == end ? (Smachine *)NULL : *found ;
}

Smachine *MachineGroup::
add_machine(
	const ChioTerm& name)
{
	Smachine *machine = find_machine(name) ;
	if (!machine)
	{
		machine = new Smachine(this, state_machines().end(), name) ;
		_machine_list.push_back(machine) ;
		_modified = true ;
		notify() ;
	}
	return machine ;
}

Smachine *MachineGroup::
add_machine()
{
	return add_machine(new_machine_name()) ;
}

Smachine *MachineGroup::
insert_machine(
	Smachine *current,
	const ChioTerm& name)
{
	Smachine *machine = find_machine(name) ;
	if (!machine)
	{
		machine = new Smachine(this, current->find_place(), name) ;
		SmachineListIter sm_pos = find(_machine_list.begin(),
			_machine_list.end(), current) ;
		assert(sm_pos != _machine_list.end()) ;
		_machine_list.insert(sm_pos, machine) ;
		_modified = true ;
		notify() ;
	}
	return machine ;
}

Smachine *MachineGroup::
insert_machine(
	Smachine *current)
{
	return insert_machine(current, new_machine_name()) ;
}

void MachineGroup::
remove_machine(
	Smachine *machine)
{
	machine->orphan() ;
	_machine_list.remove(machine) ;
	_modified = true ;
	notify() ;
}

ChioTerm MachineGroup::
new_machine_name()
{
	int trial_number = _machine_list.size() ;
	for (;;)
	{
		ostrstream name_str ;
		name_str << "Machine" << trial_number << ends ;
		ChioTerm m_name(name_str.str()) ;
		Smachine *machine = find_machine(m_name) ;
		if (machine == NULL)
			return m_name ;

		++trial_number ;
	}
}

/*
FRIEND FUNCTIONS
*/

/*
PRIVATE FUNCTIONS
*/

/*
TEMPLATE INSTANTIATION
*/
