/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:

$RCSfile: GsmDocument.cc,v $
$Revision: 1.11 $
$Date: 1997/07/02 04:45:06 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: GsmDocument.cc,v $
Revision 1.11  1997/07/02 04:45:06  andrewm
Added copyright and license notices to the tops of the files.

Revision 1.10  1997/06/06 04:34:24  andrewm
Checkpoint.  Changed the Subject / Observer code to include a
pointer to the changed subject in the call to "update".  This allowed
a given observer to observe muliple subjects.  Then I modified
various dialogs, especially those in that control the various fields
in the state machine to observe multiple subjects as necessary to
insure that all their fields were updated properly.

Revision 1.9  1997/05/31 21:12:41  andrewm
Checkpoint.  Things are working well.

Revision 1.8  1997/05/20 05:15:30  andrewm
Checkpoint.  Improved the structure of the "State" class and this
had quite some ripple effects.  However, now there is an abstract
class "State" with two concrete classes "PseudoState" to represent
error and ignore and "RealState" to represent the user specified
states.  Also improved the text display of event names on the transitions.

Revision 1.7  1997/04/16 04:06:14  andrewm
Checkpoint, as last major dialog is in.

Revision 1.6  1997/04/12 21:15:00  andrewm
Checkpoint after adding the event dialog.

Revision 1.5  1997/04/08 04:34:29  andrewm
Checkpoint as dialogs are added.
Decided to make sure that the name part of a name/value pair in
the hierarchical file is not used for id purposes.  Specifically
state machine names need to be contained within the machine not as
the key to the machine.

Revision 1.4  1997/03/30 02:07:28  andrewm
Checkpoint.  Things working well. About to change the way
machines are stored in the backing file.

Revision 1.3  1997/03/18 06:51:02  andrewm
Checkpoint.  Mouse select, insert, and delete working.
Some changes to improve robustness in the face of an arbitrary input file.

Revision 1.2  1997/03/12 03:13:05  andrewm
Checkpoint.  Things are working rather well.

Revision 1.1  1997/03/04 06:32:52  andrewm
Another check point.  The editor can draw output from files.
The crashing during the dtor for MachineGroup is fixed.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include "GsmDocument.h"
#include "Application.h"
#include "DocPageState.h"
#include "MachineGroupDialogDirector.h"
#include "SmachineDialogDirector.h"
#include "MachEventDialogDirector.h"
#include "StateDialogDirector.h"
#include "Smachine.h"
#include "PageGroup.h"
#include "Page.h"
#include "Chio.h"
#include "GraphicRenderer.h"
#include "StateCheckVisitor.h"
#include "TransitionCheckVisitor.h"

#include <fstream.h>
#include <algorithm>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: GsmDocument.cc,v $ $Revision: 1.11 $" ;

/*
STATIC MEMBER DEFINITIONS
*/

/*
FUNCTION DEFINITIONS
*/

GsmDocument::
GsmDocument() :
		Document(),
		_machine_group(NULL),
		_machine_list(NULL),
		_page_group(NULL),
		_doc_page_state(NULL),
		_mg_dialog(NULL),
		_sm_dialog(NULL),
		_me_dialog(NULL),
		_st_dialog(NULL)
			
{
	Application *appl = Application::instance() ;
	_mg_dialog = new MachineGroupDialogDirector("mg_dialog", appl, *this) ;
	_sm_dialog = new SmachineDialogDirector("sm_dialog", appl, *this) ;
	_me_dialog = new MachEventDialogDirector("me_dialog", appl, *this) ;
	_st_dialog = new StateDialogDirector("st_dialog", appl, *this) ;
}

GsmDocument::
~GsmDocument()
{
	delete _st_dialog ;
	delete _me_dialog ;
	delete _sm_dialog ;
	delete _mg_dialog ;
	delete _doc_page_state ;
	delete _machine_group ;
}

void GsmDocument::
open(
	const char *filename,
	bool draw)
{
	close() ;

	_filename = FileName(filename) ;
	if (!check_file())
		return ;

	cerr << "Opening document \"" << _filename << "\"." << endl ;
	ifstream in_file(_filename.c_str()) ;
	if (!in_file)
	{
		cerr << "Cannot open file \"" << _filename
			<< "\" for reading, " << strerror(errno) << '.' << endl ;
		return ;
	}

	Application::instance()->title(_filename.c_str()) ;
	ChioMap file_map ;
	in_file >> file_map ;
	_machine_group = new MachineGroup(file_map) ;
	_machine_list = &_machine_group->machine_list() ;
	_machine_iter = _machine_list->begin() ;
	if (_machine_iter == _machine_list->end()) // a file that has no machines
	{
		_machine_group->add_machine() ;
		_machine_iter = _machine_list->begin() ;
		assert(_machine_iter != _machine_list->end()) ;
		set_insert_mode() ;
	}
	else
		set_change_mode() ;
	_page_group = new PageGroup(_machine_group) ;

	if (draw)
		refresh() ;
}

void GsmDocument::
new_doc()
{
	close() ;
	_filename = FileName("NoName.gsm") ;
	Application::instance()->title(_filename.c_str()) ;
	cerr << "Creating new document \"" << _filename << "\"." << endl ;

	_machine_group = new MachineGroup() ;
	_machine_group->add_machine() ;
		/*
		Despite the fact that the "new" MachineGroup is "modified" and
		would need to be saved, the fact that it can be easily created
		here and that to the user it appears empty, then we mark it
		as unmodified so that subsequent actions won't try to save it.
		*/
	set_modified(false) ;
	_machine_list = &_machine_group->machine_list() ;
	_machine_iter = _machine_list->begin() ;
	set_insert_mode() ;
	_page_group = new PageGroup(_machine_group) ;
	refresh() ;
}

void GsmDocument::
close()
{
	if (_machine_group)
	{
		cerr << "Closing document \"" << _filename << "\"." << endl ;
		Application::instance()->title("") ;
		delete _doc_page_state ;
		_doc_page_state = NULL ;
		delete _machine_group ;
		_machine_group = NULL ;
		_machine_list = NULL ;
		_page_group = NULL ;
	}
}

void GsmDocument::
save()
{
	if (_machine_group)
	{
		ofstream out_file(_filename.c_str()) ;
		if (!out_file)
		{
			cerr << "Cannot open file \"" << _filename
				<< "\" for writing, " << strerror(errno) << '.' << endl ;
			return ;
		}
		Application::instance()->title(_filename.c_str()) ;
		cerr << "Saving document to \"" << _filename << "\"." << endl ;
		out_file << _machine_group->file_map() ;
		set_modified(false) ;
		notify() ;
	}
}

void GsmDocument::
save(
	const char *filename)
{
	_filename = FileName(filename) ;
	save() ;
}

bool GsmDocument::
modified()
{
	bool needs = false ;

	if (needs == false && _mg_dialog)
		needs = _mg_dialog->modified() ;
	if (needs == false && _machine_group)
		needs = _machine_group->modified() ;

	return needs ;
}

void GsmDocument::
set_modified(
	bool new_modified)
{
	if (_mg_dialog)
		_mg_dialog->modified(new_modified) ;
	if (_machine_group)
		_machine_group->modified(new_modified) ;
}

void GsmDocument::
set_page_mode()
{
	Smachine *cm = current_machine() ;
	if (cm)
	{
		if (cm->state_list().size() == 2) // error and ignore are always
			set_insert_mode() ;
		else
			set_change_mode() ;
	}
}

void GsmDocument::
redraw()
{
	if (_machine_group)
		if (Smachine *cm = current_machine())
			if (Page *current_page = _page_group->find_page(cm))
				current_page->draw() ;
}

void GsmDocument::
refresh()
{
	notify() ;
	GraphicRenderer::instance().clear() ;
	redraw() ;
}

void GsmDocument::
button_1_press(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_1_press(p) ;
}

void GsmDocument::
button_1_release(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_1_release(p) ;
}

void GsmDocument::
button_1_motion(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_1_motion(p) ;
}

void GsmDocument::
button_2_press(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_2_press(p) ;
}

void GsmDocument::
button_2_release(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_2_release(p) ;
}

void GsmDocument::
button_2_motion(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_2_motion(p) ;
}

void GsmDocument::
button_3_press(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_3_press(p) ;
}

void GsmDocument::
button_3_release(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_3_release(p) ;
}

void GsmDocument::
button_3_motion(
	const Point& p)
{
	if (_doc_page_state)
		_doc_page_state->button_3_motion(p) ;
}

void GsmDocument::
select_all()
{
	if (Smachine *cm = current_machine())
		cm->select_all(true) ;
}

void GsmDocument::
deselect_all()
{
	if (Smachine *cm = current_machine())
		cm->select_all(false) ;
}

void GsmDocument::
set_change_mode()
{
	delete _doc_page_state ;
	_doc_page_state = new DocPageChangeState(*this) ;
	notify() ;
}

void GsmDocument::
set_insert_mode()
{
	delete _doc_page_state ;
	_doc_page_state = new DocPageInsertState(*this) ;
	notify() ;
}

void GsmDocument::
set_delete_mode()
{
	delete _doc_page_state ;
	_doc_page_state = new DocPageDeleteState(*this) ;
	notify() ;
}

void GsmDocument::
first_page()
{
	Smachine *cm = current_machine() ;
	if (cm)
		cm->select_all(false) ;
	_machine_iter = _machine_list->begin() ;
	set_page_mode() ;
	refresh() ;
}

void GsmDocument::
last_page()
{
	Smachine *cm = current_machine() ;
	if (cm)
		cm->select_all(false) ;
	cm = _machine_list->back() ;
	_machine_iter = find(_machine_list->begin(), _machine_list->end(), cm) ;
	assert(_machine_iter != _machine_list->end()) ;
	set_page_mode() ;
	refresh() ;
}

void GsmDocument::
next_page()
{
	Smachine *cm = current_machine() ;
	if (cm)
		cm->select_all(false) ;
	++_machine_iter ;
	if (_machine_iter == _machine_list->end())
	{
		if (cm && cm->state_list().size() == 2) // error and ignore are always
		{
			_machine_iter = _machine_list->begin() ;
		}
		else
		{
			Smachine *machine = _machine_group->add_machine() ;
			if (machine)
			{
				_machine_iter = find(_machine_list->begin(),
					_machine_list->end(), machine) ;
				assert(_machine_iter != _machine_list->end()) ;
				set_insert_mode() ;
			}
			else
			{
				cerr << "Cannot add new machine." << endl ;
				_machine_iter = _machine_list->begin() ;
			}
		}
	}
	set_page_mode() ;
	refresh() ;
}

void GsmDocument::
previous_page()
{
	Smachine *cm = current_machine() ;
	if (cm)
		cm->select_all(false) ;
	if (_machine_iter == _machine_list->begin())
	{
		cm = _machine_list->back() ;
		_machine_iter = find(_machine_list->begin(), _machine_list->end(), cm) ;
		assert(_machine_iter != _machine_list->end()) ;
	}
	else
		--_machine_iter ;
	set_page_mode() ;
	refresh() ;
}

void GsmDocument::
delete_page()
{
	Smachine *cm = current_machine() ;
	if (cm)
	{
		cm->select_all(false) ;
		++_machine_iter ;
		delete cm ;
	}
	if (_machine_list->empty())
	{
		_machine_group->add_machine() ;
		_machine_iter = _machine_list->begin() ;
		assert(_machine_iter != _machine_list->end()) ;
		set_insert_mode() ;
	}
	if (_machine_iter == _machine_list->end())
	{
		cm = _machine_list->back() ;
		_machine_iter = find(_machine_list->begin(), _machine_list->end(), cm) ;
		assert(_machine_iter != _machine_list->end()) ;
	}
	set_page_mode() ;
	refresh() ;
}

void GsmDocument::
edit_document_params()
{
	_mg_dialog->show_dialog() ;
}

void GsmDocument::
edit_page_params()
{
	_sm_dialog->show_dialog() ;
}

void GsmDocument::
edit_page_events()
{
	_me_dialog->show_dialog() ;
}

void GsmDocument::
edit_component_params()
{
	_st_dialog->show_dialog() ;
}

void GsmDocument::
check_states()
{
	assert(_machine_group != NULL) ;
	StateCheckVisitor scv ;
	_machine_group->accept_visitor(scv) ;
	cerr << scv ;
	cerr << scv.discrepancies() << " state errors found." << endl ;
}

void GsmDocument::
check_transitions()
{
	assert(_machine_group != NULL) ;
	TransitionCheckVisitor tcv ;
	_machine_group->accept_visitor(tcv) ;
	cerr << tcv ;
	cerr << tcv.discrepancies() << " transition errors found." << endl ;
}

Smachine *GsmDocument::
current_machine() const
{
	return _machine_iter == _machine_list->end()
		? (Smachine *)NULL : *_machine_iter ;
}
