/*
 *++
COPYRIGHT:
This file is part of the GSM Suite, a set of programs for
manipulating state machines in a graphical fashion.
Copyright (C) 1996, 1997  G. Andrew Mangogna.

LICENSE:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.,
59 Temple Place - Suite 330,
Boston, MA  02111-1307, USA.

MODULE:
	gsm2ps -- generate PostScript from GSM files

$RCSfile: gsm2ps.cc,v $
$Revision: 1.5 $
$Date: 1997/07/14 02:30:33 $

ABSTRACT:

CONDITIONAL COMPILATION:

MODIFICATION HISTORY:
$Log: gsm2ps.cc,v $
Revision 1.5  1997/07/14 02:30:33  andrewm
Checkpoint.  Rework makefiles.  Rework man pages.  Fixed PostScript problems.

 *--
 */

/*
PRAGMAS
*/
#ifdef __GNUG__
#	pragma implementation
#endif /* __GNUG__ */

/*
INCLUDE FILES
*/
#include <fstream.h>
#include <strstream.h>
#include <GetOpt.h>
#include <stdlib.h>

#include "Chio.h"
#include "MachineGroup.h"
#include "Smachine.h"
#include "PageGroup.h"
#include "Page.h"
#include "PSGraphicRenderer.h"
#include "PageSize.h"

#include <string>

/*
MACRO DEFINITIONS
*/

/*
TYPE DEFINITIONS
*/

/*
EXTERNAL FUNCTION REFERENCES
*/

/*
FORWARD FUNCTION REFERENCES
*/
static Rectangle bounding_all_pages(MachineGroup& machine_group) ;

/*
FORWARD CLASS REFERENCES
*/

/*
EXTERNAL DATA REFERENCES
*/

/*
EXTERNAL DATA DEFINITIONS
*/

/*
STATIC DATA ALLOCATION
*/
static char rcsid[] = "@(#) $RCSfile: gsm2ps.cc,v $ $Revision: 1.5 $" ;

/*
FUNCTION DEFINITIONS
*/

main(
	int argc,
	char **argv)
{
	char *progname = *argv ;
	string media_name ;
	float page_width ;
	float page_height ;
	float top_margin ;
	float bottom_margin ;
	float left_margin ;
	float right_margin ;
	PSGraphicRenderer::Orientation orient = PSGraphicRenderer::BestFit ;
	char *start_page_arg = NULL ;
	char *end_page_arg = NULL ;

	char *size_name = getenv("GSM2PS_PAGESIZE") ;
	if (size_name == NULL)
		size_name = "letter" ;
	char *margins = getenv("GSM2PS_MARGINS") ;
	if (margins == NULL)
		margins = "12.7:12.7:12.7:12.7" ; // in mm top:bot:left:right

	GetOpt getopt(argc, argv, "plbz:m:s:e:") ;
	int option_char = EOF ;

	while ((option_char = getopt()) != EOF)
	{
		switch (option_char)
		{
		case 'p':
			orient = PSGraphicRenderer::Portrait ;
			break ;

		case 'l':
			orient = PSGraphicRenderer::Landscape ;
			break ;

		case 'b':
			orient = PSGraphicRenderer::BestFit ;
			break ;

		case 'z':
			size_name = getopt.optarg ;
			break ;

		case 'm':
			margins = getopt.optarg ;
			break ;

		case 's':
			start_page_arg = getopt.optarg ;
			break ;

		case 'e':
			end_page_arg = getopt.optarg ;
			break ;

		case ':':
		case '?':
		default:
			cerr << "Usage: " << progname <<
				" [-p | -l | -b] "
				"[-z page_size] "
				"[-m tm:bm:lm:rm] "
				"[-s start_page] "
				"[-e end_page] "
				"gsmfile ..."
				<< endl ;
			return 1 ;
		}
	}

	PageSize *size = PageSize::find(size_name) ;
	if (size)
	{
		media_name = size->name() ;
		page_width = size->width() ;
		page_height = size->height() ;
	}
	else
	{
		istrstream size_stream(size_name) ;
		size_stream >> page_width ;
		size_stream.ignore() ;
		size_stream >> page_height ;
		if (!size_stream)
		{
			cerr << progname << ": Unknown paper size \"" << size_name
				<< "\"." << endl ;
			return 2 ;
		}
		else
			media_name = "custom" ;
	}

	istrstream margin_stream(margins) ;
	margin_stream >> top_margin ;
	margin_stream.ignore() ;
	margin_stream >> bottom_margin ;
	margin_stream.ignore() ;
	margin_stream >> left_margin ;
	margin_stream.ignore() ;
	margin_stream >> right_margin ;
	if (!margin_stream)
	{
		cerr << progname << ": Bad margin specification \"" <<
			margins << "\"." << endl ;
		return 2 ;
	}

	PSGraphicRenderer psr(cout) ;
	psr.page_width(page_width) ;
	psr.page_height(page_height) ;
	psr.orientation(orient) ;
	psr.top_margin(top_margin) ;
	psr.bottom_margin(bottom_margin) ;
	psr.left_margin(left_margin) ;
	psr.right_margin(right_margin) ;
	GraphicRenderer::instance(&psr) ;

	int start_page = -1 ;
	int end_page = -1 ;
	if (start_page_arg)
	{
		istrstream sp_stream(start_page_arg) ;
		sp_stream >> start_page ;
		if (!sp_stream || start_page <= 0)
		{
			cerr << progname << ": Bad starting page argument \"" <<
				start_page_arg << "\"." << endl ;
			return 2 ;
		}
	}
	if (end_page_arg)
	{
		istrstream end_stream(end_page_arg) ;
		end_stream >> end_page ;
		if (!end_stream || end_page <= 0)
		{
			cerr << progname << ": Bad ending page argument \"" <<
				end_page_arg << "\"." << endl ;
			return 2 ;
		}
	}
	if (start_page > 0)
	{
		if (end_page > 0 && end_page < start_page)
		{
			cerr << progname << ": Ending page number \"" << end_page <<
				"\" is less than starting page number \"" <<
				start_page << "\"." << endl ;
			return 2 ;
		}
	}

	int return_code = 0 ;
	while (getopt.optind < argc)
	{
		string filename(argv[getopt.optind++]) ;

		ifstream in_file(filename.c_str()) ;
		if (!in_file)
		{
			cerr << progname << ": Cannot open file \"" << filename
				<< "\" for reading." << endl ;
			return_code = 3 ;
			continue ;
		}

		ChioMap file_map ;
		in_file >> file_map ;
		MachineGroup machine_group(file_map) ;
		PageGroup *page_group = new PageGroup(&machine_group) ;

		PageGroup::PageList& pl = page_group->page_list() ;
		PageGroup::PageListIter p_iter = pl.begin() ;
		int max_page = pl.size() ;
		if (start_page > max_page)
		{
			cerr << progname << ": Starting page number of \"" << start_page
				<< "\" is greater than the number of pages in the file."
				<< endl ;
			return_code = 3 ;
			continue ;
		}
		if (end_page > max_page)
		{
			cerr << progname << ": Ending page number of \"" << end_page
				<< "\" is greater than the number of pages in the file."
				<< endl ;
			return_code = 3 ;
			continue ;
		}
		int current_end = end_page < 0 ? max_page : end_page ;
		int current_page = 1 ;
		if (start_page > 0)
			while (current_page < current_end && current_page < start_page)
			{
				++p_iter ;
				++current_page ;
			}

		Rectangle bounding(bounding_all_pages(machine_group)) ;
		psr.start(media_name, progname, bounding) ;

		for ( ; p_iter != pl.end() ; ++p_iter)
		{
			Page *page = *p_iter ;
			psr.start_page(page->machine()->bounding_box()) ;
			page->draw() ;
			psr.end_page() ;
			if (++current_page >= current_end)
				break ;
		}
		psr.end() ;
	}
	return return_code ;
}

static Rectangle
bounding_all_pages(
	MachineGroup& machine_group)
{
	Rectangle r ;
	MachineGroup::SmachineList& sm_list = machine_group.machine_list() ;
	MachineGroup::SmachineListIter sm_iter = sm_list.begin() ;
	MachineGroup::SmachineListIter sm_end = sm_list.end() ;
	if (sm_iter != sm_end)
	{
		Smachine *machine = *sm_iter ;
		r = machine->bounding_box() ;
		++sm_iter ;

		for ( ; sm_iter != sm_end ; ++sm_iter)
		{
			machine = *sm_iter ;
			r.enclose(machine->bounding_box()) ;
		}
	}
	return r ;
}
