/*
 * Copyright (c) 1980 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* Modified for the Linux SCSI tape driver by Kai Makisara */
/* Modifications by tino@augsburg.net:
 *	Status on unknown tape drives
 *	Now knows linux MT_TAPE_INFO (so just recompile)
 *	MTFSS, MTBSS, MTRAS? etc.
 *	Numeric command numbers
 */

#ifndef lint
char copyright[] =
"@(#) Copyright (c) 1980 The Regents of the University of California.\n\
 All rights reserved.\n";
#endif /* not lint */

#ifndef lint
static char sccsid[] = "@(#)mt.c	5.6 (Berkeley) 6/6/91";
#endif /* not lint */

/*
 * mt --
 *   magnetic tape manipulation program
 */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

struct commands
  {
    char	*c_name;
    int 	c_code;
    int		c_ronly;
    char	*help;
  } com[] =
  {
    /* pseudo commands	*/
    { "status",		MTNOP,		-1,	"print status"			},
#ifdef	MTTELL
    { "tell",		MTTELL,		-1,	"tell position"			},
#endif
    /* standard commands	*/
    { "bsf",		MTBSF,		1,	"backward space filemarks"	},
    { "bsfm",		MTBSFM,		1,	"ditto"				},
    { "bsr",		MTBSR,		1,	"backward space records"	},
#ifdef	MTBSS
    { "bss",		MTBSS,		1,	"backward space setmarks"	},
#endif
    { "fsf",		MTFSF,		1,	"forward  space filemarks"	},
    { "fsfm",		MTFSFM,		1,	"ditto"				},
    { "fsr",		MTFSR,		1,	"forward  space records"	},
#ifdef	MTFSS
    { "fss",		MTFSS,		1,	"forward  space setmarks"	},
#endif
#ifdef	MTRESET
    { "reset",		MTRESET,	1,	"reset tape (if supported)"	},
#endif
#ifdef	MTRETEN
    { "retension",	MTRETEN,	1,	"retension tape (if supported)"	},
#endif
    { "rewind",		MTREW,		1,	"rewind tape"			},
    { "rewoffl",	MTOFFL,		1,	"rewind and eject tape"		},
    { "offline",	MTOFFL,		1,	NULL				},
    { "eject",		MTOFFL,		1,	NULL				},
    { "nop",		MTNOP,		1,	"no operation"			},
#ifdef	MTSEEK
    { "seek",		MTSEEK,		1,	"seek to position"		},
#endif
    { "eom",		MTEOM,		1,	"seek to End Of Media"		},
    { "eod",		MTEOM,		1,	NULL				},
    { "seod",		MTEOM,		1,	NULL				},
    { "setblk",		MTSETBLK,	1,	"set blocksize"			},
#ifdef	MTSETDENSITY
    { "setdensity",	MTSETDENSITY,	1,	"set writing density"		},
#endif
#ifdef	MTSETDRVBUFFER
    { "setdrvbuffer",	MTSETDRVBUFFER, 1,	"set drive buffering"		},
    { "drvbuffer",	MTSETDRVBUFFER, 1,	NULL				},
#endif
    /* writing options	*/
    { "weof",		MTWEOF,		0,	"write End Of File mark"	},
    { "eof",		MTWEOF,		0,	NULL				},
    { "wsm",		MTWSM,		0,	"write setmark"			},
    { "erase",		MTERASE,	0,	"erase tape (if supported)"	},
    /* divers	*/
#ifdef	MTRAS1
    { "ras1",		MTRAS1,		0,	"run self test 1 (nondest.)"	},
#endif
#ifdef	MTRAS2
    { "ras2",		MTRAS2,		0,	"run self test 2 (destructive)"	},
#endif
#ifdef	MTRAS3
    { "ras3",		MTRAS3,		0,	"run self test 3 (?)"		},
#endif
    { 0 }
  };

static int		mtfd;
static struct mtop	mt_com;
static struct mtget	mt_status;
static struct mtpos	mt_pos;
static char		*tape;

static void		status(struct mtget *bp);
static void		printreg(char *s, unsigned short v, char *bits);

int
main(int argc, char **argv)
{
  char			line[80], *getenv();
  struct commands	*comp;

  if (argc > 2 && (!strcmp(argv[1], "-t") || !strcmp(argv[1], "-f")))
    {
      argc -= 2;
      tape = argv[2];
      argv += 2;
    }
  else if ((tape = getenv("TAPE")) == NULL)
    tape = DEFTAPE;
  if (argc < 2)
    {
      fprintf(stderr, "usage: mt [ -f device ] command|number [ count ]");
      for (comp=com; comp->c_name; comp++)
	{
	  if (comp->help)
	    fprintf(stderr, "\n\t%2d %-29s", comp->c_code, comp->help);
	  fprintf(stderr, "%12s,", comp->c_name);
	}
      fprintf(stderr, "\n");
      return 1;
    }
  mt_com.mt_op = atoi(argv[1]);
  for (comp=com;; comp++)
    if (!comp->c_name)
      {
	if (mt_com.mt_op)
	  break;
	fprintf(stderr, "mt: don't grok \"%s\"\n", argv[1]);
	return 1;
      }
    else if (strncmp(argv[1], comp->c_name, strlen(argv[1])) == 0 ||
	     (mt_com.mt_op && mt_com.mt_op==comp->c_code))
      {
	mt_com.mt_op = comp->c_code;
	break;
      }
  if ((mtfd = open(tape, comp->c_ronly ? O_RDONLY : O_RDWR)) < 0)
    {
      perror(tape);
      return 1;
    }
  if (comp->c_ronly >= 0)
    {
      mt_com.mt_count = (argc > 2 ? atoi(argv[2]) : 1);
      if (mt_com.mt_count < 0)
	{
	  fprintf(stderr, "mt: negative repeat count\n");
	  return 1;
	}
      if (ioctl(mtfd, MTIOCTOP, (char *)&mt_com) < 0)
	{
	  fprintf(stderr, "%s %s %d ", tape, comp->c_name,
		  mt_com.mt_count);
	  perror("failed");
	  return 2;
	}
    }
  else
    switch (comp->c_code)
      {
      case MTTELL:
	if (ioctl(mtfd, MTIOCPOS, (char *)&mt_pos) < 0)
	  {
	    perror("mt");
	    return 2;
	  }
	printf("At block %d.\n", mt_pos.mt_blkno);
	break;

      default:
	if (ioctl(mtfd, MTIOCGET, (char *)&mt_status) < 0)
	  {
	    perror("mt");
	    return 2;
	  }
	status(&mt_status);
      }
  return 0;
}

#ifdef vax
#include <vaxmba/mtreg.h>
#include <vaxmba/htreg.h>

#include <vaxuba/utreg.h>
#include <vaxuba/tmreg.h>
#undef b_repcnt		/* argh */
#include <vaxuba/tsreg.h>
#endif

#ifdef sun
#include <sundev/tmreg.h>
#include <sundev/arreg.h>
#endif

#ifdef tahoe
#include <tahoe/vba/cyreg.h>
#endif

struct tape_desc
  {
    short	t_type;		/* type of magtape device */
    char	*t_name;	/* printing name */
    char	*t_dsbits;	/* "drive status" register */
    char	*t_erbits;	/* "error" register */
  } tapes[] =
  {
#ifdef vax
    { MT_ISTS,	"ts11",		0,		TSXS0_BITS },
    { MT_ISHT,	"tm03",		HTDS_BITS,	HTER_BITS },
    { MT_ISTM,	"tm11",		0,		TMER_BITS },
    { MT_ISMT,	"tu78",		MTDS_BITS,	0 },
    { MT_ISUT,	"tu45",		UTDS_BITS,	UTER_BITS },
#endif
#ifdef sun
    { MT_ISCPC,	"TapeMaster",	TMS_BITS,	0 },
    { MT_ISAR,	"Archive",	ARCH_CTRL_BITS,	ARCH_BITS },
#endif
#ifdef tahoe
    { MT_ISCY,	"cipher",	CYS_BITS,	CYCW_BITS },
#endif
    { 0,	NULL }
  };

#ifdef	MT_TAPE_INFO
static struct mt_tape_info tape_info[]=MT_TAPE_INFO;

static void
stripper(char *buf, char *what)
{
  char	*tmp;

  if ((tmp=strstr(buf, what))!=0)
    strcpy(tmp, tmp+strlen(what));
}
#endif

/*
 * Interpret the status buffer returned
 */
static void
status(struct mtget *bp)
{
  struct tape_desc	*mt;
  char			buf[80], *name;

  for (mt = tapes; mt->t_type; mt++)
    if (mt->t_type == bp->mt_type)
      break;
  if ((name=mt->t_name)==0)
    {
      sprintf(buf, "unknown %ld", bp->mt_type);
#ifdef	MT_TAPE_INFO
      {
	struct mt_tape_info *ptr;

	for (ptr=tape_info; ptr->t_name; ptr++)
	  if (ptr->t_type==bp->mt_type)
	    {
	      strcpy(buf, ptr->t_name);
	      break;
	    }
	stripper(buf, " drive");
	stripper(buf, " device");
	stripper(buf, " tape");
	stripper(buf, " of");
      }
#endif
      name	= buf;
    }
  printf("%s tape drive, residual=%d, file=%ld, block=%ld, stat=%ld, blks=%ld, dens=%ld\n",
	 name, bp->mt_resid,
	 (long)bp->mt_fileno, (long)bp->mt_blkno, bp->mt_gstat,
	 ((unsigned long)bp->mt_dsreg&MT_ST_BLKSIZE_MASK)>>MT_ST_BLKSIZE_SHIFT,
	 ((unsigned long)bp->mt_dsreg&MT_ST_DENSITY_MASK)>>MT_ST_DENSITY_SHIFT);
  printreg("ds", bp->mt_dsreg, mt->t_dsbits);
  printreg("\ner", bp->mt_erreg, mt->t_erbits);
  putchar('\n');
}

/*
 * Print a register a la the %b format of the kernel's printf
 */
static void
printreg(char *s, unsigned short v, char *bits)
{
  int	i, any = 0;
  char	c;

  if (bits && *bits == 8)
    printf("%s=%o", s, v);
  else
    printf("%s=%x", s, v);
  if (!bits)
    return;
  if (v)
    {
      putchar('<');
      while (i = *bits++)
	{
	  if (v & (1 << (i-1)))
	    {
	      if (any)
		putchar(',');
	      any = 1;
	      for (; (c = *bits) > 32; bits++)
		putchar(c);
	    }
	  else
	    for (; *bits > 32; bits++)
	      ;
	}
      putchar('>');
    }
}
