/*
 * mscan - Mustek Flatbet Scanner Driver.
 * Copyright (C) 1996 David Mosberger-Tang.
 * This file is part of the mscan package.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * mscan is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mscan; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <alloca.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include <scsi/sg.h>

#include <sys/ioctl.h>
#include <sys/param.h>

#include "mscan.h"
#include "scsi.h"


int
scsi_open (const char * dev)
{
    int fd, timeout;

    TRACE(1, "scsi_open(%s)\n", dev);

    fd = open(dev, O_RDWR | O_EXCL);
    if (fd < 0) {
	perror(dev);
	return -1;
    }

    /*
     * Set large timeout since some scanners are slow but do not
     * disconnect... ;-(
     */
    timeout = 10*60*HZ;		/* how about 10 minutes? ;-) */
    if (ioctl(fd, SG_SET_TIMEOUT, &timeout) != 0) {
	perror("ioctl(SG_SET_TIMEOUT)");
	close(fd);
	return -1;
    }

    return fd;
}


ssize_t
scsi_cmd (int fd, const void * src, size_t src_size,
	  void * dst, size_t dst_size)
{
    /*
     * The Linux generic scsi interface is braindead as it forces us
     * to copy things around like mad.  Fortunately, most command
     * requests are small.
     */
    struct {
	struct sg_header	hdr;
	u_int8_t		data[SG_BIG_BUFF];
    } * req;
    ssize_t nread;
    int i;

    req = alloca(sizeof(*req));	/* works around gdb bug... ;-( */

    TRACE(2, "scsi_cmd 0x%02x (write len=%ld, read len=%ld)\n",
	  ((u_int8_t*) src)[0], (long) src_size, (long) dst_size);

    memset(&req->hdr, 0, sizeof(req->hdr));
    req->hdr.pack_len  = src_size + sizeof(req->hdr);
    req->hdr.reply_len = dst_size + sizeof(req->hdr);
    memcpy(&req->data, src, src_size);

    if (write(fd, req, req->hdr.pack_len) != req->hdr.pack_len) {
	perror("scsi_cmd: write");
	return -1;
    }

    nread = read(fd, req, dst_size + sizeof(req->hdr));
    if (nread < 0) {
	perror("scsi_cmd: read");
	return -1;
    }
    nread -= sizeof(req->hdr);

    if (req->hdr.result != 0) {
	fprintf(stderr, "scsi_cmd: error (%s):\n  sense=[",
		strerror(req->hdr.result));
	for (i = 0; i < sizeof(req->hdr.sense_buffer); ++i) {
	    fprintf(stderr, "%s%02x", (i > 0) ? "," : "",
		    req->hdr.sense_buffer[i]);
	}
	fprintf(stderr, "]\n");
	return -1;
    }

    if (dst) {
	memcpy(dst, req->data, nread);
    }
    return nread;
}


void
scsi_close (int fd)
{
    close(fd);
}
