#include <alloca.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "mscan.h"
#include "msci.h"
#include "scsi.h"

#define COLOR_CODE_GRAY		0
#define COLOR_CODE_RED		1
#define COLOR_CODE_GREEN	2
#define COLOR_CODE_BLUE		3

/*
 * SCSI commands that the Mustek scanners understand:
 */
#define MUSTEK_SCSI_TEST_UNIT_READY	0x00
#define MUSTEK_SCSI_REQUEST_SENSE	0x03
#define MUSTEK_SCSI_AREA_AND_WINDOWS	0x04
#define MUSTEK_SCSI_READ_SCANNED_DATA	0x08
#define MUSTEK_SCSI_GET_IMAGE_STATUS	0x0f
#define MUSTEK_SCSI_ADF_AND_BACKTRACK	0x10
#define MUSTEK_SCSI_CCD_DISTANCE	0x11
#define MUSTEK_SCSI_INQUIRY		0x12
#define MUSTEK_SCSI_MODE_SELECT		0x15
#define MUSTEK_SCSI_MODE_SENSE		0x1a	/* XXX what's this??? */
#define MUSTEK_SCSI_START_STOP		0x1b
#define MUSTEK_SCSI_RECV_DIAGNOSTICS	0x1c	/* XXX what's this??? */
#define MUSTEK_SCSI_SEND_DIAGNOSTICS	0x1d	/* XXX what's this??? */
#define MUSTEK_SCSI_READ_SCANNED_DATA_HANDY 0x28 /* XXX what's this??? */
#define MUSTEK_SCSI_LOOKUP_TABLE	0x55

#define INQ_LEN	0x60
static const u_int8_t inquiry[] = {
    MUSTEK_SCSI_INQUIRY, 0x00, 0x00, 0x00, INQ_LEN, 0x00
};

static const u_int8_t test_unit_ready[] = {
    MUSTEK_SCSI_TEST_UNIT_READY, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const u_int8_t sense[] = {
    MUSTEK_SCSI_REQUEST_SENSE, 0x00, 0x00, 0x00, 0x04, 0x00
};

static const u_int8_t stop[] = {
    MUSTEK_SCSI_START_STOP, 0x00, 0x00, 0x00, 0x00, 0x00
};

static const u_int8_t distance[] = {
    MUSTEK_SCSI_CCD_DISTANCE, 0x00, 0x00, 0x00, 0x05, 0x00
};

static const u_int8_t status[] = {
    MUSTEK_SCSI_GET_IMAGE_STATUS, 0x00, 0x00, 0x00, 0x06, 0x00
};

#if 0

/*
 * Don't know anything about these commands:
 */
static const u_int8_t mode_sense[] = {
    MUSTEK_SCSI_MODE_SENSE, 0x00, 0x00, 0x00, 0x0a, 0x00
}

static const u_int8_t send_diagnostics[] = {
    MUSTEK_SCSI_SEND_DIAGNOSTICS, 0x04, 0x00, 0x00, 0x00, 0x00
}

static const u_int8_t receive_diagnostics[] = {
    MUSTEK_SCSI_RECV_DIAGNOSTICS, 0x00, 0x00, 0x00, 0x01, 0x00
}

#endif

#define STORE16(p,v)				\
do {						\
    int value = (v);				\
						\
    *(cp)++ = value & 0xff;			\
    *(cp)++ = (value >> 8) & 0xff;		\
} while (0)


#if __GNU_LIBRARY__ - 0 < 6

char *
strndup (const char * s, size_t n)
{
    char * clone;

    clone = malloc(n + 1);
    strncpy(clone, s, n);
    clone[n] = '\0';
    return clone;
}

#endif /* __GNU_LIBRARY__ */


long
msci_open (Scanner s, const char * dev_name)
{
    int fd, mustek_scanner;
    u_int8_t result[INQ_LEN];
    const char * model_name = result + 44;

    fd = scsi_open(dev_name);
    if (fd < 0) {
	fprintf(stderr, "%s: open failed\n", dev_name);
	return -1;
    }

    if (scsi_cmd(fd, inquiry, sizeof(inquiry), result, sizeof(result))
	!= INQ_LEN)
    {
	fprintf(stderr, "%s: scanner does not respond\n", dev_name);
	goto scsi_close_and_fail;
    }

    if (scsi_cmd(fd, test_unit_ready, sizeof(test_unit_ready), 0, 0) != 0) {
	fprintf(stderr, "%s: scanner not ready\n", dev_name);
	goto scsi_close_and_fail;
    }

    /* first check for new firmware format: */
    mustek_scanner = (strncmp(result + 8, "SCANNER", 7) == 0
		      && strncmp(result + 36, "MUSTEK", 6) == 0);

    if (!mustek_scanner) {
	/* now check for old format: */
	mustek_scanner |= (strncmp(result + 8, "MUSTEK", 6) == 0);
	model_name = result + 16;
    }

    if (!mustek_scanner) {
	fprintf(stderr, "%s: not a Mustek scanner\n", dev_name);
	goto scsi_close_and_fail;
    }

    s->hw.name = strndup(model_name, 11);

    if (verbose) {
	fprintf(stderr, "%s: scanner model %s (firmware version %.4s)\n",
		prog_name, s->hw.name, result + 32);
    }

    if (strncmp(model_name, "MFS-12000CX", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    = 1200;
    } else if (strncmp(model_name, "MFS-06000CX", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  600;
    } else if (strncmp(model_name, "MFS-12000SP", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    = 1200;
    } else if (strncmp(model_name, "MFS-08000SP", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  800;
    } else if (strncmp(model_name, "MFS-06000SP", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  600;
    } else if (strncmp(model_name, "MFS-08000CZ", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  800;
    } else if (strncmp(model_name, "MFS-06000CZ", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  600;
    } else if (strncmp(model_name, "MFC-08000CZ", 11) == 0) {
	s->hw.max_size.x =  8.5 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_size.y = 14.0 * MM_PER_INCH;	/* is this correct? */
	s->hw.max_dpi    =  800;
    } else if (strncmp(model_name, "MFC-06000CZ", 11) == 0) {
	s->hw.max_size.x = 220.0;			/* measured */
	s->hw.max_size.y = 300.0;			/* measured */
	s->hw.max_dpi    = 600;
    } else {
	fprintf(stderr, "%s: scanner type %.11s unknown\n",
		dev_name, result + 57);
	goto scsi_close_and_fail;
    }

    if (result[57] & (1 << 6)) {
	TRACE(1, "msci_open:  single pass\n");
	s->hw.flags |= MSCI_FLAG_SINGLE_PASS;
    }
    if (result[63] & (1 << 2)) {
	TRACE(1, "msci_open:  automatic document feeder (ADF)\n");
	s->hw.flags |= MSCI_FLAG_ADF;
    }
    if (result[63] & (1 << 3)) {
	TRACE(1, "msci_open:  ADF is ready (has paper)\n");
	s->hw.flags |= MSCI_FLAG_ADF_READY;
    }
    if (result[63] & (1 << 6)) {
	TRACE(1, "msci_open:  transparency adapter\n");
	s->hw.flags |= MSCI_FLAG_TA;
    }
    s->hw.fd = fd;
    return 0;

  scsi_close_and_fail:
    scsi_close(fd);
    return -1;
}


/*
 * Quantize s->req.resolution and return the resolution code for
 * the quantized resolution.  Quantization depends on scanner
 * type (single pass vs. three-pass).
 */
static long
encode_resolution (Scanner s)
{
    float max_dpi, dpi;
    int code;

    dpi = s->req.resolution;

    if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	if (dpi < 1) {
	    dpi = 1;
	}
	code = dpi;
    } else {
	int quant, half_res;

	max_dpi = s->hw.max_dpi;
	half_res = max_dpi / 2;

	if (dpi <= half_res) {
	    quant = half_res / 100;
	    code  = dpi / quant;
	    if (code < 1) {
		code = 1;
	    }
	    dpi = code * quant;
	} else {
	    /* quantize to 10% step: */
	    quant = (half_res / 10);
	    code = dpi / quant;
	    dpi  = code * quant;

	    /* and now encode: */
	    code = ((code - 5) * 30) % 100;
	    if (code == 0) {
		code = 0x64;
	    }
	    code |= 0x100;	/* indicate 10% quantization */
	}
    }
    s->req.resolution = dpi;
    return code;
}


static long
encode_percentage (Scanner s, float value, float quant)
{
    int max, code, sign = 0;

    if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	if (value < 0) {
	    value = -value;
	    sign = 0x80;
	}
	code = (int) (value / 100.0 * 127 + 0.5);
	code |= sign;
	max = 0xff;
    } else {
	code = (int) ((value / 100.0 / quant) + 0.5) + 0x0c;
	max = 0x18;
    }
    if (code > max) {
	code = max;
    }
    if (code < 0) {
	code = 0x00;
    }
    return code;
}


static long
request_sense (Scanner s)
{
    u_int8_t result[4];
    size_t len;

    len = scsi_cmd(s->hw.fd, sense, sizeof(sense), result, sizeof(result));
    if (len != sizeof(result)) {
	fprintf(stderr, "request_sense: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    switch (result[0]) {
      case 0x00:
	  break;

      case 0x82:
	  if ((s->hw.flags & MSCI_FLAG_ADF) && result[1] & 0x80) {
	      fprintf(stderr, "%s: paper jam in automatic document feeder\n",
		      prog_name);
	      return -1;
	  }
	  break;

      case 0x83:
	  if ((s->hw.flags & MSCI_FLAG_ADF) && result[2] & 0x02) {
	      fprintf(stderr, "%s: automatic document feeder out of paper\n",
		      prog_name);
	      return -1;
	  }
	  break;

      case 0x84:
	  if ((s->hw.flags & MSCI_FLAG_TA) && result[1] & 0x10) {
	      /* ignore transparency adapter cover open message */
	      return 0;
	  }
	  break;

      default:
	  fprintf(stderr, "%s: unknown error code %x\n", prog_name, result[0]);
	  return -1;
	  break;
    }
    return 0;
}


static long
scan_area_and_windows (Scanner s)
{
    float pixels_per_mm = s->hw.max_dpi / MM_PER_INCH;
    u_int8_t cmd[117], *cp;
    int i, j, dim, len;

    /* setup SCSI command (except length): */
    memset(cmd, 0, sizeof(cmd));
    cmd[0] = MUSTEK_SCSI_AREA_AND_WINDOWS;

    cp = cmd + 6;

    /* fill in frame header: */

    /* pixel unit and halftoning: */
    *cp++ = 0x8 | ((s->req.mode & MSCI_HALFTONE) ? 0x01 : 0x00);

    /* fill in scanning area: */
    if (s->req.area.top_left.x < 0.0
	|| s->req.area.top_left.x > s->req.area.bottom_right.x
	|| s->req.area.bottom_right.x > s->hw.max_size.x
	|| s->req.area.top_left.y < 0.0
	|| s->req.area.top_left.y > s->req.area.bottom_right.y
	|| s->req.area.bottom_right.y > s->hw.max_size.y)
    {
	fprintf(stderr, "%s: area exceeds scanner size\n", prog_name);
	return -1;
    }
    STORE16(cp, s->req.area.top_left.x	   * pixels_per_mm);
    STORE16(cp, s->req.area.top_left.y	   * pixels_per_mm);
    STORE16(cp, s->req.area.bottom_right.x * pixels_per_mm);
    STORE16(cp, s->req.area.bottom_right.y * pixels_per_mm);

    for (i = 0; i < s->req.num_windows; ++i) {
	*cp++ = 0x88;	/* window header w/pixel unit */
	STORE16(cp, s->req.window[i].top_left.x     * pixels_per_mm);
	STORE16(cp, s->req.window[i].top_left.y     * pixels_per_mm);
	STORE16(cp, s->req.window[i].bottom_right.x * pixels_per_mm);
	STORE16(cp, s->req.window[i].bottom_right.y * pixels_per_mm);
    }

    dim = s->req.halftone_pattern_dimension;
    if (dim) {
	*cp++ = 0x40;			/* mark presence of user pattern */
	*cp++ = (dim << 4) | dim;	/* set pattern length */
	for (i = 0; i < dim; ++i) {
	    for (j = 0; j < dim; ++j) {
		*cp++ = s->req.halftone_pattern[i][j];
	    }
	}
    }

    cmd[4] = (cp - cmd) - 6;
    len = scsi_cmd(s->hw.fd, cmd, (cp - cmd), 0, 0);
    if (len != 0) {
	fprintf(stderr, "scan_area_and_windows: scsi command failed (%d)\n",
		len);
	return -1;
    }
    return 0;
}


static long
mode_select (Scanner s, long color_code, long index)
{
    float pixels_per_mm = s->hw.max_dpi / MM_PER_INCH;
    int paperlength_code, grain_code, speed_code;
    u_int8_t mode[19], * cp;
    size_t len;

    /* the scanners use a funky code for the grain size, let's compute it: */
    grain_code = s->req.grain;
    if (grain_code < 2) {
	grain_code = 2;
    } else if (grain_code > 7) {
	grain_code = 7;
    }
    grain_code = 7 - grain_code;	/* code 0 is 8x8, not 7x7 */

    /* same goes for speed: */
    speed_code = (int) (4.0 * s->req.speed / 100.0 + 0.5);
    if (speed_code > 4) {
	speed_code = 4;
    } else if (speed_code < 0) {
	speed_code = 0;
    }
    speed_code = 4 - speed_code;	/* 0 is fast, 4 is slow */

    memset(mode, 0, sizeof(mode));
    mode[0] = MUSTEK_SCSI_MODE_SELECT;

    /* set command length and resolution code: */
    if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	mode[4] = 0x0d;
	cp = mode + 17;
	STORE16(cp, s->tmp.resolution_code);
    } else {
	mode[4] = 0x0b;
	mode[7] = s->tmp.resolution_code;
    }
    /*
     * Use `pixel' unit, select whether user-defined half-tone pattern
     * should be used and tell scanner what color this mode select
     * command refers to (used for 1-pass scanners only; 3-pass scanners
     * always use COLOR_CODE_GRAY).
     */
    mode[ 6] = (0x8b | (s->req.halftone_pattern_dimension ? 0x10 : 0x00)
		| (color_code << 5));
    mode[ 8] = encode_percentage(s, s->req.brightness[index], 3.0);
    mode[ 9] = encode_percentage(s, s->req.contrast[index], 7.0);
    mode[10] = grain_code;
    mode[11] = speed_code | ((s->req.lamp_setting & 0xf) << 4);
    mode[12] = s->req.shadow[index];	/* XXX what's this?? */
    mode[13] = s->req.highlight[index];	/* XXX what's this?? */

    paperlength_code = (s->req.paper_length * pixels_per_mm + 0.5);
    cp = mode + 14;
    STORE16(cp, paperlength_code);

    mode[16] = 0;	/* XXX midtone setting---is this really used? */

    len = scsi_cmd(s->hw.fd, mode, 6 + mode[4], 0, 0);
    if (len != 0) {
	fprintf(stderr, "mode_select: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    return 0;
}


static long
gamma_correction (Scanner s, long color_code, long index)
{
    u_int8_t gamma[266];
    size_t len;

    memset(gamma, 0, sizeof(gamma));
    gamma[0] = MUSTEK_SCSI_LOOKUP_TABLE;
    gamma[9] = (color_code << 6);

    len = s->req.gamma_size[index];
    if (len) {
	gamma[2] = 0x27;	/* indicate user-selected gamma table */
	gamma[7] = (len >> 8) & 0xff;	/* big endian! */
	gamma[8] = (len >> 0) & 0xff;
	memcpy(gamma + 10, &s->req.gamma_corr[index][0], len);
	len += 2;
    } else {
	/* XXX what do these gamma tables look like? */
	gamma[2] = (int) (s->req.gamma[index] + 0.5);
    }
    len += 10;

    len = scsi_cmd(s->hw.fd, gamma, len, 0, 0);
    if (len != 0) {
	fprintf(stderr, "gamma_correction: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    return 0;
}


static long
start_scan (Scanner s, long pass)
{
    u_int8_t start[6];
    size_t len;

    memset(start, 0, sizeof(start));
    start[0] = MUSTEK_SCSI_START_STOP;
    start[4] = 0x01;
    if (s->req.mode & MSCI_COLOR) {
	if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	    start[4] |= 0x20;
	} else {
	    start[4] |= ((pass << 3) + 1);
	}
    }
    /* or in single/multi bit: */
    start[4] |= (s->req.mode & MSCI_MULTIBIT) ? (1 << 6) : 0;

    if (!s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	/* or in expanded resolution bit: */
	start[4] |= (s->tmp.resolution_code & 0x100) >> 1;
    }

    len = scsi_cmd(s->hw.fd, start, sizeof(start), 0, 0);
    if (len != 0) {
	fprintf(stderr, "start_scan: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    return 0;
}


static long
stop_scan (Scanner s)
{
    size_t len;

    len = scsi_cmd(s->hw.fd, stop, sizeof(stop), 0, 0);
    if (len != 0) {
	fprintf(stderr, "stop_scan: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    return 0;
}


/*
 * Determine the CCD's distance between the primary color lines.
 */
static long
line_distance (Scanner s)
{
    u_int8_t result[5];
    size_t len;
    int factor, peak_res, color;

    len = scsi_cmd(s->hw.fd, distance, sizeof(distance),
		   result, sizeof(result));
    if (len != sizeof(result)) {
	fprintf(stderr, "line_distance: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }

#ifdef LINE_DISTANCE_WORKAROUND
    /*
     * These are the correct values for the Paragon 600 II SP.  YMMV.
     */
    if (s->req.mode & MSCI_COLOR) {
	if (s->req.resolution < 72) {
	    /* 1-71 dpi: */
	    result[0] = 4;
	    result[1] = 0;
	    result[2] = 0;
	    result[3] = 3;
	    result[4] = 5;
	} else if (s->req.resolution <= 300) {
	    /* 72-300 dpi: */
	    result[0] = 2;
	    result[1] = 0;
	    result[2] = 0;
	    result[3] = 5;
	    result[4] = 9;
	} else {
	    /*301-600 dpi: */
	    result[0] = 1;
	    result[1] = 0;
	    result[2] = 0;
	    result[3] = 9;
	    result[4] = 17;
	}
    } else {
	result[0] = 0xff;	/* greyscale scanning */
	result[1] = 0xff;
    }
#endif

    factor = result[0] | (result[1] << 8);
    if (factor != 0xffff) {
	/* need to do line-distance adjustment ourselves... */

	s->ld.max_value = s->hw.max_dpi;

	peak_res = s->req.resolution;
	if (factor == 0) {
	    if (peak_res <= s->hw.max_dpi / 2) {
		peak_res *= 2;
	    }
	} else {
	    peak_res *= factor;
	}
	s->ld.peak_res = peak_res;
	for (color = 0; color < 3; ++color) {
	    s->ld.c[color].dist = result[2 + color];
	    s->ld.c[color].Qk   = s->ld.max_value;
	}

	TRACE(1, "line_distance: factor=%u, r=%u, g=%u, b=%u\n",
	      factor, s->ld.c[0].dist, s->ld.c[1].dist, s->ld.c[2].dist);
    }
    return 0;
}


static long
get_image_status (Scanner s, int * bytes_per_line, int * lines)
{
    u_int8_t result[6];
    size_t len;
    int busy;

    do {
	len = scsi_cmd(s->hw.fd, status, sizeof(status),
		       result, sizeof(result));
	if (len != sizeof(result)) {
	    fprintf(stderr, "get_image_status: scsi command failed (%ld)\n",
		    (long) len);
	    return -1;
	}
	busy = result[0];
	if (busy) {
	    fprintf(stderr, "get_image_status: napping a little...\n");
	    usleep(1000000);
	}
    } while (busy);

    *bytes_per_line = result[1] | (result[2] << 8);
    *lines = result[3] | (result[4] << 8) | (result[5] << 16);

    TRACE(1, "get_image_status: %d bytes/line, %d lines\n",
	  *bytes_per_line, *lines);

    return 0;
}


static long
backtrack_and_adf (Scanner s)
{
    u_int8_t backtrack[6];
    int code = 0x80;
    size_t len;

    if (s->req.backtrack) {
	code |= 0x02;
    }
    switch (s->req.source) {
      case MSCI_SOURCE_FLATBED:	code |= 0x00; break;
      case MSCI_SOURCE_ADF:	code |= 0x01; break;
      case MSCI_SOURCE_TA:	code |= 0x04; break;
      default:
	  fprintf(stderr, "backtrack_and_adf: invalid source %d\n",
		  s->req.source);
	  return -1;
    }
    memset(backtrack, 0, sizeof(backtrack));
    backtrack[0] = MUSTEK_SCSI_ADF_AND_BACKTRACK;
    backtrack[4] = code;

    len = scsi_cmd(s->hw.fd, backtrack, sizeof(backtrack), 0, 0);
    if (len != 0) {
	fprintf(stderr, "backtrack_and_adf: scsi command failed (%ld)\n",
		(long) len);
	return -1;
    }
    return 0;
}


static void
fix_line_distance (Scanner s, int num_lines, int bpl,
		   u_int8_t * raw, u_int8_t * out)
{
    u_int8_t * raw_end = raw + num_lines * bpl;
    int bpc = bpl / 3;	/* bytes per color (per line) */
    int index[3];	/* index of the next output line for color C */
    int i, c;
    static const int color_seq[] = {1, 2, 0};	/* green, blue, red */

    /*
     * Initialize the indices with the line distances that were returned
     * by the CCD linedistance command.  We want to skip data for the
     * first OFFSET rounds, so we initialize the indices to the negative
     * of this offset.
     */
    for (c = 0; c < 3; ++c) {
	index[c] = -s->ld.c[c].dist;
    }

    while (1) {
	for (i = 0; i < 3; ++i) {
	    c = color_seq[i];
	    if (index[c] < 0) {
		++index[c];
	    } else if (index[c] < num_lines) {
		s->ld.c[c].Qk += s->ld.peak_res;
		if (s->ld.c[c].Qk > s->ld.max_value) {
		    s->ld.c[c].Qk -= s->ld.max_value;
		    memcpy(out + c * bpc + index[c] * bpl, raw, bpc);
		    ++index[c];
		    raw += bpc;
		    if (raw >= raw_end) {
			return;
		    }
		}
	    }
	}
    }
}


static long
read_data (Scanner s, int height, int bpl, u_int8_t * image, long pass)
{
    /* expand a 1 bit pixel into an 8 bit pixel: */
#   define EXPAND(b, v) (((v) & (1 << (bit))) ? 0xff : 0x00)
    int bit, x, y, i, l, lines_per_buffer;
    u_int8_t readlines[6];
    u_int8_t * buf, * src, * dst;
    size_t len;

    lines_per_buffer = SCSI_MAX_REQUEST_SIZE / bpl;
    if (lines_per_buffer <= 0) {
	fprintf(stderr,
		"%s: scan line size (%d) exceeds maximum scsi buffer\n"
		"%s: request size (%d); please reduce with or resolution\n",
		prog_name, bpl, prog_name, SG_BIG_BUFF);
	return -1;
    }
    memset(readlines, 0, sizeof(readlines));
    readlines[0] = MUSTEK_SCSI_READ_SCANNED_DATA;
    readlines[2] = (lines_per_buffer >> 16) & 0xff;
    readlines[3] = (lines_per_buffer >>  8) & 0xff;
    readlines[4] = (lines_per_buffer >>  0) & 0xff;

    buf = alloca(2 * lines_per_buffer * bpl);

    for (l = 0; l < height; l += lines_per_buffer) {
	if (l + lines_per_buffer > height) {
	    /* do the last few lines: */
	    lines_per_buffer = height - l;
	    readlines[2] = (lines_per_buffer >> 16) & 0xff;
	    readlines[3] = (lines_per_buffer >>  8) & 0xff;
	    readlines[4] = (lines_per_buffer >>  0) & 0xff;
	}
	len = scsi_cmd(s->hw.fd, readlines, sizeof(readlines),
		       buf, lines_per_buffer * bpl);
	if (len != lines_per_buffer * bpl) {
	    fprintf(stderr, "read_data: scsi command failed (%ld)\n",
		    (long) len);
	    return -1;
	}

	/* convert to pixel-interleaved format: */
	if (s->req.mode & MSCI_COLOR) {
	    src = buf;
	    if (s->ld.max_value) {
		/* need to correct for distance between r/g/b sensors: */
		src = buf + lines_per_buffer * bpl;
		fix_line_distance(s, lines_per_buffer, bpl, buf, src);
	    }

	    if (s->req.mode & MSCI_MULTIBIT) {
		/* each r/g/b sample is 8 bits */
		if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
		    /* the scanner data is line-interleaved */
		    dst = image + l * bpl;
		    for (y = 0; y < lines_per_buffer; ++y) {
			for (x = 0; x < bpl/3; ++x) {
			    *dst++ = src[0 * bpl/3 + x];
			    *dst++ = src[1 * bpl/3 + x];
			    *dst++ = src[2 * bpl/3 + x];
			}
			src += bpl;
		    }
		} else {
		    /*the scanner data is band-interleaved */
		    dst = image + l * bpl * 3 + pass;
		    for (i = 0; i < bpl * lines_per_buffer; ++i, dst += 3) {
			*dst = *src++;
		    }
		}
	    } else {
		/* each r/g/b sample is one bit */
		if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
		    /* the scanner data is line-interleaved */
		    dst = image + l * bpl;
		    for (y = 0; y < lines_per_buffer; ++y) {
			for (x = 0; x < bpl/3; ++x) {
			    for (bit = 7; bit >= 0; --bit) {
				*dst++ = EXPAND(bit, src[0 * bpl/3 + x]);
				*dst++ = EXPAND(bit, src[1 * bpl/3 + x]);
				*dst++ = EXPAND(bit, src[2 * bpl/3 + x]);
			    }
			}
			src += bpl;
		    }
		} else {
		    /*the scanner data is band-interleaved */
		    dst = image + l * bpl * 3 + pass;
		    for (i = 0; i < bpl * lines_per_buffer; ++i, ++src) {
			for (bit = 7; bit >= 0; --bit, dst += 3) {
			    *dst = EXPAND(bit, *src);
			}
		    }
		}
	    }
	} else {
	    if (s->req.mode & MSCI_MULTIBIT) {
		memcpy(image + l * bpl, buf, lines_per_buffer * bpl);
	    } else {
		/* in singlebit mode, the scanner returns 1 for black. ;-( */
		for (i = 0; i < lines_per_buffer * bpl; ++i) {
		    image[l * bpl + i] = ~buf[i];
		}
	    }
	}
    }
    return 0;
}


void *
msci_scan_image (Scanner s, size_t * widthp, size_t * heightp)
{
    int pass, num_passes = 1, bpl, bpp, height = 0, width = 0, image_bpp;
    int scan_color_in_one_pass, saved_bpl = 0, saved_height = 0;
    void * image = 0;

    scan_color_in_one_pass = ((s->req.mode & MSCI_COLOR)
			      && (s->hw.flags & MSCI_FLAG_SINGLE_PASS));

    /* check and quantize resolution: */
    if (s->req.resolution < 1 || s->req.resolution > s->hw.max_dpi) {
	fprintf(stderr, "%s: scanner does not support resolution of %d dpi\n",
		prog_name, s->req.resolution);
	return 0;
    }
    s->tmp.resolution_code = encode_resolution(s);

    if (scan_area_and_windows(s) < 0) {
	goto stop_scanner_and_return;
    }
    if (request_sense(s) < 0) {
	goto stop_scanner_and_return;
    }

    if (backtrack_and_adf(s) < 0) {
	goto stop_scanner_and_return;
    }
    if (request_sense(s) < 0) {
	goto stop_scanner_and_return;
    }

    if ((s->req.mode & MSCI_COLOR) && !scan_color_in_one_pass) {
	num_passes = 3;
    }

    /* # of bits per pixel for input data and image: */
    bpp  = 1;
    if (s->req.mode & MSCI_MULTIBIT) {
	bpp = 8;
    }
    if (scan_color_in_one_pass) {
	bpp *= 3;
    }
    image_bpp = 1;
    if (s->req.mode & MSCI_COLOR) {
	image_bpp = 24;
    } else if (s->req.mode & MSCI_MULTIBIT) {
	image_bpp = 8;
    }

    for (pass = 0; pass < num_passes; ++pass) {
	if (scan_color_in_one_pass) {
	    mode_select(s, COLOR_CODE_RED,   0);
	    mode_select(s, COLOR_CODE_GREEN, 1);
	    mode_select(s, COLOR_CODE_BLUE,  2);
	} else {
	    mode_select(s, COLOR_CODE_GRAY, pass);
	}
	if (request_sense(s) < 0) {
	    goto stop_scanner_and_return;
	}

	start_scan(s, pass);
	if (request_sense(s) < 0) {
	    goto stop_scanner_and_return;
	}

	if (scan_color_in_one_pass) {
	    gamma_correction(s, COLOR_CODE_RED,   0);
	    gamma_correction(s, COLOR_CODE_GREEN, 1);
	    gamma_correction(s, COLOR_CODE_BLUE,  2);
	} else {
	    gamma_correction(s, COLOR_CODE_GRAY, pass);
	}
	if (request_sense(s) < 0) {
	    goto stop_scanner_and_return;
	}

	s->ld.max_value = 0;
	if (s->hw.flags & MSCI_FLAG_SINGLE_PASS) {
	    line_distance(s);
	    if (request_sense(s) < 0) {
		goto stop_scanner_and_return;
	    }
	}

	if (backtrack_and_adf(s) < 0) {
	    goto stop_scanner_and_return;
	}
	if (request_sense(s) < 0) {
	    goto stop_scanner_and_return;
	}

	get_image_status(s, &bpl, &height);
	if (bpl == 0 || height == 0) {
	    fprintf(stderr, "%s: ignoring zero sized image\n", prog_name);
	    goto stop_scanner_and_return;
	}

	if (!image) {
	    width = 8 * bpl / bpp;	/* image width in pixels */
	    image = malloc(width * height * image_bpp / 8);
	    if (!image) {
		fprintf(stderr, "msci_scan_image: out of memory\n");
		goto stop_scanner_and_return;
	    }
	    saved_bpl = bpl;
	    saved_height = height;
	} else if (bpl != saved_bpl || height != saved_height) {
	    fprintf(stderr, "msci_scan_image: bytes-per-line/height changed"
		    "from %d/%d to %d/%d between passes\n",
		    saved_bpl, saved_height, bpl, height);
	    goto stop_scanner_and_return;
	}

	if (read_data(s, height, bpl, image, pass) < 0) {
	    goto free_image;
	}
    }
    *widthp  = width;
    *heightp = height;
    return image;

  free_image:
    free(image);
  stop_scanner_and_return:
    stop_scan(s);
    return 0;
}


void
msci_close (Scanner s)
{
    scsi_close(s->hw.fd);
    s->hw.fd = -1;
}
