/*
 * mscan - Mustek Flatbet Scanner Driver.
 * Copyright (C) 1996 David Mosberger-Tang.
 * This file is part of the mscan package.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * mscan is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mscan; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "mscan.h"
#include "msci.h"

const struct option opts[] = {
    {"backtrack",	no_argument, NULL,	 'b'},
    {"brightness",	required_argument, NULL, 'B'},
    {"color",		no_argument, NULL,	 'c'},
    {"contrast",	required_argument, NULL, 'C'},
    {"debug",		optional_argument, NULL, 'd'},
    {"grain",		required_argument, NULL, 'g'},
    {"halftone",	no_argument, NULL,	 'm'},
    {"height",		required_argument, NULL, 'y'},
    {"help",		no_argument, NULL,	 'h'},
    {"left-edge",	required_argument, NULL, 'l'},
    {"one-bit",		no_argument, NULL,	 'a'},
    {"output-file",	required_argument, NULL, 'f'},
    {"parameter-set",	required_argument, NULL, 'p'},
    {"resolution",	required_argument, NULL, 'r'},
    {"speed",		required_argument, NULL, 's'},
    {"upper-edge",	required_argument, NULL, 'u'},
    {"verbose",		no_argument, NULL,	 'v'},
    {"version",		no_argument, NULL,	 'V'},
    {"width",		required_argument, NULL, 'x'},
    {0}
};

int trace_level;
int verbose;
const char * prog_name = "mscan";
const char * device = "/dev/scanner";
Scanner scanner;


static void
usage (void)
{
    fprintf(stderr,
"usage: %s [-abBcCdfghlmprsuvxy] [-backtrack] [-brightness r[/[g/b]]\n"
"       [--color] [--contrast r[/[g/b]] [--debug [l]] [--grain g]\n"
"       [--halftone] [--height h] [--help] [--left-edge x] [--one-bit]\n"
"       [--output-file=filename] [--parameter-set=b:c:g:s]\n"
"       [--resolution=dpi] [--speed s] [--upper-edge=y-pos] [--verbose]\n"
"       [--version] [--width w] [devicename]\n", prog_name);
}


/*
 * Scans a string as a position.  A position consists of a float
 * optionally followed by a unit (mm for millimeter, cm for
 * centimeter, or in or " or nothing for inches.
 */
float
scan_pos (const char * arg)
{
    char * endptr;
    float v;

    v = strtod(arg, &endptr);
    if (*endptr == '\0' || strcmp(endptr, "\"") == 0
	|| strcmp(endptr, "in") == 0)
    {
	v *= MM_PER_INCH;	/* inch -> mm */
    } else if (strcmp(endptr, "cm") == 0) {
	v *= 10.0;		/* cm -> mm */
    } else if (strcmp(endptr, "mm") != 0) {
	if (endptr == optarg) {
	    fprintf(stderr, "%s: not a float %s\n", prog_name, optarg);
	} else {
	    fprintf(stderr, "%s: illegal unit %s\n", prog_name, endptr);
	}
	exit(1);
    }
    return v;
}


void
scan_float_vector (const char * optarg, float v[3])
{
    char * endp;
    int i;

    for (i = 0; i < 3; ++i) {
	v[i] = strtod(optarg, &endp);
	if (endp == optarg) {
	    fprintf(stderr, "%s: bad float vector %s\n", prog_name, optarg);
	    exit(1);
	}
	if (!*endp) {
	    while (++i < 3) {
		v[i] = v[i - 1];
	    }
	    break;
	}
	if (*endp != '/') {
	    fprintf(stderr, "%s: vector elements should be separated by /\n",
		    prog_name);
	    exit(1);
	}
	optarg = endp + 1;
    }
}


int
main (int argc, char ** argv)
{
    FILE * ofp = stdout;
    size_t height, width;
    u_int8_t * image;
    struct Scanner s;
    int i, ch;

    prog_name = argv[0];

    memset(&s, 0, sizeof(s));
    s.req.resolution = 18;
    s.req.mode = MSCI_MULTIBIT;

    while ((ch = getopt_long(argc, argv, "B:bC:cd:g:my:hl:af:p:r:s:u:vw:x:?",
			     opts, NULL))
	   != EOF)
    {
	switch (ch) {
	  case 'a':
	      s.req.mode &= ~MSCI_MULTIBIT;
	      break;

	  case 'B':
	      scan_float_vector(optarg, s.req.brightness);
	      break;

	  case 'b':
	      s.req.backtrack ^= 1;
	      break;

	  case 'C':
	      scan_float_vector(optarg, s.req.contrast);
	      break;

	  case 'c':
	      s.req.mode |= MSCI_COLOR;
	      break;

	  case 'd':
	      if (optarg) {
		  trace_level = atoi(optarg);
	      } else {
		  ++trace_level;
	      }
	      break;

	  case 'g':
	      s.req.grain = atoi(optarg);	/* size of low-pass filter */
	      break;

	  case 'm':
	      s.req.mode &= ~MSCI_MULTIBIT;
	      s.req.mode |=  MSCI_HALFTONE;
	      break;

	  case 'y':
	      s.req.area.bottom_right.y = scan_pos(optarg);
	      break;

	  case '?':
	  case 'h':
	      usage();
	      exit(0);

	  case 'l':
	      s.req.area.top_left.x = scan_pos(optarg);
	      break;

	  case 'f':
	      if (ofp != stdout) {
		  fclose(ofp);
	      }
	      ofp = fopen(optarg, "w");
	      if (!ofp) {
		  perror(optarg);
		  exit(1);
	      }
	      break;

	  case 'p':
	  {
	      /* this is strictly for compatibility with muscan only */
	      int brightness, contrast, grain_code, speed_code;

	      if (sscanf(optarg, "%d:%d:%d:%d", &brightness, &contrast,
			 &grain_code, &speed_code) != 4)
	      {
		  fprintf(stderr, "%s: bad parameter set `%s'\n",
			  prog_name, optarg);
		  exit(1);
	      }
	      for (i = 0; i < 3; ++i) {
		  s.req.brightness[i] = 100.0 * (brightness - 12.0) / 12.0;
		  s.req.contrast[i]   = 100.0 * (contrast   - 12.0) / 12.0;
		  s.req.grain         = 7 - grain_code;
		  s.req.speed         = (4 - speed_code) * 100.0 / 4.0;
	      }
	  }

	  case 'r':
	      s.req.resolution = atoi(optarg);
	      break;

	  case 's':
	      s.req.speed = strtod(optarg, 0);	/* speed 0%=slow, 100%=fast */
	      break;

	  case 'u':
	      s.req.area.top_left.y = scan_pos(optarg);
	      break;

	  case 'v':
	      ++verbose;
	      break;

	  case 'V':
	      fprintf(stderr, "%s: This is mscan version 0.1.\n", prog_name);
	      break;

	  case 'x':
	      s.req.area.bottom_right.x = scan_pos(optarg);
	      break;

	  default:
	      usage();
	      exit(1);
	}
    }

    if (optind < argc) {
	device = argv[optind];
    }

    /* convert width/height into bottom-right corner position: */
    s.req.area.bottom_right.x += s.req.area.top_left.x;
    s.req.area.bottom_right.y += s.req.area.top_left.y;

    if (msci_open(&s, device) < 0) {
	exit(1);
    }

    if (s.req.area.bottom_right.x == 0.0) {
	s.req.area.bottom_right.x = s.hw.max_size.x;
    }
    if (s.req.area.bottom_right.y == 0.0) {
	s.req.area.bottom_right.y = s.hw.max_size.y;
    }

    if (verbose) {
	float width, height;

	width  = s.req.area.bottom_right.x - s.req.area.top_left.x;
	height = s.req.area.bottom_right.y - s.req.area.top_left.y;
	fprintf(stderr,
		"%s: scanning %gx%gmm^2 at (%g,%g) at %ddpi (%dx%d pixels)\n",
		prog_name, width, height,
		s.req.area.top_left.x, s.req.area.top_left.y, s.req.resolution,
		(int) (width  / MM_PER_INCH * s.req.resolution + 0.5),
		(int) (height / MM_PER_INCH * s.req.resolution + 0.5));
    }

    image = msci_scan_image(&s, &width, &height);
    if (!image) {
	goto error_exit;
    }
    if (s.req.mode & MSCI_COLOR) {
	/* write image as a raw ppm image: */
	fprintf(ofp, "P6\n%ld %ld 255\n", (long) width, (long) height);
	if (fwrite(image, 3 * width * height, 1, ofp) != 1) {
	    perror("fwrite");
	}
    } else {
	if (s.req.mode & MSCI_MULTIBIT) {
	    /* write image as a raw pgm image: */
	    fprintf(ofp, "P5\n%ld %ld 255\n", (long) width, (long) height);
	    if (fwrite(image, width * height, 1, ofp) != 1) {
		perror("fwrite");
	    }
	} else {
	    /* write image as a raw pbm image: */
	    fprintf(ofp, "P4\n%ld %ld\n", (long) width, (long) height);
	    if (fwrite(image, width * height / 8, 1, ofp) != 1) {
		perror("fwrite");
	    }
	}
    }
    free(image);
    return 0;

  error_exit:
    msci_close(&s);
    if (ofp != stdout) {
	fclose(ofp);
    }
    return -1;
}
