/*
 *  ac4096.c - Scanner-driver for A4 Tech AC 4096
 *
 *  Copyright (c) 1994 Michael Beck (beck@informatik.hu-berlin.de)
 *  Some parts are (c) Thomas Faehnle (Thomas.Faehnle@student.uni-ulm.de)
 *             and (c) Andreas Beck (becka@hp.rz.uni-duesseldorf.de)
 *
 *  This is the driver for the A4 Tech AC 4096 handheld scanner.
 *  I/O-ports and DMA channel are defined in ac4096.h.
 */

/* 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
 
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/termios.h>
#include <linux/malloc.h>
#include <linux/module.h>
#include <linux/fcntl.h>

#include <asm/dma.h>
#include <asm/system.h>
#include <asm/segment.h>

#include "ac4096.h"
 
#define AC_OFF		0
#define AC_ON		2
#define AC_INIT_OK	4

/*
 * define some usefull macros
 */
 
/* first the 'DMA transfer ready' makro */

#if (AC4096_DMA < 4)
#define DMA_READY	( inb(DMA1_STAT_REG) & (1 << AC4096_DMA) )
#else
#define DMA_READY	( inb(DMA2_STAT_REG) & (1 << (AC4096_DMA & 3)) )
#endif

/* now some address calculations */

#define RD_ADDR         (&(rd.blkptr->data[rd.index - rd.blkptr->blkoff]))
#define WR_ADDR         (&(wr.blkptr->data[wr.index - wr.blkptr->blkoff]))
#define SELECT_TH       (hw_modeinfo.bpl * select_threshold)

#define RD_BLKEND       (rd.blkptr->blkoff + rd.blkptr->blksiz)
#define WR_BLKEND       (wr.blkptr->blkoff + wr.blkptr->blksiz)

#define RD_BUFSPC	(wr.index >= rd.index ? ((wr.index == rd.index) && ! empty_buffer ? \
	bufsiz : 0) + wr.index - rd.index : bufsiz - rd.index + wr.index)
#define WR_BUFSPC	(wr.index < rd.index ? rd.index - wr.index : \
	((wr.index == rd.index) && ! empty_buffer ? 0 : bufsiz) - wr.index + rd.index)

#define RD_ADDR		(&(rd.blkptr->data[rd.index - rd.blkptr->blkoff]))
#define WR_ADDR		(&(wr.blkptr->data[wr.index - wr.blkptr->blkoff]))
#define SELECT_TH	(hw_modeinfo.bpl * select_threshold)

/* needed for module installer */

char kernel_version[] = KERNEL_VERSION;

/* static data */

static struct wait_queue *scan_sleep = NULL;
static struct timer_list timer;
static char xfer_going = 0, empty_buffer = 0;
static int bytes_requested = 0, buflines = 0, bufsiz = 0, req_lines = 0;
static struct buffer_block *first_buf;
static struct buffer_pointer rd = { NULL, 0 }, wr = { NULL, 0 };
static int select_threshold = 1;

static char copy_buffer[684];	/* 3/2 * bpl */
static int inbuffer = 0;

static struct ac4096 {
	char		active;
	unsigned char	scan_cmd;
	unsigned short	scan_port;
} ac;

static struct scanner_type sctxt = { "A4 Tech", "AC 4096" };

static struct modeinfo_struct ac4096_modeinfo = 
{ 90, 90, 		/* xdpi,ydpi  */
  SCM_TRUE, 		/* sc_mode    */
  0x0F0F0F,		/* depth      */
  456,			/* bpl        */
  0,0,			/* left,right */
  0,0,			/* top,bottom */
  0,0,0,0,		/* bright,... */
  0			/* options    */
};

static struct hw_modeinfo_struct 
		{ int xdpi,ydpi, sc_mode,depth, bpl,bps; } 
  hw_modeinfo = {       90,  90,SCM_TRUE, 4095, 456,  4  };


struct scanner_capabilities scc_ac4096 = {

	SCC_SOFTSELECT | SCC_EMULATE,	/* O_monochrome mode */
	SCC_SOFTSELECT | SCC_EMULATE,	/* O_greyscale  mode */
	SCC_SOFTSELECT | SCC_SOFTDETECT,/* O_true-color mode */

	SCC_SOFTDETECT,			/* Options for x resolution */
	SCC_SOFTDETECT,			/* Options for y resolution */

	SCC_HARDSWITCH,			/* Options for brightness */
	{0,0},
	0,				/* Options for contrast */
	{0,0},
	0,				/* Options for hue */
	{0,0},
	0,				/* Options for saturation */
	{0,0},

	67,				/* Width in mm */
	0,				/* Length in mm (unlimited) */
	0				/* No options */
};


/* mem functions */

static void free_scan_mem(void)
{
	struct buffer_block *tmp = first_buf, *next;
	unsigned int size;
	if (! buflines)
		return;
	while (tmp) {
		size = tmp->blksiz + HEADERSIZ;
		next = tmp->next;
		kfree_s((char*) tmp, size);
		tmp = next;
	};
	first_buf = NULL;
	buflines = 0; bufsiz = 0;
}

/*
 * This may allocate memory after 16M, so DMA will fail on ISA
 * so we need a better version in the future
 */

static int get_scan_mem(int lines)
{
	struct buffer_block *tmp, **lastptr;
	unsigned int blksize;

	if ((lines * hw_modeinfo.bpl) > MAX_BUFSIZ)
		return -EINVAL;
	if (buflines)
		free_scan_mem();
	lastptr = &first_buf;
	while (lines > 0) {
		blksize = lines * hw_modeinfo.bpl;
		if (blksize > MAX_BLK) 
			blksize = MAX_BLK - (MAX_BLK % hw_modeinfo.bpl);
		tmp = (struct buffer_block*)kmalloc(blksize + HEADERSIZ, GFP_KERNEL);
		if (! tmp) {
			free_scan_mem();
			return -ENOMEM;
		}
		tmp->next = NULL;
		*lastptr = tmp;
		lastptr = &(tmp->next);
		tmp->blksiz = blksize;
		tmp->blkoff = buflines * hw_modeinfo.bpl;
		lines -= blksize / hw_modeinfo.bpl;
		buflines += blksize / hw_modeinfo.bpl;
		bufsiz += blksize;
		rd.blkptr = first_buf; rd.index = 0;
		wr.blkptr = first_buf; wr.index = 0;
		empty_buffer = 1;
	}
	return 0;
}

/* set the modeindo struct to 'mode' or correct any settings */

static void set_mode_info(int mode)
{
	if (mode)
		ac4096_modeinfo.sc_mode = mode;

	ac4096_modeinfo.xdpi = 90;	/* 90 DPI */
	ac4096_modeinfo.ydpi = 90;
	if ( (ac4096_modeinfo.sc_mode != SCM_MONO) &&
	     (ac4096_modeinfo.sc_mode != SCM_GREY) &&
	     (ac4096_modeinfo.sc_mode != SCM_TRUE) )
		ac4096_modeinfo.sc_mode = SCM_TRUE;
	switch (ac4096_modeinfo.sc_mode) {
		case SCM_MONO :
			ac4096_modeinfo.depth = 0;
			ac4096_modeinfo.bpl   = hw_modeinfo.bpl >> 4;
			if (! ac4096_modeinfo.bright)
				/* 70 seems a good value */
				ac4096_modeinfo.bright = 70;
			break;
		case SCM_GREY :
			ac4096_modeinfo.depth  = 0xFF;
			ac4096_modeinfo.bpl    = hw_modeinfo.bpl >> 1;
			ac4096_modeinfo.bright = 0;
			break;
		case SCM_TRUE :
			ac4096_modeinfo.depth  = 0x0F0F0F;
			ac4096_modeinfo.bpl    = (hw_modeinfo.bpl * 3) >> 1;
			ac4096_modeinfo.bright = 0;
			break;
	}
	ac4096_modeinfo.left     = ac4096_modeinfo.right   =
	ac4096_modeinfo.top      = ac4096_modeinfo.bottom  =
	ac4096_modeinfo.contrast = ac4096_modeinfo.hue     =
	ac4096_modeinfo.sat      = ac4096_modeinfo.options = 0;
}

/* scanner funcs */

static void wait_end(unsigned long dummy)
{
	wake_up(&scan_sleep);
}

static inline void wait_clks(long clks)
{
	timer.expires = clks;
	timer.function = wait_end;
	add_timer(&timer);
	sleep_on(&scan_sleep);
}

/* scanner functions */

static inline void stop_scanner(void)
{
	outb_p(AC_OFF, ac.scan_port);
	outb_p(AC_OFF, ac.scan_port); 
}

static int init_scanner(void)
{
	unsigned char cmd, in;

	outb(AC_ON, ac.scan_port);
	outb(AC_ON, ac.scan_port);
	outb(AC_ON, ac.scan_port);
	wait_clks(27);		/* wait 0.27 sec */
	cmd = (ac.scan_cmd & 0xFD) | AC_ON;
	outb_p(cmd, ac.scan_port);
	in = inb(ac.scan_port);
	if ( !(in & AC_INIT_OK) ) {
		stop_scanner();
		return 8;
	}
	wait_clks(10);		/* wait 0.1 sec */
	in = inb(ac.scan_port);
	if ( (in & 0xF8) != (cmd & 0xF8) ) {
		stop_scanner();
		return 0x80;
	}
	return 0;		/* ok, scanner initialised */
}

static void test_dma_rdy(unsigned long dummy);

static inline void start_snooping(void)
{
	timer.expires = 2;
	timer.function = test_dma_rdy;
	add_timer(&timer);
}

static void start_dma_xfer(char *buf)
{
	disable_dma(AC4096_DMA);
	clear_dma_ff(AC4096_DMA);
	set_dma_mode(AC4096_DMA, DMA_MODE_READ);
	set_dma_addr(AC4096_DMA, (unsigned int) buf);
	set_dma_count(AC4096_DMA, hw_modeinfo.bpl);
	enable_dma(AC4096_DMA);
}

/* test if dma transfer is done */
static void test_dma_rdy(unsigned long dummy)
{
	static int needed_bytes;
	char cmd;

	if (! xfer_going)
		return;
	start_snooping();	/* restart snooping */

	if ( DMA_READY ) {
		if (bytes_requested) {
			needed_bytes = bytes_requested;
			bytes_requested = 0;
		}
		wr.index += hw_modeinfo.bpl;
		if (wr.index >= WR_BLKEND) {
			wr.blkptr = wr.blkptr->next;
			if (! wr.blkptr) {
				wr.blkptr = first_buf;
				wr.index -= bufsiz;
			}
		}
		if (wr.index == rd.index) /* Shouldn't happen here ...*/
			empty_buffer = 0;

		if (needed_bytes > 0)
			needed_bytes -= hw_modeinfo.bpl;
		if (needed_bytes <= 0 || WR_BUFSPC <= hw_modeinfo.bpl * FREE_LN)
			wake_up_interruptible(&scan_sleep);

		stop_scanner();

		/* if there is enough bufspace */
		if (WR_BUFSPC >= hw_modeinfo.bpl) {		
			cmd = (ac.scan_cmd & 0xFD) | AC_ON;
			outb(cmd, ac.scan_port);
			outb(cmd, ac.scan_port);
			outb(cmd, ac.scan_port);
			start_dma_xfer(WR_ADDR);
		}
		else
			xfer_going = 0;
	}
}

static void scanner_operate(void)
{
	cli();
	start_dma_xfer(WR_ADDR);
	sti();
	xfer_going = 1;
	start_snooping();	
	enable_dma(AC4096_DMA);
}

static void ac4096_release(struct inode * inode, struct file * file)
{
	xfer_going = 0;
	stop_scanner();
	free_dma(AC4096_DMA);
	if (buflines)
		free_scan_mem();
	ac.active = 0;
	MOD_DEC_USE_COUNT;
}

static int ac4096_open(struct inode * inode, struct file * file)
{
	int err;

	if (MINOR(inode->i_rdev) != 0)
		return -ENODEV;

	if (ac.active)
		return -EBUSY;

	if ( (err = request_dma(AC4096_DMA,"scanner")) ) {
		printk("AC 4096: unable to get DMA%d\n", AC4096_DMA);
		return err;
	}
	if (init_scanner() ) {
		free_dma(AC4096_DMA);
		return -EIO;
	}
	MOD_INC_USE_COUNT;

	/* after a open the scanner is in TRUE COLOR mode */
	set_mode_info(SCM_TRUE);

	xfer_going = 0;
	inbuffer   = 0;
	ac.active  = 1;

	return 0;
}

static int ac4096_ioctl(struct inode * inode, struct file * file,
                        unsigned int cmd, unsigned long arg)
{

	int r;
	unsigned long tmp;

	if (!arg)
		return -EINVAL;
	switch (cmd) {
		case HSIOCGOPT:		/* get scanner_options struct */
			return 0;	/* NOTHING */
		case HSIOCGSCC:		/* get scanner_capabilities struct */
			memcpy_tofs((char*) arg, (char*) &scc_ac4096, 
		      		sizeof(struct scanner_capabilities));
			return 0;
		case HSIOCGSCT:		/* get scanner_type struct */
			memcpy_tofs((char*) arg, (char*) &sctxt,
				sizeof(struct scanner_type));
			return 0;
		case HSIOCSMOD:		/* set modeinfo struct */
			memcpy_fromfs((char*) &ac4096_modeinfo, (char*) arg,
				sizeof(struct modeinfo_struct));
			set_mode_info(0);
			memcpy_tofs((char*) arg, (char*) &ac4096_modeinfo,
				sizeof(struct modeinfo_struct));
			return 0;
		case HSIOCGMOD:		/* get modeinfo struct */
			set_mode_info(0);
			memcpy_tofs((char*) arg, (char*) &ac4096_modeinfo,
				sizeof(struct modeinfo_struct));
			return 0;
		case HSIOCSBUF:		/* set size of internal buffer */
			if ( !(r = get_fs_long(arg)) )
				free_scan_mem();
			else {
				req_lines = r;
				get_scan_mem(req_lines);
			}
			return 0;
		case HSIOCGBUF:		/* get current buffer size */
			put_fs_long((unsigned long) req_lines, arg);
			return 0;
		case HSIOCSSTH:		/* set select(2) threshold */
			select_threshold = get_fs_long(arg);
			return 0;
		case HSIOCGSTH:		/* get current select(2) threshold */
			put_fs_long((unsigned long) select_threshold, arg);
			return 0;
		case HSIOCGSIB:		/* get number of lines in buffer */
			tmp = RD_BUFSPC / hw_modeinfo.bpl;
			put_fs_long(tmp , arg);
			return 0;
		case HSIOCGREG:		/* get setting of misc registers */
		case HSIOCSREG:		/* set misc registers */
			/* NOT IMPLEMENTED */
		default:
			return -EINVAL;
		}
}

/* translate the read data */
static int refill_buffer(void)
{
	int i, k, j = 0;
	unsigned long hue;
	char mask;

	if (RD_BUFSPC < hw_modeinfo.bpl)
		return 0;

	switch (ac4096_modeinfo.sc_mode) {
	case SCM_MONO:
		for (i = 0; i < (hw_modeinfo.bpl & 0xFFFFFFE0);) {
			mask = 0;
			for (k = 0; k < 8; ++k, i += 2) {
				hue = (RD_ADDR[i] & 0xF0)        * 49 + 
			 	      ((RD_ADDR[i] & 0x0F) << 4) * 30 +
				      (RD_ADDR[i+1] & 0xF0)      * 11;
				hue /= 300;
			        mask <<= 1;
				mask |= (hue > ac4096_modeinfo.bright) ? 1 : 0;
			}
			copy_buffer[j++] = mask;
		}
		break;
	case SCM_GREY:
		for (i = 0; i < hw_modeinfo.bpl; i += 2) {
			hue = (RD_ADDR[i] & 0xF0)        * 49 + 
			      ((RD_ADDR[i] & 0x0F) << 4) * 30 +
			      (RD_ADDR[i+1] & 0xF0)      * 11;
			copy_buffer[j++] = (hue / 300);
		}
		break;
	case SCM_TRUE:
		for (i = 0; i < hw_modeinfo.bpl; i += 2) {
			copy_buffer[j++] = RD_ADDR[i] & 0xF0;
			copy_buffer[j++] = (RD_ADDR[i] & 0x0F) << 4;
			copy_buffer[j++] = RD_ADDR[i+1] & 0xF0;
		}
	}
	inbuffer = j;
	rd.index += hw_modeinfo.bpl;
	if (rd.index >= RD_BLKEND) {
		rd.blkptr = rd.blkptr->next;
		if (! rd.blkptr) {
			rd.blkptr = first_buf;
			rd.index -= bufsiz;
		}
	}
	cli();
	if (rd.index == wr.index)
		empty_buffer = 1;
	sti();
	return 0;
}

static inline int get_available_bytes(char **user_buf, int count)
{
	int chunk, copied = 0, to_copy = count;
	static int offset = 0;

	while (to_copy > 0) {
		if (! inbuffer) {
			refill_buffer();
			offset = 0;
		}
		if (! inbuffer)
			break;

		chunk = (inbuffer > to_copy) ? to_copy : inbuffer;
		memcpy_tofs(*user_buf, copy_buffer + offset, (long)chunk);

		inbuffer -= chunk;
		to_copy  -= chunk;
		copied   += chunk;
		offset   += chunk;
		*user_buf+= chunk;
	}
	return copied;
}

static int ac4096_read(struct inode * inode, struct file * file,
                      char * buf, int size)
{
	int err, needed = size;

	if (! buflines)	{	/* allocate mem if needed */
		if ( (err = get_scan_mem(DEFAULT_BUFLINES)) )
			return err;
		req_lines = DEFAULT_BUFLINES;
	}

	needed -= get_available_bytes(&buf, size);
	
	if (! xfer_going && (WR_BUFSPC >= hw_modeinfo.bpl)) 
		scanner_operate();

	if (file->f_flags & O_NONBLOCK)
		return size - needed;

	while (needed > 0) {
		bytes_requested = needed * hw_modeinfo.bpl/ac4096_modeinfo.bpl;
		if (! xfer_going)
			scanner_operate();
		interruptible_sleep_on(&scan_sleep);
		needed -= get_available_bytes(&buf, needed);
		if (current->signal & ~current->blocked)
			if (size - needed)
				return (size - needed);
			else
				return -EINTR;
	}
	return size;
}

static int ac4096_select(struct inode *inode, struct file *file,
	int type, select_table *st)
{
	int tmp, err;

	if (type != SEL_IN)
		return 0;
	if (! buflines)
		if ( (err = get_scan_mem(req_lines = DEFAULT_BUFLINES)) )
			return err;
	cli();
	tmp = RD_BUFSPC;
	sti();
	if (tmp >= SELECT_TH) 
		return 1;		/* device is ready */

	if (st) {
		bytes_requested = SELECT_TH;
		if (! xfer_going)
			scanner_operate();
	}
	select_wait(&scan_sleep, st);
	return 0;
}

struct file_operations ac4096_fops = {
        NULL,		/* ac4096_seek */
        ac4096_read,
        NULL,		/* ac4096_write */
        NULL,		/* ac4096_readdir */
        ac4096_select,	/* ac4096_select */
        ac4096_ioctl,
        NULL,           /* ac4096_mmap */
        ac4096_open,
        ac4096_release,
};

int init_module(void)
{
	ac.scan_port = AC4096_IOBASE;
#if (AC4096_DMA == 1)
	ac.scan_cmd  = 0xE4;
#else
	ac.scan_cmd  = 0xE5;
#endif

	if ( init_scanner() ) {
		return -EIO;
	}
	stop_scanner();

	ac.active  = 0;
	printk("A4 Tech AC 4096 on DMA%d found\n", AC4096_DMA);
	if (register_chrdev(AC4096_SCANNER_MAJOR, AC4096_SCANNER_NAME,
				&ac4096_fops)) {
		printk("AC 4096: cannot register major number\n");
		return -EIO;
	}
	return 0;
}

void cleanup_module(void)
{
	printk("Removing A4 Tech handheld scanner driver from memory\n");
	if (MOD_IN_USE)
		printk("AC 4096: device is busy, delaying driver removal\n");
	if (unregister_chrdev(AC4096_SCANNER_MAJOR, AC4096_SCANNER_NAME))
		printk("AC 4096: unregister_chrdev() failed\n");
}
