
/* qddb/Lib/LibQddb/WordList.c
 *
 * Copyright (C) 1996 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 * Qddb_InitWordList
 * Qddb_UpdateWordList
 */

static int GetVersion _ANSI_ARGS_((int));

static WordList *FindWordInWordList _ANSI_ARGS_((Schema *, char *));
static void AddKeyListToWordList _ANSI_ARGS_((Schema *, char *, KeyList *));
static void AddWordToWordList _ANSI_ARGS_((Schema *, KeyListType, int, KeyNode *));
static void AddWordsToWordList _ANSI_ARGS_((Schema *, KeyListType, int, KeyNode *));

static void AddEntryToWordList _ANSI_ARGS_((Schema *, KeyListType, int, Entry));
static Entry BufferToEntry _ANSI_ARGS_((char *, size_t));
static void ReadWordList _ANSI_ARGS_((Schema *, int, off_t));
static void WriteWordList _ANSI_ARGS_((Schema *, int));
static void DeleteNodesInWordList _ANSI_ARGS_((KeyList **, KeyListType, int));
static void InvalidateEntryInWordList _ANSI_ARGS_((Schema *, KeyListType, int));
static void ParseEntryToWordList _ANSI_ARGS_((Schema *, Entry));
static int CompareWordListNodes _ANSI_ARGS_((const void *, const void *));

void Qddb_InitWordList(schema)
    Schema			*schema;
{
    schema->word_list = NULL;
    schema->word_list_version = 0;
    schema->word_list_size = 0;
    schema->word_list_number_words = 0;
}

void Qddb_UpdateWordList(schema, type, number, entry)
    Schema			*schema;
    KeyListType			type;
    int				number;
    Entry			entry;
{
    int				fd;
    unsigned int		current_version;
    char			file_name[BUFSIZ];
    struct stat                 st;

    strcpy(file_name, schema->RelationName);
    strcat(file_name, "/SecondaryCache");
    current_version = 0;
    /* 1. word list file doesn't exist
     * 2. word list is out of date or hasn't been read
     * 3. word list is current
     */
    if (stat(file_name, &st) == -1) { /* word list doesn't exist */
	if (entry == NULL)
	    return;
	if ((fd = open(file_name, O_WRONLY|O_CREAT, 0666)) == -1) {
	    perror("Error: cannot open SecondaryCache");
	    return;
	}
	if (entry != NULL) {
	    LockSection(fd, F_WRLCK, (off_t)0, (off_t)0, 1);
	} else {
	    LockSection(fd, F_RDLCK, (off_t)0, (off_t)0, 1);
	}
	if (fstat(fd, &st) == -1) {
	    PANIC("cannot stat word list file");
	}
	if (st.st_size != 0) {
	    ReadWordList(schema, fd, st.st_size);
	}
    } else {
	if ((fd = open(file_name, O_RDWR)) == -1)
	    return;
	if (entry != NULL || QDDB_KEYLIST_ISDELNUM(type)) {
	    LockSection(fd, F_WRLCK, (off_t)0, (off_t)0, 1);	
	} else {
	    LockSection(fd, F_RDLCK, (off_t)0, (off_t)0, 1);
	}
	if (fstat(fd, &st) == -1) {
	    PANIC("cannot stat word list file");
	}
	current_version = GetVersion(fd);
	if (st.st_size != 0 && 
	    (current_version > schema->word_list_version || schema->word_list == NULL)) {
	    ReadWordList(schema, fd, st.st_size);
	}
    }
    if (entry != NULL) {
#if defined(HAVE_FTRUNCATE)
	ftruncate(fd, (off_t)0);
#else
	/* we lose the lock here */
	close(Open(file_name, O_RDWR|O_CREAT|O_TRUNC, 0666));
	LockSection(fd, F_WRLCK, (off_t)0, (off_t)0, 1);
#endif
	AddEntryToWordList(schema, QDDB_KEYLIST_TYPENUM(type), number, entry);
	WriteWordList(schema, fd);
    } else if (QDDB_KEYLIST_ISDELNUM(type)) {
#if defined(HAVE_FTRUNCATE)
	ftruncate(fd, (off_t)0);
#else
	/* we lose the lock here */
	close(Open(file_name, O_RDWR|O_CREAT|O_TRUNC, 0666));
	LockSection(fd, F_WRLCK, (off_t)0, (off_t)0, 1);
#endif
	InvalidateEntryInWordList(schema, QDDB_KEYLIST_TYPENUM(type), number);
	WriteWordList(schema, fd);	
    }
    UnlockSection(fd, (off_t)0, (off_t)0);
    close(fd);
}

static int GetVersion(fd)
    int				fd;
{
    char			buf[21];

    if (read(fd, buf, (size_t)20) != (size_t)20) {
	return 0;
    }
    buf[20] = '\0';
    return atoi(buf);
}

static int CompareWordListNodes(n1, n2)
    const void			*n1, *n2;
{
    WordList			*wn1, *wn2;

    wn1 = (WordList *)n1;
    wn2 = (WordList *)n2;
    return strcmp(wn1->word, wn2->word);
}

static WordList *FindWordInWordList(schema, word)
    Schema			*schema;
    char			*word;
{
    WordList			node;

    node.word = word;
    return (WordList *)bsearch((void *)&node, (void *)schema->word_list, 
			       schema->word_list_number_words, sizeof(WordList),
			       (int (*) _ANSI_ARGS_((const void *, const void *)))CompareWordListNodes);
}

static void AddKeyListToWordList(schema, word, list)
    Schema			*schema;
    char			*word;
    KeyList			*list;
{
    int				current;

    if (schema->word_list == NULL) {
	schema->word_list_size = QDDB_WORDLIST_INCR;
	schema->word_list_number_words = 0;
	schema->word_list = (WordList *)Malloc(sizeof(WordList)*QDDB_WORDLIST_INCR);
    } else if (schema->word_list_size-2 <= schema->word_list_number_words) {
	schema->word_list_size += QDDB_WORDLIST_INCR;
	schema->word_list = (WordList *)Realloc(schema->word_list,
						sizeof(WordList)*schema->word_list_size);
    }
    current = schema->word_list_number_words++;
    schema->word_list[current].word = word;
    schema->word_list[current].list = list;
    schema->word_list[current+1].word = NULL;
}

static void AddWordToWordList(schema, type, number, node)
    Schema			*schema;
    KeyListType			type;
    int				number;
    KeyNode			*node;
{
    WordList			*wordlist = NULL;
    KeyList			*keylist;
    int				current, i, j;

    if (schema->word_list == NULL) {
	schema->word_list_size = QDDB_WORDLIST_INCR;
	schema->word_list_number_words = 0;
	schema->word_list = (WordList *)Malloc(sizeof(WordList)*QDDB_WORDLIST_INCR);
    } else if (schema->word_list_size-2 <= schema->word_list_number_words) {
	schema->word_list_size += QDDB_WORDLIST_INCR;
	schema->word_list = (WordList *)Realloc(schema->word_list,
						sizeof(WordList)*schema->word_list_size);
    }
    if ((wordlist = FindWordInWordList(schema, node->s)) == NULL) {
	current = schema->word_list_number_words++;
	for (i = 0; i < current; i++) {
	    if (strcmp(schema->word_list[i].word, node->s) > 0) {
		break;
	    }
	}
	for (j = current; j > i; j--) {
	    schema->word_list[j].word = schema->word_list[j-1].word;
	    schema->word_list[j].list = schema->word_list[j-1].list;
	}
	schema->word_list[current+1].word = NULL;
	schema->word_list[i].word = node->s;
	keylist = (KeyList *)Malloc(sizeof(KeyList));
	keylist->Type = type;
	keylist->Number = number;
	keylist->Start = 0;
	keylist->Length = 0;
	keylist->Attribute = node->attr;
	keylist->Instance = node->inst;
	keylist->next = NULL;
	schema->word_list[i].list = keylist;
    } else {
	keylist = (KeyList *)Malloc(sizeof(KeyList));
	keylist->Type = type;
	keylist->Number = number;
	keylist->Start = 0;
	keylist->Length = 0;
	keylist->Attribute = node->attr;
	keylist->Instance = node->inst;
	keylist->next = wordlist->list;
	wordlist->list = keylist;
	Free(node->s);
    }
    Free(node);
}

static void AddWordsToWordList(schema, type, number, list)
    Schema			*schema;
    KeyListType			type;
    int				number;
    KeyNode			*list;
{
    KeyNode			*next;

    while (list != NULL) {
	next = list->next;
	AddWordToWordList(schema, type, number, list); /* frees (*list) */
	list = next;
    }
}

static void AddEntryToWordList(schema, type, number, entry)
    Schema			*schema;
    KeyListType			type;
    int				number;
    Entry			entry;
{
    KeyNode			*list;
    int				error;
    Boolean			use_redattr;

    InvalidateEntryInWordList(schema, type, number);
    /* changes/additions don't use reduced attribute
     * identifiers, so we need to temporarily disable
     * them.
     */
    use_redattr = schema->UseReducedAttributeIdentifiers;
    schema->UseReducedAttributeIdentifiers = False; 
    list = Qddb_IndexParse(schema, entry+1, &error);
#if defined(DIAGNOSTIC)
    if (error) {
	PANIC("Cannot parse entry\n");
    }
#endif
    AddWordsToWordList(schema, type, number, list);
    schema->UseReducedAttributeIdentifiers = use_redattr;
}

static void DeleteNodesInWordList(list, type, number)
    KeyList			**list;
    KeyListType			type;
    int				number;
{
    KeyList			*thisnode;

    if (*list == NULL)
	return;
    if ((*list)->Type == type && (*list)->Number == number) {
	thisnode = *list;
	*list = thisnode->next;
	thisnode->next = NULL;
	Qddb_Free(QDDB_TYPE_KEYLIST, thisnode);
	DeleteNodesInWordList(list, type, number);
    } else {
	DeleteNodesInWordList(&((*list)->next), type, number);
    }
}

static void InvalidateEntryInWordList(schema, type, number)
    Schema			*schema;
    KeyListType			type;
    int				number;
{
    int				i;
    int				num_words = schema->word_list_number_words;

    for (i = 0; i < num_words; i++) {
	DeleteNodesInWordList(&(schema->word_list[i].list), type, number);
    }
}

static void ParseEntryToWordList(schema, entry)
    Schema			*schema;
    Entry			entry;
{
    KeyList			*list;
    char			*place, *word;

    while (*entry != NULL) {
	list = Qddb_ParseKey2(schema, *entry, &place);
	*place = '\0';
	word = (char *)Malloc(strlen(*entry)+1);
	strcpy(word, *entry);
	*place = ' ';
	AddKeyListToWordList(schema, word, list);
	entry++;
    }
}

static Entry BufferToEntry(buf, size)
    char			*buf;
    size_t			size;
{
    Entry			entry;
    size_t			newlines, i, j, len;
    char			*spot, *ch;
    
    newlines = 0;
    for (spot = buf, i = 0; i < size; i++, spot++) {
	if (*spot == '\n') {
	    newlines++;
	}
    }
    entry = (Entry)Calloc(sizeof(char *)*(newlines+1));
    entry[newlines] = NULL;
    for (spot = buf, i = 0, j = 0; i < size; i++, buf++) {
	if (*buf == '\n') {
	    len = (size_t)(buf - spot);
	    entry[j] = (char *)Malloc(len+2);
	    ch = entry[j++];
	    while (*spot != '\n')
		*ch++ = *spot++;
	    *ch++ = '\n';
	    *ch = '\0';
	    spot = buf+1;
	}
    }
    return entry;
}

static void ReadWordList(schema, fd, size)
    Schema			*schema;
    int				fd;
    off_t			size;
{
    Entry			entry;
    char			*buf;
    int				current_version;

    lseek(fd, (off_t)0, 0);
    buf = (char *)Malloc((size_t)size+1);
    Read(fd, buf, (size_t)size);
    buf[size] = '\0';
    entry = BufferToEntry(buf, (size_t)size);
    Free(buf);
    Qddb_Free(QDDB_TYPE_WORDLIST, schema->word_list);
    Qddb_InitWordList(schema);
    current_version = atoi(*entry);
    ParseEntryToWordList(schema, entry+1);
    Qddb_Free(QDDB_TYPE_ENTRY, entry);
    schema->word_list_version = current_version;
}

static void WriteWordList(schema, fd)
    Schema			*schema;
    int				fd;
{
    WordList			*word_list;
    KeyList			*keylist;
    char			*word, *qbuf, buf[64];
    size_t			qlen;
    int				i, num_words;

    qsort(schema->word_list, schema->word_list_number_words, sizeof(WordList),
	  (int (*) _ANSI_ARGS_((const void *, const void *)))CompareWordListNodes);
    word_list = schema->word_list;
    num_words = schema->word_list_number_words;
    Qddb_InitBuffer();
    lseek(fd, (off_t)0, 0);
    sprintf(buf, "%20d\n", ++(schema->word_list_version));
    Qddb_ConcatBuffer(buf);
    for (i = 0; i < num_words; i++) {
	/* build the buffer 
	 */
	if (word_list[i].list == NULL)
	    continue;
	word = word_list[i].word;
	buf[1] = '\0';
	while (*word != '\0') {
	    if (*word == ' ') {
		Qddb_ConcatBuffer("\\ ");
	    } else if (*word == '\\') {
		Qddb_ConcatBuffer("\\\\");
	    } else {
		*buf = *word;
		Qddb_ConcatBuffer(buf);
	    }
	    word++;
	}
	for (keylist = word_list[i].list; keylist != NULL; keylist = keylist->next) {
	    Qddb_ConcatBuffer(" ");
	    Qddb_ConcatBuffer(FastIntegerToString((int)keylist->Type));
	    Qddb_ConcatBuffer(" ");
	    Qddb_ConcatBuffer(FastIntegerToString((int)keylist->Number));
	    Qddb_ConcatBuffer(" ");
	    Qddb_ConcatBuffer(FastIntegerToString((int)keylist->Attribute));
	    Qddb_ConcatBuffer(" ");
	    Qddb_ConcatBuffer(keylist->Instance);
	}
	Qddb_ConcatBuffer("\n");
	qbuf = Qddb_GetBufferWithoutResize();
	qlen = Qddb_GetBufferLength();
	if (write(fd, qbuf, qlen) == -1) {
	    fprintf(stderr, "Qddb_WriteWordList: write failed, errno %d", errno);
	}
	Qddb_SetBufferLength((size_t)0);
    }
    qbuf = Qddb_GetBufferWithoutResize();
    Free(qbuf);
}




