
/* qddb/Lib/LibQddb/Utils.c
 *
 * Copyright (C) 1993-1997 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2 
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	Open(char *, int, int)
 *	OpenDatabase(char *, int)
 *	....
 */

int Open(Name, Flags, Mode)
    char		*Name;
    int			Flags;
    int			Mode;
{
    int			FileDescriptor;

    FileDescriptor = open(Name, Flags, 0666);
    return FileDescriptor;
}

int OpenDatabase(RelationFN, Mode)
    char		*RelationFN;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    strcpy(TmpRelationFN, RelationFN);
    strcat(TmpRelationFN, "/Database");
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenChangeFile(RelationFN, Number, Mode)
    char		*RelationFN;
    size_t		Number;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    sprintf(TmpRelationFN, "%s/Changes/%d", RelationFN, (int)Number);
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenAdditionFile(RelationFN, Number, Mode)
    char		*RelationFN;
    size_t		Number;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    sprintf(TmpRelationFN, "%s/Additions/%d", RelationFN, (int)Number);
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenHashTable(RelationFN, Mode)
    char		*RelationFN;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];

    strcpy(TmpRelationFN, RelationFN);
    strcat(TmpRelationFN, "/HashTable");
    return Open(TmpRelationFN, O_RDONLY, Mode);
}

void Close(FileDesc)
    int		FileDesc;
{
    if (close(FileDesc) == -1)
    	perror("Close");
}

/* Qddb_FindRelation --
 *	Return value must copied if routine reentered before the
 * string is used.
 */
char *Qddb_FindRelation(RelationName)
    char		*RelationName;
{
    static char		NewName[MAXPATHLEN];
    static char		*QddbDirs = NULL; 
    static char		*RealQddbDirs = NULL;
    char		*getenv();
    char		*Dir, *TmpDirs, *last_ch, TmpRelationName[MAXPATHLEN], OldName[MAXPATHLEN];

    if (RelationName[0] == '\0')
	return NULL;
    strcpy(TmpRelationName, RelationName);
    last_ch = TmpRelationName + strlen(TmpRelationName) - 1;
    while (*last_ch == '/')
	*last_ch-- = '\0';
    last_ch = strrchr(TmpRelationName, (int)'/');
    GETPWD(OldName);
    if (chdir(TmpRelationName) == 0) {
	GETPWD(NewName);
	if (access("HashTable", 00) != -1) {
	    chdir(OldName);
	    return NewName;
	}
    }
    if (RealQddbDirs == NULL) {
	RealQddbDirs = getenv("QDDBDIRS");
	if (RealQddbDirs == NULL) {
	    return NULL;
	}
	QddbDirs = Malloc(strlen(RealQddbDirs)+1);
    }
    strcpy(QddbDirs, RealQddbDirs);
    TmpDirs = QddbDirs;
    Dir = strtok(TmpDirs, ":");
    if (Dir == NULL) {
	strcpy(NewName, TmpRelationName);
	if (chdir(NewName) == 0) {
	    GETPWD(NewName);
	    if (access("HashTable", 00) != -1) {
		chdir(OldName);
		return NewName;
	    }
	}
    }
    TmpDirs = NULL;
    do {
	if (Dir[0] != '\0') {
	    strcpy(NewName, Dir);
	    strcat(NewName, "/");
	    strcat(NewName, TmpRelationName);
	    if (chdir(NewName) == 0) {
		GETPWD(NewName);
		if (access("HashTable", 00) != -1) {
		    chdir(OldName);
		    return NewName;
		}
	    }
	}
    } while ((Dir=strtok(TmpDirs, ":")) != NULL);
    chdir(OldName);
    return NULL;
}
    	
void Read(File, Buffer, SizeOfBuffer)
    int			File;
    char		*Buffer;
    size_t		SizeOfBuffer;
{
    int			NumberRead;

    if ((NumberRead=read(File, Buffer, SizeOfBuffer)) != SizeOfBuffer) {
    	Buffer[NumberRead] = 0;
    	fprintf(stderr, "Attempted %d, actually read %d of filedesc %d\n", 
    		(int)SizeOfBuffer, NumberRead, File);
	fflush(stderr);
    	perror("Read()");
    	exit(FILE_ERROR);
    }
}

void Write(File, Buffer, SizeOfBuffer)
    int			File;
    char		*Buffer;
    size_t		SizeOfBuffer;
{
    if (write(File, Buffer, SizeOfBuffer) != SizeOfBuffer) {
    	perror("Write():");
    	exit(FILE_ERROR);
    }
}

size_t SizeOfFile(File)
    int			File;
{
    struct stat		FileStatus;

    if (fstat(File, &FileStatus) == -1) {
    	perror("SizeOfFile()");
    	exit(FILE_ERROR);
    }
    return (size_t)FileStatus.st_size;
}

void PANIC(string)
    char		*string;
{
    fprintf(stderr, "%s, errno = %d\n", string, errno);
    fprintf(stderr, "Dumping core for analysis....\n");
    fflush(stderr);
    abort();
    exit(2);
}

void Qddb_SplitBufferIntoLines(Lines, Buffer, BufLen)
    Entry		*Lines;
    char		*Buffer;
    size_t		BufLen;
{
    int			i, j;
    size_t		LineTop;
    char		*ch;
    
    /* Split the buffer into lines.  A line is
     * a set of characters ending in a newline.
     * Newlines embedded in
     * the line must be backslashed ('\').
     */

    /* Free up any traces of an old entry. 
     */
    if (*Lines != NULL)
	Qddb_Free(QDDB_TYPE_ENTRY, *Lines);
    /* Mark the newlines that aren't backslashed and
     * count the number of lines.
     */
    for (i = 0, LineTop = 0; i < BufLen; i++) {
	if (Buffer[i] == '\\' && (Buffer[i+1] == '\\' || Buffer[i+1] == '\n')) {
	    i++;
	} else if (Buffer[i] == '\n') {
	    Buffer[i] = '\0';
	    LineTop++;
	}
    }
    LineTop++;
    *Lines = (Entry)Malloc(sizeof(char *)*(size_t)LineTop);
    for (i = 0, j = 0, ch = Buffer; i < BufLen; i++) {
	if (Buffer[i] == '\0') { /* End of a line */
	    size_t	len;

	    if (i == BufLen - 1 && Buffer[i-1] == '\0') {
		break;
	    }
	    len = strlen(ch) + 1;
	    (*Lines)[j] = Malloc(sizeof(char)*len);
	    strcpy((*Lines)[j], ch);
	    ch = Buffer+i+1;
	    j++;
	}
    }
#if defined(DIAGNOSTIC)
    if (j >= LineTop)
	PANIC("SplitBufferIntoLines(Utils.c): j >= LineTop\n");
#endif
    (*Lines)[j] = NULL;
}

void Qddb_ReadEntry(File, OldEntry, Start, Length, DoLocking)
    int			File;
    Entry		*OldEntry;
    off_t 		Start;
    size_t		Length;
    Boolean		DoLocking;
{
    char		*Buffer;
    struct stat		stat_buf;

#if defined(DIAGNOSTIC)
    if (OldEntry == NULL)
	PANIC("Qddb_ReadEntry: OldEntry == NULL\n");
#endif
#if defined(RECORDLOCKING)
    if (DoLocking == True)
	LockSection(File, F_RDLCK, Start, (off_t)Length, True);
#endif
    if (Length == 0) {
	if (fstat(File, &stat_buf) != 0) {
	    PANIC("Qddb_ReadEntry: cannot stat file\n");
	}
	Length = (size_t)stat_buf.st_size;
    }
    Buffer = Malloc(sizeof(char)*(size_t)(Length+1));
    lseek(File, Start, 0);
    Read(File, Buffer, Length);
    Buffer[Length] = '\0';
#if defined(RECORDLOCKING)
    if (DoLocking == True)
	UnlockSection(File, Start, (off_t)Length);
#endif
    Qddb_SplitBufferIntoLines(OldEntry, Buffer, Length);
    Free(Buffer);
}


int Qddb_ReadEntryByKeyList(FileDesc, RelationFN, ThisEntry, TheKeyList, DoLocking)
    int			FileDesc;
    char		*RelationFN;
    Entry		*ThisEntry;
    KeyList		*TheKeyList;
    Boolean		DoLocking;
{
    int			fd;

    if (TheKeyList == NULL)
	return -1;
    switch(QDDB_KEYLIST_TYPE(TheKeyList)) {
    case ORIGINAL:
    	if (FileDesc == -1) {
	    fd = OpenDatabase(RelationFN, 0);
	    if (fd == -1)
		return -1;
	    Qddb_ReadEntry(fd, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	    Close(fd);
    	} else {
	    Qddb_ReadEntry(FileDesc, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	}
	if (EntryInvalid(*ThisEntry) == True) {
	    return -1;
	}
    	break;
    case CHANGE:
    	fd = OpenChangeFile(RelationFN, TheKeyList->Number, 0);
	if (fd == -1)
	    return -1;
    	Qddb_ReadEntry(fd, ThisEntry, (off_t)0, (size_t)0, DoLocking);
    	Close(fd);
    	break;
    case ADDITION:
    	fd = OpenAdditionFile(RelationFN, TheKeyList->Number, 0);
	if (fd == -1)
	    return -1;
    	Qddb_ReadEntry(fd, ThisEntry, (off_t)0, (size_t)0, DoLocking);
    	Close(fd);
    	break;
    default:
	fprintf(stderr, "Qddb_ReadEntryByKeyList: type %d\n", QDDB_KEYLIST_TYPE(TheKeyList));
	fflush(stderr);
    	PANIC("Qddb_ReadEntryByKeyList: BAD TYPE IN KeyList");
    }	
    return 0;
}

int Qddb_ReadEntryByKeyList2(FileDesc, RelationFN, ThisEntry, TheKeyList, DoLocking)
    int			FileDesc;
    char		*RelationFN;
    Entry		*ThisEntry;
    KeyList		*TheKeyList;
    Boolean		DoLocking;
{
    int			fd;

    switch(QDDB_KEYLIST_TYPE(TheKeyList)) {
    case ORIGINAL:
    	if (FileDesc == -1) {
	    fd = OpenDatabase(RelationFN, 0);
	    if (fd == -1)
		return -1;
	    Qddb_ReadEntry(fd, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	    Close(fd);
    	} else {
	    Qddb_ReadEntry(FileDesc, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	}
	if (EntryInvalid(*ThisEntry) == True) {
	    return -1;
	}
    	break;
    case CHANGE:
    	Qddb_ReadEntry(FileDesc, ThisEntry, (off_t)0, (size_t)0, DoLocking);
    	break;
    case ADDITION:
    	Qddb_ReadEntry(FileDesc, ThisEntry, (off_t)0, (size_t)0, DoLocking);
    	break;
    default:
    	PANIC("Qddb_ReadEntryByKeyList2: BAD TYPE IN KeyList");
    }	
    return 0;
}

/* WriteEntry doesn't use record locking because it writes from the current
 * position until it is finished.  Usually used by programs that have
 * complete control over the relation or already have it locked down.
 */
void WriteEntry(schema, File, WhichEntry)
    Schema		*schema;
    int			File;
    Entry		WhichEntry;
{
    Entry		AttributePtr = WhichEntry;
    size_t		len = 0;
    static char		*buffer = NULL;
    static size_t	buffer_size = 0, content_size = 0;
#define QDDB_BUFFER_SIZE (1024*1024)

    if (schema == NULL && qddb_stabilizing != 0) {
	/* flush */
	if (content_size > 0) {
	    Write(File, buffer, content_size);
	    content_size = 0;
	    buffer[0] = '\0';
	}
	return;
    }
    if (buffer == NULL) {
	buffer = Malloc(QDDB_BUFFER_SIZE);
	buffer_size = QDDB_BUFFER_SIZE;
	content_size = 0;
	buffer[0] = '\0';
    }
    len = strlen(*AttributePtr) + 1;
    AttributePtr++;
    while (*AttributePtr != NULL) {
	len += strlen(*AttributePtr) + 1;
    	AttributePtr++;
    }    
    while (len >= buffer_size) { /* increase buffer if necessary */
	buffer_size *= 2;
	buffer = Realloc(buffer, buffer_size);
    }
    if (qddb_stabilizing != 0) {
	if (len > (buffer_size - content_size)) {
	    Write(File, buffer, content_size);
	    content_size = 0;
	    buffer[0] = '\0';
	}
	AttributePtr = WhichEntry;
	while (*AttributePtr != NULL) {
	    strcat(buffer+content_size, *AttributePtr);
	    content_size += strlen(*AttributePtr);
	    strcat(buffer+content_size, "\n");
	    content_size++;
	    AttributePtr++;
	}
    } else { /* not stabilizing, don't buffer */
	AttributePtr = WhichEntry;
	while (*AttributePtr != NULL) {
	    strcat(buffer+content_size, *AttributePtr);
	    content_size += strlen(*AttributePtr);
	    strcat(buffer+content_size, "\n");
	    content_size++;
	    AttributePtr++;
	}
	Write(File, buffer, content_size);
	content_size = 0;
	buffer[0] = '\0';
    }
}

size_t GetEntryNumber(WhichEntry)
    Entry			WhichEntry;
{
    int				EntryNumber;

    sscanf(WhichEntry[0], "%%0 %*c %d\n", &EntryNumber);
    return (size_t)EntryNumber;
}

Boolean EntryInvalid(WhichEntry)
    Entry			WhichEntry;
{
    char			*Valid = WhichEntry[0];

    while (*Valid != 'I' && *Valid != 'V' && *Valid != '\0')
    	Valid++;
    if (*Valid == '\0') {
        fprintf(stderr, "%s\n", WhichEntry[0]);
        PANIC("EntryInvalid(): %0 entry doesn't contain 'V' or 'I'");
    }
    return *Valid == 'I'? True : False;
}

int EntryChanged(ChangeFN, Length)
    char		*ChangeFN; 
    size_t		*Length;
{
    int			ChangeFile;

    if ((ChangeFile=Open(ChangeFN, O_RDWR, 0)) == -1)
    	return -1;
    *Length = SizeOfFile(ChangeFile);
    return ChangeFile;
}

int yyerror(string)
    char		*string;
{
    extern int		LineNumber;
    char		buf[BUFSIZ], *ch;

    sprintf(buf, "syntax error at line %d\n", LineNumber);
    if (qddb_errmsg != NULL) {
	ch = qddb_errmsg;
    } else {
	ch = Calloc(1);
    }
    qddb_errmsg = Malloc(strlen(buf)+strlen(ch)+2);
    strcpy(qddb_errmsg, ch);
    strcat(qddb_errmsg, "\n");
    strcat(qddb_errmsg, buf);
    Free(ch);
    LineNumber = 1;
    return 1;
}

int StrcmpNoupper(string1, string2)
    char		*string1, *string2;
{
    int			i;

    if (strcmp(string1, string2) == 0)
        return 0;
    if (strlen(string1) != strlen(string2))
        return 1;
    for (i = 0; i < (int)strlen(string1); i++)
        if (tolower(string1[i]) != tolower(string2[i]))
            return 1;
    return 0;
}

#if defined(RECORDLOCKING)
int LockSection(File, LockType, Start, Length, Wait)
    int			File;
    int			LockType;
    off_t		Start, Length;
    Boolean		Wait;
{
    struct flock	Lock;

    Lock.l_type = LockType;
    Lock.l_whence = 0;
    Lock.l_start = Start;
    Lock.l_len = Length;
    if (Wait == True)
    	return fcntl(File, F_SETLKW, &Lock);
    else
    	return fcntl(File, F_SETLK, &Lock);
}

int UnlockSection(File, Start, Length)
    int			File;
    off_t		Start, Length;
{
    struct flock	Lock;

    Lock.l_type = F_UNLCK;
    Lock.l_whence = 0;
    Lock.l_start = Start;
    Lock.l_len = Length;
    return fcntl(File, F_SETLKW, &Lock);
}
#else
int LockSection(File, LockType, Start, Length, Wait)
    int			File;
    int			LockType;
    off_t		Start, Length;
    Boolean		Wait;
{
    return 0;
}

int UnlockSection(File, Start, Length)
    int			File;
    off_t		Start, Length;
{
    return 0;
}
#endif

Boolean Qddb_IsNumericString(string)
    char		*string;
{
    char			*endptr, *strend;
    double			retval;

    retval = (double)strtol(string, &endptr, 0);
    strend = string;
    while (*strend != '\0') {
	strend++;
    }
    if (endptr == strend) {
	return True;
    }
    retval = strtod(string, &endptr);
    if (endptr == strend) {
	return True;
    } else {
	return False;
    }
}


void SkipSpaces(string)
    char		**string;
{
    while (*string != NULL && isspace((**string)))
	(*string)++;
}

void Qddb_InvalidateEntry(schema, Start, Length)
    Schema		*schema;
    off_t		Start;
    size_t		Length;
{
    char		*RelationFN = schema->RelationName, DatabaseFN[MAXPATHLEN];
    int			DatabaseFile = schema->database_fd;
    char		*Buffer;
    int			i;

#if defined(RECORDLOCKING)
    LockSection(DatabaseFile, F_WRLCK, Start, (off_t)Length, True);
#endif
    Buffer = Malloc(Length);
    lseek(DatabaseFile, Start, 0);
    Read(DatabaseFile, Buffer, Length);
    for (i = 0; i < Length; i++)
	if (Buffer[i] == 'V') {
	    Buffer[i] = 'I';
	    break;
	} 
    lseek(DatabaseFile, Start, 0);
    Write(DatabaseFile, Buffer, Length);
    strcpy(DatabaseFN, RelationFN);
    strcat(DatabaseFN, "/Database.key");
#if defined(HAVE_UTIME)
#if defined(HAVE_UTIME_NULL)
    utime(DatabaseFN, NULL);
#endif
#endif
#if defined(RECORDLOCKING)
    UnlockSection(DatabaseFile, Start, (off_t)Length);
#endif
    Free(Buffer);
}



void Qddb_InvalidateEntryByNumber(schema, Number)
    Schema		*schema;
    size_t		Number;
{
    char		*RelationFN = schema->RelationName;
    char		DatabaseFN[MAXPATHLEN], *entry;
    FILE		*DatabaseKeyFile;
    int			i, base;
    off_t		Start;
    size_t		Length;
    char		buf[BUFSIZ], *ptr = NULL;

    strcpy(DatabaseFN, RelationFN);
    strcat(DatabaseFN, "/Database.key");
    DatabaseKeyFile = fopen(DatabaseFN, "r");
    i = 1;
    while ((entry=fgets(buf, BUFSIZ, DatabaseKeyFile)) != NULL && i != Number)
	i++;
    if (entry == NULL)
	return;	/* must be an addition */
    if (schema->UseCondensedIndexing == False) {
	base = 10;
    } else {
	base = 62;
    }
    ptr = buf;
    while (*ptr++ != ' ');
    *ptr = '\0';
    Start = Qddb_BaseToOffset(buf, base);
    Length = Qddb_BaseToSize(ptr+1, base);
    Qddb_InvalidateEntry(schema, Start, Length);
    fclose(DatabaseKeyFile);
}

void DeleteEntryByNumber(schema, Number)
    Schema		*schema;
    size_t		Number;
{
    char		DatabaseFN[MAXPATHLEN];
    char		*RelationFN = schema->RelationName;

    Qddb_UpdateWordList(schema, DELETION|CHANGE, (int)Number, NULL);
    sprintf(DatabaseFN, "%s/Changes/%d", RelationFN, (int)Number);
    unlink(DatabaseFN);
    Qddb_InvalidateEntryByNumber(schema, Number);
}

void DeleteEntryByKeyList(schema, node)
    Schema		*schema;
    KeyList		*node;
{
    char		*RelationFN = schema->RelationName;
    char		DatabaseFN[MAXPATHLEN];

    switch (QDDB_KEYLIST_TYPE(node)) {
    case ORIGINAL:
	Qddb_InvalidateEntry(schema, node->Start, node->Length);
	break;
    case CHANGE:
	DeleteEntryByNumber(schema, node->Number);
	break;
    case ADDITION:
	Qddb_UpdateWordList(schema, DELETION|ADDITION, (int)node->Number, NULL);
	sprintf(DatabaseFN, "%s/Additions/%d", RelationFN, (int)node->Number);
	unlink(DatabaseFN);
	break;
    default:
	;
    }
}

char **ParseKeyFile(fd)
    int			fd;
{
    struct stat		stat_buf;
    char		**retval, *ch, *buf;
    int			i, retval_top;

    if (fstat(fd, &stat_buf) == -1)
	return NULL;
    retval = (char **)Malloc(sizeof(char *)*(size_t)(stat_buf.st_size/4));
    buf = (char *)Malloc(sizeof(char)*(size_t)stat_buf.st_size);
    if (read(fd, buf, (size_t)stat_buf.st_size) != stat_buf.st_size)
	PANIC("ERROR: Cannot read key file\n");
    ch = buf;
    retval_top = 0;
    for (i = 0; i < stat_buf.st_size; i++) {
	if (*buf == '\n') {
	    *buf++ = '\0';
	    retval[retval_top++] = ch;
	    ch = buf;
	} else
	    buf++;
    }
    retval[retval_top] = NULL;
    return retval;
}

KeyList *CatRelation(schema, RelationFN)
    Schema              *schema;
    char		*RelationFN;
{
    char		KeyFN[BUFSIZ], **key_contents, **idx, buf[BUFSIZ];
    KeyList		*retval = NULL, *tmp, *last;
    int			fd;
    struct stat		stat_buf;
    off_t		start;
    size_t		len, number;
    struct dirent	*dir_ptr;
    DIR			*DIR_PTR;
    char		*ptr = NULL;
    int                 base;

    strcpy(KeyFN, RelationFN);
    strcat(KeyFN,"/Database.key");
    if ((fd = Open(KeyFN, O_RDONLY, 0)) == -1)
	return NULL;
    idx = key_contents = ParseKeyFile(fd);
    number = 1;
    last = NULL;
    if (schema->UseCondensedIndexing == False) {
	base = 10;
    } else {
	base = 62;
    }
    while (*idx != NULL) {
	ptr = *idx;
	while (*ptr++ != ' ');
	*ptr = '\0';
	start = Qddb_BaseToOffset(*idx, base);
	len = Qddb_BaseToSize(ptr+1, base);
	tmp = (KeyList *)Malloc(sizeof(KeyList));
	tmp->Start = start;
	tmp->Length = len;
	tmp->Number = number++;
	QDDB_KEYLIST_SET_TYPE(tmp, ORIGINAL);
	sprintf(buf, "%s/Changes/%d", RelationFN, (int)tmp->Number);
	if (stat(buf, &stat_buf) != -1) {
	    QDDB_KEYLIST_SET_TYPE(tmp, CHANGE);
	}
	tmp->next = NULL;
	if (last == NULL) {
	    last = tmp;
	    retval = last;
	} else {
	    last->next = tmp;
	    last = tmp;
	}
	idx++;
    }
    sprintf(KeyFN, "%s/Additions", RelationFN);
    DIR_PTR = opendir(KeyFN);
    while ((dir_ptr = readdir(DIR_PTR)) != NULL) {
	int			dirlen;

	/* nothing beginning with '.' is supposed to be in this directory except '.' & '..' */
	dirlen = NLENGTH(dir_ptr);
	if (dirlen != 0 && dir_ptr->d_name[0] != '.' && dir_ptr->d_name[0] != 'N') {
	    tmp = (KeyList *)Malloc(sizeof(KeyList));
	    tmp->Start = (off_t)0;
	    tmp->Length = (size_t)0;
	    tmp->Number = atoi(dir_ptr->d_name);
	    QDDB_KEYLIST_SET_TYPE(tmp, ADDITION);
	    tmp->next = NULL;
	    if (last == NULL) {
		last = tmp;
		retval = last;
	    } else {
		last->next = tmp;
		last = tmp;
	    }
	}	    
    }
    closedir(DIR_PTR);
    Free(*key_contents);
    Free(key_contents);
    close(fd);
    return retval;
}

char *InstanceToString(Inst, Len)
    size_t		Inst[];
    size_t		Len;
{
    char		retval[BUFSIZ];
    char		*ptr;
    int			i, j;

    retval[0] = '\0';
    j = 0;
    for (i = 0; i < Len; i++) {
	if (i == Len-1)
	    sprintf(retval+j, "%d", (int)Inst[i]);
	else
	    sprintf(retval+j, "%d.", (int)Inst[i]);
	j = strlen(retval);
    }
    ptr = Malloc((size_t)j+1);
    strcpy(ptr, retval);
    return ptr;
}

int StringToInstance(str, Inst)
    char		*str;
    size_t		Inst[];
{
    char		*ch;
    char		*buf = (char *)alloca(strlen(str)+1);
    int			i;

    strcpy(buf, str);
    ch = strtok(buf, ".");
    for (i = 0; ch != NULL; i++) {
	Inst[i] = atoi(ch);
	ch = strtok(NULL, ".");
    }
    return i;
}

char *IntegerToString(integer)
    int			integer;
{
    char		*retval, *ch;

    retval = (char *)Malloc((size_t)MAXINTEGERLEN);
    ch = FastIntegerToString(integer);
    strcpy(retval, ch);
    return retval;
}

char *FastIntegerToString(integer)
    int			integer;
{
    static char		retval[MAXINTEGERLEN];
    char		*forw, *back;
    int			neg;

    forw = retval;
    if (integer < 0) {
	*forw++ = '-';
	integer = -integer;
	neg = 1;
    } else {
	neg = 0;
    }
    while (integer > 0) {
	*forw++ = '0' + (integer % 10);
	integer /= 10;
    }
    *forw = '\0';
    back = forw - 1;
    forw = retval + neg;
    while (forw < back) {
	*forw ^= *back;
	*back ^= *forw;
	*forw++ ^= *back--;
    }
    return retval;
}

/* Rough date checking; don't actually check for a valid date */
int Qddb_CheckDate(s)
    char			*s;
{
    time_t			mytime, clockVal;
    long			zone;

    mytime = time(NULL);
    zone = Qddb_GetTimeZone(mytime);
    if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
	return 1;
    }
    return 0;
}

/* Accept dates in any form */
char *Qddb_DateStringToTime(s)
    char			*s;
{
    char			buf[BUFSIZ], *retval;
    time_t			mytime, clockVal;
    long			zone;

    if (strcmp(s, " ") == 0 || s[0] == '\0') {
	retval = Malloc(strlen(s)+1);
	strcpy(retval, s);
	return retval;
    }	
    mytime = time(NULL);
    zone = Qddb_GetTimeZone(mytime);
    if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
	retval = Malloc(2);
	strcpy(retval, "");
	return retval;
    }
    if (sizeof(time_t) == sizeof(int))
	sprintf(buf, "%d", (int)clockVal);
    else if (sizeof(time_t) == sizeof(long))
	sprintf(buf, "%ld", (long)clockVal);
    else
	PANIC("sizeof(time_t) != sizeof(int) or sizeof(long)");
    retval = Malloc(sizeof(char)*(strlen(buf)+1));
    strcpy(retval, buf);
    return retval;
}

char *Qddb_DateTimeToString(s, format)
    char			*s;
    char 			*format;
{
    char			buf[BUFSIZ], *retval;
    int                         numchars;
    time_t			mytime;
    struct tm			*tm_buf;

    if (strcmp(s, " ") == 0 || s[0] == '\0') {
	retval = Malloc(2);
	strcpy(retval, " ");
	return retval;
    }
    mytime = (time_t)strtoul(s, NULL, 10);
    if (strcmp(format, "%s") == 0) {
	sprintf(buf, "%d", (int)mytime);
	numchars = strlen(buf);
    } else {
	tm_buf = localtime(&mytime);
	numchars = strftime(buf, BUFSIZ, format, tm_buf);
    }
    retval = Malloc(sizeof(char)*(numchars+1));
    if (numchars > 0) {
	strcpy(retval, buf);
    } else {
	retval[0] = '\0';
    }
    return retval;
}

char *Qddb_FieldString(string, length)
    char			*string;
    size_t			length;
{
    size_t			str_len;
    int				i;
    char			*buf;

    str_len = strlen(string);
    buf = Malloc(length+1);
    if (length > str_len) {
	for (i = 0; i < length-str_len; i++)
	    buf[i] = ' ';
	strncpy(buf+i, string, str_len);
	buf[length] = '\0';
    } else {
	strncpy(buf, string, length);
	buf[length] = '\0';
    }
    return buf;
}

int Qddb_TranslateExternalID(schema, id, attr, inst)
    Schema		*schema;
    char		*id;
    size_t		*attr;
    char		**inst;
{
    char		*tok, inst_buf[BUFSIZ], attr_buf[BUFSIZ];
    int			tmpattr;
    size_t		inst_buflen, attr_buflen;

    if (*id == '%') {
	id++;
    } else {
	return -1;
    }
    inst_buf[0] = '\0';
    inst_buflen = 0;
    attr_buf[0] = '\0';
    attr_buflen = 0;
    tok = id;
    while (*id != '\0' && *id != '.')
	id++;
    if (*id != '\0') {
	*id = '\0';
	id++;
    }
    while (tok != NULL && *tok != '\0') {
	strcpy(attr_buf+attr_buflen, tok);
	attr_buflen += strlen(tok);
	attr_buf[attr_buflen++] = '.';
        attr_buf[attr_buflen] = '\0';
	tok = id;
	while (*id != '\0' && *id != '.')
	    id++;
	if (*id != '\0') {
	    *id = '\0';
	    id++;
	}
	strcpy(inst_buf+inst_buflen, tok);
	inst_buflen += strlen(tok);
	inst_buf[inst_buflen++] = '.';
        inst_buf[inst_buflen] = '\0';
	tok = id;
	while (*id != '\0' && *id != '.')
	    id++;
	if (*id != '\0') {
	    *id = '\0';
	    id++;
	}
    }
    attr_buflen--;
    attr_buf[attr_buflen] = '\0';
    if ((tmpattr = Qddb_FindAttributeNumberFromString(schema, attr_buf)) == -1)
	return -1;
    inst_buflen--;
    inst_buf[inst_buflen] = '\0';
    *inst = Malloc(inst_buflen+1);
    strcpy(*inst, inst_buf);
    *attr = tmpattr;
    return 0;
}


int Qddb_FindAttributeNumber(schema, Attributes, HowMany)
    Schema		*schema;
    size_t		Attributes[];
    int			HowMany;
{
    size_t		i, j;

    for (i = 1; i <= schema->NumberOfAttributes; i++) {
	for (j = 1; j < HowMany; j++) {
	    if (schema->Entries[i].AncestorNumber[j] != Attributes[j-1])
		break;
	}
	if (j == HowMany) {
	    if (schema->Entries[i].Level != HowMany)
		continue;
	    if (Attributes[j-1] == schema->Entries[i].Number)
		return i;
	}
    }
    return -1;
}

int Qddb_FindAttributeNumberFromString(schema, str)
    Schema			*schema;
    char			*str;
{
#if defined(USE_TCL)
    Tcl_HashEntry		*hash_entry;

    hash_entry = Tcl_FindHashEntry(&(schema->TclHashTable), str);
    if (hash_entry == NULL)
	return -1;
    return (int)Tcl_GetHashValue(hash_entry);
#else
    size_t			i;

    for (i = 1; i <= schema->NumberOfAttributes; i++) {
	if (strcmp(str, schema->Entries[i].Attribute) == 0) {
	    return i;
	}
    }
    return -1;
#endif
}

int Qddb_GetInt(buf)
    char			*buf;
{
    return (int)strtol(buf, NULL, 0);
}

double Qddb_GetNum(str)
    char			*str;
{
    char			*endptr, *strend;
    double			retval;

    strend = str;
    while (*strend != '\0') {
	strend++;
    }
    retval = (double)strtol(str, &endptr, 0);
    if (endptr == strend) {
	return retval;
    }
    retval = strtod(str, NULL);
    return retval;
}

char *Qddb_GetFile(fd)
    int				fd;
{
    char			*buf;
    size_t			bufsize, place, num_read;

    bufsize = BUFSIZ;
    place = 0;
    buf = Malloc(bufsize);
    while ((num_read = read(fd, buf+place, BUFSIZ)) > 0) {
	if (num_read >= BUFSIZ-1) {
	    bufsize += BUFSIZ;
	    buf = Realloc(buf, bufsize);
	}
	place += num_read;
    }
    buf[place] = '\0';
    buf = Realloc(buf, place+1);
    return buf;
}

static char   *qddb_setbuf_buffer;
static size_t qddb_setbuf_bufsiz;
static size_t qddb_setbuf_buflen;

void Qddb_SetBuffer(buf)
    char			*buf;
{
    qddb_setbuf_buffer = buf;
}

size_t Qddb_ReadBuffer(buf, n)
    char			*buf;
    size_t			n;
{
    int				i;

    for (i = 0; i < n-1 && *qddb_setbuf_buffer != '\0'; i++)
	*buf++ = *qddb_setbuf_buffer++;
    *buf = '\0';
    return i;
}

char *Qddb_GetBuffer _ANSI_ARGS_((void))
{
    qddb_setbuf_buffer = Realloc(qddb_setbuf_buffer, qddb_setbuf_buflen+1);
    qddb_setbuf_bufsiz = qddb_setbuf_buflen+1;
    return qddb_setbuf_buffer;
}

char *Qddb_GetBufferWithoutResize _ANSI_ARGS_((void))
{
    return qddb_setbuf_buffer;
}

size_t Qddb_GetBufferLength _ANSI_ARGS_((void))
{
    return qddb_setbuf_buflen;
}

void Qddb_SetBufferLength(len)
    size_t			len;
{
    qddb_setbuf_buffer[len] = '\0';
    qddb_setbuf_buflen = len;
}

void Qddb_SetBufferSize(len)
    size_t			len;
{
    qddb_setbuf_bufsiz = len;
}


void Qddb_InitBuffer _ANSI_ARGS_((void))
{
    qddb_setbuf_buffer = Malloc(BUFSIZ*sizeof(char));
    qddb_setbuf_buffer[0] = '\0';
    qddb_setbuf_bufsiz = BUFSIZ;
    qddb_setbuf_buflen = 0;
}

void Qddb_ConcatBuffer(newbuf)
    char			*newbuf;
{
    size_t			len = strlen(newbuf);

    if (qddb_setbuf_buflen+len >= qddb_setbuf_bufsiz) {
	qddb_setbuf_bufsiz += MAX(len+10, BUFSIZ);
	qddb_setbuf_buffer = Realloc(qddb_setbuf_buffer, qddb_setbuf_bufsiz);
    }
    strcpy(qddb_setbuf_buffer+qddb_setbuf_buflen, newbuf);
    qddb_setbuf_buflen += len;
}

void Qddb_ConcatPrintableBuffer(newbuf)
    char			*newbuf;
{
    size_t			len = strlen(newbuf);
    char			*ptr;

    if (qddb_setbuf_buflen+len >= qddb_setbuf_bufsiz) {
	qddb_setbuf_bufsiz += MAX(len+10, BUFSIZ);
	qddb_setbuf_buffer = Realloc(qddb_setbuf_buffer, qddb_setbuf_bufsiz);
    }
    ptr = qddb_setbuf_buffer+qddb_setbuf_buflen;
    while (*newbuf) {
	if (isprint(*newbuf)) {
	    *ptr++ = *newbuf;
	} else {
	    *ptr++ = ' ';
	}
	newbuf++;
    }
    *ptr = *newbuf;
    qddb_setbuf_buflen += len;
}

char *Qddb_PrintErrno(qerrno) 
    int				qerrno;
{
    switch (qerrno) {
    case QDDB_ERRNO_INVALID_ARG:
	return "Invalid argument";
    case QDDB_ERRNO_INVALID_REGEXP:
	return "Invalid regular expression";
    default:
	return "";
    }
}

char *Qddb_FindKey(string, len)
    char			*string;
    size_t			*len;
{
    int				state;
    size_t			nlen;

    state = 0;
    nlen = 0;
    for (; *string != '\0'; string++, nlen++) {
	if (*string == '\\') {
	    if (state == 1)
		state = 0;
	    else
		state = 1;
	} else if (state == 1) {
	    state = 0;
	} else if (*string == ' ') {
	    break;
	}
    }
    *len = nlen;
    return string;
}

void Qddb_StripBackslashes(buf)
    char		*buf;
{
    char		*tmp, *tmpbuf;

    tmp = tmpbuf = buf;
    while (*tmp != '\0') {
	if (*tmp == '\\' && *(tmp+1) == '\n') {
	    tmp++;
	} else if (*tmp == '\\' && (*(tmp+1) == '\\' || *(tmp+1) == ' ')) {
	    tmp++;
	    *tmpbuf++ = *tmp++;
	} else {
	    *tmpbuf++ = *tmp++;
	}
    }
    *tmpbuf = '\0';
}	    

char *Qddb_StripKey(buf, len)
    char			*buf;
    size_t			len;
{
    char		*tmp, *tmpbuf, *retval;

    tmp = buf;
    retval = tmpbuf = Malloc(len+1);
    while (*tmp != '\0') {
	if (*tmp == '\\' && *(tmp+1) == '\n') {
	    tmp++;
	} else if (*tmp == '\\' && (*(tmp+1) == '\\' || *(tmp+1) == ' ')) {
	    tmp++;
	    *tmpbuf++ = *tmp++;
	} else {
	    *tmpbuf++ = *tmp++;
	}
    }
    *tmpbuf = '\0';
    return retval;
}

void Qddb_FillChars(buf, ch, len)
    char			*buf;
    int				ch, len;
{
    int				i;

    for (i = 0; i < len; i++) {
	buf[i] = (char)ch;
    }
}

static char			*alphabet = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
static int			indices[128] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 0, 0, 0, 0, 0,
    0, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50,
    51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 0, 0, 0, 0, 0,
    0, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,
    25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 0, 0, 0, 0, 0
};

char *Qddb_SizeToBase(num, base)
    size_t                      num;
    int                         base;
{
    static char                 retval[sizeof(off_t)*3], ch[sizeof(off_t)*3];
    char                        *ch_ptr, *retval_ptr;
    int                         i;

    i = 0;
    if (num == 0) {
	retval[0] = '0';
	retval[1] = '\0';
	return retval;
    }
    ch_ptr = ch;
    while (num > 0) {
	*ch_ptr++ = alphabet[num % base];
	num  /= base;
	i++;
    }
    *ch_ptr = '\0';
    retval_ptr = retval + i;
    *retval_ptr-- = '\0';
    ch_ptr = ch;
    for (ch_ptr = ch; *ch_ptr != '\0'; ch_ptr++) {
	*retval_ptr-- = *ch_ptr;
    }
    return retval;
}


size_t Qddb_BaseToSize(ch, base)
    char                        *ch;
    int                         base;
{
    size_t                      retval;
    int				digit;

    for (retval = 0; *ch != '\0' ; ch++) {
	digit = indices[(int)(*ch)];
	retval *= base;
	retval += digit;
    }
    return retval;
}

char *Qddb_OffsetToBase(num, base)
    off_t                       num;
    int                         base;
{
    static char                 retval[sizeof(off_t)*3], ch[sizeof(off_t)*3];
    char                        *ch_ptr, *retval_ptr;
    int                         i;

    i = 0;
    if (num == 0) {
	retval[0] = '0';
	retval[1] = '\0';
	return retval;
    }
    ch_ptr = ch;
    while (num > 0) {
	*ch_ptr++ = alphabet[num % base];
	num  /= base;
	i++;
    }
    *ch_ptr = '\0';
    retval_ptr = retval + i;
    *retval_ptr-- = '\0';
    ch_ptr = ch;
    for (ch_ptr = ch; *ch_ptr != '\0'; ch_ptr++) {
	*retval_ptr-- = *ch_ptr;
    }
    return retval;
}

off_t Qddb_BaseToOffset(ch, base)
    char                        *ch;
    int                         base;
{
    off_t                      retval;
    int                        digit;

    for (retval = 0; *ch != '\0'; ch++) {
	digit = indices[(int)(*ch)];
	retval *= base;
	retval += digit;
    }
    return retval;
}

Entry Qddb_ReadExcludeWords(schema, num)
    Schema                     *schema;
    size_t                     *num;
{
    size_t                     nmemb;
    Entry                      idx, ExcludeEntry;
    int                        ExcludeFile;
    char                       ExcludeFN[MAXPATHLEN];

    if (schema->UseExcludeWords == False)
	return NULL;
    *num = 0;
    ExcludeEntry = NULL;
    strcpy(ExcludeFN, schema->RelationName);
    strcat(ExcludeFN, "/ExcludeWords");
    if ((ExcludeFile = Open(ExcludeFN, O_RDONLY, 0)) != -1) {
	Qddb_ReadEntry(ExcludeFile, &ExcludeEntry, (off_t)0, (size_t)0, True);
	nmemb = 0;
	idx = ExcludeEntry;
	while (*idx != NULL) {
	    char      *ch;

	    ch = *idx;
	    while (*ch != '\0') {
		if (isupper(*ch))
		    *ch = tolower(*ch);
		ch++;
	    }
	    idx++;
	    nmemb++;
	}
	Close(ExcludeFile);
	*num = nmemb;
	return ExcludeEntry;
    }
    return NULL;
}


/* Cache routines */
#if defined(USE_TCL)
typedef struct centry {
    Entry			e;
    time_t			tm;
} qddb_cache_entry;

static Tcl_HashTable 		qddb_cache;
static int 			qddb_cache_init = 0;

static qddb_cache_entry *Qddb_AddEntryToCache(fn)
    char			*fn;
{
    qddb_cache_entry		*c;
    int				FileDesc;
    struct stat			stat_buf;
    Tcl_HashEntry		*hash_entry;
    int				newPtr;

    c = (qddb_cache_entry *)Malloc(sizeof(qddb_cache_entry));
    c->e = NULL;
    /* open it up, lock it down */
    FileDesc = open(fn, O_RDONLY);
    LockSection(FileDesc, F_RDLCK, (off_t)0, (off_t)0, True);
    Qddb_ReadEntry(FileDesc, &(c->e), (off_t)0, (size_t)0, False);
    if (fstat(FileDesc, &stat_buf) != 0) {
	PANIC("Qddb_GetEntryFromCache: cannot stat file\n");
    }
    c->tm = stat_buf.st_mtime;
    UnlockSection(FileDesc, (off_t)0, (off_t)0);
    close(FileDesc);
    if (qddb_cache_init == 0) {
	qddb_cache_init = 1;
	Tcl_InitHashTable(&qddb_cache, TCL_STRING_KEYS);
    }
    hash_entry = Tcl_CreateHashEntry(&qddb_cache, fn, &newPtr);
    Tcl_SetHashValue(hash_entry, (ClientData)c);
    return c;
}

static qddb_cache_entry *Qddb_LookupEntryInCache(fn)
    char			*fn;
{
    Tcl_HashEntry		*hash_entry;

    if (qddb_cache_init != 0) {
	hash_entry = Tcl_FindHashEntry(&qddb_cache, fn);
	if (hash_entry == NULL)
	    return NULL;
	return (qddb_cache_entry *)Tcl_GetHashValue(hash_entry);
    }
    return NULL;
}

Entry Qddb_GetEntryFromCache(fn)
    char			*fn;
{
    struct stat			stat_buf;
    char			buf[BUFSIZ];
    qddb_cache_entry		*cache;
    int				FileDesc;

    if (stat(fn, &stat_buf) != 0) {
	sprintf(buf, "%s: %s", "Cannot stat file: ", fn);
	PANIC(buf);
    }
    /* Check cache for entry:
     *	1. If it doesn't, exist, add and return it.
     *  2. If it does exist, check the time.
     */
    if ((cache = Qddb_LookupEntryInCache(fn)) == NULL) {
	/* Doesn't exist, add it */
	cache = Qddb_AddEntryToCache(fn);
    } else {
	if (cache->tm < stat_buf.st_mtime) { /* cache entry is old */
	    /* open it up, lock it down */
	    FileDesc = open(fn, O_RDONLY);
	    LockSection(FileDesc, F_RDLCK, (off_t)0, (off_t)0, True);
	    Qddb_ReadEntry(FileDesc, &(cache->e), (off_t)0, (size_t)0, False);
	    if (fstat(FileDesc, &stat_buf) != 0) {
		PANIC("Qddb_GetEntryFromCache: cannot stat requested file\n");
	    }
	    cache->tm = stat_buf.st_mtime;
	    UnlockSection(FileDesc, (off_t)0, (off_t)0);
	    close(FileDesc);
	}
    }
    return cache->e;
}
#endif
