
/* Tree.c - DataTree functions.
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	void *Qddb_DataTreeProcess(Schema *, DataTree **, void *, int, int);
 *
 *
 *
 */
static DataTree **DataTreeNewInstance _ANSI_ARGS_((SchemaTreeNode **));
static void AddNodeToDataTree _ANSI_ARGS_((Schema *, DataTree **, InCoreEntry *));
static void DataTreeSetSequenceNumber _ANSI_ARGS_((DataTree **, int));
static void DataTreeSetMark _ANSI_ARGS_((DataTree **, int));
static void DataTreeSetAttr _ANSI_ARGS_((DataTree **, char *, char *));
static char *TreeDataToString _ANSI_ARGS_((DataTree *));

void *Qddb_DataTreeProcess(schema, tree, args, op, flags)
    Schema			*schema;
    DataTree			**tree;
    void			*args;
    int				op;
    int				flags;
{
    void			*retval = NULL;

    switch (op) {
    case QDDB_DATATREE_PROC_NEWINSTANCE:
	retval = DataTreeNewInstance((SchemaTreeNode **)args);
	break;
    case QDDB_DATATREE_PROC_ADDNODE:
	AddNodeToDataTree(schema, tree, (InCoreEntry *)args);
	break;
    case QDDB_DATATREE_PROC_SETMARK:
	DataTreeSetMark(tree, *(int *)args);
	break;
    case QDDB_DATATREE_PROC_SETATTR:
	DataTreeSetAttr(tree, "", (char *)args);
	break;
    case QDDB_DATATREE_PROC_SETSEQ:
	DataTreeSetSequenceNumber(tree, *(int *)args);
	break;
    case QDDB_DATATREE_PROC_GETDATA:
	retval = TreeDataToString((DataTree *)args);
	break;
    default:
	;
    }
    return retval;
}

static void AddNodeToDataTree(schema, tree, entry)
    Schema			*schema;
    DataTree			**tree;
    InCoreEntry			*entry;
{
    SchemaNode			*schema_entry;
    DataTree			**where, *whereinst, **placeinst;
    size_t			maxinst;
    int				i, j, level;
    
    /* Assume that the entry is valid and follow the attribute down the DataTree
     * until a point is reached where a new instance is required.   We assume
     * that at least the first instance has been built.
     */
    schema_entry = schema->Entries+entry->AttributeNumber;
    level = schema_entry->Level;
    where = tree;
    for (i = 1; i <= level; i++) {
#if defined(DIAGNOSTIC)
	if (where == NULL) {
	    fprintf(stderr, "AddNodeToDataTree: premature where == NULL\n");
	    fflush(stderr);
	}
#endif
	if (i == level) {
#if defined(DIAGNOSTIC)
	    for (j = 0; where[j] != NULL; j++);
	    if (schema_entry->Number-1 > j) {
		fprintf(stderr, "1. Uh, there aren't enough pieces here\n");
		fflush(stderr);
	    }
#endif
	    placeinst = where+schema_entry->Number-1;
	    whereinst = *placeinst;
	} else {
#if defined(DIAGNOSTIC)
	    for (j = 0; where[j] != NULL; j++);
	    if (schema_entry->AncestorNumber[i]-1 > j) {
		fprintf(stderr, "2. Uh, there aren't enough pieces here\n");
		fflush(stderr);
	    }
#endif
	    placeinst = where+schema_entry->AncestorNumber[i]-1;
	    whereinst = *placeinst;
	}
	for (maxinst = 0; whereinst[maxinst].datatree_type != DATATREE_END; maxinst++);
	if (entry->Instance[i-1] > maxinst) {
	    /* reallocate instance */
	    if (entry->Instance[i-1] > maxinst+1) {
		int		newmax;
		char		*default_value;

		newmax = entry->Instance[i-1] - 1;
		whereinst = (DataTree *)Realloc(whereinst, sizeof(DataTree)*(newmax+2));
		if (whereinst[0].datatree_schema->schemanode->IsLeaf == True) {
		    for (j = maxinst; j < newmax; j++) {
			whereinst[j].datatree_schema = whereinst[0].datatree_schema;
			default_value = whereinst[j].datatree_schema->schemanode->DefaultValue;
			switch (whereinst[j].datatree_schema->schemanode->Type) {
			case Real:
			    if (default_value[0] == '\0') {
				whereinst[j].modified = False;
			    } else {
				whereinst[j].datatree_real = 
				    strtod(whereinst[j].datatree_schema->schemanode->DefaultValue, NULL);
				whereinst[j].modified = True;
			    }
			    whereinst[j].datatree_type = DATATREE_REAL;
			    break;
			case Integer:
			    if (default_value[0] == '\0') {
				whereinst[j].modified = False;
			    } else {
				whereinst[j].datatree_int = 
				    strtol(whereinst[j].datatree_schema->schemanode->DefaultValue, NULL, 0);
				whereinst[j].modified = True;
			    }
			    whereinst[j].datatree_type = DATATREE_INT;
			    break;
			case Date:
			    whereinst[j].datatree_date = Malloc(strlen(default_value)+1);
			    strcpy(whereinst[j].datatree_date, default_value);
			    whereinst[j].modified = True;
			    whereinst[j].datatree_type = DATATREE_DATE;
			    break;
			case String:
			default:
			    whereinst[j].datatree_string = Malloc(strlen(default_value)+1);
			    strcpy(whereinst[j].datatree_string, default_value);
			    whereinst[j].modified = True;
			    whereinst[j].datatree_type = DATATREE_STRING;
			    break;
			}
			whereinst[j].marked = Inapplicable;
		    }
		} else {
		    for (j = maxinst; j < newmax; j++) {
			whereinst[j].datatree_type = DATATREE_CHILDREN;
			whereinst[j].datatree_schema = whereinst[0].datatree_schema;
			whereinst[j].datatree_children =
			    DataTreeNewInstance(whereinst[0].datatree_schema->children);
		    }
		}
		maxinst = entry->Instance[i-1] - 1;
	    } else {
		whereinst = (DataTree *)Realloc(whereinst, sizeof(DataTree)*(maxinst+2));
	    }
	    *placeinst = whereinst;
	    whereinst[maxinst].datatree_schema = whereinst[0].datatree_schema;
	    if (whereinst[0].datatree_schema->schemanode->IsLeaf == True) {
#if defined(DIAGNOSTIC)
		if (whereinst[maxinst].datatree_schema->schemanode->IsLeaf == False)
		    PANIC("*** ISLEAF IS FALSE ***");
		if (i != level)
		    PANIC("*** BAD LEVEL, ISLEAF IS TRUE ***");
#endif
		switch (whereinst[maxinst].datatree_schema->schemanode->Type) {
		case Real:
		    whereinst[maxinst].datatree_type = DATATREE_REAL;
		    if (entry->Data[0] == '\0') {
			whereinst[maxinst].datatree_real = 0.0;
			whereinst[maxinst].modified = False;
		    } else {
			whereinst[maxinst].datatree_real = strtod(entry->Data, NULL);
			whereinst[maxinst].modified = True;
		    }
		    break;
		case Integer:
		    whereinst[maxinst].datatree_type = DATATREE_INT;
		    if (entry->Data[0] == '\0') {
			whereinst[maxinst].datatree_int = 0;
			whereinst[maxinst].modified = False;
		    } else {
			whereinst[maxinst].datatree_int = Qddb_GetInt(entry->Data);
			whereinst[maxinst].modified = True;
		    }
		    break;
		case Date:
		    whereinst[maxinst].datatree_type = DATATREE_DATE;
		    whereinst[maxinst].datatree_date = Malloc(strlen(entry->Data)+1);
		    strcpy(whereinst[maxinst].datatree_date, entry->Data);
		    whereinst[maxinst].modified = True;
		    break;
		case String:
		default:
		    whereinst[maxinst].datatree_type = DATATREE_STRING;
		    whereinst[maxinst].datatree_string = Malloc(strlen(entry->Data)+1);
		    strcpy(whereinst[maxinst].datatree_string, entry->Data);
		    whereinst[maxinst].modified = True;
		    break;
		}
		whereinst[maxinst].marked = entry->Marked;
	    } else {
		whereinst[maxinst].datatree_type = DATATREE_CHILDREN;
	        whereinst[maxinst].datatree_children =
		    DataTreeNewInstance(whereinst[maxinst].datatree_schema->children);
	    }
	    whereinst[maxinst+1].datatree_type = DATATREE_END;
	    if (whereinst[maxinst].datatree_type == DATATREE_CHILDREN)
		where = whereinst[maxinst].datatree_children;
	    else
		where = NULL;
	} else {
	    int				myinst = entry->Instance[i-1] - 1;

	    /* Use an existing instance */
	    if ((whereinst[myinst].datatree_type != DATATREE_CHILDREN) && i == level) {
		myinst = entry->Instance[i-1]-1;
		if (myinst < 0)
		    myinst = 0;
		switch (whereinst[myinst].datatree_schema->schemanode->Type) {
		case Real:
		    if (entry->Data[0] == '\0') {
			whereinst[myinst].datatree_real = 0.00;
			whereinst[myinst].modified = False;
		    } else {
			whereinst[myinst].datatree_real = strtod(entry->Data, NULL);
			whereinst[myinst].modified = True;
		    }
		    whereinst[myinst].datatree_type = DATATREE_REAL;
		    break;
		case Integer:
		    if (entry->Data[0] == '\0') {
			whereinst[myinst].datatree_int = 0;
			whereinst[myinst].modified = False;
		    } else {
			whereinst[myinst].datatree_int = Qddb_GetInt(entry->Data);
			whereinst[myinst].modified = True;
		    }
		    whereinst[myinst].datatree_type = DATATREE_INT;
		    break;
		case Date:
		    whereinst[myinst].datatree_type = DATATREE_DATE;
		    if (whereinst[myinst].datatree_date != NULL)
			Free(whereinst[myinst].datatree_date);
		    whereinst[myinst].datatree_date = Malloc(strlen(entry->Data)+1);
		    strcpy(whereinst[myinst].datatree_date, entry->Data);
		    whereinst[myinst].modified = True;
		    break;
		case String:
		default:
		    whereinst[myinst].datatree_type = DATATREE_STRING;
		    if (whereinst[myinst].datatree_string != NULL)
			Free(whereinst[myinst].datatree_string);
		    whereinst[myinst].datatree_string = Malloc(strlen(entry->Data)+1);
		    strcpy(whereinst[myinst].datatree_string, entry->Data);
		    whereinst[myinst].modified = True;
		    break;
		}
		whereinst[myinst].marked = entry->Marked;
		where = NULL;
	    } else {
		if (whereinst[myinst].datatree_type == DATATREE_CHILDREN)
		    where = whereinst[myinst].datatree_children;
		else
		    abort(); /* where = NULL; */
	    }
	}
    }
}

static DataTree **DataTreeNewInstance(schema_tree)
    SchemaTreeNode		**schema_tree;
{
    DataTree			**retval, **children;
    int				i, len;
    char			*default_value;

    if (schema_tree == NULL || *schema_tree == NULL) {
	return NULL;
    }
    for (len = 0; schema_tree[len] != NULL; len++);
    len++;
    retval = (DataTree **)Calloc(sizeof(DataTree *)*(len+1));
    for (i = 0; schema_tree[i] != NULL; i++) {
#if defined(DIAGNOSTIC)
	if (i > len)
	    abort();
#endif
	retval[i] = (DataTree *)Calloc(sizeof(DataTree)*2);	/* new instances have two nodes */
	retval[i][0].datatree_schema = schema_tree[i];
	children = DataTreeNewInstance(schema_tree[i]->children);
	retval[i][0].datatree_sequence_number = -1;
	if (children != NULL) {
	    retval[i][0].datatree_type = DATATREE_CHILDREN;
	    retval[i][0].datatree_children = children;
	} else {
	    default_value = schema_tree[i]->schemanode->DefaultValue;
	    switch (schema_tree[i]->schemanode->Type) {
	    case Real:
		retval[i][0].datatree_type = DATATREE_REAL;
		if (default_value[0] == '\0') {
		    retval[i][0].modified = False;
		    retval[i][0].datatree_real = 0.0;
		} else {
		    retval[i][0].datatree_real = strtod(default_value, NULL);
		    retval[i][0].modified = True;
		}
		break;
	    case Integer:
		retval[i][0].datatree_type = DATATREE_INT;
		retval[i][0].datatree_int = 0;
		if (default_value[0] == '\0') {
		    retval[i][0].modified = False;
		    retval[i][0].datatree_int = 0;
		} else {
		    retval[i][0].datatree_int = strtol(default_value, NULL, 0);
		    retval[i][0].modified = True;
		}
		break;
	    case Date:
		retval[i][0].datatree_type = DATATREE_DATE;
		if (default_value[0] == '\0') {
		    retval[i][0].datatree_date = Calloc(1);
		} else {
		    retval[i][0].datatree_date = Malloc(strlen(default_value)+1);
		    strcpy(retval[i][0].datatree_date, default_value);
		}
		retval[i][0].modified = True;
		break;
	    case String:
	    default:
		retval[i][0].datatree_type = DATATREE_STRING;
		if (default_value[0] == '\0') {
		    retval[i][0].datatree_date = Calloc(1);
		} else {
		    retval[i][0].datatree_string = Malloc(strlen(default_value)+1);
		    strcpy(retval[i][0].datatree_string, default_value);
		}
		retval[i][0].modified = True;
		break;
	    }
	    retval[i][0].marked = Inapplicable;
	}
	retval[i][1].datatree_type = DATATREE_END;
    }
    return retval;
}

static void DataTreeSetSequenceNumber(tree, seq)
    DataTree			**tree;
    int				seq;
{
    int				i, j;

    for (i = 0; tree[i] != NULL; i++) {
	for (j = 0; tree[i][j].datatree_type != DATATREE_END; j++) {
	    tree[i][j].datatree_sequence_number = seq;
	    if (tree[i]->datatree_schema->schemanode->IsLeaf == False) {
		DataTreeSetSequenceNumber(tree[i][j].datatree_children, seq);
	    }
	}
    }
}

static void DataTreeSetMark(tree, mark)
    DataTree			**tree;
    int				mark;
{
    int				i, j;

    for (i = 0; tree[i] != NULL; i++) {
	for (j = 0; tree[i][j].datatree_type != DATATREE_END; j++) {
	    tree[i][j].marked = mark;
	    if (tree[i]->datatree_schema->schemanode->IsLeaf == False) {
		DataTreeSetMark(tree[i][j].datatree_children, mark);
	    }
	}
    }
}

static void DataTreeSetAttr(tree, path, attr)
    DataTree			**tree;
    char			*path, *attr;
{
    int				i, j, isleaf;
    char			newpath[BUFSIZ];


    for (i = 0; tree[i] != NULL; i++) { /* attributes */
	isleaf = tree[i]->datatree_schema->schemanode->IsLeaf;
	strcpy(newpath, path);
	if (path[0] != '\0')
	    strcat(newpath, ".");
	strcat(newpath, tree[i]->datatree_schema->schemanode->Name);
	for (j = 0; tree[i][j].datatree_type != DATATREE_END; j++) { /* instances */
	    if (isleaf == False) {
		DataTreeSetAttr(tree[i][j].datatree_children, newpath, attr);
	    } else if (strcmp(newpath, attr) == 0) {
		if (tree[i][j].marked != True)
		    tree[i][j].marked = False;
	    }
	}
    }
}


static char *TreeDataToString(tree_entry)
    DataTree			*tree_entry;
{
    char			*retval;
    char			buf[BUFSIZ];

    switch(tree_entry->datatree_type) {
    case DATATREE_STRING:
	retval = Malloc(strlen(tree_entry->datatree_string)+1);
	strcpy(retval, tree_entry->datatree_string);
	break;
    case DATATREE_DATE:
	retval = Malloc(strlen(tree_entry->datatree_date)+1);
	strcpy(retval, tree_entry->datatree_date);
	break;
    case DATATREE_INT:
	if (tree_entry->modified == False) {
	    retval = Calloc(1);
	} else {
	    sprintf(buf, tree_entry->datatree_schema->schemanode->Format, tree_entry->datatree_int);
	    retval = Malloc(strlen(buf)+1);
	    strcpy(retval, buf);
	}
	break;
    case DATATREE_REAL:
	if (tree_entry->modified == False) {
	    retval = Calloc(1);
	} else {
	    sprintf(buf, tree_entry->datatree_schema->schemanode->Format, tree_entry->datatree_real);
	    retval = Malloc(strlen(buf)+1);
	    strcpy(retval, buf);
	}
	break;
    case DATATREE_NOVAL:
    default:
	return NULL;
    }
    return retval;
}
