
/* Table.c - Table functions
 *
 * Copyright (C) 1996 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	void *Qddb_Table(int op, void *arg);
 *      Qddb_TableCalc *Qddb_TableCalcParse(char *string);
 *
 */

static Qddb_Table *TableDefine _ANSI_ARGS_((size_t, size_t));
static Qddb_TableHeaderInfo *CopyHeaderInfo _ANSI_ARGS_((Qddb_TableHeaderInfo *));
static Qddb_Table *TableCopy _ANSI_ARGS_((Qddb_Table *));
static Qddb_Table *TableCopyRow _ANSI_ARGS_((Qddb_Table *, size_t, size_t));
static Qddb_Table *TableCopyCol _ANSI_ARGS_((Qddb_Table *, size_t, size_t));
static Qddb_Table *TableCopyCell _ANSI_ARGS_((Qddb_Table *, size_t, size_t, size_t, size_t));
static Qddb_Table *TableRemove _ANSI_ARGS_((Qddb_Table *));
static Qddb_Table *TableRemoveRow _ANSI_ARGS_((Qddb_Table *, size_t));
static Qddb_Table *TableRemoveCol _ANSI_ARGS_((Qddb_Table *, size_t));
static Qddb_Table *TableInsertRow _ANSI_ARGS_((Qddb_Table *, size_t));
static Qddb_Table *TableInsertCol _ANSI_ARGS_((Qddb_Table *, size_t));
static Qddb_TableAttrs *TableGetAttrs _ANSI_ARGS_((Qddb_Table *, size_t, size_t));
static Qddb_Table *TableSetAttrs _ANSI_ARGS_((Qddb_Table *, size_t, size_t, Qddb_TableAttrs *));
static Qddb_Table *TableSort _ANSI_ARGS_((Qddb_Table *, Qddb_TableOpSort *));
static Qddb_Table *TableAppend _ANSI_ARGS_((Qddb_Table *, Qddb_Table *));
static Qddb_Table *TableSummary _ANSI_ARGS_((Qddb_Table *, char *, char *, char *, char *, char*));

static void ClearCell _ANSI_ARGS_((Qddb_Table *, size_t, size_t));
static void ClearRow _ANSI_ARGS_((Qddb_Table *, size_t));
static void ClearColumn _ANSI_ARGS_((Qddb_Table *, size_t));
static void ClearTable _ANSI_ARGS_((Qddb_Table *));
static void ClearHeaderInfo _ANSI_ARGS_((Qddb_TableHeaderInfo *));
static size_t TableFindRow _ANSI_ARGS_((Qddb_Table *, char *));
static size_t TableFindCol _ANSI_ARGS_((Qddb_Table *, char *));
static Qddb_TableRowCol *TableFindCell _ANSI_ARGS_((Qddb_Table *, char *, char *));
static int TableCellType _ANSI_ARGS_((Qddb_Table *, size_t, size_t));

void *Qddb_TableOp(table, op, arg)
    Qddb_Table		*table;
    int			op;
    void		*arg;
{
    switch (op) {
    case QDDB_TABLE_OP_DEFINE: {
	Qddb_TableRowCol		*define_arg = (Qddb_TableRowCol *)arg;

	return (void *)TableDefine(define_arg->row, define_arg->column);
    }
    case QDDB_TABLE_OP_SUMMARY: {
	char                            **argv = (char **)arg;

	return (void *)TableSummary(table, argv[0], argv[1], argv[2], argv[3], argv[4]);
    }
    case QDDB_TABLE_OP_COPY:
	return (void *)TableCopy(table);
    case QDDB_TABLE_OP_COPYROW: {
	Qddb_TableFromTo		*copy_arg = (Qddb_TableFromTo *)arg;

	return (void *)TableCopyRow(table, copy_arg->from, copy_arg->to);
    }
    case QDDB_TABLE_OP_COPYCOL: {
	Qddb_TableFromTo		*copy_arg = (Qddb_TableFromTo *)arg;

	return (void *)TableCopyCol(table, copy_arg->from, copy_arg->to);
    }
    case QDDB_TABLE_OP_COPYCELL: {
	Qddb_TableRowCol		*copy_arg = (Qddb_TableRowCol *)arg;

	return (void *)TableCopyCell(table, copy_arg[0].row, copy_arg[0].column,
				     copy_arg[1].row, copy_arg[1].column);
    }
    case QDDB_TABLE_OP_REMOVE: {
	TableRemove(table);
	return (void *)NULL;
    }
    case QDDB_TABLE_OP_REMOVEROW: {
	return (void *)TableRemoveRow(table, *(size_t *)arg);
    }
    case QDDB_TABLE_OP_REMOVECOL: {
	return (void *)TableRemoveCol(table, *(size_t *)arg);
    }
    case QDDB_TABLE_OP_INSERTROW: {
	return (void *)TableInsertRow(table, *(size_t *)arg);
    }
    case QDDB_TABLE_OP_INSERTCOL: {
	return (void *)TableInsertCol(table, *(size_t *)arg);
    }
    case QDDB_TABLE_OP_GETROWATTRS: {
	return (void *)TableGetAttrs(table, *(size_t *)arg, (size_t)0);
    }
    case QDDB_TABLE_OP_GETCOLATTRS: {
	return (void *)TableGetAttrs(table, (size_t)0, *(size_t *)arg);
    }
    case QDDB_TABLE_OP_GETCELLATTRS: {
	Qddb_TableRowCol		*attr_arg = (Qddb_TableRowCol *)arg;

	return (void *)TableGetAttrs(table, attr_arg->row, attr_arg->column);
    }
    case QDDB_TABLE_OP_SETROWATTRS: {
	Qddb_TableSetAttr		*attr_arg = (Qddb_TableSetAttr *)arg;

	return (void *)TableSetAttrs(table, attr_arg->row, attr_arg->column, attr_arg->attrs);
    }
    case QDDB_TABLE_OP_SETCOLATTRS: {
	Qddb_TableSetAttr		*attr_arg = (Qddb_TableSetAttr *)arg;

	return (void *)TableSetAttrs(table, attr_arg->row, attr_arg->column, attr_arg->attrs);
    }
    case QDDB_TABLE_OP_SETCELLATTRS: {
	Qddb_TableSetAttr		*attr_arg = (Qddb_TableSetAttr *)arg;

	return (void *)TableSetAttrs(table, attr_arg->row, attr_arg->column, attr_arg->attrs);
    }
    case QDDB_TABLE_OP_SORT: {
	Qddb_TableOpSort		*attr_arg = (Qddb_TableOpSort *)arg;

	return (void *)TableSort(table, attr_arg);
    }
    case QDDB_TABLE_OP_CLEAR: {
	ClearTable(table);
	return (void *)NULL;
    }
    case QDDB_TABLE_OP_CLEARROW: {
	ClearRow(table, *(size_t *)arg);
	return (void *)NULL;
    }
    case QDDB_TABLE_OP_CLEARCOL: {
	ClearColumn(table, *(size_t *)arg);
	return (void *)NULL;
    }
    case QDDB_TABLE_OP_CLEARCELL: {
	Qddb_TableRowCol		*conf_arg = (Qddb_TableRowCol *)arg;
	
	ClearCell(table, conf_arg->row, conf_arg->column);
	return (void *)NULL;
    }
    case QDDB_TABLE_OP_FINDROW: {
	size_t		*ptr;

	ptr = (size_t *)Malloc(sizeof(size_t));
	*ptr = TableFindRow(table, (char *)arg);
	return (void *)ptr;
    }
    case QDDB_TABLE_OP_FINDCOL: {
	size_t		*ptr;

	ptr = (size_t *)Malloc(sizeof(size_t));
	*ptr = TableFindCol(table, (char *)arg);
	return (void *)ptr;
    }
    case QDDB_TABLE_OP_FINDCELL: {
	Qddb_TableRowColNames		*conf_arg = (Qddb_TableRowColNames *)arg;

	return (void *)TableFindCell(table, conf_arg->row, conf_arg->column);
    }
    case QDDB_TABLE_OP_APPEND: {
	Qddb_Table                      *table2 = (Qddb_Table *)arg;

	return (void *)TableAppend(table, table2);
    }
    case QDDB_TABLE_OP_JOIN: {
    }
    default:
	;
    }
    return (void *)NULL;
}

/* TableDefine - initially build a table of specified size.
 */

static Qddb_Table *TableDefine(rows, columns)
    size_t		rows, columns;
{
    Qddb_Table		*table;
    size_t		i;

    table = (Qddb_Table *)Malloc(sizeof(Qddb_Table));
    table->info = NULL;
    table->order = QDDB_TABLE_ORDER_ROW; /* default is manual row */
    table->autoeval = False;
    table->rows = rows;
    table->columns = columns;
    table->table = (Qddb_TableNode **)Malloc(sizeof(Qddb_TableNode *)*(rows+1));
    for (i = 0; i <= rows; i++) {
	table->table[i] = (Qddb_TableNode *)Calloc(sizeof(Qddb_TableNode)*(columns+1));
    }
    return table;
}

/* CopyHeaderInfo - copy header information for a cell.
 */
static Qddb_TableHeaderInfo *CopyHeaderInfo(info)
    Qddb_TableHeaderInfo	*info;
{
    Qddb_TableHeaderInfo	*ninfo;

    if (info == NULL)
	return NULL;
    ninfo = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
    if (info->title != NULL) {
	ninfo->title = Malloc(strlen(info->title)+1);
	strcpy(ninfo->title, info->title);
    }
    if (info->comment != NULL) {
	ninfo->comment = Malloc(strlen(info->comment)+1);
	strcpy(ninfo->comment, info->comment);
    }
    if (info->name != NULL) {
	ninfo->name = Malloc(strlen(info->name)+1);
	strcpy(ninfo->name, info->name);
    }
    return ninfo;
}

void Qddb_TableCalcCopy(dest, src)
    Qddb_TableCalc	**dest, *src;
{
    Qddb_TableCalc	*tmp;
    int		     	i, maxnum;

    for (maxnum = 0; src[maxnum].op != 0; maxnum++);
    tmp = *dest = (Qddb_TableCalc *)Calloc(sizeof(Qddb_TableCalc)*(maxnum+1));
    for (i = 0; i <= maxnum; i++) {
	tmp[i].op = src[i].op;
	tmp[i].numval = src[i].numval;
	if (src[i].strval != NULL) {
	    tmp[i].strval = (char *)Malloc(strlen(src[i].strval)+1);
	    strcpy(tmp[i].strval, src[i].strval);
	}
	if (src[i].table != NULL) {
	    tmp[i].table = (char *)Malloc(strlen(src[i].table)+1);
	    strcpy(tmp[i].table, src[i].table);
	}
    }
}

static Qddb_Table *TableCopy(table)
    Qddb_Table		*table;
{
    Qddb_Table		*ntable;
    int			celltype;
    size_t		i, j;

    ntable = TableDefine(table->rows, table->columns);
    for (i = 0; i <= ntable->rows; i++) {
	for (j = 0; j <= ntable->columns; j++) {
	    if (table->table[i][j].info != NULL)
		ntable->table[i][j].info = CopyHeaderInfo(table->table[i][j].info);
	    ntable->table[i][j].data = table->table[i][j].data;
	    if (table->table[i][j].data.calc != NULL) {
		ntable->table[i][j].data.calc = Malloc(strlen(table->table[i][j].data.calc)+1);
		ntable->table[i][j].data.compiled = NULL;
		ntable->table[i][j].data.refs = NULL;
		strcpy(ntable->table[i][j].data.calc, table->table[i][j].data.calc);
		if (table->table[i][j].data.compiled != NULL) {
		    Qddb_TableCalcCopy(&(ntable->table[i][j].data.compiled), 
				       table->table[i][j].data.compiled);
		}
	    }
	    celltype = TableCellType(table, i, j);
	    if (QDDB_TABLE_TYPE_USESTRING(celltype) && table->table[i][j].data.data.string != NULL) {
		ntable->table[i][j].data.data.string = 
		    Malloc(strlen(table->table[i][j].data.data.string)+1);
		strcpy(ntable->table[i][j].data.data.string, table->table[i][j].data.data.string);
	    }
	}
    }
    return ntable;
}

static Qddb_Table *TableCopyRow(table, from, to)
    Qddb_Table		*table;
    size_t		from, to;
{
    Qddb_TableNode	*from_row, *to_row;
    char		*calc;
    int			celltype;
    size_t		i;

    /* 1. Does the 'to' row exist?  If not, create it.
     * 2. Make sure the 'from' row exists.
     * 3. Overwrite the 'to' row completely.
     */
    if (from > table->rows)
	return NULL;
    if (to > table->rows) { /* add rows if necessary */
	table->table = (Qddb_TableNode **)Realloc(table->table, sizeof(Qddb_TableNode *)*(to+1));
	for (i = table->rows+1; i <= to; i++) {
	    table->table[i] = (Qddb_TableNode *)Calloc(sizeof(Qddb_TableNode)*(table->columns+1));
	}
	table->rows = to;
    }
    from_row = table->table[from];
    to_row = table->table[to];
    for (i = 0; i <= table->columns; i++) {
	ClearCell(table, to, i);
	to_row[i].info = CopyHeaderInfo(from_row[i].info);
	to_row[i].data = from_row[i].data;
	calc = from_row[i].data.calc;
	if (calc != NULL) {
	    to_row[i].data.calc = Malloc(strlen(calc)+1);
	    to_row[i].data.compiled = NULL;
	    to_row[i].data.refs = NULL;
	    strcpy(to_row[i].data.calc, calc);
	    if (from_row[i].data.compiled != NULL) {
		Qddb_TableCalcCopy(&(to_row[i].data.compiled), from_row[i].data.compiled);
	    }
	}
	celltype = TableCellType(table, from, i);
	if (QDDB_TABLE_TYPE_USESTRING(celltype) && from_row[i].data.data.string != NULL) {
	    to_row[i].data.data.string = Malloc(strlen(from_row[i].data.data.string)+1);
	    strcpy(to_row[i].data.data.string, from_row[i].data.data.string);
	}
    }
    return table;
}

static Qddb_Table *TableCopyCol(table, from, to)
    Qddb_Table		*table;
    size_t		from, to;
{
    Qddb_TableNode	*row;
    char		*calc;
    int			celltype;
    size_t		i, j;
    size_t		rows, columns;

    /* 1. Does the new column exist?   If not, create it.
     * 2. Make sure the 'from' column exists.
     * 3. overwrite 'to' column completely.
     */
    if (from > table->columns)
	return NULL;
    rows = table->rows;
    columns = table->columns;
    if (to > columns) {
	for (i = 0; i <= rows; i++) {
	    table->table[i] = (Qddb_TableNode *)Realloc(table->table[i], sizeof(Qddb_TableNode)*(to+1));
	    for (j = columns+1; j <= to; j++) {
		bzero(&table->table[i][j], sizeof(Qddb_TableNode));
	    }
	}
	table->columns = to;
    }
    for (i = 0; i <= rows; i++) {
	row = table->table[i];
	ClearCell(table, i, to);
	row[to].info = CopyHeaderInfo(row[from].info);
	row[to].data = row[from].data;
	calc = row[from].data.calc;
	if (calc != NULL) {
	    row[to].data.calc = Malloc(strlen(calc)+1);
	    row[to].data.compiled = NULL;
	    row[to].data.refs = NULL;
	    strcpy(row[to].data.calc, calc);
	    if (row[from].data.compiled != NULL) {
		Qddb_TableCalcCopy(&(row[to].data.compiled), row[from].data.compiled);
	    }
	}
	celltype = TableCellType(table, i, from);
	if (QDDB_TABLE_TYPE_USESTRING(celltype) && row[from].data.data.string != NULL) {
	    row[to].data.data.string = Malloc(strlen(row[from].data.data.string)+1);
	    strcpy(row[to].data.data.string, row[from].data.data.string);
	}
    }
    return table;
}

static Qddb_Table *TableCopyCell(table, fromrow, fromcol, torow, tocol)
    Qddb_Table		*table;
    size_t		fromrow, fromcol, torow, tocol;
{
    Qddb_TableNode	*torowptr, *fromrowptr;
    int			celltype;
    char		*calc;

    if (table->rows < fromrow || table->columns < fromcol || 
	table->rows < torow || table->columns < tocol)
	return NULL;
    ClearCell(table, torow, tocol);
    torowptr = table->table[torow];
    fromrowptr = table->table[fromrow];
    torowptr[tocol].info = CopyHeaderInfo(fromrowptr[fromcol].info);
    torowptr[tocol].data = fromrowptr[fromcol].data;
    calc = fromrowptr[fromcol].data.calc;
    torowptr[tocol].data.refs = NULL;
    torowptr[tocol].data.compiled = NULL;
    if (calc != NULL) {
	torowptr[tocol].data.calc = Malloc(strlen(calc)+1);
	strcpy(torowptr[tocol].data.calc, calc);
	if (torowptr[tocol].data.compiled != NULL) {
	    Qddb_TableCalcCopy(&(torowptr[tocol].data.compiled), fromrowptr[fromcol].data.compiled);
	}
	/* FIXME: need Refs here */
    }
    celltype = TableCellType(table, fromrow, fromcol);
    if (QDDB_TABLE_TYPE_USESTRING(celltype) && fromrowptr[fromcol].data.data.string != NULL) {
	torowptr[tocol].data.data.string = Malloc(strlen(fromrowptr[fromcol].data.data.string)+1);
	strcpy(torowptr[tocol].data.data.string, fromrowptr[fromcol].data.data.string);
    }
    return table;
}

static Qddb_Table *TableRemove(table)
    Qddb_Table		*table;
{
    size_t		rows, columns;
    size_t		i, j;

    rows = table->rows;
    columns = table->columns;
    for (i = 1; i <= rows; i++) {
	for (j = 0; j <= columns; j++) {
	    ClearCell(table, i, j);
	}
	Free(table->table[i]);
    }
    ClearRow(table, 0);
    Free(table->table[0]);
    Free(table->table);
    if (table->info != NULL) {
	ClearHeaderInfo(table->info);
	Free(table->info);
    }
    Free(table);
    return NULL;
}

static Qddb_Table *TableRemoveRow(table, row)
    Qddb_Table		*table;
    size_t		row;
{
    size_t		rows, columns;
    size_t		i;

    if (row > table->rows)
	return table;
    columns = table->columns;
    rows = table->rows;
    for (i = 0; i <= columns; i++) {
	ClearCell(table, row, i);
    }
    Free(table->table[row]);
    for (i = row; i < rows; i++) {
	table->table[i] = table->table[i+1];
    }
    table->rows--;
    return NULL;
}

static Qddb_Table *TableRemoveCol(table, column)
    Qddb_Table		*table;
    size_t		column;
{
    size_t		rows, columns;
    size_t		i, j;

    if (column > table->columns)
	return table;
    columns = table->columns;
    rows = table->rows;
    for (i = 0; i <= rows; i++) {
	ClearCell(table, i, column);
	for (j = column; j < columns; j++) {
	    table->table[i][j] = table->table[i][j+1];
	}
    }
    table->columns--;
    return NULL;
}

static Qddb_Table *TableInsertRow(table, atrow)
    Qddb_Table		*table;
    size_t		atrow;
{
    size_t		i, rows;

    if (atrow > table->rows) {
	table->table = (Qddb_TableNode **)Realloc(table->table, sizeof(Qddb_TableNode *)*(atrow+1));
	for (i = table->rows+1; i <= atrow; i++) {
	    table->table[i] = (Qddb_TableNode *)Calloc(sizeof(Qddb_TableNode)*(table->columns+1));
	}
	table->rows = atrow;
    } else {
	rows = ++(table->rows);
	table->table = (Qddb_TableNode **)Realloc(table->table, 
						  sizeof(Qddb_TableNode *)*(table->rows+1));	
	for (i = rows-1; i >= atrow; i--) {
	    table->table[i+1] = table->table[i];
	}
	table->table[atrow] = (Qddb_TableNode *)Calloc(sizeof(Qddb_TableNode)*(table->columns+1));
    }
    return table;
}

static Qddb_Table *TableInsertCol(table, atcolumn)
    Qddb_Table		*table;
    size_t		atcolumn;
{
    size_t		rows, columns;
    size_t		i, j;

    rows = table->rows;
    columns = table->columns;
    if (atcolumn > table->columns) {
	for (i = 0; i <= rows; i++) {
	    table->table[i] = (Qddb_TableNode *)Realloc(table->table[i], 
							sizeof(Qddb_TableNode)*(atcolumn+1));
	    for (j = columns+1; j <= atcolumn; j++) {
		bzero(&table->table[i][j], sizeof(Qddb_TableNode));
	    }
	}
	table->columns = atcolumn;
    } else {
	columns = ++(table->columns);
	for (i = 0; i <= rows; i++) {
	    table->table[i] = (Qddb_TableNode *)Realloc(table->table[i], 
							sizeof(Qddb_TableNode)*(columns+1));
	    for (j = columns; j > atcolumn; j--) {
		table->table[i][j] = table->table[i][j-1];
	    }
	    bzero(&table->table[i][atcolumn], sizeof(Qddb_TableNode));
	}
    }
    return table;
}

/* TableGetAttrs -- 
 *	Get the row/col/cell attributes.   Don't allocate any space.
 */
static Qddb_TableAttrs *TableGetAttrs(table, row, column)
    Qddb_Table			*table;
    size_t			row, column;
{
    Qddb_TableAttrs		*retval;
    Qddb_TableHeaderInfo	*info = table->table[row][column].info;
    Qddb_TableDatatype		*data = &(table->table[row][column].data);

    if ((row == 0 && column == 0) || row > table->rows || column > table->columns) {
	return NULL;
    }
    retval = (Qddb_TableAttrs *)Calloc(sizeof(Qddb_TableAttrs));
    retval->title = info->title;
    retval->name = info->name;
    retval->comment = info->comment;
    retval->variable = info->variable;
    retval->type = data->type;
    retval->calc = data->calc;
    return retval;
}

/* TableSetAttrs -- 
 *	Get the row/col/cell attributes.   Don't allocate any space.
 */
static Qddb_Table *TableSetAttrs(table, row, column, attrs)
    Qddb_Table			*table;
    size_t			row, column;
    Qddb_TableAttrs		*attrs;
{
    Qddb_TableHeaderInfo	*info = table->table[row][column].info;
    Qddb_TableDatatype		*data = &(table->table[row][column].data);
    int				celltype;

    if ((row == 0 && column == 0) || row > table->rows || column > table->columns) {
	return NULL;
    }
    if (info->title != NULL)
	Free(info->title);
    info->title = attrs->title;
    if (info->name != NULL)
	Free(info->name);
    info->name = attrs->name;
    if (info->comment != NULL)
	Free(info->comment);
    info->comment = attrs->comment;
    if (info->variable != NULL)
	Free(info->variable);
    info->variable = attrs->variable;
    celltype = TableCellType(table, row, column);
    if (celltype != attrs->type) {
	if (QDDB_TABLE_TYPE_USESTRING(celltype)) {
	    if (QDDB_TABLE_TYPE_USESTRING(attrs->type)) {
		if (data->data.string != NULL) {
		    Free(data->data.string);
		    data->data.string = NULL;
		}
	    } else { /* new type doesn't use string field */
		if (data->data.string != NULL) {
		    Free(data->data.string);
		    data->data.string = NULL;
		}
	    }
	} else {
	    if (QDDB_TABLE_TYPE_USESTRING(attrs->type)) {
		data->data.string = NULL;
	    }
	}
	switch (attrs->type) {
	case QDDB_TABLE_TYPE_INTEGER:
	    data->data.integer = 0;
	    break;
	case QDDB_TABLE_TYPE_REAL:
	    data->data.real = 0.0;
	    break;
	case QDDB_TABLE_TYPE_CALC:
	    data->data.real = 0.0;
	    break;
	default:
	    ;
	}
    }
    data->type = attrs->type;
    if (data->calc != NULL) {
	Free(data->calc);
	if (data->compiled != NULL) {
	    Free(data->compiled);
	}
	if (data->refs != NULL) {
	    Free(data->refs);
	}
    }
    data->calc = attrs->calc;
    data->compiled = NULL;
    data->refs = NULL;
    return table;
}

static size_t *table_compare_col_info = NULL;
static Boolean *table_compare_col_ascending = NULL;
static size_t table_compare_col_info_size = 0;
static Qddb_TableNode *table_compare_row0;
static int TableCompareRows(r1, r2)
    void		*r1, *r2;
{
    size_t		i, idx;
    Boolean		ascending;
    Qddb_TableNode	*nr1, *nr2, *row0 = table_compare_row0;
    int			mytype;

    for (i = 0; i < table_compare_col_info_size; i++) {
	idx = table_compare_col_info[i];
	if (idx == 0)
	    continue;
	ascending = table_compare_col_ascending[i];
	nr1 = *(Qddb_TableNode **)r1 + idx;
	nr2 = *(Qddb_TableNode **)r2 + idx;
	if (nr1->data.type != nr2->data.type) {
	    continue;
	}
	mytype = nr1->data.type;
	if (mytype == QDDB_TABLE_TYPE_DEFAULT) {
	    mytype = row0[idx].data.type;
	}
	switch (mytype) {
	case QDDB_TABLE_TYPE_INTEGER: {
	    int			result;

	    result = nr1->data.data.integer - nr2->data.data.integer;
	    if (result != 0) {
		return ascending == 0? -result: result;
	    }
	    break;
	}
	case QDDB_TABLE_TYPE_REAL:
	case QDDB_TABLE_TYPE_CALC: {
	    double		result;

	    result = nr1->data.data.real - nr2->data.data.real;
	    if (result != 0.0) {
		if (result > 0.0) {
		    return ascending == 0? -1:1;
		} else {
		    return ascending == 0? 1:-1;
		}
	    }
	    break;
	}
	case QDDB_TABLE_TYPE_DATE: {
	    char		*str1, *str2;
	    time_t		t1, t2;
	    long		result;

	    if (nr1->data.data.string == NULL) {
		if (nr2->data.data.string == NULL) {
		    return 0;
		} else {
		    return ascending == 0? 1 : -1;
		}
	    } else if (nr2->data.data.string == NULL) {
		return ascending == 0? -1 : 1;
	    }
	    str1 = Qddb_DateStringToTime(nr1->data.data.string);
	    str2 = Qddb_DateStringToTime(nr2->data.data.string);
	    t1 = strtol(str1, NULL, 10);
	    t2 = strtol(str2, NULL, 10);
	    Free(str1);
	    Free(str2);
	    result = (long)t1 - (long)t2;
	    if (result != 0) {
		if (result > 0) {
		    return ascending == 0? -1 : 1;
		} else {
		    return ascending == 0? 1 : -1;
		}
	    }
	    break;
	}
	case QDDB_TABLE_TYPE_DEFAULT:
	case QDDB_TABLE_TYPE_STRING: {
	    int			result;

	    if (nr1->data.data.string == NULL) {
		if (nr2->data.data.string == NULL) {
		    return 0;
		} else {
		    return ascending == 0? 1 : -1;
		}
	    } else if (nr2->data.data.string == NULL) {
		return ascending == 0? -1 : 1;
	    }
	    result = strcmp(nr1->data.data.string, nr2->data.data.string);
	    if (result != 0) {
		return ascending == 0? -result : result;
	    }
	    break;
	}
	default: ;
	}
    }
    return 0;
}

static Qddb_Table *TableSort(table, info)
    Qddb_Table			*table;
    Qddb_TableOpSort		*info;
{
    size_t			i, columns = table->columns, startrow, endrow;

    table_compare_col_info = (size_t *)Malloc(sizeof(size_t)*columns);
    table_compare_col_ascending = (Boolean *)Malloc(sizeof(Boolean)*columns);
    for (i = 0;  i < info->number && i < columns; i++) {
	table_compare_col_info[i] = TableFindCol(table, info->sortby[i].column_name);
	if (table_compare_col_info[i] == 0) { /* couldn't find column name, ignore it */
	    continue;
	}
	table_compare_col_ascending[i] = info->sortby[i].ascending;
    }
    if (info->startrow == NULL) {
	startrow = 1;
    } else {
	startrow = TableFindRow(table, info->startrow);
	if (startrow == 0)
	    startrow = 1;
    }
    if (info->endrow == NULL) {
	endrow = table->rows;
    } else {
	endrow = TableFindRow(table, info->endrow);
	if (endrow == 0)
	    endrow = table->rows;
    }
    if (startrow < endrow) {
	table_compare_row0 = table->table[0];
	table_compare_col_info_size = i;
	(void)qsort(table->table + startrow, endrow, sizeof(Qddb_TableNode *),
		    (int (*) _ANSI_ARGS_((const void *, const void *)))TableCompareRows);
    }
    Free(table_compare_col_info);
    Free(table_compare_col_ascending);
    table_compare_col_info = NULL;
    table_compare_col_ascending = NULL;
    return table;
}

/* Extra support routines
 */
static void ClearHeaderInfo(info)
    Qddb_TableHeaderInfo	*info;
{
    if (info->title != NULL)
	Free(info->title);
    if (info->name != NULL)
	Free(info->name);
    if (info->comment != NULL)
	Free(info->comment);
    if (info->variable != NULL)
	Free(info->variable);
    if (info->print != NULL) {
	if (info->print->format != NULL)
	    Free(info->print->format);
	if (info->print->separator != NULL)
	    Free(info->print->separator);
	Free(info->print);
    }
    info->title = NULL;
    info->name = NULL;
    info->comment = NULL;
    info->variable = NULL;
    info->print = NULL;
} 

static void ClearCell(table, row, column)
    Qddb_Table			*table;
    size_t			row, column;
{
    Qddb_TableNode		*node = &table->table[row][column], *row0;
    Qddb_TableHeaderInfo	*info = node->info;

    if (info != NULL) {
	ClearHeaderInfo(info);
	Free(info);
	node->info = NULL;
    }
    if (node->data.calc != NULL) {
	Free(node->data.calc);
	node->data.calc = NULL;
	if (node->data.compiled != NULL) {
	    Qddb_Free(QDDB_TYPE_TABLECALC, node->data.compiled);
	    node->data.compiled = NULL;
	}
	if (node->data.refs != NULL) {
	    Qddb_Free(QDDB_TYPE_TABLEREFS, node->data.refs);
	    node->data.refs = NULL;
	}
    }
    row0 = table->table[0] + column;
    if (node->data.type == QDDB_TABLE_TYPE_DEFAULT) {
	if (QDDB_TABLE_TYPE_USESTRING(row0->data.type) && node->data.data.string != NULL) {
	    Free(node->data.data.string);
	    node->data.data.string = NULL;
	}
    } else {
	if (QDDB_TABLE_TYPE_USESTRING(node->data.type) && node->data.data.string != NULL) {
	    Free(node->data.data.string);
	    node->data.data.string = NULL;
	}
    }
    node->data.type = QDDB_TABLE_TYPE_DEFAULT;
    node->data.marked = 0;
    switch(row0->data.type) {
    case QDDB_TABLE_TYPE_DEFAULT:
    case QDDB_TABLE_TYPE_STRING:
    case QDDB_TABLE_TYPE_DATE:
	node->data.data.string = NULL;
	break;
    case QDDB_TABLE_TYPE_REAL:
    case QDDB_TABLE_TYPE_CALC:
	node->data.data.real = 0.0;
	break;
    case QDDB_TABLE_TYPE_INTEGER:
	node->data.data.integer = 0;
	break;
    }
}

static void ClearColumn(table, column)
    Qddb_Table			*table;
    size_t			column;
{
    size_t			i;
    size_t			rows;

    rows = table->rows;
    for (i = 1; i <= rows; i++) {
	ClearCell(table, i, column);
    }
    ClearCell(table, 0, column);
}

static void ClearRow(table, row)
    Qddb_Table			*table;
    size_t			row;
{
    size_t			i;
    size_t			columns;

    columns = table->columns;
    for (i = 0; i <= columns; i++) {
	ClearCell(table, row, i);
    }
}

static void ClearTable(table)
    Qddb_Table			*table;
{
    size_t			i;
    size_t			rows;

    rows = table->rows;
    for (i = 1; i <= rows; i++) {
	ClearRow(table, i);
    }
    ClearRow(table, 0);
    if (table->info != NULL) {
	ClearHeaderInfo(table->info);
	Free(table->info);
	table->info = NULL;
    }
}

/* TableFindRow - find the row with the given name.  If the name is an
 * integer, return the number converted to integer; otherwise look for
 * the row with the given name.
 */
static size_t TableFindRow(table, name)
    Qddb_Table 			*table;
    char			*name;
{
    Qddb_TableNode		**ptr;
    size_t			i, max;
    unsigned long		num;
    char			*endptr = NULL;

    if (name == NULL || *name == '\0')
	return 0;
    num = strtoul(name, &endptr, 10);
    if (name == endptr || endptr != name + strlen(name)) {
	ptr = table->table + 1; /* skip info row */
	max = table->rows;
	for (i = 1; i <= max; i++, ptr++) {
#if defined(DIAGNOSTIC)
	    if (*ptr == NULL)
		PANIC("TableFindRow: null row found");
#endif
	    if ((*ptr)->info != NULL && ((*ptr)->info->name != NULL)) {
		if (strcmp((*ptr)->info->name, name) == 0) {
		    return i;
		}
	    }
	}
	return 0;
    }
    if (num > table->rows)
	return 0;
    return num;
}


static size_t TableFindCol(table, name)
    Qddb_Table			*table;
    char			*name;
{
    Qddb_TableNode		*ptr;
    size_t			i, max;
    unsigned long		num;
    char			*endptr = NULL;

    if (name == NULL || *name == '\0')
	return 0;
    num = strtoul(name, &endptr, 10);
    if (name == endptr || endptr != name + strlen(name)) {
	ptr = table->table[0] + 1;
	max = table->columns;
	for (i = 1; i <= max; i++, ptr++) {
	    if (ptr->info != NULL && ptr->info->name != NULL) {
		if (strcmp(ptr->info->name, name) == 0) {
		    return i;
		}
	    }
	}
	return 0;
    }
    if (num > table->columns)
	return 0;
    return num;
}

static Qddb_TableRowCol *TableFindCell(table, rowname, colname)
    Qddb_Table			*table;
    char			*rowname, *colname;
{
    Qddb_TableRowCol		*retval;

    retval = (Qddb_TableRowCol *)Calloc(sizeof(Qddb_TableRowCol));
    retval->row = TableFindRow(table, rowname);
    retval->column = TableFindCol(table, colname);
    if (retval->row == 0 && retval->column == 0) {
	Qddb_TableNode		**rowptr, *colptr;
	char			*cellname;
	size_t			rowlen, collen;
	size_t			i, j;

	if (rowname == NULL)
	    rowlen = 0;
	else
	    rowlen = strlen(rowname);
	if (colname == NULL)
	    collen = 0;
	else
	    collen = strlen(colname);
	cellname = Malloc(rowlen+collen+1);
	strcpy(cellname, rowname);
	strcat(cellname, colname);
	for (i = 1, rowptr = table->table+1; i <= table->rows; i++, rowptr++) {
	    for (j = 1, colptr = (*rowptr)+1; j <= table->columns; j++, colptr++) {
		if (colptr->info != NULL && colptr->info->name != NULL) {
		    if (strcmp(colptr->info->name, cellname) == 0) {
			retval->row = i;
			retval->column = j;
			Free(cellname);
			return retval;
		    }
		}
	    }
	}
	Free(cellname);
    }
    return retval;
}

static int TableCellType(table, row, col)
    Qddb_Table			*table;
    size_t			row, col;
{
    Qddb_TableNode		*row0 = table->table[0];

    if (table->table[row][col].data.type == QDDB_TABLE_TYPE_DEFAULT) {
	return row0->data.type;
    } else {
	return table->table[row][col].data.type;
    }
}


Qddb_TableCalc *Qddb_TableCalcParse(calc)
    char			*calc;
{
    char			*tmp;

    tmp = (char *)Malloc(strlen(calc)+1);
    strcpy(tmp, calc);
    Qddb_ResetTableParser();
    Qddb_SetBuffer(tmp);
    if (TableParse() != 0) {
	return NULL;
    }
    Free(tmp);
    return Qddb_GetTableParserResults();
}

static Qddb_Table *TableAppend(table, table2)
    Qddb_Table		*table, *table2;
{
    Qddb_TableNode	*torowptr, *fromrowptr;
    int			celltype;
    char		*calc;
    size_t		i, j;

    for (i = 1; i <= table2->rows; i++) {
	table = TableInsertRow(table, table->rows+1);
	for (j = 0; j <= table2->columns; j++) {
	    torowptr = table->table[table->rows];
	    fromrowptr = table2->table[i];
	    torowptr[j].info = CopyHeaderInfo(fromrowptr[j].info);
	    torowptr[j].data = fromrowptr[j].data;
	    calc = fromrowptr[j].data.calc;
	    torowptr[j].data.refs = NULL;
	    torowptr[j].data.compiled = NULL;
	    if (calc != NULL) {
		torowptr[j].data.calc = Malloc(strlen(calc)+1);
		torowptr[j].data.compiled = NULL;
		torowptr[j].data.refs = NULL;
		strcpy(torowptr[j].data.calc, calc);
		if (torowptr[j].data.compiled != NULL) {
		    Qddb_TableCalcCopy(&(torowptr[j].data.compiled), fromrowptr[j].data.compiled);
		}
		/* FIXME: need Refs here */
	    }
	    celltype = TableCellType(table2, i, j);
	    if (QDDB_TABLE_TYPE_USESTRING(celltype) && fromrowptr[j].data.data.string != NULL) {
		torowptr[j].data.data.string = Malloc(strlen(fromrowptr[j].data.data.string)+1);
		strcpy(torowptr[j].data.data.string, fromrowptr[j].data.data.string);
	    }
	}
    }
    return table;
}

static Qddb_Table *TableSummary(table, x_name, y_name, x_range, x_format, range_type)
    Qddb_Table             *table;
    char                   *x_name, *y_name, *x_range, *x_format, *range_type;
{
    Qddb_Table             *retval = NULL, *ntable;
    Qddb_TableOpSort       info;
    Qddb_TableOpSortNode   infonode;
    char                   *sort_name, *tmpbuf, *tmpbuf2;
    size_t                 x_idx, y_idx, sort_idx, fmt_idx;
    size_t                 num_unique = 1;
    int                    x_type, y_type, maxrow;
    int                    i, j, curr, ncurr;
    int                    range_op;

    x_idx = TableFindCol(table, x_name);
    if (x_idx == 0) {
	return NULL;
    }
    y_idx = TableFindCol(table, y_name);
    if (y_idx == 0) {
	return NULL;
    }
    if (table->rows == 0 || table->columns == 0) {
	return NULL;
    }
    if (strcmp(range_type, "sum") == 0) {
	range_op = QDDB_TABLE_CALC_OP_SUM;
    } else if (strcmp(range_type, "avg") == 0) {
	range_op = QDDB_TABLE_CALC_OP_AVG;
    } else if (strcmp(range_type, "min") == 0) {
	range_op = QDDB_TABLE_CALC_OP_MIN;
    } else if (strcmp(range_type, "max") == 0) {
	range_op = QDDB_TABLE_CALC_OP_MAX;
    } else if (strcmp(range_type, "stddev") == 0) {
	range_op = QDDB_TABLE_CALC_OP_STDDEV;
    } else if (strcmp(range_type, "prod") == 0) {
	range_op = QDDB_TABLE_CALC_OP_PROD;
    } else if (strcmp(range_type, "count") == 0) {
	range_op = QDDB_TABLE_CALC_OP_COUNT;
    } else {
	return NULL;
    }
    sort_idx = table->columns + 1;
    fmt_idx = sort_idx + 1;
    sort_name = IntegerToString((int)sort_idx);
    x_type = table->table[0][x_idx].data.type;
    y_type = table->table[0][y_idx].data.type;
    if (x_type != QDDB_TABLE_TYPE_STRING) {
	TableInsertCol(table, sort_idx);
	TableInsertCol(table, fmt_idx);
    }
    maxrow = table->rows;
    /* Build the sort_idx column */
    switch (x_type) {
    case QDDB_TABLE_TYPE_DATE: {
	char         *tmp, *last;

	for (i = 1; i <= maxrow; i++) {
	    if (table->table[i][x_idx].data.type != QDDB_TABLE_TYPE_DATE && \
		table->table[i][x_idx].data.type != QDDB_TABLE_TYPE_DEFAULT) {
		continue;
	    }
	    tmp = Qddb_DateStringToTime(table->table[i][x_idx].data.data.string);
	    table->table[i][sort_idx].data.data.string = Qddb_DateTimeToString(tmp, x_range);
	    table->table[i][fmt_idx].data.data.string = Qddb_DateTimeToString(tmp, x_format);
	    Free(tmp);
	}
	info.sortby = &infonode;
	info.startrow = info.endrow = NULL;
	info.number = 1;
	infonode.column_name = sort_name;
	infonode.ascending = True;
	TableSort(table, &info);    
	last = table->table[1][sort_idx].data.data.string;
	for (i = 2; i <= maxrow; i++) {
	    tmp = table->table[i][sort_idx].data.data.string;
	    if (strcmp(last, tmp) != 0) {
		last = tmp;
		num_unique++;
	    }
	}
	break;
    }
    case QDDB_TABLE_TYPE_STRING: {
	char         *tmp, *last;

	info.sortby = &infonode;
	info.startrow = info.endrow = NULL;
	info.number = 1;
	infonode.column_name = x_name;
	infonode.ascending = True;
	TableSort(table, &info);    
	sort_idx = x_idx;
	fmt_idx = x_idx;
	last = table->table[1][sort_idx].data.data.string;
	for (i = 2; i <= maxrow; i++) {
	    tmp = table->table[i][sort_idx].data.data.string;	    
	    if (strcmp(last, tmp) != 0) {
		last = tmp;
		num_unique++;
	    }
	}
	break;
    }
    case QDDB_TABLE_TYPE_INTEGER: {
	int        range, current, new;
	char       *buf, *buf2;
	
	info.sortby = &infonode;
	info.startrow = info.endrow = NULL;
	info.number = 1;
	infonode.column_name = x_name;
	infonode.ascending = True;
	TableSort(table, &info);
	range = atoi(x_range);
	if (range <= 0) {
	    range = 1;
	}
	current = table->table[1][x_idx].data.data.integer;
	buf = table->table[1][sort_idx].data.data.string = IntegerToString(current);
	table->table[1][fmt_idx].data.data.string = IntegerToString(current);
	for (i = 2; i <= maxrow; i++) {
	    new = table->table[i][x_idx].data.data.integer;
	    if (current+range <= new) {
		current = new;
		buf = table->table[i][sort_idx].data.data.string = IntegerToString(current);
		table->table[i][fmt_idx].data.data.string = IntegerToString(current);
		num_unique++;
	    } else {
		buf2 = table->table[i][sort_idx].data.data.string = Malloc(MAXINTEGERLEN);
		strcpy(buf2, buf);
		buf2 = table->table[i][fmt_idx].data.data.string = Malloc(MAXINTEGERLEN);
		strcpy(buf2, buf);
	    }
	}
	break;
    }
    case QDDB_TABLE_TYPE_REAL:
    case QDDB_TABLE_TYPE_CALC: {
	double     range, current, new;
	char       *buf, *buf2, realbuf[BUFSIZ];
	size_t     realbuf_len;
	int        precision = 0;
	
	info.sortby = &infonode;
	info.startrow = info.endrow = NULL;
	info.number = 1;
	infonode.column_name = x_name;
	infonode.ascending = True;
	TableSort(table, &info);
	range = atof(x_range);
	if (range <= 0.0) {
	    range = 1.0;
	}
	precision = atoi(x_format);
	if (precision < 0) {
	    precision = 0;
	}
	current = table->table[1][x_idx].data.data.real;
	sprintf(realbuf, "%.*f", precision, current);
	realbuf_len = strlen(realbuf) + 1;
	buf = table->table[1][sort_idx].data.data.string = Malloc(realbuf_len);
	strcpy(buf, realbuf);
	buf = table->table[1][fmt_idx].data.data.string = Malloc(realbuf_len);
	strcpy(buf, realbuf);
	for (i = 2; i <= maxrow; i++) {
	    new = table->table[i][x_idx].data.data.real;
	    if (current+range <= new) {
		current = new;
		sprintf(realbuf, "%.*f", precision, current);
		realbuf_len = strlen(realbuf) + 1;
		buf = table->table[i][sort_idx].data.data.string = Malloc(realbuf_len);
		strcpy(buf, realbuf);
		buf = table->table[i][fmt_idx].data.data.string = Malloc(realbuf_len);
		strcpy(buf, realbuf);
		num_unique++;
	    } else {
		buf2 = table->table[i][sort_idx].data.data.string = Malloc(realbuf_len);
		strcpy(buf2, realbuf);
		buf2 = table->table[i][fmt_idx].data.data.string = Malloc(realbuf_len);
		strcpy(buf2, realbuf);
	    }
	}
	break;
    }
    default:
	Free(sort_name);
	return NULL;
    }
    if (y_type == QDDB_TABLE_TYPE_INTEGER) {
	for (i = 1; i <= maxrow; i++) {
	    table->table[i][y_idx].data.data.real = (double)table->table[i][y_idx].data.data.integer;
	}
    }
    retval = ntable = TableDefine(num_unique, 3);
    curr = 1;
    ncurr = 0;
    ntable->table[0][3].data.type = QDDB_TABLE_TYPE_REAL;
    tmpbuf = table->table[1][sort_idx].data.data.string;
    tmpbuf2 = table->table[1][fmt_idx].data.data.string;
    for (i = 1; i <= maxrow; i++) {
	char        *nstr, *nfmt;
	int         cmp;

	nstr = table->table[i][sort_idx].data.data.string;
	nfmt = table->table[i][fmt_idx].data.data.string;
	cmp = strcmp(tmpbuf, nstr);
	if (cmp != 0 || (cmp == 0 && i == maxrow)) {
	    ncurr++;
#if defined(DIAGNOSTIC)
	    if (ncurr > ntable->rows)
		PANIC("TableSummary: ncurr > ntable->rows");
#endif
	    if (i == maxrow && cmp == 0) { /* include last row in stats */
		i++;
	    }
	    ntable->table[ncurr][1].data.data.string = Malloc(strlen(tmpbuf)+1);
	    strcpy(ntable->table[ncurr][1].data.data.string, tmpbuf);
	    ntable->table[ncurr][2].data.data.string = Malloc(strlen(tmpbuf2)+1);
	    strcpy(ntable->table[ncurr][2].data.data.string, tmpbuf2);
	    tmpbuf = nstr;
	    tmpbuf2 = nfmt;
	    if (curr != 0) {
		switch (range_op) {
		case QDDB_TABLE_CALC_OP_SUM:
		    ntable->table[ncurr][3].data.data.real = 0.0;
		    for (j = curr; j < i; j++) {
			ntable->table[ncurr][3].data.data.real +=
			    table->table[j][y_idx].data.data.real;
		    }
		    break;
		case QDDB_TABLE_CALC_OP_AVG:
		    ntable->table[ncurr][3].data.data.real = 0.0;
		    for (j = curr; j < i; j++) {
			ntable->table[ncurr][3].data.data.real +=
			    table->table[j][y_idx].data.data.real;
		    }
		    ntable->table[ncurr][3].data.data.real /= (double)((double)j - (double)curr);
		    break;
		case QDDB_TABLE_CALC_OP_STDDEV: {
		    double     mean, num, num2, count;

		    num = 0.0;
		    count = 0.0;
		    for (j = curr; j < i; j++) {
			num += table->table[j][y_idx].data.data.real;
			count += 1.0;
		    }
		    mean = num / count;
		    num = 0.0;
		    for (j = curr; j < i; j++) {
			num2 = (table->table[j][y_idx].data.data.real - mean);
			num += num2*num2;
		    }		    
		    ntable->table[ncurr][3].data.data.real = sqrt(num  / (count - 1.0));
		    break;
		}
		case QDDB_TABLE_CALC_OP_MIN: {
		    double      nval, val;

		    nval = table->table[curr][y_idx].data.data.real;
		    ntable->table[ncurr][3].data.data.real = nval;
		    for (j = curr+1; j < i; j++) {
			val  = table->table[j][y_idx].data.data.real;
			if (nval > val) {
			    ntable->table[ncurr][3].data.data.real = val;
			    nval = val;
			}
		    }
		    break;
		}
		case QDDB_TABLE_CALC_OP_MAX: {
		    double      nval, val;

		    nval = table->table[curr][y_idx].data.data.real;
		    ntable->table[ncurr][3].data.data.real = nval;
		    for (j = curr+1; j < i; j++) {
			val  = table->table[j][y_idx].data.data.real;
			if (nval < val) {
			    ntable->table[ncurr][3].data.data.real = val;
			    nval = val;
			}
		    }
		    break;
		}
		case QDDB_TABLE_CALC_OP_PROD:
		    ntable->table[ncurr][3].data.data.real = 1.0;
		    for (j = curr; j < i; j++) {
			ntable->table[ncurr][3].data.data.real *=
			    table->table[j][y_idx].data.data.real;
		    }
		    break;
		case QDDB_TABLE_CALC_OP_COUNT:
		    ntable->table[ncurr][3].data.data.real = (double)(i - curr);
		    if (i >= maxrow)
			ntable->table[ncurr][3].data.data.real += 1.0;
		    break;
		}
	    }
	    curr = i;
	}
	if (i == maxrow && cmp != 0) {
	    tmpbuf = nstr;
	    tmpbuf2 = nfmt;
	    ncurr++;
	    ntable->table[ncurr][1].data.data.string = Malloc(strlen(tmpbuf)+1);
	    strcpy(ntable->table[ncurr][1].data.data.string, tmpbuf);
	    ntable->table[ncurr][2].data.data.string = Malloc(strlen(tmpbuf2)+1);
	    strcpy(ntable->table[ncurr][2].data.data.string, tmpbuf2);
	    switch (range_op) {
	    case QDDB_TABLE_CALC_OP_SUM:
	    case QDDB_TABLE_CALC_OP_AVG:
	    case QDDB_TABLE_CALC_OP_MIN:
	    case QDDB_TABLE_CALC_OP_MAX:
	    case QDDB_TABLE_CALC_OP_PROD:
		ntable->table[ncurr][3].data.data.real = table->table[maxrow][y_idx].data.data.real;
		break;
	    case QDDB_TABLE_CALC_OP_STDDEV:
		ntable->table[ncurr][3].data.data.real = DBL_MAX;
		break;
	    case QDDB_TABLE_CALC_OP_COUNT:
		ntable->table[ncurr][3].data.data.real = 1.0;
		break;
	    }
	}
    }
#if defined(DIAGNOSTIC)
    if (ncurr != ntable->rows)
	printf("TableSummary warning: ncurr (%d) != ntable->rows (%d)\n", ncurr, ntable->rows);
#endif
    if (x_type != QDDB_TABLE_TYPE_STRING) {
	TableRemoveCol(table, fmt_idx);
	TableRemoveCol(table, sort_idx);
    }
    Free(sort_name);
    return retval;
}
