
/* qddb/Lib/LibQddb/Schema.c 
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	Schema *Qddb_InitSchema(char *)
 *	Schema *Qddb_RebuildSchema(Schema *, int);
 */

static Schema *ReadRelationSchema _ANSI_ARGS_((char *));
static Schema *RebuildFromSchemaNodes _ANSI_ARGS_((Schema *));
static Schema *RebuildFromSchemaTree _ANSI_ARGS_((Schema *));
static void   FillInAttributes _ANSI_ARGS_((Schema *));

Schema *Qddb_InitSchema(RelationFN)
    char		*RelationFN;
{
    Schema		*retval;

    qddb_errno = 0;
    if (qddb_errmsg != NULL) {
	Free(qddb_errmsg);
	qddb_errmsg = NULL;
    }
    retval = ReadRelationSchema(RelationFN);
    if (retval == NULL)
	return NULL;
    Qddb_InitHash(retval);
    Qddb_MakeHash(retval, RelationFN);
    FillInAttributes(retval);
    retval = Qddb_RebuildSchema(retval, QDDB_REBUILDSCHEMA_FROMNODES);
    Qddb_InitWordList(retval);
    return retval;
}

static void FillInAttributes(schema)
    Schema		*schema;
{
    int			i, j;
    char		buf[BUFSIZ];
    size_t		ancestor, HowMany, buf_len;
#if defined(USE_TCL)
    Tcl_HashEntry	*hash_entry;
    int			newPtr;
#endif

    for (i = 1; i <= schema->NumberOfAttributes; i++) {
	HowMany = schema->Entries[i].Level;
	buf_len = 0;
	for (j = 1; j < HowMany; j++) {
	    ancestor = schema->Entries[i].AncestorNumber[j];
	    sprintf(buf+buf_len, "%d.", (int)ancestor);
	    buf_len += strlen(buf+buf_len);
	}
	sprintf(buf+buf_len, "%d", (int)schema->Entries[i].Number);
	buf_len += strlen(buf+buf_len);
#if defined(USE_TCL)
	hash_entry = Tcl_CreateHashEntry(&(schema->TclHashTable), buf, &newPtr);
	Tcl_SetHashValue(hash_entry, (ClientData)i);
#else
	schema->Entries[i].Attribute = Malloc(buf_len+1);
	strcpy(schema->Entries[i].Attribute, buf);
#endif
    }
}

Schema *Qddb_RebuildSchema(schema, how)
    Schema		*schema;
    int			how; /* 0 == from SchemaNodes, 1 == from SchemaTree */
{
    if (how == QDDB_REBUILDSCHEMA_FROMNODES) {
	return RebuildFromSchemaNodes(schema);
    } else if (how == QDDB_REBUILDSCHEMA_FROMTREE) {
	return RebuildFromSchemaTree(schema);
    }
    return NULL;
}

static SchemaTreeNode **AllocateSchemaTreeSlot(tree)
    SchemaTreeNode	***tree;
{
    SchemaTreeNode	**retval;
    int			i;

    if (*tree != NULL) {
	for (i = 1, retval = *tree; (*retval) != NULL; retval++, i++);
	*tree = (SchemaTreeNode **)Realloc(*tree, sizeof(SchemaTreeNode *) * (i+1));
	(*tree)[i] = NULL;
    } else {
	*tree = (SchemaTreeNode **)Malloc(sizeof(SchemaTreeNode *) * 2);
	(*tree)[1] = NULL;
	i = 1;
    }
    return (*tree) + i - 1;
}

static SchemaTreeNode ***FindPlaceInTree(tree, entry)
    SchemaTreeNode	***tree;
    SchemaNode		*entry;
{
    SchemaTreeNode	***place;
    int			level = entry->Level;
    int			i;

    place = tree;
    for (i = 1; i < level; i++) {
#if defined(DIAGNOSTIC)
	if (*place == NULL || (*place)[entry->AncestorNumber[i]-1] == NULL) {
	    fprintf(stderr, "FindPlaceInTree: aborted at i == %d\n", i);
	    exit(1);
	}
#endif
	place = &((*place)[entry->AncestorNumber[i]-1]->children);
    }
    return place;
}

static void AddNodeToTree(tree, entry)
    SchemaTreeNode      ***tree;
    SchemaNode		*entry;
{
    SchemaTreeNode	***slots, **newslot;

    if (*tree == NULL)
	*tree = (SchemaTreeNode **)Calloc(sizeof(SchemaTreeNode *)*2);
    slots = FindPlaceInTree(tree, entry);
    newslot = AllocateSchemaTreeSlot(slots);
    *newslot = (SchemaTreeNode *)Malloc(sizeof(SchemaTreeNode));
    (*newslot)->schemanode = entry;
    (*newslot)->children = NULL;
}

static Schema *RebuildFromSchemaNodes(schema)
    Schema		*schema;
{
    int			i;
    SchemaTreeNode	**Tree = NULL;

    for (i = 1; i <= schema->NumberOfAttributes; i++)
	AddNodeToTree(&Tree, schema->Entries+i);
    schema->Tree = Tree;
    return schema;
}

static Schema *RebuildFromSchemaTree(schema)
    Schema		*schema;
{
    return NULL;	/* not yet implemented */
}

static Schema *ReadRelationSchema(RelationFN)
    char		*RelationFN;
{	
    int			RelationFile, read_only = 0;
    FILE		*RelationFilePtr;
    Schema		*CurrentSchema;
    char		*CurrentSchemaName, *buf;

    /* Initialize the data structure before passing it to the parser */
    CurrentSchemaName = Malloc(MAXPATHLEN);
    CurrentSchema = (Schema *)Calloc(sizeof(Schema));
    CurrentSchema->ReducedAttributeTable = NULL;
    CurrentSchema->TopOfReducedAttributeTable = 0;
    CurrentSchema->NumberOfAttributes = 0;
    CurrentSchema->HashSize = DEFAULT_HASHSIZE;
    CurrentSchema->HashType = QDDB_HASHTYPE_DEFAULT;
    CurrentSchema->UseReducedAttributeIdentifiers = False;
    CurrentSchema->UseCachedHashing = False;
    CurrentSchema->UseCachedSecondarySearch = False;
    CurrentSchema->UseExcludeWords = False;
    CurrentSchema->UseCondensedIndexing = False;
    CurrentSchema->UseLargeDatabaseOptimizations = False;
    CurrentSchema->CacheSize = DEFAULT_CACHESIZE;
    CurrentSchema->TopOfCache = 0;
    CurrentSchema->Entries = (SchemaNode *)Malloc(sizeof(SchemaNode)*MAXIMUM_SCHEMA_ENTRY);
    CurrentSchema->RelationName = Malloc(strlen(RelationFN)+1);
    CurrentSchema->default_date_format = Malloc(strlen(qddb_default_date_format)+1);
    strcpy(CurrentSchema->default_date_format, qddb_default_date_format);
    strcpy(CurrentSchema->RelationName, RelationFN);
    strcpy(CurrentSchemaName, RelationFN);
    strcat(CurrentSchemaName, "/Schema");
#if defined(USE_TCL)
    Tcl_InitHashTable(&(CurrentSchema->TclHashTable), TCL_STRING_KEYS);
#endif

    RelationFilePtr = fopen(CurrentSchemaName, "r");
    Free(CurrentSchemaName);
    if (RelationFilePtr == NULL) {
	Qddb_Free(QDDB_TYPE_SCHEMA, CurrentSchema);
	qddb_errno = QDDB_ERRNO_INVALID_ARG;
	qddb_errmsg = Malloc(BUFSIZ);
	sprintf(qddb_errmsg, "Cannot open Schema for relation '%s'", RelationFN);
	return NULL;
    }
    RelationFile = (int)((unsigned int)fileno(RelationFilePtr));
    Qddb_ResetSchemaParser();
    Qddb_SchemaParse_Set(CurrentSchema);
    buf = Qddb_GetFile(RelationFile);
    Qddb_SetBuffer(buf);
    if (SchemaParse() != 0) {
	Qddb_Free(QDDB_TYPE_SCHEMA, CurrentSchema);
	return NULL;
    }
    Free(buf);
    fclose(RelationFilePtr);
    CurrentSchema = Qddb_SchemaParse_Get();
    if ((CurrentSchema->database_fd = OpenDatabase(RelationFN, 0)) == -1) {
	fprintf(stderr, "Cannot open Database for relation \"%s\"\n", RelationFN);
	exit(1);
    }
    if ((CurrentSchema->hashtable_fd = OpenHashTable(RelationFN, 0)) == -1) {
	fprintf(stderr, "Cannot open HashTable for relation \"%s\"\n", RelationFN);
	exit(1);
    }
    errno = 0;
    if (LockSection(CurrentSchema->hashtable_fd, F_RDLCK, (off_t)0, (off_t)0, False) == -1) {
#if defined(EOPNOTSUPP)
	if (errno == EOPNOTSUPP) {
	    fprintf(stderr, "Warning: cannot lock HashTable, all accesses are read-only.\n");
	    read_only = 1;
	}
#endif
#if defined(EINVAL)
	if (errno == EINVAL) {
	    fprintf(stderr, "Warning: cannot lock HashTable, all accesses are read-only.\n");
	    read_only = 1;
	}
#endif
#if defined(ENOLCK)
	if (errno == ENOLCK) {
	    fprintf(stderr, "Warning: cannot lock HashTable, all accesses are read-only.\n");
	    read_only = 1;
	}
#endif
	if (!read_only) {
	    /* A stabilization is occurring */
	    qddb_errno = QDDB_ERRNO_WRLOCKED;
	    qddb_errmsg = Malloc(128);
	    sprintf(qddb_errmsg, "Cannot open schema: HashTable is locked for writing");
	    Qddb_Free(QDDB_TYPE_SCHEMA, CurrentSchema);
	    return NULL;
	}
    }
    CurrentSchema->ReducedAttributeTable = Qddb_ReadReducedAttrTable(CurrentSchema);
    if (qddb_stabilizing == 0) {
	if (SizeOfFile(CurrentSchema->database_fd) <= 1) {
	    /* If this is a new database, ignore reduced attribute identifiers */
	    CurrentSchema->UseReducedAttributeIdentifiers = False;
	} else if (CurrentSchema->ReducedAttributeTable == NULL &&
	    CurrentSchema->UseReducedAttributeIdentifiers == True) {
	    qddb_errno = QDDB_ERRNO_NEEDSTAB;
	    qddb_errmsg = Malloc(128);
	    sprintf(qddb_errmsg, "Cannot open schema: Relation needs stabilizing\n\
Reduced attribute index doesn't exist, but \n\
\"Use Reduced Attribute Identifiers\" is specified in Schema.\n");
	    Qddb_Free(QDDB_TYPE_SCHEMA, CurrentSchema);
	    return NULL;
	} else if (CurrentSchema->ReducedAttributeTable != NULL &&
		   CurrentSchema->UseReducedAttributeIdentifiers == False) {
	    qddb_errno = QDDB_ERRNO_NEEDSTAB;
	    qddb_errmsg = Malloc(128);
	    sprintf(qddb_errmsg, "Cannot open schema: Relation needs stabilizing\n\
Reduced attribute index exists, but \n\
\"Use Reduced Attribute Identifiers\" not specified in Schema.\n");
	    Qddb_Free(QDDB_TYPE_SCHEMA, CurrentSchema);
	    return NULL;
	}
    }
    return CurrentSchema;
}

