
/* qddb/Lib/LibQddb/Rows.c 
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* RowListCompareRows:
 *
 * This routine takes advantage of the fact that all rows are the same
 * length and they are sorted.
 */
Boolean RowListCompareRows(TheEntry, Row1, Row2, Nums, Top)
    InCoreEntry		*TheEntry;
    RowList		*Row1, *Row2;
    size_t		*Nums;
    size_t		Top;
{
    RowList		*r1, *r2;
    Boolean		Duplicate;
    int			i;

    /* FIXME: This routine needs to be cleaned up for efficiency */
    Duplicate = True;
    for (i = 0; i < Top && Duplicate == True; i++) {
	r1 = Row1;
	while (r1 != NULL) {
	    if (Nums[i] == TheEntry[r1->Attribute].AttributeNumber) {
		r2 = Row2;
		while (r2 != NULL) {
		    if (Nums[i] == TheEntry[r2->Attribute].AttributeNumber) {
			if (r1->Attribute != r2->Attribute) {
			    return False;
			}
			break;
		    }
		    r2 = r2->next;
		}
		if (r2 != NULL)
		    break;
	    }
	    r1 = r1->next;
	}
    }
    return Duplicate;
}

int RowListShouldSkipRow(TheEntry, Row, Nums, Top)
    InCoreEntry		*TheEntry;
    RowList		*Row;
    size_t		*Nums;
    size_t		Top;
{
    int			*Match, i;

    Match = (int *)Malloc(sizeof(int)*Top);
    for (i = 0; i < Top; i++)
	Match[i] = -1;
    while (Row != NULL) {
	Boolean		mark;

	mark = TheEntry[Row->Attribute].Marked;
	if (mark != False) {
	    for (i = 0; i < Top; i++) {
		if (Nums[i] == TheEntry[Row->Attribute].AttributeNumber) {
		    Match[i] = Nums[i];
		    break;
		}
	    }
	}
	Row = Row->next;
    }
    for (i = 0; i < Top; i++) {
	if ((int)Nums[i] != Match[i]) {
#if defined(DEBUG)
	    fprintf(stderr, "skipping row\n");
#endif
	    Free(Match);
	    return True;
	}
    }
    Free(Match);
    return False;
}


void RowListDeleteDuplicateRows(TheEntry, Rows, Nums, Top)
    InCoreEntry		*TheEntry;
    RowList		*Rows;
    size_t		*Nums;
    size_t		Top;
{
    RowList		*r1, *rp1, *back;

    while (Rows != NULL && Rows->down != NULL) {
	back = Rows;
	rp1 = Rows->down;
	while (rp1 != NULL) {
	    if (RowListCompareRows(TheEntry, Rows, rp1, Nums, Top) == True) {
		/* Same row, keep the first one */
		r1 = rp1;
		back->down = rp1->down;
		rp1 = r1->down;
		r1->down = NULL;
		Qddb_Free(QDDB_TYPE_ROWLIST, r1);
	    } else {
		rp1 = rp1->down;
		back = back->down;
	    }
	}
	Rows = Rows->down;
    }
}

/* WARNING: RowList processing depends on the list being sorted.  Disaster may
 * occur otherwise.
 */

RowList *RowListProcessEntry(schema, ThisEntry)
    Schema		*schema;
    InCoreEntry		*ThisEntry;
{
    InCoreEntry		*OldThisEntry;
    size_t		i, j;
    RowList		*List = NULL;
    size_t		Attrs[2];
    SchemaNode		*ThisNode;

    OldThisEntry = ThisEntry;
    i = 0;
    while (OldThisEntry->SequenceNumber != 0) {
	OldThisEntry->Index = i++;
	OldThisEntry++;
    }
    for (i = 1; i <= schema->NumberOfAttributes; i++) {
	ThisNode = schema->Entries+i;
	if (ThisEntry->SequenceNumber == 0)
	    break;
	if (ThisNode->Level == 1) {
	    Attrs[1] = ThisNode->Number;
	    j = 0;
	    OldThisEntry = ThisEntry;
	    if (schema->Entries[ThisEntry->AttributeNumber].Level == 1) {
		while (ThisEntry->SequenceNumber != 0 &&
		       ThisEntry->AttributeNumber == i)
		    ThisEntry++, j++;		
	    } else {
		while (ThisEntry->SequenceNumber != 0 &&
		       schema->Entries[ThisEntry->AttributeNumber].AncestorNumber[1] == 
		       ThisNode->Number)
		    ThisEntry++, j++;
	    }
	    if (j == 0)
		continue;
	    if (List == NULL) {
		List = RowListProcessAttr(schema, OldThisEntry, j, Attrs, 1);
	    } else {
		List = RowListCartesianProduct(List, RowListProcessAttr(schema, OldThisEntry, j, Attrs, 1));
	    }
	}
    }
    RowListPruneUntrueRows(&List);
    return List;
}

RowList *RowListProcessAttr(schema, ThisEntry, EntryLength, Attrs, AttrsLength)
    Schema		*schema;
    InCoreEntry		*ThisEntry;
    size_t		EntryLength, Attrs[], AttrsLength;
{
    InCoreEntry		*OldThisEntry;
    size_t		Inst, i, j;
    RowList		*List = NULL;

    /* For each instance in this attribute */
    for (i = 0; i < EntryLength; i+=j) {
	j = 0;
	OldThisEntry = ThisEntry;
	Inst = ThisEntry->Instance[AttrsLength-1];
	while (i+j < EntryLength && ThisEntry->Instance[AttrsLength-1] == Inst)
	    ThisEntry++, j++;
	if (List == NULL)
	    List = RowListProcessInst(schema, OldThisEntry, j, Attrs, AttrsLength);
	else
	    List = RowListUnion(List, RowListProcessInst(schema, OldThisEntry, j, Attrs, AttrsLength));
    }
    return List;
}

RowList *RowListProcessInst(schema, ThisEntry, EntryLength, Attrs, AttrsLength)
    Schema		*schema;
    InCoreEntry		*ThisEntry;
    size_t		EntryLength, Attrs[], AttrsLength;
{
    InCoreEntry		*OldThisEntry;
    RowList 		*List;
    size_t		Inst, i, j;
    size_t		*NewAttrs, NewAttrsLength, First;

    if (schema->Entries[ThisEntry->AttributeNumber].Level == AttrsLength) { /* at a leaf */
	if (ThisEntry->Marked != False) {
	    List = (RowList *)Malloc(sizeof(RowList));
	    List->next = List->down = NULL;
	    if (ThisEntry->Marked == True) {
		List->Flags = ROWLIST_CONTAINS_TRUE;
	    } else
		List->Flags = 0;
	    List->Attribute = ThisEntry->Index;
	} else {
	    List = NULL;
	}
	return List;
    }
    List = NULL;
    First = True;
    NewAttrsLength = AttrsLength+1;
    NewAttrs = (size_t *)alloca(sizeof(size_t)*(NewAttrsLength+1));
    for (i = 1; i <= AttrsLength; i++)
	NewAttrs[i] = Attrs[i];
    Inst = ThisEntry->Instance[AttrsLength-1];
    /* For each sub-attribute in this instance */
    for (i = 0; i < EntryLength; i+=j) {
	j = 0;
	OldThisEntry = ThisEntry;
	if (NewAttrsLength == schema->Entries[ThisEntry->AttributeNumber].Level) {
	    NewAttrs[NewAttrsLength] = 
		schema->Entries[ThisEntry->AttributeNumber].Number;
	    while (i+j < EntryLength && 
		   schema->Entries[ThisEntry->AttributeNumber].Level == NewAttrsLength &&
		   schema->Entries[ThisEntry->AttributeNumber].Number == NewAttrs[NewAttrsLength])
		ThisEntry++, j++;
	} else {
	    NewAttrs[NewAttrsLength] = 
		schema->Entries[ThisEntry->AttributeNumber].AncestorNumber[NewAttrsLength];
	    while (i+j < EntryLength &&
		   schema->Entries[ThisEntry->AttributeNumber].AncestorNumber[NewAttrsLength] == 
		   NewAttrs[NewAttrsLength])
		ThisEntry++, j++;
	}
	if (First == True)
	    List = RowListProcessAttr(schema, OldThisEntry, j, NewAttrs, NewAttrsLength);
	else
	    List = RowListCartesianProduct(List, RowListProcessAttr(schema, OldThisEntry, j, 
								    NewAttrs, NewAttrsLength));
	First = False;
    }
#if defined(ROWDEBUG)
    PrintRowDebug(List);
#endif 
    return List;
}

#if defined(ROWDEBUG)
void PrintRowDebug(List)
    RowList		*List;
{
    RowList		*tmp;

    while (List != NULL) {
	tmp = List;
	printf("{ ");
	while (tmp != NULL) {
	    printf("%d ", tmp->Attribute);
	    tmp = tmp->next;
	}
	printf("}\n");
	List = List->down;
    }
}
#endif

RowList *RowListCopy(List)
    RowList		*List;
{
    RowList		*NewList, *tmp;

    if (List == NULL)
	return NULL;
    tmp = NewList = NULL;
    while (List != NULL) {
	if (NewList == NULL)
	    NewList = tmp = (RowList *)Malloc(sizeof(RowList));
	else {
	    tmp->next = (RowList *)Malloc(sizeof(RowList));
	    tmp = tmp->next;
	}
	tmp->Attribute = List->Attribute;
	tmp->Flags = List->Flags;
	tmp->down = tmp->next = NULL;
	List = List->next;
    }
    return NewList;
}

RowList *RowListCartesianProduct(List1, List2)
    RowList 		*List1, *List2;
{
    RowList	    	*tmp1, *tmp2, *NewList = NULL;
    RowList		*OrigList1 = List1;

    if (List1 == NULL || List2 == NULL) {
	return NULL;
    }
    tmp1 = NULL;
    while (List1 != NULL) {
	tmp2 = List2;
	while (tmp2 != NULL) {
	    RowList		*tmplist;

	    if (NewList == NULL) {
		NewList = tmp1 = RowListCopy(List1);
	    } else {
		tmp1->down = RowListCopy(List1);
		tmp1 = tmp1->down;
		tmp1->down = NULL;
	    }
	    tmplist = tmp1;
	    while (tmplist->next != NULL)
		tmplist = tmplist->next;
	    tmplist->next = RowListCopy(tmp2);
	    tmp2 = tmp2->down;
	}
	List1 = List1->down;
    }
    Qddb_Free(QDDB_TYPE_ROWLIST, OrigList1);
    Qddb_Free(QDDB_TYPE_ROWLIST, List2);
    return NewList;
}

RowList *RowListUnion(List1, List2)
    RowList 		*List1, *List2;
{
    RowList		*tmp;

    if (List1 == NULL)
	return List2;
    if (List2 == NULL)
	return List1;
    tmp = List1;
    while (tmp->down != NULL)
	tmp = tmp->down;
    tmp->down = List2;
    return List1;
}

void RowListPruneUntrueRows(List)
    RowList		**List;
{
    RowList		**list, *tmp;
    Boolean		true_flag;

    if (List == NULL || *List == NULL)
	return;
    list = List;
    while (*list != NULL) {
	true_flag = False;
	tmp = *list;
	while (tmp != NULL) {
	    if ((tmp->Flags & ROWLIST_CONTAINS_TRUE) != 0) {
		true_flag = True;
		break;
	    }
	    tmp = tmp->next;
	}
	if (true_flag == False) {
	    tmp = *list;
	    *list = tmp->down;
	    tmp->down = NULL;
	    Qddb_Free(QDDB_TYPE_ROWLIST, tmp);
	} else {
	    list = &(*list)->down;
	}
    }
}


#if defined(USE_TCL)
static void QueryPrintRow _ANSI_ARGS_((Tcl_Interp *, Schema *, KeyList *, InCoreEntry *, \
				       RowList *, char *, size_t *, size_t *, size_t, size_t *, size_t, \
				       Boolean, Boolean));
static char *PrintRowInstances _ANSI_ARGS_((Schema *, InCoreEntry *, RowList *));
static void QueryPrintRowAsElements _ANSI_ARGS_((Tcl_Interp *, Schema *, KeyList *, InCoreEntry *, \
						 RowList *, char *, size_t *, size_t *, size_t, \
						 size_t *, size_t, Boolean, Boolean));
#else
static void QueryPrintRow _ANSI_ARGS_((Schema *, KeyList *, InCoreEntry *, RowList *, char *, \
				       size_t *, size_t *, size_t, size_t *, size_t, Boolean, Boolean));
#endif

#if defined(USE_TCL)
int Qddb_RowListQuery(interp, schema, list, SepString, 
		      AttrNames, AttrWidth, AttrTop, AttrsUsed, AttrsUsedTop, 
		      Any, TCLprint, DontPrintRows, print_tcl_elements, suppress_printing)
    Tcl_Interp		*interp;
    Schema		*schema;
    KeyList		*list;
    char		*SepString;
    /* printing attributes */
    char		*AttrNames[];
    size_t		*AttrWidth, AttrTop;
    /* attributes used to generate 'list' */
    char		*AttrsUsed[];
    size_t		AttrsUsedTop;
    /* was this an 'Any' search? */
    Boolean		Any;
    Boolean		TCLprint;
    Boolean		DontPrintRows;
    Boolean		print_tcl_elements;
    Boolean		suppress_printing;
#else
int Qddb_RowListQuery(schema, list, SepString, 
		      AttrNames, AttrWidth, AttrTop, AttrsUsed, AttrsUsedTop, 
		      Any, TCLprint)
    Schema		*schema;
    KeyList		*list;
    char		*SepString;
    /* printing attributes */
    char		*AttrNames[];
    size_t		*AttrWidth,AttrTop;
    /* attributes used to generate 'list' */
    char		*AttrsUsed[];
    size_t		AttrsUsedTop;
    /* was this an 'Any' search? */
    Boolean		Any;
    Boolean		TCLprint;
#endif
{
    Entry		ThisEntry = NULL;
    InCoreEntry		*CoreEntry = NULL;
    char		*RelationFN = schema->RelationName;
    int			DBFile, tmpattr;
    size_t		*AttrNums, i;
    size_t		*AttrNumsUsed;
    RowList		*Row;
    KeyList		**SplitList, **OrigSplitList, *tmplist, Key;

    AttrNums = (size_t *)Malloc(sizeof(size_t)*AttrTop);
    AttrNumsUsed = (size_t *)Malloc(sizeof(size_t)*AttrsUsedTop);
    if (list == NULL)
	return 0;
    DBFile = schema->database_fd;
    for (i = 0; i < AttrTop; i++) {
	tmpattr = Qddb_ConvertAttributeNameToNumber(schema, AttrNames[i]);
	if (tmpattr == -1) {
	    Free(AttrNums);
	    Free(AttrNumsUsed);
	    return -1;
	}
	AttrNums[i] = (size_t)tmpattr;
    }
    for (i = 0; i < AttrsUsedTop; i++) {
	tmpattr = Qddb_ConvertAttributeNameToNumber(schema, AttrsUsed[i]);
	if (tmpattr == -1) {
	    Free(AttrNums);
	    Free(AttrNumsUsed);
	    return -1;
	}
	AttrNumsUsed[i] = (size_t)tmpattr;
    }
    /* Here we copy the keylist into a split list; each element of the
     * split list is deleted by MARKFROMINCORE
     */
    (void)Qddb_KeyListProcess(schema, list, &SplitList, QDDB_KEYLIST_PROC_SPLIT, QDDB_KEYLIST_FLAG_COPY);
    OrigSplitList = SplitList;
    while (*SplitList != NULL) {
	tmplist = *SplitList;
#if defined(DEBUG)
	printf("*SplitList == (%d,%d,%d,%s)\n", tmplist->Start, tmplist->Length, tmplist->Attribute,
	       tmplist->Instance);
#endif
	ThisEntry = NULL;
	if (Qddb_ReadEntryByKeyList(DBFile, RelationFN, &ThisEntry, tmplist, True) == -1) { 
	    /* invalid entry */
	    Qddb_Free(QDDB_TYPE_KEYLIST, *SplitList);
	    SplitList++;
	    continue;
	}
	Qddb_ReducedAttrToFullAttr(schema, ThisEntry);
	CoreEntry = (InCoreEntry *)Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, 
						ThisEntry, QDDB_ENTRYTYPE_INTERNAL);
	Qddb_Free(QDDB_TYPE_ENTRY, ThisEntry);
	Key = **SplitList;
	Key.Instance = NULL;
	if (Any == False)
	    (void)Qddb_KeyListProcess(schema, *SplitList, CoreEntry, QDDB_KEYLIST_PROC_MARK, 
				       QDDB_KEYLIST_FLAG_MARKFROMINCORE);
	else { 
	    InCoreEntry 	*e;
	    
	    (void)Qddb_KeyListProcess(schema, *SplitList, CoreEntry, QDDB_KEYLIST_PROC_MARK, 
				       QDDB_KEYLIST_FLAG_MARKFROMINCORE);
	    e = CoreEntry;
	    while (e->SequenceNumber != 0) {
		e->Marked = True;
		e++;
	    }
	}
	/* *SplitList was deleted by Qddb_KeyListProcess */
	Row = RowListProcessEntry(schema, CoreEntry);
#if defined(DEBUG)
	if (Row == NULL) fprintf(stderr, "Row is NULL!!\n");
	else fprintf(stderr, "Row is *not* NULL!!\n");
#endif
	RowListDeleteDuplicateRows(CoreEntry, Row, AttrNums, AttrTop);
#if defined(USE_TCL)
	if (print_tcl_elements || suppress_printing) {
	    QueryPrintRowAsElements(interp, schema, &Key, CoreEntry, Row, SepString, 
				    AttrNums, AttrWidth, AttrTop, 
				    AttrNumsUsed, AttrsUsedTop, DontPrintRows, 
				    suppress_printing);
	} else {
	    QueryPrintRow(interp, schema, &Key, CoreEntry, Row, SepString, AttrNums, AttrWidth, AttrTop, 
			  AttrNumsUsed, AttrsUsedTop, TCLprint, DontPrintRows);
	}
#else
	QueryPrintRow(schema, &Key, CoreEntry, Row, SepString, AttrNums, AttrWidth, AttrTop, 
		 AttrNumsUsed, AttrsUsedTop, TCLprint, False);
#endif
	Qddb_Free(QDDB_TYPE_INCOREENTRY, CoreEntry);
	Qddb_Free(QDDB_TYPE_ROWLIST, 	 Row);
	SplitList++;
    }
    Free(AttrNums);
    Free(AttrNumsUsed);
    Free(OrigSplitList);
    return 0;
}

#if defined(USE_TCL)
static void QueryPrintRow(interp, schema, Key, ThisEntry, Row, SepString, 
			  AttrNums, AttrWidth, AttrTop, AttrNumsUsed, AttrUsedTop,
			  TCLprint, DontPrintRows)
    Tcl_Interp		*interp;
    Schema		*schema;
    KeyList		*Key;
    InCoreEntry		*ThisEntry;
    RowList		*Row;
    char		*SepString;
    size_t		*AttrNums, *AttrWidth, AttrTop;
    size_t		*AttrNumsUsed, AttrUsedTop;
    Boolean		TCLprint;
    Boolean		DontPrintRows;
#else
static void QueryPrintRow(schema, Key, ThisEntry, Row, SepString, 
			  AttrNums, AttrWidth, AttrTop, AttrNumsUsed, AttrUsedTop,
			  TCLprint, DontPrintRows)
    Schema		*schema;
    KeyList		*Key;
    InCoreEntry		*ThisEntry;
    RowList		*Row;
    char		*SepString;
    size_t		*AttrNums, *AttrWidth, AttrTop;
    size_t		*AttrNumsUsed, AttrUsedTop;
    Boolean		TCLprint;
    Boolean		DontPrintRows;
#endif
{
    RowList		*tmp, *thisrow;
    char		*buf_ptr;
    int			i;

    while (Row != NULL) {
	if (RowListShouldSkipRow(ThisEntry, Row, AttrNumsUsed, AttrUsedTop) == True) {
	    Row = Row->down;
	    continue;
	}
	thisrow = Row;
	Qddb_InitBuffer();
	for (i = 0; i < AttrTop; i++) {

	    tmp = Row;
	    while (tmp != NULL) {
		if (AttrNums[i] == ThisEntry[tmp->Attribute].AttributeNumber) {
		    break;
		}
		tmp = tmp->next;
	    }
	    if (tmp == NULL) {
		if (AttrWidth[i] == 0) {
		    Qddb_ConcatBuffer(SepString);
		} else {
		    buf_ptr = Malloc(AttrWidth[i]+strlen(SepString)+2);
		    sprintf(buf_ptr, "%*s%s", (int)AttrWidth[i], " ", SepString);
		    Qddb_ConcatBuffer(buf_ptr);
		    Free(buf_ptr);
		}
	    } else {
		if (AttrWidth[i] != 0) {
		    ThisEntry[tmp->Attribute].Marked = True;
		    Qddb_ConcatBuffer(buf_ptr = Qddb_FieldString(ThisEntry[tmp->Attribute].Data, 
								 (size_t)AttrWidth[i]));
		    Free(buf_ptr);
		    Qddb_ConcatBuffer(SepString);
		} else {
		    ThisEntry[tmp->Attribute].Marked = True;
		    Qddb_ConcatBuffer(buf_ptr = Qddb_FieldString(ThisEntry[tmp->Attribute].Data, 
								 strlen(ThisEntry[tmp->Attribute].Data)));
		    Free(buf_ptr);
		    Qddb_ConcatBuffer(SepString);
		}
	    }
	}
	buf_ptr = Qddb_GetBuffer();
	if (TCLprint == True) {
#if defined(USE_TCL)
	    char		key_buf[BUFSIZ], *row_buf = NULL, *merged_buf;
	    int			nargc = 0;
	    char		*nargv[4];

	    nargv[nargc++] = buf_ptr;
	    sprintf(key_buf, "%d %d %d %d",
		    (int)Key->Start, (int)Key->Length, (int)Key->Number, (int)QDDB_KEYLIST_TYPE(Key));
	    nargv[nargc++] = key_buf;
	    if (DontPrintRows == False) {
		row_buf = nargv[nargc++] = PrintRowInstances(schema, ThisEntry, Row);
	    }
	    nargv[nargc] = NULL;
	    merged_buf = Tcl_Merge(nargc, nargv);
	    Free(buf_ptr);
	    if (row_buf != NULL)
		Free(row_buf);
	    if (interp == NULL)
		printf("{ %s }\n", merged_buf);
	    else
		Tcl_AppendElement(interp, merged_buf);
	    Free(merged_buf);
#else
	    fprintf(stderr, "-tcl option is unusable unless qddb is compiled with \
--with-tcl or --with-tk\n");
	    exit(1);
#endif
	} else {
	    printf("%s\n", buf_ptr);
	    Free(buf_ptr);
	}
	Row = Row->down;
    }
}

#if defined(USE_TCL)
static void QueryPrintRowAsElements(interp, schema, Key, ThisEntry, Row, SepString, 
			  AttrNums, AttrWidth, AttrTop, AttrNumsUsed, AttrUsedTop,
			  DontPrintRows, suppress_printing)
    Tcl_Interp		*interp;
    Schema		*schema;
    KeyList		*Key;
    InCoreEntry		*ThisEntry;
    RowList		*Row;
    char		*SepString;
    size_t		*AttrNums, *AttrWidth, AttrTop;
    size_t		*AttrNumsUsed, AttrUsedTop;
    Boolean		DontPrintRows;
    Boolean		suppress_printing;
{
    RowList		*tmp, *thisrow;
    char		**merge_buf;
    char		*buf_ptr;
    int			i;
    char		key_buf[BUFSIZ], *row_buf = NULL, *merged_buf;
    int			nargc = 0;
    char		*nargv[4];

    while (Row != NULL) {
	if (RowListShouldSkipRow(ThisEntry, Row, AttrNumsUsed, AttrUsedTop) == True) {
	    Row = Row->down;
	    continue;
	}
	thisrow = Row;
	if (!suppress_printing) {
	    merge_buf = (char **)Malloc(sizeof(char *)*(AttrTop+1));
	    for (i = 0; i < AttrTop; i++) {
		tmp = Row;
		while (tmp != NULL) {
		    if (AttrNums[i] == ThisEntry[tmp->Attribute].AttributeNumber) {
			break;
		    }
		    tmp = tmp->next;
		}
		if (tmp == NULL) {
		    if (AttrWidth[i] == 0) {
			merge_buf[i] = Malloc(QDDB_MIN_MALLOC_SIZE);
			*(merge_buf[i]) = '\0';
		    } else {
			buf_ptr = Malloc((size_t)AttrWidth[i]+1);
			sprintf(buf_ptr, "%*s%s", (int)AttrWidth[i], " ", SepString);
			merge_buf[i] = buf_ptr;
		    }
		} else {
		    if (AttrWidth[i] != 0) {
			ThisEntry[tmp->Attribute].Marked = True;
			buf_ptr = Qddb_FieldString(ThisEntry[tmp->Attribute].Data, (size_t)AttrWidth[i]);
			merge_buf[i] = buf_ptr;
		    } else {
			ThisEntry[tmp->Attribute].Marked = True;
			buf_ptr = Qddb_FieldString(ThisEntry[tmp->Attribute].Data, 
						   strlen(ThisEntry[tmp->Attribute].Data));
			merge_buf[i] = buf_ptr;
		    }
		}
	    }
	    merge_buf[AttrTop] = NULL;
	    buf_ptr = Tcl_Merge((int)AttrTop, merge_buf);
	    for (i = 0; i < AttrTop; i++)
		Free(merge_buf[i]);
	    Free(merge_buf);
	} else {
	    buf_ptr = Calloc(QDDB_MIN_MALLOC_SIZE);
	}
	nargv[nargc++] = buf_ptr;
	sprintf(key_buf, "%d %d %d %d",
		(int)Key->Start, (int)Key->Length, (int)Key->Number, (int)QDDB_KEYLIST_TYPE(Key));
	nargv[nargc++] = key_buf;
	if (DontPrintRows == False) {
	    row_buf = nargv[nargc++] = PrintRowInstances(schema, ThisEntry, Row);
	}
	nargv[nargc] = NULL;
	merged_buf = Tcl_Merge(nargc, nargv);
	Free(buf_ptr);
	if (row_buf != NULL)
	    Free(row_buf);
	Tcl_AppendElement(interp, merged_buf);
	Free(merged_buf);
	Row = Row->down;
    }
}


static char *PrintRowInstances(schema, ThisEntry, Row)
    Schema			*schema;
    InCoreEntry			*ThisEntry;
    RowList			*Row;
{
    char			*name, *instance;
    char			buf[BUFSIZ];

    Qddb_InitBuffer();
    while (Row != NULL) {
	name = Qddb_ConvertAttributeNumberToName(schema, (int)ThisEntry[Row->Attribute].AttributeNumber);
	instance = InstanceToString(ThisEntry[Row->Attribute].Instance, 
				(size_t)schema->Entries[ThisEntry[Row->Attribute].AttributeNumber].Level);
	sprintf(buf, "%s,%s ", name, instance);
	Free(name);
	Free(instance);
	Qddb_ConcatBuffer(buf);
	Row = Row->next;
    }
    return Qddb_GetBuffer();
}
#endif
