
/* qddb/Lib/LibQddb/ReducedAttr.c
 *
 * Copyright (C) 1995 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	void Qddb_InitReducedAttrTable(Schema *);
 *      void Qddb_AddToReducedAttrTable(Schema *, char *);
 *      void Qddb_GetReducedAttrTable(Schema *, char *);
 *	ReducedAttributeTable *Qddb_ReadReducedAttrTable(Schema *);
 *	void Qddb_WriteReducedAttrTable(Schema *);
 *	char *Qddb_IndexToReducedAttr(int);
 *	int  Qddb_ReducedAttrToIndex(char *);
 *	void Qddb_FreeReducedAttrTable();
 *	void Qddb_ReducedAttrToFullAttr(Schema *, Entry)
 *	void Qddb_FullAttrToReducedAttr(Schema *, Entry)
 */

#define REDATTRNUM (16*1024)

#if defined(USE_TCL)
static Tcl_HashTable		attr_hashtable;
#endif
static ReducedAttrTable 	*attr_table = NULL;
static size_t 			attr_top;
static size_t 			attr_size;
static char			*alphabet = "0123456789abcdefghijklmnopqrstuvwxyz";
static int			indices[128] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24,
    25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 0, 0, 0, 0, 0
};

void Qddb_InitReducedAttrTable(schema)
    Schema			*schema;
{
    if (schema->UseReducedAttributeIdentifiers == False)	
	return;
#if defined(USE_TCL)
    Tcl_InitHashTable(&attr_hashtable, TCL_STRING_KEYS);
#endif
    attr_table = (ReducedAttrTable *)Calloc(sizeof(ReducedAttrTable)*REDATTRNUM);
    attr_table[0] = Malloc(2);
    attr_table[0][0] = '0';
    attr_table[0][1] = '\0';
    attr_top = 1;
    attr_size = REDATTRNUM;
}

char *Qddb_AddToReducedAttrTable(schema, value)
    Schema			*schema;
    char			*value;
{
#if defined(USE_TCL)
    int				newPtr;
    Tcl_HashEntry		*hash_ptr;
    char			*ch1;
#else
    int				i;
    char			*ch1, *ch2;
#endif

    if (schema->UseReducedAttributeIdentifiers == False)	
	return NULL;
#if defined(USE_TCL)
    hash_ptr = Tcl_FindHashEntry(&attr_hashtable, value);
    if (hash_ptr != NULL) {
	return (char *)Tcl_GetHashValue(hash_ptr);
    }
#else
    for (i = 0; i < attr_top; i++) {
	ch1 = value;
	ch2 = attr_table[i];
	while (*ch1 != '\0') {
	    if (*ch1 != *ch2)
		break;
	    ch1++, ch2++;
	}
	if (*ch1 == *ch2)
	    return Qddb_IndexToReducedAttr(i);
    }
#endif
    if (attr_top >= attr_size) {
	attr_size += REDATTRNUM;
	attr_table = (ReducedAttrTable *)Realloc(attr_table, sizeof(ReducedAttrTable)*attr_size);
    }
    attr_table[attr_top] = Malloc(strlen(value)+1);
    strcpy(attr_table[attr_top], value);
    attr_top++;
#if defined(USE_TCL)
    ch1 = Qddb_IndexToReducedAttr((int)(attr_top-1));
    hash_ptr = Tcl_CreateHashEntry(&attr_hashtable, value, &newPtr);
    Tcl_SetHashValue(hash_ptr, (ClientData)ch1);
    return ch1;
#else
    return Qddb_IndexToReducedAttr((int)(attr_top-1));
#endif
}

ReducedAttrTable *Qddb_GetReducedAttrTable(schema)
    Schema			*schema;
{
    if (schema->UseReducedAttributeIdentifiers == False)
	return NULL;
    return attr_table;
}

/* ReadReducedAttrTable()
 * 	Reads external table.
 */
ReducedAttrTable *Qddb_ReadReducedAttrTable(schema)
    Schema			*schema;
{
    char			redattr_fn[MAXFILENAMELEN];
    int				fd, i, j, numlines;
    size_t			len;
    char			*buf, *ch;
    ReducedAttrTable		*retval;

    strcpy(redattr_fn, schema->RelationName);
    strcat(redattr_fn, "/RedAttrIndex");
    if ((fd = open(redattr_fn, O_RDONLY)) == -1) {
	return NULL;
    }
    len = SizeOfFile(fd);
    if (len == 0) {
	close(fd);
	return NULL;
    }
    ch = buf = Malloc(len);
    if (read(fd, buf, len) != len) {
	PANIC("Cannot read reduced attribute identifier index (RedAttrIndex)");
    }
    close(fd);
    *(ch+len-1) = '\0';
    numlines = 0;
    while (*ch != '\0') {
	if (*ch == '\n') {
	    *ch = '\0';
	    numlines++;
	}
	ch++;
    }
    retval = (ReducedAttrTable *)Calloc(sizeof(ReducedAttrTable)*(numlines+2));
    for (i = 0, j = 0; i < len; i+=(strlen(buf+i)+1), j++) {
	retval[j] = buf + i;
    }
    return retval;
}

/* Qddb_WriteReducedAttrTable()
 * 	Write internal table.
 */
int Qddb_WriteReducedAttrTable(schema)
    Schema			*schema;
{
    int				i, fd, real_exists;
    char			redattr_fn[MAXFILENAMELEN], real_redattr_fn[MAXFILENAMELEN];
    char			tmp_fn[MAXFILENAMELEN];

    if (schema->UseReducedAttributeIdentifiers == False)
	return 0;
    strcpy(redattr_fn, schema->RelationName);
    strcpy(real_redattr_fn, schema->RelationName);
    strcpy(tmp_fn, schema->RelationName);
    strcat(redattr_fn, "/.RedAttrIndex");
    strcat(real_redattr_fn, "/RedAttrIndex");
    strcat(tmp_fn, "/RedAttrIndex.old");
    if ((fd = open(redattr_fn, O_CREAT|O_WRONLY|O_TRUNC, 0666)) == -1) {
	PANIC("Cannot write reduce attribute identifier index (RedAttrIndex)");
    }
    for (i = 0; i < attr_top; i++) {
#if defined(DIAGNOSTIC)
	if (attr_table[i] == NULL || attr_table[i][0] == '\0') {
	    fprintf(stderr, "writing null entry at index %d\n", i);
	}
#endif
	if (write(fd, attr_table[i], strlen(attr_table[i])) < 0)
	    return -1;
	if (write(fd, "\n", 1) < 0)
	    return -1;
    }
    close(fd);
#if defined(HAVE_RENAME)
    if ((real_exists = access(tmp_fn, F_OK)) != -1 && rename(real_redattr_fn, tmp_fn) == -1) {
	return -1;
    }	    
    if (rename(redattr_fn, real_redattr_fn) == -1) {
	return -1;
    }
    if (real_exists != -1 && unlink(tmp_fn) == -1) {
	fprintf(stderr, "Warning: Failed to remove %s file.\nContact your system administrator.\n", tmp_fn);
    }
#else
    if ((real_exists = access(tmp_fn, F_OK) != -1) && link(real_redattr_fn, tmp_fn) == -1) {
	fprintf(stderr, "Link of %s to %s failed, RedAttrIndex not modified\n",
		real_redattr_fn, tmp_fn);
	return -1;
    }
    if (real_exists != -1 && unlink(real_redattr_fn) == -1) {
	fprintf(stderr, "Unlink of %s failed, RedAttrIndex not modified\n",
		real_redattr_fn);
	unlink(tmp_fn);
	return -1;
    }
    if (link(redattr_fn, real_redattr_fn) == -1) {
	fprintf(stderr, "Link of %s to %s failed, RedAttrIndex not modified\n",
		redattr_fn, real_redattr_fn);
	fprintf(stderr, "Please remedy situation and try again\n");
	return -1;
    }
    if (real_exists != -1 && unlink(tmp_fn) == -1) {
	fprintf(stderr, "Warning: Failed to remove %s file.\nContact your system administrator.\n", tmp_fn);
    }
    if (unlink(redattr_fn) == -1) {
	fprintf(stderr, "Unlink of %s failed, please remove manually\n", redattr_fn);
    }
#endif
    return 0;
}

char *Qddb_IndexToReducedAttr(index)
    int				index;
{
    char			ch[sizeof(int)*3], *retval;
    int				i, j;

    bzero(ch, sizeof(int)*3);
    i = 0;
    if (index == 0) {
	retval = Malloc(2);
	retval[0] = '0';
	retval[1] = '\0';
	return retval;
    }
    while (index > 0) {
	ch[i++] = alphabet[index % 36];
	index  /= 36;
    }
    retval = Malloc((size_t)i+1);
    retval[i] = '\0';
    for (j = 0; j < i; j++) {
	retval[j] = ch[i-j-1];
    }
    return retval;
}

int Qddb_ReducedAttrToIndex(redattr)
    char			*redattr;
{
    int				i, len, retval, digit;

    len = strlen(redattr);
    retval = 0;
    for (i = 0; i < len; i++) {
	digit = indices[(int)redattr[i]];
	retval *= 36;
	retval += digit;
    }
    return retval;
}

/* Qddb_FreeReducedAttrTable()
 *	Frees internal table.
 */
void Qddb_FreeReducedAttrTable()
{
    int				i;

    for (i = 0; i < attr_top; i++)
	Free(attr_table[i]);
    Free(attr_table);
}

/* Qddb_ReducedAttrToFullAttr
 *	Convert from reduced attributes to full attributes.
 */
void Qddb_ReducedAttrToFullAttr(schema, this_entry)
    Schema			*schema;
    Entry			this_entry;
{
    char			*freebuf, *ch, *endptr, *fullattr;

    if (this_entry == NULL || *this_entry == NULL || schema->ReducedAttributeTable == NULL)
	return;
    while (*this_entry != NULL) {
	freebuf = ch = *this_entry;
	ch++;
	endptr = ch;
	while (*endptr != ' ') {
	    if (*endptr == '.')
		return;
	    endptr++;
	}
	*endptr = '\0';
	fullattr = schema->ReducedAttributeTable[Qddb_ReducedAttrToIndex(ch)];
	*this_entry = Malloc(strlen(fullattr)+strlen(endptr+1)+3);
	strcpy(*this_entry, "%");
	strcat(*this_entry, fullattr);
	strcat(*this_entry, " ");
	strcat(*this_entry, endptr+1);
	Free(freebuf);
	this_entry++;
    }
}

/* Qddb_FullAttrToReducedAttr
 *	For stabilization only.
 */
void Qddb_FullAttrToReducedAttr(schema, this_entry)
    Schema			*schema;
    Entry			this_entry;
{
    char			*freebuf, *ch, *endptr, *redattr;

    if (*this_entry == NULL || attr_table == NULL || schema->UseReducedAttributeIdentifiers == False)
	return;
    while (*this_entry != NULL) {
	freebuf = ch = *this_entry;
	ch++;
	endptr = ch;
	while (*endptr != ' ')
	    endptr++;
	*endptr = '\0';
	redattr = Qddb_AddToReducedAttrTable(schema, ch);
#if defined(DIAGNOSTIC)
	if (redattr == NULL)
	    PANIC("Qddb_AddToReducedAttrTable returned NULL");
#endif
	*this_entry = Malloc(strlen(redattr)+strlen(endptr+1)+3);
	strcpy(*this_entry, "%");
	strcat(*this_entry, redattr);
#if !defined(USE_TCL)
	Free(redattr);
#endif
	strcat(*this_entry, " ");
	strcat(*this_entry, endptr+1);
	Free(freebuf);
	this_entry++;
    }    
}

