
/* Lib/LibQddb/Memory.c -- Miscellaneous memory-handling functions.
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

#if defined(QDDB_MEM_DEBUG)
/* Qddb memory checking routines
 */
#define QDDB_MEM_DEBUG_MAX 200000
static void *mem_debug_array[QDDB_MEM_DEBUG_MAX];
static int  mem_debug_num = 0;

static void memdebug_addval(arg)
    void		*arg;
{
    if (arg == NULL)
	abort();
    if (mem_debug_num > QDDB_MEM_DEBUG_MAX)
	abort();
    mem_debug_array[mem_debug_num++] = arg;
}

static void memdebug_check(arg)
    void		*arg;
{
    int			i;

    if (arg == NULL)
	abort();
    for (i = 0; i < mem_debug_num; i++)
	if (arg == mem_debug_array[i])
	    break;
    if (i == mem_debug_num)
	abort();
}

static void memdebug_delval(arg)
    void		*arg;
{
    int			i;

    if (arg == NULL)
	abort();
    for (i = 0; i < mem_debug_num; i++)
	if (arg == mem_debug_array[i]) {
	    mem_debug_array[i] = NULL;
	    break;
	}
    if (i == mem_debug_num)
	abort();
}
#endif

static void FreeKeyList _ANSI_ARGS_((KeyList *));
static void FreeEntry _ANSI_ARGS_((Entry));
static void FreeRowList _ANSI_ARGS_((RowList *));
static void FreeSingleRow _ANSI_ARGS_((RowList *));
static void FreeInCoreEntry _ANSI_ARGS_((InCoreEntry *));
static void FreeSchema _ANSI_ARGS_((Schema *));
static void FreeDataTree _ANSI_ARGS_((DataTree **));
static void FreeSchemaTree _ANSI_ARGS_((SchemaTreeNode **));
static void FreeDoubleList _ANSI_ARGS_((Qddb_DoubleList *));
static void FreeTableCalc _ANSI_ARGS_((Qddb_TableCalc *));
static void FreeTableRefs _ANSI_ARGS_((Qddb_TableDependencies *));
static void FreeWordList _ANSI_ARGS_((WordList *));
static void FreeAttrs _ANSI_ARGS_((Qddb_AttrList *));
static void FreeAttrList _ANSI_ARGS_((Qddb_AttrHead *));

#if !defined(USE_GNU_MALLOC)
size_t		qddb_malloc_bytes_used = 0;
#endif


#if defined(USE_MALLOC_FUNCTIONS)
char *Malloc(Size)
    size_t			Size;
{
    char			*ReturnValue;

#if defined(BIG_DEBUG)
    if (Size > (1024*1024)) {
	PANIC("Malloc: Size too big, dumping for analysis");
    }
#endif
    if (Size < QDDB_MIN_MALLOC_SIZE) {
#if !defined(USE_GNU_MALLOC)
	qddb_malloc_bytes_used += QDDB_MIN_MALLOC_SIZE;
#endif
	Size = QDDB_MIN_MALLOC_SIZE;
    }
    if ((ReturnValue = malloc(Size)) == NULL) {
    	char 		buf[BUFSIZ];

    	sprintf(buf, "Malloc(): maximum size exceeded");
    	perror(buf);
	PANIC("Malloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += Size;
#endif
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Malloc: (size %d) %x\n", Size, ReturnValue);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}


char *Calloc(Size)
    size_t		Size;
{
    char		*ReturnValue; 

    if (Size <= QDDB_MIN_MALLOC_SIZE) {
#if !defined(USE_GNU_MALLOC)
	qddb_malloc_bytes_used += QDDB_MIN_MALLOC_SIZE;
#endif
	return calloc((size_t)QDDB_MIN_MALLOC_SIZE, (size_t)1);
    }
    if ((ReturnValue = calloc(Size, (size_t)1)) == NULL) {
    	char 		buf[BUFSIZ];

    	sprintf(buf, "Calloc(): Maximum core size exceeded");
    	perror(buf);
	PANIC("Calloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += Size;
#endif
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Malloc: (size %d) %x\n", Size, ReturnValue);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}

char *Realloc(OldPtr, NewSize)
    void       		*OldPtr;
    size_t		NewSize;
{
    char		*ReturnValue;

#if defined(DIAGNOSTIC)
    if (NewSize == 0) {
	fprintf(stderr, "reallocing zero length\n");
	fflush(stderr);
	abort();
    }
#endif
    if (NewSize < QDDB_MIN_MALLOC_SIZE)
	NewSize = QDDB_MIN_MALLOC_SIZE;
    if (OldPtr == NULL) {
	if ((ReturnValue = malloc(NewSize)) == NULL) {
	    PANIC("Realloc");
	}
    } else if ((ReturnValue = realloc(OldPtr, NewSize)) == NULL) {
    	perror("Realloc():");
    	if (Malloc(NewSize) != NULL)
    		printf("Malloc worked though!\n");
	PANIC("Realloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += NewSize;
#endif
#if defined(MALLOC_DEBUG)
    if (ReturnValue != OldPtr) {
	fprintf(stderr, "Malloc: (Size %d) %x\n", NewSize, ReturnValue);
	fprintf(stderr, "Free: %x\n", OldPtr);
    }
#endif
#if defined(QDDB_MEM_DEBUG)
    if (ReturnValue != OldPtr && OldPtr != NULL)
	memdebug_delval(OldPtr);
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}

void Free(ptr)
    void		*ptr;
{
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Free: %x\n", ptr);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_delval(ptr);
#endif
    if (ptr != NULL)
	free(ptr);
}

#endif /* USE_MALLOC_FUNCTIONS */

void Qddb_Free(type, ptr)
    int			type;
    void		*ptr;
{
    if (ptr == NULL)
	return;
    switch (type) {
    case QDDB_TYPE_GENERIC:
	Free((void *)ptr);
	break;
    case QDDB_TYPE_ENTRY:
	FreeEntry((Entry)ptr);
	break;
    case QDDB_TYPE_INCOREENTRY:
	FreeInCoreEntry((InCoreEntry *)ptr);
	break;
    case QDDB_TYPE_KEYLIST:
	FreeKeyList((KeyList *)ptr);
	break;
    case QDDB_TYPE_ROWLIST:
	FreeRowList((RowList *)ptr);
	break;
    case QDDB_TYPE_DATATREE:
	FreeDataTree((DataTree **)ptr);
	break;
    case QDDB_TYPE_SCHEMA:
	FreeSchema((Schema *)ptr);
	break;
    case QDDB_TYPE_SCHEMATREE:
	FreeSchemaTree((SchemaTreeNode **)ptr);
	break;
    case QDDB_TYPE_DOUBLELIST:
	FreeDoubleList((Qddb_DoubleList *)ptr);
	break;
    case QDDB_TYPE_TABLECALC:
	FreeTableCalc((Qddb_TableCalc *)ptr);
	break;
    case QDDB_TYPE_TABLEREFS:
	FreeTableRefs((Qddb_TableDependencies *)ptr);
	break;
    case QDDB_TYPE_WORDLIST:
	FreeWordList((WordList *)ptr);
	break;
    case QDDB_TYPE_ATTRLIST:
	FreeAttrList((Qddb_AttrHead *)ptr);
	break;
    default:
	printf("DEFAULT FREE, type %d\n", type);
	Free(ptr);	
    }

}

static void FreeSchema(schema)
    Schema		*schema;
{
    int			i;
    size_t		num;
    SchemaNode		*nodes;

#if defined(DIAGNOSTIC)
    if (schema == NULL)
	return;
#endif
    Free(schema->RelationName);
    if (schema->HashTable != NULL)
	Free(schema->HashTable);
    if (schema->ReducedAttributeTable != NULL) {
	Free(schema->ReducedAttributeTable[0]);
	Free(schema->ReducedAttributeTable);
    }
    if (schema->word_list != NULL) {
	FreeWordList(schema->word_list);
    }
    num = schema->NumberOfAttributes;
    nodes = schema->Entries;
    for (i = 1; i <= num; i++) {
	if (nodes[i].Name != NULL)
	    Free(nodes[i].Name);
	if (nodes[i].VerboseName != NULL)
	    Free(nodes[i].VerboseName);
	if (nodes[i].Alias != NULL)
	    Free(nodes[i].Alias);
	if (nodes[i].Format != NULL)
	    Free(nodes[i].Format);
	if (nodes[i].Separators != NULL)
	    Free(nodes[i].Separators);
#if !defined(USE_TCL)
	if (nodes[i].Attribute != NULL)
	    Free(nodes[i].Attribute);
#endif
    }
    Free(nodes);
    FreeSchemaTree(schema->Tree);
    Free(schema->default_date_format);
#if defined(USE_TCL)
    Tcl_DeleteHashTable(&(schema->TclHashTable));
#endif
    Free(schema);
}

static void FreeSchemaTree(tree)
    SchemaTreeNode	**tree;
{
    int			i;

#if defined(DIAGNOSTIC)
    if (tree == NULL)
	return;
#endif
    for (i = 0; tree[i] != NULL; i++) {
	FreeSchemaTree(tree[i]->children);
	Free(tree[i]);
    }
    Free(tree);
}

static void FreeDataTree(tree)
    DataTree		**tree;
{
    DataTree		**i, *j;

    i = tree;
    while (*i != NULL) {
	for (j = *i; j->datatree_type != DATATREE_END; j++) {
	    switch (j->datatree_type) {
	    case DATATREE_CHILDREN:
		if (j->datatree_children != NULL)
		    FreeDataTree(j->datatree_children);
		break;
	    case DATATREE_DATE:
		if (j->datatree_date != NULL)
		    Free(j->datatree_date);
		break;
	    case DATATREE_STRING:
		if (j->datatree_string != NULL)
		    Free(j->datatree_string);
		break;
	    case DATATREE_NOVAL:
		if (j->datatree_string != NULL)
		    Free(j->datatree_string);
		break;
	    default:
		;
	    }
	}
	Free(*i);
	i++;
    }
    Free(tree);
}

static void FreeKeyList(list)
    KeyList		*list;
{
    KeyList		*tmp;

    while (list != NULL) {
	tmp = list;
	list = list->next;
	Free(tmp->Instance);
	Free(tmp);
    }
}

static void FreeEntry(ThisEntry)
    Entry		ThisEntry;
{
    int			i;

    for (i = 0; ThisEntry[i] != NULL; i++)
	Free(ThisEntry[i]);
    Free(ThisEntry);
}

static void FreeSingleRow(Row)
    RowList		*Row;
{
    RowList		*r;

    while (Row != NULL) {
	r = Row;
	Row = Row->next;
	Free(r);
    }
}

static void FreeRowList(Row)
    RowList		*Row;
{
    RowList		*r;

    while (Row != NULL) {
	r = Row;
	Row = Row->down;
	FreeSingleRow(r);
    }
}

static void FreeInCoreEntry(entry)
    InCoreEntry		*entry;
{
    InCoreEntry		*tmp;

    if (entry == NULL)
	return;
    tmp = entry;
    while (tmp->SequenceNumber != 0) {
	Free(tmp->Data);
	tmp++;
    }
    Free(entry);
}

static void FreeDoubleList(list)
    Qddb_DoubleList		*list;
{
    Qddb_DoubleList		*ptr;

    if (list == NULL)
	return;
    ptr = list->next;
    while (ptr != NULL) {
	Free(list);
	list = ptr;
	ptr = ptr->next;
    }
    Free(list);
}

static void FreeTableCalc(compiled)
    Qddb_TableCalc		*compiled;
{
    Qddb_TableCalc		*tmp;

    tmp = compiled;
    while (tmp->op != 0) {
	if (tmp->table != NULL)
	    Free(tmp->table);
	if (tmp->strval != NULL)
	    Free(tmp->strval);
	tmp++;
    }
    Free(compiled);
}

static void FreeTableRefs(refs)
    Qddb_TableDependencies	*refs;
{
    Qddb_TableDependencies	*tmp;

    while (refs != NULL) {
	if (refs->table != NULL)
	    Free(refs->table);
	if (refs->cell.row != NULL)
	    Free(refs->cell.row);
	if (refs->cell.col != NULL)
	    Free(refs->cell.col);
	if (refs->depends.from.row != NULL)
	    Free(refs->depends.from.row);
	if (refs->depends.from.col != NULL)
	    Free(refs->depends.from.col);
	if (refs->depends.to.row != NULL)
	    Free(refs->depends.to.row);
	if (refs->depends.to.col != NULL)
	    Free(refs->depends.to.col);
	tmp = refs;
	refs = refs->next;
	Free(tmp);
    }
}

static void FreeWordList(wordlist)
    WordList			*wordlist;
{
    WordList			*ptr = wordlist;

    if (wordlist == NULL)
	return;
    while (ptr->word != NULL) {
	Free(ptr->word);
	FreeKeyList(ptr->list);
	ptr++;
    }
    Free(wordlist);
}

static void FreeAttrs(attrs)
    Qddb_AttrList              *attrs;
{
    Qddb_AttrList              *ptr;

    while (attrs != NULL) {
	ptr = attrs->next;
	Free(attrs);
	attrs = ptr;
    }
}

static void FreeAttrList(attrlist)
    Qddb_AttrHead              *attrlist;
{
    int                        i;

    for (i = 0; i < attrlist->Number; i++) {
	FreeAttrs(attrlist->Nodes[i]);
    }
    Free(attrlist->Nodes);
}
