
/* qddb/Lib/LibQddb/Hash.c
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	void Qddb_InitHash(Schema *)
 *	void Qddb_MakeHash(Schema *, char *)
 * 	int Qddb_HashValue(char *)
 *	unsigned int Qddb_GetHashTableEntry(Schema *, size_t)
 */

static int Qddb_ReadHashEntry _ANSI_ARGS_((Schema *, size_t, size_t));


void Qddb_InitHash(schema)
    Schema		*schema;
{
    if (schema->UseCachedHashing == False) {
	schema->HashTable = (Hash *)Calloc(sizeof(Hash)*schema->HashSize);
    } else {
	schema->HashTable = (Hash *)Calloc(sizeof(Hash)*schema->CacheSize);
    }
}

int Qddb_HashValue(schema, string)
    Schema		*schema;
    char		*string;
{

    unsigned int	RetVal;
    static unsigned int	hash_slots[] = {
	2, 3, 5, 7, 13, 17, 19, 31, 61, 89, 107, 127, 521, 607,
	1279, 2203, 2281, 3217, 4253, 4423, 9689, 9941, 11213,
	19937, 21701, 23209, 44497, 86243, 110503, 132049, 216091
    };
    static unsigned int	number_hashslots = sizeof(hash_slots)/sizeof(int);

    RetVal = 0;
    if (schema->HashType == QDDB_HASHTYPE_MOD1) {
	unsigned int	i, nch;
	char		*ch;

	ch = string;
	i = 0;
	while (*ch) {
	    nch = (unsigned int)(*ch);
	    nch += (nch>>3);
	    RetVal += (nch + nch*hash_slots[((i+1)*7)%number_hashslots]);
	    ch++, i++;	    
	}
	RetVal += i*31;
	return (int)(RetVal % schema->HashSize);
    } else { /* default (0 or invalid #) */
	int		len;

	len = strlen(string);
	if (len > 0)
	    RetVal = (int)string[0]*17576;
	else
	    return 0;
	if (len > 1)
	    RetVal += (int)string[1]*676;
	else
	    return RetVal % schema->HashSize;
	if (len > 2)
	    RetVal += (int)string[2]*26;
	else
	    return RetVal % schema->HashSize;
	if (len > 3)
	    RetVal += (int)string[3];
	return RetVal % schema->HashSize;
    }
}

void Qddb_MakeHash(schema, RelationFN)
    Schema		*schema;
    char		*RelationFN;
{
    char		HashFN[MAXFILENAMELEN], *Buffer;
    char		*Line, *EndOfLine;
    int			HashFile = schema->hashtable_fd, base;
    size_t		num, place;
    off_t		SizeOfHashFile;

    if (schema->UseCachedHashing == True || qddb_stabilizing == True)
	return;
    if (schema->UseCondensedIndexing == False) {
	base = 10;
    } else {
	base = 62;
    }
    strcpy(HashFN, RelationFN);
    strcat(HashFN, "/HashTable");
    SizeOfHashFile = SizeOfFile(HashFile);
    if (SizeOfHashFile == 0)
	return;
    Buffer = Malloc(sizeof(char)*(size_t)SizeOfHashFile);
    Read(HashFile, Buffer, (size_t)SizeOfHashFile);
    Line = EndOfLine = Buffer; 
    while (EndOfLine < Buffer+SizeOfHashFile) {
	SkipSpaces(&EndOfLine);
	Line = EndOfLine;
	while (!isspace(*EndOfLine))
	    EndOfLine++;
	*EndOfLine = '\0';
	EndOfLine++;
	SkipSpaces(&EndOfLine);
	num = Qddb_BaseToSize(Line, base);
	Line = EndOfLine;
	while (!isspace(*EndOfLine))
	    EndOfLine++;
	*EndOfLine = '\0';
	EndOfLine++;
	SkipSpaces(&EndOfLine);
	place = Qddb_BaseToSize(Line, base);
	Line = EndOfLine;
	schema->HashTable[place].num = num;
	while (!isspace(*EndOfLine))
	    EndOfLine++;
	*EndOfLine = '\0';
	EndOfLine++;
	SkipSpaces(&EndOfLine);
	schema->HashTable[place].pos = Qddb_BaseToOffset(Line, base);
	Line = EndOfLine;
	while (!isspace(*EndOfLine))
	    EndOfLine++;
	*EndOfLine = '\0';
	EndOfLine++;
	SkipSpaces(&EndOfLine);
	schema->HashTable[place].len = Qddb_BaseToSize(Line, base);
	Line = EndOfLine;
    }
    Free(Buffer);
}


int Qddb_GetHashTableEntry(schema, HashVal)
    Schema			*schema;
    size_t			HashVal;
{
    size_t			i;
    static int			NextCacheEntryToPurge = -1;

    if (schema->UseCachedHashing == False)
	return HashVal;
    for (i = 0; i < schema->TopOfCache; i++) {
	if (schema->HashTable[i].index == HashVal)
	    break;
    }
    if (i == schema->TopOfCache) { /* Entry not in cache */
	if (schema->TopOfCache == schema->CacheSize) {
	    if (NextCacheEntryToPurge >= schema->CacheSize) {
		if (Qddb_ReadHashEntry(schema, 0, HashVal) == -1)
		    return -1;
		i = 0;
		NextCacheEntryToPurge = 1;
	    } else {
		if (Qddb_ReadHashEntry(schema, (size_t)NextCacheEntryToPurge, HashVal) == -1)
		    return -1;
		i  = NextCacheEntryToPurge;
		NextCacheEntryToPurge++;
	    }
	} else {
	    if (Qddb_ReadHashEntry(schema, i, HashVal) == -1) {
		return -1;
	    }
	    schema->TopOfCache++;
	}
    }
    return i;
}

static int Qddb_ReadHashEntry(schema, CacheVal, HashVal)
    Schema		*schema;
    size_t		CacheVal, HashVal;
{
    char		HashLine[BASE10_HASHLINESIZE+1];
    Hash		*HashEntry = &schema->HashTable[CacheVal];
    int			HashFile = schema->hashtable_fd, base;
    size_t              hashlinesize, entrysize;

    if (SizeOfFile(HashFile) == 0) {
	return -1;
    }
    if (schema->UseCondensedIndexing == False) {
	base = 10;
	hashlinesize = BASE10_HASHLINESIZE;
    } else {
	base = 62;
	hashlinesize = BASE62_HASHLINESIZE;
    }
    entrysize = hashlinesize / 4;
    lseek(HashFile, (off_t)(hashlinesize*HashVal), 0);
    Read(HashFile, HashLine, hashlinesize);
    HashLine[hashlinesize] = '\0';
    HashLine[entrysize-1]  = '\0';
    HashLine[(entrysize*2)-1]  = '\0';
    HashLine[(entrysize*3)-1]  = '\0';
    HashLine[(entrysize*4)-1]  = '\0';
    HashEntry->num = Qddb_BaseToSize(HashLine, base);
    HashEntry->index = Qddb_BaseToSize(HashLine+entrysize, base);
    HashEntry->pos = Qddb_BaseToOffset(HashLine+(2*entrysize), base);
    HashEntry->len = Qddb_BaseToSize(HashLine+(3*entrysize), base);
    return 0;
}

