
/* TclQddb_Util.c - Misc utility routines.
 *
 * Copyright (C) 1995 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

/* qddb_util --
 *	qddb_util isdate <value>
 *	qddb_util formatdate <format> <value>
 *      qddb_util datestyle ?us|european?
 *      qddb_util isregexp <value>
 *      qddb_util flock <filedesc>
 */

static char *commands_string = "isdate, formatdate, isregexp, or flock";

int TclQddb_UtilProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 2 || argc > 4) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (strcmp(argv[1], "isdate") == 0 && argc == 3) {
	if (Qddb_CheckDate(argv[2]) != 0) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "formatdate") == 0 && argc == 4) {
	/* qddb_util formatdate format date */
	char		*date, *time;

	time = Qddb_DateStringToTime(argv[3]);
	if (strlen(time) == 0) {
	    Free(time);
	    Tcl_AppendResult(interp, argv[0], ": date conversion failed, bad input", NULL);
	    return TCL_ERROR;
	}
	date = Qddb_DateTimeToString(time, argv[2]);
	Free(time);
	if (strlen(date) >= 1) {
	    Tcl_SetResult(interp, date, TCL_DYNAMIC);
	} else {
	    Free(date);
	    Tcl_AppendResult(interp, argv[0], ": date conversion failed, bad format", NULL);
	    return TCL_ERROR;
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "datestyle") == 0) {
	if (argc == 2) {
	    if (qddb_datestyle == QDDB_DATE_STYLE_US) {
		Tcl_SetResult(interp, "us", TCL_STATIC);
		return TCL_OK;
	    } else {
		Tcl_SetResult(interp, "european", TCL_STATIC);
		return TCL_OK;
	    }
	} else if (argc == 3) {
	    if (strcmp(argv[2], "us") == 0) {
		qddb_default_date_format = "%m/%d/%y";
		qddb_datestyle = QDDB_DATE_STYLE_US;
	    } else if (strcmp(argv[2], "european") == 0) {
		qddb_default_date_format = "%d/%m/%y";
		qddb_datestyle = QDDB_DATE_STYLE_EUROPEAN;
	    } else {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad date style \"", 
				 argv[2], "\"", NULL);
		return TCL_ERROR;
	    }
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	    return TCL_ERROR;
	}
    } else if (strcmp(argv[1], "isregexp") == 0 && argc == 3) {
	regexp                   *reg;

	reg = regcomp(argv[2]);
	if (reg == NULL) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	} else {
	    Free(reg);
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "flock") == 0 && argc == 4) {
	FILE			*fp;
	int			flags = 0;
	char			*ptr;

#if TCL_MAJOR_VERSION >= 4 && TCL_MINOR_VERSION >= 1
	if (Tcl_GetOpenFile(interp, argv[2], 0, 0, (ClientData)&fp) != TCL_OK) {
#else
	if (Tcl_GetOpenFile(interp, argv[2], 0, 0, &fp) != TCL_OK) {
#endif
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot get open file \"", 
			     argv[2], "\"", NULL);
	    return TCL_ERROR;
	}
	ptr = strtok(argv[3], "|");
	while (ptr != NULL) {
	    if (strcmp(ptr, "LOCK_SH") == 0) {
		flags |= LOCK_SH;
	    } else if (strcmp(ptr, "LOCK_EX") == 0) {
		flags |= LOCK_EX;
	    } else if (strcmp(ptr, "LOCK_NB") == 0) {
		flags |= LOCK_NB;
	    } else if (strcmp(ptr, "LOCK_UN") == 0) {
		flags |= LOCK_UN;
	    } else {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad option \"", ptr, "\"", NULL);
		return TCL_ERROR;
	    }
	    ptr = strtok(NULL, "|");
	}
#if defined(HAVE_FLOCK)
	if (flock(fileno(fp), flags) != 0) {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	}
#else
	if ((flags&LOCK_UN) != 0) {
	    if (UnlockSection(fileno(fp), 0, 0) != 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		Tcl_SetResult(interp, "1", TCL_STATIC);
	    }
	} else {
	    if (LockSection(fileno(fp), (flags&LOCK_SH) != 0? F_RDLCK:F_WRLCK, 0, 0, 
			    (flags&LOCK_NB) != 0? 0:1) != 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		Tcl_SetResult(interp, "1", TCL_STATIC);
	    }
	}
#endif
	return TCL_OK;
    }
    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\": should be ",
		     commands_string, NULL);
    return TCL_ERROR;
}
