
/* TclTable.c - TCL interface routines for manipulating tables and.
 * 	performing calculations on cells.
 *
 * Copyright (C) 1996 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static Tcl_HashTable 	TclQddb_TableHashTable;
static int		TclQddb_TableHashTableInit = 0;
static unsigned int	TclQddb_TableNextNumber = 0;
static Tcl_HashTable	TclQddb_TableNameHashTable;
static int		TclQddb_TableNameHashTableInit = 0;

static int TclQddb_ProcessTableDefine _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ProcessTableDelete _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ProcessTableSort   _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_SetTableName       _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, char *));
static void TclQddb_DeleteTableName   _ANSI_ARGS_((Tcl_Interp *, char *));

static int TclQddb_TableDefine _ANSI_ARGS_((Tcl_Interp *, size_t, size_t, \
					    Qddb_TableHeaderInfo *, int, int));
static int TclQddb_TableClear _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableSetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableGetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableGetVal_list _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, int, int));
static int TclQddb_TableGetVal_text _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, int, int));
static int TclQddb_TableGetVal_rows _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, int, int));
static int TclQddb_TableCget _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableConfigure _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableCopy _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableExpr _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableExprOk _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableEval _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_TableSummary _ANSI_ARGS_((Tcl_Interp *, int, char **));

static int TclQddb_ProcessRowCommands _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowInsert _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowDelete _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowSetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowGetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowCget _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowConfigure _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowClear _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowCopy _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_RowEval _ANSI_ARGS_((Tcl_Interp *, int, char **));

static int TclQddb_ProcessColCommands _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColInsert _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColDelete _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColSetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColGetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColCget _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColConfigure _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColClear _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColCopy _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_ColEval _ANSI_ARGS_((Tcl_Interp *, int, char **));

static int TclQddb_ProcessCellCommands _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellSetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellGetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellCget _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellConfigure _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellClear _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellCopy _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_CellEval _ANSI_ARGS_((Tcl_Interp *, int, char **));

static int TclQddb_SetTable _ANSI_ARGS_((Tcl_Interp *, char *, Qddb_Table *));

static char *TclQddb_HeaderGetInfo _ANSI_ARGS_((Qddb_TableHeaderInfo *, Qddb_TableHeaderInfo *, char *));
static char *TclQddb_GetType _ANSI_ARGS_((Qddb_TableNode *, int));
static int TclQddb_SetCellValue _ANSI_ARGS_((Tcl_Interp *, Qddb_TableNode *, int, char *));
static char ***TclQddb_TableGetStrings _ANSI_ARGS_((Qddb_Table *));

static int TclQddb_RelinkTable _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *));
static int TclQddb_RelinkRow _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t));
static int TclQddb_RelinkCol _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t));
static int TclQddb_RelinkCell _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t, size_t, char *));

static void TclQddb_UnlinkTable _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *));
static void TclQddb_UnlinkRow _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t));
static void TclQddb_UnlinkCol _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t));
static void TclQddb_UnlinkCell _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t, size_t, char *));

static void TclQddb_FreeStringAndCalcData _ANSI_ARGS_((Qddb_TableNode *, int));

/* Still to do:
 *
 * 1. Calculated fields: autoevaluation
 */


/* TclQddb_TableProc -- main parsing routine for all table commands
 *
 * qddb_table define <rows> <columns> ?-variable <variable>?
 *     ?-order row|column? ?-autoeval on|off? 
 *     ?-name <table name>? ?-title <title>? ?-comment <comment>?
 * qddb_table delete <table_desc>|all
 * qddb_table clear <table_desc>
 * qddb_table copy <table_desc>
 * qddb_table eval <table_desc>
 * qddb_table getval <table_desc> ?-format list|text|rows?
 *	?-rowtitles on|off? ?-coltitles on|off?
 * qddb_table setval <table_desc> <list>
 * qddb_table cget <table_desc> ?options?
 * qddb_table configure <table_desc> ?-variable <variable>? ?-order <order>?
 * qddb_table sort <table_desc> ?-start <row name>? ?-end <row name>?
 *	?-ascending <column list>? ?-sortby <column list>?
 *
 * qddb_table row insert <table_desc> ?-before <row name>? ?-numrows number?
 * qddb_table row delete <table_desc> <row name>
 * qddb_table row clear <table_desc> <row name>
 * qddb_table row copy <table_desc> <from> <to>
 * qddb_table row maxnum <table_desc>
 * qddb_table row getval <table_desc> <row name>
 * qddb_table row setval <table_desc> <row name> <value list>
 * qddb_table row cget <table_desc> <row name> ?options?
 * qddb_table row configure <table_desc>
 *     -comment, -title, -name, -variable,
 *
 * qddb_table col insert <table_desc> ?-before <column name>? ?-numcols number?
 * qddb_table col delete <table_desc> <column name>
 * qddb_table col clear <table_desc> <column name>
 * qddb_table col copy <table_desc> <from> <to>
 * qddb_table col maxnum <table_desc>
 * qddb_table col getval <table_desc> <column name>
 * qddb_table col setval <table_desc> <row name> <value list>
 * qddb_table col cget <table_desc> <column name> ?options?
 * qddb_table col configure <table_desc> <column name>
 *     -comment, -title, -name, -variable, -type,
 *     -justify, -width, -precision, -separator
 *
 * qddb_table cell clear <table_desc> <row name> <column name>
 * qddb_table cell copy <table_desc> <from row> <from column> <to row> <to column>
 * qddb_table cell getval <table_desc> <row name> <column name>
 * qddb_table cell setval <table_desc> <row name> <column name> <value>
 * qddb_table cell cget <table_desc> <row name> <column name>
 * qddb_table cell configure <table_desc> <row name> <column name>
 *     -comment, -title, -name, -type, -calc, -variable
 *     -justify, -width, -precision, -separator
 *     (overrides any column defs and not used in column operations)
 * 
 * type = string|date|real|integer|calc
 * 
 */

int TclQddb_TableProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    switch (argv[1][0]) {
    case 'd': {
	if (strcmp(argv[1], "define") == 0) {
	    if (TclQddb_ProcessTableDefine(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "delete") == 0) {
	    if (TclQddb_ProcessTableDelete(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    }
    case 'r':
	if (strcmp(argv[1], "row") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ProcessRowCommands(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'c':
	if (strcmp(argv[1], "col") == 0) {
	    if (TclQddb_ProcessColCommands(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "cell") == 0) {
	    if (TclQddb_ProcessCellCommands(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "cget") == 0) {
	    if (TclQddb_TableCget(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "configure") == 0) {
	    if (TclQddb_TableConfigure(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "clear") == 0) {
	    if (TclQddb_TableClear(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "copy") == 0) {
	    if (TclQddb_TableCopy(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    case 'g':
	if (strcmp(argv[1], "getval") == 0) {
	    if (TclQddb_TableGetVal(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    case 's':
	if (strcmp(argv[1], "sort") == 0) {
	    if (TclQddb_ProcessTableSort(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "setval") == 0) {
	    if (TclQddb_TableSetVal(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "summary") == 0) {
	    if (TclQddb_TableSummary(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    case 'e':
	if (strcmp(argv[1], "expr") == 0) {
	    if (TclQddb_TableExpr(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "expr_ok") == 0) {
	    if (TclQddb_TableExprOk(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "eval") == 0) {
	    if (TclQddb_TableEval(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    default:
	Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/* BEGIN GENERIC TABLE COMMANDS AND INTERNAL UTILS */

static int TclQddb_ProcessTableDefine(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char 			*argv[];
{	
    size_t			rows, columns;
    char			*endptr;
    Qddb_TableHeaderInfo	info;
    int				order = QDDB_TABLE_ORDER_ROW;
    int				autoeval = 0;

    if (argc < 4 || (argc%2) != 0) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], 
			 ": wrong # args for \"", argv[0], "\"", NULL);
	return TCL_ERROR;
    }
    endptr = NULL;
    rows = strtoul(argv[2], &endptr, 10);
    if (endptr != argv[2]+strlen(argv[2])) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], 
			 ": bad row number \"", argv[2], "\"", NULL);
	return TCL_ERROR;		
    }
    endptr = NULL;
    columns = strtoul(argv[3], &endptr, 10);
    if (endptr != argv[3]+strlen(argv[3])) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], 
			 ": bad column number \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    }
    info.title = info.comment = info.name = info.variable = NULL;
    info.print = NULL;
    if (argc >= 6) {
	int			i;

	for (i = 4; i < argc; i+=2) {
	    if (i+1 < argc) {
		if (strcmp(argv[i], "-variable") == 0) {
		    info.variable = argv[i+1];
		} else if (strcmp(argv[i], "-order") == 0) {
		    if (strcmp(argv[i+1], "row") == 0) {
			order = QDDB_TABLE_ORDER_ROW;
		    } else if (strcmp(argv[i+1], "column") == 0) {
			order = QDDB_TABLE_ORDER_COL;
		    } else {
			Tcl_AppendResult(interp, argv[0], " ", argv[1], 
					 ": bad arg to option ", argv[i], 
					 ", must be manual, row, or col", NULL);
			return TCL_ERROR;
		    }
		} else if (strcmp(argv[i], "-autoeval") == 0) {
		    if (Tcl_GetBoolean(interp, argv[++i], &autoeval) != TCL_OK)
			return TCL_ERROR;
		} else if (strcmp(argv[i], "-name") == 0) {
		    info.name = argv[i+1];
		} else if (strcmp(argv[i], "-comment") == 0) {
		    info.comment = argv[i+1];
		} else if (strcmp(argv[i], "-title") == 0) {
		    info.title = argv[i+1];
		} else {
		    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad option \"", 
				     argv[i], "\"", NULL);
		    return TCL_ERROR;
		}
	    } else {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": no arg to option ", argv[i], NULL);
		return TCL_ERROR;
	    }
	}
    }
    if (TclQddb_TableDefine(interp, rows, columns, &info, order, autoeval) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_ProcessTableDelete(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char 			*argv[];
{	
    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], 
			 "wrong # args, should be 3", NULL);
	return TCL_ERROR;		
    }
    if (TclQddb_TableDelete(interp, argv[2]) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_ProcessTableSort(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table = NULL;
    char			*startrow = NULL, *endrow = NULL;
    char			**ascending = NULL, **sortby = NULL;
    int				i, ascending_num = 0, sortby_num = 0;

    if (argc < 3 || argc > 11 || (argc%2) != 1) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;	
    }
    if ((table = TclQddb_GetTable(interp, argv[2])) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot find table \"", 
			 argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 3; i < argc; i++) {
	if (strcmp("-start", argv[i]) == 0) {
	    startrow = argv[++i];
	} else if (strcmp("-end", argv[i]) == 0) {
	    endrow = argv[++i];
	} else if (strcmp("-ascending", argv[i]) == 0) {
	    if (ascending != NULL)
		Free(ascending);
	    if (Tcl_SplitList(interp, argv[++i], &ascending_num, &ascending) != TCL_OK) {
		if (sortby != NULL)
		    Free(sortby);
		return TCL_ERROR;
	    }
	} else if (strcmp("-sortby", argv[i]) == 0) {
	    if (sortby != NULL)
		Free(sortby);
	    if (Tcl_SplitList(interp, argv[++i], &sortby_num, &sortby) != TCL_OK) {
		if (ascending != NULL)
		    Free(ascending);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad argument \"",
			     argv[i], "\"");
	    if (ascending != NULL)
		Free(ascending);
	    if (sortby != NULL)
		Free(sortby);
	    return TCL_ERROR;
	}
    }
    if (TclQddb_TableSort(interp, table, startrow, endrow,
			  ascending, ascending_num, sortby, sortby_num) != TCL_OK) {
	if (ascending != NULL)
	    Free(ascending);
	if (sortby != NULL)
	    Free(sortby);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_TableDefine(interp, rows, columns, info, order, autoeval)
    Tcl_Interp			*interp;
    size_t			rows, columns;
    Qddb_TableHeaderInfo	*info;
    int				order;
    int				autoeval;
{
    Qddb_TableRowCol		rowcol;
    Qddb_Table			*table;
    char			*table_desc;
    int				relink = 0;

    rowcol.row = rows;
    rowcol.column = columns;
    table = Qddb_TableOp(NULL, QDDB_TABLE_OP_DEFINE, (void *)&rowcol);
    if (table == NULL) {
	Tcl_AppendResult(interp, "qddb_table define failed", NULL);
	return TCL_ERROR;
    }
    table->info = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
    if (info->variable != NULL) {
	table->info->variable = (char *)Malloc(strlen(info->variable)+1);
	strcpy(table->info->variable, info->variable);
	relink = 1;
    }
    if (info->name != NULL) {
	table->info->name = (char *)Malloc(strlen(info->name)+1);
	strcpy(table->info->name, info->name);
    }
    if (info->title != NULL) {
	table->info->title = (char *)Malloc(strlen(info->title)+1);
	strcpy(table->info->title, info->title);
    }
    if (info->comment != NULL) {
	table->info->comment = (char *)Malloc(strlen(info->comment)+1);
	strcpy(table->info->comment, info->comment);
    }
    table->order = order;
    table->autoeval = autoeval;
    if (relink) {
	if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	    return TCL_ERROR;
	if (info->variable != NULL && info->variable[0] == '\0') {
	    Free(table->info->variable);
	    table->info->variable = NULL;
	}
    }
    table_desc = TclQddb_NewTable(interp, table);
    if (table_desc == NULL)
	return TCL_ERROR;
    Tcl_SetResult(interp, table_desc, TCL_DYNAMIC);
    if (info->name != NULL) {
	if (TclQddb_SetTableName(interp, table, info->name) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_TableDeleteOne(interp, table_name)
    Tcl_Interp			*interp;
    char			*table_name;
{
    Qddb_Table			*table;
    Tcl_HashEntry		*hash_ptr, *named_hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_TableHashTable, table_name);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    table = (Qddb_Table *)Tcl_GetHashValue(hash_ptr);
    if (table != NULL && table->info != NULL && table->info->name != NULL &&
	TclQddb_TableNameHashTableInit != 0) {
	named_hash_ptr = Tcl_FindHashEntry(&TclQddb_TableNameHashTable, table->info->name);
	Tcl_DeleteHashEntry(named_hash_ptr);
    }
    TclQddb_UnlinkTable(interp, table);
    Tcl_DeleteHashEntry(hash_ptr);
    if (table != NULL) {
	(void)Qddb_TableOp(table, QDDB_TABLE_OP_REMOVE, NULL);
    }
    return TCL_OK;    
}

int TclQddb_TableDelete(interp, table_name)
    Tcl_Interp			*interp;
    char			*table_name;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;

    if (table_name == NULL)
	return TCL_OK;
    if (TclQddb_TableHashTableInit == 0)
	return TCL_OK;
    if (strcmp(table_name, "all") == 0) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_TableHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    hash_key = Tcl_GetHashKey(&TclQddb_TableHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_TableDelete: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_TableDeleteOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	TclQddb_TableHashTableInit = 0;
	Tcl_DeleteHashTable(&TclQddb_TableHashTable);
    } else if (TclQddb_TableDeleteOne(interp, table_name) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

void TclQddb_DeleteTableProc(clientData)
    ClientData			clientData;
{
    (void)TclQddb_TableDelete((Tcl_Interp *)clientData, "all");
}

int TclQddb_TableSort(interp, table, startname, endname, 
		      ascending, ascending_num, sortby, sortby_num)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    char			*startname, *endname;
    char			*ascending[];
    int				ascending_num;
    char			*sortby[];
    int				sortby_num;
{
    int				i, j;
    Qddb_TableOpSort		sortnode;
    size_t			*sizeptr;

    if (sortby_num < ascending_num) {
	Tcl_AppendResult(interp, "qddb_table sort: # elements in -sortby < # elements in -ascending", NULL);
	return TCL_ERROR;
    }
    if (sortby_num > 0) {
	sortnode.sortby = (Qddb_TableOpSortNode *)Malloc(sizeof(Qddb_TableOpSortNode)*sortby_num);
    } else {
	sortnode.sortby = NULL;
    }
    sortnode.number = sortby_num;
#if defined(notyet)
    for (i = 0; i < ascending_num; i++) { /* check ascending list for validity */
	sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, ascending[i]);
	if (*sizeptr == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "qddb_table sort: bad column name \"", ascending[i], "\"", NULL);
	    Free(sortnode.sortby);
	    return TCL_ERROR;
	}
	Free(sizeptr);
    }
#endif
    if (startname != NULL) { /* validate start row */
	sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, startname);
	if (*sizeptr == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "qddb_table sort: bad row name \"", startname, "\"", NULL);
	    Free(sortnode.sortby);
	    return TCL_ERROR;
	}
	Free(sizeptr);
    }
    sortnode.startrow = startname;
    if (endname != NULL) { /* validate end row */
	sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, endname);
	if (*sizeptr == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "qddb_table sort: bad row name \"", endname, "\"", NULL);
	    Free(sortnode.sortby);
	    return TCL_ERROR;
	}
	Free(sizeptr);
    }
    sortnode.endrow = endname;
    for (i = 0; i < sortby_num; i++) {
	sortnode.sortby[i].column_name = sortby[i];
	sortnode.sortby[i].ascending = False;
	for (j = 0; j < ascending_num; j++) {
	    if (strcmp(ascending[j], sortby[i]) == 0) {
		sortnode.sortby[i].ascending = True;
		break;
	    }
	}
    }
    TclQddb_UnlinkTable(interp, table);
    if (Qddb_TableOp(table, QDDB_TABLE_OP_SORT, &sortnode) == NULL) {
	Free(sortnode.sortby);
	Tcl_AppendResult(interp, "qddb_table sort: bad column names (", NULL);
	for (i = 0; i < sortby_num; i++) {
	    sizeptr = Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, sortby[i]);
	    if (*sizeptr == 0) {
		Tcl_AppendResult(interp, sortby[i], NULL);
	    }
	    Free(sizeptr);
	}
	Tcl_AppendResult(interp, ")", NULL);
	TclQddb_RelinkTable(interp, table);
	return TCL_ERROR;
    }
    TclQddb_RelinkTable(interp, table);
    if (sortnode.sortby != NULL)
	Free(sortnode.sortby);
    return TCL_OK;
}

char *TclQddb_NewTable(interp, table)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
{
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ], *retval;
    int				newPtr;

    if (TclQddb_TableHashTableInit == 0) {
	TclQddb_TableHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TableHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_table%d", TclQddb_TableNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_TableHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)table);
    retval = (char *)Malloc(strlen(token)+1);
    strcpy(retval, token);
    return retval;
}

static int TclQddb_SetTable(interp, table_name, table)
    Tcl_Interp			*interp;
    char			*table_name;
    Qddb_Table			*table;
{
    Tcl_HashEntry		*hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_TableHashTable, table_name);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)table);
    return TCL_OK;
}

Qddb_Table *TclQddb_GetTable(interp, table_name)
    Tcl_Interp			*interp;
    char			*table_name;
{
    Qddb_Table			*table;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_TableHashTableInit == 0) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TableHashTable, table_name);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return NULL;
    }
    table = (Qddb_Table *)Tcl_GetHashValue(hash_ptr);
    return table;
}

static void TclQddb_DeleteTableName(interp, table_name)
    Tcl_Interp			*interp;
    char			*table_name;
{
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_TableNameHashTableInit == 0) {
	return;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TableNameHashTable, table_name);
    if (hash_ptr == NULL) {
	return;
    }
    Tcl_DeleteHashEntry(hash_ptr);
}

static int TclQddb_SetTableName(interp, table, table_name)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    char			*table_name;
{
    Tcl_HashEntry		*hash_ptr;
    int				newPtr;

    if (TclQddb_TableNameHashTableInit == 0) {
	TclQddb_TableNameHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_TableNameHashTable, TCL_STRING_KEYS);
    }
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_TableNameHashTable, table_name, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", table_name, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    if (newPtr == 0) {
	Tcl_AppendResult(interp, "table name \"", table_name, "\" already exists", NULL);
	return TCL_ERROR;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)table);
    return TCL_OK;
}

Qddb_Table *TclQddb_GetTableByName(interp, table_name)
    Tcl_Interp			*interp;
    char			*table_name;
{
    Qddb_Table			*table;
    Tcl_HashEntry		*hash_ptr;

    if ((table = TclQddb_GetTable(interp, table_name)) != NULL) {
	return table;
    } else {
	Tcl_ResetResult(interp);
    }
    if (TclQddb_TableNameHashTableInit == 0) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_TableNameHashTable, table_name);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find table \"", table_name, "\" (TCL error)", NULL);
	return NULL;
    }
    table = (Qddb_Table *)Tcl_GetHashValue(hash_ptr);
    return table;
}

static int TclQddb_TableClear(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name;

    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    TclQddb_UnlinkTable(interp, table);
    (void)Qddb_TableOp(table, QDDB_TABLE_OP_CLEAR, NULL);
    TclQddb_RelinkTable(interp, table);
    return TCL_OK;
}

static int TclQddb_TableSetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *rowlist, **rows;
    int				i, num_rows;
    char			*nargv[6], buf[MAXINTEGERLEN];
    char			nargc = 6;

    if (argc != 4) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    rowlist = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (Tcl_SplitList(interp, rowlist, &num_rows, &rows) != TCL_OK)
	return TCL_ERROR;
    if (num_rows != table->rows) {
	Tcl_AppendResult(interp, "qddb_table setval: wrong # rows in list", NULL);
	return TCL_ERROR;
    }
    nargv[0] = "qddb_table";
    nargv[1] = "row";
    nargv[2] = "setval";
    nargv[3] = table_name;
    for (i = 0; i < num_rows; i++) {
	sprintf(buf, "%d", i+1);
	nargv[4] = buf;
	nargv[5] = rows[i];
	if (TclQddb_RowSetVal(interp, nargc, nargv) != TCL_OK) {
	    Free(rows);
	    Tcl_AppendResult(interp, "qddb_table setval cannot set row", NULL);
	    return TCL_ERROR;
	}
    }
    Free(rows);
    return TCL_OK;
}

static int TclQddb_TableGetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *format = "list";
    int				rowtitles = 0, coltitles = 0;
    int				i;

    if (argc < 3 || argc > 9 || (argc%2) != 1) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    for (i = 3; i < argc; i++) {
	if (strcmp(argv[i], "-format") == 0) {
	    format = argv[++i];
	} else if (strcmp(argv[i], "-rowtitles") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &rowtitles) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-coltitles") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &coltitles) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad arg \"", argv[i], "\"", NULL);
	    return TCL_ERROR;	    
	}
    }
    if (strcmp(format, "list") == 0) {
	if (TclQddb_TableGetVal_list(interp, table, rowtitles, coltitles) != TCL_OK)
	    return TCL_ERROR;
    } else if (strcmp(format, "text") == 0) {
	if (TclQddb_TableGetVal_text(interp, table, rowtitles, coltitles) != TCL_OK)
	    return TCL_ERROR;	
    } else if (strcmp(format, "rows") == 0) {
	if (TclQddb_TableGetVal_rows(interp, table, rowtitles, coltitles) != TCL_OK)
	    return TCL_ERROR;
    } else {
	Tcl_AppendResult(interp, "qddb_table getval: bad format type \"", format, "\"", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_TableGetVal_list(interp, table, rowtitles, coltitles)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    int				rowtitles, coltitles;
{
    char			***rawtable;
    char			**argv;
    char			*ptr;
    int				i, j, kludge;

    rawtable = TclQddb_TableGetStrings(table);
    if (rawtable == NULL) {
	Tcl_AppendResult(interp, "TclQddb_TableGetStrings failed", NULL);
	return TCL_ERROR;
    }
    argv = (char **)Malloc(sizeof(char *)*(table->columns+2));
    argv[table->columns+1] = NULL;
    kludge = (rowtitles == 0)? 1:0;
    for (i = (coltitles == 0)? 1:0; i <= table->rows; i++) {
	for (j = kludge; j <= table->columns; j++) {
	    argv[j-kludge] = rawtable[i][j];
	}
	ptr = Tcl_Merge((int)(table->columns+1-kludge), argv);
	Tcl_AppendElement(interp, ptr);
	Free(ptr);
    }
    for (i = 0; i <= table->rows; i++) {
	for (j = 0; j <= table->columns; j++) {
	    Free(rawtable[i][j]);
	}
	Free(rawtable[i]);
    }
    Free(argv);
    Free(rawtable);
    return TCL_OK;
}

static int TclQddb_TableGetVal_text(interp, table, rowtitles, coltitles)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    int				rowtitles, coltitles;
{
    char			**orig_rawtable, **rawtable, *ptr;

    orig_rawtable = rawtable = TclQddb_GetFormattedTable(interp, table, NULL, coltitles, rowtitles);
    if (rawtable == NULL) {
	Tcl_AppendResult(interp, "TclQddb_GetFormattedTable failed", NULL);
	return TCL_ERROR;
    }
    if (!coltitles) {
	rawtable++;
    }
    Qddb_InitBuffer();
    while (*rawtable != NULL) {
	Qddb_ConcatBuffer(*rawtable);
	Qddb_ConcatBuffer("\n");
	Free(*rawtable);
	rawtable++;
    }
    ptr = Qddb_GetBuffer();
    Tcl_SetResult(interp, ptr, TCL_DYNAMIC);
    Free(orig_rawtable);
    return TCL_OK;
}

static int TclQddb_TableGetVal_rows(interp, table, rowtitles, coltitles)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    int				rowtitles, coltitles;
{
    char			**orig_rawtable, **rawtable;

    orig_rawtable = rawtable = TclQddb_GetFormattedTable(interp, table, NULL, coltitles, rowtitles);
    if (rawtable == NULL) {
	Tcl_AppendResult(interp, "TclQddb_GetFormattedTable failed", NULL);
	return TCL_ERROR;
    }
    if (!coltitles) {
	rawtable++;
    }
    while (*rawtable != NULL) {
	Tcl_AppendElement(interp, *rawtable);
	Free(*rawtable);
	rawtable++;
    }
    Free(orig_rawtable);
    return TCL_OK;
}

static int TclQddb_TableCget(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *option;

    if (argc != 4) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (strcmp(argv[3], "-order") == 0) {
	switch (table->order) {
	case QDDB_TABLE_ORDER_COL:
	    option = "column";
	    break;
	case QDDB_TABLE_ORDER_ROW:
	default:
	    option = "row";
	}
    } else {
	option = TclQddb_HeaderGetInfo(table->info, NULL, argv[3]);
    }
    if (option == NULL) {
	Tcl_AppendResult(interp, "qddb_table cget: bad option \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    } else {
	Tcl_SetResult(interp, option, TCL_VOLATILE);
    }
    return TCL_OK;
}

static int TclQddb_TableConfigure(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*old_table_name = NULL, *table_name;
    int				i;

    if (argc < 3 || (argc%2) != 1) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    table_name = NULL;
    if (table->info != NULL && table->info->name != NULL) {
	old_table_name = (char *)alloca(strlen(table->info->name)+1);
	strcpy(old_table_name, table->info->name);
    }
    TclQddb_UnlinkTable(interp, table);
    for (i = 3; i < argc; i+=2) {
	if (strcmp(argv[i], "-order") == 0) {
	    if (strcmp(argv[i+1], "row") == 0) {
		table->order = QDDB_TABLE_ORDER_ROW;
	    } else if (strcmp(argv[i+1], "column") == 0) {
		table->order = QDDB_TABLE_ORDER_COL;
	    } else {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[i], 
				 ": bad order argument \"", argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	} else if (TclQddb_TableSetHeaderInfo(interp, &table->info, argv[i], argv[i+1], 0) != TCL_OK) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad option \"", argv[i], "\"", NULL);
	    return TCL_ERROR;
	}
	if (strcmp(argv[i], "-name") == 0)
	    table_name = argv[i+1];
    }
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    if (table_name != NULL) {
	if (old_table_name != NULL) {
	    TclQddb_DeleteTableName(interp, old_table_name);
	}
	if (table_name[0] != '\0' && TclQddb_SetTableName(interp, table, table_name) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_TableCopy(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table, *ntable;
    char			*table_name;

    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if ((ntable = (Qddb_Table *)Qddb_TableOp(table, QDDB_TABLE_OP_COPY, NULL)) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot copy table", NULL);
	return TCL_ERROR;
    }
    if ((table_name = TclQddb_NewTable(interp, ntable)) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot create new table", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, table_name, TCL_DYNAMIC);
    return TCL_OK;
}

static int TclQddb_TableExpr(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_TableCalc		*calc;
    double			num;
    char			retval[4*1024];

    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 ": wrong # args, should be 3", NULL);
	return TCL_ERROR;
    }
    calc = Qddb_TableCalcParse(argv[2]);
    if (calc == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 ": syntax error in \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_TableExprInterp(interp, calc, NULL, 0, 0, NULL, &num) != TCL_OK) {
	Qddb_Free(QDDB_TYPE_TABLECALC, calc);
	return TCL_ERROR;
    }
    Tcl_PrintDouble(interp, num, retval);
    Tcl_SetResult(interp, retval, TCL_VOLATILE);
    Qddb_Free(QDDB_TYPE_TABLECALC, calc);
    return TCL_OK;
}

static int TclQddb_TableExprOk(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_TableCalc		*calc;

    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 ": wrong # args, should be 3", NULL);
	return TCL_ERROR;
    }
    calc = Qddb_TableCalcParse(argv[2]);
    if (calc == NULL) {
	Tcl_SetResult(interp, "0", TCL_STATIC);
    } else {
	Tcl_SetResult(interp, "1", TCL_STATIC);
	Qddb_Free(QDDB_TYPE_TABLECALC, calc);
    }
    return TCL_OK;
}

static int TclQddb_TableEval(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row, *row0, *cell;
    Qddb_TableCalc		*cell_calc;
    char			*table_name;
    int				default_type;
    double			num;
    size_t			i, j;
    int				retval = TCL_OK;

    if (argc != 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[2];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    row0 = table->table[0];
    if (table->order == QDDB_TABLE_ORDER_COL) {
	for (j = 1; j <= table->columns; j++) {
	    for (i = 1; i <= table->rows; i++) {
		row = table->table[i];
		cell = row + j;
		default_type = cell->data.type;
		cell_calc = cell->data.compiled;
		if (default_type == QDDB_TABLE_TYPE_DEFAULT) {
		    default_type = row0[j].data.type;
		    if (default_type == QDDB_TABLE_TYPE_CALC) {
			cell_calc = row0[j].data.compiled;
		    }
		}
		if (default_type == QDDB_TABLE_TYPE_CALC) {
		    if (TclQddb_TableExprInterp(interp, cell_calc, table, i, j,
						NULL, &num) != TCL_OK) {
			cell->data.data.real = 0.00;
			retval = TCL_ERROR;
		    } else {
			cell->data.data.real = num;
		    }
		}
	    }
	}
    } else {
	for (i = 1; i <= table->rows; i++) {
	    row = table->table[i];
	    for (j = 1; j <= table->columns; j++) {
		cell = row + j;
		default_type = cell->data.type;
		cell_calc = cell->data.compiled;
		if (default_type == QDDB_TABLE_TYPE_DEFAULT) {
		    default_type = row0[j].data.type;
		    if (default_type == QDDB_TABLE_TYPE_CALC) {
			cell_calc = row0[j].data.compiled;
		    }
		}
		if (default_type == QDDB_TABLE_TYPE_CALC) {
		    if (TclQddb_TableExprInterp(interp, cell_calc, table, i, j,
						NULL, &num) != TCL_OK) {
			cell->data.data.real = 0.00;
			retval = TCL_ERROR;
		    } else {
			cell->data.data.real = num;
		    }
		}
	    }
	}
    }
    return retval;
}

static int TclQddb_TableSummary(interp, argc, argv)
    Tcl_Interp                  *interp;
    int                         argc;
    char                        *argv[];
{
    Qddb_Table                  *table, *ntable;
    char                        *table_name;

    if (argc != 8) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args, should be 8", NULL);
	return TCL_ERROR;
    }
    if ((table = TclQddb_GetTable(interp, argv[2])) == NULL) {
	Tcl_AppendResult(interp, "cannot find table \"", argv[2], "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    ntable = (Qddb_Table *)Qddb_TableOp(table, QDDB_TABLE_OP_SUMMARY, (void *)(argv+3));
    if (ntable == NULL) {
	Tcl_SetResult(interp, "", TCL_STATIC);
	return TCL_OK;
    }
    if ((table_name = TclQddb_NewTable(interp, ntable)) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot create new table", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, table_name, TCL_DYNAMIC);
    return TCL_OK;
}

/* BEGIN ROW COMMANDS */

static int TclQddb_ProcessRowCommands(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 ": wrong # args, should be >= 3", NULL);
	return TCL_ERROR;
    }
    switch (argv[2][0]) {
    case 'i': /* insert */
	if (strcmp(argv[2], "insert") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_RowInsert(interp, argc-3, argv+3) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'd': /* delete */
	if (strcmp(argv[2], "delete") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_RowDelete(interp, argc-3, argv+3) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'm': { /* maxnum */
	char		buf[MAXINTEGERLEN];
	Qddb_Table	*table;

	if (strcmp(argv[2], "maxnum") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if ((table = TclQddb_GetTable(interp, argv[3])) == NULL) {
	    Tcl_AppendResult(interp, "cannot find table \"", argv[3], "\" (TCL error)", NULL);	    
	    return TCL_ERROR;
	}
	sprintf(buf, "%d", (int)table->rows);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	break;
    }
    case 'g': /* getval */
	if (strcmp("getval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_RowGetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 's': /* setval */
	if (strcmp("setval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_RowSetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'c': /* cget, configure, clear, copy */
	if (strcmp("cget", argv[2]) == 0) {
	    if (TclQddb_RowCget(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("configure", argv[2]) == 0) {
	    if (TclQddb_RowConfigure(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("clear", argv[2]) == 0) {
	    if (TclQddb_RowClear(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("copy", argv[2]) == 0) {
	    if (TclQddb_RowCopy(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	break;
    case 'e':
	if (strcmp("eval", argv[2]) == 0) {
	    if (TclQddb_RowEval(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	break;
    default:
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_RowInsert(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *endptr;
    size_t			num_rows = 1, before_num = 0, *sizeptr;
    int				i;

    if (argc != 1 && argc != 3 && argc != 5) {
	Tcl_AppendResult(interp, "qddb_table row insert: wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[0];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    TclQddb_UnlinkTable(interp, table);
    for (i = 1; i < argc; i+=2) {
	if (strcmp(argv[i], "-before") == 0) {
	    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[i+1])) == 0) {
		Free(sizeptr);
		Tcl_AppendResult(interp, "qddb_table row insert: bad row name \"", 
				 argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	    before_num = *sizeptr;
	    Free(sizeptr);
	    if (before_num == 0) {
		Tcl_AppendResult(interp, "qddb_table row insert: bad row name \"", 
				 argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	} else if (strcmp(argv[i], "-numrows") == 0) {
	    num_rows = strtoul(argv[i+1], &endptr, 10);
	    if (argv[i+1]+strlen(argv[i+1]) != endptr) {
		Tcl_AppendResult(interp, "qddb_table row insert: bad number for -numrows \"", 
				 argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "qddb_table row insert: bad option ", argv[i], NULL);
	    return TCL_ERROR;
	}
    }
    if (before_num == 0) {
	before_num = table->rows + 1;
    }
    for (i = 0; i < num_rows; i++) {
	table = (Qddb_Table *)Qddb_TableOp(table, QDDB_TABLE_OP_INSERTROW, &before_num);
    }
    if (TclQddb_SetTable(interp, table_name, table) != TCL_OK)
	return TCL_ERROR;
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_RowDelete(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    char			*table_name;
    char			*row_name;
    Qddb_Table			*table;
    size_t			*rowptr;

    if (argc != 2) {
	Tcl_AppendResult(interp, "qddb_table row delete: wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[0];
    row_name = argv[1];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table row delete: bad table descriptor \"",
			 table_name, "\"", NULL);
	return TCL_ERROR;
    }
    TclQddb_UnlinkTable(interp, table);
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, "qddb_table row delete: bad row name \"", 
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    Qddb_TableOp(table, QDDB_TABLE_OP_REMOVEROW, rowptr);
    Free(rowptr);
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}


static int TclQddb_RowSetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, *thisrow, **cells;
    char			*table_name, *row_name;
    char			*value_list, **value_splitlist;
    size_t			*rowptr, rownum;
    int				i, value_num, num_cols, cell_type, retval = TCL_OK;

    /* qddb_table row setval <table_desc> <row name> <value list> */
    if (argc != 6) {
	Tcl_AppendResult(interp, "qddb_table row setval: wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    row_name = argv[4];
    value_list = argv[5];
    if (Tcl_SplitList(interp, value_list, &value_num, &value_splitlist) != TCL_OK)
	return TCL_ERROR;
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table row setval: bad table descriptor\"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    num_cols = table->columns;
    if (num_cols != value_num) {
	char			buf1[MAXINTEGERLEN], buf2[MAXINTEGERLEN];

	sprintf(buf1, "%d", value_num);
	sprintf(buf2, "%d", num_cols);
	Tcl_AppendResult(interp, "qddb_table row setval: number of elements in value list \"", buf1,
			 "\" does not match number of columns in table ", "\"", buf2, "\"", NULL);
	return TCL_ERROR;
    }
    cells = table->table;
    row0 = cells[0] + 1;
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, "qddb_table row setval: bad row name \"",
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    rownum = *rowptr;
    thisrow = cells[rownum];
    Free(rowptr);
    for (i = 1; i <= num_cols; i++, row0++) {
	cell_type = thisrow[i].data.type;
	if (cell_type == QDDB_TABLE_TYPE_DEFAULT) {
	    cell_type = row0->data.type;
	}
	if (TclQddb_SetCellValue(interp, thisrow+i, row0->data.type, value_splitlist[i-1]) != TCL_OK) {
	    retval = TCL_ERROR;
	}
	if (cell_type == QDDB_TABLE_TYPE_CALC) {
	    double		num;

	    if (TclQddb_TableExprInterp(interp, thisrow[i].data.compiled, table, rownum, 
					(size_t)i, &(thisrow[i].data.refs), &num) != TCL_OK) {
		thisrow[i].data.data.real = 0.00;
		retval = TCL_ERROR;
	    } else {
		thisrow[i].data.data.real = num;
	    }
	}
    }
    Free(value_splitlist);
    return retval;
}

static int TclQddb_RowGetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, *thisrow, **cells;
    char			*table_name, *row_name, *value;
    size_t			*rowptr;
    int				i, num_cols;

    /* qddb_table row getval <table_desc> <row name> */
    if (argc != 5) {
	Tcl_AppendResult(interp, "qddb_table row getval: wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    row_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table row getval: bad table \"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    num_cols = table->columns;
    cells = table->table;
    row0 = cells[0] + 1;
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, "qddb_table row getval: bad row name \"",
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    thisrow = cells[*rowptr];
    Free(rowptr);
    for (i = 1; i <= num_cols; i++, row0++) {
	if (row0->info != NULL)
	    value = TclQddb_GetCellValue(thisrow+i, row0->data.type, row0->info->print);
	else
	    value = TclQddb_GetCellValue(thisrow+i, row0->data.type, NULL);
	Tcl_AppendElement(interp, value);
    }
    return TCL_OK;
}

static int TclQddb_RowCget(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *option, *row_name;
    size_t			*sizeptr;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    option = argv[5];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    sizeptr = Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*sizeptr == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, "qddb_table row cget: bad row name \"", row_name, "\"", NULL);
	return TCL_ERROR;
    }
    option = TclQddb_HeaderGetInfo(table->table[*sizeptr]->info, NULL, option);
    Free(sizeptr);
    if (option == NULL) {
	Tcl_AppendResult(interp, "qddb_table row cget: bad option \"", argv[5], "\"", NULL);
	return TCL_ERROR;
    } else {
	Tcl_SetResult(interp, option, TCL_VOLATILE);
    }
    return TCL_OK;
}

static int TclQddb_RowConfigure(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row;
    char			*table_name, *row_name;
    size_t			*rowptr;
    int				i, relink = 0;

    if (argc < 5 || (argc%2) != 1) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name)) == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			 ": bad row name \"", row_name, "\"", NULL);
	return TCL_ERROR;
    }
    row = table->table[*rowptr];
    Free(rowptr);
    for (i = 5; i < argc; i+=2) {
	if (!relink && (strcmp(argv[i], "-name") == 0 || strcmp(argv[i], "-variable") == 0)) {
	    TclQddb_UnlinkTable(interp, table);
	    relink = 1;
	}
	if (TclQddb_TableSetHeaderInfo(interp, &row->info, argv[i], argv[i+1], 0) != TCL_OK) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			     ": bad option \"", argv[i], "\"", NULL);
	    return TCL_ERROR;
	}
    }
    if (relink) {
	if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_RowClear(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 5) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    (void)Qddb_TableOp(table, QDDB_TABLE_OP_CLEARROW, sizeptr);
    Free(sizeptr);
    return TCL_OK;
}

static int TclQddb_RowCopy(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableFromTo		fromto;
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    fromto.from = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[5])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[5], "\"", NULL);
	return TCL_ERROR;
    }
    fromto.to = *sizeptr;
    Free(sizeptr);
    TclQddb_UnlinkTable(interp, table);
    if (Qddb_TableOp(table, QDDB_TABLE_OP_COPYROW, &fromto) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot copy row", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_RowEval(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row, *row0, *cell;
    Qddb_TableCalc		*cell_calc;
    char			*table_name, *row_name;
    int				default_type, retval = TCL_OK;
    double			num;
    size_t			i, *rowptr, rownum;

    if (argc != 5) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    row0 = table->table[0];
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad row name \"",
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    rownum = *rowptr;
    Free(rowptr);
    row = table->table[rownum];
    for (i = 1; i <= table->columns; i++) {
	cell = row + i;
	default_type = cell->data.type;
	cell_calc = cell->data.compiled;
	if (default_type == QDDB_TABLE_TYPE_DEFAULT) {
	    default_type = row0[i].data.type;
	    if (default_type == QDDB_TABLE_TYPE_CALC) {
		cell_calc = row0[i].data.compiled;
	    }
	}
	if (default_type == QDDB_TABLE_TYPE_CALC) {
	    if (TclQddb_TableExprInterp(interp, cell_calc, table, rownum, i,
					NULL, &num) != TCL_OK) {
		cell->data.data.real = 0.00;
		retval = TCL_ERROR;
	    } else {
		cell->data.data.real = num;
	    }
	}
    }
    return retval;
}



/* BEGIN COL COMMANDS */

static int TclQddb_ProcessColCommands(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 "wrong # args, should >= 3", NULL);
	return TCL_ERROR;
    }
    switch (argv[2][0]) {
    case 'i': /* insert */
	if (strcmp(argv[2], "insert") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ColInsert(interp, argc-3, argv+3) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'd': /* delete */
	if (strcmp(argv[2], "delete") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ColDelete(interp, argc-3, argv+3) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'm': { /* maxnum */
	char		buf[MAXINTEGERLEN];
	Qddb_Table	*table;

	if (strcmp(argv[2], "maxnum") != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;	    
	}
	if ((table = TclQddb_GetTable(interp, argv[3])) == NULL) {
	    Tcl_AppendResult(interp, "cannot find table \"", argv[3], "\" (TCL error)", NULL);	    
	    return TCL_ERROR;
	}
	sprintf(buf, "%d", (int)table->columns);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	break;
    }
    case 'g': /* getval */
	if (strcmp("getval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ColGetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 's': /* setval */
	if (strcmp("setval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_ColSetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'c': /* cget, configure */
	if (strcmp("cget", argv[2]) == 0) {
	    if (TclQddb_ColCget(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("configure", argv[2]) == 0) {
	    if (TclQddb_ColConfigure(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("clear", argv[2]) == 0) {
	    if (TclQddb_ColClear(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("copy", argv[2]) == 0) {
	    if (TclQddb_ColCopy(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'e':
	if (strcmp("eval", argv[2]) == 0) {
	    if (TclQddb_ColEval(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	break;
    default:
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_ColInsert(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *endptr;
    size_t			num_cols = 1, before_num = 0, *sizeptr;
    int				i;

    if (argc != 1 && argc != 3 && argc != 5) {
	Tcl_AppendResult(interp, "qddb_table col insert: wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[0];
    if ((table = TclQddb_GetTable(interp, argv[0])) == NULL)
	return TCL_ERROR;
    TclQddb_UnlinkTable(interp, table);
    for (i = 1; i < argc; i+=2) {
	if (strcmp(argv[i], "-before") == 0) {
	    sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[i+1]);
	    before_num = *sizeptr;
	    Free(sizeptr);
	    if (before_num == 0) {
		Tcl_AppendResult(interp, "qddb_table col insert: bad col name \"", 
				 argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	} else if (strcmp(argv[i], "-numcols") == 0) {
	    num_cols = strtoul(argv[i+1], &endptr, 10);
	    if (argv[i+1]+strlen(argv[i+1]) != endptr) {
		Tcl_AppendResult(interp, "qddb_table col insert: bad number for -numcols\"", 
				 argv[i+1], "\"", NULL);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "qddb_table col insert: bad option ", argv[i], NULL);
	    return TCL_ERROR;
	}
    }
    if (before_num == 0) {
	before_num = table->columns+1;
    }
    for (i = 0; i < num_cols; i++) {
	table = (Qddb_Table *)Qddb_TableOp(table, QDDB_TABLE_OP_INSERTCOL, &before_num);
    }
    if (TclQddb_SetTable(interp, table_name, table) != TCL_OK)
	return TCL_ERROR;
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_ColDelete(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    char			*table_name;
    char			*col_name;
    Qddb_Table			*table;
    size_t			*colptr;

    if (argc != 2) {
	Tcl_AppendResult(interp, "qddb_table col delete: wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[0];
    col_name = argv[1];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table col delete: bad table descriptor \"",
			 table_name, "\"", NULL);
	return TCL_ERROR;
    }
    TclQddb_UnlinkTable(interp, table);
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, "qddb_table col delete: bad col name \"", 
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    Qddb_TableOp(table, QDDB_TABLE_OP_REMOVECOL, colptr);
    Free(colptr);
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_ColSetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, **cells, **thisrow;
    char			*table_name, *col_name;
    char			*value_list, **value_splitlist;
    size_t			*colptr, col_num, cell_type;
    int				i, value_num, num_rows, retval = TCL_OK;

    /* qddb_table col setval <table_desc> <column name> <value list> */
    if (argc != 6) {
	Tcl_AppendResult(interp, "qddb_table col setval: wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    col_name = argv[4];
    value_list = argv[5];
    if (Tcl_SplitList(interp, value_list, &value_num, &value_splitlist) != TCL_OK)
	return TCL_ERROR;
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table col setval: bad table descriptor\"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    num_rows = table->rows;
    if (num_rows != value_num) {
	char			buf1[MAXINTEGERLEN], buf2[MAXINTEGERLEN];

	sprintf(buf1, "%d", value_num);
	sprintf(buf2, "%d", num_rows);
	Tcl_AppendResult(interp, "qddb_table col setval: number of elements in value list \"", buf1,
			 "\" does not match number of columns in table ", "\"", buf2, "\"", NULL);
	return TCL_ERROR;
    }
    cells = table->table;
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, "qddb_table col setval: bad column name \"",
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    col_num = *colptr;
    row0 = cells[0] + col_num;
    Free(colptr);
    thisrow = cells + 1;
    for (i = 1; i <= num_rows; i++, thisrow++) {
	cell_type = (*thisrow)[col_num].data.type;
	if (cell_type == QDDB_TABLE_TYPE_DEFAULT) {
	    cell_type = row0->data.type;
	}
	if (TclQddb_SetCellValue(interp, (*thisrow)+col_num, row0->data.type,
				 value_splitlist[i-1]) != TCL_OK) {
	    Free(value_splitlist);
	    return TCL_ERROR;
	}
	if (cell_type == QDDB_TABLE_TYPE_CALC) {
	    double		num;

	    if (TclQddb_TableExprInterp(interp, (*thisrow)[col_num].data.compiled, 
					table, (size_t)i, col_num,
					&((*thisrow)[col_num].data.refs), &num) != TCL_OK) {
		(*thisrow)[col_num].data.data.real = 0.00;
		retval = TCL_ERROR;
	    } else {
		(*thisrow)[col_num].data.data.real = num;
	    }
	}
    }
    Free(value_splitlist);
    return retval;
}

static int TclQddb_ColGetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, **cells, **thisrow;
    char			*table_name, *col_name, *value;
    size_t			*colptr, col_num;
    int				i, num_rows;

    /* qddb_table col getval <table_desc> <column name> */
    if (argc != 5) {
	Tcl_AppendResult(interp, "qddb_table col setval: wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    col_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, "qddb_table col setval: bad table descriptor\"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    num_rows = table->rows;
    cells = table->table;
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, "qddb_table col setval: bad column name \"",
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    col_num = *colptr;
    row0 = cells[0] + col_num;
    Free(colptr);
    thisrow = cells + 1;
    for (i = 1; i <= num_rows; i++, thisrow++) {
	if (row0->info != NULL)
	    value = TclQddb_GetCellValue((*thisrow)+col_num, row0->data.type, row0->info->print);
	else
	    value = TclQddb_GetCellValue((*thisrow)+col_num, row0->data.type, NULL);
	Tcl_AppendElement(interp, value);
    }
    return TCL_OK;
}

static int TclQddb_ColCget(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *option, *col_name;
    size_t			*sizeptr;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    col_name = argv[4];
    option = argv[5];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    sizeptr = Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*sizeptr == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, "qddb_table cget: bad column name \"", col_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (strcmp(option, "-type") == 0) {
	option = TclQddb_GetType(table->table[0]+*sizeptr, table->table[0][*sizeptr].data.type);
    } else {
	option = TclQddb_HeaderGetInfo(table->table[0][*sizeptr].info, NULL, option);
    }
    Free(sizeptr);
    if (option == NULL) {
	Tcl_AppendResult(interp, "qddb_table col cget: bad option \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    } else {
	Tcl_SetResult(interp, option, TCL_VOLATILE);
    }
    return TCL_OK;
}

static int TclQddb_ColConfigure(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*col;
    Qddb_TableNode		*cell;
    char			*table_name, *col_name;
    size_t			*colptr, num_rows, col_num;
    int				i, relink = 0;
    int				j;

    if (argc < 5 || (argc%2) != 1) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    col_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name)) == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			 ": bad col name \"", col_name, "\"", NULL);
	return TCL_ERROR;
    }
    col = table->table[0] + *colptr;
    col_num = *colptr;
    Free(colptr);
    num_rows = table->rows;
    for (i = 5; i < argc; i+=2) {
	if (strcmp(argv[i], "-type") == 0) {
	    TclQddb_UnlinkTable(interp, table);
	    relink = 1;
	    for (j = 1; j <= num_rows; j++) {
		cell = table->table[j] + col_num;
		if (cell->data.type == QDDB_TABLE_TYPE_DEFAULT &&
		    TclQddb_TableConfigureColType(interp, cell, col->data.type, argv[i+1]) != TCL_OK)
		    return TCL_ERROR;
	    }
	    if (TclQddb_TableConfigureType(interp, col, col->data.type, argv[i+1]) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-calc") == 0) {
	    char	*my_argv[5];

	    my_argv[0] = "qddb_table";
	    my_argv[1] = "col";
	    my_argv[2] = "configure";
	    my_argv[3] = table_name;
	    my_argv[4] = col_name;

	    TclQddb_UnlinkTable(interp, table);
	    relink = 1;
	    for (j = 1; j <= num_rows; j++) {
		cell = table->table[j] + col_num;
		if (cell->data.type == QDDB_TABLE_TYPE_DEFAULT &&
		    TclQddb_TableConfigureColType(interp, cell, col->data.type, "calc") != TCL_OK)
		    return TCL_ERROR;
	    }
	    if (TclQddb_TableConfigureType(interp, col, col->data.type, "calc") != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (TclQddb_SetCellValue(interp, col, col->data.type, argv[i+1]) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (TclQddb_ColEval(interp, 5, my_argv) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    if (!relink && (strcmp(argv[i], "-name") == 0 || strcmp(argv[i], "-variable") == 0)) {
		TclQddb_UnlinkTable(interp, table);
		relink = 1;
	    }
	    if (TclQddb_TableSetHeaderInfo(interp, &col->info, argv[i], argv[i+1], 1) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
				 ": bad option \"", argv[i], "\"", NULL);
		return TCL_ERROR;
	    }
	}
    }
    if (relink) {
	if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_ColClear(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 5) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    (void)Qddb_TableOp(table, QDDB_TABLE_OP_CLEARCOL, sizeptr);
    Free(sizeptr);
    return TCL_OK;
}


static int TclQddb_ColCopy(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableFromTo		fromto;
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    fromto.from = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[5])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[5], "\"", NULL);
	return TCL_ERROR;
    }
    fromto.to = *sizeptr;
    Free(sizeptr);
    TclQddb_UnlinkTable(interp, table);
    if (Qddb_TableOp(table, QDDB_TABLE_OP_COPYCOL, &fromto) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot copy column", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_ColEval(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		**rows, *row0, *cell;
    Qddb_TableCalc		*cell_calc;
    char			*table_name, *col_name;
    int				default_type, retval = TCL_OK;
    double			num;
    size_t			i, *colptr, colnum;

    if (argc != 5) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    col_name = argv[4];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    row0 = table->table[0];
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad column name \"",
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    colnum = *colptr;
    Free(colptr);
    rows = table->table;
    for (i = 1; i <= table->rows; i++) {
	cell = rows[i] + colnum;
	default_type = cell->data.type;
	cell_calc = cell->data.compiled;
	if (default_type == QDDB_TABLE_TYPE_DEFAULT) {
	    default_type = row0[colnum].data.type;
	    if (default_type == QDDB_TABLE_TYPE_CALC) {
		cell_calc = row0[colnum].data.compiled;
	    }
	}
	if (default_type == QDDB_TABLE_TYPE_CALC) {
	    if (TclQddb_TableExprInterp(interp, cell_calc, table, i, colnum,
					NULL, &num) != TCL_OK) {
		cell->data.data.real = 0.00;
		retval = TCL_ERROR;
	    } else {
		cell->data.data.real = num;
	    }
	}
    }
    return retval;
}


/* BEGIN CELL COMMANDS */

static int TclQddb_ProcessCellCommands(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1],
			 "wrong # args, should >= 3", NULL);
    }
    switch (argv[2][0]) {
    case 'c': /* cget, configure, clear*/
	if (strcmp("cget", argv[2]) == 0) {
	    if (TclQddb_CellCget(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("configure", argv[2]) == 0) {
	    if (TclQddb_CellConfigure(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("clear", argv[2]) == 0) {
	    if (TclQddb_CellClear(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp("copy", argv[2]) == 0) {
	    if (TclQddb_CellCopy(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	break;
    case 'g': /* getval */
	if (strcmp("getval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_CellGetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 's': /* setval */
	if (strcmp("setval", argv[2]) != 0) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1],
			     ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_CellSetVal(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'e':
	if (strcmp("eval", argv[2]) == 0) {
	    if (TclQddb_CellEval(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	    return TCL_ERROR;
	}
	break;
    default:
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid command ", argv[2], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_CellSetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, *cell;
    char			*table_name, *row_name, *col_name;
    char			*value;
    size_t			*rowptr, *colptr, rownum, colnum;
    int				cell_type;

    /* qddb_table cell setval <table_desc> <row> <column <value> */
    if (argc != 7) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], 
			 ": wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    row_name = argv[4];
    col_name = argv[5];
    value = argv[6];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], 
			 ": bad table descriptor\"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad row name \"",
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(rowptr);
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad column name \"",
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    rownum = *rowptr;
    colnum = *colptr;
    Free(colptr);
    Free(rowptr);
    cell = table->table[rownum] + colnum;
    row0 = table->table[0] + colnum;
    cell_type = cell->data.type;
    if (cell_type == QDDB_TABLE_TYPE_DEFAULT) {
	cell_type = row0->data.type;
    }
    if (TclQddb_SetCellValue(interp, cell, row0->data.type, value) != TCL_OK) {
	return TCL_ERROR;
    }
    if (cell_type == QDDB_TABLE_TYPE_CALC) {
	double		num;

	if (TclQddb_TableExprInterp(interp, cell->data.compiled, table, rownum, colnum, 
				    &(cell->data.refs), &num) != TCL_OK) {
	    cell->data.data.real = 0.0;
	    return TCL_ERROR;
	} else {
	    cell->data.data.real = num;
	}
    }
    return TCL_OK;
}

static int TclQddb_CellGetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *row_name, *column_name, *value;
    size_t			*rowptr, *colptr;

    /* qddb_table cell getval <table_desc> <row name> <column name> */
    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # of args", NULL);
	return TCL_ERROR;	
    }
    table_name = argv[3];
    row_name = argv[4];
    column_name = argv[5];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			 ": bad table \"", table_name, "\"", NULL);
	return TCL_ERROR;
    }
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], 
			 ": bad row name \"", row_name, "\"", NULL);
	return TCL_ERROR;
    }
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, column_name);
    if (*colptr == 0) {
	Free(rowptr);
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], 
			 ": bad column name \"", column_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (table->table[0][*colptr].info != NULL)
	value = TclQddb_GetCellValue(table->table[*rowptr] + *colptr,
				     table->table[0][*colptr].data.type,
				     table->table[0][*colptr].info->print);
    else
	value = TclQddb_GetCellValue(table->table[*rowptr] + *colptr, 
				     table->table[0][*colptr].data.type, NULL);
    Tcl_SetResult(interp, value, TCL_VOLATILE);
    Free(rowptr);
    Free(colptr);
    return TCL_OK;
}

static int TclQddb_CellCget(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    char			*table_name, *option, *row_name, *col_name;
    size_t			*rowptr, *colptr;
    int				default_type;

    /* qddb_table cell cget <table> <row> <col> <option> */
    if (argc != 7) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    col_name = argv[5];
    option = argv[6];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    rowptr = Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, "qddb_table cget: bad row name \"", row_name, "\"", NULL);
	return TCL_ERROR;
    }
    colptr = Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(rowptr);
	Free(colptr);
	Tcl_AppendResult(interp, "qddb_table cell cget: bad column name \"", col_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (strcmp(option, "-type") == 0) {
	option = TclQddb_GetType(table->table[*rowptr]+*colptr, table->table[0][*colptr].data.type);
    } else if (strcmp(option, "-calc") == 0) {
	default_type = table->table[*rowptr][*colptr].data.type;
	if (default_type == QDDB_TABLE_TYPE_DEFAULT)
	    default_type = table->table[0][*colptr].data.type;
	if (default_type != QDDB_TABLE_TYPE_CALC) {
	    Tcl_AppendResult(interp, "qddb_table cell cget: cell is not of type \"calc\"", NULL);
	    option = NULL;
	} else {
	    option = table->table[*rowptr][*colptr].data.calc;
	    if (option == NULL)
		option = Calloc(1);
	}
    } else {
	option = TclQddb_HeaderGetInfo(table->table[*rowptr][*colptr].info, 
				       table->table[0][*colptr].info, option);
    }
    Free(rowptr);
    Free(colptr);
    if (option == NULL) {
	Tcl_AppendResult(interp, "qddb_table cell cget: bad option \"", option, "\"", NULL);
	return TCL_ERROR;
    } else {
	Tcl_SetResult(interp, option, TCL_VOLATILE);
    }
    return TCL_OK;
}

static int TclQddb_CellConfigure(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*cell;
    char			*table_name, *col_name, *row_name;
    size_t			*rowptr, *colptr, row, col;
    int				i, relink = 0, default_type, retval = TCL_OK;

    /* qddb_table cell configure <table> <row> <col> <options> */
    if (argc < 6 || (argc%2) != 0) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    col_name = argv[5];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if ((row=*(rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name))) == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			 ": bad row name \"", row_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((col=*(colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name))) == 0) {
	Free(rowptr);
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
			 ": bad column name \"", col_name, "\"", NULL);
	return TCL_ERROR;
    }
    cell = table->table[*rowptr] + *colptr;
    Free(rowptr);
    Free(colptr);
    default_type = table->table[0][col].data.type;
    for (i = 6; i < argc; i+=2) {
	if (strcmp(argv[i], "-type") == 0) {
	    TclQddb_UnlinkTable(interp, table);
	    relink = 1;
	    if (TclQddb_TableConfigureType(interp, cell, default_type, argv[i+1]) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-calc") == 0) {
	    double		num;

	    TclQddb_UnlinkTable(interp, table);
	    relink = 1;
	    if (TclQddb_TableConfigureType(interp, cell, default_type, "calc") != TCL_OK)
		return TCL_ERROR;
	    if (TclQddb_SetCellValue(interp, cell, default_type, argv[i+1]) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (TclQddb_TableExprInterp(interp, cell->data.compiled, table, row, col, 
					&(cell->data.refs), &num) != TCL_OK) {
		cell->data.data.real = 0.00;
		retval = TCL_ERROR;
	    } else {
		cell->data.data.real = num;
	    }
	} else {
	    if (!relink && (strcmp(argv[i], "-name") == 0 || strcmp(argv[i], "-variable") == 0)) {
		TclQddb_UnlinkTable(interp, table);
		relink = 1;
	    }
	    if (TclQddb_TableSetHeaderInfo(interp, &cell->info, argv[i], argv[i+1], 1) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
				 ": bad option \"", argv[i], "\"", NULL);
		return TCL_ERROR;
	    }
	}
    }
    if (relink)
	TclQddb_RelinkTable(interp, table);
    return retval;
}

static int TclQddb_CellClear(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableRowCol		rowcol;
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol.row = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[5])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[5], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol.column = *sizeptr;
    Free(sizeptr);
    (void)Qddb_TableOp(table, QDDB_TABLE_OP_CLEARCELL, &rowcol);
    return TCL_OK;
}

static int TclQddb_CellCopy(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableRowCol		rowcol[2];
    char			*table_name;
    size_t			*sizeptr;

    if (argc != 8) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[4])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[4], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol[0].row = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[5])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[5], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol[0].column = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, argv[6])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad row name \"", argv[6], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol[1].row = *sizeptr;
    Free(sizeptr);
    if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, argv[7])) == 0) {
	Free(sizeptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": bad column name \"", argv[7], "\"", NULL);
	return TCL_ERROR;
    }
    rowcol[1].column = *sizeptr;
    Free(sizeptr);
    TclQddb_UnlinkTable(interp, table);
    if (Qddb_TableOp(table, QDDB_TABLE_OP_COPYCELL, rowcol) == NULL) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], ": cannot copy cell", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_RelinkTable(interp, table) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

/* MISC SUPPORT ROUTINES */

static char *TclQddb_HeaderGetInfo(info, default_info, option)
    Qddb_TableHeaderInfo	*info, *default_info;
    char			*option;
{
    char			*val = NULL;
    static char			buf[MAXINTEGERLEN];

    if (info != NULL) { /* things without a column default */
	if (strcmp(option, "-variable") == 0) {
	    val = info->variable;
	    return val == NULL? "":val;
	} else if (strcmp(option, "-name") == 0) {
	    val = info->name;
	    return val == NULL? "":val;
	} else if (strcmp(option, "-title") == 0) {
	    val = info->title;
	    return val == NULL? "":val;
	} else if (strcmp(option, "-comment") == 0) {
	    val = info->comment;
	    return val == NULL? "":val;
	}
    }
    if (info == NULL || info->print == NULL)
	info = default_info;
    if (strcmp(option, "-width") == 0) {
	if (info == NULL || info->print == NULL)
	    return "0";
	sprintf(buf, "%d", info->print->width);
	return buf;
    } else if (strcmp(option, "-justify") == 0) {
	if (info == NULL || info->print == NULL)
	    return "left";
	switch (info->print->justify) {
	case QDDB_TABLE_JUSTIFY_RIGHT:
	    return "right";
	case QDDB_TABLE_JUSTIFY_CENTER:
	    return "center";
	case QDDB_TABLE_JUSTIFY_NONE:
	    return "none";
	case QDDB_TABLE_JUSTIFY_DEFAULT:
	case QDDB_TABLE_JUSTIFY_LEFT:
	default:
	    return "left";
	}
    } else if (strcmp(option, "-precision") == 0) {
	if (info == NULL || info->print == NULL)
	    return "0";
	sprintf(buf, "%d", info->print->precision);
	return buf;
    } else if (strcmp(option, "-separator") == 0) {
	if (info == NULL || info->print == NULL || info->print->separator == NULL)
	    return " ";
	return info->print->separator;
    }
    return NULL;
}

int TclQddb_TableSetHeaderInfo(interp, info, option, value, print_options)
    Tcl_Interp			*interp;
    Qddb_TableHeaderInfo	**info;
    char			*option, *value;
    int				print_options;
{
    Qddb_TableHeaderInfo	*linfo;

    if (*info == NULL) {
	*info = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
    }
    linfo = *info;
    if (value == NULL)
	return TCL_ERROR;
    if (strcmp(option, "-variable") == 0) {
	if (linfo->variable != NULL) {
	    Free(linfo->variable);
	}
	if (value[0] == '\0') {
	    linfo->variable = NULL;
	} else {
	    linfo->variable = (char *)Malloc(strlen(value)+1);
	    strcpy(linfo->variable, value);
	}
    } else if (strcmp(option, "-name") == 0) {
	if (linfo->name != NULL) {
	    Free(linfo->name);
	}
	if (value[0] == '\0') {
	    linfo->name = NULL;
	} else {
	    linfo->name = (char *)Malloc(strlen(value)+1);
	    strcpy(linfo->name, value);
	}
    } else if (strcmp(option, "-title") == 0) {
	if (linfo->title != NULL) {
	    Free(linfo->title);
	}
	if (value[0] == '\0') {
	    linfo->name = NULL;
	} else {
	    linfo->title = (char *)Malloc(strlen(value)+1);
	    strcpy(linfo->title, value);
	}
    } else if (strcmp(option, "-comment") == 0) {
	if (linfo->comment != NULL) {
	    Free(linfo->comment);
	}
	if (value[0] == '\0') {
	    linfo->name = NULL;
	} else {
	    linfo->comment = (char *)Malloc(strlen(value)+1);
	    strcpy(linfo->comment, value);
	}
    } else if (print_options) {
	if (linfo->print == NULL) {
	    linfo->print = (Qddb_TableHeaderFormat *)Calloc(sizeof(Qddb_TableHeaderFormat));
	}
	if (strcmp(option, "-justify") == 0) {
	    if (strcmp("left", value) == 0) {
		linfo->print->justify = QDDB_TABLE_JUSTIFY_LEFT;
	    } else if (strcmp("right", value) == 0) {
		linfo->print->justify = QDDB_TABLE_JUSTIFY_RIGHT;
	    } else if (strcmp("center", value) == 0) {
		linfo->print->justify = QDDB_TABLE_JUSTIFY_CENTER;
	    } else if (strcmp("none", value) == 0) {
		linfo->print->justify = QDDB_TABLE_JUSTIFY_NONE;
	    } else {
		Tcl_AppendResult(interp, "bad justification parameter \"", value, "\"", NULL);
	    }
	} else if (strcmp(option, "-width") == 0) {
	    if (Tcl_GetInt(interp, value, &(linfo->print->width)) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(option, "-precision") == 0) {
	    if (Tcl_GetInt(interp, value, &(linfo->print->precision)) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(option, "-separator") == 0) {
	    if (linfo->print->separator != NULL) {
		Free(linfo->print->separator);
	    }
	    linfo->print->separator = (char *)Malloc(strlen(value)+1);
	    strcpy(linfo->print->separator, value);
	}
    } else {
	return TCL_ERROR;
    }
    return TCL_OK;
}

static char *TclQddb_GetType(node, coltype)
    Qddb_TableNode		*node;
    int				coltype;
{
    int				type;

    type = node->data.type;
    if (type == QDDB_TABLE_TYPE_DEFAULT) {
	type = coltype;
    }
    switch (type) {
    case QDDB_TABLE_TYPE_INTEGER:
	return "integer";
    case QDDB_TABLE_TYPE_REAL:
	return "real";
    case QDDB_TABLE_TYPE_DATE:
	return "date";
    case QDDB_TABLE_TYPE_CALC:
	return "calc";
    case QDDB_TABLE_TYPE_STRING:
    default:
	return "string";
    }
}

static void TclQddb_FreeStringAndCalcData(node, type)
    Qddb_TableNode		*node;
    int				type;
{
    if (QDDB_TABLE_TYPE_USESTRING(type)) {
	if (node->data.data.string != NULL)
	    Free(node->data.data.string);
    } else if (type == QDDB_TABLE_TYPE_CALC) {
	if (node->data.calc != NULL) {
	    Free(node->data.calc);
	    node->data.calc = NULL;
	}
	if (node->data.compiled != NULL) {
	    Qddb_Free(QDDB_TYPE_TABLECALC, node->data.compiled);
	    node->data.compiled = NULL;
	}
    }    
}

int TclQddb_TableConfigureType(interp, node, coltype, value)
    Tcl_Interp			*interp;
    Qddb_TableNode		*node;
    int				coltype;
    char			*value;
{
    int				retval = TCL_OK, type;

    type = node->data.type;
    if (type == QDDB_TABLE_TYPE_DEFAULT) {
	type = coltype;
    }
    switch (value[0]) {
    case 'i':
	if (strcmp("integer", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	if (type == QDDB_TABLE_TYPE_INTEGER)
	    break;
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.type = QDDB_TABLE_TYPE_INTEGER;
	node->data.data.integer = 0;
	break;
    case 'r':
	if (strcmp("real", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	if (type == QDDB_TABLE_TYPE_REAL)
	    break;
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.type = QDDB_TABLE_TYPE_REAL;
	node->data.data.real = 0.0;
	break;
    case 's':
	if (strcmp("string", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	if (type == QDDB_TABLE_TYPE_STRING)
	    break;
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.type = QDDB_TABLE_TYPE_STRING;
	node->data.data.string = NULL;
	break;
    case 'd':
	if (strcmp("date", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	if (type == QDDB_TABLE_TYPE_DATE)
	    break;
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.type = QDDB_TABLE_TYPE_DATE;
	node->data.data.string = NULL;
	break;
    case 'c':
	if (strcmp("calc", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	if (type == QDDB_TABLE_TYPE_CALC)
	    break;
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.type = QDDB_TABLE_TYPE_CALC;
	node->data.data.real = 0.0;
	break;
    default:
	retval = TCL_ERROR;
    }
    if (retval != TCL_OK) {
	Tcl_AppendResult(interp, "invalid type: \"", value, "\"", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

int TclQddb_TableConfigureColType(interp, node, coltype, value)
    Tcl_Interp			*interp;
    Qddb_TableNode		*node;
    int				coltype;
    char			*value;
{
    int				retval = TCL_OK, type;

    if (node->data.type != QDDB_TABLE_TYPE_DEFAULT) {
	return retval;
    }
    type = coltype;
    switch (value[0]) {
    case 'i':
	if (strcmp("integer", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.data.integer = 0;
	break;
    case 'r':
	if (strcmp("real", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.data.real = 0.0;
	break;
    case 's':
	if (strcmp("string", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.data.string = NULL;
	break;
    case 'd':
	if (strcmp("date", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.data.string = NULL;
	break;
    case 'c':
	if (strcmp("calc", value) != 0) {
	    retval = TCL_ERROR;
	    break;
	}
	TclQddb_FreeStringAndCalcData(node, type);
	node->data.data.real = 0.0;
	break;
    default:
	retval = TCL_ERROR;
    }
    if (retval != TCL_OK) {
	Tcl_AppendResult(interp, "invalid type: \"", value, "\"", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


char *TclQddb_GetCellValue(cell, default_type, default_format)
    Qddb_TableNode		*cell;
    int				default_type;
    Qddb_TableHeaderFormat	*default_format;
{
    int				thistype;
    static char			buf[4*1024]; /* a little overkill */

    thistype = cell->data.type;
    if (thistype == QDDB_TABLE_TYPE_DEFAULT) {
	thistype = default_type;
    }
    switch (thistype) {
    case QDDB_TABLE_TYPE_INTEGER:
	if (cell->info != NULL && cell->info->print != NULL && cell->info->print->format != NULL) {
	    sprintf(buf, cell->info->print->format, cell->data.data.integer);
	} else if (default_format != NULL && default_format->format != NULL) {
	    sprintf(buf, default_format->format, cell->data.data.integer);
	} else {
	    sprintf(buf, "%d", cell->data.data.integer);
	}
	return buf;
    case QDDB_TABLE_TYPE_CALC:
    case QDDB_TABLE_TYPE_REAL:
	if (cell->info != NULL && cell->info->print != NULL) {
	    if (cell->info->print->format != NULL)
		sprintf(buf, cell->info->print->format, cell->data.data.real);
	    else
		sprintf(buf, "%.*f", cell->info->print->precision, cell->data.data.real);
	} else if (default_format != NULL) {
	    if (default_format->format != NULL) {
		sprintf(buf, default_format->format, cell->data.data.real);
	    } else {
		sprintf(buf, "%.*f", default_format->precision, cell->data.data.real);
	    }
	} else {
	    sprintf(buf, "%f", cell->data.data.real);
	}
	return buf;
    case QDDB_TABLE_TYPE_DATE:
    case QDDB_TABLE_TYPE_STRING:
    default: /* default type is string */
	if (cell->data.data.string != NULL)
	    return cell->data.data.string;
	else
	    return "";
    }
}

static int TclQddb_SetCellValue(interp, cell, default_type, value)
    Tcl_Interp			*interp;
    Qddb_TableNode		*cell;
    int				default_type;
    char			*value;
{
    int				thistype;

    thistype = cell->data.type;
    if (thistype == QDDB_TABLE_TYPE_DEFAULT) {
	thistype = default_type;
    }
    switch (thistype) {
    case QDDB_TABLE_TYPE_INTEGER:
	if (Tcl_GetInt(interp, value, &cell->data.data.integer) != TCL_OK) {
	    Tcl_AppendResult(interp, "cannot set element: \"", value,
			     "\" must be an integer", NULL);
	    return TCL_ERROR;
	}
	break;
    case QDDB_TABLE_TYPE_REAL:
	if (Tcl_GetDouble(interp, value, &cell->data.data.real) != TCL_OK) {
	    Tcl_AppendResult(interp, "cannot set element: \"", value,
			     "\" must be a real number", NULL);
	    return TCL_ERROR;
	}
	break;
    case QDDB_TABLE_TYPE_DATE:
	if (cell->data.data.string != NULL)
	    Free(cell->data.data.string);
	if (Qddb_CheckDate(value)) {
	    Tcl_AppendResult(interp, "cannot set element: \"", value,
			     "\" must be a valid date", NULL);
	    return TCL_ERROR;
	}
	cell->data.data.string = (char *)Malloc(strlen(value)+1);
	strcpy(cell->data.data.string, value);
	break;
    case QDDB_TABLE_TYPE_CALC:
	if (cell->data.calc != NULL)
	    Free(cell->data.calc);
	if (cell->data.compiled != NULL)
	    Qddb_Free(QDDB_TYPE_TABLECALC, cell->data.compiled);
	cell->data.calc = (char *)Malloc(strlen(value)+1);
	strcpy(cell->data.calc, value);
	cell->data.compiled = Qddb_TableCalcParse(cell->data.calc);
	if (cell->data.compiled == NULL) {
	    Tcl_AppendResult(interp, "syntax error in expression", NULL);
	    return TCL_ERROR;
	}
	cell->data.type = QDDB_TABLE_TYPE_CALC;
	cell->data.data.real = 0.00;
	break;
    case QDDB_TABLE_TYPE_STRING:
    default:
	if (cell->data.data.string != NULL)
	    Free(cell->data.data.string);
	cell->data.data.string = (char *)Malloc(strlen(value)+1);
	strcpy(cell->data.data.string, value);
	break;
    }
    return TCL_OK;
}

static int TclQddb_CellEval(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Qddb_Table			*table;
    Qddb_TableNode		*row0, *cell;
    Qddb_TableCalc		*cell_calc;
    char			*table_name, *col_name, *row_name;
    int				default_type;
    double			num;
    size_t			*colptr, colnum, *rowptr, rownum;

    if (argc != 6) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    table_name = argv[3];
    row_name = argv[4];
    col_name = argv[5];
    if ((table = TclQddb_GetTable(interp, table_name)) == NULL)
	return TCL_ERROR;
    rowptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW, row_name);
    if (*rowptr == 0) {
	Free(rowptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad row name \"",
			 row_name, "\"", NULL);
	return TCL_ERROR;
    }
    rownum = *rowptr;
    Free(rowptr);
    colptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL, col_name);
    if (*colptr == 0) {
	Free(colptr);
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2], ": bad column name \"",
			 col_name, "\"", NULL);
	return TCL_ERROR;
    }
    colnum = *colptr;
    Free(colptr);
    row0 = table->table[0];
    cell = table->table[rownum] + colnum;
    default_type = cell->data.type;
    cell_calc = cell->data.compiled;
    if (default_type == QDDB_TABLE_TYPE_DEFAULT) {
	default_type = row0[colnum].data.type;
	if (default_type == QDDB_TABLE_TYPE_CALC) {
	    cell_calc = row0[colnum].data.compiled;
	}
    }
    if (default_type == QDDB_TABLE_TYPE_CALC) {
	if (TclQddb_TableExprInterp(interp, cell_calc, table, rownum, colnum,
				    NULL, &num) != TCL_OK) {
	    cell->data.data.real = 0.00;
	    return TCL_ERROR;
	}
	cell->data.data.real = num;
    }
    return TCL_OK;
}

/* END COMMANDS */

static char ***TclQddb_TableGetStrings(table)
    Qddb_Table			*table;
{
    Qddb_TableNode		*cell;
    int				i, j, maxwidth;
    char			***retval, *cellval;

    retval = (char ***)Malloc(sizeof(char **)*(table->rows+2));
    for (i = 0; i <= table->rows; i++) {
	retval[i] = (char **)Malloc(sizeof(char *)*(table->columns+1));
    }
    retval[0][0] = Calloc(QDDB_MIN_MALLOC_SIZE);
    for (i = 1; i <= table->columns; i++) { /* column titles */
	cell = table->table[0] + i;
	if (cell->info != NULL && cell->info->title != NULL) {
	    cellval = cell->info->title;
	    retval[0][i] = (char *)Malloc(strlen(cellval)+1);
	    strcpy(retval[0][i], cellval);
	} else {
	    retval[0][i] = (char *)Malloc((size_t)MAXINTEGERLEN);
	    sprintf(retval[0][i], "%d", i);
	}
    }
    maxwidth = 0;
    for (i = table->rows; i > 0; i /= 10)
	maxwidth++;
    for (i = 1; i <= table->rows; i++) { /* row titles */
	cell = table->table[i];
	if (cell->info != NULL && cell->info->title != NULL) {
	    cellval = cell->info->title;
	    retval[i][0] = (char *)Malloc(strlen(cellval)+1);
	    strcpy(retval[i][0], cellval);
	} else {
	    retval[i][0] = (char *)Malloc((size_t)MAXINTEGERLEN);
	    sprintf(retval[i][0], "%0*d", maxwidth, i);
	}
    }
    for (i = 1; i <= table->rows; i++) {
	for (j = 1; j <= table->columns; j++) {
	    cell = table->table[i] + j;
	    if (table->table[0][j].info != NULL)
		cellval = TclQddb_GetCellValue(cell, table->table[0][j].data.type, 
					       table->table[0][j].info->print);
	    else
		cellval = TclQddb_GetCellValue(cell, table->table[0][j].data.type, NULL);
	    if (cellval == NULL) {
		retval[i][j] = (char *)Calloc(1);
	    } else {
		retval[i][j] = (char *)Malloc(strlen(cellval)+1);
		strcpy(retval[i][j], cellval);
	    }
	}
    }
    return retval;
}

char **TclQddb_GetFormattedTable(interp, table, sepstring, coltitles, rowtitles)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    char			*sepstring;
    int				coltitles, rowtitles;
{
    Qddb_TableNode		*cell;
    char			**retval, ***rawtbl, *buf, *ch, *colsep, **colseparators;
    size_t			len;
    int				*colwidth, *coljustify, justify, width, maxwidth;
    int				i, j;

    if (sepstring == NULL)
	sepstring = " ";
    retval = (char **)Malloc(sizeof(char *)*(table->rows+2));
    retval[0] = NULL;
    retval[table->rows+1] = NULL;
    rawtbl = TclQddb_TableGetStrings(table);
    colwidth = (int *)Malloc(sizeof(int)*(table->columns+1));
    coljustify = (int *)Calloc(sizeof(int)*(table->columns+1));
    colseparators = (char **)Calloc(sizeof(char *)*(table->columns+1));
    maxwidth = 0;
    for (i = 0; i <= table->columns; i++) {
	cell = table->table[0] + i;
	if (cell->info != NULL && cell->info->title != NULL)
	    colwidth[i] = strlen(cell->info->title);
	else
	    colwidth[i] = 0;
	if (cell->info != NULL && cell->info->print != NULL) {
	    if (cell->info->print->width != 0) {
		colwidth[i] = cell->info->print->width;
	    } else {
		for (j = 1; j <= table->rows; j++) {
		    if ((ch = index(rawtbl[j][i], (int)'\n')) != NULL)
			*ch = '\0';
		    colwidth[i] = MAX(colwidth[i], strlen(rawtbl[j][i]));
		}
	    }
	    coljustify[i] = cell->info->print->justify;
	    if (cell->info->print->separator == NULL)
		colseparators[i] = sepstring;
	    else
		colseparators[i] = cell->info->print->separator;
	} else {
	    for (j = 1; j <= table->rows; j++) {
		if ((ch = index(rawtbl[j][i], (int)'\n')) != NULL)
		    *ch = '\0';
		colwidth[i] = MAX(colwidth[i], strlen(rawtbl[j][i]));
	    }
	    coljustify[i] = QDDB_TABLE_JUSTIFY_DEFAULT;
	    colseparators[i] = sepstring;
	}
	maxwidth = MAX(maxwidth, colwidth[i]);
    }
    buf = (char *)Malloc((size_t)MAX(maxwidth+1, BUFSIZ));
    for (i = (coltitles == 0)? 1:0; i <= table->rows; i++) {
	Qddb_InitBuffer();
	for (j = (rowtitles == 0)? 1:0; j <= table->columns; j++) {
	    cell = table->table[i] + j;
	    if (cell->info == NULL || cell->info->print == NULL) {
		justify = coljustify[j];
	    } else {
		justify = cell->info->print->justify;
	    }
	    colsep = colseparators[j];
	    width = colwidth[j];
	    if (width == 0)
		continue;
	    if (justify != QDDB_TABLE_JUSTIFY_NONE) {
		Qddb_FillChars(buf, (int)' ', width);
	    } else {
		Qddb_FillChars(buf, (int)'\0', width);
	    }
	    buf[width] = '\0';
	    len = strlen(rawtbl[i][j]);
	    switch (justify) {
	    case QDDB_TABLE_JUSTIFY_DEFAULT:
	    case QDDB_TABLE_JUSTIFY_LEFT:
	    case QDDB_TABLE_JUSTIFY_NONE:
		bcopy(rawtbl[i][j], buf, MIN(width, len));
		Qddb_ConcatPrintableBuffer(buf);
		if (j < table->columns)
		    Qddb_ConcatPrintableBuffer(colsep);
		break;
	    case QDDB_TABLE_JUSTIFY_RIGHT:
		bcopy(rawtbl[i][j], buf+MAX(width-len, 0), MIN(width, len));
		Qddb_ConcatPrintableBuffer(buf);
		if (j < table->columns)
		    Qddb_ConcatPrintableBuffer(colsep);
		break;
	    case QDDB_TABLE_JUSTIFY_CENTER:
		len = MIN(width, len);
		bcopy(rawtbl[i][j], buf+(width/2)-(MIN(len, width)/2), MIN(width, len));
		Qddb_ConcatPrintableBuffer(buf);
		if (j < table->columns)
		    Qddb_ConcatPrintableBuffer(colsep);
		break;
	    default:
		PANIC("TclQddb_TableGetFormattedTable: bad justification setting for table node");
	    }
	}
	retval[i] = Qddb_GetBuffer();
    }
    for (i = 0; i <= table->rows; i++) {
	for (j = 0; j <= table->columns; j++) {
	    Free(rawtbl[i][j]);
	}
	Free(rawtbl[i]);
    }
    Free(rawtbl);
    Free(buf);
    Free(colwidth);
    Free(coljustify);
    Free(colseparators);
    return retval;
}

static int TclQddb_RelinkTable(interp, table)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
{
    Qddb_TableNode	*cell, *row0;
    int			dovar = 0;
    char		variable[BUFSIZ], *rowname, *colname;
    size_t		i, j, varlen = 0, namelen;

    if (table->info == NULL || table->info->variable == NULL || table->info->variable[0] == '\0') {
	return TCL_OK;
    }
    for (i = 1; i <= table->rows; i++) {
	if (TclQddb_RelinkRow(interp, table, i) != TCL_OK)
	    return TCL_ERROR;
    }
    for (i = 1; i <= table->columns; i++) {
	if (TclQddb_RelinkCol(interp, table, i) != TCL_OK)
	    return TCL_ERROR;
    }
    row0 = table->table[0];
    if (table->info != NULL && table->info->variable != NULL && table->info->variable[0] != '\0') {
	strcpy(variable, table->info->variable);
	varlen = strlen(variable);
	strcat(variable, "(");
	varlen++;
	dovar = 1;
    }
    for (i = 1; i <= table->rows; i++) {
	if (table->table[i][0].info != NULL) {
	    rowname = table->table[i][0].info->name;
	} else {
	    rowname = NULL;
	}
	for (j = 1; j <= table->columns; j++) {
	    if (TclQddb_RelinkCell(interp, table, i, j, NULL) != TCL_OK)
		return TCL_ERROR;
	    if (row0[j].info != NULL) {
		colname = row0[j].info->name;
	    } else {
		colname = NULL;
	    }
	    if (dovar) {
		sprintf(variable+varlen, "%d,%d", (int)i, (int)j);
		strcat(variable, ")");
		if (TclQddb_RelinkCell(interp, table, i, j, variable) != TCL_OK)
		    return TCL_ERROR;
		if (rowname != NULL && colname != NULL) {
		    sprintf(variable+varlen, "%s,%s", rowname, colname);
		    strcat(variable, ")");
		    if (TclQddb_RelinkCell(interp, table, i, j, variable) != TCL_OK)
			return TCL_ERROR;
		}
		if (rowname != NULL) {
		    sprintf(variable+varlen, "%s,%d", rowname, (int)j);
		    strcat(variable, ")");
		    if (TclQddb_RelinkCell(interp, table, i, j, variable) != TCL_OK)
			return TCL_ERROR;
		}
		if (colname != NULL) {
		    sprintf(variable+varlen, "%d,%s", (int)i, colname);
		    strcat(variable, ")");
		    if (TclQddb_RelinkCell(interp, table, i, j, variable) != TCL_OK)
			return TCL_ERROR;
		}
		cell = table->table[i] + j;
		if (cell->info != NULL && cell->info->name != NULL) {
		    namelen = strlen(cell->info->name);
		    if (varlen+namelen+2 > BUFSIZ) {
			continue;
		    }
		    strcpy(variable+varlen, cell->info->name);
		    variable[varlen+namelen] = ')';
		    variable[varlen+namelen+1] = '\0';
		    if (TclQddb_RelinkCell(interp, table, i, j, variable) != TCL_OK)
			return TCL_ERROR;
		}
	    }
	}
    }
    return TCL_OK;
}

static int TclQddb_RelinkRow(interp, table, row)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		row;
{
    Qddb_TableNode	*rowptr, *cell;
    char		variable[BUFSIZ];
    size_t		i, varlen, namelen;

    rowptr = table->table[row];
    if (rowptr->info == NULL || rowptr->info->variable == NULL || rowptr->info->variable[0] == '\0')
	return TCL_OK;
    strcpy(variable, rowptr->info->variable);
    varlen = strlen(variable);
    strcat(variable, "(");
    varlen++;
    for (i = 1; i <= table->columns; i++) {
	cell = table->table[row] + i;
	sprintf(variable+varlen, "%d", (int)i);
	strcat(variable, ")");
	if (TclQddb_RelinkCell(interp, table, row, i, variable) != TCL_OK)
	    return TCL_ERROR;
	if (cell->info != NULL && cell->info->name != NULL) {
	    namelen = strlen(cell->info->name);
	    if (varlen+namelen+2 > BUFSIZ) {
		Tcl_AppendResult(interp, "TclQddb_RelinkRow: variable name too long", NULL);
		return TCL_ERROR;
	    }
	    strcpy(variable+varlen, cell->info->name);
	    variable[varlen+namelen] = ')';
	    variable[varlen+namelen+1] = '\0';
	    if (TclQddb_RelinkCell(interp, table, row, i, variable) != TCL_OK)
		return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static int TclQddb_RelinkCol(interp, table, col)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		col;
{
    Qddb_TableNode	*colptr, *cell;
    char		variable[BUFSIZ];
    size_t		i, varlen, namelen;

    colptr = table->table[0] + col;
    if (colptr->info == NULL || colptr->info->variable == NULL || colptr->info->variable[0] == '\0')
	return TCL_OK;
    strcpy(variable, colptr->info->variable);
    varlen = strlen(variable);
    strcat(variable, "(");
    varlen++;
    for (i = 1; i <= table->rows; i++) {
	cell = table->table[i] + col;
	sprintf(variable+varlen, "%d", (int)i);
	strcat(variable, ")");
	if (TclQddb_RelinkCell(interp, table, i, col, variable) != TCL_OK)
	    return TCL_ERROR;
	if (cell->info != NULL && cell->info->name != NULL) {
	    namelen = strlen(cell->info->name);
	    if (varlen+namelen+2 > BUFSIZ) {
		Tcl_AppendResult(interp, "TclQddb_RelinkCol: variable name too long", NULL);
		return TCL_ERROR;
	    }
	    strcpy(variable+varlen, cell->info->name);
	    variable[varlen+namelen] = ')';
	    variable[varlen+namelen+1] = '\0';
	    if (TclQddb_RelinkCell(interp, table, i, col, variable) != TCL_OK)
		return TCL_ERROR;
	}
    }
    return TCL_OK;
}

static int TclQddb_RelinkCell(interp, table, row, col, variable)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		row, col;
    char		*variable;
{
    Qddb_TableNode	*cell, *row0 = table->table[0];
    int			celltype;
    char		*vartouse;

    cell = table->table[row] + col;
    if (variable == NULL) {
	if (cell->info == NULL || cell->info->variable == NULL || cell->info->variable[0] == '\0')
	    return TCL_OK;
	vartouse = cell->info->variable;
    } else {
	vartouse = variable;
    }
    celltype = cell->data.type;
    if (celltype == QDDB_TABLE_TYPE_DEFAULT) {
	celltype = row0[col].data.type;
    }
    Tcl_UnlinkVar(interp, vartouse);
    switch (celltype) {
    case QDDB_TABLE_TYPE_INTEGER:
	if (Tcl_LinkVar(interp, vartouse, (char *)&(cell->data.data.integer), TCL_LINK_INT) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_TYPE_REAL:
	if (Tcl_LinkVar(interp, vartouse, (char *)&(cell->data.data.real), TCL_LINK_DOUBLE) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_TYPE_CALC:
	if (Tcl_LinkVar(interp, vartouse, (char *)&(cell->data.data.real), 
			TCL_LINK_DOUBLE|TCL_LINK_READ_ONLY) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_TYPE_STRING:
    case QDDB_TABLE_TYPE_DATE:
    case QDDB_TABLE_TYPE_DEFAULT:
    default:
	if (Tcl_LinkVar(interp, vartouse, (char *)&(cell->data.data.string), TCL_LINK_STRING) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}



static void TclQddb_UnlinkTable(interp, table)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
{
    Qddb_TableNode	*cell, *row0;
    int			dovar = 0;
    char		variable[BUFSIZ], *rowname, *colname;
    size_t		i, j, varlen = 0, namelen;

    if (table->info == NULL || table->info->variable == NULL || table->info->variable[0] == '\0') {
	return;
    }
    for (i = 1; i <= table->rows; i++) {
	TclQddb_UnlinkRow(interp, table, i);
    }
    for (i = 1; i <= table->columns; i++) {
	TclQddb_UnlinkCol(interp, table, i);
    }
    row0 = table->table[0];
    if (table->info != NULL && table->info->variable != NULL) {
	strcpy(variable, table->info->variable);
	strcat(variable, "(");
	varlen = strlen(variable);
	dovar = 1;
    }
    for (i = 1; i <= table->rows; i++) {
	if (table->table[i][0].info != NULL) {
	    rowname = table->table[i][0].info->name;
	} else {
	    rowname = NULL;
	}
	for (j = 1; j <= table->columns; j++) {
	    TclQddb_UnlinkCell(interp, table, i, j, NULL);
	    if (row0[j].info != NULL) {
		colname = row0[j].info->name;
	    } else {
		colname = NULL;
	    }
	    if (dovar) {
		sprintf(variable+varlen, "%d,%d", (int)i, (int)j);
		strcat(variable, ")");
		TclQddb_UnlinkCell(interp, table, i, j, variable);
		if (rowname != NULL && colname != NULL) {
		    sprintf(variable+varlen, "%s,%s", rowname, colname);
		    strcat(variable, ")");
		    TclQddb_UnlinkCell(interp, table, i, j, variable);
		}
		if (rowname != NULL) {
		    sprintf(variable+varlen, "%s,%d", rowname, (int)j);
		    strcat(variable, ")");
		    TclQddb_UnlinkCell(interp, table, i, j, variable);
		}
		if (colname != NULL) {
		    sprintf(variable+varlen, "%d,%s", (int)i, colname);
		    strcat(variable, ")");
		    TclQddb_UnlinkCell(interp, table, i, j, variable);
		}
		cell = table->table[i] + j;
		if (cell->info != NULL && cell->info->name != NULL) {
		    namelen = strlen(cell->info->name);
		    if (varlen+namelen+2 > BUFSIZ) {
			continue;
		    }
		    strcpy(variable+varlen, cell->info->name);
		    variable[varlen+namelen] = ')';
		    variable[varlen+namelen+1] = '\0';
		    TclQddb_UnlinkCell(interp, table, i, j, variable);
		}
	    }
	}
    }
}

static void TclQddb_UnlinkRow(interp, table, row)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		row;
{
    Qddb_TableNode	*rowptr, *cell;
    char		variable[BUFSIZ];
    size_t		i, varlen, namelen;

    rowptr = table->table[row];
    if (rowptr->info == NULL || rowptr->info->variable == NULL)
	return;
    strcpy(variable, rowptr->info->variable);
    varlen = strlen(variable);
    strcat(variable, "(");
    varlen++;
    for (i = 1; i <= table->columns; i++) {
	cell = table->table[row] + i;
	sprintf(variable+varlen, "%d", (int)i);
	strcat(variable, ")");
	TclQddb_UnlinkCell(interp, table, row, i, variable);
	if (cell->info != NULL && cell->info->name != NULL) {
	    namelen = strlen(cell->info->name);
	    if (varlen+namelen+2 > BUFSIZ) {
		continue;
	    }
	    strcpy(variable+varlen, cell->info->name);
	    variable[varlen+namelen] = ')';
	    variable[varlen+namelen+1] = '\0';
	    TclQddb_UnlinkCell(interp, table, row, i, variable);
	}
    }
}

static void TclQddb_UnlinkCol(interp, table, col)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		col;
{
    Qddb_TableNode	*colptr, *cell;
    char		variable[BUFSIZ];
    size_t		i, varlen, namelen;

    colptr = table->table[0] + col;
    if (colptr->info == NULL || colptr->info->variable == NULL)
	return;
    strcpy(variable, colptr->info->variable);
    varlen = strlen(variable);
    strcat(variable, "(");
    varlen++;
    for (i = 1; i <= table->rows; i++) {
	cell = table->table[i] + col;
	sprintf(variable+varlen, "%d", (int)i);
	strcat(variable, ")");
	TclQddb_UnlinkCell(interp, table, i, col, variable);
	if (cell->info != NULL && cell->info->name != NULL) {
	    namelen = strlen(cell->info->name);
	    if (varlen+namelen+2 > BUFSIZ) {
		continue;
	    }
	    strcpy(variable+varlen, cell->info->name);
	    variable[varlen+namelen] = ')';
	    variable[varlen+namelen+1] = '\0';
	    TclQddb_UnlinkCell(interp, table, i, col, variable);
	}
    }
}

static void TclQddb_UnlinkCell(interp, table, row, col, variable)
    Tcl_Interp		*interp;
    Qddb_Table		*table;
    size_t		row, col;
    char		*variable;
{
    Qddb_TableNode	*cell;
    char		*vartouse;

    cell = table->table[row] + col;
    if (variable == NULL) {
	if (cell->info == NULL || cell->info->variable == NULL)
	    return;
	vartouse = cell->info->variable;
    } else {
	vartouse = variable;
    }
    Tcl_UnlinkVar(interp, vartouse);
}



