
/* TclQddb_Search.c - TCL interface routines for Qddb schemas.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

/* ParseRange -- Parse argc/argv for range search arguments.
 */
static int ParseRange(interp, argc, argv, args)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
    Qddb_SearchArg		*args;
{
    if (argc == 4) {
	/* all keys */
	if (argv[3][0] != '-' || argv[3][1] != '\0') {
	    Tcl_AppendResult(interp, "Bad range separator \"", argv[3], "\": should be '-'", NULL);
	    return TCL_ERROR;
	}
	args->LowerExists = False;
	args->UpperExists = False;
	args->UpperStr = NULL;
	args->LowerStr = NULL;
    } else if (argc == 5 && argv[3][0] == '-' && argv[3][1] == '\0') { 
	/* no lower limit */
	args->UpperStr = argv[4];
	args->UpperExists = True;
	args->LowerStr = NULL;
	args->LowerExists = False;
    } else if (argc == 5 && argv[4][0] == '-' && argv[4][1] == '\0') {
	/* no upper limit */
	args->UpperStr = NULL;
	args->UpperExists = False;
	args->LowerStr = argv[3];
	args->LowerExists = True;
    } else {
	if (argc != 6 || argv[4][0] != '-' || argv[4][1] != '\0') {
	    if (argc <= 4) {
		Tcl_AppendResult(interp, "Bad range separator: should be '-'", NULL);
	    } else {
		Tcl_AppendResult(interp, "Bad range separator \"", argv[4], "\": should be '-'", NULL);
	    }
	    return TCL_ERROR;
	}
	args->LowerStr = argv[3];
	args->UpperStr = argv[5];
	args->LowerExists = True;
	args->UpperExists = True;
    }
    return TCL_OK;
}

int TclQddb_Search(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Schema			*schema;
    KeyList			*result;
    char			*pruneattr;
    int				pruneattrnum, parsing;
    Qddb_SearchArg		args;
    size_t			n;

    if (argc < 3 || argc > 8) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(argv[1])) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }
    pruneattr = NULL;
    if ((argc == 4 || argc == 6 || argc == 8) && (strcmp(argv[argc-2], "word") == 0 ||
						  strcmp(argv[argc-2], "w") == 0)) {
	int                     i;

	parsing = 1; /* default */
	for (i = 2; i < argc-2; i+=2) {
	    if (strcmp("-prunebyattr", argv[i]) == 0) {
		pruneattr = argv[i+1];
	    } else if (strcmp("-parsing", argv[i]) == 0) {
		if (Tcl_GetBoolean(interp, argv[++i], &parsing) != TCL_OK) {
		    Tcl_AppendResult(interp, argv[0], ": bad argument to \"-parsing\"", NULL);
		    return TCL_ERROR;
		}
	    } else {
		Tcl_AppendResult(interp, argv[0], ": bad option \"", argv[i], "\"", NULL);
		return TCL_ERROR;
	    }
	}
	args.Alpha = argv[argc-1];
	if (parsing) {
	    args.Type = ALPHAPARSE;
	} else {
	    args.Type = ALPHANOPARSE;
	}
    } else if ((argc == 4 || argc == 6) && (strcmp(argv[argc-2], "numeric") == 0 ||
					    strcmp(argv[argc-2], "n") == 0)) {
	if (argc == 6) {
	    if (strcmp(argv[argc-4], "-prunebyattr") != 0) {
		Tcl_AppendResult(interp, argv[0], ": bad option \"", argv[argc-4], "\"", NULL);
		return TCL_ERROR;
	    }
	    pruneattr = argv[argc-3];
	    args.Number = Qddb_GetNum(argv[argc-1]);
	} else {
	    args.Number = Qddb_GetNum(argv[argc-1]);
	}
	args.Type = NUMERIC;
    } else if ((argc == 4 || argc == 6) && (strcmp(argv[argc-2], "regexp") == 0 || 
					    strcmp(argv[argc-2], "r") == 0)) {
	if (argc == 6) {
	    if (strcmp(argv[2], "-prunebyattr") != 0) {
		Tcl_AppendResult(interp, argv[0], ": bad option \"", argv[2], "\"", NULL);
		return TCL_ERROR;
	    }
	    pruneattr = argv[3];
	    args.Alpha = argv[5];
	} else {
	    args.Alpha = argv[argc-1];
	}
	args.Type = REGEXP_V8;
    } else if ((argc >= 6 && 
	       (strcmp(argv[4], "word_range") == 0 || strcmp(argv[4], "wr") == 0)) ||
	       (argc >= 4 && 
	       (strcmp(argv[2], "word_range") == 0 || strcmp(argv[2], "wr") == 0))) {
	/* qddb_search schema ?-prunebyattr <attr>? wr <keylow> - <keyhi> */
	if (strcmp(argv[2], "-prunebyattr") == 0) {
	    if (ParseRange(interp, argc-2, argv+2, &args) != TCL_OK)
		return TCL_ERROR;
	    pruneattr = argv[3];
	} else {
	    if (ParseRange(interp, argc, argv, &args) != TCL_OK)
		return TCL_ERROR;
	}
	args.Type = ALPHARANGE;
    } else if ((argc >= 6 && 
	       (strcmp(argv[4], "numeric_range") == 0 || strcmp(argv[4], "nr") == 0)) ||
	       (argc >= 4 && 
	       (strcmp(argv[2], "numeric_range") == 0 || strcmp(argv[2], "nr") == 0))) {
	/* qddb_search schema ?-prunebyattr <attr>? nr <numlow> - <numhi> */
	if (strcmp(argv[2], "-prunebyattr") == 0) {
	    if (ParseRange(interp, argc-2, argv+2, &args) != TCL_OK)
		return TCL_ERROR;
	    if (args.LowerExists == True) {
		args.LowerNum = Qddb_GetNum(args.LowerStr);
	    }
	    if (args.UpperExists == True) {
		args.UpperNum = Qddb_GetNum(args.UpperStr);
	    }
	    args.LowerStr = NULL;
	    args.UpperStr = NULL;
	    pruneattr = argv[3];
	} else {
	    if (ParseRange(interp, argc, argv, &args) != TCL_OK)
		return TCL_ERROR;
	    if (args.LowerExists == True)
		args.LowerNum = Qddb_GetNum(args.LowerStr);
	    if (args.UpperExists == True)
		args.UpperNum = Qddb_GetNum(args.UpperStr);
	    args.LowerStr = NULL;
	    args.UpperStr = NULL;
	}
	args.Type = NUMERICRANGE;
    } else if ((argc >= 6 && 
	       (strcmp(argv[4], "date_range") == 0 || strcmp(argv[4], "dr") == 0)) ||
	       (argc >= 4 && 
	       (strcmp(argv[2], "date_range") == 0 || strcmp(argv[2], "dr") == 0))) {
	/* qddb_search schema ?-prunebyattr <attr>? dr <datelow> - <datehi> */
	if (strcmp(argv[2], "-prunebyattr") == 0) {
	    if (ParseRange(interp, argc-2, argv+2, &args) != TCL_OK)
		return TCL_ERROR;
	    if (args.LowerExists == True) {
		args.LowerStr = Qddb_DateStringToTime(args.LowerStr);
		args.LowerNum = Qddb_GetNum(args.LowerStr);
		Free(args.LowerStr);
	    }
	    if (args.UpperExists == True) {
		args.UpperStr = Qddb_DateStringToTime(args.UpperStr);
		args.UpperNum = Qddb_GetNum(args.UpperStr);
		Free(args.UpperStr);
	    }
	    args.LowerStr = NULL;
	    args.UpperStr = NULL;
	    pruneattr = argv[3];
	} else {
	    if (ParseRange(interp, argc, argv, &args) != TCL_OK)
		return TCL_ERROR;
	    if (args.LowerExists == True) {
		args.LowerStr = Qddb_DateStringToTime(args.LowerStr);
		args.LowerNum = Qddb_GetNum(args.LowerStr);
		Free(args.LowerStr);
	    }
	    if (args.UpperExists == True) {
		args.UpperStr = Qddb_DateStringToTime(args.UpperStr);
		args.UpperNum = Qddb_GetNum(args.UpperStr);
		Free(args.UpperStr);
	    }
	    args.LowerStr = NULL;
	    args.UpperStr = NULL;
	}
	args.Type = NUMERICRANGE;
    } else if (argc == 3 || argc == 5) {
	/* qddb_search schema ?-prunebyattr <attr>? <key> */
	/* default to word */
	if (argc == 5) {
	    if (strcmp(argv[argc-3], "-prunebyattr") != 0) {
		Tcl_AppendResult(interp, argv[0], ": bad option \"", argv[argc-3], "\"", NULL);
		return TCL_ERROR;
	    }
	    args.Alpha = argv[4];
	    pruneattr = argv[3];
	} else {
	    args.Alpha = argv[2];
	}
	args.Type = ALPHA;
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[argc-2], "\", \
should be w, n, r, dr, nr, or wr\n", NULL);
	return TCL_ERROR;
    }
    if (pruneattr != NULL) {
	if (strcmp(pruneattr, "any") == 0) {
	    pruneattrnum = -1;
	} else {
	    pruneattrnum = Qddb_ConvertAttributeNameToNumber(schema, pruneattr);
	    if (pruneattrnum == -1) {
		Tcl_AppendResult(interp, argv[0], ": bad attribute name \"", pruneattr, "\"", NULL);
		return TCL_ERROR;
	    }
	}
    } else {
	pruneattrnum = -1;
    }
    result = Qddb_Search(schema, &args, &n, pruneattrnum);
    if (result == NULL && qddb_errno != 0) {
	Tcl_AppendResult(interp, Qddb_PrintErrno(qddb_errno), ": ", qddb_errmsg, NULL);
	return TCL_ERROR;
    }
    if (TclQddb_NewKeyList(interp, argv[1], result, pruneattrnum, 0) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

