
/* TclQddb_Schema.c - TCL interface routines for Qddb schemas.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static Tcl_HashTable 	TclQddb_SchemaHashTable;
static int		TclQddb_SchemaHashTableInit = 0;
static unsigned int	TclQddb_SchemaNextNumber = 0;

static int TclQddb_SchemaOpen _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_SchemaOption _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *));
static int TclQddb_SchemaPrint _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_SchemaPath _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_SchemaLeaves _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_SchemaClose _ANSI_ARGS_((Tcl_Interp *, char *));


/* TclQddb_SchemaProc -- Open/close Qddb schemas.
 *
 * <schema_desc> <- qddb_schema open Relation
 * <schema list> <- qddb_schema print <schema_desc>
 * <leaf list>   <- qddb_schema leaves <schema_desc> ?<attr name>?
 * <schema path> <- qddb_schema path <schema_desc>
 * qddb_schema option type|verbosename|alias|isexpandable|format <schema_desc> <attr name>
 * qddb_schema excludewords <schema_desc>
 * qddb_schema secondarycaching <schema_desc> on|off
 * qddb_schema datestyle <schema_desc> us|european
 * qddb_schema close <schema_desc>|all
 */
int TclQddb_SchemaProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 3 || argc > 5) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_SchemaHashTableInit == 0) {
	TclQddb_SchemaHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_SchemaHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'o':
	if (strcmp(argv[1], "option") == 0) {
	    if (argc != 5) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_SchemaOption(interp, argv[2], argv[3], argv[4]) != TCL_OK)
		return TCL_ERROR;
	} else if (TclQddb_SchemaOpen(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'c':
	if (TclQddb_SchemaClose(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'l':
	if (strcmp(argv[1], "leaves") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	if (argc == 3) {
	    if (TclQddb_SchemaLeaves(interp, argv[2], NULL) != TCL_OK)
		return TCL_ERROR;
	} else if (argc == 4) {
	    if (TclQddb_SchemaLeaves(interp, argv[2], argv[3]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'p':
	if (strcmp(argv[1], "path") == 0) {
	    if (TclQddb_SchemaPath(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    if (TclQddb_SchemaPrint(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 's':
	if (strcmp(argv[1], "secondarycaching") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	} else {
	    Schema            *schema;
	    int               option;

	    if ((schema = TclQddb_GetSchema(argv[2])) == NULL) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], 
				 ": bad schema descriptor \"", argv[2], "\"", NULL);
		return TCL_ERROR;
	    }
	    if (Tcl_GetBoolean(interp, argv[3], &option) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": option must be on|off", NULL);
		return TCL_ERROR;
	    }
	    if (option) {
		schema->UseCachedSecondarySearch = True;
	    } else {
		schema->UseCachedSecondarySearch = False;
	    }
	}
	break;
    case 'd':
	if (strcmp(argv[1], "datestyle") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	} else {
	    Schema            *schema;
	    int               i;

	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (strcmp(argv[3], "us") != 0 && strcmp(argv[3], "european") != 0) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": option must be us|european", NULL);
		return TCL_ERROR;
	    }
	    if ((schema = TclQddb_GetSchema(argv[2])) == NULL) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], 
				 ": bad schema descriptor \"", argv[2], "\"", NULL);
		return TCL_ERROR;
	    }
	    if (strcmp(schema->default_date_format, "%m/%d/%y") == 0 ||
		strcmp(schema->default_date_format, "%d/%m/%y") == 0) {
		if (strcmp(argv[3], "us") == 0) {
		    Free(schema->default_date_format);
		    schema->default_date_format = Malloc(16);
		    strcpy(schema->default_date_format, "%m/%d/%y");
		    for (i = 1; i <= schema->NumberOfAttributes; i++) {
			if (strcmp(schema->Entries[i].Format, "%d/%m/%y") == 0) {
			    Free(schema->Entries[i].Format);
			    schema->Entries[i].Format = Malloc(16);
			    strcpy(schema->Entries[i].Format, "%m/%d/%y");
			}
		    }
		} else {
		    Free(schema->default_date_format);
		    schema->default_date_format = Malloc(16);
		    strcpy(schema->default_date_format, "%d/%m/%y");
		    for (i = 1; i <= schema->NumberOfAttributes; i++) {
			if (strcmp(schema->Entries[i].Format, "%m/%d/%y") == 0) {
			    Free(schema->Entries[i].Format);
			    schema->Entries[i].Format = Malloc(16);
			    strcpy(schema->Entries[i].Format, "%d/%m/%y");
			}
		    }
		}
	    }
	}
	break;
    case 'e': {
	Schema                  *schema;
	Entry                   ExcludeEntry;
	size_t                  exclude_num;
	char                    *retval;

	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": wrong # args", NULL);
	    return TCL_ERROR;
	}
	if (strcmp(argv[1], "excludewords") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	if ((schema = TclQddb_GetSchema(argv[2])) == NULL) {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], 
			     ": bad schema descriptor \"", argv[2], "\"", NULL);
	    return TCL_ERROR;
	}
	if ((ExcludeEntry = Qddb_ReadExcludeWords(schema, &exclude_num)) == NULL) {
	    Tcl_SetResult(interp, "", TCL_STATIC);
	    return TCL_OK;
	}
	if (exclude_num > 0) {
	    retval = Tcl_Merge((int)exclude_num, ExcludeEntry);
	} else {
	    retval = Calloc(1);
	}
	if (ExcludeEntry != NULL)
	    Qddb_Free(QDDB_TYPE_ENTRY, ExcludeEntry);
	Tcl_SetResult(interp, retval, TCL_DYNAMIC);
	break;
    }
    default:
	Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

Schema *TclQddb_GetSchema(Token)
    char			*Token;
{
    Schema			*retval;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_SchemaHashTableInit == 0) {
	TclQddb_SchemaHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_SchemaHashTable, TCL_STRING_KEYS);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_SchemaHashTable, Token);
    if (hash_ptr == NULL)
	return NULL;
    retval = (Schema *)Tcl_GetHashValue(hash_ptr);
    return retval;
}

static int TclQddb_SchemaOption(interp, option, schema_name, attr_name)
    Tcl_Interp			*interp;
    char			*option, *schema_name, *attr_name;
{
    Schema			*schema;
    int				attr_num;
    char			*retval;

    schema = TclQddb_GetSchema(schema_name);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "qddb_schema option: cannot find schema \"", schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr_name)) == -1) {
	Tcl_AppendResult(interp, "qddb_schema option: bad attribute name \"", attr_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (strcmp(option, "isexpandable") == 0) {
	if (schema->Entries[attr_num].AsteriskPresent == True)
	    retval = "yes";
	else
	    retval = "no";
    } else if (strcmp(option, "verbosename") == 0) {
	retval = schema->Entries[attr_num].VerboseName;
    } else if (strcmp(option, "alias") == 0) {
	retval = schema->Entries[attr_num].Alias;
    } else if (strcmp(option, "format") == 0) {
	retval = schema->Entries[attr_num].Format;
    } else if (strcmp(option, "type") == 0) {
	if (schema->Entries[attr_num].IsLeaf == False) {
	    Tcl_AppendResult(interp, "qddb_schema option: \"", attr_name, "\" must be a leaf", NULL);
	    return TCL_ERROR;
	}
	switch (schema->Entries[attr_num].Type) {
	case Integer:
	    retval = "integer";
	    break;
	case Real:
	    retval = "real";
	    break;
	case Date:
	    retval = "date";
	    break;
	case String:
	case Typeless:
	default:
	    retval = "string";
	    break;
	}
    } else if (strcmp(option, "separators") == 0) {
	retval = schema->Entries[attr_num].Separators;
    } else if (strcmp(option, "exclude") == 0) {
	if (schema->Entries[attr_num].Exclude == True) {
	    retval = "1";
	} else {
	    retval = "0";
	}
    } else {
	Tcl_AppendResult(interp, "qddb_schema option: bad option \"", option, "\"", NULL);
	return TCL_ERROR;	
    }
    Tcl_SetResult(interp, retval, TCL_VOLATILE);
    return TCL_OK;
}

static int TclQddb_SchemaOpen(interp, Relation)
    Tcl_Interp			*interp;
    char			*Relation;
{
    char			*RelationFN = Qddb_FindRelation(Relation);
    char			token[BUFSIZ];
    Schema			*schema;
    Tcl_HashEntry		*hash_ptr;
    int				newPtr;

    if (RelationFN == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", Relation, "\", set QDDBDIRS", NULL);
	return TCL_ERROR;
    }
    sprintf(token, "qddb_schema%d", TclQddb_SchemaNextNumber++);
    schema = Qddb_InitSchema(RelationFN);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "Error while reading Schema:\n",  qddb_errmsg, NULL);
	return TCL_ERROR;
    }
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_SchemaHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)schema);
    Tcl_ResetResult(interp);
    Tcl_SetResult(interp, token, TCL_VOLATILE);
    return TCL_OK;
}

static char *TclQddb_TraverseSchemaTree(interp, tree)
    Tcl_Interp			*interp;
    SchemaTreeNode		**tree;
{
    char			*argv[5], **nargv = NULL, *buf = NULL;
    int				nargc = 0;

    while (*tree != NULL) {
	argv[0] = (*tree)->schemanode->Name;
	argv[2] = (*tree)->schemanode->VerboseName;
	argv[3] = (*tree)->schemanode->AsteriskPresent == True? "yes":"no";
	argv[4] = NULL;
	argv[1] = Tcl_Merge(2, argv+2);
	nargv = (char **)Realloc(nargv, sizeof(char *)*(nargc+2));
	if ((*tree)->children == NULL || *((*tree)->children) == NULL) {
	    argv[2] = "";
	    nargv[nargc++] = Tcl_Merge(3, argv);
	    if (argv[1] != NULL)
		Free(argv[1]);
	} else {
	    if ((buf = TclQddb_TraverseSchemaTree(interp, (*tree)->children)) == NULL) {
		if (argv[1] != NULL)
		    Free(argv[1]);
		return NULL;
	    }
	    argv[2] = buf;
	    nargv[nargc++] = Tcl_Merge(3, argv);
	    if (argv[1] != NULL)
		Free(argv[1]);
	    if (buf != NULL)
		Free(buf);
	}
	tree++;
    }
    nargv[nargc] = NULL;
    buf = Tcl_Merge(nargc, nargv);
    while (--nargc >= 0)
	Free(nargv[nargc]);
    if (nargv != NULL)
	Free(nargv);
    return buf;
}

static int TclQddb_SchemaPrint(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Schema			*schema;
    char			*buf;

    schema = TclQddb_GetSchema(token);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "qddb_schema print: cannot find schema \"", token, "\"", NULL);
	return TCL_ERROR;
    }
    if ((buf = TclQddb_TraverseSchemaTree(interp, schema->Tree)) == NULL) {
	Tcl_AppendResult(interp, "qddb_schema print: cannot print schema \"", 
			 token, "\" (Qddb error)", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, buf, TCL_DYNAMIC);
    return TCL_OK;
}

static int TclQddb_SchemaPath(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Schema			*schema;
    char			*buf;

    schema = TclQddb_GetSchema(token);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "qddb_schema path: cannot find schema \"", token, "\"", NULL);
	return TCL_ERROR;
    }
    buf = Malloc(strlen(schema->RelationName)+1);
    strcpy(buf, schema->RelationName);
    Tcl_SetResult(interp, buf, TCL_DYNAMIC);
    return TCL_OK;
}


static int TclQddb_SchemaLeaves(interp, schema_token, attribute)
    Tcl_Interp			*interp;
    char			*schema_token, *attribute;
{
    Schema			*schema;
    int				attr_num, level;

    schema = TclQddb_GetSchema(schema_token);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "qddb_schema print: cannot find schema \"", schema_token, "\"", NULL);
	return TCL_ERROR;
    }
    if (attribute == NULL) {
	for (attr_num = 1; attr_num <= schema->NumberOfAttributes; attr_num++) {
	    if (schema->Entries[attr_num].IsLeaf == True) {
		char		*attr_str;

		attr_str = Qddb_ConvertAttributeNumberToName(schema, attr_num);
		if (attr_str == NULL) {
		    Tcl_ResetResult(interp);
		    Tcl_AppendResult(interp, "cannot find attribute number", NULL);
		    return TCL_ERROR;
		}
		Tcl_AppendResult(interp, " ", attr_str, NULL);
		Free(attr_str);
	    }
	}
	return TCL_OK;
    }
    attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute);
    if (attr_num == -1) {
	Tcl_AppendResult(interp, "qddb_schema: cannot find attribute \"", attribute,
			 "\" in schema \"", schema_token, "\"", NULL);
	return TCL_ERROR;
    }
    if (schema->Entries[attr_num].IsLeaf == True) {
	Tcl_SetResult(interp, attribute, TCL_VOLATILE);
	return TCL_OK;
    }
    level = schema->Entries[attr_num].Level;
    for (attr_num++; attr_num <= schema->NumberOfAttributes && 
	 schema->Entries[attr_num].Level > level; attr_num++) {
	if (schema->Entries[attr_num].IsLeaf == True) {
	    char		*attr_str;
	    
	    attr_str = Qddb_ConvertAttributeNumberToName(schema, attr_num);
	    if (attr_str == NULL) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "cannot find attribute number", NULL);
		return TCL_ERROR;
	    }
	    Tcl_AppendResult(interp, " ", attr_str, NULL);
	    Free(attr_str);
	}
    }
    return TCL_OK;
}


static int TclQddb_SchemaCloseOne(interp, Token)
    Tcl_Interp			*interp;
    char			*Token;
{
    Schema			*schema;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_DeleteTuple(interp, Token) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot delete tuples associated with schema \"", 
			 Token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_DeleteKeyList(interp, Token, True) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot delete keylists associated with schema \"", 
			 Token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_SchemaHashTable, Token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", Token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    schema = (Schema *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    Close(schema->database_fd);
    Close(schema->hashtable_fd);
    Qddb_Free(QDDB_TYPE_SCHEMA, schema);
    return TCL_OK;
}

static int TclQddb_SchemaClose(interp, Token)
    Tcl_Interp			*interp;
    char			*Token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;

    if (TclQddb_SchemaHashTableInit == 0)
	return TCL_OK;
    if (strcmp(Token, "all") == 0) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_SchemaHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    hash_key = Tcl_GetHashKey(&TclQddb_SchemaHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_SchemaClose: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_SchemaCloseOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	TclQddb_SchemaHashTableInit = 0;
	Tcl_DeleteHashTable(&TclQddb_SchemaHashTable);
    } else if (TclQddb_SchemaCloseOne(interp, Token) != TCL_OK)
	    return TCL_ERROR;
    return TCL_OK;
}


void TclQddb_DeleteSchemaProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteSchemaProc\n");
#endif
    (void)TclQddb_SchemaClose((Tcl_Interp *)clientData, "all");
}
