
/* TclQddb_Rows.c - TCL interface routines for Qddb KeyLists.
 *
 * Copyright (C) 1994, 1995, 1996 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"
#include "tclQddb.h"
#define DEBUG 1
/* EXPORTED:
 *	TclQddb_RowsProc
 */

static Tcl_HashTable 	TclQddb_RowsHashTable;
static int		TclQddb_RowsHashTableInit = 0;
static unsigned int	TclQddb_RowsNextNumber = 0;

static int TclQddb_SelectRows _ANSI_ARGS_((Tcl_Interp *, TclQddb_KeyListHeader *, int, char **));
static int TclQddb_RowsTable _ANSI_ARGS_((Tcl_Interp *, TclQddb_KeyListHeader *, int, char **));
static int TclQddb_TranslateRows _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_AllRows _ANSI_ARGS_((Tcl_Interp *, int, char **));
static void TclQddb_FreeRow _ANSI_ARGS_((TclQddb_Rows *));
static TclQddb_Rows **TclQddb_BuildAllRows _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, \
							char **, int, char *, char *));
static TclQddb_Rows *TclQddb_RowCopy _ANSI_ARGS_((TclQddb_Rows *, TclQddb_Rows **));
static TclQddb_Rows *TclQddb_RowConcat _ANSI_ARGS_((TclQddb_Rows *, TclQddb_Rows *));
static TclQddb_Rows **TclQddb_RowCartesianProduct _ANSI_ARGS_((TclQddb_Rows **, TclQddb_Rows **));
static TclQddb_Rows **TclQddb_RowUnion _ANSI_ARGS_((TclQddb_Rows **, TclQddb_Rows **));
static TclQddb_Rows **TclQddb_RowInst _ANSI_ARGS_((Schema *, DataTree **, char **, int, char *, char *));
static TclQddb_Rows **TclQddb_RowAttr _ANSI_ARGS_((Schema *, DataTree *, char **, int, char *, char *));
static char *TclQddb_AddRow _ANSI_ARGS_((Tcl_Interp *, char *, char *, TclQddb_Rows *));
static Qddb_Table *TclQddb_BuildTableFromRows _ANSI_ARGS_((Tcl_Interp *, TclQddb_Tuple *, \
							   char *, TclQddb_Rows **, \
							   char *, int, char **, size_t *, int));
/*static char *TclQddb_PrintRow _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, TclQddb_Rows *, \
					   char *, int, char **, size_t *));*/

static int TclQddb_GetVal _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_DoSortRows _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_InterestingAttr _ANSI_ARGS_((char *, char **, int));
static int TclQddb_InterestingAttrsBelow _ANSI_ARGS_((char *, char **, int));

static void TclQddb_GenerateAttrList _ANSI_ARGS_((Schema *, Qddb_AttrList *, int *, char ***));

static Qddb_Table *TclQddb_GenerateTable _ANSI_ARGS_((Tcl_Interp *, char *, Schema *, KeyList *, \
						      int, char **, int, char **, int *, int, int));
static int TclQddb_BuildTable _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_KeyListHeader *, \
					   int, char **, int, char **, int *, int, int, char **));
static int TclQddb_TableToQuery _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, char *, Boolean, Boolean, Boolean));
static int TclQddb_TableToRows _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, char *, Boolean, Boolean));
static char *TclQddb_PrintRowVals _ANSI_ARGS_((Tcl_Interp *, TclQddb_RowHeader *));
void TclQddb_SetTableNodeFromDataTree _ANSI_ARGS_((Qddb_Table *, DataTree *, size_t, size_t));

/* TclQddb_RowsProc -- manipulate rows.
 *
 * <row_desc list>              <- qddb_rows all ?-format rows|table?
 * 					?-instance <inst>? ?<options>? 
 *                                      ?-sortby <attr list>? ?-ascending <? <tuple desc>
 *   -format rows (default)
 *	{ <row_desc> "row value" }
 *	{ <row_desc> "row value" }
 *	...
 *   -format table
 *      <table_desc>
 * <row_desc list> <- qddb_rows select ?-format query|rows|table? ?<options>? <keylist desc>
 *   -format query
 *	{ { row value } { Start Length Number Type } { attr,inst attr,inst ... } }
 *	{ { row value } { Start Length Number Type } { attr,inst attr,inst ... } }
 *	...
 *   -format rows (default)
 *	{ <table_desc> <row_desc> "row value" }
 *	{ <table_desc> <row_desc> "row value" }
 *	...
 *   -format table
 *      <table_desc>
 * NOTE: -norows is ignored unless the format is 'query'.
 *
 * { attr,inst attr,inst }      <- qddb_rows get <row desc>
 * <values>                     <- qddb_rows getval <attr_list> <row_desc>
 * { <row_desc> <row_desc> }    <- qddb_rows sort ?-ascending on|off? <attr_list> <row_desc list>
 *                                 qddb_rows delete <row_desc>|all
 *
 * inst -- <attr>,<inst> pair
 * <options>:
 *	-flush on|off (default: off)		- flush tuples from memory as they are read
 *	-rowdescs on|off (default: on)		- return row descriptors for later use (table format)
 *	-deldup_rows on|off (default: off)	- delete duplicate rows
 *	-attrs <attribute list>			- attributes used to generate keylist
 *	-print <print list>			- attributes printed as 'row value'
 *	   { attr:width attr:width ... }
 */
int TclQddb_RowsProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    TclQddb_KeyListHeader	*keylist_header = NULL;
    TclQddb_RowHeader		*rows;
    
    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'a':
	if (strcmp(argv[1], "all") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc < 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for all", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_AllRows(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 's':
	if (strcmp(argv[1], "select") != 0) {
	    if (strcmp(argv[1], "sort") != 0) {
		Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
		return TCL_ERROR;
	    }
	    if (argc < 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for sort", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DoSortRows(interp, argc, argv) != TCL_OK)
		return TCL_ERROR;
	} else {
	    if (argc < 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for select", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_GetKeyList(interp, argv[argc-1], &keylist_header) != TCL_OK) {
		Tcl_AppendResult(interp, argv[0], ": bad keylist \"", argv[argc-1], "\"", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_SelectRows(interp, keylist_header, argc, argv) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'g':
	if (strcmp(argv[1], "get") != 0) {
	    if (strcmp(argv[1], "getval") != 0) {
		Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
		return TCL_ERROR;
	    }
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for getval", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_GetVal(interp, argc, argv) != TCL_OK) 
		return TCL_ERROR;
	} else {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for get", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_TranslateRows(interp, argv[2]) != TCL_OK)
		return TCL_ERROR;
	}
	break;
    case 'd':
	if (strcmp(argv[1], "delete") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args for delete", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_DeleteRows(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 't':
	if (strcmp(argv[1], "tuplename") == 0) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args for tuplename", NULL);
		return TCL_ERROR;
	    }
	    if ((rows = TclQddb_GetRows(interp, argv[2])) == NULL)
		return TCL_ERROR;
	    Tcl_SetResult(interp, rows->tuple_name, TCL_VOLATILE);
	} else {
	    Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	    return TCL_ERROR;
	}
	break;
    default:
	Tcl_AppendResult(interp, argv[0], ": invalid command ", argv[1], NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_AllRows(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    TclQddb_Tuple		*tuple;
    Schema			*schema;
    TclQddb_Rows		**row_array = NULL;
    Qddb_Table			*table = NULL;
    int				deldup_rows = 0;
    int				attr_argc = 0, print_argc = 0, sort_argc = 0, ascending_argc = 0, i, j;
    int				print_tcl_elements = 0, suppress_printing = 0, use_rowdescs = 0;
    size_t			*print_widths = NULL;
    int				getint, error_occurred = 0, retval;
    char			**attr_argv = NULL, **print_argv = NULL, **sort_argv = NULL;
    char			**ascending_argv = NULL;
    char			*sepstring = " ", *table_name = NULL, *format = "rows";
    char			*check_attr = NULL, *check_inst = NULL;

    if ((tuple = TclQddb_GetTuple(interp, argv[argc-1])) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad tuple \"", argv[argc-1], "\"", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", tuple->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 2; i < argc-2; i++) {
	if (strcmp(argv[i], "-attrs") == 0) {
	    if (attr_argc != 0) {
		Free(attr_argv);
		attr_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &attr_argc, &attr_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-rowdescs") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &use_rowdescs) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-format") == 0) {
	    format = argv[++i];
	    if (strcmp(format, "rows") != 0 && strcmp(format, "table") != 0) {
		Tcl_AppendResult(interp, argv[0], " ", argv[1], ": invalid format \"", argv[i], "\"", NULL);
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-sortby") == 0) {
	    if (sort_argc != 0) {
		Free(sort_argv);
		sort_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &sort_argc, &sort_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-ascending") == 0) {
	    if (ascending_argc != 0) {
		Free(ascending_argv);
		ascending_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &ascending_argc, &ascending_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-print") == 0) {
	    if (print_argc != 0) {
		Free(print_argv);
		print_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &print_argc, &print_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	    print_widths = (size_t *)Malloc(sizeof(size_t)*print_argc);
	    for (j = 0; j < print_argc; j++) {
		char		*c_buf, *n_buf;

		c_buf = strtok(print_argv[j], ":");
		n_buf = strtok(NULL, ":");
		if (n_buf == NULL) {
		    print_widths[j] = 0;
		} else if (Tcl_GetInt(interp, n_buf, &getint) != TCL_OK) {
		    error_occurred = 1;
		    break;
		} else
		    print_widths[j] = getint;
	    }
	    if (error_occurred) {
		break;
	    }
	} else if (strcmp(argv[i], "-instance") == 0) {
	    check_attr = strtok(argv[++i], ",");
	    if (check_attr == NULL || check_attr[0] == '\0' || 
		Qddb_ConvertAttributeNameToNumber(schema, check_attr) == -1) {
		Tcl_AppendResult(interp, "bad instance \"", argv[i], "\"", NULL);
		error_occurred = 1;
		break;
	    }
	    check_inst = strtok(NULL, ",");
	    if (check_inst == NULL)
		check_inst = "";
	} else if (strcmp(argv[i], "-sep") == 0) {
	    sepstring = argv[++i];
	} else 	if (strcmp(argv[i], "-deldup_rows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &deldup_rows) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], "\": must be -attrs or -print", NULL);
	    error_occurred = 1;
	    break;
	}
    }
    if (strcmp(format, "rows") == 0) {
	use_rowdescs = 1;
    }
    if (error_occurred == 0 && 
	(row_array = TclQddb_BuildAllRows(interp, schema, tuple, attr_argv, attr_argc,
					  check_attr, check_inst)) == NULL) {
	error_occurred = 1;
    }
    /* Now we have a complete set of existing rows starting at (if specified)
     * check_attr,check_inst for this tuple.   We have to prune the list of 
     * rows based upon the attrs specified in attr_argv by deleting duplicates.
     * Generally, if you want a unique list of rows, then you specify
     * the same attrs in attr_argv as in print_argv.   If you want all rows, then
     * you specify nothing in attr_argv.   You CAN specify different values for
     * attr_argv & print_argv if you want to prune duplicate rows based on
     * attributes different from those being printed.
     */
    if (error_occurred == 0) {
	if ((table=TclQddb_BuildTableFromRows(interp, tuple, argv[argc-1], 
				       row_array, sepstring,
				       print_argc, print_argv, print_widths, use_rowdescs)) == NULL) {
	    error_occurred = 1;
	    Free(row_array);
	} else if (table->rows < 1) {
	    table = NULL;
	    table_name = Calloc(QDDB_MIN_MALLOC_SIZE);
	} else if ((table_name = TclQddb_NewTable(interp, table)) == NULL) {
	    error_occurred = 1;
	    Free(row_array);
	} else if (sort_argc > 0) {
	    if (TclQddb_TableSort(interp, table, NULL, NULL, ascending_argv, ascending_argc,
				  sort_argv, sort_argc) != TCL_OK) {
		(void)TclQddb_TableDelete(interp, table_name);
		error_occurred = 1;
		Free(row_array);
	    }
	}
    }
    if (error_occurred == 0) {
	if (strcmp(format, "rows") == 0) {
	    if (TclQddb_TableToRows(interp, table, sepstring, (Boolean)print_tcl_elements, 
				    (Boolean)suppress_printing) != TCL_OK) {
		(void)TclQddb_TableDelete(interp, table_name);
		retval = TCL_ERROR;
	    } else {
		retval = TCL_OK;
	    }
	} else { /* table */
	    Tcl_SetResult(interp, table_name, TCL_DYNAMIC);
	    table_name = NULL;
	    retval = TCL_OK;
	}
	Free(row_array);
    } else {
	retval = TCL_ERROR;
    }
    if (table_name != NULL) {
	(void)TclQddb_TableDelete(interp, table_name);
	Free(table_name);
    }
    if (print_argv != NULL)
	Free(print_argv);
    if (print_widths != NULL)
	Free(print_widths);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return retval;
}

/* TclQddb_BuildTableFromRows --
 *	Takes the rows, prunes them via argv, then keeps the individual rows
 * for later use.   The rows (rows[i]) should not be used again.
 */
static Qddb_Table *TclQddb_BuildTableFromRows(interp, tuple, tuple_name, rows, 
					      sep, argc, argv, widths, use_rowdescs)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    char			*tuple_name;
    TclQddb_Rows		**rows;
    char			*sep;
    int				argc;
    char			**argv;
    size_t			*widths;
    int				use_rowdescs;
{
    Schema			*schema;
    DataTree			*dtnode;
    Qddb_Table			*table;
    Qddb_TableNode		*table_node;
    TclQddb_Rows		*ptr;
    Qddb_TableRowCol		rowcol;
    char			*tmpch, *tmpch2;
    size_t			num_rows = 0;
    int				*argv_nums;
    int				i;

    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", tuple->schema_name, "\"", NULL);
	return NULL;
    }
    if (argc > 0)
	argv_nums = (int *)Malloc(sizeof(size_t)*argc);
    else
	argv_nums = NULL;
    for (i = 0; i < argc; i++) {
	if ((argv_nums[i] = Qddb_ConvertAttributeNameToNumber(schema, argv[i])) == -1) {
	    Free(argv_nums);
	    Tcl_AppendResult(interp, "bad attribute: \"", argv[i], "\"", NULL);
	    return NULL;
	}
    }
    rowcol.row = 0;
    rowcol.column = argc;
    table = Qddb_TableOp(NULL, QDDB_TABLE_OP_DEFINE, &rowcol);
    if (table == NULL) {
	if (argc > 0)
	    Free(argv_nums);
	Tcl_AppendResult(interp, "cannot create new table", NULL);
	return NULL;
    }
    /* Need to set up the column types according to the print arguments
     */
    table_node = table->table[0] + 1;
    for (i = 0; i < argc; i++, table_node++) {
	switch (schema->Entries[argv_nums[i]].Type) {
	case Real:
	    table_node->data.type = QDDB_TABLE_TYPE_REAL;
	    table_node->data.data.real = 0.0;
	    break;
	case Integer:
	    table_node->data.type = QDDB_TABLE_TYPE_INTEGER;
	    table_node->data.data.integer = 0;
	    break;
	case Date:
	    table_node->data.type = QDDB_TABLE_TYPE_DATE;
	    table_node->data.data.string = NULL;
	    break;
	case Typeless:
	case String:
	default:
	    table_node->data.type = QDDB_TABLE_TYPE_STRING;
	    table_node->data.data.string = NULL;
	    break;
	}
	table_node->info = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
	tmpch = schema->Entries[argv_nums[i]].VerboseName;
	if (tmpch == NULL || tmpch[0] == '\0') {
	    tmpch = argv[i];
	    tmpch2 = rindex(tmpch, (int)'.');
	} else {
	    tmpch2 = NULL;
	}
	table_node->info->title = Malloc(strlen(tmpch)+1);
	if (tmpch2 == NULL)
	    tmpch2 = tmpch;
	else
	    tmpch2++;
	strcpy(table_node->info->title, tmpch2);
	table_node->info->name = (char *)Malloc(strlen(argv[i])+1);
	strcpy(table_node->info->name, argv[i]);
	table_node->info->print = (Qddb_TableHeaderFormat *)Calloc(sizeof(Qddb_TableHeaderFormat));
	table_node->info->print->format =
	    (char *)Malloc(strlen(schema->Entries[argv_nums[i]].Format)+1);
	strcpy(table_node->info->print->format, schema->Entries[argv_nums[i]].Format);
    }
    if (argc > 0)
	Free(argv_nums);
    while (*rows != NULL) {
	num_rows = table->rows + 1;
	table = Qddb_TableOp(table, QDDB_TABLE_OP_INSERTROW, &num_rows);
	for (i = 0; i < argc; i++) {
	    ptr = *rows;
	    while (ptr != NULL) {
		if (strcmp(argv[i], ptr->attr_name) == 0) {
		    dtnode = TclQddb_DataTreeRef(tuple, ptr);
		    TclQddb_SetTableNodeFromDataTree(table, dtnode, (size_t)num_rows, (size_t)(i+1));
		}
		ptr = ptr->next;
	    }
	}
	if (table->table[num_rows]->info == NULL)
	    table->table[num_rows]->info = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
	if (use_rowdescs) {
	    table->table[num_rows]->info->comment = 
		TclQddb_AddRow(interp, tuple->schema_name, tuple_name, *rows);
	}
	rows++;
    }
    return table;
}

static TclQddb_Rows *TclQddb_RowCopy(row, lastnode)
    TclQddb_Rows		*row, **lastnode;
{
    TclQddb_Rows		*retval = NULL, *lastretval = NULL;

    while (row != NULL) {
	if (retval == NULL) {
	    retval = lastretval = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    lastretval->next = NULL;
	} else {
	    lastretval->next = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    lastretval = lastretval->next;
	    lastretval->next = NULL;
	}
	lastretval->attr_name = Malloc(strlen(row->attr_name)+1);
	strcpy(lastretval->attr_name, row->attr_name);
	lastretval->instance = Malloc(strlen(row->instance)+1);
	strcpy(lastretval->instance, row->instance);
	row = row->next;
    }
    if (lastnode != NULL)
	*lastnode = lastretval;
    return retval;
}

static TclQddb_Rows *TclQddb_RowConcat(row1, row2)
    TclQddb_Rows		*row1, *row2;
{
    TclQddb_Rows		*lastnode;

    row1 = TclQddb_RowCopy(row1, &lastnode);
    row2 = TclQddb_RowCopy(row2, NULL);
    lastnode->next = row2;
    return row1;
}

static TclQddb_Rows **TclQddb_RowCartesianProduct(row1, row2)
    TclQddb_Rows		**row1, **row2;
{
    TclQddb_Rows		**ptr1, **ptr2, **retval, **place;
    int				row1_len, row2_len;

    if (row1 == NULL && row2 == NULL) {
	return NULL;
    }
    if (row1 == NULL) {
	for (ptr2 = row2; *ptr2 != NULL; ptr2++) {
	    TclQddb_FreeRow(*ptr2);
	}
	Free(row2);
	return NULL;
    }
    if (row2 == NULL) {
	for (ptr1 = row1; *ptr1 != NULL; ptr1++) {
	    TclQddb_FreeRow(*ptr1);
	}
	Free(row1);
	return NULL;
    }
    for (ptr1 = row1, row1_len = 0; *ptr1 != NULL; ptr1++, row1_len++);
    for (ptr2 = row2, row2_len = 0; *ptr2 != NULL; ptr2++, row2_len++);
    retval = (TclQddb_Rows **)Malloc(sizeof(TclQddb_Rows *)*(row1_len*row2_len+1));
    place = retval;
    for (ptr1 = row1; *ptr1 != NULL;  ptr1++) {
	for (ptr2 = row2; *ptr2 != NULL; ptr2++) {
	    *place++ = TclQddb_RowConcat(*ptr1, *ptr2);
	}
	TclQddb_FreeRow(*ptr1);
    }
    *place = NULL;
    for (ptr2 = row2; *ptr2 != NULL; ptr2++)
	TclQddb_FreeRow(*ptr2);
    Free(row1);
    Free(row2);
    return retval;
}

/* TclQddb_RowUnion --
 * 	Take the union of two rows.
 */
static TclQddb_Rows **TclQddb_RowUnion(row1, row2)
    TclQddb_Rows		**row1, **row2;
{
    TclQddb_Rows		**ptr, **retval, **place;
    int				row1_len, row2_len;

    if (row1 == NULL)
	return row2;
    if (row2 == NULL)
	return row1;
    for (ptr = row1, row1_len = 0; *ptr != NULL; ptr++, row1_len++);
    for (ptr = row2, row2_len = 0; *ptr != NULL; ptr++, row2_len++);
    retval = (TclQddb_Rows **)Malloc(sizeof(TclQddb_Rows *)*(row1_len+row2_len+1));
    place = retval;
    for (ptr = row1; *ptr != NULL; ptr++)
	*place++ = *ptr;
    for (ptr = row2; *ptr != NULL; ptr++)
	*place++ = *ptr;
    *place = NULL;
    Free(row1);
    Free(row2);
    return retval;
}

static TclQddb_Rows **TclQddb_RowInst(schema, attrs, attr_argv, attr_argc, name, inst_name)
    Schema			*schema;
    DataTree			**attrs;
    char			**attr_argv;
    int				attr_argc;
    char			*name, *inst_name;
{
    TclQddb_Rows		**retval = NULL;
    int				i;

    for (i = 0; attrs[i] != NULL; i++) {
	if (retval == NULL) {
	    retval = TclQddb_RowAttr(schema, attrs[i], attr_argv, attr_argc, name, inst_name);
	    if (retval == NULL)
		break;
	} else {
	    retval = TclQddb_RowCartesianProduct(retval, 
						 TclQddb_RowAttr(schema, attrs[i], attr_argv, attr_argc,
								 name, inst_name));
	    if (retval == NULL)
		break;
	}
    }
    return retval;
}

static int TclQddb_InterestingAttr(name, attr_argv, attr_argc)
    char			*name, **attr_argv;
    int				attr_argc;
{
    int				i;

    if (attr_argc == 0)
	return True; /* All nodes are interesting */
    for (i = 0; i < attr_argc; i++)
	if (strcmp(name, attr_argv[i]) == 0) 
	    return True;
    return False;
}

static int TclQddb_InterestingAttrsBelow(name, attr_argv, attr_argc)
    char			*name, **attr_argv;
    int				attr_argc;
{
    int				i;

    if (attr_argc == 0)
	return True; /* All nodes are interesting */
    for (i = 0; i < attr_argc; i++)
	if (strncmp(name, attr_argv[i], strlen(name)) == 0) 
	    return True;
    return False;
}

static TclQddb_Rows **TclQddb_RowAttr(schema, insts, attr_argv, attr_argc, name, inst_name)
    Schema			*schema;
    DataTree			*insts;
    char 			**attr_argv;
    int				attr_argc;
    char			*name, *inst_name;
    
{
    TclQddb_Rows		**retval = NULL;
    int				i, retval_top;
    int				interesting_attr, interesting_attrs_below;
    size_t			namelen, instlen;
    char			*thisname, *thisinstname, *comp;

    /* Compose this level's attribute name */
    comp = insts->datatree_schema->schemanode->Name;
    if (name[0] == '\0') { /* top level */
	thisname = alloca(strlen(comp)+1);
	strcpy(thisname, comp);
    } else {
	namelen = strlen(name);
	thisname = alloca(namelen+strlen(comp)+2);
	strcpy(thisname, name);
	thisname[namelen] = '.';
	strcpy(thisname+namelen+1, comp);
    }
    instlen = strlen(inst_name);
    thisinstname = alloca(instlen+MAXINTEGERLEN+2);
    strcpy(thisinstname, inst_name);
    if (instlen > 0) {
	thisinstname[instlen++] = '.';
    }
    interesting_attr = TclQddb_InterestingAttr(thisname, attr_argv, attr_argc);
    interesting_attrs_below = TclQddb_InterestingAttrsBelow(thisname, attr_argv, attr_argc);
    retval_top = 0;
    for (i = 0; insts[i].datatree_type != DATATREE_END; i++) {
	sprintf(thisinstname+instlen, "%d", i+1);
	if (insts[i].datatree_type != DATATREE_CHILDREN) {
	    /* This node is a leaf node; if we are interested in this
	     * attribute, then walk through all instances.  Otherwise,
	     * just return the first non-false one.
	     */
	    if (insts[i].marked == False) {
		continue;
	    }
	    retval = (TclQddb_Rows **)Realloc(retval, sizeof(TclQddb_Rows *)*(retval_top+2));
	    retval[retval_top] = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    retval[retval_top]->attr_name = Malloc(strlen(thisname)+1);
	    strcpy(retval[retval_top]->attr_name, thisname);
	    retval[retval_top]->instance = Malloc(strlen(thisinstname)+1);
	    strcpy(retval[retval_top]->instance, thisinstname);
	    retval[retval_top]->next = NULL;
	    retval[++retval_top] = NULL;
	    if (!interesting_attr)
		break;
	} else {
	    if (retval == NULL) {
		retval = TclQddb_RowInst(schema, insts[i].datatree_children, attr_argv, attr_argc,
					 thisname, thisinstname);
	    } else {
		retval = TclQddb_RowUnion(retval, TclQddb_RowInst(schema, insts[i].datatree_children, 
								  attr_argv, attr_argc,
								  thisname, thisinstname));
	    }
	    if (!interesting_attrs_below && retval != NULL)
		break;
	}
    }
    return retval;
}

static void *TclQddb_FindSpot(tree, attr, inst, do_attr)
    DataTree			**tree;
    char			*attr, *inst;
    int				*do_attr;
{
    DataTree			**dt;
    char			**attr_arr;
    size_t			inst_arr[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    size_t			num_inst, cur;
    int				i;

    if (attr == NULL || attr[0] == '\0') {
	*do_attr = 0;
	return (void *)tree;
    }
    attr_arr = TclQddb_SplitAttributeName(attr);
    num_inst = StringToInstance(inst, inst_arr);
    dt = tree;
    for (i = 0; attr_arr[i] != NULL; i++) {
	cur = TclQddb_FindAttribute(dt, attr_arr[i]);
	if (i >= num_inst) {
	    *do_attr = 1;
	    TclQddb_FreeArgs(-1, attr_arr);
	    return (void *)dt[cur];
	}
#if defined(DIAGNOSTIC)
	if (dt[cur][inst_arr[i]-1].datatree_type != DATATREE_CHILDREN) {
	    fprintf(stderr, "bad attr,inst specified (%s,%s)\n", attr, inst);
	    fflush(stderr);
	    abort();
	}
#endif
	dt = dt[cur][inst_arr[i]-1].datatree_children;
    }
    TclQddb_FreeArgs(-1, attr_arr);
    *do_attr = 0;
    return (void *)dt;
}

/* TclQddb_BuildAllRows -- 
 *	Build an array of (TclQddb_Rows *) consisting of all interesting
 * rows based upon check_attr,check_inst.   If check_attr,check_inst is
 * non-existant, then just return a list of ALL rows in the tuple matching
 * the attributes in attr_argv.
 */
static TclQddb_Rows **TclQddb_BuildAllRows(interp, schema, tuple, attr_argv, attr_argc, 
					   check_attr, check_inst)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    char			**attr_argv;
    int				attr_argc;
    char			*check_attr, *check_inst;
{
    TclQddb_Rows		**rows = NULL;
    void			*dt;
    int				do_attr;
    char			*lastdot;

    if (check_attr != (char *)NULL) {
	dt = TclQddb_FindSpot(tuple->datatree, check_attr, check_inst, &do_attr);
	if (do_attr == 1) {
	    lastdot = rindex(check_attr, '.');
	    if (lastdot == (char *)NULL)
		check_attr = "";
	    else
		*lastdot = '\0';
	    rows = TclQddb_RowAttr(schema, (DataTree *)dt, attr_argv, attr_argc, check_attr, check_inst);
	    if (lastdot != (char *)NULL)
		*lastdot = '.';
	} else {
	    rows = TclQddb_RowInst(schema, (DataTree **)dt, attr_argv, attr_argc, check_attr, check_inst);
	}
    } else {
	rows = TclQddb_RowInst(schema, tuple->datatree, attr_argv, attr_argc, "", "");
    }
    return rows;
}

/* TclQddb_SelectRows -- Build a list of <row_desc,entry_desc> pairs
 * describing the rows from the given keylist.
 */
static int TclQddb_SelectRows(interp, keylist_header, argc, argv)
    Tcl_Interp			*interp;
    TclQddb_KeyListHeader	*keylist_header;
    int				argc;
    char			*argv[];
{
    Schema			*schema;
    Qddb_Table			*table;
    int				query_mode = 0, deldup_rows = 0;
    int				attr_argc = 0, print_argc = 0, sort_argc = 0, ascending_argc = 0;
    int				i, j;
    int				*print_nums = NULL;
    int				getint, error_occurred = 0, retval;
    char			**attr_argv = NULL, **print_argv = NULL;
    char			**sort_argv = NULL, **ascending_argv = NULL;
    char			*sepstring = " ", *table_name = NULL, *format = NULL;
    int				norows = 0;
    int				suppress_printing = 0, print_tcl_elements = 0;
    int				flush = 0;

    for (i = 2; i <= argc-2; i++) {
	if (strcmp(argv[i], "-format") == 0) {
	    if (strcmp(argv[i+1], "table") == 0) {
		if (TclQddb_RowsTable(interp, keylist_header, argc, argv) != TCL_OK)
		    return TCL_ERROR;
		return TCL_OK;
	    }
	}
    }
    if ((schema = TclQddb_GetSchema(keylist_header->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 2; i <= argc-2; i++) {
	if (strcmp(argv[i], "-attrs") == 0) {
	    if (attr_argc != 0) {
		Free(attr_argv);
		attr_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &attr_argc, &attr_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-format") == 0) {
	    format = argv[++i];
	    if (strcmp(format, "query") == 0) {
		query_mode = 1;
	    } else if (strcmp(format, "rows") == 0) {
		query_mode = 0;
	    } else {
		Tcl_AppendResult(interp, "bad format \"", format, "\": must be query, rows, or table", NULL);
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-suppress") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &suppress_printing) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-tclelements") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &print_tcl_elements) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-print") == 0) {
	    if (print_argc != 0) {
		Free(print_argv);
		print_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &print_argc, &print_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	    print_nums = (int *)Malloc(sizeof(size_t)*print_argc);
	    for (j = 0; j < print_argc; j++) {
		char		*c_buf, *n_buf;

		c_buf = strtok(print_argv[j], ":");
		n_buf = strtok(NULL, ":");
		if (n_buf == NULL) {
		    print_nums[j] = 0;
		} else if (Tcl_GetInt(interp, n_buf, &getint) != TCL_OK) {
		    error_occurred = 1;
		    break;
		} else
		    print_nums[j] = getint;
	    }
	    if (error_occurred) {
		break;
	    }
	}  else if (strcmp(argv[i], "-sortby") == 0) {
	    if (sort_argc != 0) {
		Free(sort_argv);
		sort_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &sort_argc, &sort_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-ascending") == 0) {
	    if (ascending_argc != 0) {
		Free(ascending_argv);
		ascending_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &ascending_argc, &ascending_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-sep") == 0) {
	    sepstring = argv[++i];
	} else 	if (strcmp(argv[i], "-deldup_rows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &deldup_rows) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-query") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &query_mode) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-norows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &norows) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-suppress") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &suppress_printing) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-tclelements") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &print_tcl_elements) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-flush") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &flush) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], "\": must be -query, -attrs or -print", NULL);
	    error_occurred = 1;
	    break;
	}
    }
    if (error_occurred == 0) {
	if (query_mode == 0 && norows != 0) {
	    Tcl_AppendResult(interp, "-norows is only valid if -query is 'on'", NULL);
	    error_occurred = 1;
	} else if (TclQddb_BuildTable(interp, schema, keylist_header, 
				      attr_argc, attr_argv, print_argc, print_argv, print_nums, 
				      True, flush, &table_name) != TCL_OK) {
	    error_occurred = 1;
	} else if (table_name == NULL) {
	    Tcl_SetResult(interp, "", TCL_STATIC);
	} else if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
	    error_occurred = 1;
	} else if (TclQddb_TableSort(interp, table, NULL, NULL, ascending_argv, ascending_argc,
				     sort_argv, sort_argc) != TCL_OK) {
	    (void)TclQddb_TableDelete(interp, table_name);
	    error_occurred = 1;
	} else if (query_mode == 1) {
	    if (TclQddb_TableToQuery(interp, table, sepstring, (Boolean)print_tcl_elements,
				     (Boolean)suppress_printing, (Boolean)norows) != TCL_OK) {
		(void)TclQddb_TableDelete(interp, table_name);
		error_occurred = 1;
	    }
	} else if (TclQddb_TableToRows(interp, table, sepstring, (Boolean)print_tcl_elements, 
				       (Boolean)suppress_printing) != TCL_OK) {
	    (void)TclQddb_TableDelete(interp, table_name);
	    error_occurred = 1;
	}
    }
    if (error_occurred == 0) {
	(void)TclQddb_TableDelete(interp, table_name);
	retval = TCL_OK;
    } else {
	retval = TCL_ERROR;
    }
    if (table_name != NULL)
	Free(table_name);
    if (print_argv != NULL)
	Free(print_argv);
    if (print_nums != NULL)
	Free(print_nums);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return TCL_OK;
}

static int TclQddb_TableToQuery(interp, table, sepstring, print_tcl_elements, suppress_printing, norows)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    char			*sepstring;
    Boolean			print_tcl_elements, suppress_printing, norows;
{
    Qddb_TableNode		*row0;
    TclQddb_Tuple		*tuple;
    size_t			rows = table->rows, cols = table->columns, i, j;
    char			**tmprowval, *rowval[4], **colval, *ch;
    char			**formatted_table = NULL;
    char			*tuple_desc, *row_desc;
    char			keybuf[MAXINTEGERLEN*5];
    int				num;

    rowval[3] = NULL;
    colval = (char **)Malloc(sizeof(char *)*(cols+1));
    colval[cols] = NULL;
    row0 = table->table[0];
    if (!suppress_printing && !print_tcl_elements) {
	formatted_table = TclQddb_GetFormattedTable(interp, table, sepstring, 0, 0);
    }
    for (i = 1; i <= rows; i++) {
#if defined(DIAGNOSTIC)
	if (table->table[i][0].info == NULL || table->table[i][0].info->comment == NULL)
	    PANIC("TclQddb_TableToQuery: empty comment field in row after processing");
#endif
	if (Tcl_SplitList(interp, table->table[i][0].info->comment, &num, &tmprowval) != TCL_OK) {
	    Free(colval);
	    return TCL_ERROR;
	}
	tuple_desc = tmprowval[0];
	row_desc = tmprowval[1];
	if (!suppress_printing) {
	    if (print_tcl_elements) {
		for (j = 1; j <= cols; j++) {
		    if (row0[j].info != NULL)
			ch = TclQddb_GetCellValue(table->table[i]+j, row0[j].data.type, row0[j].info->print);
		    else
			ch = TclQddb_GetCellValue(table->table[i]+j, row0[j].data.type, NULL);
		    colval[j-1] = (char *)Malloc(strlen(ch)+1);
		    strcpy(colval[j-1], ch);
		}
		rowval[0] = Tcl_Merge((int)cols, colval);
		for (j = 1; j <= cols; j++) {
		    Free(colval[j-1]);
		}
	    } else {
		rowval[0] = formatted_table[i];
	    }
	}
	if ((tuple = TclQddb_GetTuple(interp, tuple_desc)) == NULL) {
	    Tcl_AppendResult(interp, "TclQddb_TableToQuery: cannot find tuple \"", tuple_desc, "\"", NULL);
	    if (!suppress_printing)
		Free(rowval[0]);
	    Free(tmprowval);
	    Free(colval);
	    return TCL_ERROR;
	}
	sprintf(keybuf, "%d %d %d %d", (int)tuple->keylist.Start, (int)tuple->keylist.Length, 
		(int)tuple->keylist.Number, (int)tuple->keylist.Type);
	rowval[1] = keybuf;
	if (norows) {
	    rowval[2] = Calloc(QDDB_MIN_MALLOC_SIZE);
	} else {
	    TclQddb_RowHeader 	*rowptr;

	    if ((rowptr = TclQddb_GetRows(interp, row_desc)) == NULL) {
		Tcl_AppendResult(interp, "TclQddb_TableToQuery: cannot find row \"", row_desc, "\"", NULL);
		if (!suppress_printing)
		    Free(rowval[0]);
		Free(tmprowval);
		Free(colval);
		return TCL_ERROR;
	    }
	    rowval[2] = TclQddb_PrintRowVals(interp, rowptr);
	}
	ch = Tcl_Merge(3, rowval);
	Free(tmprowval);
	Free(rowval[2]);
	if (!suppress_printing) {
	    if (!print_tcl_elements) {
		Free(formatted_table[i]);
		formatted_table[i] = NULL;
	    } else {
		Free(rowval[0]);
	    }
	}
	Tcl_AppendElement(interp, ch);
	Free(ch);
    }
    if (formatted_table != NULL)
	Free(formatted_table);
    Free(colval);
    return TCL_OK;
}

static char *TclQddb_PrintRowVals(interp, rowptr)
    Tcl_Interp			*interp;
    TclQddb_RowHeader		*rowptr;
{
    TclQddb_Rows		*r;

    Qddb_InitBuffer();
    for (r = rowptr->row; r != NULL; r = r->next) {
	if (r->next == NULL) {
	    Qddb_ConcatBuffer(r->attr_name);
	    Qddb_ConcatBuffer(",");
	    Qddb_ConcatBuffer(r->instance);
	} else {
	    Qddb_ConcatBuffer(r->attr_name);
	    Qddb_ConcatBuffer(",");
	    Qddb_ConcatBuffer(r->instance);
	    Qddb_ConcatBuffer(" ");
	}
    }
    return Qddb_GetBuffer();
}

static int TclQddb_TableToRows(interp, table, sepstring, print_tcl_elements, suppress_printing)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    char			*sepstring;
    Boolean			print_tcl_elements, suppress_printing;
{
    Qddb_TableNode		*row0;
    size_t			rows = table->rows, cols = table->columns, i, j;
    char			**tmprowval, **formatted_table = NULL, *rowval[4], **colval, *ch;
    int				num;

    rowval[3] = NULL;
    colval = (char **)Malloc(sizeof(char *)*(cols+1));
    colval[cols] = NULL;
    row0 = table->table[0];
    if (!suppress_printing && !print_tcl_elements) {
	formatted_table = TclQddb_GetFormattedTable(interp, table, sepstring, 0, 0);
    }
    for (i = 1; i <= rows; i++) {
#if defined(DIAGNOSTIC)
	if (table->table[i][0].info == NULL || table->table[i][0].info->comment == NULL)
	    PANIC("TclQddb_TableToRows: empty comment field in row after processing");
#endif
	if (Tcl_SplitList(interp, table->table[i][0].info->comment, &num, &tmprowval) != TCL_OK) {
	    Free(colval);
	    return TCL_ERROR;
	}
	if (num > 0) {
	    rowval[0] = tmprowval[0];
	    if (num > 1) {
		rowval[1] = tmprowval[1];
	    }
#if defined(DIAGNOSTIC)
	    if (num > 2) {
		PANIC("TclQddb_TableToRows: too many Tcl elements in table comment field");
	    }
#endif
	}
	if (!suppress_printing) {
	    if (print_tcl_elements) {
		for (j = 1; j <= cols; j++) {
		    if (row0[j].info != NULL)
			ch = TclQddb_GetCellValue(table->table[i]+j, row0[j].data.type, row0[j].info->print);
		    else
			ch = TclQddb_GetCellValue(table->table[i]+j, row0[j].data.type, NULL);
		    colval[j-1] = (char *)Malloc(strlen(ch)+1);
		    strcpy(colval[j-1], ch);
		}
		rowval[num] = Tcl_Merge((int)cols, colval);
		for (j = 1; j <= cols; j++) {
		    Free(colval[j-1]);
		}
	    } else {
		rowval[num] = formatted_table[i];
	    }
	    ch = Tcl_Merge(num+1, rowval);
	} else {
	    ch = Tcl_Merge(num, rowval);
	}
	Free(tmprowval);
	if (!suppress_printing)
	    Free(rowval[num]);
	Tcl_AppendElement(interp, ch);
	Free(ch);
    }
    if (!suppress_printing && !print_tcl_elements) {
	Free(formatted_table);
    }
    Free(colval);
    return TCL_OK;
}

TclQddb_RowHeader *TclQddb_GetRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*rows;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_RowsHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find row \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    rows = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
    return rows;
}

static int TclQddb_TranslateRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*rows;
    TclQddb_Rows		*r;
    char			*buf;

    rows = TclQddb_GetRows(interp, token);
    for (r = rows->row; r != NULL; r = r->next) {
	Qddb_InitBuffer();
	if (r->next == NULL) {
	    Qddb_ConcatBuffer(r->attr_name);
	    Qddb_ConcatBuffer(",");
	    Qddb_ConcatBuffer(r->instance);
	} else {
	    Qddb_ConcatBuffer(r->attr_name);
	    Qddb_ConcatBuffer(",");
	    Qddb_ConcatBuffer(r->instance);
	    Qddb_ConcatBuffer(" ");
	}
	buf = Qddb_GetBuffer();
	Tcl_AppendResult(interp, buf, NULL);
	Free(buf);
    }
    return TCL_OK;
}


int TclQddb_SetRow(interp, row_header)
    Tcl_Interp			*interp;
    TclQddb_RowHeader		*row_header;
{
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ];
    int				newPtr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_row%d", TclQddb_RowsNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_RowsHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)row_header);
    Tcl_SetResult(interp, token, TCL_VOLATILE);
    return TCL_OK;
}


static char *TclQddb_AddRow(interp, schema_desc, tuple_desc, row)
    Tcl_Interp			*interp;
    char			*schema_desc, *tuple_desc;
    TclQddb_Rows		*row;
{
    TclQddb_RowHeader		*row_header;
    Tcl_HashEntry		*hash_ptr;
    char			token[BUFSIZ], *retval;
    int				newPtr;

    if (TclQddb_RowsHashTableInit == 0) {
	TclQddb_RowsHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_RowsHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_row%d", TclQddb_RowsNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_RowsHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    row_header = (TclQddb_RowHeader *)Malloc(sizeof(TclQddb_RowHeader));
    row_header->schema_name = Malloc(strlen(schema_desc)+1);
    strcpy(row_header->schema_name, schema_desc);
    row_header->tuple_name = Malloc(strlen(tuple_desc)+1);
    strcpy(row_header->tuple_name, tuple_desc);
    row_header->row = row;
    Tcl_SetHashValue(hash_ptr, (ClientData)row_header);
    retval = Malloc(strlen(token)+1);
    strcpy(retval, token);
    return retval;
}

static void TclQddb_FreeRow(row)
    TclQddb_Rows		*row;
{
    TclQddb_Rows		*old_row;

    while (row != NULL) {
	old_row = row;
	row = row->next;
	Free(old_row->attr_name);
	Free(old_row->instance);
	Free(old_row);
    }
}

static int TclQddb_DeleteRowsOne(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_RowHeader		*row_header;
    Tcl_HashEntry		*hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_RowsHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find row \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    row_header = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    TclQddb_FreeRow(row_header->row);
    Free(row_header->schema_name);
    Free(row_header->tuple_name);
    Free(row_header);
    return TCL_OK;
}

int TclQddb_DeleteRows(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;
    int				deletebytuple = 0;

    if (TclQddb_RowsHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_tuple", token, 10) == 0)
	deletebytuple = 1;
    if (strcmp(token, "all") == 0 || deletebytuple == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_RowsHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    if (deletebytuple == 1) {
		TclQddb_RowHeader *rows;

		rows = (TclQddb_RowHeader *)Tcl_GetHashValue(hash_ptr);
		if (strcmp(rows->tuple_name, token) != 0) {
		    hash_ptr = Tcl_NextHashEntry(&hash_search);
		    continue;
		}		
	    }
	    hash_key = Tcl_GetHashKey(&TclQddb_RowsHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_DeleteRows: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DeleteRowsOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	if (deletebytuple == 0) {
	    TclQddb_RowsHashTableInit = 0;
	    Tcl_DeleteHashTable(&TclQddb_RowsHashTable);
	}
    } else if (TclQddb_DeleteRowsOne(interp, token) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

void TclQddb_DeleteRowProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteRowProc\n");
#endif
    (void)TclQddb_DeleteRows((Tcl_Interp *)clientData, "all");
}


static int TclQddb_GetVal(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    TclQddb_RowHeader		*row_header;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*row;
    DataTree			*node;
    int				attr_argc;
    char			**attr_argv = NULL, *element;
    int				i;

    row_header = TclQddb_GetRows(interp, argv[3]);
    if (row_header == NULL) {
	Tcl_AppendResult(interp, "Cannot find row for row descriptor \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    }
    if ((tuple = TclQddb_GetTuple(interp, row_header->tuple_name)) == NULL) {
	Tcl_AppendResult(interp, "Cannot find tuple \"", row_header->tuple_name,
			 "\" for row descriptor \"", argv[3], "\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_SplitList(interp, argv[2], &attr_argc, &attr_argv) != TCL_OK)
	return TCL_ERROR;
    for (i = 0; i < attr_argc; i++) {
	row = row_header->row;
	while (row != NULL && strcmp(attr_argv[i], row->attr_name) != 0)
	    row = row->next;
	if (row == NULL) {
	    Tcl_AppendElement(interp, "");
	    continue;
	}
	if ((node = TclQddb_DataTreeLookupNode(interp, tuple, row->attr_name, row->instance)) == NULL) {
	    if (attr_argv != NULL)
		Free(attr_argv);
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "Cannot find part of datatree node \"", row->attr_name, ",",
			     row->instance, "\"", NULL);

	    return TCL_ERROR;
	}
	if ((element = (char *)Qddb_DataTreeProcess(NULL, NULL, (void *)node, 
						    QDDB_DATATREE_PROC_GETDATA, 0)) == NULL) {
	    if (attr_argv != NULL)
		Free(attr_argv);
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "Datatree node corrupted \"", row->attr_name, ",",
			     row->instance, "\"", NULL);

	    return TCL_ERROR;	    
	}
	Tcl_AppendElement(interp, element);
	Free(element);
    }
    return TCL_OK;
}

/* TclQddb_DoSortRows -- Front-end wrapper for TclQddb_SortRows.
 */
static int TclQddb_DoSortRows(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			**argv;
{
    int				attr_argc, row_argc, ascending_argc = 0;
    char			**attr_argv = NULL, **row_argv = NULL, **ascending_argv = NULL, *retval;
    char			*attrs_str = NULL, *rows_str = NULL;
    int				error_occurred = 0, intretval;

    /* qddb_rows sort ?-ascending <attr list>? <attr list> <row_desc list> */
    if (argc == 6) {
	if (Tcl_SplitList(interp, argv[3], &ascending_argc, &ascending_argv) != TCL_OK) {
	    error_occurred = 1;
	} else {
	    attrs_str = argv[4];
	    rows_str = argv[5];
	}
    } else if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args for sort", NULL);
	return TCL_ERROR;
    } else {
	attrs_str = argv[2];
	rows_str = argv[3];
    }
    if (error_occurred == 0) {
	if (Tcl_SplitList(interp, attrs_str, &attr_argc, &attr_argv) != TCL_OK) {
	    error_occurred = 1;
	} else if (Tcl_SplitList(interp, rows_str, &row_argc, &row_argv) != TCL_OK) {
	    error_occurred = 1;
	} else if (TclQddb_SortRows(interp, row_argc, row_argv, attr_argc, attr_argv, 
				    ascending_argc, ascending_argv) != TCL_OK) {
	    error_occurred = 1;
	}
    }
    if (error_occurred == 0) {
	/* package result */
	retval = Tcl_Merge(row_argc, row_argv);
	Tcl_SetResult(interp, retval, TCL_DYNAMIC);
	intretval = TCL_OK;
    } else {
	intretval = TCL_ERROR;
    }
    if (attr_argv != NULL)
	Free(attr_argv);
    if (row_argv != NULL)
	Free(row_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    return intretval;
}

DataTree *TclQddb_DataTreeLookupNode(interp, tuple, attr, inst)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    char			*attr, *inst;
{
    DataTree			**dt, *dinst;
    char			**attr_array;
    size_t			Instance[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    int				spot, level;
    int				i;

    dinst = NULL;
    attr_array = TclQddb_SplitAttributeName(attr);
    StringToInstance(inst, Instance);
    dt = tuple->datatree; 
    level = 0;
    while (dt != NULL) {
	spot = TclQddb_FindAttribute(dt, attr_array[level]);
	if (spot == -1) {
	    break;
	}
        for (i = 0; dt[spot][i].datatree_type != DATATREE_END; i++);
	if (Instance[level] > i) {
	    break;
	}
	dinst = dt[spot] + (Instance[level] - 1);
	if (dinst->datatree_type == DATATREE_CHILDREN) {
	    dt = dinst->datatree_children;
	} else if (level == dinst->datatree_schema->schemanode->Level-1) {
	    TclQddb_FreeArgs(-1, attr_array);
	    return dinst;
	} else {
	    break;
	}
	level++;
    }
    TclQddb_FreeArgs(-1, attr_array);
    Tcl_AppendResult(interp, "bad instance number \"", inst, "\"", NULL);
    return NULL;
}

/* BEGIN TABLE */

static int TclQddb_RowsTable(interp, keyheader, argc, argv)
    Tcl_Interp			*interp;
    TclQddb_KeyListHeader	*keyheader;
    int				argc;
    char			*argv[];
{
    Schema			*schema;
    Qddb_Table			*table;
    int				deldup_rows = 0;
    int				attr_argc = 0, print_argc = 0, sort_argc = 0, ascending_argc = 0;
    int				i, j;
    size_t			*print_nums = NULL;
    int				getint;
    char			**attr_argv = NULL, **print_argv = NULL;
    char			**sort_argv = NULL, **ascending_argv = NULL;
    char			*table_name = NULL;
    int				flush = 0, use_rowdescs = 0;
    int				error_occurred = 0;

    if ((schema = TclQddb_GetSchema(keyheader->schema_name)) == NULL) {
	Tcl_AppendResult(interp, argv[0], ": bad schema \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 2; i <= argc-2; i++) {
	if (strcmp(argv[i], "-attrs") == 0) {
	    if (attr_argc != 0) {
		Free(attr_argv);
		attr_argv = NULL;
		attr_argc = 0;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &attr_argc, &attr_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-format") == 0) { /* skip table format, we already know that */
	    i++;
	} else if (strcmp(argv[i], "-print") == 0) {
	    if (print_argc != 0) {
		Free(print_argv);
		print_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &print_argc, &print_argv) != TCL_OK) {
		error_occurred = 1;
	    }
	    print_nums = (size_t *)Malloc(sizeof(size_t)*print_argc);
	    for (j = 0; j < print_argc; j++) {
		char		*c_buf, *n_buf;

		c_buf = strtok(print_argv[j], ":");
		n_buf = strtok(NULL, ":");
		if (n_buf == NULL) {
		    print_nums[j] = 0;
		} else if (Tcl_GetInt(interp, n_buf, &getint) != TCL_OK) {
		    error_occurred = 1;
		    break;
		} else {
		    print_nums[j] = getint;
		}
	    }
	}  else if (strcmp(argv[i], "-sortby") == 0) {
	    if (sort_argc != 0) {
		Free(sort_argv);
		sort_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &sort_argc, &sort_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-ascending") == 0) {
	    if (ascending_argc != 0) {
		Free(ascending_argv);
		ascending_argv = NULL;
	    }
	    if (Tcl_SplitList(interp, argv[++i], &ascending_argc, &ascending_argv) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else 	if (strcmp(argv[i], "-deldup_rows") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &deldup_rows) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-flush") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &flush) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else if (strcmp(argv[i], "-rowdescs") == 0) {
	    if (Tcl_GetBoolean(interp, argv[++i], &use_rowdescs) != TCL_OK) {
		error_occurred = 1;
		break;
	    }
	} else {
	    Tcl_AppendResult(interp, argv[0], " ", argv[1], ":bad option \"", argv[i], "\"", NULL);
	    error_occurred = 1;
	    break;
	}
    }
#if defined(DEBUG)
    if (error_occurred != 0)
	fprintf(stderr, "an error occurred in parsing\n");
#endif
    if (error_occurred == 0) {
	if (TclQddb_BuildTable(interp, schema, keyheader, attr_argc, attr_argv, print_argc, print_argv, 
			       NULL, use_rowdescs, flush, &table_name) != TCL_OK) {

#if defined(DEBUG)
    fprintf(stderr, "an error occurred in TclQddb_BuildTable\n");
#endif
	    error_occurred = 1;
	} else if (table_name == NULL) {
	    table_name = Calloc(QDDB_MIN_MALLOC_SIZE);
	} else if ((table = TclQddb_GetTable(interp, table_name)) == NULL) {
#if defined(DEBUG)
    fprintf(stderr, "an error occurred in TclQddb_GetTable\n");
#endif
	    error_occurred = 1;
	} else if (TclQddb_TableSort(interp, table, NULL, NULL, ascending_argv, ascending_argc,
			      sort_argv, sort_argc) != TCL_OK) {
	    (void)Qddb_TableOp(table, QDDB_TABLE_OP_REMOVE, NULL);
#if defined(DEBUG)
    fprintf(stderr, "an error occurred in TclQddb_TableSort\n");
#endif
	    error_occurred = 1;
	}
    }
    if (print_argv != NULL)
	Free(print_argv);
    if (print_nums != NULL)
	Free(print_nums);
    if (attr_argv != NULL)
	Free(attr_argv);
    if (sort_argv != NULL)
	Free(sort_argv);
    if (ascending_argv != NULL)
	Free(ascending_argv);
    if (error_occurred == 0) {
	Tcl_SetResult(interp, table_name, TCL_DYNAMIC);
	return TCL_OK;
    } else {
	return TCL_ERROR;
    }
}
    
static int TclQddb_BuildTable(interp, schema, keyheader, 
			      attr_argc, attr_argv, print_argc, print_argv, print_nums,
			      use_rowdescs, flush, table_name)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_KeyListHeader	*keyheader;
    int				attr_argc;
    char			*attr_argv[];
    int				print_argc;
    char			*print_argv[];
    int				print_nums[];
    int				use_rowdescs, flush;
    char			**table_name;
{
    Qddb_Table			*table, *table2;
    int                         i;

    if (keyheader->keylist == NULL) {
	*table_name = NULL;
	return TCL_OK;
    }
    if (attr_argc == 0 && keyheader->attributes.Number > 0) {
	table = NULL;
	for (i = 0; i < keyheader->attributes.Number; i++) {
	    TclQddb_GenerateAttrList(schema, keyheader->attributes.Nodes[i], &attr_argc, &attr_argv);
#if defined(REMOVE)
/* REMOVE */
printf("table# %d\n", i);
fflush(stdout);
#endif
	    table2 = TclQddb_GenerateTable(interp, keyheader->schema_name, schema, 
					   keyheader->keylist, attr_argc, attr_argv,
					   print_argc, print_argv, print_nums,
					   use_rowdescs, flush);
	    Qddb_Free(QDDB_TYPE_ENTRY, attr_argv);
	    if (table == NULL) {
		table = table2;
	    } else {
		table = Qddb_TableOp(table, QDDB_TABLE_OP_APPEND, (void *)table2);
		(void)Qddb_TableOp(table2, QDDB_TABLE_OP_REMOVE, NULL);
	    }
	}
	if (table == NULL) {
	    *table_name = NULL;
	    return TCL_OK;
	}
    } else if ((table=TclQddb_GenerateTable(interp, keyheader->schema_name, schema, 
				     keyheader->keylist, attr_argc, attr_argv,
				     print_argc, print_argv, print_nums,
				     use_rowdescs, flush)) == NULL) {
	*table_name = NULL;
	return TCL_OK;
    }
    if (table->rows < 1) {
	(void)Qddb_TableOp(table, QDDB_TABLE_OP_REMOVE, NULL);
	*table_name = NULL;
	return TCL_OK;
    }
    *table_name = TclQddb_NewTable(interp, table);
    if (*table_name == NULL)
	return TCL_ERROR;
    return TCL_OK;
}

static void TclQddb_GenerateAttrList(schema, attrlist, argc_ptr, argv_ptr)
    Schema                      *schema;
    Qddb_AttrList               *attrlist;
    int                         *argc_ptr;
    char                        ***argv_ptr;
{
    Qddb_AttrList               *ptr;
    int                         i;

    *argc_ptr = 0;
    ptr = attrlist;
    while (ptr != NULL) {
	(*argc_ptr)++;
	ptr = ptr->next;
    }
    *argv_ptr = (char **)Malloc(sizeof(char *) * ((*argc_ptr)+1));
    for (i = 0, ptr = attrlist; i < (*argc_ptr); i++, ptr = ptr->next) {
	(*argv_ptr)[i] = Qddb_ConvertAttributeNumberToName(schema, ptr->Attribute);
    }
    (*argv_ptr)[*argc_ptr] = NULL;
}

static Qddb_Table *TclQddb_GenerateTable(interp, schema_desc, schema, list, 
					 attr_argc, attr_argv, 
					 print_argc, print_argv, print_nums, 
					 use_rowdescs, flush)
    Tcl_Interp			*interp;
    char			*schema_desc;
    Schema			*schema;
    KeyList			*list;
    int				attr_argc;
    char			*attr_argv[];
    int				print_argc;
    char			*print_argv[];
    int				print_nums[];
    int				use_rowdescs, flush;
{
    Entry			this_entry;
    InCoreEntry			*core_entry = NULL;
    DataTree			**datatree, *node;
    TclQddb_Rows		**rows, *thisrow;
    Qddb_Table			*table;
    Qddb_TableNode		*table_node;
    Qddb_TableRowCol		rowcol;
    char			*relation_fn = schema->RelationName;
    int				*print_argv_nums = NULL;
    int				db_file;
    int				i, j;
    int				inapplicable = Inapplicable;
    TclQddb_Tuple		*tuple;
    char			*tuple_desc, *row_desc, *tmpch, *tmpch2;
    KeyList			**split_list, **orig_split_list, *tmplist, key, *keylist;
    Qddb_PruneArgs		prune_args;
    
    if (list == NULL)
	return NULL;
    /* Check the attributes at this level for validity */
    for (i = 0; i < attr_argc; i++) {
	if (Qddb_ConvertAttributeNameToNumber(schema, attr_argv[i]) == -1) {
	    Tcl_AppendResult(interp, "bad attribute: \"", attr_argv[i], "\"", NULL);
	    return NULL;
	}
    }
    if (print_argc > 0)
	print_argv_nums = (int *)Malloc(sizeof(size_t)*print_argc);
    for (i = 0; i < print_argc; i++) {
	if ((print_argv_nums[i] = Qddb_ConvertAttributeNameToNumber(schema, print_argv[i])) == -1) {
	    Free(print_argv_nums);
	    Tcl_AppendResult(interp, "bad attribute: \"", print_argv[i], "\"", NULL);
	    return NULL;
	}
    }
    /* Here we copy the keylist into a split list; each element of the
     * split list is deleted from MARKFROMINCORE   We need to make sure that
     * we use a copy because TclQddb probably has a token set up for this KeyList.
     */
    keylist = Qddb_KeyListProcess(schema, list, NULL, QDDB_KEYLIST_PROC_NULLOP, QDDB_KEYLIST_FLAG_COPY);
    if (attr_argc > 0) {
	prune_args.attr_len = attr_argc;
	prune_args.attrs = attr_argv;
	keylist = Qddb_KeyListProcess(schema, keylist, &prune_args, QDDB_KEYLIST_PROC_PRUNE, 
				      QDDB_KEYLIST_FLAG_PRUNEBYROW);
    }
    if (keylist == NULL) { /* This is ok? */
	Tcl_AppendResult(interp, "No list found after pruning by attribute ",
			 "(did you specify the same attributes you searched on?)", NULL);
	if (print_argc > 0)
	    Free(print_argv_nums);
	return NULL;
    }
    (void)Qddb_KeyListProcess(schema, keylist, &split_list, QDDB_KEYLIST_PROC_SPLIT, 0);
    orig_split_list = split_list;
    db_file = schema->database_fd;
    rowcol.row = 0;
    rowcol.column = print_argc;
    table = Qddb_TableOp(NULL, QDDB_TABLE_OP_DEFINE, &rowcol);
    if (table == NULL) {
	if (print_argc > 0)
	    Free(print_argv_nums);
	Tcl_AppendResult(interp, "cannot create new table", NULL);
	return NULL;
    }
    /* Need to set up the column types according to the print arguments
     */
    table_node = table->table[0] + 1;
    for (i = 0; i < print_argc; i++, table_node++) {
	switch (schema->Entries[print_argv_nums[i]].Type) {
	case Real:
	    table_node->data.type = QDDB_TABLE_TYPE_REAL;
	    table_node->data.data.real = 0.0;
	    break;
	case Integer:
	    table_node->data.type = QDDB_TABLE_TYPE_INTEGER;
	    table_node->data.data.integer = 0;
	    break;
	case Date:
	    table_node->data.type = QDDB_TABLE_TYPE_DATE;
	    table_node->data.data.string = NULL;
	    break;
	case Typeless:
	case String:
	default:
	    table_node->data.type = QDDB_TABLE_TYPE_STRING;
	    table_node->data.data.string = NULL;
	    break;
	}
	table_node->info = (Qddb_TableHeaderInfo *)Calloc(sizeof(Qddb_TableHeaderInfo));
	tmpch = schema->Entries[print_argv_nums[i]].VerboseName;
	if (tmpch == NULL || tmpch[0] == '\0') {
	    tmpch = print_argv[i];
	    tmpch2 = rindex(tmpch, (int)'.');
	} else {
	    tmpch2 = NULL;
	}
	table_node->info->title = Malloc(strlen(tmpch)+1);
	if (tmpch2 == NULL)
	    tmpch2 = tmpch;
	else
	    tmpch2++;
	strcpy(table_node->info->title, tmpch2);
	table_node->info->name = (char *)Malloc(strlen(print_argv[i])+1);
	strcpy(table_node->info->name, print_argv[i]);
	table_node->info->print = (Qddb_TableHeaderFormat *)Calloc(sizeof(Qddb_TableHeaderFormat));
	table_node->info->print->format =
	    (char *)Malloc(strlen(schema->Entries[print_argv_nums[i]].Format)+1);
	strcpy(table_node->info->print->format, schema->Entries[print_argv_nums[i]].Format);
	if (print_nums != NULL) {
	    table_node->info->print->width = print_nums[i];
	}
    }
    while (*split_list != NULL) {
	rows = NULL;
	tmplist = *split_list;
	this_entry = NULL;
	if (Qddb_ReadEntryByKeyList(db_file, relation_fn, &this_entry, tmplist, True) == -1) { 
	    /* invalid entry */
	    Qddb_Free(QDDB_TYPE_ENTRY, this_entry);
	    Qddb_Free(QDDB_TYPE_KEYLIST, *split_list);
	    split_list++;
	    continue;
	}
	Qddb_ReducedAttrToFullAttr(schema, this_entry);
	core_entry = (InCoreEntry *)Qddb_Convert(schema, QDDB_ENTRYTYPE_EXTERNAL, this_entry, 
						 QDDB_ENTRYTYPE_INTERNAL);
	Qddb_Free(QDDB_TYPE_ENTRY, this_entry);
	key = **split_list;
	key.Instance = NULL;
	key.next = NULL;
	(void)Qddb_KeyListProcess(schema, *split_list, core_entry, QDDB_KEYLIST_PROC_MARK, 
				  QDDB_KEYLIST_FLAG_MARKFROMINCORE);
	/* *split_list was deleted by Qddb_KeyListProcess */
	*split_list = NULL;
	datatree = (DataTree **)Qddb_Convert(schema, QDDB_ENTRYTYPE_INTERNAL, core_entry,
						 QDDB_ENTRYTYPE_DATATREE);
	Qddb_Free(QDDB_TYPE_INCOREENTRY, core_entry);
	tuple_desc = TclQddb_NewTuple(interp, schema_desc, datatree, &key);
	tuple = TclQddb_GetTuple(interp, tuple_desc);
#if defined(REMOVE)
/* REMOVE */
printf("tree# %d\n", datatree[0]->datatree_sequence_number);
fflush(stdout);
#endif
	for (i = 0; i < attr_argc; i++) {
	    /* Unfortunately, this is necessary to take care of NULL values.
	     */
	    (void)Qddb_DataTreeProcess(schema, tuple->datatree, attr_argv[i], QDDB_DATATREE_PROC_SETATTR, 0);
	}
	rows = TclQddb_BuildAllRows(interp, schema, tuple, print_argv, print_argc, NULL, NULL);
	if (rows == NULL) {
	    /* This can occur when doing unattributed searches, since the keylist was not pruned
	     * by row.
	     */
	    TclQddb_DeleteTuple(interp, tuple_desc); 
	} else {
	    size_t		num_rows = table->rows + 1;
	    char		buf[BUFSIZ];

	    (void)Qddb_DataTreeProcess(schema, tuple->datatree, 
				       &inapplicable, QDDB_DATATREE_PROC_SETMARK, 0);
	    for (i = 0; rows[i] != NULL; i++, num_rows++) {
#if defined(REMOVE)
/* REMOVE */
printf("\trow# %d\n", i);
#endif
		if ((row_desc = TclQddb_AddRow(interp, schema_desc, tuple_desc, rows[i])) == NULL) {
		    if (print_argc > 0)
			Free(print_argv_nums);
		    /* FIXME: free more memory */
		    return NULL;
		}
		table = Qddb_TableOp(table, QDDB_TABLE_OP_INSERTROW, &num_rows);
		if (use_rowdescs) {
		    sprintf(buf, "%s %s", tuple_desc, row_desc);
		    if (TclQddb_TableSetHeaderInfo(interp, &table->table[num_rows]->info, 
						   "-comment", buf, 0) != TCL_OK) {
			if (print_argc > 0)
			    Free(print_argv_nums);
			/* FIXME: free up stuff */
			Tcl_AppendResult(interp, "could not set new table's header information", NULL);
			return NULL;
		    }
		}
		Free(row_desc);
		for (j = 1; j <= print_argc; j++) {
		    thisrow = rows[i];
		    while (thisrow != NULL) {
			if (strcmp(thisrow->attr_name, print_argv[j-1]) == 0) {
			    break;
			}
			thisrow = thisrow->next;
		    }
#if defined(DIAGNOSTIC)
		    if (thisrow == NULL) {
			PANIC("TclQddb_GenerateTable: cannot find attribute in row!");
		    }
#endif
		    node = TclQddb_DataTreeLookupNode(interp, tuple, thisrow->attr_name, thisrow->instance);
#if defined(DIAGNOSTIC)
		    if (node == NULL) {
			PANIC("TclQddb_GenerateTable: cannot find data for a node");
		    }
#endif
		    TclQddb_SetTableNodeFromDataTree(table, node, (size_t)num_rows, (size_t)j);
		}
	    }
	    Free(rows);
	}
	if (flush) {
	    TclQddb_FlushTuple(interp, tuple_desc);
	}
	Free(tuple_desc);
	split_list++;
    }
    if (print_argc > 0)
	Free(print_argv_nums);
    Free(orig_split_list);
    return table;
}

void TclQddb_SetTableNodeFromDataTree(table, node, row, col)
    Qddb_Table			*table;
    DataTree			*node;
    size_t			row, col;
{
    int				thisnodetype;

    if (table->table[row][col].data.type == QDDB_TABLE_TYPE_DEFAULT) {
	thisnodetype = table->table[0][col].data.type;
    } else {
	thisnodetype = table->table[row][col].data.type;
    }
    switch (thisnodetype) {
    case QDDB_TABLE_TYPE_REAL:
#if defined(DIAGNOSTIC)
	if (node->datatree_type != DATATREE_REAL) {
	    fprintf(stderr, "TclQddb_SetTableNodeFromDataTree: type %d node\n", node->datatree_type);
	    PANIC("TclQddb_SetTableNodeFromDataTree: wrong node type, should be real");
	}
#endif
	if (node->modified == False) {
	    table->table[row][col].data.type = QDDB_TABLE_TYPE_STRING;
	    table->table[row][col].data.data.string = Calloc(QDDB_MIN_MALLOC_SIZE);
	} else {
	    table->table[row][col].data.data.real = node->datatree_real;
	}
	break;
    case QDDB_TABLE_TYPE_INTEGER:
#if defined(DIAGNOSTIC)
	if (node->datatree_type != DATATREE_INT) {
	    fprintf(stderr, "TclQddb_SetTableNodeFromDataTree: type %d node\n", node->datatree_type);
	    PANIC("TclQddb_SetTableNodeFromDataTree: wrong node type, should be integer");
	}
#endif
	if (node->modified == False) {
	    table->table[row][col].data.type = QDDB_TABLE_TYPE_STRING;
	    table->table[row][col].data.data.string = Calloc(QDDB_MIN_MALLOC_SIZE);
	} else {
	    table->table[row][col].data.data.integer = node->datatree_int;
	}
	break;
    case QDDB_TABLE_TYPE_DATE:
#if defined(DIAGNOSTIC)
	if (node->datatree_type != DATATREE_DATE) {
	    fprintf(stderr, "TclQddb_SetTableNodeFromDataTree: type %d node\n", node->datatree_type);
	    PANIC("TclQddb_SetTableNodeFromDataTree: wrong node type, should be date");
	}
#endif
	table->table[row][col].data.data.string = 
	    (char *)Malloc(strlen(node->datatree_date)+1);
	strcpy(table->table[row][col].data.data.string, node->datatree_date);
	break;
    case QDDB_TABLE_TYPE_STRING:
    default:
#if defined(DIAGNOSTIC)
	if (node->datatree_type != DATATREE_STRING) {
	    fprintf(stderr, "TclQddb_SetTableNodeFromDataTree: type %d node\n", node->datatree_type);
	    PANIC("TclQddb_SetTableNodeFromDataTree: wrong node type, should be string");
	}
#endif
	table->table[row][col].data.data.string =
	    (char *)Malloc(strlen(node->datatree_string)+1);
	strcpy(table->table[row][col].data.data.string, node->datatree_string);
	break;
    }
}
