
/* TclQddb_KeyList.c - TCL interface routines for Qddb schemas.
 *
 * Copyright (C) 1994, 1995 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static Tcl_HashTable 	TclQddb_KeyListHashTable;
static int		TclQddb_KeyListHashTableInit = 0;
static unsigned int	TclQddb_KeyListNextNumber = 0;

static int TclQddb_KeyListOp _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_KeyListProcess _ANSI_ARGS_((Tcl_Interp *, int, char **));
static int TclQddb_KeyListLength _ANSI_ARGS_((Tcl_Interp *, char *, int *));
static int TclQddb_KeyListRange _ANSI_ARGS_((Tcl_Interp *, int, char **));

static KeyList *KeyListCopyNode _ANSI_ARGS_((KeyList *));
static Qddb_AttrHead *KeyListCopyAttrHead _ANSI_ARGS_((Qddb_AttrHead *));
static Qddb_AttrList *KeyListCopyAttrList _ANSI_ARGS_((Qddb_AttrList *));
static Qddb_AttrHead *AttrListIntersection _ANSI_ARGS_((Qddb_AttrHead *, Qddb_AttrHead *));
static Qddb_AttrHead *AttrListUnion _ANSI_ARGS_((Qddb_AttrHead *, Qddb_AttrHead *));
static Qddb_AttrHead *AttrListExclusion _ANSI_ARGS_((Qddb_AttrHead *, Qddb_AttrHead *));
static int AttrListCompareNodes _ANSI_ARGS_((Qddb_AttrList *, Qddb_AttrList *));

/* TclQddb_KeyList -- Manipulate a Qddb KeyList.
 *
 * qddb_keylist op [intersection|union] ?<options>? <keylist_desc1> <keylist_desc2>
 * qddb_keylist op exclusion ?<options>? <inlist_desc> <notinlist_desc>
 * qddb_keylist process <type> ?<options>? <keylist_desc>
 *	<type>:
 *		prune
 *			-prunebyattr attr
 *			-prunebyrow entry
 *		sort
 *		split
 *		nullop
 *	<options>:
 *		-deldup_sameentry on|off
 *		-deldup_sameattr  on|off
 *		-copy on|off
 * qddb_keylist get <keylist_desc>
 * qddb_keylist length <keylist_desc>
 * qddb_keylist delete <keylist_desc>|all
 */

int TclQddb_KeyList(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    TclQddb_KeyListHeader	*keylist_header;
    KeyList			*list;
    char			**nargv;
    int				i, nargc, length;
    char			*merged_buf, buf[BUFSIZ];

    if (argc < 3) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_KeyListHashTableInit == 0) {
	TclQddb_KeyListHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_KeyListHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'o':	/* operation  */
	if (TclQddb_KeyListOp(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'p':	/* processing */
	if (TclQddb_KeyListProcess(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'g':	/* get */
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);	
	    return TCL_ERROR;
	}
	if (strcmp("get", argv[1]) != 0) {
	    goto do_error;
	}
	if (TclQddb_GetKeyList(interp, argv[2], &keylist_header) != TCL_OK)
	    return TCL_ERROR;
	if (keylist_header == NULL || keylist_header->keylist == NULL)
	    return TCL_OK;
	nargv = (char **)Malloc(sizeof(char *));
	nargc = 0;
	list = keylist_header->keylist;
	while (list != NULL) {
	    nargv = (char **)Realloc(nargv, sizeof(char *)*(nargc+2));
	    sprintf(buf, "%d %d %d %d", (int)list->Start, (int)list->Length, 
		    (int)list->Number, (int)list->Type);
	    nargv[nargc] = Malloc(strlen(buf)+1);
	    strcpy(nargv[nargc], buf);
	    nargc++;
	    list = list->next;
	}
	nargv[nargc] = NULL;
	if (nargc > 0) {
	    merged_buf = Tcl_Merge(nargc, nargv);
	} else {
	    merged_buf = Calloc(1);
	}
	for (i = 0; i < nargc; i++) {
	    Free(nargv[i]);
	}
	Free(nargv);
	Tcl_SetResult(interp, merged_buf, TCL_DYNAMIC);
	break;
    case 'r':
	if (strncmp("range", argv[1], 5) != 0) {
	    goto do_error;
	}
	if (TclQddb_KeyListRange(interp, argc, argv) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'l':
	if (strncmp("length", argv[1], 6) != 0) {
	    goto do_error;
	}
	if (TclQddb_KeyListLength(interp, argv[2], &length) != TCL_OK)
	    return TCL_ERROR;
	sprintf(buf, "%d", length);
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	break;
    case 'd':
	if (strncmp("del", argv[1], 3) != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\": should be ",
			     "op, process, get, length, or delete", NULL);
	    return TCL_ERROR;
	}
	TclQddb_DeleteKeyList(interp, argv[2], 1);
	break;	/* delete */
    default:
	goto do_error;
    }
    return TCL_OK;
do_error:
    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\": should be ",
		     "op, process, get, length, or delete", NULL);
    return TCL_ERROR;
}

int TclQddb_GetKeyList(interp, Token, List)
    Tcl_Interp			*interp;
    char			*Token;
    TclQddb_KeyListHeader	**List;
{
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_KeyListHashTableInit == 0) {
	TclQddb_KeyListHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_KeyListHashTable, TCL_STRING_KEYS);
	*List = NULL;
	Tcl_AppendResult(interp, "bad keylist \"", Token, "\"", NULL);
	return TCL_ERROR;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_KeyListHashTable, Token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "bad keylist \"", Token, "\"", NULL);
	return TCL_ERROR;
    }
    *List = (TclQddb_KeyListHeader *)Tcl_GetHashValue(hash_ptr);
    return TCL_OK;
}


int TclQddb_NewKeyList(interp, schema_desc, list, attr, append)
    Tcl_Interp			*interp;
    char			*schema_desc;
    KeyList			*list;
    int                         attr;
    int				append;
{
    char			token[BUFSIZ];
    TclQddb_KeyListHeader	*header;
    Tcl_HashEntry		*hash_ptr;
    int				newPtr;

    if (TclQddb_KeyListHashTableInit == 0) {
	TclQddb_KeyListHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_KeyListHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_keylist%d", TclQddb_KeyListNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_KeyListHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    header = (TclQddb_KeyListHeader *)Malloc(sizeof(TclQddb_KeyListHeader));
    header->keylist = list;
    header->schema_name = Malloc(strlen(schema_desc)+1);
    if (attr >= 0) {
	header->attributes.Number = 1;
	header->attributes.Nodes = (Qddb_AttrList **)Malloc(sizeof(Qddb_AttrList *));
	header->attributes.Nodes[0] = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	header->attributes.Nodes[0]->Attribute = attr;
	header->attributes.Nodes[0]->next = NULL;
    } else {
	header->attributes.Number = 0;
	header->attributes.Nodes = NULL;
    }
    strcpy(header->schema_name, schema_desc);
    Tcl_SetHashValue(hash_ptr, (ClientData)header);
    if (append) {
	Tcl_AppendElement(interp, token);
    } else {
	Tcl_ResetResult(interp);
	Tcl_SetResult(interp, token, TCL_VOLATILE);
    }
    return TCL_OK;
}

int TclQddb_NewKeyListWithAttrList(interp, schema_desc, list, attr, append)
    Tcl_Interp			*interp;
    char			*schema_desc;
    KeyList			*list;
    Qddb_AttrHead               *attr;
    int				append;
{
    char			token[BUFSIZ];
    TclQddb_KeyListHeader	*header;
    Tcl_HashEntry		*hash_ptr;
    int				newPtr;

    if (TclQddb_KeyListHashTableInit == 0) {
	TclQddb_KeyListHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_KeyListHashTable, TCL_STRING_KEYS);
    }
    sprintf(token, "qddb_keylist%d", TclQddb_KeyListNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_KeyListHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    header = (TclQddb_KeyListHeader *)Malloc(sizeof(TclQddb_KeyListHeader));
    header->keylist = list;
    header->schema_name = Malloc(strlen(schema_desc)+1);
    if (attr != NULL) {
	header->attributes.Number = attr->Number;
	header->attributes.Nodes = attr->Nodes;
    } else {
	header->attributes.Number = 0;
	header->attributes.Nodes = NULL;
    }
    strcpy(header->schema_name, schema_desc);
    Tcl_SetHashValue(hash_ptr, (ClientData)header);
    if (append) {
	Tcl_AppendElement(interp, token);
    } else {
	Tcl_ResetResult(interp);
	Tcl_SetResult(interp, token, TCL_VOLATILE);
    }
    return TCL_OK;
}

static KeyList *KeyListCopyNode(node)
    KeyList		*node;
{
    KeyList		*retval;

    retval = (KeyList *)Malloc(sizeof(KeyList));
    *retval = *node;
    retval->Instance = Malloc(strlen(node->Instance)+1);
    strcpy(retval->Instance, node->Instance);
    retval->next = NULL;
    return retval;
}

static Qddb_AttrList *KeyListCopyAttrList(attr_list)
    Qddb_AttrList       *attr_list;
{
    Qddb_AttrList       *retval = NULL, *lastnode = NULL;

    while (attr_list != NULL) {
	if (retval == NULL) {
	    retval = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	    retval->Attribute = attr_list->Attribute;
	    retval->next = NULL;
	    lastnode = retval;
	} else {
	    lastnode->next = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	    lastnode = lastnode->next;
	    lastnode->Attribute = attr_list->Attribute;
	    lastnode->next = NULL;
	}
	attr_list = attr_list->next;
    }
    return retval;
}


static Qddb_AttrHead *KeyListCopyAttrHead(attr_head)
    Qddb_AttrHead       *attr_head;
{
    Qddb_AttrHead       *retval;
    int                 i, max;

    retval = (Qddb_AttrHead *)Malloc(sizeof(Qddb_AttrHead));
    max = retval->Number = attr_head->Number;
    retval->Nodes = (Qddb_AttrList **)Malloc(sizeof(Qddb_AttrList *) * max);
    for (i = 0; i < max; i++) {
	retval->Nodes[i] = KeyListCopyAttrList(attr_head->Nodes[i]);
    }
    return retval;
}

static int TclQddb_KeyListRange(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char                        *argv[];
{
    TclQddb_KeyListHeader	*header;
    Qddb_AttrHead               *attr_head;
    KeyList			*list, *lastnode = NULL;
    KeyList                     *newnodes = NULL, *lastnewnode = NULL;
    char                        *token;
    int                         first, last;
    int                         i;

    /* qddb_keylist range <keylist_desc> first last */
    token=argv[2];
    first = atoi(argv[3]);
    if (first < 0) {
	Tcl_AppendResult(interp, argv[0], " ", argv[1], " range: bad 'first' argument", NULL);
	return TCL_ERROR;
    }
    if (strcmp(argv[4], "end") == 0) {
	last = -1;
    } else {
	last = atoi(argv[4]);
    }
    if (TclQddb_GetKeyList(interp, token, &header) != TCL_OK)
	return TCL_ERROR;
    list = header->keylist;
    lastnode = NULL;
    i = 0;
    while (i < first && list != NULL) {
	if (lastnode == NULL || !QDDB_KEYLIST_SAMEENTRY(lastnode, list)) {
	    i++;
	}
	lastnode = list;
	list = list->next;	    
    }
    if (lastnode != NULL)
	list = lastnode;
    lastnode = NULL;
    while (list != NULL && (i <= last || last == -1)) {
	if (lastnode == NULL || !QDDB_KEYLIST_SAMEENTRY(lastnode, list)) {
	    i++;
	}
	/* copy this node to the new keylist */
	if (newnodes == NULL) {
	    newnodes = KeyListCopyNode(list);
	    lastnewnode = newnodes;
	} else {
	    lastnewnode->next = KeyListCopyNode(list);
	    lastnewnode = lastnewnode->next;
	}
	lastnode = list;
	list = list->next;
    }
    if (newnodes == NULL) {
	Tcl_SetResult(interp, "", TCL_STATIC);
	return TCL_OK;
    }
    attr_head = KeyListCopyAttrHead(&header->attributes);
    if (TclQddb_NewKeyListWithAttrList(interp, header->schema_name, newnodes, attr_head, 0) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_KeyListLength(interp, token, length)
    Tcl_Interp			*interp;
    char			*token;
    int				*length;
{
    TclQddb_KeyListHeader	*header;
    KeyList			*list;

    if (TclQddb_GetKeyList(interp, token, &header) != TCL_OK)
	return TCL_ERROR;
    *length = 0;
    list = header->keylist;
    while (list != NULL) {
	(*length)++;
	list = list->next;
    }
    return TCL_OK;
}

static int TclQddb_DeleteKeyListNode(interp, Token, freeit)
    Tcl_Interp			*interp;
    char			*Token;
    int				freeit;
{
    TclQddb_KeyListHeader	*header;
    Tcl_HashEntry		*hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_KeyListHashTable, Token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find keylist \"", Token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    header = (TclQddb_KeyListHeader *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    if (header != NULL) {
	if (freeit != 0)
	    Qddb_Free(QDDB_TYPE_KEYLIST, header->keylist);
	Free(header->schema_name);
	if (header->attributes.Number > 0)
	    Qddb_Free(QDDB_TYPE_ATTRLIST, &header->attributes);
	Free(header);
    }
    return TCL_OK;
}

int TclQddb_DeleteKeyList(interp, Token, Free)
    Tcl_Interp			*interp;
    char			*Token;
    int				Free;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;
    int				deletebyschema = 0;

    if (TclQddb_KeyListHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_schema", Token, 11) == 0)
	deletebyschema = 1;
    if (strcmp(Token, "all") == 0 || deletebyschema == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_KeyListHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    if (deletebyschema == 1) {
		TclQddb_KeyListHeader	*keylist_header;

		keylist_header = (TclQddb_KeyListHeader *)Tcl_GetHashValue(hash_ptr);
		if (strcmp(keylist_header->schema_name, Token) != 0) {
		    hash_ptr = Tcl_NextHashEntry(&hash_search);
		    continue;
		}
	    }
	    hash_key = Tcl_GetHashKey(&TclQddb_KeyListHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_DeleteKeyList: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DeleteKeyListNode(interp, hash_key, Free) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	if (deletebyschema == 0) {
	    /* deleting all keylists */
	    TclQddb_KeyListHashTableInit = 0;
	    Tcl_DeleteHashTable(&TclQddb_KeyListHashTable);
	}
    } else {
	if (TclQddb_DeleteKeyListNode(interp, Token, Free) != TCL_OK)
	    return TCL_ERROR;
    }
    return TCL_OK;
}

void TclQddb_DeleteKeyListProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteKeyListProc\n");
#endif
    (void)TclQddb_DeleteKeyList((Tcl_Interp *)clientData, "all", True);
}


static int SetOpFlags(interp, str, boolstr, flags, val)
    Tcl_Interp			*interp;
    char			*str, *boolstr;
    int				*flags;
    int				val;
{
    int				op_bool;

    op_bool = 0;
    if (Tcl_GetBoolean(interp, boolstr, &op_bool) != TCL_OK) {
	Tcl_AppendResult(interp, "\n", str, " should be followed by on, off, 1, 0, true, or false", NULL);
	return TCL_ERROR;
    }
    if (op_bool == 1)
	(*flags) |= val;
    else
	(*flags) &= ~val;
    return TCL_OK;
}

static int AttrListCompareNodes(attr1, attr2)
    Qddb_AttrList               *attr1, *attr2;
{
    Qddb_AttrList               *ptr;

    while (attr1 != NULL) {
	ptr = attr2;
	while (ptr != NULL) {
	    if (ptr->Attribute != attr1->Attribute)
		return 1;
	    ptr = ptr->next;
	}
	attr1 = attr1->next;
    }
    return 0;
}

static Qddb_AttrList *AttrListMerge(attr1, attr2)
    Qddb_AttrList               *attr1, *attr2;
{
    Qddb_AttrList               *retval = NULL, *ptr, **next = NULL;
    int                         skip_flag;

    while (attr1 != NULL) {
	if (retval == NULL) {
	    retval = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	    next = &retval->next;
	    retval->Attribute = attr1->Attribute;
	} else {
	    *next = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	    (*next)->Attribute = attr1->Attribute;
	    next = &(*next)->next;
	}
	attr1 = attr1->next;
	*next = NULL;
    }
    while (attr2 != NULL) {
	skip_flag = 0;
	if (retval == NULL) {
	    retval = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
	    next = &retval->next;
	    retval->Attribute = attr2->Attribute;
	} else {
	    for (ptr = retval; ptr != NULL; ptr = ptr->next) {
		if (ptr->Attribute == attr2->Attribute) {
		    skip_flag = 1;
		    break;
		}
	    }
	    if (!skip_flag) {
		*next = (Qddb_AttrList *)Malloc(sizeof(Qddb_AttrList));
		(*next)->Attribute = attr2->Attribute;
		next = &(*next)->next;
	    }
	}
	attr2 = attr2->next;
	*next = NULL;
    }
    return retval;
}

static Qddb_AttrHead *AttrListIntersection(attr1, attr2)
    Qddb_AttrHead               *attr1, *attr2;
{
    Qddb_AttrHead               *retval;
    int                         i, j, idx;

    retval = (Qddb_AttrHead *)Malloc(sizeof(Qddb_AttrHead));
    retval->Number = attr1->Number * attr2->Number;
    if (retval->Number == 0) {
	Free(retval);
	return NULL;
    }
    retval->Nodes = (Qddb_AttrList **)Malloc(sizeof(Qddb_AttrList *) * retval->Number);
    idx = 0;
    for (i = 0; i < attr1->Number; i++) {
	for (j = 0; j < attr2->Number; j++, idx++) {
	    retval->Nodes[idx] = AttrListMerge(attr1->Nodes[i], attr2->Nodes[j]);
	}
    }
    return retval;
}

static Qddb_AttrHead *AttrListUnion(attr1, attr2)
    Qddb_AttrHead               *attr1, *attr2;
{
    Qddb_AttrHead               *retval;
    int                         i, j, idx;

    retval = (Qddb_AttrHead *)Malloc(sizeof(Qddb_AttrHead));
    retval->Number = attr1->Number + attr2->Number;
    if (retval->Number == 0) {
	Free(retval);
	return NULL;
    }
    retval->Nodes = (Qddb_AttrList **)Malloc(sizeof(Qddb_AttrList *) * retval->Number);
    idx = 0;
    for (i = 0; i < attr1->Number; i++, idx++) {
	retval->Nodes[idx] = AttrListMerge(attr1->Nodes[i], NULL);
    }
    for (i = 0; i < attr2->Number; i++) {
	for (j = 0; j < attr1->Number; j++) {
	    if (AttrListCompareNodes(retval->Nodes[j], attr2->Nodes[i]) != 0) {
		break;
	    }
	}
	if (j < idx) {
	    retval->Nodes[idx] = AttrListMerge(NULL, attr2->Nodes[i]);
	    idx++;
	}
    }
    retval->Number = idx;
    return retval;
}

static Qddb_AttrHead *AttrListExclusion(attr1, attr2)
    Qddb_AttrHead               *attr1, *attr2;
{
    Qddb_AttrHead               *retval;
    int                         i, idx;

    retval = (Qddb_AttrHead *)Malloc(sizeof(Qddb_AttrHead));
    retval->Number = attr1->Number;
    if (retval->Number == 0) {
	Free(retval);
	return NULL;
    }
    retval->Nodes = (Qddb_AttrList **)Malloc(sizeof(Qddb_AttrList *) * retval->Number);
    idx = 0;
    for (i = 0; i < attr1->Number; i++, idx++) {
	retval->Nodes[idx] = AttrListMerge(attr1->Nodes[i], NULL);
    }
    return retval;
}

static int TclQddb_KeyListOp(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Schema			*schema;
    TclQddb_KeyListHeader	*l1, *l2;
    char			tmp_schema_name[BUFSIZ];
    KeyList			*tmp_keylist;
    int				i, op_type, op_flags;
    Qddb_AttrHead               *attr_head;

    if (argc < 5) {
	Tcl_AppendResult(interp, "wrong # args", NULL);
	return TCL_ERROR;
    }
    if (strncmp(argv[1], "operation", strlen(argv[1])) != 0) {
	Tcl_AppendResult(interp, "bad arg \"", argv[1], "\", should be operation", NULL);
	return TCL_ERROR;	
    }
    switch (argv[2][0]) {
    case 'i':
	if (strncmp(argv[2], "intersection", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_OP_INTERSECTION;
	    break;
	}
    case 'u':
	if (strncmp(argv[2], "union", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_OP_UNION;
	    break;
	}
    case 'e':
	if (strncmp(argv[2], "exclusion", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_OP_EXCLUSION;
	    break;
	}
    case 'c':
        if (strncmp(argv[2], "concat", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_OP_CONCAT;
        }
    default:
	Tcl_AppendResult(interp, "bad arg \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_GetKeyList(interp, argv[argc-2], &l1) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot find keylist \"", argv[argc-2], "\"", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_GetKeyList(interp, argv[argc-1], &l2) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot find keylist \"", argv[argc-1], "\"", NULL);
	return TCL_ERROR;
    }
    if (strcmp(l1->schema_name, l2->schema_name) != 0) {
	Tcl_AppendResult(interp, "keylists have mismatched schemas: \"", l1->schema_name,
			 "\", ", l2->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(l1->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", argv[argc-3], "\", use qddb_schema open", NULL);
	return TCL_ERROR;
    }
    op_flags = 0;
    for (i = 3; i < argc-3; i++) {
	/* parse options */
	if (strcmp(argv[i], "-exact") == 0) {
	    if (SetOpFlags(interp, "-exact", argv[++i], &op_flags, QDDB_KEYLIST_FLAG_EXACT) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-copy") == 0) {
	    if (SetOpFlags(interp, "-copy", argv[++i], &op_flags, QDDB_KEYLIST_FLAG_COPY) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-deldup_sameentry") == 0) {
	    if (SetOpFlags(interp, "-deldup_sameentry", argv[++i], &op_flags, 
			 QDDB_KEYLIST_FLAG_DELDUP_SAMEENTRY) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[i], "-deldup_sameattr") == 0) {
	    if (SetOpFlags(interp, "-deldup_sameattr", argv[++i], &op_flags, 
			 QDDB_KEYLIST_FLAG_DELDUP_SAMEATTR) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], NULL);
	    return TCL_ERROR;
	}
    }
    switch (op_type) {
    case QDDB_KEYLIST_OP_INTERSECTION:
	attr_head = AttrListIntersection(&l1->attributes, &l2->attributes);
	break;
    case QDDB_KEYLIST_OP_UNION:
	attr_head = AttrListUnion(&l1->attributes, &l2->attributes);
	break;
    case QDDB_KEYLIST_OP_EXCLUSION:
	attr_head = AttrListExclusion(&l1->attributes, &l2->attributes);
	break;
    default: 
	attr_head = NULL;
	;
    }
    tmp_keylist = Qddb_KeyListOp(schema, l1->keylist, l2->keylist, op_type, op_flags);
    strcpy(tmp_schema_name, l1->schema_name);
    if ((op_flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
	int			r1, r2;

	/* If the copy flag wasn't enabled, then the two arguments are automatically
	 * deleted.
	 */
	r1 = TclQddb_DeleteKeyList(interp, argv[argc-1], 0);
	r2 = TclQddb_DeleteKeyList(interp, argv[argc-2], 0);
	if (r1 != TCL_OK || r2 != TCL_OK)
	    return TCL_ERROR;
    }
    if (TclQddb_NewKeyListWithAttrList(interp, tmp_schema_name, tmp_keylist, attr_head, 0) != TCL_OK)
	return TCL_ERROR;
    if (attr_head != NULL) {
	Free(attr_head);
    }
    return TCL_OK;
}

static int TclQddb_KeyListProcess(interp, argc, argv)
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    Schema			*schema;
    TclQddb_KeyListHeader	*l;
    char			tmp_schema_name[BUFSIZ];
    KeyList			*retval;
    int				i, op_type, op_flags;
    char			*prune_arg = NULL;

    if (argc < 4) {
	Tcl_AppendResult(interp, "wrong # args", NULL);
	return TCL_ERROR;
    }
    if (strncmp(argv[1], "process", strlen(argv[1])) != 0) {
	Tcl_AppendResult(interp, "bad arg \"", argv[1], "\", should be operation", NULL);
	return TCL_ERROR;	
    }
    if (TclQddb_GetKeyList(interp, argv[argc-1], &l) != TCL_OK) {
	Tcl_AppendResult(interp, "cannot find keylist \"", argv[argc-1], "\"", NULL);
	return TCL_ERROR;
    }
    if ((schema = TclQddb_GetSchema(l->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", l->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    strcpy(tmp_schema_name, l->schema_name);
    switch (argv[2][0]) {
    case 'm':
	if (strncmp(argv[2], "mark", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_PROC_MARK;
	    break;
	}
    case 'p':
	if (strncmp(argv[2], "prune", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_PROC_PRUNE;
	    break;
	}
    case 's':
	if (argv[2][1] == 'o' && strncmp(argv[2], "sort", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_PROC_SORT;
	    break;
	} else if (argv[2][1] == 'p' && strncmp(argv[2], "split", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_PROC_SPLIT;
	    break;
	}
    case 'n':
	if (strncmp(argv[2], "nullop", strlen(argv[2])) == 0) {
	    op_type = QDDB_KEYLIST_PROC_NULLOP;
	    break;
	}
    default:
	Tcl_AppendResult(interp, "bad arg \"", argv[2], "\"", NULL);
	return TCL_ERROR;
    }
    op_flags = 0;
    for (i = 3; i < argc-2; i++) {
	/* parse options */
	if (strncmp(argv[i], "-prunebyattr", strlen(argv[i])) == 0) {
	    if ((op_flags & QDDB_KEYLIST_FLAG_PRUNEBYROW) != 0 || 
		(op_flags & QDDB_KEYLIST_FLAG_PRUNEBYATTR) != 0) {
		Tcl_AppendResult(interp, "multiple -pruneby* flags not allowed", NULL);
		return TCL_ERROR;
	    } else if (op_type != QDDB_KEYLIST_PROC_PRUNE) {
		Tcl_AppendResult(interp, "-pruneby* flags are only allowed for proc prune", NULL);
		return TCL_ERROR;
	    }
	    op_flags |= QDDB_KEYLIST_FLAG_PRUNEBYATTR;
	    prune_arg = argv[++i];
	} else if (strncmp(argv[i], "-prunebyrow", strlen(argv[i])) == 0) {
	    if ((op_flags & QDDB_KEYLIST_FLAG_PRUNEBYROW) != 0 || 
		(op_flags & QDDB_KEYLIST_FLAG_PRUNEBYATTR) != 0) {
		Tcl_AppendResult(interp, "multiple -pruneby* flags not allowed", NULL);
		return TCL_ERROR;
	    } else if (op_type != QDDB_KEYLIST_PROC_PRUNE) {
		Tcl_AppendResult(interp, "-pruneby* flags are only allowed for proc prune", NULL);
		return TCL_ERROR;
	    }
	    op_flags |= QDDB_KEYLIST_FLAG_PRUNEBYROW;
	    prune_arg = argv[++i];
	} else if (strncmp(argv[i], "-deldup_sameentry", strlen(argv[i])) == 0) {
	    if (SetOpFlags(interp, "-deldup_sameentry", argv[++i], &op_flags, 
			 QDDB_KEYLIST_FLAG_DELDUP_SAMEENTRY) != TCL_OK)
		return TCL_ERROR;
	} else if (strncmp(argv[i], "-deldup_sameattr", strlen(argv[i])) == 0) {
	    if (SetOpFlags(interp, "-deldup_sameattr", argv[++i], &op_flags, 
			 QDDB_KEYLIST_FLAG_DELDUP_SAMEATTR) != TCL_OK)
		return TCL_ERROR;
	} else if (strncmp(argv[i], "-copy", strlen(argv[i])) == 0) {
	    if (SetOpFlags(interp, "-copy", argv[++i], &op_flags, 
			 QDDB_KEYLIST_FLAG_COPY) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad option \"", argv[i], NULL);
	    return TCL_ERROR;
	}
    }
    switch (op_type) {
    case QDDB_KEYLIST_PROC_MARK:
#if defined(notyet)

#else
	/* FIXME -- Implement the mark function.
	 */
	Tcl_AppendResult(interp, "mark not yet implemented", NULL);
	return TCL_ERROR;
#endif
    case QDDB_KEYLIST_PROC_PRUNE:
	if ((op_flags & QDDB_KEYLIST_FLAG_PRUNEBYATTR) != 0) {
	    retval = Qddb_KeyListProcess(schema, l->keylist, prune_arg, op_type, op_flags);
	} else if ((op_flags & QDDB_KEYLIST_FLAG_PRUNEBYROW) != 0) {
	    Qddb_PruneArgs		parg;
	    
	    if (Tcl_SplitList(interp, prune_arg, &parg.attr_len, &parg.attrs) != TCL_OK)
		return TCL_ERROR;
	    retval = Qddb_KeyListProcess(schema, l->keylist, &parg, op_type, op_flags);
	    Free(parg.attrs);
	} else {
	    Tcl_AppendResult(interp, "\"", argv[0], " proc prune\" requires -pruneby{row,attr}", NULL);
	    return TCL_ERROR;
	}
	break;
    case QDDB_KEYLIST_PROC_SPLIT: {
	KeyList				**list_arr, **index_arr;

	retval = Qddb_KeyListProcess(schema, l->keylist, (void *)&list_arr, op_type, op_flags);
	index_arr = list_arr;
	/* 'list_arr' now points to an array of KeyLists; add them all to
	 * TclQddb_KeyListHashTable and return the list of new keylists.
	 */
	if (list_arr != NULL) {
	    Tcl_ResetResult(interp);
	    while (*index_arr != NULL) {
		if (TclQddb_NewKeyList(interp, tmp_schema_name, *index_arr++, -1, 1) != TCL_OK) {
		    Tcl_ResetResult(interp);
		    Tcl_SetResult(interp, "Internal QDDB error: qddb_keylist process", TCL_STATIC);
		    return TCL_ERROR;
		}
	    }
	    Free(list_arr);
	    if ((op_flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
		/* If the copy flag wasn't enabled, then the argument is automatically
		 * deleted.
		 */
		if (TclQddb_DeleteKeyList(interp, argv[argc-1], 0) != TCL_OK)
		    return TCL_ERROR;
	    }
	    return TCL_OK;
	}
	Tcl_ResetResult(interp);
	break;
    }
    case QDDB_KEYLIST_PROC_SORT:
    case QDDB_KEYLIST_PROC_NULLOP:
	retval = Qddb_KeyListProcess(schema, l->keylist, NULL, op_type, op_flags);	
	break;
    default:
	Tcl_AppendResult(interp, "Internal QDDB error: qddb_keylist process", NULL);
	return TCL_ERROR;
    }
    if ((op_flags & QDDB_KEYLIST_FLAG_COPY) == 0) {
	/* If the copy flag wasn't enabled, then the argument is automatically
	 * deleted.
	 */
	if (TclQddb_DeleteKeyList(interp, argv[argc-1], 0) != TCL_OK)
	    return TCL_ERROR;
    }
    if (TclQddb_NewKeyList(interp, tmp_schema_name, retval, -1, 0) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}
