
/* TclQddb_Instance.c - TCL interface routines for Qddb instances.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static int TclQddb_InstanceCreate _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_InstanceGetCurrent _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_InstanceMaxNum _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_InstanceGetVal _ANSI_ARGS_((Tcl_Interp *, char *, char *, int));
static int TclQddb_InstanceMove _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *, char *));
static int TclQddb_InstanceSetVal _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *, char *));
static int TclQddb_InstanceSwitch _ANSI_ARGS_((Tcl_Interp *, char *, char *, size_t));
static int TclQddb_InstanceRemove _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *));
static int TclQddb_InstanceIsEmpty _ANSI_ARGS_((Tcl_Interp *, char *, char *, char *));

/* TclQddb_Instance -- Define and manipulate instances.
 *
 * <instance number> <- qddb_instance current <view desc> <attribute name>
 * <instance value>  <- qddb_instance getval <view desc> <leaf name> ?<instance number>?
 * <instance number> <- qddb_instance maxnum <view desc> <attribute name>
 *                      qddb_instance move <view desc> <attribute name> <numfrom> <numto>
 * <instance number> <- qddb_instance new <view desc> <attribute name>
 * <1|0>	     <- qddb_instance isempty <view desc> <attribute name> <instance number>
 *			qddb_instance remove <view desc> <attribute name> <instance number>
 *                      qddb_instance setval <view desc> <leaf name> <instance number> <instance value>
 *                      qddb_instance switch <view desc> <attribute name> <instance number>
 *                      qddb_instance delete <view desc> <attribute name>
 */
int TclQddb_Instance(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{
    if (argc < 4) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    switch (argv[1][0]) {
    case 'r':
	if (strcmp(argv[1], "remove") == 0) {
	    if (argc != 5) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceRemove(interp, argv[2], argv[3], argv[4]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'i':
	if (strcmp(argv[1], "isempty") == 0) {
	    if (argc != 5) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceIsEmpty(interp, argv[2], argv[3], argv[4]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'c':
	if (strcmp(argv[1], "current") == 0) {
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceGetCurrent(interp, argv[2], argv[3]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'd':
	if (strcmp(argv[1], "delete") == 0) {
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'g':
	if (strcmp(argv[1], "getval") == 0) {
	    if (argc == 4) {
		if (TclQddb_InstanceGetVal(interp, argv[2], argv[3], 0) != TCL_OK)
		    return TCL_ERROR;
	    } else if (argc == 5) {
		if (TclQddb_InstanceGetVal(interp, argv[2], argv[3], atoi(argv[4])) != TCL_OK)
		    return TCL_ERROR;
	    } else {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'm':
	if (strcmp(argv[1], "maxnum") == 0) {
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceMaxNum(interp, argv[2], argv[3]) != TCL_OK)
		return TCL_ERROR;
	} else if (strcmp(argv[1], "move") == 0) {
	    if (argc != 6) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceMove(interp, argv[2], argv[3], argv[4], argv[5]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    case 'n':
	if (strcmp(argv[1], "new") == 0) {
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceCreate(interp, argv[2], argv[3]) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break; 
    case 's':
	if (strcmp(argv[1], "setval") == 0) {
	    if (argc != 6 && argc != 5) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (argc == 5) {
		if (TclQddb_InstanceSetVal(interp, argv[2], argv[3], NULL, argv[4]) != TCL_OK)
		    return TCL_ERROR;
	    } else {
		if (TclQddb_InstanceSetVal(interp, argv[2], argv[3], argv[4], argv[5]) != TCL_OK)
		    return TCL_ERROR;
	    }
	} else if (strcmp(argv[1], "switch") == 0) {
	    if (argc != 5) {
		Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InstanceSwitch(interp, argv[2], argv[3], (size_t)atoi(argv[4])) != TCL_OK)
		return TCL_ERROR;
	} else {
	    Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	break;
    default:
 	Tcl_AppendResult(interp, "bad instance command \"", argv[1], "\": should be ", 
			 "new, current, getval, isempty, maxnum, move, ",
			 "setval, switch or remove", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static int TclQddb_InstanceGetCurrent(interp, view_token, attribute)
    Tcl_Interp			*interp;
    char			*view_token, *attribute;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    char			buf[BUFSIZ];
    int				attr_num;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute: \"", attribute, "\"", NULL);
	return TCL_ERROR;
    }
    if (attr_num > view_ptr->num_attrs) {
	Tcl_AppendResult(interp, "trying to access an attribute outside the range of view: \"",
			 attribute, "\" (internal Qddb error)", NULL);
	return TCL_ERROR;
    }
    sprintf(buf, "%d", view_ptr->curinst[attr_num]);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

static int TclQddb_InstanceMaxNum(interp, view_token, attribute)
    Tcl_Interp			*interp;
    char			*view_token, *attribute;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    DataTree			*dt;
    char			buf[BUFSIZ];
    int				attr_num, i;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute: \"", attribute, "\"", NULL);
	return TCL_ERROR;
    }
    if (attr_num > view_ptr->num_attrs) {
	Tcl_AppendResult(interp, "trying to access an attribute outside the range of view: \"",
			 attribute, "\" (internal Qddb error)", NULL);
	return TCL_ERROR;
    }
    dt = view_ptr->view[attr_num] - (view_ptr->curinst[attr_num] - 1);
    for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
    sprintf(buf, "%d", i);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}


static int TclQddb_InstanceGetVal(interp, view_token, attribute, instance)
    Tcl_Interp			*interp;
    char			*view_token, *attribute;
    int				instance;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    DataTree			*dt;
    char			*buf;
    int				attr_num, i;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute: \"", attribute, "\"", NULL);
	return TCL_ERROR;
    }
    if (attr_num > view_ptr->num_attrs) {
	Tcl_AppendResult(interp, "trying to access an attribute outside the range of view: \"",
			 attribute, "\" (internal Qddb error)", NULL);
	return TCL_ERROR;
    }
    if (schema->Entries[attr_num].IsLeaf != True) {
	Tcl_AppendResult(interp, "attribute \"", attribute, "\" is not a leaf", NULL);
	return TCL_ERROR;	
    }
    if (instance == 0)
	instance = view_ptr->curinst[attr_num];
    dt = view_ptr->view[attr_num] - (view_ptr->curinst[attr_num] - 1);
    for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
    if (instance > i) {
	char			buf[MAXINTEGERLEN];

	sprintf(buf, "%d", instance);
	Tcl_AppendResult(interp, "no such instance (", buf, ") in attribute \"", attribute, NULL);
	return TCL_ERROR;
    }
    buf = Qddb_DataTreeProcess(schema, NULL, dt+instance-1, QDDB_DATATREE_PROC_GETDATA, 0);
    if (buf == NULL) {
	Tcl_AppendResult(interp, "cannot allocate space for data", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, buf, TCL_DYNAMIC);
    return TCL_OK;
}


static int TclQddb_InstanceSetVal(interp, view_token, attribute, instance, value)
    Tcl_Interp			*interp;
    char			*view_token, *attribute, *instance, *value;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    DataTree			*dt;
    int				attr_num, inst_num, i;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute: \"", attribute, "\"", NULL);
	return TCL_ERROR;
    }
    if (attr_num > view_ptr->num_attrs) {
	Tcl_AppendResult(interp, "trying to access an attribute outside the range of view: \"",
			 attribute, "\" (internal Qddb error)", NULL);
	return TCL_ERROR;
    }
    if (schema->Entries[attr_num].IsLeaf != True) {
	Tcl_AppendResult(interp, "attribute \"", attribute, "\" is not a leaf", NULL);
	return TCL_ERROR;	
    }
    inst_num = 0;
    if (instance != NULL && Tcl_GetInt(interp, instance, &inst_num) != TCL_OK) {
	return TCL_ERROR;
    }
    if (inst_num == 0)
	inst_num = view_ptr->curinst[attr_num];
    dt = view_ptr->view[attr_num] - (view_ptr->curinst[attr_num] - 1);
    for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
    if (inst_num > i) {
	char			buf[MAXINTEGERLEN];

	sprintf(buf, "%d", inst_num);
	Tcl_AppendResult(interp, "no such instance (", buf, ") in attribute \"", attribute, NULL);
	return TCL_ERROR;
    }
    if (TclQddb_ViewLockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	return TCL_ERROR;
    switch (dt[inst_num-1].datatree_type) {
    case DATATREE_INT:
	if (value[0] == '\0') {
	    dt[inst_num-1].modified = False;
	    dt[inst_num-1].datatree_int = 0;
	    break;
	}
	if (Tcl_GetInt(interp, value, &dt[inst_num-1].datatree_int) != TCL_OK) {
	    TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name);
	    return TCL_ERROR;
	}
	dt[inst_num-1].modified = True;
	break;
    case DATATREE_REAL:
	if (value[0] == '\0') {
	    dt[inst_num-1].modified = False;
	    dt[inst_num-1].datatree_real = 0;
	    break;
	}
	if (Tcl_GetDouble(interp, value, &dt[inst_num-1].datatree_real) != TCL_OK) {
	    TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name);
	    return TCL_ERROR;
	}
	dt[inst_num-1].modified = True;
	break;
    case DATATREE_DATE:
	if (value[0] != '\0' && Qddb_CheckDate(value) != 0) {
	    TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name);
	    Tcl_AppendResult(interp, "Bad date: \"", value, "\"", NULL);
	    return TCL_ERROR;
	}
	if (dt[inst_num-1].datatree_date != NULL)
	    Free(dt[inst_num-1].datatree_date);
	dt[inst_num-1].datatree_date = Malloc(strlen(value)+1);
	strcpy(dt[inst_num-1].datatree_date, value);
	dt[inst_num-1].modified = True;
	break;
    case DATATREE_STRING:
	if (dt[inst_num-1].datatree_string != NULL)
	    Free(dt[inst_num-1].datatree_string);
	dt[inst_num-1].datatree_string = Malloc(strlen(value)+1);
	strcpy(dt[inst_num-1].datatree_string, value);
	dt[inst_num-1].modified = True;
	break;
    default:
	Tcl_AppendResult(interp, "Found bad leaf type at instance \"", instance, "\"", NULL);
	TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name);
	return TCL_ERROR;
    }
    if (TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}


static int TclQddb_InstanceSwitch(interp, view_token, attribute, instance)
    Tcl_Interp			*interp;
    char			*view_token, *attribute;
    size_t			instance;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;

    if (instance == 0)
	return TCL_OK;
    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_SetInstance(interp, schema, tuple_ptr, view_ptr, attribute, instance) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_InstanceCreate(interp, view_token, attribute)
    Tcl_Interp			*interp;
    char			*view_token, *attribute;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;
    DataTree			*dt;
    char			**attr_buf, *attr_cat = NULL, buffer[BUFSIZ], *default_value;
    int				orig_attr_num, attr_num, child_num, newinst, i;
    int				level_one_node = 0;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    attr_num = child_num = -1;
    /* 1. Find the correct attribute.
     * 2. Find its parent in the DataTree.
     * 3. Lock/unlink all the tuple's views.
     * 4. Reallocate the space for the new instance.
     * 5. Unlock/relink all the tuple's views.
     */
    if ((orig_attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute \"", attribute, "\" in schema \"",
			 view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (schema->Entries[orig_attr_num].AsteriskPresent == False) {
	Tcl_AppendResult(interp, "attribute \"", attribute, "\" is not expandable", NULL);
	return TCL_ERROR;
    }
    attr_buf = TclQddb_SplitAttributeName(attribute);
    for (i = 0; attr_buf[i] != NULL; i++);
    if (i > 1) {
	attr_cat = TclQddb_ConcatAttributeName(attr_buf, i-1);
	if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr_cat)) == -1) {
	    Tcl_AppendResult(interp, "cannot find attribute \"", attr_cat, "\" in schema \"",
			     view_ptr->schema_name, "\"", NULL);
	    goto error_ret;
	}
    }
    if (schema->Entries[orig_attr_num].Level == 1) {
	level_one_node = 1;
	child_num = TclQddb_FindAttribute(tuple_ptr->datatree, attr_buf[0]);
	if (child_num == -1) {
	    Tcl_AppendResult(interp, "cannot find level 1 child attribute \"", 
			     attr_buf[0], "\" in view \"",
			     view_token, "\"", NULL);
	    goto error_ret;
	}
    } else if (view_ptr->view[attr_num]->datatree_type == DATATREE_CHILDREN &&
	       (child_num = 
		TclQddb_FindAttribute(view_ptr->view[attr_num]->datatree_children, attr_buf[i-1])) == -1) {
	Tcl_AppendResult(interp, "cannot find child attribute \"", attr_buf[i-1], "\" in view \"",
			 view_token, "\"", NULL);
	goto error_ret;
    }
    /* Now we have the parent attribute (view_ptr->view[attr_num]) and the child
     * instance array (view_ptr->view[attr_num]->datatree_children[child_num]).
     * We're getting ready to change the datatree from under a view, so we need
     * to lock down any variables first.  This means we unlink any TCL variables
     * and save state.   Afterwards, we simply relink them to the proper instance 
     * numbers.  TclQddb_View{Lock,Unlock}Tuple handle this for all views associated
     * with this view's tuple.
     */
    if (TclQddb_ViewLockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	goto error_ret;
    if (level_one_node == 0) {
	/* > level one node */
	dt = view_ptr->view[attr_num]->datatree_children[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	dt = (DataTree *)Realloc(dt, sizeof(DataTree)*(i+2));
	view_ptr->view[attr_num]->datatree_children[child_num] = dt;
    } else {
	/* level one node */
	dt = tuple_ptr->datatree[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	dt = (DataTree *)Realloc(dt, sizeof(DataTree)*(i+2));
	tuple_ptr->datatree[child_num] = dt;
    }
    dt[i+1].datatree_type = DATATREE_END;
    dt[i].datatree_sequence_number = dt[0].datatree_sequence_number;
    dt[i].datatree_schema = dt[0].datatree_schema;
    dt[i].datatree_children = Qddb_DataTreeProcess(schema, NULL, dt[0].datatree_schema->children, 
						   QDDB_DATATREE_PROC_NEWINSTANCE, 0);
    newinst = i+1;
    if (dt[i].datatree_children != NULL) {
	Qddb_DataTreeProcess(schema, dt[i].datatree_children, &dt[0].datatree_sequence_number,
			 QDDB_DATATREE_PROC_SETSEQ, 0);
	dt[i].datatree_type = DATATREE_CHILDREN;
    } else {
	default_value = dt[i].datatree_schema->schemanode->DefaultValue;
	switch (dt[i].datatree_schema->schemanode->Type) {
	case Real:
	    if (default_value[0] == '\0') {
		dt[i].modified = False;
		dt[i].datatree_real = 0.00;
	    } else {
		dt[i].modified = True;
		dt[i].datatree_real = strtod(default_value, NULL);
	    }
	    dt[i].datatree_type = DATATREE_REAL;
	    break;
	case Integer:
	    if (default_value[0] == '\0') {
		dt[i].modified = False;
		dt[i].datatree_int = 0;
	    } else {
		dt[i].modified = True;
		dt[i].datatree_int = strtol(default_value, NULL, 0);
	    }
	    dt[i].datatree_type = DATATREE_INT;
	    break;
	case Date:
	    dt[i].datatree_type = DATATREE_DATE;
	    dt[i].datatree_date = Malloc(strlen(default_value)+1);
	    strcpy(dt[i].datatree_date, default_value);
	    dt[i].modified = True;
	    break;
	case String:
	default:
	    dt[i].datatree_type = DATATREE_STRING;
	    dt[i].datatree_string = Malloc(strlen(default_value)+1);
	    strcpy(dt[i].datatree_string, default_value);
	    dt[i].modified = True;
	    break;
	}
	dt[i].marked = Inapplicable;
    }
    if (TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	goto error_ret;
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    sprintf(buffer, "%d", newinst);
    Tcl_SetResult(interp, buffer, TCL_VOLATILE);
    return TCL_OK;
error_ret:
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_ERROR;
}

static int TclQddb_InstanceMove(interp, view_token, attribute, instfrom, instto)
    Tcl_Interp			*interp;
    char			*view_token, *attribute, *instfrom, *instto;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;
    DataTree			*dt, tmpdt1;
    char			**attr_buf, *attr_cat = NULL;
    int				orig_attr_num, attr_num, child_num, i, j;
    int				instfrom_num, instto_num;
    int				level_one_node = 0, error_occurred = 0;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, instfrom, &instfrom_num) != TCL_OK) {
	Tcl_AppendResult(interp, " (bad instance number)", NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, instto, &instto_num) != TCL_OK) {
	Tcl_AppendResult(interp, " (bad instance number)", NULL);
	return TCL_ERROR;
    }
    attr_num = child_num = -1;
    /* 1. Find the correct attribute.
     * 2. Find its parent in the DataTree.
     * 3. Lock/unlink all the tuple's views.
     * 4. Delete the instance.
     * 5. Unlock/relink all the tuple's views.
     */
    if ((orig_attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute \"", attribute, "\" in schema \"",
			 view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    attr_buf = TclQddb_SplitAttributeName(attribute);
    for (i = 0; attr_buf[i] != NULL; i++);
    if (i > 1) {
	attr_cat = TclQddb_ConcatAttributeName(attr_buf, i-1);
	if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr_cat)) == -1) {
	    Tcl_AppendResult(interp, "cannot find attribute \"", attr_cat, "\" in schema \"",
			     view_ptr->schema_name, "\"", NULL);
	    goto error_ret;
	}
    }
    if (schema->Entries[orig_attr_num].Level == 1) {
	level_one_node = 1;
	child_num = TclQddb_FindAttribute(tuple_ptr->datatree, attr_buf[0]);
	if (child_num == -1) {
	    Tcl_AppendResult(interp, "cannot find level 1 child attribute \"", 
			     attr_buf[0], "\" in view \"",
			     view_token, "\"", NULL);
	    goto error_ret;
	}
    } else if (view_ptr->view[attr_num]->datatree_type == DATATREE_CHILDREN &&
	       (child_num = 
		TclQddb_FindAttribute(view_ptr->view[attr_num]->datatree_children, attr_buf[i-1])) == -1) {
	Tcl_AppendResult(interp, "cannot find child attribute \"", attr_buf[i-1], "\" in view \"",
			 view_token, "\"", NULL);
	goto error_ret;
    }
    if (TclQddb_ViewLockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	goto error_ret;
    if (level_one_node == 0) {
	/* > level one node */
	dt = view_ptr->view[attr_num]->datatree_children[child_num];
    } else {
	/* level one node */
	dt = tuple_ptr->datatree[child_num];
    }
    for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
    if (i > 1 && i >= instfrom_num && i >= instto_num) {
	if (instfrom_num < instto_num) {
	    tmpdt1 = dt[instfrom_num-1];
	    for (j = instfrom_num-1; j < instto_num-1; j++)
		dt[j] = dt[j+1]; /* move everything down */
	    dt[instto_num-1] = tmpdt1;
	} else if (instfrom_num > instto_num) {
	    tmpdt1 = dt[instfrom_num-1];
	    for (j = instfrom_num-1; j > instto_num-1; j--)
		dt[j] = dt[j-1]; /* move everything up */
	    dt[instto_num-1] = tmpdt1;
	}
    } else {
	if (i > instfrom_num) {
	    Tcl_AppendResult(interp, "qddb_instance insert: bad instance number \"", instfrom,
			     "\"", NULL);
	    error_occurred = 1;
	} else if (i > instto_num) {
	    Tcl_AppendResult(interp, "qddb_instance insert: bad instance number \"", instto, 
			     "\"", NULL);
	    error_occurred = 1;
	}
    }
    if (TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name) != TCL_OK || error_occurred == 1)
	goto error_ret;
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_OK;
error_ret:
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_ERROR;
}

static int TclQddb_InstanceRemove(interp, view_token, attribute, instance)
    Tcl_Interp			*interp;
    char			*view_token, *attribute, *instance;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;
    DataTree			*dt;
    char			**attr_buf, *attr_cat = NULL;
    int				orig_attr_num, attr_num, child_num, i, j;
    int				instance_number;
    int				level_one_node = 0;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, instance, &instance_number) != TCL_OK) {
	Tcl_AppendResult(interp, " (bad instance number)", NULL);
	return TCL_ERROR;
    }
    attr_num = child_num = -1;
    /* 1. Find the correct attribute.
     * 2. Find its parent in the DataTree.
     * 3. Lock/unlink all the tuple's views.
     * 4. Delete the instance.
     * 5. Unlock/relink all the tuple's views.
     */
    if ((orig_attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute \"", attribute, "\" in schema \"",
			 view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    attr_buf = TclQddb_SplitAttributeName(attribute);
    for (i = 0; attr_buf[i] != NULL; i++);
    if (i > 1) {
	attr_cat = TclQddb_ConcatAttributeName(attr_buf, i-1);
	if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr_cat)) == -1) {
	    Tcl_AppendResult(interp, "cannot find attribute \"", attr_cat, "\" in schema \"",
			     view_ptr->schema_name, "\"", NULL);
	    goto error_ret;
	}
    }
    if (schema->Entries[orig_attr_num].Level == 1) {
	level_one_node = 1;
	child_num = TclQddb_FindAttribute(tuple_ptr->datatree, attr_buf[0]);
	if (child_num == -1) {
	    Tcl_AppendResult(interp, "cannot find level 1 child attribute \"", 
			     attr_buf[0], "\" in view \"",
			     view_token, "\"", NULL);
	    goto error_ret;
	}
    } else if (view_ptr->view[attr_num]->datatree_type == DATATREE_CHILDREN &&
	       (child_num = 
		TclQddb_FindAttribute(view_ptr->view[attr_num]->datatree_children, attr_buf[i-1])) == -1) {
	Tcl_AppendResult(interp, "cannot find child attribute \"", attr_buf[i-1], "\" in view \"",
			 view_token, "\"", NULL);
	goto error_ret;
    }
    if (TclQddb_ViewLockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	goto error_ret;
    if (level_one_node == 0) {
	/* > level one node */
	dt = view_ptr->view[attr_num]->datatree_children[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	if (i == 1) {
	    if (dt->datatree_type == DATATREE_CHILDREN) {
		Qddb_Free(QDDB_TYPE_DATATREE, dt->datatree_children);
		dt->datatree_children = Qddb_DataTreeProcess(schema, NULL, dt->datatree_schema->children, 
							     QDDB_DATATREE_PROC_NEWINSTANCE, 0);
		if (dt[i].datatree_children != NULL)
		    Qddb_DataTreeProcess(schema, dt->datatree_children, &dt->datatree_sequence_number,
					 QDDB_DATATREE_PROC_SETSEQ, 0);
	    } else {
		TclQddb_NullOutNode(dt, 1);
	    }
	} else if (i >= instance_number) {
	    if (dt[instance_number-1].datatree_type == DATATREE_CHILDREN) {
		Qddb_Free(QDDB_TYPE_DATATREE, dt[instance_number-1].datatree_children);
	    } else {
		TclQddb_NullOutNode(dt+instance_number-1, 0);
	    }
	    for (j = instance_number-1; j < i; j++)
		dt[j] = dt[j+1]; /* just move everything down */
	}
    } else {
	/* level one node */
	dt = tuple_ptr->datatree[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	if (i == 1) {
	    if (dt->datatree_type == DATATREE_CHILDREN) {
		Qddb_Free(QDDB_TYPE_DATATREE, dt->datatree_children);
		dt->datatree_children = Qddb_DataTreeProcess(schema, NULL, dt->datatree_schema->children, 
							     QDDB_DATATREE_PROC_NEWINSTANCE, 0);
		if (dt[i].datatree_children != NULL)
		    Qddb_DataTreeProcess(schema, dt->datatree_children, &dt->datatree_sequence_number,
					 QDDB_DATATREE_PROC_SETSEQ, 0);
	    } else {
		TclQddb_NullOutNode(dt, 1);
	    }
	} else if (i >= instance_number) {
	    if (dt[instance_number-1].datatree_type == DATATREE_CHILDREN) {
		Qddb_Free(QDDB_TYPE_DATATREE, dt[instance_number-1].datatree_children);
	    } else {
		TclQddb_NullOutNode(dt+instance_number-1, 0);
	    }
	    for (j = instance_number-1; j < i; j++)
		dt[j] = dt[j+1];
	}
    }
    if (TclQddb_ViewUnlockTuple(interp, view_ptr->tuple_name) != TCL_OK)
	goto error_ret;
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_OK;
error_ret:
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_ERROR;
}

static int IsEmptyNode(interp, dt)
    Tcl_Interp			*interp;
    DataTree			*dt;
{
    switch (dt->datatree_type) {
    case DATATREE_INT:
	if (dt->modified == True)
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	else 
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	break;
    case DATATREE_REAL:
	if (dt->modified == True)
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	else
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	break;
    case DATATREE_DATE:
	if (dt->datatree_date[0] != '\0')
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	else
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	break;
    case DATATREE_STRING:
	if (dt->datatree_string[0] != '\0')
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	else
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	break;
    default:
	Tcl_AppendResult(interp, "Bad datatree type in tuple", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}


static int TclQddb_InstanceIsEmpty(interp, view_token, attribute, instance)
    Tcl_Interp			*interp;
    char			*view_token, *attribute, *instance;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;
    DataTree			*dt;
    char			**attr_buf, *attr_cat = NULL;
    int				orig_attr_num, attr_num, child_num, i;
    int				instance_number;
    int				level_one_node = 0;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, instance, &instance_number) != TCL_OK) {
	Tcl_AppendResult(interp, " (bad instance number)", NULL);
	return TCL_ERROR;
    }
    attr_num = child_num = -1;
    if ((orig_attr_num = Qddb_ConvertAttributeNameToNumber(schema, attribute)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute \"", attribute, "\" in schema \"",
			 view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    attr_buf = TclQddb_SplitAttributeName(attribute);
    for (i = 0; attr_buf[i] != NULL; i++);
    if (i > 1) {
	attr_cat = TclQddb_ConcatAttributeName(attr_buf, i-1);
	if ((attr_num = Qddb_ConvertAttributeNameToNumber(schema, attr_cat)) == -1) {
	    Tcl_AppendResult(interp, "cannot find attribute \"", attr_cat, "\" in schema \"",
			     view_ptr->schema_name, "\"", NULL);
	    goto error_ret;
	}
    }
    if (schema->Entries[orig_attr_num].Level == 1) {
	level_one_node = 1;
	child_num = TclQddb_FindAttribute(tuple_ptr->datatree, attr_buf[0]);
	if (child_num == -1) {
	    Tcl_AppendResult(interp, "cannot find level 1 child attribute \"", 
			     attr_buf[0], "\" in view \"",
			     view_token, "\"", NULL);
	    goto error_ret;
	}
    } else if (view_ptr->view[attr_num]->datatree_type == DATATREE_CHILDREN &&
	       (child_num = 
		TclQddb_FindAttribute(view_ptr->view[attr_num]->datatree_children, attr_buf[i-1])) == -1) {
	Tcl_AppendResult(interp, "cannot find child attribute \"", attr_buf[i-1], "\" in view \"",
			 view_token, "\"", NULL);
	goto error_ret;
    }
    /* We're just peeking, so don't need to lock */
    if (level_one_node == 0) {
	/* > level one node */
	dt = view_ptr->view[attr_num]->datatree_children[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	if (i == 1) {
	    if (dt->datatree_type == DATATREE_CHILDREN) {
		if (TclQddb_IsEmptyDataTree(dt->datatree_children) == True)
		    Tcl_SetResult(interp, "1", TCL_STATIC);
		else
		    Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		if (IsEmptyNode(interp, dt) != TCL_OK)
		    goto error_ret;
	    }
	} else if (i >= instance_number) {
	    if (dt[instance_number-1].datatree_type == DATATREE_CHILDREN) {
		if (TclQddb_IsEmptyDataTree(dt[instance_number-1].datatree_children) == True)
		    Tcl_SetResult(interp, "1", TCL_STATIC);
		else
		    Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		if (IsEmptyNode(interp, dt) != TCL_OK)
		    goto error_ret;
	    }
	}
    } else {
	/* level one node */
	dt = tuple_ptr->datatree[child_num];
	for (i = 0; dt[i].datatree_type != DATATREE_END; i++);
	if (i == 1) {
	    if (dt->datatree_type == DATATREE_CHILDREN) {
		if (TclQddb_IsEmptyDataTree(dt->datatree_children) == True)
		    Tcl_SetResult(interp, "1", TCL_STATIC);
		else
		    Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		if (IsEmptyNode(interp, dt) != TCL_OK)
		    goto error_ret;
	    }
	} else if (i >= instance_number) {
	    if (dt[instance_number-1].datatree_type == DATATREE_CHILDREN) {
		if (TclQddb_IsEmptyDataTree(dt[instance_number-1].datatree_children) == True)
		    Tcl_SetResult(interp, "1", TCL_STATIC);
		else
		    Tcl_SetResult(interp, "0", TCL_STATIC);
	    } else {
		if (IsEmptyNode(interp, dt+instance_number-1) != TCL_OK)
		    goto error_ret;
	    }
	}
    }
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_OK;
error_ret:
    TclQddb_FreeArgs(-1, attr_buf);
    if (attr_cat != NULL)
	Free(attr_cat);
    return TCL_ERROR;
}
