
/* TclExpr.c - TCL interface routines for evaluating table
 * 	expressions.
 *
 * Copyright (C) 1996 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static int TclQddb_TableExprEval _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t, size_t, \
					      Qddb_TableCalc *, int *, int, \
					      Qddb_TableDependencies **));
static Qddb_Table *TclQddb_FindOverridingTable _ANSI_ARGS_((Tcl_Interp *, Qddb_TableCalc *));
static int TclQddb_TableExprEvalRange _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, size_t, size_t, \
						   Qddb_TableCalc *, int *, Qddb_DoubleList **));
static void TclQddb_TableExprRunRangeCommand _ANSI_ARGS_((Tcl_Interp *, Qddb_TableCalc *, int *, int, \
						     Qddb_DoubleList *));
static void TclQddb_TableExprRunCommand _ANSI_ARGS_((Tcl_Interp *, Qddb_TableCalc *, int *));
static size_t TclQddb_NormalizeRowCol _ANSI_ARGS_((Qddb_Table *, double, size_t, size_t));
static int TclQddb_TableExprGetRangeVals _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, \
						      size_t, size_t, Qddb_TableCalc *, \
						      int *, Qddb_DoubleList **));
static int TclQddb_TableExprGetRowVals _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, \
						    size_t, size_t, Qddb_TableCalc *, \
						    int *, Qddb_DoubleList **));
static int TclQddb_TableExprGetColVals _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, \
						    size_t, size_t, Qddb_TableCalc *, \
						    int *, Qddb_DoubleList **));
static int TclQddb_TableExprGetCellVals _ANSI_ARGS_((Tcl_Interp *, Qddb_Table *, \
						     size_t, size_t, Qddb_TableCalc *, \
						     int *, Qddb_DoubleList **));

static struct {
    Boolean		israngefunc;
    int			numargs;
    Boolean		isrange;
} TableCalcFunctions[] = {
    {False,	0, False}, /* LPAREN 300  */
    {False,	0, False}, /* RPAREN 301  */
    {False, 	0, False}, /* COMMA 302   */
    {False,	0, False}, /* COLON 303   */
    {False,	0, False}, /* ATSIGN 304  */
    {False,	2, False}, /* MOD 305     */
    {False,	2, False}, /* PLUS 306    */
    {False,	2, False}, /* MINUS 307   */
    {False,	2, False}, /* MULT 308    */
    {False,	2, False}, /* DIV 309     */
    {False,	1, False}, /* NEGATE 310  */
    {True,	0, False}, /* SUM 311     */
    {True,	0, False}, /* AVG 312     */
    {True,	0, False}, /* MIN 313     */
    {True,	0, False}, /* MAX 314     */
    {True,	0, False}, /* COUNT 315   */
    {True, 	0, False}, /* STDDEV 316  */
    {False,	1, False}, /* SQRT 317    */
    {False,	1, False}, /* EXP 318     */
    {False,	1, False}, /* LN 319      */
    {False,	1, False}, /* LOG 320     */
    {False,	1, False}, /* FLOOR 321   */
    {False,	1, False}, /* CEIL 322    */
    {False,	1, False}, /* RND 323     */
    {False,	2, False}, /* ROUND 324   */
    {False,	2, False}, /* POWER 325   */
    {False,	0, False}, /* PI 326      */
    {False,	1, False}, /* ABS 327     */
    {False,	2, False}, /* HYPOT 328   */
    {False,	1, False}, /* DEGREES 329 */
    {False,	1, False}, /* RADIANS 330 */
    {False,	1, False}, /* SIN 331     */
    {False,	1, False}, /* COS 332     */
    {False,	1, False}, /* TAN 333     */
    {False,	1, False}, /* ASIN 334    */
    {False,	1, False}, /* ACOS 335    */
    {False,	1, False}, /* ATAN 336    */
    {False,	1, False}, /* ATAN2 337   */
    {False,	0, False}, /* THISROW 338 */
    {False,	0, False}, /* THISCOL 339 */
    {False,	2, False}, /* CELLVAL 340 */
    {False,	0, False}, /* STRING 341  */
    {False,	0, False}, /* CSTRING 342 */
    {False,	0, False}, /* SVALUE 343  */
    {False,	0, False}, /* NVALUE 344  */
    {False,	0, True},  /* RANGE 345   */
    {False,	0, True},  /* CELL 346    */
    {False,	0, True},  /* ROW 347     */
    {False,	0, True},  /* COL 348     */
    {False,	0, False}, /* NUMBER 349  */
    {False, 	1, False}, /* LOGICAL_NOT 350 */
    {False,	2, False}, /* EQ 351      */
    {False,	2, False}, /* LT 352      */
    {False,	2, False}, /* GT 353      */
    {False,	2, False}, /* LE 354      */
    {False,	2, False}, /* GE 355      */
    {False,	2, False}, /* NE 356      */
    {False,	2, False}, /* LOGICAL_AND 357 */
    {False,	2, False}, /* LOGICAL_OR 358 */
    {False,	3, False}, /* IF 359      */
    {False,	0, False}, /* MAXROW 360  */
    {False,	0, False}, /* MAXCOL 361  */
    {True,	0, False}, /* PROD 362    */
};


static void PrintStack(stack, stacktop)
    Qddb_TableCalc		*stack;
    int				stacktop;
{
    int				i;

    fprintf(stderr, "->Stack trace:\n");
    for (i = stacktop-1; i >= 0; i--) {
	fprintf(stderr, "\t\t%3d: op(%d) table(%s) val(%f)\n", i, stack[i].op, 
		stack[i].table == NULL?"":stack[i].table,
		stack[i].numval);
    }
}

int TclQddb_TableExprInterp(interp, calc, table, thisrow, thiscol, dep, retval)
    Tcl_Interp			*interp;
    Qddb_TableCalc		*calc;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableDependencies	**dep;
    double			*retval;
{
    Qddb_Table			*table_override;
    Qddb_TableCalc		*calcptr;
    Qddb_TableCalc		*stack;
    int				stacktop, stackbot, max;

    *retval = 0.0;
    if (calc == NULL)
	return TCL_OK;
    calcptr = calc;
    for (max = 0; calc[max].op != 0; max++);
    stack = (Qddb_TableCalc *)Malloc(sizeof(Qddb_TableCalc)*(max+2));
    stackbot = stacktop = 0;
    while (calcptr->op != 0) {
	if (calcptr->op == QDDB_TABLE_CALC_OP_RANGE || 
	    calcptr->op == QDDB_TABLE_CALC_OP_CELL || 
	    calcptr->op == QDDB_TABLE_CALC_OP_ROW || 
	    calcptr->op == QDDB_TABLE_CALC_OP_COL ||
	    calcptr->op == QDDB_TABLE_CALC_OP_NVALUE ||
	    calcptr->op == QDDB_TABLE_CALC_OP_SVALUE) {
	    stack[stacktop].op = calcptr->op; /* push */
	    stack[stacktop].table = calcptr->table;
	    stack[stacktop].strval = calcptr->strval;
	    stack[stacktop].numval = calcptr->numval;
	    stacktop++;
	} else if (calcptr->op == QDDB_TABLE_CALC_OP_THISROW) { /* must belong to a cell */
	    stack[stacktop] = *calcptr;
	    stack[stacktop].op = QDDB_TABLE_CALC_OP_NVALUE;
	    stack[stacktop].numval = (double)thisrow;
	    stacktop++;
	} else if (calcptr->op == QDDB_TABLE_CALC_OP_THISCOL) { /* must belong to a cell */
	    stack[stacktop] = *calcptr;
	    stack[stacktop].op = QDDB_TABLE_CALC_OP_NVALUE;
	    stack[stacktop].numval = (double)thiscol;
	    stacktop++;
	} else if (calcptr->op == QDDB_TABLE_CALC_OP_MAXROW) {
	    table_override = TclQddb_FindOverridingTable(interp, calcptr);
	    if (table_override == NULL) {
		if (table == NULL) {
		    Tcl_AppendResult(interp, "error: no default table for @maxrow", NULL);
		    return TCL_ERROR;
		} else {
		    Tcl_ResetResult(interp);
		    table_override = table;
		}
	    }
	    stack[stacktop].table = calcptr->table;
	    stack[stacktop].strval = calcptr->strval;
	    stack[stacktop].op = QDDB_TABLE_CALC_OP_NVALUE;
	    stack[stacktop].numval = (double)table_override->rows;
	    stacktop++;
	} else if (calcptr->op == QDDB_TABLE_CALC_OP_MAXCOL) {
	    table_override = TclQddb_FindOverridingTable(interp, calcptr);
	    if (table_override == NULL) {
		if (table == NULL) {
		    Tcl_AppendResult(interp, "error: no default table for @maxrow", NULL);
		    return TCL_ERROR;
		} else {
		    Tcl_ResetResult(interp);
		    table_override = table;
		}
	    }
	    stack[stacktop].table = calcptr->table;
	    stack[stacktop].strval = calcptr->strval;
	    stack[stacktop].op = QDDB_TABLE_CALC_OP_NVALUE;
	    stack[stacktop].numval = (double)table_override->columns;
	    stacktop++;
	} else {
	    stack[stacktop].op = calcptr->op; /* push */
	    stack[stacktop].table = calcptr->table;
	    stack[stacktop].strval = calcptr->strval;
	    stack[stacktop].numval = calcptr->numval;
	    if (TclQddb_TableExprEval(interp, table, thisrow, thiscol, 
				      stack, &stacktop, stackbot, dep) != TCL_OK) {
		*retval = 0.0;
		return TCL_ERROR;
	    }
	    stacktop++;
	    stackbot = stacktop;
	}
	calcptr++;
    }
#if defined(DIAGNOSTIC)
    if (stack[0].op != QDDB_TABLE_CALC_OP_NVALUE) {
	fprintf(stderr, "stacktop is %d\n", stacktop);
	PrintStack(stack, stacktop);
	PANIC("TclQddb_TableExprInterp: Internal error, result not NVALUE");
    }
#endif
    *retval = stack[0].numval;
/*PrintStack(stack, stacktop);
fprintf(stderr, "returning '%f'\n", *retval);*/
    Free(stack);
    return TCL_OK;
}

static Qddb_Table *TclQddb_FindOverridingTable(interp, calcptr)
    Tcl_Interp			*interp;
    Qddb_TableCalc		*calcptr;
{
    int				range = 0, funcnum;

    while (calcptr->op != 0) {
	funcnum = calcptr->op - QDDB_TABLE_CALC_OP_START;
	if (!range && (TableCalcFunctions[funcnum].isrange || calcptr->op == QDDB_TABLE_CALC_OP_CELLVAL)) {
	    if (calcptr->table != NULL) {
		return TclQddb_GetTableByName(interp, calcptr->table);
	    }
	    range = 1;
	} else if (TableCalcFunctions[funcnum].israngefunc) {
	    if (calcptr->table != NULL) {
		return TclQddb_GetTableByName(interp, calcptr->table);
	    }
	    return NULL;
	}
	calcptr++;
    }
    return NULL;
}

static char *TclQddb_StackToString(elem)
    Qddb_TableCalc		*elem;
{
    char			*retval;

    if (elem->op == QDDB_TABLE_CALC_OP_NVALUE) {
	retval = IntegerToString((int)elem->numval);
    } else if (elem->op == QDDB_TABLE_CALC_OP_SVALUE) {
	retval = (char *)Malloc(strlen(elem->strval)+1);
	strcpy(retval, elem->strval);
    } else {
	fprintf(stderr, "TclQddb_StackToString: bad op %d, should be numeric or string\n", elem->op);
	retval = Calloc(1);
    }
    return retval;
}

static void TclQddb_PushDependentRange(dep, stack, stacktop, thisrow, thiscol)
    Qddb_TableDependencies	**dep;
    Qddb_TableCalc		*stack;
    int				stacktop;
    size_t			thisrow, thiscol;
{
    Qddb_TableCalc		*topofstack = stack + stacktop;
    Qddb_TableDependencies	*list;

    if (dep == NULL)
	return;
    list = (Qddb_TableDependencies *)Calloc(sizeof(Qddb_TableDependencies));
    list->next = *dep;
    *dep = list;
    list->table = topofstack->table;
    list->cell.row = NULL;
    list->cell.col = NULL;
    switch (topofstack->op) {
    case QDDB_TABLE_CALC_OP_RANGE:
	list->depends.from.row = TclQddb_StackToString(topofstack - 4);
	list->depends.from.col = TclQddb_StackToString(topofstack - 3);
	list->depends.to.row = TclQddb_StackToString(topofstack - 2);
	list->depends.to.col = TclQddb_StackToString(topofstack - 1);
	break;
    case QDDB_TABLE_CALC_OP_ROW:
	list->depends.from.row = TclQddb_StackToString(topofstack - 1);
	break;
    case QDDB_TABLE_CALC_OP_COL:
	list->depends.from.col = TclQddb_StackToString(topofstack - 1);
	break;
    case QDDB_TABLE_CALC_OP_CELL:
	list->depends.from.row = TclQddb_StackToString(topofstack - 2);
	list->depends.from.col = TclQddb_StackToString(topofstack - 1);
	break;
    }
}

static void TclQddb_PushDependentCell(dep, stack, stacktop, thisrow, thiscol)
    Qddb_TableDependencies	**dep;
    Qddb_TableCalc		*stack;
    int				stacktop;
    size_t			thisrow, thiscol;
{
    Qddb_TableCalc		*topofstack = stack + stacktop;
    Qddb_TableDependencies	*list;

    if (dep == NULL)
	return;
    list = (Qddb_TableDependencies *)Calloc(sizeof(Qddb_TableDependencies));
    list->next = (*dep);
    *dep = list;
    list->table = topofstack->table;
    list->cell.row = NULL;
    list->cell.col = NULL;
    list->depends.from.row = TclQddb_StackToString(topofstack - 2);
    list->depends.from.col = TclQddb_StackToString(topofstack - 1);
}

static int TclQddb_TableExprEval(interp, table, thisrow, thiscol, stack, stacktop, stackbot, dep)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*stacktop, stackbot;
    Qddb_TableDependencies	**dep;
{
    Qddb_DoubleList		*list = NULL;
    int				top = *stacktop - 1;
    
    if (stack[*stacktop].table != NULL) { /* override default table if necessary */
	if ((table = TclQddb_GetTableByName(interp, stack[*stacktop].table)) == NULL) {
	    Tcl_AppendResult(interp, "bad table name: \"", stack[*stacktop].table, "\"", NULL);
	    return TCL_ERROR;
	}
    }
#if defined(DEBUG)
    PrintStack(stack, top+2);
#endif
    if (TableCalcFunctions[stack[*stacktop].op - QDDB_TABLE_CALC_OP_START].israngefunc) {
	for (; (top >= stackbot || 
		(top >= 0 && top >= TableCalcFunctions[stack[top].op - QDDB_TABLE_CALC_OP_START].isrange)) &&
		 top >= 0; top--) {
	    TclQddb_PushDependentRange(dep, stack, top);
#if defined(DEBUG)
	    fprintf(stderr, "evaluating operation %d at stack spot %d\n", stack[top].op, top);
#endif
	    if (TclQddb_TableExprEvalRange(interp, table, thisrow, thiscol, stack, &top, &list) != TCL_OK) {
		Qddb_Free(QDDB_TYPE_DOUBLELIST, list);
		return TCL_ERROR;
	    }
#if defined(DEBUG)
	    fprintf(stderr, "top is now %d, waiting for %d\n", top-1, stackbot);
#endif
	}
	top++;
	if (top < 0)
	    top = 0;
	TclQddb_TableExprRunRangeCommand(interp, stack, stacktop, top, list);
	Qddb_Free(QDDB_TYPE_DOUBLELIST, list);
    } else {
	if (stack[*stacktop].op == QDDB_TABLE_CALC_OP_CELLVAL) {
	    if (stack[*stacktop].table != NULL) { /* override table if necessary */
		if ((table = TclQddb_GetTableByName(interp, stack[*stacktop].table)) == NULL) {
		    Tcl_AppendResult(interp, "bad table name: \"", stack[*stacktop].table, "\"", NULL);
		    return TCL_ERROR;
		}
	    }
	    if (table == NULL) {
		Tcl_AppendResult(interp, "no table for cellval function", NULL);
		return TCL_ERROR;
	    }
	    top = *stacktop;
	    TclQddb_PushDependentCell(dep, stack, *stacktop);
	    if (TclQddb_TableExprGetCellVals(interp, table, thisrow, thiscol, 
					     stack, &top, &list) != TCL_OK)
		return TCL_ERROR;
	    stack[top].op = QDDB_TABLE_CALC_OP_NVALUE;
	    stack[top].table = NULL;
	    if (list != NULL) {
		stack[top].numval = list->val;
		Qddb_Free(QDDB_TYPE_DOUBLELIST, list);
	    } else {
		stack[top].numval = 0.00;
	    }
	    *stacktop = top;
	} else {
	    TclQddb_TableExprRunCommand(interp, stack, stacktop);
	}
    }
    return TCL_OK;
}

static int TclQddb_TableExprEvalRange(interp, table, thisrow, thiscol, stack, top, list)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*top;
    Qddb_DoubleList		**list;
{
    if (stack[*top].table != NULL) { /* override table if necessary */
	if ((table = TclQddb_GetTableByName(interp, stack[*top].table)) == NULL) {
	    Tcl_AppendResult(interp, "bad table name: \"", stack[*top].table, "\"", NULL);
	    return TCL_ERROR;
	}
    }
    if (table == NULL) {
	Tcl_AppendResult(interp, "no table for range function", NULL);
	return TCL_ERROR;
    }
    switch (stack[*top].op) {
    case QDDB_TABLE_CALC_OP_RANGE:
	if (TclQddb_TableExprGetRangeVals(interp, table, thisrow, thiscol, stack, top, list) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_CALC_OP_ROW:
	if (TclQddb_TableExprGetRowVals(interp, table, thisrow, thiscol, stack, top, list) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_CALC_OP_COL:
	if (TclQddb_TableExprGetColVals(interp, table, thisrow, thiscol, stack, top, list) != TCL_OK)
	    return TCL_ERROR;
	break;
    case QDDB_TABLE_CALC_OP_CELL:
	if (TclQddb_TableExprGetCellVals(interp, table, thisrow, thiscol, stack, top, list) != TCL_OK)
	    return TCL_ERROR;
	break;
    default:
	Tcl_AppendResult(interp, "unexpected value during calculation, must be a range", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static size_t TclQddb_NormalizeRowCol(table, val, thisrow, thiscol)
    Qddb_Table			*table;
    double			val;
    size_t			thisrow, thiscol;
{
    if (val >= 1.0)
	return (size_t)val;
    return 0.0;
}

/* TclQddb_TableExprGetRangeVals - retrieve range vals from specified table.
 */
static int TclQddb_TableExprGetRangeVals(interp, table, thisrow, thiscol, stack, top, list)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*top;
    Qddb_DoubleList		**list;
{
    Qddb_DoubleList		*tmplist;
    Qddb_TableNode		*row0;
    int				tabletype;
    size_t			i, j;
    size_t			startcol, endcol, startrow, endrow;
    size_t			*sizeptr;
    
    if (stack[(*top)-4].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startrow = TclQddb_NormalizeRowCol(table, stack[(*top)-4].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW,
					       stack[(*top)-4].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad row name in calculation \"", 
			     stack[(*top)-4].strval, "\"", NULL);
	    return TCL_ERROR;
	}
	startrow = *sizeptr;
	Free(sizeptr);
    }
    if (stack[(*top)-3].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startcol = TclQddb_NormalizeRowCol(table, stack[(*top)-3].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL,
					       stack[(*top)-3].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad column name in calculation\"", 
			     stack[(*top)-3], "\"", NULL);
	    return TCL_ERROR;
	}
	startcol = *sizeptr;
	Free(sizeptr);
    }
    if (stack[(*top)-2].op == QDDB_TABLE_CALC_OP_NVALUE) {
	endrow = TclQddb_NormalizeRowCol(table, stack[(*top)-2].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW,
					       stack[(*top)-2].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad row name in calculation\"", 
			     stack[(*top)-2].strval, "\"", NULL);
	    return TCL_ERROR;
	}
	endrow = *sizeptr;
	Free(sizeptr);
    }
    if (stack[(*top)-1].op == QDDB_TABLE_CALC_OP_NVALUE) {
	endcol = TclQddb_NormalizeRowCol(table, stack[(*top)-1].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL,
					       stack[(*top)-1].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad column name in calculation\"", 
			     stack[(*top)-1], "\"", NULL);
	    return TCL_ERROR;
	}
	endcol = *sizeptr;
	Free(sizeptr);
    }
    if (startrow <= 0 || endrow <= 0 || startcol <= 0 || endcol <= 0) {
	Tcl_AppendResult(interp, "row/col index too small, must be >= 1", NULL);
	return TCL_ERROR;
    }
    if (startrow > table->rows || endrow > table->rows) {
	Tcl_AppendResult(interp, "table row index too large", NULL);
	return TCL_ERROR;
    }
    if (startcol > table->columns || endcol > table->columns) {
	Tcl_AppendResult(interp, "table column index too large", NULL);
	return TCL_ERROR;
    }
    row0 = table->table[0];
    for (i = startrow; i <= endrow; i++) {
#if defined(DEBUG)
	fprintf(stderr, "evaluating row %d for @range\n", i);
#endif
	for (j = startcol; j <= endcol; j++) {
#if defined(DEBUG)
	fprintf(stderr, "evaluating col %d for @range\n", j);
#endif
	    tabletype = table->table[i][j].data.type;
	    if (tabletype == QDDB_TABLE_TYPE_DEFAULT) {
		tabletype = row0[j].data.type;
	    }
	    switch (tabletype) {
	    case QDDB_TABLE_TYPE_REAL:
	    case QDDB_TABLE_TYPE_CALC:
		tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
		tmplist->val = table->table[i][j].data.data.real;
		tmplist->next = *list;
		*list = tmplist;
		break;
	    case QDDB_TABLE_TYPE_INTEGER:
		tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
		tmplist->val = (double)(table->table[i][j].data.data.integer);
		tmplist->next = *list;
		*list = tmplist;
		break;
	    }
	}
    }
    (*top) -= 4;
    return TCL_OK;
}

static int TclQddb_TableExprGetRowVals(interp, table, thisrow, thiscol, stack, top, list)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*top;
    Qddb_DoubleList		**list;
{
    Qddb_DoubleList		*tmplist;
    Qddb_TableNode		*row0;
    int				tabletype;
    size_t			i;
    size_t			startrow, numcolumns;
    size_t			*sizeptr;
    
    if (stack[(*top)-1].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startrow = TclQddb_NormalizeRowCol(table, stack[(*top)-1].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW,
					       stack[(*top)-1].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad row name in calculation\"", 
			     stack[(*top)-1].strval, "\"", NULL);
	    return TCL_ERROR;
	}
	startrow = *sizeptr;
	Free(sizeptr);
    }
    if (startrow <= 0) {
	Tcl_AppendResult(interp, "no context for \"thisrow\" and \"thiscol\"", NULL);
	return TCL_ERROR;
    }
    if (startrow > table->rows) {
	Tcl_AppendResult(interp, "table row index too large", NULL);
	return TCL_ERROR;
    }
    row0 = table->table[0];
    numcolumns = table->columns;
    for (i = 1; i <= numcolumns; i++) {
#if defined(DEBUG)
	fprintf(stderr, "evaluating col for @row\n");
#endif
	tabletype = table->table[startrow][i].data.type;
	if (tabletype == QDDB_TABLE_TYPE_DEFAULT) {
	    tabletype = row0[i].data.type;
	}
	switch (tabletype) {
	case QDDB_TABLE_TYPE_REAL:
	case QDDB_TABLE_TYPE_CALC:
	    tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	    tmplist->val = table->table[startrow][i].data.data.real;
	    tmplist->next = *list;
	    *list = tmplist;
	    break;
	case QDDB_TABLE_TYPE_INTEGER:
	    tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	    tmplist->val = (double)(table->table[startrow][i].data.data.integer);
	    tmplist->next = *list;
	    *list = tmplist;
	    break;
	}
    }
    (*top)--;
    return TCL_OK;
}

static int TclQddb_TableExprGetColVals(interp, table, thisrow, thiscol, stack, top, list)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*top;
    Qddb_DoubleList		**list;
{
    Qddb_DoubleList		*tmplist;
    Qddb_TableNode		*row0;
    int				tabletype;
    size_t			i;
    size_t			startcol, numrows;
    size_t			*sizeptr;
    
    if (stack[(*top)-1].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startcol = TclQddb_NormalizeRowCol(table, stack[(*top)-1].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL,
					       stack[(*top)-1].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad column name in calculation\"", 
			     stack[(*top)-1].strval, "\"", NULL);
	    return TCL_ERROR;
	}
	startcol = *sizeptr;
	Free(sizeptr);
    }
    if (startcol <= 0) {
	Tcl_AppendResult(interp, "no context for \"thisrow\" and \"thiscol\"", NULL);
	return TCL_ERROR;
    }
    if (startcol > table->columns) {
	Tcl_AppendResult(interp, "table column index too large", NULL);
	return TCL_ERROR;
    }
    row0 = table->table[0];
    numrows = table->rows;
    for (i = 1; i <= numrows; i++) {
#if defined(DEBUG)
	fprintf(stderr, "evaluating row for @col\n");
#endif
	tabletype = table->table[i][startcol].data.type;
	if (tabletype == QDDB_TABLE_TYPE_DEFAULT) {
	    tabletype = row0[startcol].data.type;
	}
	switch (tabletype) {
	case QDDB_TABLE_TYPE_REAL:
	case QDDB_TABLE_TYPE_CALC:
	    tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	    tmplist->val = table->table[i][startcol].data.data.real;
	    tmplist->next = *list;
	    *list = tmplist;
	    break;
	case QDDB_TABLE_TYPE_INTEGER:
	    tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	    tmplist->val = (double)(table->table[i][startcol].data.data.integer);
	    tmplist->next = *list;
	    *list = tmplist;
	    break;
	}
    }
    (*top)--;
    return TCL_OK;
}

static int TclQddb_TableExprGetCellVals(interp, table, thisrow, thiscol, stack, top, list)
    Tcl_Interp			*interp;
    Qddb_Table			*table;
    size_t			thisrow, thiscol;
    Qddb_TableCalc		*stack;
    int				*top;
    Qddb_DoubleList		**list;
{
    Qddb_DoubleList		*tmplist;
    Qddb_TableNode		*row0;
    int				tabletype;
    size_t			startcol, startrow;
    size_t			*sizeptr;
    
    if (stack[(*top)-2].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startrow = TclQddb_NormalizeRowCol(table, stack[(*top)-2].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDROW,
					       stack[(*top)-2].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad row name in calculation\"", 
			     stack[(*top)-2].strval, "\"", NULL);
	    return TCL_ERROR;
	}
	startrow = *sizeptr;
	Free(sizeptr);
    }
    if (stack[(*top)-1].op == QDDB_TABLE_CALC_OP_NVALUE) {
	startcol = TclQddb_NormalizeRowCol(table, stack[(*top)-1].numval, thisrow, thiscol);
    } else {
	if (*(sizeptr = (size_t *)Qddb_TableOp(table, QDDB_TABLE_OP_FINDCOL,
					       stack[(*top)-1].strval)) == 0) {
	    Free(sizeptr);
	    Tcl_AppendResult(interp, "bad column name in calculation\"", 
			     stack[(*top)-1], "\"", NULL);
	    return TCL_ERROR;
	}
	startcol = *sizeptr;
	Free(sizeptr);
    }
    if (startrow <= 0 || startcol <= 0) {
	Tcl_AppendResult(interp, "no context for \"thisrow\" and \"thiscol\"", NULL);
	return TCL_ERROR;
    }
    if (startrow > table->rows) {
	Tcl_AppendResult(interp, "table row index too large", NULL);
	return TCL_ERROR;
    }
    if (startcol > table->columns) {
	Tcl_AppendResult(interp, "table column index too large", NULL);
	return TCL_ERROR;
    }
    row0 = table->table[0];
    tabletype = table->table[startrow][startcol].data.type;
    if (tabletype == QDDB_TABLE_TYPE_DEFAULT) {
	tabletype = row0[startcol].data.type;
    }
    switch (tabletype) {
    case QDDB_TABLE_TYPE_REAL:
    case QDDB_TABLE_TYPE_CALC:
	tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	tmplist->val = table->table[startrow][startcol].data.data.real;
	tmplist->next = *list;
	*list = tmplist;
	break;
    case QDDB_TABLE_TYPE_INTEGER:
	tmplist = (Qddb_DoubleList *)Malloc(sizeof(Qddb_DoubleList));
	tmplist->val = (double)(table->table[startrow][startcol].data.data.integer);
	tmplist->next = *list;
	*list = tmplist;
	break;
    }
    (*top) -= 2;
    return TCL_OK;
}

static void TclQddb_TableExprRunRangeCommand(interp, stack, top, newtop, list)
    Tcl_Interp			*interp;
    Qddb_TableCalc		*stack;
    int				*top, newtop;
    Qddb_DoubleList		*list;
{
    Qddb_DoubleList		*tmplist;
    double			mean, num, num2, count;

    if (list == NULL)
	return;
    num = 0.0;
    count = 0.0;
    switch (stack[(*top)].op) {
    case QDDB_TABLE_CALC_OP_SUM:
	while (list != NULL) {
	    num += list->val;
	    list = list->next;
	}
	break;
    case QDDB_TABLE_CALC_OP_PROD:
	num = list->val;
	list = list->next;
	while (list != NULL) {
	    num *= list->val;
	    list = list->next;
	}
	break;
    case QDDB_TABLE_CALC_OP_AVG:
	while (list != NULL) {
	    num += list->val;
	    count++;
	    list = list->next;
	}
	num /= count;
	break;
    case QDDB_TABLE_CALC_OP_MIN:
	num = list->val;
	list = list->next;
	while (list != NULL) {
	    if (list->val < num) {
		num = list->val;
	    }
	    list = list->next;
	}
	break;
    case QDDB_TABLE_CALC_OP_MAX:
	num = list->val;
	list = list->next;
	while (list != NULL) {
	    if (list->val > num) {
		num = list->val;
	    }
	    list = list->next;
	}
	break;
    case QDDB_TABLE_CALC_OP_COUNT:
	while (list != NULL) {
	    num++;
	    list = list->next;
	}
	break;
    case QDDB_TABLE_CALC_OP_STDDEV:
	tmplist = list;
	while (list != NULL) {
	    num += list->val;
	    count++;
	    list = list->next;
	}
	mean = num / count;
	list = tmplist;
	num = 0.0;
	while (list != NULL) {
	    num2 = (list->val - mean);
	    num += num2*num2;
	    list = list->next;
	}
	num = sqrt(num  / (count - 1.0));
	break;
    default:
	;
    }
    stack[newtop].op = QDDB_TABLE_CALC_OP_NVALUE;
    stack[newtop].table = NULL;
    stack[newtop].numval = num;
    *top = newtop;
}

static void TclQddb_TableExprRunCommand(interp, stack, top)
    Tcl_Interp			*interp;
    Qddb_TableCalc		*stack;
    int				*top;
{
    int				mytop = *top;
    double			num, num2, num3;

    num = 0.0;
    switch (stack[mytop].op) {
    case QDDB_TABLE_CALC_OP_IF:
	if (stack[mytop-3].numval != 0.0) {
	    num = stack[mytop-2].numval;
	} else {
	    num = stack[mytop-1].numval;
	}
	break;
    case QDDB_TABLE_CALC_OP_EQ:
	if (stack[mytop-2].numval == stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_NE:
	if (stack[mytop-2].numval != stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_GT:
	if (stack[mytop-2].numval > stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_GE:
	if (stack[mytop-2].numval >= stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_LT:
	if (stack[mytop-2].numval < stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_LE:
	if (stack[mytop-2].numval <= stack[mytop-1].numval) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_LOGICAL_AND:
	if (stack[mytop-2].numval != 0.0 && stack[mytop-1].numval != 0.0) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_LOGICAL_OR:
	if (stack[mytop-2].numval != 0.0 || stack[mytop-1].numval != 0.0) {
	    num = 1.0;
	} else {
	    num = 0.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_LOGICAL_NOT:
	if (stack[mytop-1].numval != 0.0) {
	    num = 0.0;
	} else {
	    num = 1.0;
	}
	break;
    case QDDB_TABLE_CALC_OP_NEGATE:
	num = -stack[mytop-1].numval;
	break;
    case QDDB_TABLE_CALC_OP_PLUS:
	num = stack[mytop-2].numval + stack[mytop-1].numval;
	break;
    case QDDB_TABLE_CALC_OP_MINUS:
	num = stack[mytop-2].numval - stack[mytop-1].numval;
	break;
    case QDDB_TABLE_CALC_OP_MULT:
	num = stack[mytop-2].numval * stack[mytop-1].numval;
	break;
    case QDDB_TABLE_CALC_OP_DIV:
	num = stack[mytop-2].numval / stack[mytop-1].numval;
	break;
    case QDDB_TABLE_CALC_OP_MOD:
	num = fmod(stack[mytop-2].numval, stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_SQRT:
	num = sqrt(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_EXP:
	num = exp(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_LN:
	num = log(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_LOG:
	num = log10(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_FLOOR:
	num = floor(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_CEIL:
	num = ceil(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_RND:
#if defined(HAVE_RINT)
	num = rint(stack[mytop-1].numval);
#else
	num = floor(stack[mytop-1].numval+0.5);
#endif
	break;
    case QDDB_TABLE_CALC_OP_ROUND: /* FIXME: this is a bit stupid */
	num2 = modf(stack[mytop-2].numval, &num);
	num3 = pow(10.0, floor(stack[mytop-1].numval));
	num  *= num3;
	num2 *= num3;
	num2 = floor(num2);
	num += num2;
	num /= num3;
	break;
    case QDDB_TABLE_CALC_OP_POWER:
	num = pow(stack[mytop-2].numval, stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_ABS:
	num = fabs(stack[mytop-1].numval);
	break;
#if defined(HAVE_HYPOT)
    case QDDB_TABLE_CALC_OP_HYPOT:
	num = hypot(stack[mytop-2].numval, stack[mytop-1].numval);
	break;
#endif
    case QDDB_TABLE_CALC_OP_DEGREES:
	num = (stack[mytop-1].numval / M_PI_4) * 45.0;
	break;
    case QDDB_TABLE_CALC_OP_RADIANS:
	num = (stack[mytop-1].numval / 45.0) * M_PI_4;
	break;
    case QDDB_TABLE_CALC_OP_SIN:
	num = sin(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_COS:
	num = cos(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_TAN:
	num = tan(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_ASIN:
	num = asin(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_ACOS:
	num = acos(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_ATAN:
	num = atan(stack[mytop-1].numval);
	break;
    case QDDB_TABLE_CALC_OP_ATAN2:
	num = atan2(stack[mytop-2].numval, stack[mytop-1].numval);
	break;
    default:
	fprintf(stderr, "Warning: invalid command %d\n", stack[mytop].op);
    }
    *top = mytop - TableCalcFunctions[stack[mytop].op - QDDB_TABLE_CALC_OP_START].numargs;
    if (*top < 0) {
	fprintf(stderr, "Warning: top is %d on operator %d\n", *top, stack[mytop].op);
	*top = 0;
    }
    stack[*top].op = QDDB_TABLE_CALC_OP_NVALUE;
    stack[*top].table = NULL;
    stack[*top].numval = num;
}
