/***************************************************************************
                          toplevel.cpp  -  Main application class
                             -------------------
    begin                : Thu Apr 1 1999
    copyright            : (C) 1999 by Mutiny Bay Software
    			   (C) 1998 by Jochen Wilhelmy (kwrite portions)
    email                : info@mutinybaysoftware.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qkeycode.h>
#include <qmsgbox.h>

#include <kapp.h>
#include <kiconloader.h>
#include <kurl.h>
#include <kfiledialog.h>

#include "toplevel.h"
#include "highlight.h"
#include "connectdialog.h"
#include "outputdialog.h"
#include "cprintdlg.h"

// StatusBar field IDs
#define ID_LINE_COLUMN 1
#define ID_INS_OVR 2
#define ID_MODIFIED 3
#define ID_GENERAL 4

// Toolbar button IDs
enum toolbarButtons {
  toolNew = 1,
  toolOpen,
  toolSave,
  toolPrint,
  toolCut,
  toolCopy,
  toolPaste,
  toolUndo,
  toolRedo,
  toolSearch,
  toolBookmark,
  toolHelp,
  toolConnect,
  toolExecute,
  toolCancel,
  toolOptions,
  toolClear,
  toolSpool
};

QList<KWriteDoc> docList; 	//documents
HlManager hlManager; 		//highlight manager

TopLevel::TopLevel (KWriteDoc *doc) : KTMainWindow("Kpsql") {

  setMinimumSize(180,120);

  recentFiles.setAutoDelete(TRUE);

  statusbarTimer = new QTimer(this);
  connect(statusbarTimer,SIGNAL(timeout()),this,SLOT(timeout()));

  setupEditWidget(doc);

  connection = new DBAccess(this, "Connection");

  setupMenuBar();
  setupToolBars();
  setupStatusBar();

  KDNDDropZone *dropZone = new KDNDDropZone(this,DndURL);
  connect(dropZone,SIGNAL(dropAction(KDNDDropZone *)),
    this,SLOT(dropAction(KDNDDropZone *)));
}

TopLevel::~TopLevel() {
  if (kWrite->isLastView()) docList.remove(kWrite->doc());
  delete connection;
}

void TopLevel::init() {

  hideToolBar = !hideToolBar;
  toggleToolBar();
  hideToolBarSql = !hideToolBarSql;
  toggleToolBarSql();
  hideStatusBar = !hideStatusBar;
  toggleStatusBar();
  toggleQueryState();
  newCurPos();
  newStatus();
  newCaption();
  newUndo();

  results->updateTextFont();
  results->updateColors();

  connection->closeConnection();

  show();
}

bool TopLevel::queryClose() {
  int query;

  if (kWrite->isModified() && kWrite->isLastView()) {
    query = QMessageBox::information(this,
      i18n("Message"),
      i18n("This query has been modified.\nWould you like to save it?"),
      i18n("Yes"),
      i18n("No"),
      i18n("Cancel"),
      0,2);

    switch (query) {
      case 0: //yes
        kWrite->save();
        if (kWrite->isModified()) return false;
        break;
      case 2: //cancel
        return false;
    }
  }
  return true;
}

bool TopLevel::queryExit() {
  writeConfig();
  return true;
}

void TopLevel::loadURL(const char *url, int flags) {
  kWrite->loadURL(url,flags);
}


void TopLevel::setupEditWidget(KWriteDoc *doc) {

  if (!doc) {
    doc = new KWriteDoc(&hlManager);
    docList.append(doc);
  }

  // Create the splitter widget	
  splitter = new KNewPanner(this, "splitter", KNewPanner::Horizontal, KNewPanner::Percent, 65);	

  // Create the splitter's children	
  kWrite = new KWrite(doc, splitter);
  results = new KResultWindow(0, splitter, "result", 0);	

  // Begin managing the child widgets with the splitter widget
  splitter ->activate(kWrite, results);

  connect(kWrite,SIGNAL(newCurPos()),this,SLOT(newCurPos()));
  connect(kWrite,SIGNAL(newStatus()),this,SLOT(newStatus()));
  connect(kWrite,SIGNAL(statusMsg(const char *)),this,SLOT(statusMsg(const char *)));
  connect(kWrite,SIGNAL(newCaption()),this,SLOT(newCaption()));
  connect(kWrite,SIGNAL(newUndo()),this,SLOT(newUndo()));

  setView(splitter, FALSE);
}

void TopLevel::setupMenuBar() {
  int z;

  file =        new QPopupMenu();
  edit =        new QPopupMenu();
  view =        new QPopupMenu();
  query =	new QPopupMenu();
  options =     new QPopupMenu();
  help =        new QPopupMenu();
  recentPopup = new QPopupMenu();

  file->insertItem(Icon("connect.xpm"), i18n("Connect to &Database..."),this, SLOT(newConnection()),CTRL+Key_D);
  file->insertSeparator ();
  file->insertItem(Icon("new.xpm"), i18n("Ne&w"),kWrite,SLOT(newDoc()),CTRL+Key_N);
  file->insertItem(Icon("open.xpm"), i18n("&Open..."),kWrite,SLOT(open()),CTRL+Key_O);
  file->insertItem(i18n("&Insert..."), kWrite,SLOT(insertFile()));
  file->insertItem(i18n("Open Recen&t"), recentPopup);
  connect(recentPopup,SIGNAL(activated(int)),SLOT(openRecent(int)));
  file->insertSeparator ();
  file->insertItem(Icon("save.xpm"), i18n("&Save"),kWrite,SLOT(save()),CTRL+Key_S);
  file->insertItem(i18n("S&ave as..."),kWrite,SLOT(saveAs()));
  file->insertItem(i18n("&Close"),this,SLOT(closeWindow()),CTRL+Key_W);
  file->insertSeparator ();
  file->insertItem(Icon("print.xpm"),i18n("&Print..."),this,SLOT(print()),CTRL+Key_P);
  file->insertSeparator ();
  file->insertItem (i18n("New &Window"),this,SLOT(newWindow()));

  edit->insertItem(Icon("cut.xpm"), i18n("C&ut"),this,SLOT(cut()),CTRL+Key_X);
  edit->insertItem(Icon("copy.xpm"), i18n("&Copy"),this,SLOT(copy()),CTRL+Key_C);
  edit->insertItem(Icon("paste.xpm"), i18n("&Paste"),kWrite,SLOT(paste()),CTRL+Key_V);
  edit->insertSeparator();
  edit->insertItem(Icon("find.xpm"), i18n("&Find..."),kWrite,SLOT(search()),CTRL+Key_F);
  edit->insertItem(i18n("&Replace..."),kWrite,SLOT(replace()),CTRL+Key_R);
  edit->insertItem(i18n("Find &Again"),kWrite,SLOT(searchAgain()),Key_F3);
  edit->insertSeparator();
  menuUndo = edit->insertItem(Icon("undo.xpm"), i18n("U&ndo"),kWrite,SLOT(undo()),CTRL+Key_Z);
  menuRedo = edit->insertItem(Icon("redo.xpm"), i18n("R&edo"),kWrite,SLOT(redo()),CTRL+Key_Y);
  edit->insertSeparator();
  edit->insertItem(Icon("indent.xpm"), i18n("&Indent"),kWrite,SLOT(indent()),CTRL+Key_I);
  edit->insertItem(Icon("unindent.xpm"), i18n("Uninden&t"),kWrite,SLOT(unIndent()),CTRL+Key_U);
  edit->insertSeparator();
  edit->insertItem(i18n("&Select All"),kWrite,SLOT(selectAll()));
  edit->insertItem(i18n("&Deselect All"),kWrite,SLOT(deselectAll()));
  edit->insertItem(i18n("In&vert Selection"),kWrite,SLOT(invertSelection()));
  edit->insertSeparator();
  edit->insertItem(Icon("clear.xpm"), i18n("Clear Result &Window"),results,SLOT(clear()));

  view->insertItem(i18n("&Line..."),kWrite,SLOT(gotoLine()),CTRL+Key_G);
  view->insertSeparator();
  menuShowTB = view->insertItem(i18n("Show Editor &Toolbar"),this,SLOT(toggleToolBar()));
  menuShowTBSql = view->insertItem(i18n("Show S&QL Toolbar"),this,SLOT(toggleToolBarSql()));
  menuShowSB = view->insertItem(i18n("Show &Statusbar"),this,SLOT(toggleStatusBar()));
  view->insertSeparator();
  view->insertItem(Icon("bookmark.xpm"), i18n("Set &Bookmark..."),kWrite,SLOT(setBookmark()),ALT+Key_S);
  view->insertItem(i18n("&Add Bookmark"),kWrite,SLOT(addBookmark()));
  view->insertItem(i18n("&Clear Bookmarks"),kWrite,SLOT(clearBookmarks()),ALT+Key_C);
  kWrite->installBMPopup(view);

  menuExecute = query->insertItem(Icon("go.xpm"), i18n("&Execute Query"),this,SLOT(executeQuery()),CTRL+Key_E);
  menuCancel = query->insertItem(Icon("stop.xpm"), i18n("&Cancel Query"),this,SLOT(cancelQuery()),CTRL+Key_Q);

  //highlight selector
  hlPopup = new QPopupMenu();
  hlPopup->setCheckable(true);
  for (z = 0; z < hlManager.highlights(); z++) {
    hlPopup->insertItem(i18n(hlManager.hlName(z)),z);
  }
  connect(hlPopup,SIGNAL(activated(int)),kWrite,SLOT(setHl(int)));

  options->setCheckable(TRUE);
  options->insertItem(i18n("Set Highlight"),hlPopup);
  connect(hlPopup,SIGNAL(aboutToShow()),this,SLOT(showHighlight()));
  options->insertSeparator();
  options->insertItem(i18n("&Editor..."),kWrite,SLOT(optDlg()));
  options->insertItem(i18n("Editor &Defaults..."),kWrite,SLOT(hlDef()));
  options->insertItem(i18n("Editor &Colors..."),kWrite,SLOT(colDlg()));
  options->insertItem(i18n("Syntax &Highlighting..."),kWrite,SLOT(hlDlg()));

  options->insertSeparator();
  options->insertItem(Icon("results.xpm"), i18n("Result &Window..."),this,SLOT(setOutputOptions()));
  options->insertItem(i18n("Result Window &Font..."),results,SLOT(setTextFont()));
  options->insertItem(i18n("&Result Window Colors..."),results,SLOT(setColors()));
  options->insertSeparator();
  options->insertItem(Icon("spool.xpm"), i18n("&Set Spool File..."),this, SLOT(setSpoolFile()));
  options->insertSeparator();
  options->insertItem(i18n("Sa&ve Configuration"),this,SLOT(writeConfig()));

  help = kapp->getHelpMenu(true, 0);
	help->removeItemAt(0);
  help->insertItem(Icon("help.xpm"), i18n("&Contents"),this,SLOT(helpSelected()),Key_F1,0,0);
  help->removeItemAt(2);
  help->insertItem(i18n("&About Kpsql..."),this,SLOT(about()),0,-1,2);

  // Right mouse button popup menu (editor)
  popup = new QPopupMenu();
  popup->insertItem(Icon("open.xpm"), i18n("&Open..."),kWrite,SLOT(open()),CTRL+Key_O);
  popup->insertItem(Icon("save.xpm"), i18n("&Save"),kWrite,SLOT(save()),CTRL+Key_S);
  popup->insertItem(i18n("S&ave as..."),kWrite,SLOT(saveAs()));
  popup->insertSeparator();
  popup->insertItem(Icon("cut.xpm"), i18n("C&ut"),this,SLOT(cut()),CTRL+Key_X);
  popup->insertItem(Icon("copy.xpm"),i18n("&Copy"),this,SLOT(copy()),CTRL+Key_C);
  popup->insertItem(Icon("paste.xpm"), i18n("&Paste"),kWrite,SLOT(paste()),CTRL+Key_V);
  popup->insertSeparator();
	popup->insertItem(Icon("find.xpm"), i18n("&Find..."),kWrite,SLOT(search()),CTRL+Key_F);
  kWrite->installRBPopup(popup);

  // Right mouse button popup menu (results)
  resultPopup = new QPopupMenu();
  resultPopup->insertItem(Icon("copy.xpm"), i18n("&Copy"),results,SLOT(copyText()));
  results->installRBPopup(resultPopup);

  // Top level menus
  menubar = menuBar();
  menuFile = menubar->insertItem(i18n("&File"),file);
  menuEdit = menubar->insertItem(i18n("&Edit"),edit);
  menuView = menubar->insertItem(i18n("&View"),view);
  menubar->insertItem(i18n("&Query"),query);
  menuOptions = menubar->insertItem(i18n("&Options"),options);
  menubar->insertSeparator();
  menuHelp = menubar->insertItem(i18n("&Help"),help);

}

void TopLevel::setupToolBars(){
  KToolBar *toolbar;
  //QFrame *separatorLine;

  toolbar = toolBar(0);

  KIconLoader *loader = kapp->getIconLoader();

  QPixmap pixmap;

  // Editor toolbar
  pixmap = loader->loadIcon("new.xpm");
  toolbar->insertButton(pixmap,toolNew,SIGNAL(clicked()),
    kWrite,SLOT(newDoc()),TRUE,i18n("New Query"));
  pixmap = loader->loadIcon("open.xpm");
  toolbar->insertButton(pixmap,toolOpen,SIGNAL(clicked()),
    kWrite,SLOT(open()),TRUE,i18n("Open Query"));
  pixmap = loader->loadIcon("save.xpm");
  toolbar->insertButton(pixmap,toolSave,SIGNAL(clicked()),
    kWrite,SLOT(save()),TRUE,i18n("Save Query"));
  pixmap = loader->loadIcon("print.xpm");
  toolbar->insertButton(pixmap,toolPrint,SIGNAL(clicked()),
    this,SLOT(print()),TRUE,i18n("Print Query"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("cut.xpm");
  toolbar->insertButton(pixmap,toolCut,SIGNAL(clicked()),
    this,SLOT(cut()),TRUE,i18n("Cut"));
  pixmap = loader->loadIcon("copy.xpm");
  toolbar->insertButton(pixmap,toolCopy,SIGNAL(clicked()),
    this,SLOT(copy()),TRUE,i18n("Copy"));
  pixmap = loader->loadIcon("paste.xpm");
  toolbar->insertButton(pixmap,toolPaste,SIGNAL(clicked()),
    kWrite,SLOT(paste()),TRUE,i18n("Paste"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("undo.xpm");
  toolbar->insertButton(pixmap,toolUndo,SIGNAL(clicked()),
    kWrite,SLOT(undo()),TRUE,i18n("Undo"));
  pixmap = loader->loadIcon("redo.xpm");
  toolbar->insertButton(pixmap,toolRedo,SIGNAL(clicked()),
    kWrite,SLOT(redo()),TRUE,i18n("Redo"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("find.xpm");
  toolbar->insertButton(pixmap,toolSearch,SIGNAL(clicked()),
    kWrite,SLOT(search()),TRUE,i18n("Find"));
  pixmap = loader->loadIcon("bookmark.xpm");
  toolbar->insertButton(pixmap,toolBookmark,SIGNAL(clicked()),
    kWrite,SLOT(setBookmark()),TRUE,i18n("Set Bookmark"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("help.xpm");
  toolbar->insertButton(pixmap,toolHelp,SIGNAL(clicked()),
    this,SLOT(helpSelected()),TRUE,i18n("Help"));

  toolbar->setBarPos(KToolBar::Top);

  // SQL toolbar
  toolbar = toolBar(1);

  pixmap = loader->loadIcon("connect.xpm");
  toolbar->insertButton(pixmap,toolConnect,SIGNAL(clicked()),
    this,SLOT(newConnection()),TRUE,i18n("Connect To Database"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("go.xpm");
  toolbar->insertButton(pixmap,toolExecute,SIGNAL(clicked()),
    this,SLOT(executeQuery()),TRUE,i18n("Execute Query (Ctrl+E)"));
  pixmap = loader->loadIcon("stop.xpm");
  toolbar->insertButton(pixmap,toolCancel,SIGNAL(clicked()),
    this,SLOT(cancelQuery()),TRUE,i18n("Cancel Executing Query (Ctrl+Q)"));
  toolbar->insertSeparator();
  pixmap = loader->loadIcon("results.xpm");
  toolbar->insertButton(pixmap,toolOptions,SIGNAL(clicked()),
    this,SLOT(setOutputOptions()),TRUE,i18n("Result Window Output Options"));
  pixmap = loader->loadIcon("spool.xpm");
  toolbar->insertButton(pixmap,toolSpool,SIGNAL(clicked()),
    this,SLOT(setSpoolFile()),TRUE,i18n("Set Spool File"));
  pixmap = loader->loadIcon("clear.xpm");
  toolbar->insertButton(pixmap,toolClear,SIGNAL(clicked()),
    results,SLOT(clear()),TRUE,i18n("Clear Result Window"));

  toolbar->setBarPos(KToolBar::Top);
}

void TopLevel::setupStatusBar(){
    KStatusBar *statusbar;
    statusbar = statusBar();
    statusbar->insertItem("Line:000000 Col: 000", ID_LINE_COLUMN);
    statusbar->insertItem("XXX", ID_INS_OVR);
    statusbar->insertItem("*", ID_MODIFIED);
    statusbar->insertItem("", ID_GENERAL);

    statusbar->setInsertOrder(KStatusBar::RightToLeft);
    statusbar->setAlignment(ID_INS_OVR,AlignCenter);
}

void TopLevel::doEvents(){
  kapp->flushX();
  kapp->processEvents();
}

void TopLevel::openRecent(int id) {
  loadURL(recentPopup->text(id));
}

void TopLevel::newWindow() {
  system("kpsql &");
}

void TopLevel::closeWindow() {
  close();
}


void TopLevel::quitEditor() {
  kapp->quit();
}

void TopLevel::toggleToolBar() {

  view->setItemChecked(menuShowTB,hideToolBar);
  if (hideToolBar) {
    hideToolBar = FALSE;
    enableToolBar(KToolBar::Show, 0);
  } else {
    hideToolBar = TRUE;
    enableToolBar(KToolBar::Hide, 0);
  }
}

void TopLevel::toggleToolBarSql() {

  view->setItemChecked(menuShowTBSql,hideToolBarSql);
  if (hideToolBarSql) {
    hideToolBarSql = FALSE;
    enableToolBar(KToolBar::Show, 1);
  } else {
    hideToolBarSql = TRUE;
    enableToolBar(KToolBar::Hide, 1);
  }
}

void TopLevel::toggleStatusBar() {

  view->setItemChecked(menuShowSB,hideStatusBar);
  if (hideStatusBar) {
    hideStatusBar = FALSE;
    enableStatusBar(KStatusBar::Show);
  } else {
    hideStatusBar = TRUE;
    enableStatusBar(KStatusBar::Hide);
  }
}

void TopLevel::toggleQueryState() {
  if(connection->queryState() == 0){
    newUndo();
		splitter->setEnabled(TRUE);
    menubar->setItemEnabled(menuFile, TRUE);
    menubar->setItemEnabled(menuEdit, TRUE);
    menubar->setItemEnabled(menuView, TRUE);
    menubar->setItemEnabled(menuOptions, TRUE);
    menubar->setItemEnabled(menuHelp, TRUE);
    query->setItemEnabled(menuExecute, TRUE);
    query->setItemEnabled(menuCancel, FALSE);
    toolBar(0)->setItemEnabled(toolNew, TRUE);
    toolBar(0)->setItemEnabled(toolOpen, TRUE);
    toolBar(0)->setItemEnabled(toolSave, TRUE);
    toolBar(0)->setItemEnabled(toolPrint, TRUE);
    toolBar(0)->setItemEnabled(toolCut, TRUE);
    toolBar(0)->setItemEnabled(toolCopy, TRUE);
    toolBar(0)->setItemEnabled(toolPaste, TRUE);
    toolBar(0)->setItemEnabled(toolSearch, TRUE);
    toolBar(0)->setItemEnabled(toolBookmark, TRUE);
    toolBar(0)->setItemEnabled(toolHelp, TRUE);
    toolBar(1)->setItemEnabled(toolConnect, TRUE);
    toolBar(1)->setItemEnabled(toolOptions, TRUE);
    toolBar(1)->setItemEnabled(toolClear, TRUE);
    toolBar(1)->setItemEnabled(toolSpool, TRUE);
    toolBar(1)->setItemEnabled(toolExecute, TRUE);
    toolBar(1)->setItemEnabled(toolCancel, FALSE);
  }else{
		splitter->setEnabled(FALSE);
    menubar->setItemEnabled(menuFile, FALSE);
    menubar->setItemEnabled(menuEdit, FALSE);
    menubar->setItemEnabled(menuView, FALSE);
    menubar->setItemEnabled(menuOptions, FALSE);
    menubar->setItemEnabled(menuHelp, FALSE);
    query->setItemEnabled(menuExecute, FALSE);
    query->setItemEnabled(menuCancel, TRUE);
    toolBar(0)->setItemEnabled(toolNew, FALSE);
    toolBar(0)->setItemEnabled(toolOpen, FALSE);
    toolBar(0)->setItemEnabled(toolSave, FALSE);
    toolBar(0)->setItemEnabled(toolPrint, FALSE);
    toolBar(0)->setItemEnabled(toolCut, FALSE);
    toolBar(0)->setItemEnabled(toolCopy, FALSE);
    toolBar(0)->setItemEnabled(toolPaste, FALSE);
    toolBar(0)->setItemEnabled(toolUndo, FALSE);
    toolBar(0)->setItemEnabled(toolRedo, FALSE);
    toolBar(0)->setItemEnabled(toolSearch, FALSE);
    toolBar(0)->setItemEnabled(toolBookmark, FALSE);
    toolBar(0)->setItemEnabled(toolHelp, FALSE);
    toolBar(1)->setItemEnabled(toolConnect, FALSE);
    toolBar(1)->setItemEnabled(toolOptions, FALSE);
    toolBar(1)->setItemEnabled(toolClear, FALSE);
    toolBar(1)->setItemEnabled(toolSpool, FALSE);
    toolBar(1)->setItemEnabled(toolExecute, FALSE);
    toolBar(1)->setItemEnabled(toolCancel, TRUE);
  }
}

void TopLevel::helpSelected() {
  kapp->invokeHTMLHelp( "" , "" );
}

void TopLevel::newCurPos() {
  char s[64];

  sprintf(s,"%1.20s: %d %1.20s: %d",i18n("Line"),kWrite->currentLine() +1,
    i18n("Col"),kWrite->currentColumn() +1);
  statusBar()->changeItem(s,ID_LINE_COLUMN);
}

void TopLevel::newStatus() {
  int config;

  config = kWrite->config();
  statusBar()->changeItem(config & cfOvr ? "OVR" : "INS",ID_INS_OVR);
  statusBar()->changeItem(kWrite->isModified() ? "*" : "",ID_MODIFIED);
}

void TopLevel::statusMsg(const char *msg) {
  statusbarTimer->stop();
  statusBar()->changeItem(msg,ID_GENERAL);
  statusbarTimer->start(10000,true); //single shot
}

void TopLevel::timeout() {
  statusBar()->changeItem("",ID_GENERAL);
}

void TopLevel::newCaption() {
  const char *caption;
  int z;

  caption = kWrite->fileName();
  if (caption != 0L && *caption != '\0') {
    setCaption(caption);
    //set recent files popup menu
    z = (int) recentPopup->count();
    while (z > 0) {
      z--;
      if (!strcmp(caption, recentPopup->text(z))) recentPopup->removeItemAt(z);
    }
    recentPopup->insertItem(caption, 0, 0);
    if (recentPopup->count() > 5) recentPopup->removeItemAt(5);
    for (z = 0; z < 5; z++) recentPopup->setId(z, z);
  } else {
    setCaption(i18n("Kpsql"));
  }
}

void TopLevel::newUndo() {
  int state;

  state = kWrite->undoState();
  edit->setItemEnabled(menuUndo,state & 1);
  edit->setItemEnabled(menuRedo,state & 2);
  toolBar(0)->setItemEnabled(toolUndo,state & 1);
  toolBar(0)->setItemEnabled(toolRedo,state & 2);
}

void TopLevel::newConnection() {
  ConnectDialog *dlg;
  LoginStruct login;

  connection->getConnectionInfo(&login);

  dlg = new ConnectDialog(&login, this, "connectdlg");

  if (dlg->exec() == QDialog::Accepted) {
    dlg->getLoginData(&login);
    connection->setConnection(&login);
  }
  delete dlg;
}

void TopLevel::executeQuery(){
  connection->executeQuery();
}

void TopLevel::cancelQuery(){
  connection->cancelQuery();
}

void TopLevel::setOutputOptions() {
  OutputDialog *dlg;
  OptionStruct opt;

  connection->getOptionInfo(&opt);

  dlg = new OutputDialog(&opt, this, "outputdlg");

  if (dlg->exec() == QDialog::Accepted) {
    dlg->getOptionData(&opt);
    connection->setOptions(&opt);
  }
  delete dlg;
}

void TopLevel::setSpoolFile(){
  KFileDialog *dlg;
  QString spoolFile;
  int status;

  dlg = new KFileDialog(connection->getSpoolFile(), "*", this, "spooldialog", TRUE, FALSE);
  dlg->setCaption(i18n("Spool File"));

  do {
    status = 0;
    dlg->exec();
    spoolFile = dlg->selectedFile();
    if (spoolFile.isEmpty()){
      delete dlg;
      return;
    }

    QFileInfo info;
    info.setFile(spoolFile);
    if (info.exists()) {
      status = QMessageBox::warning(dlg,
        i18n("Warning"),
        i18n("A file with this name already exists.\nDo you want to overwrite it?"),
        i18n("Yes"),
        i18n("No"),
        "",0,1);
    }
  } while (status == 1);

  delete dlg;
  connection->setSpoolFile(spoolFile);
}

void TopLevel::copy(){
  if(strlen(kWrite->markedText()) > 0)
    kWrite->copy();
  else if(strlen(results->markedText()) > 0)
    results->copyText();
}

void TopLevel::cut(){
  if(strlen(kWrite->markedText()) > 0)
    kWrite->cut();
}

void TopLevel::print(){
  QString file;
  kWrite->save();

  file = kWrite->fileName();
	if(strlen(file) > 0){
  	CPrintDlg* printerdlg = new CPrintDlg(this, file.replace(QRegExp("file:"), ""), "printdlg");
 	 	printerdlg->setCaption("Print");
   	printerdlg->exec();
  	delete (printerdlg);
	}
	else {
  	KMsgBox::message(this,
                     i18n("Print"),
                     i18n("No file to print!"),
                     KMsgBox::INFORMATION,
                     i18n("&OK"));
	}
}

void TopLevel::about(){
  char info[255];
  const char *d_text = i18n("\nKpsql %s\n\nA Query Tool for PostgresSQL\n\nCopyright (c) 1999 Mutiny Bay Software\n");

  sprintf(info, d_text, VERSION);

  KMsgBox::message(this,
                   i18n("About Kpsql"),
                   info,
                   KMsgBox::INFORMATION,
                   i18n("&OK"));
}

void TopLevel::dropAction(KDNDDropZone *dropZone) {
  char *s;
  int query;
  int cnt = 0;

  QStrList &list = dropZone->getURLList();
  for (s = list.first(); s != 0L && cnt < 1; s = list.next()) {
    // Load the first file in this window
    if(kWrite->isModified()){
      query = QMessageBox::information(this,
      i18n("Drop File"),
      i18n("This query has been modified.\nWould you like to save it?"),
      i18n("Yes"),
      i18n("No"),
      i18n("Cancel"),
      0,2);

      switch (query) {
        case 0: //yes
          kWrite->save();
          if (kWrite->isModified()) return;
          break;
        case 2: //cancel
          return;
      }
    }
    if(s == list.getFirst())
      loadURL(s);
    cnt++;
  }
}

void TopLevel::showHighlight()
{
  int hl=kWrite->doc()->getHighlight();

  for (int index = 0; index < (int) hlPopup->count(); index++)
    hlPopup->setItemChecked(index, hl == index);
}

//common config
void TopLevel::readConfig(KConfig *config) {
  int z;
  char name[16];
  QString s;

  hideToolBar = config->readNumEntry("HideToolBar");
  hideToolBarSql = config->readNumEntry("HideToolBarSql");
  hideStatusBar = config->readNumEntry("HideStatusBar");

  for (z = 0; z < 5; z++) {
    sprintf(name, "Recent%d", z + 1);
    s = config->readEntry(name);
    if (!s.isEmpty()) recentPopup->insertItem(s);
  }
}

void TopLevel::writeConfig(KConfig *config) {
  int z;
  char name[16];
  QString string;

  config->writeEntry("HideToolBar",hideToolBar);
  config->writeEntry("HideToolBarSql",hideToolBarSql);
  config->writeEntry("HideStatusBar",hideStatusBar);

  for (z = 0; z < (int) recentPopup->count(); z++) {
    sprintf(name, "Recent%d", z + 1);
    config->writeEntry(name, recentPopup->text(z));
  }
}

//config file
void TopLevel::readConfig() {
  KConfig *config;
  int w, h;

  config = kapp->getConfig();

  config->setGroup("General Options");
  w = config->readNumEntry("Width",550);
  h = config->readNumEntry("Height",400);
  resize(w,h);

  readConfig(config);

  kWrite->readConfig(config);
  kWrite->doc()->readConfig(config);
  results->readConfig(config);
  connection->readConfig(config);
}

void TopLevel::writeConfig() {
  KConfig *config;

  config = kapp->getConfig();

  config->setGroup("General Options");
  config->writeEntry("Width",width());
  config->writeEntry("Height",height());

  writeConfig(config);

  kWrite->writeConfig(config);
  kWrite->doc()->writeConfig(config);
  results->writeConfig(config);
  connection->writeConfig(config);
}

// session management
void TopLevel::restore(KConfig *config, int n) {
  const char *url;

  if (kWrite->isLastView()) { //in this case first view
    url = kWrite->fileName();
    if (url && *url) loadURL(url,lfNoAutoHl);
  }
  readPropertiesInternal(config,n);
  init();
}

void TopLevel::readProperties(KConfig *config) {

  readConfig(config);
  kWrite->readSessionConfig(config);
}

void TopLevel::saveProperties(KConfig *config) {

  writeConfig(config);
  config->writeEntry("DocumentNumber",docList.find(kWrite->doc()) + 1);
  kWrite->writeSessionConfig(config);
  setUnsavedData(kWrite->isModified());
}

void TopLevel::saveData(KConfig *config) { //save documents
  int z;
  char buf[16];
  KWriteDoc *doc;

  config->setGroup("Number");
  config->writeEntry("NumberOfDocuments",docList.count());

  for (z = 1; z <= (int) docList.count(); z++) {
     sprintf(buf,"Document%d",z);
     config->setGroup(buf);
     doc = docList.at(z - 1);
     doc->writeSessionConfig(config);
  }
}

//restore session
void restore() {
  KConfig *config;
  int docs, windows, z;
  char buf[16];
  KWriteDoc *doc;
  TopLevel *t;

  config = kapp->getSessionConfig();
  if (!config) return;

  config->setGroup("Number");
  docs = config->readNumEntry("NumberOfDocuments");
  windows = config->readNumEntry("NumberOfWindows");

  for (z = 1; z <= docs; z++) {
     sprintf(buf,"Document%d",z);
     config->setGroup(buf);
     doc = new KWriteDoc(&hlManager);
     doc->readSessionConfig(config);
     docList.append(doc);
  }

  for (z = 1; z <= windows; z++) {
    sprintf(buf,"%d",z);
    config->setGroup(buf);
    t = new TopLevel(docList.at(config->readNumEntry("DocumentNumber") - 1));
    t->restore(config,z);
  }
}

int main(int argc, char** argv) {
  KApplication a(argc,argv);

  docList.setAutoDelete(false);

  if (kapp->isRestored()) {
    restore();
  } else {
    TopLevel *t = new TopLevel();
    t->readConfig();
    if (argc > 1) t->loadURL(argv[1],lfNewFile);
    t->init();
  }
  return a.exec();
}








































































































































































